/**
 * @file	moncell.c
 * @author  Zach Habersang
 * @brief 	Network Monitor Cell
 *
 * Copyright (C) 2008 Zach Habersang
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include "moncells.h"
#include "moncell.h"

/**
 * Monitor Application Widget Constructor.
 * @return liqcell The newly created fullscreen liqcell
 * 
 */
liqcell* Monitor_App_Create() 
{	
	// create the fullscreen widget
	liqcell *self = liqcell_quickcreatewidget(APP_NAME, "form", 800, 480);
	
	if (self) {
		
		//
		// fonts
		//

	    font = liqfont_cache_getttf("/usr/share/fonts/nokia/nosnb.ttf", (32), 0);
		font_h = liqfont_textheight(font);

	    lfont = liqfont_cache_getttf("/usr/share/fonts/nokia/nosnb.ttf", (52), 0);
		lfont_h = liqfont_textheight(lfont);
		
	    xlfont = liqfont_cache_getttf("/usr/share/fonts/nokia/nosnb.ttf", (72), 0);
		xlfont_h = liqfont_textheight(xlfont);
		
	    xxlfont = liqfont_cache_getttf("/usr/share/fonts/nokia/nosnb.ttf", (128), 0);
		xxlfont_h = liqfont_textheight(xxlfont);
		
		// create body
		liqcell *body = liqcell_quickcreatevis("body", "grid", 0, 0, 800, 480);
		liqcell_child_append(self, body);
		liqcell_child_arrange_easycol(self);	
	}
	return self;
}

/**
 * Update the displayed moncell content
 * @param MonArgs Data Received from the server, parsed and split
 * @param Monitor_Data Moncell's data
 */
void Monitor_Updata_Data(char MonArgs[NUM_PROTOCOL_FIELDS][MAXSENDSIZE], struct Mon_App_Args *Monitor_Data)
{
	char buf[256];
	char *key = MonArgs[0];
	char *simple_data = MonArgs[1]; // if their is only one command delimiter
	
	if (!(Monitor_Data->isInit))
		Init_Monitor_Data(Monitor_Data); 
		
	// liqapp_log("STUFF: %s @ %s @ %s @ %s", MonArgs[1],MonArgs[2],MonArgs[3],MonArgs[4]);

	// --------------------------------------------------
	// SYS :: System Information
	// --------------------------------------------------
	if (!strcmp(key, SYS))
		moncell_update_sys(Monitor_Data, MonArgs);
	
	// --------------------------------------------------
	// CPU :: Number of Processes Running
	// --------------------------------------------------
	else if (!strcmp(key, CPU))
		moncell_update_cpu(Monitor_Data, MonArgs);
		
	// --------------------------------------------------
	// MEM :: RAM, SWAP
	// --------------------------------------------------
	else if (!strcmp(key, MEM))
		moncell_update_mem(Monitor_Data, MonArgs);
		
	// --------------------------------------------------
	// PROCS :: Procs, Top
	// --------------------------------------------------
	else if (!strcmp(key, PROCS))
		moncell_update_procs(Monitor_Data, MonArgs);
	
	// update content!
	liqcell_setdirty(Monitor_Data->body, 1);
	liqcell *appcell = liqcell_getlinkparent(Monitor_Data->body);
	liqcell_setdirty(appcell, 1);

}

// ################################################################################################
// ################################################################################################ updates
// ################################################################################################

void moncell_update_sys(struct Mon_App_Args *mon, char MonArgs[NUM_PROTOCOL_FIELDS][MAXSENDSIZE])
{
	char buf[256];
	
	// data: uptime, sysname, nodename, release, machine
	char *uptime   = MonArgs[1];
	char *sysname  = MonArgs[2];
	char *nodename = MonArgs[3];
	char *release  = MonArgs[4];
	char *machine  = MonArgs[5];
	
	snprintf(buf, 256, "Sysname: %s", sysname);
	liqcell_setcaption(mon->sys.sysname, buf);
	
	snprintf(buf, 256, "Nodename: %s", nodename);
	liqcell_setcaption(mon->sys.nodename, buf);
	
	snprintf(buf, 256, "Release: %s", release);
	liqcell_setcaption(mon->sys.release, buf);
	
	snprintf(buf, 256, "Machine: %s", machine);
	liqcell_setcaption(mon->sys.machine, buf);
	
	
	liqcell_setcaption(mon->sys.uptime, uptime);
		
		
	liqcell_setdirty(mon->sys.body, 1);
}

void moncell_update_cpu(struct Mon_App_Args *mon, char MonArgs[NUM_PROTOCOL_FIELDS][MAXSENDSIZE])
{
	char buf[256];
	
	int gw; // graph width
	int usei; // usage int
	
	// data: loads, cpu usage, freq, cpu count
	char *loads   = MonArgs[1];
	char *usage   = MonArgs[2];
	char *freq    = MonArgs[3];
	char *count   = MonArgs[4];
	
	snprintf(buf, 256, "Core x%s", count);
	liqcell_setcaption(mon->cpu.cpu_count, buf);
	
	snprintf(buf, 256, "Freq: %s MHz", freq);
	liqcell_setcaption(mon->cpu.freq, buf);
	
	snprintf(buf, 256, "Loadavg: %s", loads);
	liqcell_setcaption(mon->cpu.load, buf);
	
	usei = atoi(usage);
	gw = (int)(((float)usei / 100.0) * 740.0); 
	liqcell_setrect(mon->cpu.usage_graph, 30, 320, gw, xxlfont_h);
	
	snprintf(buf, 256, "%s%%", usage, freq);
	liqcell_setcaption(mon->cpu.cpu_usage, buf);

	liqcell_setdirty(mon->cpu.body, 1);
}

void moncell_update_mem(struct Mon_App_Args *mon, char MonArgs[NUM_PROTOCOL_FIELDS][MAXSENDSIZE])
{
	char buf[256];
	
	int memi, memw, swapi, swapw; // integer and width vars
	
	// data: memused, memmax, memusage, swapused, swapmax, swapusage
	char *memused    = MonArgs[1];
	char *memmax     = MonArgs[2];
	char *memusage   = MonArgs[3];
	char *swapused   = MonArgs[4];
	char *swapmax    = MonArgs[5];
	char *swapusage  = MonArgs[6];
	
	snprintf(buf, 256, "RAM: %s MiB of %s MiB", memused, memmax);
	liqcell_setcaption(mon->mem.ram_stat, buf);
	
	memi = atoi(memusage);
	memw = (int)(((float)memi / 100.0) * 740.0); 
	liqcell_setrect(mon->mem.ram_usage_graph, 30, 180, memw, xlfont_h);
	
	snprintf(buf, 256, "%s%%", memusage);
	liqcell_setcaption(mon->mem.ram_usage, buf);
	
	snprintf(buf, 256, "SWAP: %s MiB of %s MiB", swapused, swapmax);
	liqcell_setcaption(mon->mem.swap_stat, buf);
	
	swapi = atoi(swapusage);
	swapw = (int)(((float)swapi / 100.0) * 740.0); 
	liqcell_setrect(mon->mem.swap_usage_graph, 30, 340, swapw, xlfont_h);
	
	snprintf(buf, 256, "%s%%", swapusage);
	liqcell_setcaption(mon->mem.swap_usage, buf);
	

	liqcell_setdirty(mon->mem.body, 1);
}

void moncell_update_procs(struct Mon_App_Args *mon, char MonArgs[NUM_PROTOCOL_FIELDS][MAXSENDSIZE])
{
	char buf[256];
	
	// data: procs, run_procs, top 1-5
	char *procs      = MonArgs[1];
	char *run_procs  = MonArgs[2];
	char *top1       = MonArgs[3];
	char *top2       = MonArgs[4];
	char *top3       = MonArgs[5];
	char *top4       = MonArgs[6];
	char *top5       = MonArgs[7];
	
	snprintf(buf, 256, "Running Processes: %s", run_procs);
	liqcell_setcaption(mon->proc.run_procs, buf);
	
	snprintf(buf, 256, "Total Processes: %s", procs);
	liqcell_setcaption(mon->proc.procs, buf);
	
	// top
	liqcell_setcaption(mon->proc.top1, top1);
	liqcell_setcaption(mon->proc.top2, top2);
	liqcell_setcaption(mon->proc.top3, top3);
	liqcell_setcaption(mon->proc.top4, top4);
	liqcell_setcaption(mon->proc.top5, top5);
	
	
	liqcell_setdirty(mon->proc.body, 1);
}

// ################################################################################################
// ################################################################################################
// ################################################################################################

/* 
 * --------------------------------------------
 * Widget Handling Functions
 * --------------------------------------------
 */

static int widget_click(liqcell *self, liqcellclickeventargs *args)
{
	args->newdialogtoopen = liqcell_getcontent(self);
	return 1;
}

/**
 * Create initial data components for the Monitor Data
 * @param Monitor_Data Moncell's Data
 */
void Init_Monitor_Data(struct Mon_App_Args *Monitor_Data)
{
	char buf[256];
	int line_c = 0;
	
	Monitor_Data->isInit = 1; // set flag
	
	//
	// title
	//
	
	liqcell *moncell_title = liqcell_quickcreatevis(Monitor_Data->hostname, "label", 0, 200, 800, 80);
	liqcell_setfont(moncell_title, liqfont_hold(xlfont));
	snprintf(buf, 256, "%s", Monitor_Data->hostname);
	liqcell_setcaption(moncell_title, buf);
	
	liqcell_propsets(moncell_title, "textcolor", GREEN);
	liqcell_propsets(moncell_title, "backcolor", DARK);

	liqcell_propseti(moncell_title, "textalign",  2);
	liqcell_propseti(moncell_title, "textaligny", 2);

	liqcell_child_insert(Monitor_Data->body, moncell_title);	
	
	// body bg
	liqcell_propsets(Monitor_Data->body, "backcolor", DARK);
	
	//
	// system
	//
	moncell_sys_init(Monitor_Data);
	
	//
	// cpu
	//
	moncell_cpu_init(Monitor_Data);
	
	//
	// mem
	//
	moncell_mem_init(Monitor_Data);
	
	//
	// procs
	//
	moncell_procs_init(Monitor_Data);
}

// ################################################################################################
// ################################################################################################ inits
// ################################################################################################

/*
 * 
 * SYSTEM INIT (SYS)
 * 
 */
void moncell_sys_init(struct Mon_App_Args *mon)
{
	//
	// main system cell
	//
	
	
	liqcell *wrap;
	char wrap_name[64];

	snprintf(wrap_name, 64, "%s_wrap", SYS);
	mon->sys.body = liqcell_quickcreatevis(SYS, "label", 0, 0, 800, 480);
	//liqcell_setfont(mon->sys.body, liqfont_hold(large_font));
	
	liqcell_propsets(mon->sys.body, "textcolor", WHITE);
	//liqcell_propsets(mon->sys.body, "backcolor", BLUE);
	liqcell_setimage(mon->sys.body, liqimage_cache_getfile(MONCELL_BG, 0, 0, 1));

	//liqcell_propseti(mon->sys.body, "textalign",  2);
	//liqcell_propseti(mon->sys.body, "textaligny", 2);
	
	wrap = liqcell_quickcreatevis(wrap_name, NULL, 0, 0, 400, 200);
	liqcell_handleradd(wrap, "click", widget_click);
	liqcell_setcontent(wrap, mon->sys.body);
	liqcell_propsets(wrap, "backcolor", DARK);
	liqcell_propseti(wrap, "lockaspect", 1);

	liqcell_child_insert(mon->body, wrap);	
	
	
	//
	// content
	// 
	
	// data: uptime, sysname, nodename, release, machine
	
	liqcell *title = liqcell_quickcreatevis(SYS, "label", 30, 20, 800, xlfont_h);
	liqcell_setfont(title, liqfont_hold(xlfont));
	liqcell_setcaption(title, SYS);
	liqcell_child_insert(mon->sys.body, title);	

	// sysinfo

	mon->sys.sysname = liqcell_quickcreatevis("sysname", "label", 30, 120, 800, font_h);
	liqcell_setfont(mon->sys.sysname, liqfont_hold(font));
	liqcell_child_insert(mon->sys.body, mon->sys.sysname);	

	mon->sys.nodename = liqcell_quickcreatevis("nodename", "label", 30, 156, 800, font_h);
	liqcell_setfont(mon->sys.nodename, liqfont_hold(font));
	liqcell_child_insert(mon->sys.body, mon->sys.nodename);		
	
	mon->sys.release = liqcell_quickcreatevis("release", "label", 30, 192, 800, font_h);
	liqcell_setfont(mon->sys.release, liqfont_hold(font));
	liqcell_child_insert(mon->sys.body, mon->sys.release);		
	
	mon->sys.machine = liqcell_quickcreatevis("machine", "label", 30, 228, 800, font_h);
	liqcell_setfont(mon->sys.machine, liqfont_hold(font));
	liqcell_child_insert(mon->sys.body, mon->sys.machine);	
	
	
	// uptime
	
	mon->sys.uptime = liqcell_quickcreatevis("uptime", "label", 30, 320, 800, xxlfont_h);
	liqcell_setfont(mon->sys.uptime, liqfont_hold(xxlfont));
	liqcell_child_insert(mon->sys.body, mon->sys.uptime);		
}

/*
 * 
 * CENTRAL PROCESSING UNIT INIT (CPU)
 * 
 */
void moncell_cpu_init(struct Mon_App_Args *mon)
{
	//
	// main cpu cell
	//
	
	liqcell *wrap;
	char wrap_name[64];

	snprintf(wrap_name, 64, "%s_wrap", CPU);
	mon->cpu.body = liqcell_quickcreatevis(CPU, "label", 0, 0, 800, 480);
	//liqcell_setfont(mon->cpu.body, liqfont_hold(large_font));
	
	liqcell_propsets(mon->cpu.body, "textcolor", WHITE);
	//liqcell_propsets(mon->cpu.body, "backcolor", RED);
	liqcell_setimage(mon->cpu.body, liqimage_cache_getfile(MONCELL_BG, 0, 0, 1));

	//liqcell_propseti(mon->cpu.body, "textalign",  2);
	//liqcell_propseti(mon->cpu.body, "textaligny", 2);
	
	wrap = liqcell_quickcreatevis(wrap_name, NULL, 400, 0, 400, 200);
	liqcell_handleradd(wrap, "click", widget_click);
	liqcell_setcontent(wrap, mon->cpu.body);
	liqcell_propsets(wrap, "backcolor", DARK);
	liqcell_propseti(wrap, "lockaspect", 1);

	liqcell_child_insert(mon->body, wrap);	
	
	
	//
	// content
	// 
	
	// cells: loads, usage, count
	
	liqcell *title = liqcell_quickcreatevis(CPU, "label", 30, 20, 800, xlfont_h);
	liqcell_setfont(title, liqfont_hold(xlfont));
	liqcell_setcaption(title, CPU);
	liqcell_child_insert(mon->cpu.body, title);	
	
	mon->cpu.cpu_count = liqcell_quickcreatevis("cpu_count", "label", 30, 100, 800, font_h);
	liqcell_setfont(mon->cpu.cpu_count, liqfont_hold(font));
	liqcell_child_insert(mon->cpu.body, mon->cpu.cpu_count);
	
	mon->cpu.freq = liqcell_quickcreatevis("freq", "label", 30, 150, 800, lfont_h);
	liqcell_setfont(mon->cpu.freq, liqfont_hold(lfont));
	liqcell_child_insert(mon->cpu.body, mon->cpu.freq);		

	mon->cpu.load = liqcell_quickcreatevis("load", "label", 30, 210, 800, lfont_h);
	liqcell_setfont(mon->cpu.load, liqfont_hold(lfont));
	liqcell_child_insert(mon->cpu.body, mon->cpu.load);		
	
	mon->cpu.cpu_usage = liqcell_quickcreatevis("cpu_usage", "label", 520, 320, 800, xxlfont_h);
	liqcell_setfont(mon->cpu.cpu_usage, liqfont_hold(xxlfont));
	liqcell_child_insert(mon->cpu.body, mon->cpu.cpu_usage);
	
	mon->cpu.usage_graph = liqcell_quickcreatevis("usage_graph", "label", 30, 320, 740, xxlfont_h);
	liqcell_propsets(mon->cpu.usage_graph, "backcolor", GREEN);
	liqcell_child_insert(mon->cpu.body, mon->cpu.usage_graph);	
}

/*
 * 
 * MEMORY (MEM)
 * 
 */
void moncell_mem_init(struct Mon_App_Args *mon)
{
	//
	// main mem cell
	//
	
	liqcell *wrap;
	char wrap_name[64];

	snprintf(wrap_name, 64, "%s_wrap", MEM);
	mon->mem.body = liqcell_quickcreatevis(MEM, "label", 0, 0, 800, 480);
	//liqcell_setfont(mon->mem.body, liqfont_hold(large_font));
	
	liqcell_propsets(mon->mem.body, "textcolor", WHITE);
	//liqcell_propsets(mon->mem.body, "backcolor", RED);
	liqcell_setimage(mon->mem.body, liqimage_cache_getfile(MONCELL_BG, 0, 0, 1));

	//liqcell_propseti(mon->mem.body, "textalign",  2);
	//liqcell_propseti(mon->mem.body, "textaligny", 2);
	
	wrap = liqcell_quickcreatevis(wrap_name, NULL, 0, 280, 400, 200);
	liqcell_handleradd(wrap, "click", widget_click);
	liqcell_setcontent(wrap, mon->mem.body);
	liqcell_propsets(wrap, "backcolor", DARK);
	liqcell_propseti(wrap, "lockaspect", 1);

	liqcell_child_insert(mon->body, wrap);	
	
	
	//
	// content
	//
	
	liqcell *title = liqcell_quickcreatevis(MEM, "label", 30, 20, 800, xlfont_h);
	liqcell_setfont(title, liqfont_hold(xlfont));
	liqcell_setcaption(title, MEM);
	liqcell_child_insert(mon->mem.body, title);	

	mon->mem.ram_stat = liqcell_quickcreatevis("ram_stat", "label", 30, 120, 800, font_h);
	liqcell_setfont(mon->mem.ram_stat, liqfont_hold(font));
	liqcell_child_insert(mon->mem.body, mon->mem.ram_stat);	
	
	mon->mem.ram_usage = liqcell_quickcreatevis("ram_usage", "label", 520, 180, 800, xlfont_h);
	liqcell_setfont(mon->mem.ram_usage, liqfont_hold(xlfont));
	liqcell_child_insert(mon->mem.body, mon->mem.ram_usage);	
	
	mon->mem.ram_usage_graph = liqcell_quickcreatevis("ram_usage_graph", "label", 30, 180, 740, xlfont_h);
	liqcell_propsets(mon->mem.ram_usage_graph, "backcolor", RED);
	liqcell_child_insert(mon->mem.body, mon->mem.ram_usage_graph);	

	mon->mem.swap_stat = liqcell_quickcreatevis("swap_stat", "label", 30, 280, 800, font_h);
	liqcell_setfont(mon->mem.swap_stat, liqfont_hold(font));
	liqcell_child_insert(mon->mem.body, mon->mem.swap_stat);		
	
	mon->mem.swap_usage = liqcell_quickcreatevis("swap_usage", "label", 520, 340, 800, xlfont_h);
	liqcell_setfont(mon->mem.swap_usage, liqfont_hold(xlfont));
	liqcell_child_insert(mon->mem.body, mon->mem.swap_usage);
	
	mon->mem.swap_usage_graph = liqcell_quickcreatevis("swap_usage_graph", "label", 30, 340, 740, xlfont_h);
	liqcell_propsets(mon->mem.swap_usage_graph, "backcolor", BLUE);
	liqcell_child_insert(mon->mem.body, mon->mem.swap_usage_graph);	
}

/*
 * 
 * PROCESSES (PROCS)
 * 
 */
void moncell_procs_init(struct Mon_App_Args *mon)
{
	//
	// main procs cell
	//
	
	liqcell *wrap;
	char wrap_name[64];

	snprintf(wrap_name, 64, "%s_wrap", PROCS);
	mon->proc.body = liqcell_quickcreatevis(PROCS, "label", 0, 0, 800, 480);
	//liqcell_setfont(mon->proc.body, liqfont_hold(large_font));
	
	liqcell_propsets(mon->proc.body, "textcolor", WHITE);
	//liqcell_propsets(mon->proc.body, "backcolor", BLUE);
	liqcell_setimage(mon->proc.body, liqimage_cache_getfile(MONCELL_BG, 0, 0, 1));

	//liqcell_propseti(mon->proc.body, "textalign",  2);
	//liqcell_propseti(mon->proc.body, "textaligny", 2);
	
	wrap = liqcell_quickcreatevis(wrap_name, NULL, 400, 280, 400, 200);
	liqcell_handleradd(wrap, "click", widget_click);
	liqcell_setcontent(wrap, mon->proc.body);
	liqcell_propsets(wrap, "backcolor", DARK);
	liqcell_propseti(wrap, "lockaspect", 1);

	liqcell_child_insert(mon->body, wrap);	
	
	
	//
	// content
	// 
	
	liqcell *title = liqcell_quickcreatevis(PROCS, "label", 30, 20, 800, xlfont_h);
	liqcell_setfont(title, liqfont_hold(xlfont));
	liqcell_setcaption(title, "PROCESSES");
	liqcell_child_insert(mon->proc.body, title);	

	mon->proc.procs = liqcell_quickcreatevis("procs", "label", 30, 100, 800, font_h);
	liqcell_setfont(mon->proc.procs, liqfont_hold(font));
	liqcell_child_insert(mon->proc.body, mon->proc.procs);	

	mon->proc.run_procs = liqcell_quickcreatevis("run_procs", "label", 30, 140, 800, font_h);
	liqcell_setfont(mon->proc.run_procs, liqfont_hold(font));
	liqcell_child_insert(mon->proc.body, mon->proc.run_procs);
	
	//
	// top
	//
	
	liqcell *top = liqcell_quickcreatevis("top", "label", 60, 200, 800, font_h);
	liqcell_setfont(top, liqfont_hold(font));
	liqcell_setcaption(top, "NAME PID CPU% MEM%");
	liqcell_child_insert(mon->proc.body, top);
	
	mon->proc.top1 = liqcell_quickcreatevis("top1", "label", 60, 240, 800, font_h);
	liqcell_setfont(mon->proc.top1, liqfont_hold(font));
	liqcell_child_insert(mon->proc.body, mon->proc.top1);	
	
	mon->proc.top2 = liqcell_quickcreatevis("top2", "label", 60, 280, 800, font_h);
	liqcell_setfont(mon->proc.top2, liqfont_hold(font));
	liqcell_child_insert(mon->proc.body, mon->proc.top2);
	
	mon->proc.top3 = liqcell_quickcreatevis("top3", "label", 60, 320, 800, font_h);
	liqcell_setfont(mon->proc.top3, liqfont_hold(font));
	liqcell_child_insert(mon->proc.body, mon->proc.top3);
	
	mon->proc.top4 = liqcell_quickcreatevis("top4", "label", 60, 360, 800, font_h);
	liqcell_setfont(mon->proc.top4, liqfont_hold(font));
	liqcell_child_insert(mon->proc.body, mon->proc.top4);
	
	mon->proc.top5 = liqcell_quickcreatevis("top5", "label", 60, 400, 800, font_h);
	liqcell_setfont(mon->proc.top5, liqfont_hold(font));
	liqcell_child_insert(mon->proc.body, mon->proc.top5);
			
}

// ################################################################################################
// ################################################################################################
// ################################################################################################

/**
 * Build a moncell, don't add content yet, wait until the shown event executes
 * @return liqcell* Newly created moncell
 */
liqcell *moncell_create()
{
	liqcell *self = Monitor_App_Create();  
	 
	if (self)
	{
		liqcell_handleradd(self, "shown", moncell_shown);
	}
	return self;
}

/**
 * Moncell shown event handler, executes when the liqcell is first shown
 */
static moncell_shown(liqcell *self, liqcelleventargs *args, void *context)
{
	// get ip
	char *hostname = liqcell_propgets(self, "hostname", NULL);
	int servsock = liqcell_propgeti(self, "servsock", -1);
	
	// get body
	liqcell *body = liqcell_child_lookup(self, "body");
	
	// run client service
	if (Moncell_Client_Thread(hostname, body, servsock) < 0)
		liqapp_log("Failed to run Moncell connected to host: %s", hostname);

	return 1;
}

/**
 * Shutdown a socket
 * @param errorMsg The error
 * @param socket The socket to close
 * @return int Failure
 */
int SocketShutdown(char *errorMsg, int socket)
{
	liqapp_log("Shutting Down Socket! :: Error: %s", errorMsg);
	if (socket) close(socket);
	return -1;
}


/**
 * Init and run a client thread connection
 * @param hostname 
 */
int Moncell_Client_Thread(char *hostname, liqcell *body, int servsock) 
{
	pthread_t threadID;
	struct ThreadArgs *threadArgs;
		
	// Create separate memory for client argument 
	if ((threadArgs = (struct ThreadArgs *)malloc(sizeof(struct ThreadArgs))) == NULL)
		return SocketShutdown("thread malloc() failed", servsock);
		
	// Save client socket
	threadArgs->socket = servsock;
	threadArgs->body = body;
	threadArgs->hostname = hostname;
	
	// Create client thread
	if (pthread_create(&threadID, NULL, ClientThreadMain, (void *) threadArgs) != 0)
		return SocketShutdown("pthread_create() failed", servsock);
	else
		liqapp_log("Thread creation complete :: THREAD ID: %ld\n", (long int) threadID);
		
	return EXIT_SUCCESS;
}

/**
 * Thread Main Function
 * @param threadArgs The arguements passed to this thread main
 */
void *ClientThreadMain(void *threadArgs)
{
	int sock;
	liqcell *body;
	char *hostname;
	
	// Gurantees that thread resources are deallocated upon return
	pthread_detach(pthread_self());
	
	// Extract socket file descriptor from argument, free thread arugments
	sock      = ((struct ThreadArgs *) threadArgs) -> socket;
	body      = ((struct ThreadArgs *) threadArgs) -> body;
	hostname  = ((struct ThreadArgs *) threadArgs) -> hostname;
	free(threadArgs);
	
	Recv_From_Server(sock, body, hostname);
	
	return (NULL);
}

/**
 * Receive data from the monitor server daemon
 * @param sock The socket to read from
 * @param body The liqcell the data will go on
 * @param hostname Hostname connected to
 */
void Recv_From_Server(int sock, liqcell *body, char *hostname)
{
	FILE *in;
	char recvBuf[MAXSENDSIZE]; 

	in = fdopen(sock, "r");
	
	// init monitor data
	struct Mon_App_Args Monitor_Data;
	Monitor_Data.body = body;
	Monitor_Data.hostname = hostname;
 
    // Get data from server
	do 
	{
		fgets(recvBuf, RCVSIZE, in);
		Parse_Monitor_Data(recvBuf, &Monitor_Data);
		
	} while (recvBuf[0] != '\0');
	
    shutdown(sock, SHUT_RDWR);
    close(sock);
}

/**
 * Parse and split the data received from a monitor server
 * @param recvBuf Raw data received
 * @param Monitor_Data Moncell's Data
 */
void Parse_Monitor_Data(char recvBuf[], struct Mon_App_Args *Monitor_Data)
{
	char MonArgs[NUM_PROTOCOL_FIELDS][MAXSENDSIZE];

	// Pointer to each character in linebuf
    char *p; 
  
    // Counts
    int i = 0, argc = 0;
  
    // Point to first character of linebuf
    p = &recvBuf[0];  
	
	// :: PARSE EXAMPLE :: "UPTIME 1481481\n"
  
	// Go through each char in linebuf until you hit a \n || \0
    for(; ((*p != MSG_DELIMIT) && (*p != '\0') && (argc < NUM_PROTOCOL_FIELDS)); p++) 
    {
		// Split <key> and <data>
		if (*p == CMD_DELIMIT) 
		{
			MonArgs[argc][i] = '\0';
			argc++;
			i = 0;
		} 
		
		// Build the actual strings
		else 
		{
			MonArgs[argc][i] = *p;
			i++;
		}
	}
	
	// Close the argument string
	MonArgs[argc][i] = '\0';
	argc++;

	Monitor_Updata_Data(MonArgs, Monitor_Data);
}
