/**
 * @file	hostnames.c
 * @author  Zach Habersang
 * @brief 	Hostnames Config Cell
 *
 * Copyright (C) 2008 Zach Habersang
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include "moncells.h"

#define LABEL_HEIGHT 82
#define Y_SPACE 44

int generate_hostnames(liqcell *self);
void create_hostbox(liqcell *hostnames, char *hostname, int count);

/**	Save Hostnames */	
static int hostnames_save_click(liqcell *self, liqcelleventargs *args, liqcell *hostnames)
{
	FILE *f;

	f = fopen(HOSTNAME_FILE, "w");

	if (f)
	{
		liqcell *hostx;
		int count = 1;
		char cell_name[32];
		snprintf(cell_name, 32, "hostname%d", count);

		while ((hostx = liqcell_child_lookup(hostnames, cell_name)) != NULL)
		{	
			char *hostname = liqcell_getcaption(hostx);
			if (strcmp(hostname, "")) // only write to file if the line isn't ""
				fprintf(f, "%s\n", hostname);
			
			count++;
			snprintf(cell_name, 32, "hostname%d", count);
		}
		fclose(f);
	}
	
	return 1;
}

/**	Reset Hostname Textbox */	
static int hostnames_reset_click(liqcell *self, liqcelleventargs *args, liqcell *hostnames)
{
	FILE *hnconf = fopen(HOSTNAME_FILE, "r");
	
	int count = 0;
	
	if (hnconf) 
	{
		char buf[256], hostname[128];

		while (fgets(buf, 256, hnconf)) 
		{
			sscanf(buf, "%128s", hostname);
			count++;
			
			char cell_name[32];
			snprintf(cell_name, 32, "hostname%d", count);
			
			liqcell *hostx = liqcell_child_lookup(hostnames, cell_name);
			liqcell_setcaption(hostx, hostname);
		}
		fclose(hnconf);
		
		// take care of no saved boxes
		liqcell *hostx;
		count++;
		char cell_name[32];
		snprintf(cell_name, 32, "hostname%d", count);
		
		while ((hostx = liqcell_child_lookup(hostnames, cell_name)) != NULL)
		{
			liqcell_setcaption(hostx, "");
			
			count++;
			snprintf(cell_name, 32, "hostname%d", count);
		}
		
		
	} 

	return 1;
}

/**	New Hostnames */	
static int hostnames_new_click(liqcell *self, liqcelleventargs *args, liqcell *hostnames)
{
	int count = liqcell_propgeti(hostnames, "hostname_count", 0);
	
	count++;
	liqcell_propseti(hostnames, "hostname_count", count); // make sure you get the new number
	char *hostname = "";
	create_hostbox(hostnames, hostname, count);
	
	return 1;
}


liqcell *hostnames_create()
{
	liqcell *self = liqcell_quickcreatewidget("hostnames", "form", 800, 480);
	
	if(!self)
	{
		liqapp_log("liqcell error not create 'hostnames'"); 
		return NULL;
	}
	
	// divide self
	liqcell *hostnames = liqcell_quickcreatevis("hostname_body", "form", 0, 0, 650, 480);
	liqcell_propsets(hostnames, "backcolor", DARK);
	liqcell_child_append(self, hostnames);
	
	liqcell *buttons = liqcell_quickcreatevis("hostname_buttons", "form", 650, 0, 150, 480);
	liqcell_propsets(buttons, "backcolor", BLACK);
	liqcell_child_append(self, buttons);
	
	
	// title
	liqcell *title = liqcell_quickcreatevis("hostname_config_title", "label", 30, 20, 650, 60);
	liqcell_setfont(title, liqfont_cache_getttf("/usr/share/fonts/nokia/nosnb.ttf", (52), 0));
	liqcell_setcaption(title, "Autoconnect Hostnames");
	liqcell_child_insert(hostnames, title);	


	if (generate_hostnames(hostnames) < 0)
		liqapp_log("!-- UNABLE TO READ HOSTNAMES.CONF --!"); 
		
		
	// new button
	liqcell *new = liqcell_quickcreatevis("hostname_new", "commandbutton", 25, 150, 100, 50);
	liqcell_setfont(new, liqfont_cache_getttf("/usr/share/fonts/nokia/nosnb.ttf", (36), 0) );
	liqcell_setcaption(new, "New");
	liqcell_propsets(new, "textcolor", WHITE);
	liqcell_propsets(new, "backcolor", CYAN);
	liqcell_handleradd_withcontext(new, "click", hostnames_new_click, hostnames);
	liqcell_propseti(new, "textalign", 2);
	liqcell_propseti(new, "textaligny", 2);
	liqcell_child_append(buttons, new);
		
		
	// save button
	liqcell *save = liqcell_quickcreatevis("hostname_save", "commandbutton", 25, 210, 100, 50);
	liqcell_setfont(save, liqfont_cache_getttf("/usr/share/fonts/nokia/nosnb.ttf", (36), 0) );
	liqcell_setcaption(save, "Save");
	liqcell_propsets(save, "textcolor", WHITE);
	liqcell_propsets(save, "backcolor", GREEN);
	liqcell_handleradd_withcontext(save, "click", hostnames_save_click, hostnames);
	liqcell_propseti(save, "textalign", 2);
	liqcell_propseti(save, "textaligny", 2);
	liqcell_child_append(buttons, save);
		
	// reset button
	liqcell *reset = liqcell_quickcreatevis("hostname_reset", "commandbutton", 25, 270, 100, 50);
	liqcell_setfont(reset, liqfont_cache_getttf("/usr/share/fonts/nokia/nosnb.ttf", (36), 0) );
	liqcell_setcaption(reset, "Reset");
	liqcell_propsets(reset, "textcolor", WHITE);
	liqcell_propsets(reset, "backcolor", RED);
	liqcell_handleradd_withcontext(reset, "click", hostnames_reset_click, hostnames);
	liqcell_propseti(reset, "textalign", 2);
	liqcell_propseti(reset, "textaligny", 2);
	liqcell_child_append(buttons, reset);

	liqcell_handleradd(hostnames, "mouse", liqcell_easyhandler_kinetic_mouse);
	
	return self;

}


/** Generate Hostnames for Editing */
int generate_hostnames(liqcell *self)
{
	int retval = 0;
	FILE *hnconf = fopen(HOSTNAME_FILE, "r");
	
	int count = 0;
	
	if (hnconf) 
	{
		char buf[256], hostname[128];

		while (fgets(buf, 256, hnconf)) 
		{
			sscanf(buf, "%128s", hostname);
			count++;
			create_hostbox(self, hostname, count);
		}
		fclose(hnconf);
		
		// set count for use by new
		liqcell_propseti(self, "hostname_count", count);
		
	} 
	else
		retval = -1;
		
	return retval;
}

void create_hostbox(liqcell *hostnames, char *hostname, int count)
{
	int y = (Y_SPACE * count) + LABEL_HEIGHT;
			
	char cell_label[32];
	snprintf(cell_label, 32, "%d.", count);
			
	char cell_name[32];
	snprintf(cell_name, 32, "hostname%d", count);
			
	// create label
	liqcell *hostlabel = liqcell_quickcreatevis(cell_label, "label", 100, y, 200, 36);
	liqcell_setfont(hostlabel, liqfont_cache_getttf("/usr/share/fonts/nokia/nosnb.ttf", (22), 0));
	liqcell_setcaption(hostlabel, cell_label);
	liqcell_propsets(hostlabel, "textcolor", CYAN);
	liqcell_propseti(hostlabel, "textalign", 2);
	liqcell_child_append(hostnames, hostlabel);
			
	// create textbox
	liqcell *hostbox = liqcell_quickcreatevis(cell_name, "textbox", 255, y, 200, 36);
	liqcell_setfont(hostbox, liqfont_cache_getttf("/usr/share/fonts/nokia/nosnb.ttf", (22), 0) );
	liqcell_setcaption(hostbox, hostname);
	liqcell_propsets(hostbox, "textcolor", WHITE);
	liqcell_propsets(hostbox, "backcolor", BLACK);
	liqcell_propsets(hostbox, "bordercolor", CYAN);
	liqcell_propseti(hostbox, "textalign", 0);
	liqcell_propseti(hostbox, "textaligny", 2);
	liqcell_child_append(hostnames, hostbox);
}
