/*
 * Copyright: (C) 2008 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  bruce.forsberg@gmail.com
 *
 */


#include <stdlib.h>
#include <string.h>

#include <gtk/gtkmain.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtktoolbar.h>
#include <gtk/gtk.h>

#include "car_tab.h"
#include "entry_tab.h"
#include "data_tab.h"
#include "care_tab.h"
#include "trip_tab.h"
#include "graph_tab.h"
#include "common_tab.h"
#include "mileagefile.h"
#include "mileagecfg.h"

static int units = 0;
static int cost_units = 0;

static void common_tab_select_car_sub(char *, GtkWidget *, int);
static void common_tab_updateActivityData(char *);
static void common_tab_updateAlertData(char *);
static void common_tab_updateTripData(char *);

/* Loads a car into the Car selection menu. NULL means clear all entries */
void
common_tab_load_car(char * car)
{
   car_tab_load_car(car);
   entry_tab_load_car(car);
   data_tab_load_car(car);
   care_tab_load_car(car);
   trip_tab_load_car(car);
   graph_tab_load_car(car);
}

void
common_tab_select_car(char *car)
{
   GtkWidget  *car_selector;
   int         car_inserted;

   car_selector = car_tab_get_car_selector();
   car_inserted = car_tab_get_car_inserted();
   common_tab_select_car_sub(car, car_selector, car_inserted);

   car_selector = entry_tab_get_car_selector();
   car_inserted = entry_tab_get_car_inserted();
   common_tab_select_car_sub(car, car_selector, car_inserted);

   car_selector = data_tab_get_car_selector();
   car_inserted = data_tab_get_car_inserted();
   common_tab_select_car_sub(car, car_selector, car_inserted);

   car_selector = care_tab_get_car_selector();
   car_inserted = care_tab_get_car_inserted();
   common_tab_select_car_sub(car, car_selector, car_inserted);

   car_selector = trip_tab_get_car_selector();
   car_inserted = trip_tab_get_car_inserted();
   common_tab_select_car_sub(car, car_selector, car_inserted);

   car_selector = graph_tab_get_car_selector();
   car_inserted = graph_tab_get_car_inserted();
   common_tab_select_car_sub(car, car_selector, car_inserted);
}

void
common_tab_select_car_sub(
   char *car,
   GtkWidget *car_selector,
   int        car_inserted)
{
   GtkTreeModel  *model;
   GtkTreeIter    iter;
   gboolean valid;
   gint row_count = 0;

   /* Select first car in list if NULL */
   if (car == NULL)
   {
      if (car_inserted == TRUE)
      {
         gtk_combo_box_set_active(GTK_COMBO_BOX(car_selector), 0);
      }
   }
   else
   {
      model = gtk_combo_box_get_model(GTK_COMBO_BOX(car_selector));

      /* Get the first iter in the list */
      valid = gtk_tree_model_get_iter_first (model, &iter);

      while (valid)
      {
         /* Walk through the list, reading each row */
         gchar *str_data;

         /* Make sure you terminate calls to gtk_tree_model_get()
          * with a '-1' value */
         gtk_tree_model_get (model, &iter, 0, &str_data, -1);

         /* Look for the specified car and make it active */
         if (strcmp(car, str_data) == 0)
         {
            gtk_combo_box_set_active(GTK_COMBO_BOX(car_selector), row_count);
         }
         g_free (str_data);

         row_count ++;
         valid = gtk_tree_model_iter_next (model, &iter);
      }
   }
}

void
common_tab_updateCars()
{
   char *car_array;
   int  num; 
   int  i;
   char **ptr;

   /* Clear all entries */
   common_tab_load_car(NULL);

   num = returnCars(&car_array);
#ifdef DEBUG
   fprintf(stdout, "Number of cars is %d\n", num);
#endif
   for (i = 0; i < num; i++)
   {
      ptr = car_array + (i * sizeof(void *));
#ifdef DEBUG
      fprintf(stdout, "Car found:%s:\n", *ptr);
#endif
      common_tab_load_car(*ptr);
      free(*ptr);
   }
   free(car_array);

   /* Select by default the first car */
   common_tab_select_car(NULL);
}

void
common_tab_updateUnits()
{
   units = returnUnits();
   car_tab_update_units(units);
   cost_units = returnCostUnits();
   car_tab_update_cost_units(cost_units);
}

void
common_tab_update_car_combo_box(int value)
{
   car_tab_update_car_combo_box(value);
   entry_tab_update_car_combo_box(value);
   data_tab_update_car_combo_box(value);
   care_tab_update_car_combo_box(value);
   trip_tab_update_car_combo_box(value);
   graph_tab_update_car_combo_box(value);
}

void
common_tab_load_data(char *car)
{
   data_tab_load_data(car);
   graph_tab_load_data(car);
}

void
common_tab_updateDisplayTab(char *car)
{
   GtkListStore *store;
   int           selector;

   /* Clear the list */
   store = data_tab_get_store();
   if (store)
      gtk_list_store_clear(store);

   selector = data_tab_get_display_selector();
   if (selector == DATA_DISPLAY_NOITEM)
      return;

   if (selector == DATA_DISPLAY_MILEAGE)
   {
      common_tab_updateGasData(TRUE, car);
      common_tab_updateAlertDisplay(car);
   }
   else if (selector == DATA_DISPLAY_ACTIVITY)
   {
      common_tab_updateGasData(FALSE, car);
      common_tab_updateActivityData(car);
      common_tab_updateAlertDisplay(car);
   }
   else if (selector == DATA_DISPLAY_ALERTS)
   {
      common_tab_updateGasData(FALSE, car);
      common_tab_updateAlertData(car);
      common_tab_updateAlertDisplay(car);
   }
   else if (selector == DATA_DISPLAY_TRIPS)
   {
      common_tab_updateGasData(FALSE, car);
      common_tab_updateTripData(car);
   }
}

void
common_tab_updateActivityData(char *car)
{
   GSList  *lst;
   GSList *lst_iter;
   GtkTreeIter   iter;
   char buf[1024];
   char *saveptr;
   char *date;   
   char *odometer;
   char *activity;
   char *comment;
   char *length;
   int  len;
   GtkListStore *store;

   data_tab_update_columns("Date", "Odometer", "Activity", "Comment", "", "", "");

   lst = returnActivityEntries(car);

   /* sort the list in date order */
   lst_iter = g_slist_sort(lst, common_sort_date);

   store = data_tab_get_store();

   while (lst_iter != NULL)
   {
      strncpy(buf, lst_iter->data, 1024);
      date = (char *)strtok_r(buf, " ", &saveptr);
      odometer = (char *)strtok_r(NULL, " ", &saveptr);
      length = (char *)strtok_r(NULL, " ", &saveptr);
      len = atoi(length);
      activity = saveptr;
      saveptr[len] = '\0';
      saveptr += (len + 1);
      length = (char *)strtok_r(NULL, " ", &saveptr);
      len = atoi(length);
      comment = saveptr;

      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter, 0, date, 1, odometer, 2, activity, 3, comment, 4, "", -1);

      free(lst_iter->data);
      lst_iter->data = NULL;
      lst_iter = g_slist_next(lst_iter);
   }

   g_slist_foreach(lst_iter, common_delete_data, NULL);
   g_slist_free(lst_iter);
}

void
common_tab_updateAlertData(char *car)
{
   GSList  *lst;
   GSList *lst_iter;
   GtkTreeIter   iter;
   char buf[1024];
   char *saveptr;
   char *date;
   char *every;
   char *activity;
   char *comment;
   char *length;
   int  len;
   GtkListStore *store;

   data_tab_update_columns("Odometer", "Repeat Every", "Activity", "Comment", "", "", "");

   lst = returnAlertEntries(car);

   /* sort the list in date order */
   lst_iter = g_slist_sort(lst, common_sort_date);

   store = data_tab_get_store();

   while (lst_iter != NULL)
   {
      strncpy(buf, lst_iter->data, 1024);
      date = (char *)strtok_r(buf, " ", &saveptr);
      every = (char *)strtok_r(NULL, " ", &saveptr);
      length = (char *)strtok_r(NULL, " ", &saveptr);
      len = atoi(length);
      activity = saveptr;
      saveptr[len] = '\0';
      saveptr += (len + 1);
      length = (char *)strtok_r(NULL, " ", &saveptr);
      len = atoi(length);
      comment = saveptr;

      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter, 0, date, 1, every, 2, activity, 3, comment, 4, "", -1);

      free(lst_iter->data);
      lst_iter->data = NULL;
      lst_iter = g_slist_next(lst_iter);
   }

   g_slist_foreach(lst_iter, common_delete_data, NULL);
   g_slist_free(lst_iter);
}

void
common_tab_updateTripData(char *car)
{
   GSList  *lst;
   GSList *lst_iter;
   GtkTreeIter   iter;
   char buf[1024];
   char *saveptr;
   char *start_date;
   char *start_odo;
   char *end_date;
   char *end_odo;
   char *person;
   char *dest;
   char *purpose;
   char *length;
   int  len;
   GtkListStore *store;

   data_tab_update_columns("Start Date", "Start Odo", "End Date", "End Odo", "Person",
                            "Destination", "Purpose");

   lst = returnTripEntries(car);

   /* sort the list in date order */
   lst_iter = g_slist_sort(lst, common_sort_date);

   store = data_tab_get_store();

   while (lst_iter != NULL)
   {
      strncpy(buf, lst_iter->data, 1024);
      start_date = (char *)strtok_r(buf, " ", &saveptr);
      start_odo = (char *)strtok_r(NULL, " ", &saveptr);
      end_date = (char *)strtok_r(NULL, " ", &saveptr);
      end_odo = (char *)strtok_r(NULL, " ", &saveptr);
      length = (char *)strtok_r(NULL, " ", &saveptr);
      len = atoi(length);
      person = saveptr;
      saveptr[len] = '\0';
      saveptr += (len + 1);
      length = (char *)strtok_r(NULL, " ", &saveptr);
      len = atoi(length);
      dest = saveptr;
      saveptr[len] = '\0';
      saveptr += (len + 1);
      length = (char *)strtok_r(NULL, " ", &saveptr);
      len = atoi(length);
      purpose = saveptr;

      /* IF these field have empty markers then empty the strings */
      if (strcmp("Empty", end_date) == 0)
         end_date = "";
      if (strcmp("Empty", end_odo) == 0)
         end_odo = "";
      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter, 0, start_date, 1, start_odo, 2, end_date, 3, end_odo, 4, person,
        5, dest, 6, purpose, -1);

      free(lst_iter->data);
      lst_iter->data = NULL;
      lst_iter = g_slist_next(lst_iter);
   }

   g_slist_foreach(lst_iter, common_delete_data, NULL);
   g_slist_free(lst_iter);
}

void
common_tab_updateGasData(int update_tab, char *car)
{
   char     *date_string;
   char     *odo_string;
   char     *fuel_string;
   char     *mpg_string;
   char     *cost_string;
   char     *lifetime_label;
   char     *last_label;
   char     *miles_month_label;
   char     *fuel_label;
   GSList *lst;
   GSList *lst_iter;
   GtkTreeIter   iter;
   char *date;   
   char *odometer;
   char *fuel;
   char *cost;
   char buf[1024];
   char *saveptr;
   char *p;
   int  first_entry = 1;
   int  first_odo;
   int  last_odo;
   int  prev_odo;
   int  big_odo = 0;
   int  total_odo = 0;
   int  first_update = 1;
   char first_date[100];
   char last_date[100];
   char mileage[100];
   char lifeMPG[100];
   char miles_month_str[100];
   double total_fuel = 0.0;
   double gallons = 0.0;
   double last_fuel;
   double mileage_d;
   double cost_total;
   double last_cost;
   GtkListStore *store;
   double   miles_month = 0.0;

   first_date[0] = '\0';
   last_date[0] = '\0';
   mileage[0] = '\0';
   lifeMPG[0] = '\0';

   if (update_tab)
   {
      date_string = "Date";
      odo_string = "Odometer";
      cost_string = "Cost";

      /* Set correct labels for gas data */
      switch (units)
      {
         /* MPG */
         case 0:
            lifetime_label = "Lifetime MPG:";
            last_label = "Last MPG:";
            miles_month_label = "Miles / Month:";
            fuel_label = "Enter Gallons";
            fuel_string = "Gallons";
            mpg_string = "MPG";
         break;

         /* L/100km */
         case 1:
            lifetime_label = "Lifetime L/100km:";
            last_label = "Last L/100km:";
            miles_month_label = "Kilometers / Month:";
            fuel_label = "Enter Liters";
            fuel_string = "Liters";
            mpg_string = "L/100km";
         break;

         /* L/10km */
         case 2:
            lifetime_label = "Lifetime L/10km:";
            last_label = "Last L/10km:";
            miles_month_label = "Kilometers / Month:";
            fuel_label = "Enter Liters";
            fuel_string = "Liters";
            mpg_string = "L/10km";
         break;

         /* Miles / L */
         case 3:
            lifetime_label = "Lifetime Miles/L:";
            last_label = "Last Miles/L:";
            miles_month_label = "Miles / Month:";
            fuel_label = "Enter Liters";
            fuel_string = "Liters";
            mpg_string = "M/L";
         break;

         /* km / L */
         case 4:
            lifetime_label = "Lifetime km/L:";
            last_label = "Last km/L:";
            miles_month_label = "Kilometers / Month:";
            fuel_label = "Enter Liters";
            fuel_string = "Liters";
            mpg_string = "km/L";
         break;
      }
      data_tab_update_columns(date_string, odo_string, fuel_string, mpg_string, cost_string, "", "");
      car_tab_update_text_labels(lifetime_label, last_label, miles_month_label);
      entry_tab_update_text_label(fuel_label);
   }

   if (car)
   {
      lst = returnMileEntries(car);

      /* sort the list in date order */
      lst_iter = g_slist_sort(lst, common_sort_date);

      store = data_tab_get_store();

      while (lst_iter != NULL)
      {
         strncpy(buf, lst_iter->data, 1024);
         date = (char *)strtok_r(buf, " ", &saveptr);
         odometer = (char *)strtok_r(NULL, " ", &saveptr);
         fuel = (char *)strtok_r(NULL, " ", &saveptr);
         cost = (char *)strtok_r(NULL, " ", &saveptr);
         p = (char *)strtok_r(NULL, " ", &saveptr);
         /* IF no cost data */
         if (p == NULL)
         {
            p = cost;
            cost = "";
         }

         if (first_entry == 1)
         {
            first_entry = 0;
            first_odo = atoi(odometer);
            strncpy(first_date, date, sizeof(first_date));
         }
         last_odo = atoi(odometer);
         last_fuel = atof(fuel);
         last_cost = atof(cost);
         strncpy(last_date, date, sizeof(last_date));

         mileage[0] = '\0';
         if (first_update == 0)
         {
            /* Save latgest odo */
            if (last_odo > big_odo);
               big_odo = last_odo;

            if (strcmp(p, "0") == 0)
            {
               total_odo += (last_odo - prev_odo);
               total_fuel += last_fuel;

               /* MPG */
               if (units == 0)
                  mileage_d = (double)(last_odo - prev_odo) / last_fuel;
               /* L/100km */
               else if (units == 1)
                  mileage_d = 100.0 * last_fuel / (double)(last_odo - prev_odo);
               /* L/10km */
               else if (units == 2)
                  mileage_d = 10.0 * last_fuel / (double)(last_odo - prev_odo);
               /* Miles / L */
               else if (units == 3)
                  mileage_d = (double)(last_odo - prev_odo) / last_fuel;
               /* km / L */
               else if (units == 4)
                  mileage_d = (double)(last_odo - prev_odo) / last_fuel;

               sprintf(mileage, "%6.2f", mileage_d);
            }
         }

         if (update_tab)
         {
            gtk_list_store_append(store, &iter);
            gtk_list_store_set(store, &iter, 0, date, 1, odometer, 2, fuel, 3, mileage, 4, cost, -1);
         }
         free(lst_iter->data);
         lst_iter->data = NULL;
         lst_iter = g_slist_next(lst_iter);

         first_update = 0;
         prev_odo = last_odo;
         if (cost_units == 0)
            cost_total += last_cost;
         else
            cost_total += (last_cost * last_fuel);

      }
      g_slist_foreach(lst_iter, common_delete_data, NULL);
      g_slist_free(lst_iter);

      /* If there is data to calculate lifetime MPG */
      if (total_fuel > 0.0)
      {
         /* MPG */
         if (units == 0)
         {
            total_fuel = (double)total_odo / total_fuel;
            gallons = (double)big_odo / total_fuel;
         }
         /* L/100km */
         else if (units == 1)
         {
            total_fuel = 100.0 * total_fuel / (double)total_odo;
            gallons = total_fuel / 100.0 * big_odo;
         }
         /* L/10km */
         else if (units == 2)
         {
            total_fuel = 10.0 * total_fuel / (double)total_odo;
            gallons = total_fuel / 10.0 * big_odo;
         }
         /* Miles / L */
         else if (units == 3)
         {
            total_fuel = (double)total_odo / total_fuel;
            gallons = (double)big_odo / total_fuel;
         }
         /* km / L */
         else if (units == 4)
         {
            total_fuel = (double)total_odo / total_fuel;
            gallons = (double)big_odo / total_fuel;
         }

         sprintf(lifeMPG, "%6.2f", total_fuel);
      }

      miles_month_str[0] = '\0';;
      if (strcmp(first_date, last_date) != 0)
      {
         GDate *date1 = g_date_new();
         GDate *date2 = g_date_new();
         g_date_set_parse(date1, first_date);
         g_date_set_parse(date2, last_date);
         if (g_date_valid(date1) == TRUE && g_date_valid(date2) == TRUE)
         {
            miles_month = (last_odo - first_odo) * 30.0 / g_date_days_between(date1, date2);
            sprintf(miles_month_str, "%7.1f", miles_month);
         }
         g_date_free(date1);
         g_date_free(date2);
      }

      car_tab_set_life_MPG(lifeMPG);
      car_tab_set_miles_month(miles_month_str);
      car_tab_set_last_MPG(mileage);
   }
}

gint
common_sort_date(
   gconstpointer a,
   gconstpointer b)
{
   char  datea[1024];
   char  dateb[1024];
   GDate gdate_a;
   GDate gdate_b;
   gint  ret_value;

   sscanf(a, "%s", datea);
   sscanf(b, "%s", dateb);

   g_date_clear(&gdate_a, 1);
   g_date_clear(&gdate_b, 1);

   g_date_set_parse(&gdate_a, datea);
   g_date_set_parse(&gdate_b, dateb);

   ret_value = g_date_compare(&gdate_a, &gdate_b);

   return (ret_value);
}

void
common_delete_data(gpointer a, gpointer b)
{
   free(a);
}

int
common_valid_date(char *orig_date_str)
{
   int ret_value = TRUE;
   char *date_str;
   GDate  *date = NULL;
   char  *year_str = NULL;
   char  *mon_str = NULL;
   char  *day_str = NULL;

   date = g_date_new();
   if (date && orig_date_str)
   {
      date_str = (char *)strdup(orig_date_str);
      if (date_str)
      {
         year_str = strtok(date_str, "/");
         mon_str = strtok(NULL, "/");
         day_str = strtok(NULL, "/");
         if (day_str && mon_str && year_str)
         {
            g_date_set_day(date, atoi(day_str));
            g_date_set_month(date, atoi(mon_str));
            g_date_set_year(date, atoi(year_str));
         }
         free(date_str);
         date_str = NULL;
      }
   }
   if ((date == NULL) || (g_date_valid(date) == FALSE))
   {
      ret_value = FALSE;
   }
   if (date)
      g_date_free(date);
   return ret_value;
}

void
common_tab_updateAlertDisplay(char *car)
{
   GSList *lst;
   GSList *lst_iter;
   GtkWidget  *text_buf;
   GtkTextIter  start;
   GtkTextIter  end;
   char  *odo;
   char  *activity;
   char  *comment;
   char  *saveptr;
   char  *length;
   int   len;
   char  buf[1024];


   text_buf = car_tab_get_text_buffer();
   gtk_text_buffer_get_start_iter(GTK_TEXT_BUFFER(text_buf), &start);
   gtk_text_buffer_get_end_iter(GTK_TEXT_BUFFER(text_buf), &end);
   gtk_text_buffer_delete(GTK_TEXT_BUFFER(text_buf), &start, &end);

   lst = getNextAlert(car, 1000);
   if (g_slist_length(lst) == 0)
   {
      gtk_text_buffer_insert_at_cursor(GTK_TEXT_BUFFER(text_buf), "No Current Alerts for this Car", 30);
   }
   else
   {
      gtk_text_buffer_insert_at_cursor(GTK_TEXT_BUFFER(text_buf), "Alerts for this car are:\n", 25);
      lst_iter = lst;
      while (lst_iter != NULL)
      {
         odo = strtok_r(lst_iter->data, " ", &saveptr);
         length = (char *)strtok_r(NULL, " ", &saveptr);
         len = atoi(length);
         activity = saveptr;
         saveptr[len] = '\0';
         saveptr += (len + 1);
         length = (char *)strtok_r(NULL, " ", &saveptr);
         len = atoi(length);
         comment = saveptr;
         saveptr[len] = '\0';

         sprintf(buf, "%s due at %s: %s\n", activity, odo, comment);
         gtk_text_buffer_insert_at_cursor(GTK_TEXT_BUFFER(text_buf), buf, strlen(buf));

         free(lst_iter->data);
         lst_iter->data = NULL;
         lst_iter = g_slist_next(lst_iter);
      }
   }
}

