#/usr/bin/env python2.5

"""
mephemeris.py
version 1.25

This is a simple PyGTK GUI application intended to display astronomical 
ephemeris information.  The application relies heavily upon a ported
version of the 'PyEphem' library created by Brandon Craig Rhodes.
 
More information about PyEphem is available at:

http://rhodesmill.org/pyephem/

More information about what an ephemeris is all about is available at:

http://en.wikipedia.org/wiki/Ephemeris

More information about Python and PyGTK is available at:

http://www.python.org/
http://www.pygtk.org/

Copyright (C) 2008 Darren Enns <dmenns1@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
USA.
"""

import urllib2
import xml.dom.minidom

import gtk
import calendar
import ConfigParser
import os
import sys
import socket
import pango
import urllib
from math import sin, cos, tan, asin, acos, atan, pi, sqrt, atan2
import math
import time
import string
import cairo
from socket import inet_aton
import re

try:
	import ephem
	ephem_flag = 1
	ephem_version = string.split(string.split(ephem.__version__)[0], ".")
	if (float(ephem_version[0]) >= 3 and float(ephem_version[1] >= 7) and float(ephem_version[2]) >= 3 and float(ephem_version[3]) >= 2):
		print "pyephem version OK", ephem.__version__
	else:
		ephem_flag = -1
		print "pyephem version too old"
except ImportError:
	ephem_flag = -1
	print "pyephem not present"

try:
        import hildon
        hildon_flag = 1
except ImportError:
        hildon_flag = -1
        print "not on maemo platform"

try:
        import gpsbt
        gpsbt_flag = 1
except ImportError:
        gpsbt_flag = -1
        print "maemo bluetooth gps not present"

try:
	import gps
	gpsd_flag = 1
except ImportError:
	gpsd_flag = -1
	print "gpsd not present"

NAME = 'MEphemeris'
VERSION = '1.25'
AUTHORS = ['Darren Enns <dmenns1@gmail.com>\n\n' + 'Amos 5:8 (New Living Translation):\n\nIt is the Lord who created the stars,\n   the Pleiades and Orion.\nHe turns darkness into morning\n   and day into night.\nHe draws up water from the oceans\n   and pours it down as rain on the land.\nThe Lord is his name!']

timeout = 10

socket.setdefaulttimeout(timeout)

magnitude_limit = 3.0
window_in_fullscreen = False
date_in_gmt = True
star_show_flag = True
star_name_show_flag = False
planet_show_flag = True
planet_name_show_flag = False
messier_show_flag = False
messier_name_show_flag = False
ngc_show_flag = False
ngc_name_show_flag = False
inverse_show_flag = False
constellation_name_show_flag = False
grid_show_flag = False
legend_show_flag = False
constellation_show_flag = False
controls_show_flag = True
on_off_controls_show_flag = True
date_time_controls_show_flag = True
object_size = -4
object_bright = 0
polar_sky_flag = True
azimuth_center = 180
altitude_center = 0
polar_field_of_view = 90.0
equi_field_of_view = 360.0
moon_field_of_view = 1.0
grid_space = 15
horizon_grid_flag = True
rise_set_plot_flag = True
elevation_show_flag = False
font_size = 10
x_shift_degrees = 0
y_shift_degrees = 0
google_view = 0.05
google_map_type = 'roadmap'
ecliptic_show_flag = False
galactic_show_flag = False
horizon_proj_flag = True
horizon_show_flag = False
satellite_show_flag = False
satellite_name_show_flag = False
verbose_show_flag = True
sun_compass_flag = False
milkyway_show_flag = False
planet_field_of_view = 1.0
asteroid_show_flag = False
asteroid_name_show_flag = False
asteroid_read_flag = False
comet_show_flag = False
comet_name_show_flag = False
comet_read_flag = False
neo_show_flag = False
neo_name_show_flag = False
neo_read_flag = False
object_search_flag = False
planet_search_flag = False
star_search_flag = False
messier_search_flag = False
ngc_search_flag = False
satellite_search_flag = False
asteroid_search_flag = False
comet_search_flag = False
neo_search_flag = False
constellation_search_flag = False
#navigate_keys_flag = True
moon_feature_show_flag = False
moon_feature_name_show_flag = False
moon_trail_show_flag = False
planet_trail_show_flag = False
object_show_name_flag = True
daylight_show_flag = False
chart_info_toggle_flag = False

moon_zoom = 2
moon_x = 0
moon_y = 0

rotate_angle = 0
rotate_angle_x = 0
rotate_angle_y = 0
rotate_axis = (0,0,0)

select_object = ""

search_az_ra = 180 * pi/180
search_alt_dec = 90 * pi/180

daylight_plot_color = [0.015625,0.515625,0.984375]
sun_plot_color = [1,1,0]
moon_plot_color = [0.7,0.7,0.7]
mercury_plot_color = [0.4,0.4,0.4]
venus_plot_color = [0.6,0.6,0]
earth_plot_color = [0.2,0.2,1.0]
mars_plot_color = [1,0,0] 
jupiter_plot_color = [1,0.6,0]
saturn_plot_color = [0.8,0.8,0]
uranus_plot_color = [0,0.6,0.6]
neptune_plot_color = [0,0.6,0.6]
pluto_plot_color = [0.5,0.5,0.5]

star_plot_color = [1.0,0.5,1]
messier_plot_color = [0.5,0.5,1]
ngc_plot_color = [0.7,0.7,0.2]
satellite_plot_color = [1,0.5,0.5]
moon_feature_plot_color = [1.0,0.2,1]

asteroid_plot_color = [0.2,0.2,1.0]
comet_plot_color = [0.6,0.3,0.0]
neo_plot_color = [1.0,0.2,0.2]

if hildon_flag == 1:
	grid_inc = 3
else:
	grid_inc = 1

if ephem_flag == -1:
	raise ImportError("PyEPhem version 3.7.3.2 or newer is required; you have %s" % ephem.__version__)

def show_object_info_basic_start(widget,window,button,local):

	button.destroy()

	show_object_info_basic("clicked",window,local)

	return

def show_about(widget):
	dialog = gtk.AboutDialog()
	dialog.set_name('MEphemeris')
	dialog.set_logo_icon_name('system')
	dialog.set_comments('Simple Astronomical Ephemeris')
	dialog.set_copyright('(C) 2009 Darren Enns')
	dialog.set_authors(AUTHORS)
	dialog.set_version(VERSION)
	dialog.set_website('http://darethehair.googlepages.com/software.html')
	dialog.connect('response', lambda d, r: d.destroy())
	dialog.show()

def show_help(widget):
	show_text_file('doc/README')

def show_license(widget):
	show_text_file('doc/COPYING')

def show_text_file(file_name):
	global font_size

	# Create a new window
	if hildon_flag == 1:
		window = hildon.Window()
	else:
		window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title("Viewing " + file_name)
	window.set_size_request(800, 480)

	box1 = gtk.VBox(False, 0)
	window.add(box1)
	box1.show()

	box2 = gtk.VBox(False, 10)
	box2.set_border_width(10)
	box1.pack_start(box2, True, True, 0)
	box2.show()

	sw = gtk.ScrolledWindow()
	sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
	textview = gtk.TextView()
	textview.modify_font(pango.FontDescription('Courier bold ' + str(font_size)))

	textbuffer = textview.get_buffer()
	sw.add(textview)
	sw.show()
	textview.show()
	box2.pack_start(sw)

	# Load the file into the text window
	path = os.path.dirname(os.path.abspath(sys.argv[0]))
	file_name = os.path.join(path, file_name)
	infile = open(file_name, "r")

	if infile:
		string = infile.read()
		infile.close()
		textbuffer.set_text(string)

	hbox = gtk.HButtonBox()
	box2.pack_start(hbox, False, False, 0)
	hbox.show()

	if window_in_fullscreen == True:
		window.fullscreen()

	window.show_all()

def quit_cb(widget):
	gtk.main_quit()

def show_object_info_basic(widget,window,local):
	global chart_info_toggle_flag
	global ini_treeview
	global observer

	window.set_title('MEphemeris ' + str(VERSION) + ': MAIN WINDOW')

	observer = local

	save = local.date 

        if hildon_flag == True:
                menu_element = 'popup'
        else:
                menu_element = 'menubar'

        ui = '''<ui>
        <%(menu_element)s name="MenuBar">
        <menu action="File">
        <menuitem action="Quit"/>
        </menu>
        <menu action="Edit">
        <menuitem action="Edit Ini File"/>
        <menuitem action="Save Ini File"/>
        <menuitem action="Edit Messier File"/>
        <menuitem action="Save Messier File"/>
        <menuitem action="Edit Misc File"/>
        <menuitem action="Save Misc File"/>
        </menu>
        <menu action="Help">
        <menuitem action="Readme"/>
        <menuitem action="License"/>
        <menuitem action="About"/>
        </menu>
        </%(menu_element)s>
        </ui>''' % { 'menu_element': menu_element }

	# make ui layout
	vbox = gtk.VBox()

	# Create a UIManager instance
	uimanager = gtk.UIManager()

	# Add the accelerator group to the toplevel window
	accelgroup = uimanager.get_accel_group()
	window.add_accel_group(accelgroup)

	# Create an ActionGroup
	actiongroup = gtk.ActionGroup('UIManagerExample')

	# Create actions
	actiongroup.add_actions([('Quit', gtk.STOCK_QUIT, '_Quit!', None, 'Quit the Program', quit_cb),
				('File', None, '_File'),
				('Edit', None, '_Edit'),
			 	('Help', None, '_Help')])

	actiongroup.add_actions([('Edit Ini File', None, 'Edit Ini File', None, None, edit_ini),
				('Save Ini File', None, 'Save Ini File', None, None, save_ini),
				('Edit Messier File', None, 'Edit Messier Image URL File', None, None, edit_messier_url),
				('Save Messier File', None, 'Save Messier Image URL File', None, None, save_messier_url),
				('Edit Misc File', None, 'Edit Misc Image URL File', None, None, edit_misc_url),
				('Save Misc File', None, 'Save Misc Image URL File', None, None, save_misc_url)])

	actiongroup.get_action('Quit').set_property('short-label', '_Quit')

	actiongroup.add_actions([('Readme', gtk.STOCK_HELP, '_Readme', None, None, show_help),
				('License', gtk.STOCK_HELP, '_License', None, None, show_license),
				('About', gtk.STOCK_ABOUT, '_About', None, None, show_about)])

	# Add the actiongroup to the uimanager
	uimanager.insert_action_group(actiongroup, 0)

	# Add a UI description
	uimanager.add_ui_from_string(ui)

	# Create a MenuBar
	menubar = uimanager.get_widget('/MenuBar')
	if hildon_flag == True:
		window.set_menu(menubar)
	else:
		vbox.pack_start(menubar, False)

	window.add(vbox)

	if hildon_flag == True:
		settings = window.get_settings()
		settings.set_property("gtk-button-images", False)
		settings.set_property("gtk-menu-images", False)

	current_local_date = ephem.Date(ephem.localtime(local.date))
	local_time_zone_name = time.tzname

	if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
		local_time_zone_daylight_flag = 0
	else:
		local_time_zone_daylight_flag = time.daylight

	color_black = gtk.gdk.color_parse('black')

	table = gtk.Table(11,8,False)
	table.set_col_spacings(1)
	table.set_row_spacings(1)

	table2 = gtk.Table(1,9,False)
	table2.set_col_spacings(1)

	table3 = gtk.Table(1,7,False)
	table3.set_col_spacings(1)

	row = 0
	column = 0

	y = local.date.tuple()

	if date_in_gmt == True:
		title = 'DATE/TIME: '+str(local.date)+' GMT LOCATION: '+str(local.lat)+u'\xb0 '+str(local.long)+u'\xb0'
	else:
		title = 'DATE/TIME: '+str(current_local_date)+' '+local_time_zone_name[local_time_zone_daylight_flag]+' LOCATION: '+str(local.lat)+u'\xb0 '+str(local.long)+u'\xb0'

	button = gtk.Button(title)
	button.connect("clicked",toggle_date_style,window,button,local)
	table.attach(button,column,column+13,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","orange","green","red")

	row = row+1
	column = 0
	date_time_change_flag = -5
	button = gtk.Button("-1 Yr")
	button.connect("clicked",change_date_time_basic,window,table,vbox,local,date_time_change_flag)
	table2.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = -4
	button = gtk.Button("-1 Mo")
	button.connect("clicked",change_date_time_basic,window,table,vbox,local,date_time_change_flag)
	table2.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = -3
	button = gtk.Button("-1 Dy")
	button.connect("clicked",change_date_time_basic,window,table,vbox,local,date_time_change_flag)
	table2.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = -2
	button = gtk.Button("-1 Hr")
	button.connect("clicked",change_date_time_basic,window,table,vbox,local,date_time_change_flag)
	table2.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = -1
	button = gtk.Button("-1 Mi")
	button.connect("clicked",change_date_time_basic,window,table,vbox,local,date_time_change_flag)
	table2.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = 0
	button = gtk.Button('Reset Date/Time')
	button.connect("clicked",change_date_time_basic,window,table,vbox,local,date_time_change_flag)
	table2.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = 1
	button = gtk.Button("+1 Mi")
	button.connect("clicked",change_date_time_basic,window,table,vbox,local,date_time_change_flag)
	table2.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = 2
	button = gtk.Button("+1 Hr")
	button.connect("clicked",change_date_time_basic,window,table,vbox,local,date_time_change_flag)
	table2.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = 3
	button = gtk.Button("+1 Dy")
	button.connect("clicked",change_date_time_basic,window,table,vbox,local,date_time_change_flag)
	table2.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = 4
	button = gtk.Button("+1 Mo")
	button.connect("clicked",change_date_time_basic,window,table,vbox,local,date_time_change_flag)
	table2.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = 5
	button = gtk.Button("+1 Yr")
	button.connect("clicked",change_date_time_basic,window,table,vbox,local,date_time_change_flag)
	table2.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	table.attach(table2,0,column+13,row,row+1)

	row = row+1
	column = 0
	button = gtk.Button("Set Date")
	button.connect("clicked",set_date,window,button,table,vbox,local,save)
	table3.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	button = gtk.Button("Set Time")
	button.connect("clicked",set_time,window,button,table,vbox,local,save)
	table3.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	button = gtk.Button('Set Location')
	button.connect("clicked",set_location,window,table,vbox,local)
	table3.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	button = gtk.Button('Reset Location')
	button.connect("clicked",reset_location,window,table,vbox,local)
	table3.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	button = gtk.Button("Season")
	button.connect("clicked",show_season_info,local,save)
	table3.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","light blue","green","red")

	column = column+1
	button = gtk.Button("Time")
	button.connect("clicked",show_time_info,local)
	table3.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","light blue","green","red")

	column = column+1
	button = gtk.Button("Compass")
	button.connect("clicked",plot_compass_start,local)
	table3.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","light blue","green","red")

	column = column+1
	button=gtk.Button("Web")
	button.connect("clicked",select_web_image,local)
	table3.attach(button,column,column+1,row,row+1)
	button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

	column = column+1
	button=gtk.Button("Sky")
	button.connect("clicked",plot_sky_start,local)
	table3.attach(button,column,column+1,row,row+1)
	button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

	table.attach(table3,0,column+13,row,row+1)

	row = row+1
	column = 0

	button=gtk.Button("Name")
	button.connect("clicked",toggle_chart_info,window,table,vbox,local)
	table.attach(button,column,column+1,row,row+1)
	button.show()
	#set_button_color("clicked",button,"black","black","black","yellow","green","red")
	set_button_color("clicked",button,"black","black","black","light blue","green","red")

	if chart_info_toggle_flag == False:
		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","RA")

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Dec")

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Magnitude")

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Size")

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Sun Dist")

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Earth Dist")

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Phase")
	else:
		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Azimuth")

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Altitude")

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Rise Date/Time")

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Rise Azimuth")

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Set Date/Time")

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Set Azimuth")

	row = row+1
	column = 0
	sun = ephem.Sun()
	chart_info(sun,local,table,row,column)

	row = row+1
	column = 0
	moon = ephem.Moon()
	chart_info(moon,local,table,row,column)

	row = row+1
	column = 0
	mercury = ephem.Mercury()
	chart_info(mercury,local,table,row,column)

	row = row+1
	column = 0
	venus = ephem.Venus()
	chart_info(venus,local,table,row,column)

	row = row+1
	column = 0
	mars = ephem.Mars()
	chart_info(mars,local,table,row,column)

	row = row+1
	column = 0
	jupiter = ephem.Jupiter()
	chart_info(jupiter,local,table,row,column)

	row = row+1
	column = 0
	saturn = ephem.Saturn()
	chart_info(saturn,local,table,row,column)

	row = row+1
	column = 0
	uranus = ephem.Uranus()
	chart_info(uranus,local,table,row,column)

	row = row+1
	column = 0
	neptune = ephem.Neptune()
	chart_info(neptune,local,table,row,column)

	row = row+1
	column = 0
	pluto = ephem.Pluto()
	chart_info(pluto,local,table,row,column)

	local.date = save #still required for 3.7.3.2
	
	table.show()
	table2.show()
	table3.show()

	vbox.pack_start(table)

	if window_in_fullscreen == True:
		window.fullscreen()

	window.show_all()

	return

def select_web_image(widget,local):
	global map_flag
	if hildon_flag == 1:
		window = hildon.Window()
	else:
		window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title('Planet Ephemeris: WEB IMAGES/DATA')

	window.set_size_request(800,480)

        table = gtk.Table(11,14,False)
	table.set_col_spacings(1)
	table.set_row_spacings(1)

	row = 0
        column = 0

	event_box = gtk.EventBox()
	table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Topo Images")

        column = column+1
        map_flag = 3
        button=gtk.Button("Topo Map")
        button.connect("clicked",get_web_image,window,local,map_flag)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
        map_flag = 1
        button=gtk.Button("Topo Globe")
        button.connect("clicked",get_web_image,window,local,map_flag)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
        map_flag = 10
        button=gtk.Button("Google Map")
        button.connect("clicked",get_google_map_start,local,local)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
	event_box = gtk.EventBox()
	table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","")

	row = row + 1
        column = 0
	event_box = gtk.EventBox()
	table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Astro Data")

        column = column+1
	elevation_show_flag = True
        button=gtk.Button("NGC Data")
        button.connect("clicked",get_ngc_data,window)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
	elevation_show_flag = True
        button=gtk.Button("Asteroid Data")
        button.connect("clicked",get_asteroid_data,window)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
	elevation_show_flag = True
        button=gtk.Button("Comet Data")
        button.connect("clicked",get_comet_data,window)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
	elevation_show_flag = True
        button=gtk.Button("NEO Data")
        button.connect("clicked",get_neo_data,window)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

	row = row + 1
        column = 0
	event_box = gtk.EventBox()
	table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Topo Data")

        column = column+1
	elevation_show_flag = True
        button=gtk.Button("GeoNames Elevation")
        button.connect("clicked",get_geonames_elevation,local,elevation_show_flag)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
	event_box = gtk.EventBox()
	table.attach(event_box,column,column+3,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","")

	row = row + 1
        column = 0
	event_box = gtk.EventBox()
	table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Astro Images")

        column = column+1
        button=gtk.Button("APOD")
        button.connect("clicked",get_apod_image,window,local,map_flag)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
        map_flag = 9
        button=gtk.Button("Sun Spots")
        button.connect("clicked",get_web_image,window,local,map_flag)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
        button=gtk.Button("Moon Map")
        button.connect("clicked",get_moon_image_start)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
	event_box = gtk.EventBox()
	table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","")

	row = row + 1
        column = 0
	event_box = gtk.EventBox()
	table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Astro Plots")

        column = column+1
        map_flag = 5
        button=gtk.Button("Inner Planets")
        button.connect("clicked",get_web_image,window,local,map_flag)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
        map_flag = 6 
        button=gtk.Button("Outer Planets")
        button.connect("clicked",get_web_image,window,local,map_flag)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
        map_flag = 7 
        button=gtk.Button("Polar Sky")
        button.connect("clicked",get_web_image,window,local,map_flag)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
        map_flag = 8
        button=gtk.Button("Moon Phase")
        button.connect("clicked",get_web_image,window,local,map_flag)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

	row = row + 1
        column = 0
	event_box = gtk.EventBox()
	table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Satellites")

        column = column+1
        map_flag = 10
        button=gtk.Button("Satellite Position")
        button.connect("clicked",get_web_image,window,local,map_flag)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
        button=gtk.Button("Satellite Elements")
        button.connect("clicked",get_tle_elements,window)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
	event_box = gtk.EventBox()
	table.attach(event_box,column,column+2,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","")

	row = row + 1
        column = 0
	event_box = gtk.EventBox()
	table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Weather Feeds")

        column = column+1
        xml_flag = 1
        button=gtk.Button("GeoNames Weather")
        button.connect("clicked",show_local_weather,local,xml_flag)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
        xml_flag = 2
        button=gtk.Button("Yahoo Weather")
        button.connect("clicked",show_local_weather,local,xml_flag)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
        xml_flag = 3 
        button=gtk.Button("Wunderground Weather")
        button.connect("clicked",show_local_weather,local,xml_flag)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
	event_box = gtk.EventBox()
	table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","")

	row = row + 1
        column = 0
	event_box = gtk.EventBox()
	table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","User Images")

        column = column+1
        button=gtk.Button("Messier")
        button.connect("clicked",select_messier_image)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

	column = column+1
	button = gtk.Button('NGC')
	button.connect("clicked",select_ngc_image,window)
	table.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
	elevation_show_flag = True
        button=gtk.Button("Miscellaneous")
        button.connect("clicked",select_misc_image)
        table.attach(button,column,column+1,row,row+1)
        button.show()
	set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

        column = column+1
	event_box = gtk.EventBox()
	table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","")

        table.show()
        window.add(table)

	if window_in_fullscreen == True:
		window.fullscreen()

        window.show()

	return

def display_image(image,pixbuf,aspect,enlarge,screen_width,screen_height,window_width,window_height):
	if window_in_fullscreen == True:
		display_width = screen_width
		display_height = screen_height-20
	else:
		display_width = window_width
		display_height = window_height-20

	image_width, image_height = resize_image_to_fit(
		(pixbuf.get_width(), pixbuf.get_height()),
		(display_width,display_height),
		aspect,
		enlarge)

	scaled_buf = pixbuf.scale_simple(image_width,image_height,gtk.gdk.INTERP_BILINEAR)

	image.set_from_pixbuf(scaled_buf)

	return

def resize_image_to_fit(image, frame, aspect=True, enlarge=False):
    	"""Resizes a rectangle to fit within another.

    	Parameters:
    	image -- A tuple of the original dimensions (width, height).
    	frame -- A tuple of the target dimensions (width, height).
    	aspect -- Maintain aspect ratio?
    	enlarge -- Allow image to be scaled up?
    	"""
    	if aspect:
        	return scale_image_to_fit(image, frame, enlarge)
    	else:
        	return stretch_image_to_fit(image, frame, enlarge)

def scale_image_to_fit(image, frame, enlarge=False):
    	image_width, image_height = image
    	frame_width, frame_height = frame
    	image_aspect = float(image_width) / image_height
    	frame_aspect = float(frame_width) / frame_height

    	# Determine maximum width/height (prevent up-scaling).
    	if not enlarge:
        	max_width = min(frame_width, image_width)
        	max_height = min(frame_height, image_height)
    	else:
        	max_width = frame_width
        	max_height = frame_height

    	# Frame is wider than image.
    	if frame_aspect > image_aspect:
        	height = max_height
        	width = int(height * image_aspect)

    	# Frame is taller than image.
    	else:
        	width = max_width
        	height = int(width / image_aspect)
    	return (width, height)

def stretch_image_to_fit(image, frame, enlarge=False):
    	image_width, image_height = image
    	frame_width, frame_height = frame

    	# Stop image from being blown up.
    	if not enlarge:
        	width = min(frame_width, image_width)
        	height = min(frame_height, image_height)
    	else:
        	width = frame_width
        	height = frame_height
    	return (width, height)

def select_misc_image(widget):
	if hildon_flag == 1:
		window = hildon.Window()
	else:
		window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title('Planet Ephemeris: MISCELLANEOUS WEB IMAGES')

	window.set_size_request(800,480)

        table = gtk.Table(10,10,False)
	table.set_col_spacings(1)
	table.set_row_spacings(1)

	row = 0
	column = 0

	image_num = 1

	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
	data_dir = os.path.join(run_dir, 'data')
	misc_file = os.path.join(data_dir, 'misc_url.dat')
	f = open(misc_file,"rb")
	while 1:
		t = f.readline()
		if t == '':
			break
		if '\n' in t:
			t = t[:-1]
		if '\r' in t:
			t = t[:-1]
		a = t.split()

		image_name = a[0]
		image_url = a[1]

		button=gtk.Button(image_name)
		button.connect("clicked",get_misc_image,image_num,image_url)
		table.attach(button,column,column+1,row,row+1)
		button.show()
		set_button_color("clicked",button,"black","black","black","dodger blue","green","red")

		image_num = image_num + 1
		column = column + 1
		if column > 4:
			column = 0
			row = row + 1

	f.close()

	if column != 0:
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+(5-column),row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","")
	
        table.show()
        window.add(table)

	if window_in_fullscreen == True:
		window.fullscreen()

        window.show()

	return

def get_misc_image(widget,image_num,url):
        if hildon_flag == 1:
                window = hildon.Window()
        else:
                window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title('Planet Ephemeris: WEB MISCELLANEOUS IMAGES')

	window.set_size_request(800,480)

	screen = window.get_screen()
	screen_width = screen.get_width()
	screen_height = screen.get_height()
	window_width = window.get_size()[0]
	window_height = window.get_size()[1]

        color_black = gtk.gdk.color_parse('black')
	window.modify_bg(gtk.STATE_NORMAL, color_black)

	try:
		file = urllib.urlopen(url)
		print "url=",url
		misc_pic = file.read()
		run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
		data_dir = os.path.join(run_dir, 'data')
		image_file = os.path.join(data_dir, 'misc.gif')
		temp_file = open(image_file,"wb")
		temp_file.write(misc_pic)
		temp_file.close()
		print "image_file=",image_file
	except Exception:
		print "web connection timeout"

	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
	data_dir = os.path.join(run_dir, 'data')
	image_file = os.path.join(data_dir, 'misc.gif')
	image = gtk.Image()
	image.set_from_file(image_file)

	pixbuf = gtk.gdk.pixbuf_new_from_file(image_file)

	aspect = True
	enlarge = True

	display_image(image,pixbuf,aspect,enlarge,screen_width,screen_height,window_width,window_height)

	table = gtk.Table(1,1, False)
	window.add(table)
	table.attach(image,0,1,0,1)

	event_box = gtk.EventBox()
	table.attach(event_box,0,1,1,2)
	set_event_box_color("clicked",event_box,"black","yellow",url)
	table.show()

	image.show()

	if window_in_fullscreen == True:
		window.fullscreen()

	window.show()

	return

def select_messier_image(widget):
	if hildon_flag == 1:
		window = hildon.Window()
	else:
		window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title('Planet Ephemeris: MESSIER WEB IMAGES')

	window.set_size_request(800,480)

        table = gtk.Table(10,10,False)
	table.set_col_spacings(1)
	table.set_row_spacings(1)

	messier_id = 1

	for row in range(0,11):
		for column in range(0,10):
			button=gtk.Button("M"+str(messier_id))
			button.connect("clicked",get_messier_image,window,messier_id)
			table.attach(button,column,column+1,row,row+1)
			button.show()
			set_button_color("clicked",button,"black","black","black","dodger blue","green","red")
			messier_id = messier_id + 1

        table.show()
        window.add(table)

	if window_in_fullscreen == True:
		window.fullscreen()

        window.show()

	return

def get_messier_image(widget,window,messier_id):
        if hildon_flag == 1:
                window = hildon.Window()
        else:
                window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title('Planet Ephemeris: WEB MESSIER IMAGES')

	window.set_size_request(800,480)

	screen = window.get_screen()
	screen_width = screen.get_width()
	screen_height = screen.get_height()
	window_width = window.get_size()[0]
	window_height = window.get_size()[1]

        color_black = gtk.gdk.color_parse('black')
	window.modify_bg(gtk.STATE_NORMAL, color_black)

	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
	data_dir = os.path.join(run_dir, 'data')
	messier_file = os.path.join(data_dir, 'messier_url.dat')
	f = open(messier_file,"rb")
	while 1:
		t = f.readline()
		if t == '':
			break
		if '\n' in t:
			t = t[:-1]
		if '\r' in t:
			t = t[:-1]
		a = t.split()

		if str(messier_id) == a[0]:
			url = a[1]
			try:
				file = urllib.urlopen(url)
				print "url=",url
				messier_pic = file.read()
				run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
				data_dir = os.path.join(run_dir, 'data')
				image_file = os.path.join(data_dir, 'messier.gif')
				temp_file = open(image_file,"wb")
				temp_file.write(messier_pic)
				temp_file.close()
				print "image_file=",image_file
			except Exception:
				print "web connection timeout"

			run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
			data_dir = os.path.join(run_dir, 'data')
			image_file = os.path.join(data_dir, 'messier.gif')
			image = gtk.Image()
			image.set_from_file(image_file)

			pixbuf = gtk.gdk.pixbuf_new_from_file(image_file)

			aspect = True
			enlarge = True

			display_image(image,pixbuf,aspect,enlarge,screen_width,screen_height,window_width,window_height)

			table = gtk.Table(1,1, False)
			window.add(table)
			table.attach(image,0,1,0,1)

			event_box = gtk.EventBox()
			table.attach(event_box,0,1,1,2)
			set_event_box_color("clicked",event_box,"black","yellow",url)
			table.show()

			image.show()

			if window_in_fullscreen == True:
				window.fullscreen()

			window.show()

	return

def reset_location(widget,window,table,vbox,local):
        gps_position = [local.lat, local.long, local.date, local.temp, local.pressure, local.elevation]
        print 'GPS SEARCHING...PLEASE WAIT'

	gps_position = get_gps_location("clicked",window,local)

        if gps_position[0] == -999:
		on_show_note(widget,window,"GPS location/time not used -- using internal settings instead","OK")
                gps_position = [local.lat, local.long, local.date]
		print "not found gps position=",gps_position

	local.lat = str(gps_position[0])
	local.long = str(gps_position[1])
	local.date = gps_position[2]
	local.temp = local.temp
	local.pressure = local.pressure
	local.elevation = local.elevation

	table.destroy()
	vbox.destroy()

	show_object_info_basic("clicked",window,local)

	return

def get_gps_location(widget,window,local):

	if gps_mode == "gpsbt":
		if gpsbt_flag == 1:
			print "On Maemo so trying bluetooth GPS..."

			context = gpsbt.start()

			if context == None:
				print 'Problem while connecting!'
				gps_position = (-999,-999)
				return gps_position

			# ensure that GPS device is ready to connect and to receive commands
			time.sleep(2)
			gpsdevice = gpsbt.gps()

			gpsdevice.get_fix()
			
			gps_timer = 0

			print "Waiting for location..."
			while gpsdevice.get_position()[0] == 0.0:
				sys.stdout.write('.')
				time.sleep(1)
				gps_timer = gps_timer + 1
				if gps_timer > int(gps_timeout):
					break

			if gpsdevice.fix.latitude == 0.0:
				gps_position = (-999, -999)
			else:
				gps_lat = gpsdevice.fix.latitude
				gps_long = gpsdevice.fix.longitude
				gps_seconds = gpsdevice.fix.time
				gps_date_time = time.gmtime(gps_seconds)[0:6]
				gps_position = [gps_lat, gps_long, gps_date_time]

			# ends Bluetooth connection
			gpsbt.stop(context)
		else:
			on_show_note(widget,window,"GPS connection not attempted due to GPS 'gpsbt' mode setting which is only available on Maemo","OK")
			print "Not on Maemo so not trying bluetooth GPS..."
			gps_position = (-999, -999)
	else:
		if gps_mode == "gpsd":
			if gpsd_flag == 1:
				print "Now trying gpsd GPS connection..."

				gps_timer = 0

				try:
					session = gps.gps(gpsd_host, gpsd_port)
				except socket.error:
					print "Problem connecting to gpsd on host/port: ", gpsd_host, gpsd_port
					gps_position = (-999, -999, local.date)
					return gps_position

				session.query('admosy')

				while session.status == 0:
					session.query('admosy')
					time.sleep(1)
					print "Waiting for gpsd lock...", gps_timer, gps_timeout, session.status, session.fix.mode
					gps_timer = gps_timer + 1
					if gps_timer > int(gps_timeout):
						print "Timeout waiting for GPS lock!"
						break

				print session
				if session.fix.latitude == 0.0:
					gps_position = (-999, -999, local.date)
				else:
					print "Successful gpsd lock!"
					gps_lat = session.fix.latitude
					gps_long = session.fix.longitude
					gps_seconds = session.fix.time
					print "seconds=",gps_seconds
					if gps_seconds != 10000000000.0:
						gps_date_time = time.gmtime(gps_seconds)[0:6]
					else:
						print "Invalid GPS time returned...using local.date instead"
						gps_date_time = local.date
					gps_position = [gps_lat, gps_long, gps_date_time]
					print gps_position

				session.close()
				print session
				del session
				print "session closed/deleted"
			else:
				on_show_note(widget,window,"GPS connection not attempted due to GPS 'gpsd' mode setting which requires absent gps module","OK")
				print "Python gps module missing so not trying gpsd GPS..."
				gps_position = (-999, -999)
		else:
			if gps_mode == "off":
				on_show_note(widget,window,"GPS connection not attempted due to GPS 'off' mode setting in 'ini' file","OK")
				print "GPS mode set to 'none' in ini file so not trying any gps connections..."
				gps_position = (-999, -999, local.date)
			else:
				on_show_note(widget,window,"GPS connection not attempted due to invalid GPS mode setting in 'ini' file","OK")
				print "Invalid GPS mode set in ini file..."
				gps_position = (-999, -999, local.date)

        return gps_position

def chart_info(object,local,table,row,column):
	global chart_info_toggle_flag

	object.compute(local)

	object_ra = object.ra
	object_dec = object.dec
	object_az = object.az
	object_alt = object.alt
	object_mag = object.mag
	object_elong = object.elong

	object_radius = object.radius
	object_size = object.size
	object_sun_distance = object.sun_distance
	object_earth_distance = object.earth_distance
	object_phase = object.phase

        button = gtk.Button(str(object.name))
	button.connect("clicked",show_object_info_detail_start,object,local)
        table.attach(button,column,column+1,row,row+1)
        button.show()                            
	set_button_color("clicked",button,"black","black","black","light blue","green","red")

	if chart_info_toggle_flag == False:
		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",object_ra)

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",object_dec)

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",round(object_mag,1))

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",round(object_size,1))

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",round(object_sun_distance,3))

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",round(object_earth_distance,3))

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",round(object_phase,0))
	else:
		save2 = local.date

		object_next_antitransit = local.next_antitransit(object)
		object_antitransit_alt = object.alt
		local.date = save2
		object_next_rising = local.next_rising(object)
		object_rising_az = object.az
		local.date = save2
		object_next_transit = local.next_transit(object)
		object_transit_alt = object.alt
		local.date = save2
		object_next_setting = local.next_setting(object)
		object_setting_az = object.az
		local.date = save2

		local_time_zone_name = time.tzname

		if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
			local_time_zone_daylight_flag = 0
		else:
			local_time_zone_daylight_flag = time.daylight

		if date_in_gmt == True:
			object_next_antitransit_show = str(object_next_antitransit)[:19][-8:]+' GMT'
			object_next_rising_show = str(object_next_rising)[:19][-8:]+' GMT'
			object_next_setting_show = str(object_next_setting)[:19][-8:]+' GMT'
			object_next_transit_show = str(object_next_transit)[:19][-8:]+' GMT'
		else:
			object_next_antitransit_show = str(ephem.localtime(object_next_antitransit))[:19][-8:]+' '+local_time_zone_name[local_time_zone_daylight_flag]
			object_next_rising_show = str(ephem.localtime(object_next_rising))[:19][-8:]+' '+local_time_zone_name[local_time_zone_daylight_flag]
			object_next_setting_show = str(ephem.localtime(object_next_setting))[:19][-8:]+' '+local_time_zone_name[local_time_zone_daylight_flag]
			object_next_transit_show = str(ephem.localtime(object_next_transit))[:19][-8:]+' '+local_time_zone_name[local_time_zone_daylight_flag]

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",object_az)

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",object_alt)

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",object_next_rising_show)

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",object_rising_az)

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",object_next_setting_show)

		column = column+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",object_setting_az)

        return

def set_button_color(widget,button,bg_normal_color,bg_active_color,bg_prelight_color,fg_normal_color,fg_active_color,fg_prelight_color):
        button_map = button.get_colormap()
        bg_normal_color2 = button_map.alloc_color(bg_normal_color)
        bg_active_color2 = button_map.alloc_color(bg_active_color)
        bg_prelight_color2 = button_map.alloc_color(bg_prelight_color)
        fg_normal_color2 = button_map.alloc_color(fg_normal_color)
        fg_active_color2 = button_map.alloc_color(fg_active_color)
        fg_prelight_color2 = button_map.alloc_color(fg_prelight_color)
        button_style = button.get_style().copy()
        button_style.bg[gtk.STATE_NORMAL] = bg_normal_color2
        button_style.bg[gtk.STATE_ACTIVE] = bg_active_color2
        button_style.bg[gtk.STATE_PRELIGHT] = bg_prelight_color2
        button_label = button.get_child()
        button_label.modify_fg(gtk.STATE_NORMAL, gtk.gdk.color_parse(fg_normal_color))
        button_label.modify_fg(gtk.STATE_ACTIVE, gtk.gdk.color_parse(fg_active_color))
        button_label.modify_fg(gtk.STATE_PRELIGHT, gtk.gdk.color_parse(fg_prelight_color))
        button.set_style(button_style)
  	if button.get_use_stock():
     		label = button.child.get_children()[1]
  	elif isinstance(button.child, gtk.Label):
     		label = button.child
	label.modify_font(pango.FontDescription("sans 12"))
        button.show()

	return

def set_event_box_color(widget,event_box,bg_normal_color,fg_normal_color,text):
	event_box.show()
        label = gtk.Label('')
	label.set_markup('<span foreground="' + fg_normal_color + '">' + str(text) + '</span>')
	label.set_use_markup(True)
	event_box.add(label)
	event_box.modify_bg(gtk.STATE_NORMAL,event_box.get_colormap().alloc_color(bg_normal_color))
	label.modify_font(pango.FontDescription("sans 12"))
	label.show()                            

	return

def change_date_time_core(local,date_time_change_flag):
	if date_time_change_flag == -5:
		current_date = (local.date.triple()[0]-1,local.date.triple()[1],local.date.triple()[2])
	elif date_time_change_flag == -4:
		current_date = (local.date.triple()[0],local.date.triple()[1]-1,local.date.triple()[2])
	elif date_time_change_flag == -3:
		current_date = (local.date.triple()[0],local.date.triple()[1],local.date.triple()[2]-1)
	elif date_time_change_flag == -2:
		current_date = (local.date.triple()[0],local.date.triple()[1],local.date.triple()[2]-(1/24.0))
	elif date_time_change_flag == -1:
		current_date = (local.date.triple()[0],local.date.triple()[1],local.date.triple()[2]-(1/24.0/60))
	elif date_time_change_flag == 5:
		current_date = (local.date.triple()[0]+1,local.date.triple()[1],local.date.triple()[2])
	elif date_time_change_flag == 4:
		current_date = (local.date.triple()[0],local.date.triple()[1]+1,local.date.triple()[2])
	elif date_time_change_flag == 3:
		current_date = (local.date.triple()[0],local.date.triple()[1],local.date.triple()[2]+1)
	elif date_time_change_flag == 2:
		current_date = (local.date.triple()[0],local.date.triple()[1],local.date.triple()[2]+(1/24.0))
	elif date_time_change_flag == 1:
		current_date = (local.date.triple()[0],local.date.triple()[1],local.date.triple()[2]+(1/24.0/60))
	elif date_time_change_flag == 0:
		current_date = ephem.now()
	
	local.date=ephem.date(current_date)

	return

def change_date_time_basic(widget,window,table,vbox,local,date_time_change_flag):
	change_date_time_core(local,date_time_change_flag)

	table.destroy()
	vbox.destroy()

	show_object_info_basic("clicked",window,local)

	return

def change_date_time_detail(widget,window,table,object,local,date_time_change_flag):
	change_date_time_core(local,date_time_change_flag)

	table.destroy()

	show_object_info_detail("clicked",window,object,local)

	return

def change_date_solar_system_plot(widget,window,vbox,object,local,date_time_change_flag):
	change_date_time_core(local,date_time_change_flag)
	
	vbox.destroy()

	plot_solar_system_position("clicked",window,vbox,object,local)

	return

def change_date_moon_plot(widget,window,vbox,local,date_time_change_flag):
	change_date_time_core(local,date_time_change_flag)

	vbox.destroy()

	plot_moon_phase("clicked",window,vbox,local)

	return

def change_date_sky_plot(widget,window,vbox,local,date_time_change_flag):
	change_date_time_core(local,date_time_change_flag)

	vbox.destroy()

	plot_sky("clicked",window,vbox,local)

	return

def change_date_planet_moon_plot(widget,window,vbox,object,local,date_time_change_flag):
	change_date_time_core(local,date_time_change_flag)

	vbox.destroy()

	plot_planet_moon_position("clicked",window,vbox,object,local)

	return

def get_geonames_weather(weather):
	global geonames_location
	global station_name
	global country_name
	global elevation_value
	global elevation_string
	global temperature_value
	global temperature_string
	global latitude_value
	global latitude_string
	global longitude_value
	global longitude_string
	global weather_date
	global humidity_value
	global humidity_string
	global wind_speed_value
	global wind_degrees_value
	global wind_string
	global pressure_value
	global pressure_string
	global dewpoint_value
	global dewpoint_string
	global clouds_value
	global condition_value

        url = 'http://ws.geonames.org/findNearByWeatherXML?lat='+str(weather.lat*180/pi)+'&lng='+str(weather.long*180/pi)

        dom = xml.dom.minidom.parse(urllib2.urlopen(url))

        weather_name = dom.getElementsByTagName('observation')[0].firstChild.data
        weather_date = dom.getElementsByTagName('observationTime')[0].firstChild.data
        station_name = dom.getElementsByTagName('stationName')[0].firstChild.data
        station_id = dom.getElementsByTagName('ICAO')[0].firstChild.data
	country_name = dom.getElementsByTagName('countryCode')[0].firstChild.data
        elevation_value = dom.getElementsByTagName('elevation')[0].firstChild.data
        latitude_value = dom.getElementsByTagName('lat')[0].firstChild.data
        longitude_value = dom.getElementsByTagName('lng')[0].firstChild.data
        temperature_value = dom.getElementsByTagName('temperature')[0].firstChild.data
        dewpoint_value = dom.getElementsByTagName('dewPoint')[0].firstChild.data
        humidity_value = dom.getElementsByTagName('humidity')[0].firstChild.data
        clouds_value = dom.getElementsByTagName('clouds')[0].firstChild.data
        pressure_value = dom.getElementsByTagName('seaLevelPressure')[0].firstChild.data
        wind_degrees_value = dom.getElementsByTagName('windDirection')[0].firstChild.data
        wind_speed_value = dom.getElementsByTagName('windSpeed')[0].firstChild.data
        condition_value = dom.getElementsByTagName('weatherCondition')[0].firstChild.data

	latitude_value = str(round(float(latitude_value),4))
	longitude_value = str(round(float(longitude_value),4))

	temperature_string = temperature_value+u"\xb0"+ "C"
	dewpoint_string = dewpoint_value+u"\xb0"+"C"
	pressure_string = pressure_value+" mb"
	wind_string = wind_speed_value+" kph "+wind_degrees_value+u"\xb0"
	humidity_string = "n/a"
	humidity_string = humidity_value+"%"
	latitude_string = latitude_value+u"\xb0"
	longitude_string = longitude_value+u"\xb0"
	elevation_string = elevation_value+" m"

	weather.lat = str(latitude_value)
	weather.long = str(longitude_value)
	weather.elevation = float(elevation_value)
	weather.temp = float(temperature_value)
	weather.pressure = float(pressure_value)

	return

def get_yahoo_weather(weather):
	global yahoo_location
	global city_name
	global country_name
	global elevation_value
	global elevation_string
	global sunrise_time
	global sunset_time
	global temperature_value
	global temperature_unit
	global temperature_string
	global latitude_value
	global latitude_string
	global longitude_value
	global longitude_string
	global weather_date
	global humidity_value
	global humidity_string
	global wind_speed_value
	global wind_speed_unit
	global wind_degrees_value
	global wind_chill_value
	global wind_string
	global pressure_value
	global pressure_unit
	global pressure_rising_value
	global pressure_string
	global visibility_value
	global visibility_distance_unit
	global visibility_string

        url = 'http://xml.weather.yahoo.com/forecastrss?p='+yahoo_location+'&u=c'

        dom = xml.dom.minidom.parse(urllib2.urlopen(url))

        xml_value = dom.getElementsByTagName('yweather:location')[0]
        city_name = xml_value.getAttribute('city')
        country_name = xml_value.getAttribute('country')

        xml_value = dom.getElementsByTagName('yweather:astronomy')[0]
        sunrise_time = xml_value.getAttribute('sunrise')
        sunset_time = xml_value.getAttribute('sunset')

        xml_value = dom.getElementsByTagName('yweather:atmosphere')[0]
        humidity_value = xml_value.getAttribute('humidity')
        visibility_value = xml_value.getAttribute('visibility')
        pressure_value = xml_value.getAttribute('pressure')
        pressure_rising_flag = xml_value.getAttribute('rising')

        xml_value = dom.getElementsByTagName('yweather:wind')[0]
        wind_chill_value = xml_value.getAttribute('chill')
        wind_degrees_value = xml_value.getAttribute('direction')
        wind_speed_value = xml_value.getAttribute('speed')

        xml_value = dom.getElementsByTagName('yweather:units')[0]
        temperature_unit = xml_value.getAttribute('temperature')
        visibility_distance_unit = xml_value.getAttribute('distance')
        pressure_unit = xml_value.getAttribute('pressure')
        wind_speed_unit = xml_value.getAttribute('speed')

        xml_value = dom.getElementsByTagName('yweather:condition')[0]
        temperature_value = xml_value.getAttribute('temp')
        latitude_value = dom.getElementsByTagName('geo:lat')[0].firstChild.data
        longitude_value = dom.getElementsByTagName('geo:long')[0].firstChild.data
        weather_date = dom.getElementsByTagName('pubDate')[0].firstChild.data

	elevation_show_flag = False
	elevation_value = get_geonames_elevation("clicked",weather,elevation_show_flag)

	weather.lat = str(latitude_value)
	weather.long = str(longitude_value)
	weather.elevation = float(elevation_value)
	weather.temp = float(temperature_value)
	weather.pressure = float(pressure_value)

        if pressure_rising_flag == "0":
                pressure_rising_value = "Steady"

        if pressure_rising_flag == "1":
                pressure_rising_value = "Rising"

        if pressure_rising_flag == "2":
                pressure_rising_value = "Falling"

	temperature_string = temperature_value+u"\xb0"+temperature_unit
	if int(pressure_value) > 0:
		pressure_string = pressure_value+" "+pressure_unit+" "+pressure_rising_value
	else:
		pressure_string = "n/a"
	wind_string = wind_speed_value+" "+wind_speed_unit+" "+wind_degrees_value+u"\xb0"
	visibility_string = visibility_value+" "+visibility_distance_unit
	if humidity_value == "":
		humidity_string = "n/a"
	else:
		humidity_string = humidity_value+"%"
	latitude_string = latitude_value+u"\xb0"
	longitude_string = longitude_value+u"\xb0"
	elevation_string = elevation_value+" m (GeoNames)"

	return

def get_wunderground_weather(weather):
	global wunderground_location
	global station_id
	global city_name
	global temperature_value
	global temperature_string
	global latitude_value
	global latitude_string
	global longitude_value
	global elevation_value
	global elevation_string
	global longitude_string
	global weather_date
	global humidity_value
	global humidity_string
	global wind_speed_value
	global wind_degrees_value
	global wind_string
	global pressure_value
	global pressure_string
	global dewpoint_value
	global dewpoint_string

        url = 'http://api.wunderground.com/weatherstation/WXCurrentObXML.asp?ID='+wunderground_location

        dom = xml.dom.minidom.parse(urllib2.urlopen(url))

        station_id = dom.getElementsByTagName('station_id')[0].firstChild.data
        city_name = dom.getElementsByTagName('city')[0].firstChild.data
        pressure_value = dom.getElementsByTagName('pressure_mb')[0].firstChild.data
        pressure_string = dom.getElementsByTagName('pressure_string')[0].firstChild.data
        temperature_value = dom.getElementsByTagName('temp_c')[0].firstChild.data
        temperature_string = dom.getElementsByTagName('temperature_string')[0].firstChild.data
        latitude_value = dom.getElementsByTagName('latitude')[0].firstChild.data
        longitude_value = dom.getElementsByTagName('longitude')[0].firstChild.data
        elevation_value = dom.getElementsByTagName('elevation')[0].firstChild.data
        weather_date = dom.getElementsByTagName('observation_time')[0].firstChild.data
        humidity_value = dom.getElementsByTagName('relative_humidity')[0].firstChild.data
        dewpoint_value = dom.getElementsByTagName('dewpoint_c')[0].firstChild.data
        dewpoint_string = dom.getElementsByTagName('dewpoint_string')[0].firstChild.data
        wind_string = dom.getElementsByTagName('wind_string')[0].firstChild.data
        wind_direction = dom.getElementsByTagName('wind_dir')[0].firstChild.data
        wind_degrees_value = dom.getElementsByTagName('wind_degrees')[0].firstChild.data
        wind_speed_value = dom.getElementsByTagName('wind_mph')[0].firstChild.data

	wind_speed_value = str(round(float(wind_speed_value)*1.609344,1))
	elevation_value =str(round(float(elevation_value.split()[0])*0.3048,1))

	temperature_string = temperature_value+u"\xb0"+ "C"
	dewpoint_string = dewpoint_value+u"\xb0"+"C"
	pressure_string = pressure_value+" mb"
	wind_string = wind_speed_value+" kph "+wind_degrees_value+u"\xb0"
	humidity_string = humidity_value+"%"
	latitude_string = latitude_value+u"\xb0"
	longitude_string = longitude_value+u"\xb0"
	elevation_string = elevation_value+" m"

	weather.lat = str(latitude_value)
	weather.long = str(longitude_value)
	weather.elevation = float(elevation_value)
	weather.temp = float(temperature_value)
	weather.pressure = float(pressure_value)

	return

def show_local_weather(widget,local,xml_flag):
	if hildon_flag == 1:
		window = hildon.Window()
	else:
		window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title('Planet Ephemeris: WEB LOCAL WEATHER')

	window.set_size_request(800,480)

	weather = ephem.Observer()
	weather.lat = local.lat
	weather.long = local.long

	if xml_flag == 1:
		window.set_title('Weather Information: GEONAMES')
		get_geonames_weather(weather)
	if xml_flag == 2:
		window.set_title('Weather Information: YAHOO')
		get_yahoo_weather(weather)
	if xml_flag == 3:
		window.set_title('Weather Information: WUNDERGROUND')
		get_wunderground_weather(weather)

        table = gtk.Table(8,4,False)
	table.set_col_spacings(1)
	table.set_row_spacings(1)

	row = 0
	column = 0

	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Weather Date")

	if xml_flag == 1:
		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Station Name")
	if xml_flag == 2 or xml_flag == 3:	
		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","City Name")
	if xml_flag == 3:
		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Station Id")

	if xml_flag == 1 or xml_flag == 2:
		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Country Name")

	row = row+1
	event_box = gtk.EventBox()
	table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Latitude")

	row = row+1
	event_box = gtk.EventBox()
	table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Longitude")

	row = row+1
	button = gtk.Button("Elevation")
	button.connect("clicked",change_elevation,window,local,weather)
	table.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	row = row+1
	button = gtk.Button("Temperature")
	button.connect("clicked",change_temperature,window,local,weather)
	table.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	row = row+1
	button = gtk.Button("Pressure")
	button.connect("clicked",change_pressure,window,local,weather)
	table.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	if xml_flag == 1 or xml_flag == 3:
		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Dewpoint")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Humidity")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Wind")

	if xml_flag == 2:
		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Visibility")

		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Sunrise")

		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Sunset")

	if xml_flag == 1:
		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Clouds")

		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Condition")

	row = row+1
	button = gtk.Button("Distance")
        button.connect("clicked",get_google_map_start,local,weather)
	table.attach(button,column,column+1,row,row+1)
	button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	row = 0
	row = 0
	column = column+1

	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",weather_date)

	if xml_flag == 1:
		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",station_name)
	if xml_flag == 2 or xml_flag == 3:
		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",city_name)
	if xml_flag == 3:
		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",station_id)

	if xml_flag == 1 or xml_flag == 2:
		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",country_name)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",latitude_string)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",longitude_string)

	row = row+1
	event_box = gtk.EventBox()
	table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",elevation_string)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",temperature_string)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",pressure_string)

	if xml_flag == 1 or xml_flag == 3:
		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",dewpoint_string)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",humidity_string)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",wind_string)

	if xml_flag == 2:
		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",visibility_string)

		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",sunrise_time)

		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",sunset_time)

	if xml_flag == 1:
		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",clouds_value)

		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",condition_value)

	table.show()
        window.add(table)

	if window_in_fullscreen == True:
		window.fullscreen()

        window.show()

	r = 6378.7
	distance_value = r * acos(sin(local.lat) * sin(weather.lat) + cos(local.lat) * cos(weather.lat) * cos((weather.long)-(local.long)))
	distance_string = str(round(distance_value,2))+" km"

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",distance_string)

	return

def get_ngc_data(widget,window):
	global ngc_source

	url = ngc_source

        try:
                file = urllib.urlopen(url)
		print "url=",url
                ngc_data = file.read()
                run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
                data_dir = os.path.join(run_dir, 'data')
                text_file = os.path.join(data_dir, 'ngc.dat')
                temp_file = open(text_file,"wb")
                temp_file.write(ngc_data)
                temp_file.close()
		print "ngc_data:",text_file
        except Exception:
                print "web connection timeout"

	ngc_count = 0

	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
	data_dir = os.path.join(run_dir, 'data')
	ngc_file = os.path.join(data_dir, 'ngc.dat')
	f = open(ngc_file,"rb")
	while 1:
		t = f.readline()
		if t == '':
			break
		ngc_count = ngc_count + 1
	f.close()

	print "NGC count=", ngc_count
	on_show_note(widget,window,str(ngc_count)+" NGC objects successfully downloaded from: "+url,"OK")

	return

def get_asteroid_data(widget,window):
	global asteroid_source
	global asteroid_read_flag

	url = asteroid_source

        try:
                file = urllib.urlopen(url)
		print "url=",url
                asteroid_elements = file.read()
                run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
                data_dir = os.path.join(run_dir, 'data')
                text_file = os.path.join(data_dir, 'asteroid.dat')
                temp_file = open(text_file,"wb")
                temp_file.write(asteroid_elements)
                temp_file.close()
		print "asteroid_elements:",text_file
        except Exception:
                print "web connection timeout"

	asteroid_count = 0

	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
	data_dir = os.path.join(run_dir, 'data')
	asteroid_file = os.path.join(data_dir, 'asteroid.dat')
	f = open(asteroid_file,"rb")
	while 1:
		t = f.readline()
		if t == '':
			break
		if '\n' in t:
			t = t[:-1]
		if '\r' in t:
			t = t[:-1]
		a = t.split()
		line_type = a[0]
		if (line_type != "#"):
			asteroid_count = asteroid_count + 1

	asteroid_read_flag = False
	f.close()

	print "asteroid count=", asteroid_count
	on_show_note(widget,window,str(asteroid_count)+" asteroid orbitial elements successfully downloaded from: "+url,"OK")

	return

def get_comet_data(widget,window):
	global comet_source
	global comet_read_flag

	url = comet_source

        try:
                file = urllib.urlopen(url)
		print "url=",url
                comet_elements = file.read()
                run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
                data_dir = os.path.join(run_dir, 'data')
                text_file = os.path.join(data_dir, 'comet.dat')
                temp_file = open(text_file,"wb")
                temp_file.write(comet_elements)
                temp_file.close()
		print "comet_elements:",text_file
        except Exception:
                print "web connection timeout"

	comet_count = 0

	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
	data_dir = os.path.join(run_dir, 'data')
	comet_file = os.path.join(data_dir, 'comet.dat')
	f = open(comet_file,"rb")
	while 1:
		t = f.readline()
		if t == '':
			break
		if '\n' in t:
			t = t[:-1]
		if '\r' in t:
			t = t[:-1]
		a = t.split()
		line_type = a[0]
		if (line_type != "#"):
			comet_count = comet_count + 1

	comet_read_flag = False
	f.close()

	print "comet count=", comet_count
	on_show_note(widget,window,str(comet_count)+" comet orbitial elements successfully downloaded from: "+url,"OK")

	return

def get_neo_data(widget,window):
	global neo_source
	global neo_read_flag

	url = neo_source

        try:
                file = urllib.urlopen(url)
		print "url=",url
                neo_elements = file.read()
                run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
                data_dir = os.path.join(run_dir, 'data')
                text_file = os.path.join(data_dir, 'neo.dat')
                temp_file = open(text_file,"wb")
                temp_file.write(neo_elements)
                temp_file.close()
		print "neo_elements:",text_file
        except Exception:
                print "web connection timeout"

	neo_count = 0

	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
	data_dir = os.path.join(run_dir, 'data')
	neo_file = os.path.join(data_dir, 'neo.dat')
	f = open(neo_file,"rb")
	while 1:
		t = f.readline()
		if t == '':
			break
		if '\n' in t:
			t = t[:-1]
		if '\r' in t:
			t = t[:-1]
		a = t.split()
		line_type = a[0]
		if (line_type != "#"):
			neo_count = neo_count + 1

	neo_read_flag = False
	f.close()

	print "NEO count=", neo_count
	on_show_note(widget,window,str(neo_count)+" NEO orbitial elements successfully downloaded from: "+url,"OK")

	return

def get_tle_elements(widget,window):
	global tle_group

	url = tle_group

        try:
                file = urllib.urlopen(url)
		print "url=",url
                tle_elements = file.read()
                run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
                data_dir = os.path.join(run_dir, 'data')
                text_file = os.path.join(data_dir, 'tle.dat')
                temp_file = open(text_file,"wb")
                temp_file.write(tle_elements)
                temp_file.close()
		print "tle_elements:",text_file
        except Exception:
                print "web connection timeout"

	satellite_count = 0

	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
	data_dir = os.path.join(run_dir, 'data')
	tle_file = os.path.join(data_dir, 'tle.dat')
	f = open(tle_file,"rb")
	while 1:
		t = f.readline()
		if t == '':
			break
		if '\n' in t:
			t = t[:-1]
		if '\r' in t:
			t = t[:-1]
		a = t.split()
		tle_type = a[0]
		if (tle_type != "1") and (tle_type != "2"):
			satellite_count = satellite_count + 1

	f.close()

	print "sat count=", satellite_count
	on_show_note(widget,window,str(satellite_count)+" TLE elements successfully downloaded from: "+url,"OK")

	return

def show_object_info_detail_start(widget,object,local):

	if hildon_flag == 1:
		window = hildon.Window()
	else:
		window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title('Planet Ephemeris for '+object.name+': DETAIL INFO')

	window.set_size_request(800,480)

	show_object_info_detail("clicked",window,object,local)

	return

def show_object_info_detail(widget,window,object,local):
	global rise_set_plot_flag

	if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
		local_time_zone_daylight_flag = 0
	else:
		local_time_zone_daylight_flag = time.daylight

	local_time_zone_name = time.tzname

	save = local.date #still required for 3.7.3.2

        object.compute(local)

        object_ra = object.ra
        object_dec = object.dec
        object_az = object.az
        object_alt = object.alt
        object_mag = object.mag
        object_elong = object.elong

        object_radius = object.radius
        object_size = object.size
        object_sun_distance = object.sun_distance
        object_earth_distance = object.earth_distance
        object_phase = object.phase

	geo_ra = object.g_ra
	geo_dec = object.g_dec
	astro_ra = object.a_ra
	astro_dec = object.a_dec
	helio_lat = object.hlat
	helio_long = object.hlong

	constellation = ephem.constellation(object)[1]
	
	timezone_offset = ephem.date(local.date)-ephem.date(ephem.localtime(local.date))
        start_of_day = ephem.date(str(local.date.tuple()[0])+'/'+str(local.date.tuple()[1])+'/'+str(local.date.tuple()[2])+' 00:00') #reset to start of day GMT
        local.date = start_of_day+timezone_offset # compensate for differing time zone

	save2 = local.date

	object_next_antitransit = local.next_antitransit(object)
        object_antitransit_alt = object.alt
	local.date = save2
	object_next_rising = local.next_rising(object)
        object_rising_az = object.az
	local.date = save2
	object_next_transit = local.next_transit(object)
        object_transit_alt = object.alt
	local.date = save2
	object_next_setting = local.next_setting(object)
        object_setting_az = object.az
	local.date = save2

	day_length = (object_next_setting-object_next_rising)*24
	if day_length < 0:
		day_length = 24.0 + day_length

	local.date = save #still required for 3.7.3.2

	if date_in_gmt == True:
		object_next_antitransit_show = str(object_next_antitransit)[:19]+' GMT'
		object_next_rising_show = str(object_next_rising)[:19]+' GMT'
		object_next_setting_show = str(object_next_setting)[:19]+' GMT'
		object_next_transit_show = str(object_next_transit)[:19]+' GMT'
	else:
		object_next_antitransit_show = str(ephem.localtime(object_next_antitransit))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		object_next_rising_show = str(ephem.localtime(object_next_rising))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		object_next_setting_show = str(ephem.localtime(object_next_setting))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		object_next_transit_show = str(ephem.localtime(object_next_transit))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]

        table = gtk.Table(8,4,False)
	table.set_col_spacings(1)
	table.set_row_spacings(1)

        table2 = gtk.Table(1,7,False)
	table2.set_col_spacings(1)
	table2.set_row_spacings(1)

	current_local_date = ephem.Date(ephem.localtime(local.date))

	row = 0
	column = 0

        if date_in_gmt == True:
                title = 'DATE/TIME: '+str(local.date)+' GMT LOCATION: '+str(local.lat)+u'\xb0 '+str(local.long)+u'\xb0'
        else:
                title = 'DATE/TIME: '+str(current_local_date)+' '+local_time_zone_name[local_time_zone_daylight_flag]+' LOCATION: '+str(local.lat)+u'\xb0 '+str(local.long)+u'\xb0'

        button = gtk.Button(title)
	button.connect("clicked",toggle_date_style,window,button,local)
        table.attach(button,column,column+4,row,row+1)
        button.show()                            
	set_button_color("clicked",button,"black","black","black","orange","green","red")

	row = row+1
	column = 0
	date_time_change_flag = -5
        button = gtk.Button("-1 Yr")
	button.connect("clicked",change_date_time_detail,window,table,object,local,date_time_change_flag)
        table2.attach(button,column,column+1,row,row+1)
        button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = -4
        button = gtk.Button("-1 Mo")
	button.connect("clicked",change_date_time_detail,window,table,object,local,date_time_change_flag)
        table2.attach(button,column,column+1,row,row+1)
        button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = -3
        button = gtk.Button("-1 Dy")
	button.connect("clicked",change_date_time_detail,window,table,object,local,date_time_change_flag)
        table2.attach(button,column,column+1,row,row+1)
        button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = -2
        button = gtk.Button("-1 Hr")
	button.connect("clicked",change_date_time_detail,window,table,object,local,date_time_change_flag)
        table2.attach(button,column,column+1,row,row+1)
        button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = -1
        button = gtk.Button("-1 Mi")
	button.connect("clicked",change_date_time_detail,window,table,object,local,date_time_change_flag)
        table2.attach(button,column,column+1,row,row+1)
        button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = 0
        button = gtk.Button('Now')
	button.connect("clicked",change_date_time_detail,window,table,object,local,date_time_change_flag)
        table2.attach(button,column,column+1,row,row+1)
        button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = 1
        button = gtk.Button("+1 Mi")
	button.connect("clicked",change_date_time_detail,window,table,object,local,date_time_change_flag)
        table2.attach(button,column,column+1,row,row+1)
        button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = 2
        button = gtk.Button("+1 Hr")
	button.connect("clicked",change_date_time_detail,window,table,object,local,date_time_change_flag)
        table2.attach(button,column,column+1,row,row+1)
        button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = 3
        button = gtk.Button("+1 Dy")
	button.connect("clicked",change_date_time_detail,window,table,object,local,date_time_change_flag)
        table2.attach(button,column,column+1,row,row+1)
        button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = 4
        button = gtk.Button("+1 Mo")
	button.connect("clicked",change_date_time_detail,window,table,object,local,date_time_change_flag)
        table2.attach(button,column,column+1,row,row+1)
        button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

	column = column+1
	date_time_change_flag = 5
        button = gtk.Button("+1 Yr")
	button.connect("clicked",change_date_time_detail,window,table,object,local,date_time_change_flag)
        table2.attach(button,column,column+1,row,row+1)
        button.show()                            
	set_button_color("clicked",button,"black","black","black","red","green","red")

        table.attach(table2,0,column+4,row,row+1)

	row = row+1
	column = 0
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Object")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Right Ascension")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Azimuth")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Transit Time")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Antitransit Time")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Rise Date/Time")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Set Date/Time")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Sun Distance")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Visual Size")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Magnitude")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Geo Right Ascension")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Astro Right Ascension")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Helio Latitude")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Day Length")

	row = 1
	column = 2

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Constellation")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Declination")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Altitude")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Transit Altitude")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Antitransit Altitude")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Rise Azimuth")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Set Azimuth")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Earth Distance")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Phase")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Elongation")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Geo Declination")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Astro Declination")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Helio Longitude")

	row = row+1
	if object.name != "Moon":
		rise_set_plot_flag = True
		button=gtk.Button("Rise/Set Graph")
		button.connect("clicked",plot_object_rise_set_graph_start,object,local)
		table.attach(button,column,column+1,row,row+1)
		button.show()
		set_button_color("clicked",button,"black","black","black","light blue","green","red")
	else:
		button = gtk.Button("Phase")
		button.connect("clicked",plot_moon_phase_start,local)
		table.attach(button,column,column+1,row,row+1)
		button.show()                            
		set_button_color("clicked",button,"black","black","black","light blue","green","red")

	row = 1 
	column = 1 

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",object.name)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",object_ra)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",object_az)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",object_next_transit_show)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",object_next_antitransit_show)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",object_next_rising_show)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",object_next_setting_show)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",round(object_sun_distance,3))

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",round(object_size,1))

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",round(object_mag,1))

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",geo_ra)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",astro_ra)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",helio_lat)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",day_length)

	row = 1 
	column = 3

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",constellation)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",object_dec)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",object_alt)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",object_transit_alt)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",object_antitransit_alt)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",object_rising_az)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",object_setting_az)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",round(object_earth_distance,3))

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",round(object_phase,0))

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",object_elong)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",geo_dec)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",astro_dec)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",helio_long)

	if object.name in ('Mars', 'Jupiter', 'Saturn', 'Uranus'):
		row = row+1
		button=gtk.Button("Moon Positions")
		button.connect("clicked",plot_planet_moon_position_start,object,local)
		table.attach(button,column,column+1,row,row+1)
		button.show()
		set_button_color("clicked",button,"black","black","black","light blue","green","red")
	else:
		if object.name == 'Moon':
			row = row+1
			button = gtk.Button("More Info")
			button.connect("clicked",show_moon_info,local,save)
			table.attach(button,column,column+1,row,row+1)
			button.show()                            
			set_button_color("clicked",button,"black","black","black","light blue","green","red")
		else:
			if object.name == 'Sun':
				row = row+1
				button = gtk.Button("Solar System Plot")
				button.connect("clicked",plot_solar_system_position_start,object,local)
				table.attach(button,column,column+1,row,row+1)
				button.show()
				set_button_color("clicked",button,"black","black","black","light blue","green","red")
			else:
				row = row+1
				event_box = gtk.EventBox()
				table.attach(event_box,column,column+1,row,row+1)
				set_event_box_color("clicked",event_box,"black","green","")

	table.show()
	table2.show()
        window.add(table)

	if window_in_fullscreen == True:
		window.fullscreen()

        window.show()

	return

def show_moon_info(widget,local,now):
	if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
		local_time_zone_daylight_flag = 0
	else:
		local_time_zone_daylight_flag = time.daylight

	local_time_zone_name = time.tzname

	save = local.date #still required for 3.7.3.2

	moon = ephem.Moon()
	moon.compute(local)

	moon_libration_lat = moon.libration_lat
	moon_libration_long = moon.libration_long
	moon_colong = moon.colong
	moon_subsolar_lat = moon.subsolar_lat

	timezone_offset = ephem.date(local.date)-ephem.date(ephem.localtime(local.date))
        start_of_day = ephem.date(str(local.date.tuple()[0])+'/'+str(local.date.tuple()[1])+'/'+str(local.date.tuple()[2])+' 00:00') #reset to start of day GMT
        local.date = start_of_day+timezone_offset # compensate for differing time zone

	save2 = local.date

	moon_next_rising=local.next_rising(moon) #this resets the moon position!
	local.date = save2 
	moon_previous_transit=local.previous_transit(moon) #this resets the moon position!
	local.date = save2 
	moon_previous_antitransit=local.previous_antitransit(moon) #this resets the moon position!
	local.date = save2 
	moon_next_setting=local.next_setting(moon) #this resets the moon position!
	local.date = save2 
	moon_next_transit=local.next_transit(moon) #this resets the moon position!
	local.date = save2 
	moon_next_antitransit=local.next_antitransit(moon) #this resets the moon position!
	local.date = save2

	moon_previous_new=ephem.previous_new_moon(now)
	moon_next_new=ephem.next_new_moon(now)
	moon_previous_first_quarter=ephem.previous_first_quarter_moon(now)
	moon_next_first_quarter=ephem.next_first_quarter_moon(now)
	moon_previous_full=ephem.previous_full_moon(now)
	moon_next_full=ephem.next_full_moon(now)
	moon_previous_last_quarter=ephem.previous_last_quarter_moon(now)
	moon_next_last_quarter=ephem.next_last_quarter_moon(now)

	moon_prev_full=ephem.previous_full_moon(now)
	moon_next_full=ephem.next_full_moon(now)
	moon_full_diff=moon_next_full-moon_prev_full
	moon_curr_diff=moon_next_full-local.date

	if date_in_gmt == True:
		moon_next_rising_show = str(moon_next_rising)[:19]+' GMT'
		moon_next_setting_show = str(moon_next_setting)[:19]+' GMT'
		moon_previous_transit_show = str(moon_previous_transit)[:19]+' GMT'
		moon_previous_antitransit_show = str(moon_previous_antitransit)[:19]+' GMT'
		moon_next_transit_show = str(moon_next_transit)[:19]+' GMT'
		moon_next_antitransit_show = str(moon_next_antitransit)[:19]+' GMT'
		moon_previous_new_show = str(moon_previous_new)[:19]+' GMT'
		moon_previous_first_quarter_show = str(moon_previous_first_quarter)[:19]+' GMT'
		moon_previous_full_show = str(moon_previous_full)[:19]+' GMT'
		moon_previous_last_quarter_show = str(moon_previous_last_quarter)[:19]+' GMT'
		moon_next_new_show = str(moon_next_new)[:19]+' GMT'
		moon_next_first_quarter_show = str(moon_next_first_quarter)[:19]+' GMT'
		moon_next_full_show = str(moon_next_full)[:19]+' GMT'
		moon_next_last_quarter_show = str(moon_next_last_quarter)[:19]+' GMT'
	else:
		moon_next_rising_show = str(ephem.localtime(moon_next_rising))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		moon_next_setting_show = str(ephem.localtime(moon_next_setting))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		moon_previous_transit_show = str(ephem.localtime(moon_previous_transit))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		moon_previous_antitransit_show = str(ephem.localtime(moon_previous_antitransit))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		moon_next_transit_show = str(ephem.localtime(moon_next_transit))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		moon_next_antitransit_show = str(ephem.localtime(moon_next_antitransit))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		moon_previous_new_show = str(ephem.localtime(moon_previous_new))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		moon_previous_first_quarter_show = str(ephem.localtime(moon_previous_first_quarter))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		moon_previous_full_show = str(ephem.localtime(moon_previous_full))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		moon_previous_last_quarter_show = str(ephem.localtime(moon_previous_last_quarter))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		moon_next_new_show = str(ephem.localtime(moon_next_new))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		moon_next_first_quarter_show = str(ephem.localtime(moon_next_first_quarter))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		moon_next_full_show = str(ephem.localtime(moon_next_full))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		moon_next_last_quarter_show = str(ephem.localtime(moon_next_last_quarter))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]

        if hildon_flag == 1:
                window = hildon.Window()
        else:
                window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title('Moon Ephemeris: DETAIL')

	window.set_size_request(800,480)

        table = gtk.Table(2,6,False)
        table.set_col_spacings(1)
        table.set_row_spacings(1)

        row = 0
        column = 0

        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Next Moon Rise")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Next Moon Set")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Previous Moon Transit")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Previous Moon Anti-Transit")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Next Moon Transit")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Next Moon Anti-Transit")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Previous New Moon")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Previous First Quarter Moon")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Previous Full Moon")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Previous Last Quarter Moon")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Next New Moon")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Next First Quarter Moon")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Next Full Moon")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Next Last Quarter Moon")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Moon Libration Latitude")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Moon Libration Longitude")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Moon Selenographic Colongitude")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Moon subsolar latitude")

	column = column+1
	row = 0
	
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
       	set_event_box_color("clicked",event_box,"black","green",moon_next_rising_show)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
       	set_event_box_color("clicked",event_box,"black","green",moon_next_setting_show)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
       	set_event_box_color("clicked",event_box,"black","green",moon_previous_transit_show)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
       	set_event_box_color("clicked",event_box,"black","green",moon_previous_antitransit_show)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
       	set_event_box_color("clicked",event_box,"black","green",moon_next_transit_show)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
       	set_event_box_color("clicked",event_box,"black","green",moon_next_antitransit_show)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
       	set_event_box_color("clicked",event_box,"black","green",moon_previous_new_show)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",moon_previous_first_quarter_show)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",moon_previous_full_show)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",moon_previous_last_quarter_show)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",moon_next_new_show)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",moon_next_first_quarter_show)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",moon_next_full_show)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",moon_next_last_quarter_show)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",moon_libration_lat)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",moon_libration_long)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",moon_colong)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",moon_subsolar_lat)

	local.date = save #still required for 3.7.3.2

        table.show()
        window.add(table)

	if window_in_fullscreen == True:
		window.fullscreen()

        window.show()

	return

def show_time_info(widget,local):
	if hildon_flag == 1:
		window = hildon.Window()
	else:
		window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title('Earth Ephemeris: TIME DETAIL')

	window.set_size_request(800,480)

        table = gtk.Table(11,14,False)
	table.set_col_spacings(1)
	table.set_row_spacings(1)
	
	y = local.date.tuple()
        z = str(y[0]).zfill(2)+'/'+str(y[1]).zfill(2)+'/'+str(y[2]).zfill(2)

	gmt_date_time = ephem.Date(local.date)
	local_date_time = str(ephem.localtime(local.date))[:19]
	local_pretty_date_time = time.strftime("%Y/%m/%d %H:%M", time.localtime())
	local_time_zone_offset = time.altzone/60/60
	local_time_zone_hour = time.timezone/60/60
	local_time_zone_name = time.tzname
	local_sidereal_time = local.sidereal_time()
	local_julian_date = ephem.julian_date()
	local_year_day_num = time.localtime()[7]
	local_year_num = time.localtime()[0]
	local_month_num = time.localtime()[1]

	save = local.date #still required for 3.7.3.2
	object = ephem.Sun()
	object.compute(local)
	object_next_transit = local.next_transit(object)
	solar_time_zone = ((float(local.long)*180/pi)/15) - int((float(local.long)*180/pi)/15)
	transit_gmt = ephem.Date(object_next_transit).triple()[2] - int(ephem.Date(object_next_transit).triple()[2])
	transit_local = (transit_gmt*24) - time.timezone/60/60
	transit_mean = transit_local+solar_time_zone
	equation_of_time = ephem.hours("00:" + str((12-transit_mean)*60))
	local.date = save #still required for 3.7.3.2

	if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
		local_time_zone_daylight_flag = 0
	else:
		local_time_zone_daylight_flag = time.daylight

	row = 0
        column = 0

        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","GMT Date/Time")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Local Date/Time")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","True Local Date/Time")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Local Julian Date")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Local Timezone Name")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Local Timezone Hour")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Local Daylight Flag")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Local Sidereal Time")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Local Day of Year")

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","yellow","Equation of Time")

        row = 0
	column = column+1

        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",gmt_date_time)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",local_date_time)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",local_pretty_date_time)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",local_julian_date)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",local_time_zone_name)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",local_time_zone_hour)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",local_time_zone_daylight_flag)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",local_sidereal_time)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",local_year_day_num)

        row = row+1
        event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
        set_event_box_color("clicked",event_box,"black","green",equation_of_time)

        table.show()
        window.add(table)

	if window_in_fullscreen == True:
		window.fullscreen()

        window.show()

        return                                                            

def show_season_info(widget,local,now):
	global rise_set_plot_flag

	if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
		local_time_zone_daylight_flag = 0
	else:
		local_time_zone_daylight_flag = time.daylight

	local_time_zone_name = time.tzname

	object = ephem.Sun()

	spring = ephem.next_equinox(ephem.date(str(local.date.tuple()[0])))
        summer = ephem.next_solstice(spring)
        fall = ephem.next_equinox(summer)
        winter = ephem.next_solstice(fall)

	previous_solstice = ephem.previous_solstice(now)
	next_solstice = ephem.next_solstice(now)
	previous_equinox = ephem.previous_equinox(now)
	next_equinox = ephem.next_equinox(now)
	previous_vernal_equinox = ephem.previous_vernal_equinox(now)
	next_vernal_equinox = ephem.next_vernal_equinox(now)
	previous_autumnal_equinox = ephem.previous_autumnal_equinox(now)
	next_autumnal_equinox = ephem.next_autumnal_equinox(now)

	if date_in_gmt == True:
		spring_show = str(spring)[:19]+' GMT'
		summer_show = str(summer)[:19]+' GMT'
		fall_show = str(fall)[:19]+' GMT'
		winter_show = str(winter)[:19]+' GMT'

		previous_solstice_show = str(previous_solstice)[:19]+' GMT'
		next_solstice_show = str(next_solstice)[:19]+' GMT'
		previous_equinox_show = str(previous_equinox)[:19]+' GMT'
		next_equinox_show = str(next_equinox)[:19]+' GMT'
		previous_vernal_equinox_show = str(previous_vernal_equinox)[:19]+' GMT'
		next_vernal_equinox_show = str(next_vernal_equinox)[:19]+' GMT'
		previous_autumnal_equinox_show = str(previous_autumnal_equinox)[:19]+' GMT'
		next_autumnal_equinox_show = str(next_autumnal_equinox)[:19]+' GMT'
	else:
		spring_show = str(ephem.localtime(spring))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		summer_show = str(ephem.localtime(summer))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		fall_show = str(ephem.localtime(fall))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		winter_show = str(ephem.localtime(winter))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]

		previous_solstice_show = str(ephem.localtime(previous_solstice))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		next_solstice_show = str(ephem.localtime(next_solstice))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		previous_equinox_show = str(ephem.localtime(previous_equinox))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		next_equinox_show = str(ephem.localtime(next_equinox))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		previous_vernal_equinox_show = str(ephem.localtime(previous_vernal_equinox))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		next_vernal_equinox_show = str(ephem.localtime(next_vernal_equinox))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		previous_autumnal_equinox_show = str(ephem.localtime(previous_autumnal_equinox))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		next_autumnal_equinox_show = str(ephem.localtime(next_autumnal_equinox))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]

	if hildon_flag == 1:
		window = hildon.Window()
	else:
		window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title('Earth Ephemeris: SEASON DETAIL')

	window.set_size_request(800,480)

        table = gtk.Table(2,6,False)
	table.set_col_spacings(1)
	table.set_row_spacings(1)

	row = 0
	column = 0

	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Spring Equinox")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Summer Solstice")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Autumnal Equinox")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Winter Solstice")

	row = row+2
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Previous Solstice")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Previous Equinox")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Previous Vernal Equinox")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Previous Autumnal Equinox")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Next Solstice")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Next Equinox")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Next Vernal Equinox")

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","yellow","Next Autumnal Equinox")

	row = 0
	column = column+1 

	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",spring_show)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",summer_show)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",fall_show)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",winter_show)

	row = row+2
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",previous_solstice_show)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",previous_equinox_show)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",previous_vernal_equinox_show)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",previous_autumnal_equinox_show)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",next_solstice_show)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",next_equinox_show)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",next_vernal_equinox_show)

	row = row+1
	event_box = gtk.EventBox()
        table.attach(event_box,column,column+1,row,row+1)
	set_event_box_color("clicked",event_box,"black","green",next_autumnal_equinox_show)

	column = 0
	row = row+1
	button = gtk.Button("Sun Sky Angle")
	button.connect("clicked",plot_sun_angle_graph_start,object,local)
        table.attach(button,column,column+1,row,row+1)
	button.show()
	set_button_color("clicked",button,"black","black","black","light blue","green","red")

	rise_set_plot_flag = False
	column = column + 1
	button = gtk.Button("Sun Analemma")
	button.connect("clicked",plot_object_rise_set_graph_start,object,local)
        table.attach(button,column,column+1,row,row+1)
	button.show()
	set_button_color("clicked",button,"black","black","black","light blue","green","red")

        table.show()
        window.add(table)

	if window_in_fullscreen == True:
		window.fullscreen()

        window.show()
	
        return

def on_show_note(widget,window,message,choice):
	if hildon_flag == 1:
    		dialog = hildon.Note ("information", (window, message, gtk.STOCK_DIALOG_INFO) )
    		dialog.set_button_text(choice)
    		dialog.run()
    		dialog.destroy()
	else:
		dialog = gtk.MessageDialog(
    		parent         = None,
    		flags          = gtk.DIALOG_DESTROY_WITH_PARENT,
    		type           = gtk.MESSAGE_INFO,
    		buttons        = gtk.BUTTONS_OK,
    		message_format = message)
		dialog.set_title("Information")
		dialog.connect('response', lambda dialog, response: dialog.destroy())
		dialog.show()

	return

def toggle_chart_info(widget,window,table,vbox,local):
	global chart_info_toggle_flag
	if chart_info_toggle_flag == True:
		chart_info_toggle_flag = False
	else:
		chart_info_toggle_flag = True

	table.destroy()
	vbox.destroy()

	show_object_info_basic("clicked",window,local)

	return

def toggle_date_type(widget):
	global date_in_gmt
	if date_in_gmt == True:
		date_in_gmt = False
	else:
		date_in_gmt = True

	return

def toggle_date_style(widget,window,button,local):
	global date_in_gmt
	if date_in_gmt == True:
		date_in_gmt = False
	else:
		date_in_gmt = True

	current_local_date = ephem.Date(ephem.localtime(local.date))

	if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
		local_time_zone_daylight_flag = 0
	else:
		local_time_zone_daylight_flag = time.daylight

	local_time_zone_name = time.tzname
	
	if date_in_gmt == True:
		title = 'DATE/TIME: '+str(local.date)+' GMT LOCATION: '+str(local.lat)+u'\xb0 '+str(local.long)+u'\xb0'
	else:
		title = 'DATE/TIME: '+str(current_local_date)+' '+local_time_zone_name[local_time_zone_daylight_flag]+' LOCATION: '+str(local.lat)+u'\xb0 '+str(local.long)+u'\xb0'

	button.set_label(title)
	set_button_color("clicked",button,"black","black","black","orange","green","red")

	return

def get_web_image(widget,window,local,map_flag):

        if hildon_flag == 1:
                window = hildon.Window()
        else:
                window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title('Planet Ephemeris: WEB IMAGES')

	window.set_size_request(800,480)

	screen = window.get_screen()
	screen_width = screen.get_width()
	screen_height = screen.get_height()
	window_width = window.get_size()[0]
	window_height = window.get_size()[1]

        color_black = gtk.gdk.color_parse('black')
	window.modify_bg(gtk.STATE_NORMAL, color_black)

        if local.long < 0:
                east_west_direction = "West"
                longitude_unsigned_value = -local.long
        else:
                east_west_direction = "East"
                longitude_unsigned_value = local.long

        if local.lat < 0:
                north_south_direction = "South"
                latitude_unsigned_value = -local.lat
        else:
                north_south_direction = "North"
                latitude_unsigned_value = local.lat

	screen = window.get_screen()
	screen_width = screen.get_width()
	screen_height = screen.get_height()

	current_date_gmt = ephem.Date(local.date).tuple()

        local_date_time = str(local.date.tuple()[0])+"-"+str(local.date.tuple()[1])+"-"+str(local.date.tuple()[2])+"+"+str(local.date.tuple()[3])+":"+str(local.date.tuple()[4])+":"+str(int(local.date.tuple()[5]))
        local_date_time2 = str(local.date.tuple()[0])+str(local.date.tuple()[1]).zfill(2)+str(local.date.tuple()[2]).zfill(2)

	if (map_flag == 1):
		if window_in_fullscreen == True:
			url = 'http://www.fourmilab.ch/cgi-bin/Earth?img=nasa.evif&imgsize=' + str(screen_width) + '&dynimg=y&opt=-l&lat='
		else:
			url = 'http://www.fourmilab.ch/cgi-bin/Earth?img=nasa.evif&imgsize=422&dynimg=y&opt=-l&lat='
		url_show = 'http://www.fourmilab.ch/cgi-bin/Earth...'
        	url = url+str(latitude_unsigned_value)+"&ns="+north_south_direction+"&lon="+str(longitude_unsigned_value)+"&ew="+east_west_direction+"&alt=35785&tle=&date=1&utc="+local_date_time+"&jd="
	elif (map_flag == 2):
        	url = 'http://www.fourmilab.ch/cgi-bin/Earth?img=wx-cmap.bmp&imgsize=422&dynimg=y&opt=-l&lat=' # weath globe
        	url = url+str(latitude_unsigned_value)+"&ns="+north_south_direction+"&lon="+str(longitude_unsigned_value)+"&ew="+east_west_direction+"&alt=35785&tle=&date=0&utc=&jd="
		url_show = 'http://www.fourmilab.ch/cgi-bin/Earth...'
	elif (map_flag == 3):
		#url = 'http://www.fourmilab.ch/cgi-bin/Earth?img=nasa.evif&imgsize=422&dynimg=y&opt=-p&lat=&lon=&alt=&tle=&date=1&utc='+local_date_time+'&jd='
		if window_in_fullscreen == True:
			url = 'http://www.fourmilab.ch/cgi-bin/Earth?img=nasa.evif&imgsize=' + str(screen_width) + '&dynimg=y&opt=-p&lat=&lon=&alt=&tle=&date=1&utc='+local_date_time+'&jd='
		else:
			url = 'http://www.fourmilab.ch/cgi-bin/Earth?img=nasa.evif&imgsize=422&dynimg=y&opt=-p&lat=&lon=&alt=&tle=&date=1&utc='+local_date_time+'&jd='
		url_show = 'http://www.fourmilab.ch/cgi-bin/Earth...'
	elif (map_flag == 4):
		url = 'http://www.fourmilab.ch/cgi-bin/Earth?img=wx-cmap.bmp&imgsize=336&dynimg=y&opt=-p&lat=&lon=&alt=&tle=&date=0&utc=&jd=' # weath flat
		url_show = 'http://www.fourmilab.ch/cgi-bin/Earth...'
	elif (map_flag == 5):
		url = 'http://space.jpl.nasa.gov/cgi-bin/wspace?tbody=1000&vbody=1001'
		url = url+'&month='+str(current_date_gmt[1])+'&day='+str(current_date_gmt[2])+'&year='+str(current_date_gmt[0])+'&hour='+str(current_date_gmt[3])+'&minute='+str(current_date_gmt[4])+'&fovmul=1&rfov=2&bfov=100&porbs=1&brite=1'
		url_show = 'http://space.jpl.nasa.gov/cgi-bin/wspace...'
	elif (map_flag == 6):
		url = 'http://space.jpl.nasa.gov/cgi-bin/wspace?tbody=1000&vbody=1001'
		url = url+'&month='+str(current_date_gmt[1])+'&day='+str(current_date_gmt[2])+'&year='+str(current_date_gmt[0])+'&hour='+str(current_date_gmt[3])+'&minute='+str(current_date_gmt[4])+'&rfov=2&fovmul=-1&bfov=50&porbs=1&brite=1'
		url_show = 'http://space.jpl.nasa.gov/cgi-bin/wspace...'
	elif (map_flag == 7):
		url = 'http://www.fourmilab.ch/cgi-bin/Yoursky?utc='+local_date_time+'&date=2&jd=&lat='
		if window_in_fullscreen == True:
			url = url+str(latitude_unsigned_value)+"&ns="+north_south_direction+"&lon="+str(longitude_unsigned_value)+"&ew="+east_west_direction+'&coords=on&moonp=on&deepm=2.5&consto=on&limag=4&starnm=2.0&starbm=2.5&imgsize=' + str(screen_width) + '&dynimg=y&scheme=0&elements='
		else:
			url = url+str(latitude_unsigned_value)+"&ns="+north_south_direction+"&lon="+str(longitude_unsigned_value)+"&ew="+east_west_direction+'&coords=on&moonp=on&deepm=2.5&consto=on&limag=4&starnm=2.0&starbm=2.5&imgsize=422&dynimg=y&scheme=0&elements='
		url_show = 'http://www.fourmilab.ch/cgi-bin/Yoursky...'
	elif (map_flag == 8):
		if window_in_fullscreen == True:
			url = 'http://www.fourmilab.ch/cgi-bin/Earth?img=MoonTopo.evif&imgsize=' + str(screen_width) + '&dynimg=y&opt=-m&lat=&lon=&alt=&tle=&date=1&utc='+local_date_time+'&jd='
		else:
			url = 'http://www.fourmilab.ch/cgi-bin/Earth?img=MoonTopo.evif&imgsize=422&dynimg=y&opt=-m&lat=&lon=&alt=&tle=&date=1&utc='+local_date_time+'&jd='
		url_show = 'http://www.fourmilab.ch/cgi-bin/Earth...'
	elif (map_flag == 9):
		url = 'http://sohowww.nascom.nasa.gov/data/synoptic/sunspots_earth/sunspots_512_'+local_date_time2+'.jpg'
		url_show = 'http://sohowww.nascom.nasa.gov/data/synoptic/sunspots_earth/sunspots_512_...'
	elif (map_flag == 10):
		url = 'http://www.heavens-above.com/orbitdisplay.aspx?icon=iss&width=600&height=600&satid='+tle_number
		url_show = 'http://www.heavens-above.com/orbitdisplay.aspx...'
	
        try:
                file = urllib.urlopen(url)
		print "url=",url
                weather_pic = file.read()
                run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
                data_dir = os.path.join(run_dir, 'data')
                image_file = os.path.join(data_dir, 'sat.gif')
                temp_file = open(image_file,"wb")
                temp_file.write(weather_pic)
                temp_file.close()
		print "image_file=",image_file
        except Exception:
                print "web connection timeout"

        run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
        data_dir = os.path.join(run_dir, 'data')
        image_file = os.path.join(data_dir, 'sat.gif')
        image = gtk.Image()
        image.set_from_file(image_file)

	pixbuf = gtk.gdk.pixbuf_new_from_file(image_file)

	aspect = True
	enlarge = True

	display_image(image,pixbuf,aspect,enlarge,screen_width,screen_height,window_width,window_height)

	url_show = url_show + '[dynamic link]'

	table = gtk.Table(1,1, False)
	window.add(table)
	table.attach(image,0,1,0,1)

	event_box = gtk.EventBox()
	table.attach(event_box,0,1,1,2)
	set_event_box_color("clicked",event_box,"black","yellow",url_show)
	table.show()

        image.show()

	if window_in_fullscreen == True:
		window.fullscreen()

	window.show()

	return

def get_google_map_start(widget,local,weather):
        if hildon_flag == 1:
                window = hildon.Window()
        else:
                window = gtk.Window()

        image = gtk.Image()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_google, window, image, local)

	window.set_title('Planet Ephemeris: WEB GOOGLE MAPS')

	window.set_size_request(800,480)

       	get_google_map("clicked",window,image,local,local)

	return

def get_google_map(widget,window,image,local,weather):
	global google_view
	global google_map_type

	screen = window.get_screen()
	screen_width = screen.get_width()
	screen_height = screen.get_height()
	window_width = window.get_size()[0]
	window_height = window.get_size()[1]

        color_black = gtk.gdk.color_parse('black')
	window.modify_bg(gtk.STATE_NORMAL, color_black)

	if local.lat == weather.lat:
		url = 'http://maps.google.com/staticmap?center='+str(local.lat*180/pi)+','+str(local.long*180/pi)+'&markers='+str(local.lat*180/pi)+','+str(local.long*180/pi)+',midblue&size=600x400&span='+str(google_view)+','+str(google_view)+'&maptype='+google_map_type
	else:
		url = 'http://maps.google.com/staticmap?&markers='+str(local.lat*180/pi)+','+str(local.long*180/pi)+',midblue1|'+str(weather.lat*180/pi)+','+str(weather.long*180/pi)+',midyellow2&size=600x400&span='+str(google_view)+','+str(google_view)+'&maptype='+google_map_type

        try:
                file = urllib.urlopen(url)
		print "url=",url
                google_map = file.read()
                run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
                data_dir = os.path.join(run_dir, 'data')
                image_file = os.path.join(data_dir, 'google.gif')
                temp_file = open(image_file,"wb")
                temp_file.write(google_map)
                temp_file.close()
		print "image_file=",image_file
        except Exception:
                print "web connection timeout"

        run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
        data_dir = os.path.join(run_dir, 'data')
        image_file = os.path.join(data_dir, 'google.gif')
        image.set_from_file(image_file)

	pixbuf = gtk.gdk.pixbuf_new_from_file(image_file)

	aspect = True
	enlarge = True

	display_image(image,pixbuf,aspect,enlarge,screen_width,screen_height,window_width,window_height)

        image.show()
	window.add(image)

	if window_in_fullscreen == True:
		window.fullscreen()

	window.show()

	return

def get_apod_image(widget,window,local,map_flag):
        if hildon_flag == 1:
                window = hildon.Window()
        else:
                window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title('Planet Ephemeris: WEB APOD IMAGE')

	window.set_size_request(800,480)

	screen = window.get_screen()
	screen_width = screen.get_width()
	screen_height = screen.get_height()
	window_width = window.get_size()[0]
	window_height = window.get_size()[1]

        color_black = gtk.gdk.color_parse('black')
	window.modify_bg(gtk.STATE_NORMAL, color_black)

        run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
        data_dir = os.path.join(run_dir, 'data')
        image_file = os.path.join(data_dir, 'apod.jpg')
        run_file = os.path.join(data_dir, 'apod.sh')

	y = local.date.tuple()
	apod_date = str(y[0])[2:]+str(y[1]).zfill(2)+str(y[2]).zfill(2)

	if (local.date.tuple()[0] >= time.localtime()[0]) and (local.date.tuple()[1] >= time.localtime()[1]) and (local.date.tuple()[2] >= time.localtime()[2]):
		apod_main_url = "http://antwrp.gsfc.nasa.gov/apod/"
		apod_pic_page = "astropix.html"
	else:
		apod_main_url = "http://antwrp.gsfc.nasa.gov/apod/"
		apod_pic_page = "ap"+apod_date+".html"

	for line in urllib.urlopen(apod_main_url+apod_pic_page).readlines():
	  	if line.lower().find('img src=') >= 0:
	    		pos1 = line.find('"') + 1
	    		pos2 = line.find('"', pos1)
	    		pic_url = apod_main_url+line[pos1:pos2]
	    		break

	# retrieve it and save to the cache directory
	apod_pic_file = data_dir+"/"+os.path.split(pic_url)[1]
	urllib.urlretrieve(pic_url, apod_pic_file)
	print "apod_pic_file=",apod_pic_file
	os.system("mv "+apod_pic_file+" "+data_dir+"/apod.jpg")

        image = gtk.Image()
        image.set_from_file(image_file)

	pixbuf = gtk.gdk.pixbuf_new_from_file(image_file)

	aspect = True
	enlarge = True

	display_image(image,pixbuf,aspect,enlarge,screen_width,screen_height,window_width,window_height)

	table = gtk.Table(1,1, False)
	window.add(table)
	table.attach(image,0,1,0,1)

	event_box = gtk.EventBox()
	table.attach(event_box,0,1,1,2)
	set_event_box_color("clicked",event_box,"black","yellow",pic_url)
	table.show()

        image.show()

	if window_in_fullscreen == True:
		window.fullscreen()

	window.show()

	return

def get_geonames_elevation(widget,local,elevation_show_flag):

        url = 'http://ws.geonames.org/gtopo30?lat='+str(local.lat*180/pi)+'&lng='+str(local.long*180/pi) 

        file = urllib.urlopen(url)
        print "url=",url
        elevation_value = file.read()
        if '\n' in elevation_value:
                elevation_value = elevation_value[:-1]
        if '\r' in elevation_value:
                elevation_value = elevation_value[:-1]

	if elevation_show_flag == True:
		if hildon_flag == 1:
			window = hildon.Window()
		else:
			window = gtk.Window()

		window.connect("window-state-event", on_window_state_change, window)
		window.connect("key-press-event", on_key_press_basic, window)

		window.set_title('Earth Ephemeris: LOCAL ELEVATION')

		window.set_size_request(800,480)

		location = ephem.Observer()
		location.lat = local.lat
		location.long = local.long
		location.elev = float(elevation_value)

		table = gtk.Table(11,14,False)
		table.set_col_spacings(1)
		table.set_row_spacings(1)

		row = 0
		column = 0

		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Local Latitude")

        	row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","yellow","Local Longitude")

		row = row+1
		button = gtk.Button("Local Elevation")
		button.connect("clicked",change_elevation,window,local,location)
		table.attach(button,column,column+1,row,row+1)
		button.show()                            
		set_button_color("clicked",button,"black","black","black","red","green","red")

		row = 0
		column = column+1

		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",local.lat)

		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",local.long)

		row = row+1
		event_box = gtk.EventBox()
		table.attach(event_box,column,column+1,row,row+1)
		set_event_box_color("clicked",event_box,"black","green",elevation_value)

		table.show()
		window.add(table)

		if window_in_fullscreen == True:
			window.fullscreen()

		window.show()

	return(elevation_value)

def change_elevation(widget,window,local,weather):

	local.elevation = weather.elevation
	on_show_note(widget,window,"Local elevation reset to "+str(local.elevation)+" m","OK")

	return

def change_temperature(widget,window,local,weather):

	local.temp = weather.temp
	on_show_note(widget,window,"Local temperature reset to "+str(local.temp)+u"\xb0"+"C","OK")

	return

def change_pressure(widget,window,local,weather):
	global ini_treestore

	local.pressure = weather.pressure
	on_show_note(widget,window,"Local pressure reset to "+str(local.pressure)+" mb","OK")

	iter = ini_treestore.get_iter_root()
	path = ini_treestore.get_path(iter)
	while iter:
		ini_name = ini_treestore.get_value(iter, 0)
		ini_value = ini_treestore.get_value(iter, 1)
		if ini_name == "pressure:":
			print "changing!"
			ini_treestore.set_value(iter, 1, local.pressure)
		iter = ini_treestore.iter_next(iter)

	return

def plot_solar_system_position_start(widget,object,local):
	global asteroid_show_flag
	global comet_show_flag
	global neo_show_flag
	global planet_trail

        if hildon_flag == 1:
                window = hildon.Window()
        else:
                window = gtk.Window()

	vbox = gtk.HBox(False, 0)

	planet_trail = []

	window.connect("window-state-event", on_window_state_change, window)
        window.connect("key-press-event", on_key_press_plot_solar_system, window, vbox, object, local)

	window.set_title('Planet Ephemeris for Sun: SOLAR SYSTEM')

	window.set_size_request(800,480)

	plot_solar_system_position("clicked",window,vbox,object,local)

	return

def plot_solar_system_orbit_start(widget,object,local):
	global asteroid_show_flag
	global comet_show_flag
	global neo_show_flag
	global asteroid_read_flag
	global comet_read_flag
	global neo_read_flag

        if hildon_flag == 1:
                window = hildon.Window()
        else:
                window = gtk.Window()

	vbox = gtk.HBox(False, 0)

	planet_orbit = ([])

	window.connect("window-state-event", on_window_state_change, window)
        window.connect("key-press-event", on_key_press_plot_solar_orbit, window, vbox, object, local, planet_orbit)

	window.set_title('Planet Ephemeris for Sun: SOLAR SYSTEM')

	window.set_size_request(800,480)

	if hildon_flag == True:
		on_show_note(widget,window,"Please wait while lengthy solar system calculations are made","OK")

	save = local.date #still required for 3.7.3.2

	planet = ephem.Mercury()
	planet_interval = 87.969/orbit_resolution
	for degrees in range(0,orbit_resolution,1):
		planet.compute(local)
		long = planet.hlong
		lat = (pi*0.5)-planet.hlat
		radius = planet.sun_distance
		planet_x = radius * cos(long) * sin(lat)
		planet_y = radius * sin(long) * sin(lat)
		planet_z = radius * cos(lat)

		planet_orbit.append([planet.name,planet_x,planet_y, planet_z])

		local.date = local.date + planet_interval

	local.date = ephem.now()

	planet = ephem.Venus()
	planet_interval = 224.70069/orbit_resolution
	for degrees in range(0,orbit_resolution,1):
		planet.compute(local)
		long = planet.hlong
		lat = planet.hlat
		lat = (pi*0.5)-planet.hlat
		radius = planet.sun_distance
		planet_x = radius * cos(long) * sin(lat)
		planet_y = radius * sin(long) * sin(lat)
		planet_z = radius * cos(lat)

		planet_orbit.append([planet.name,planet_x,planet_y, planet_z])

		local.date = local.date + planet_interval

	local.date = ephem.now()

	planet = ephem.Sun()
	planet_interval = 365.256366/orbit_resolution
	for degrees in range(0,orbit_resolution,1):
		planet.compute(local)
		long = planet.hlong
		lat = planet.hlat
		lat = (pi*0.5)-planet.hlat
		radius = planet.earth_distance
		planet_x = radius * cos(long) * sin(lat)
		planet_y = radius * sin(long) * sin(lat)
		planet_z = radius * cos(lat)

		planet_orbit.append(["Earth",planet_x,planet_y, planet_z])

		local.date = local.date + planet_interval

	local.date = ephem.now()

	planet = ephem.Mars()
	planet_interval = 686.971/orbit_resolution
	for degrees in range(0,orbit_resolution,1):
		planet.compute(local)
		long = planet.hlong
		lat = planet.hlat
		lat = (pi*0.5)-planet.hlat
		radius = planet.sun_distance
		planet_x = radius * cos(long) * sin(lat)
		planet_y = radius * sin(long) * sin(lat)
		planet_z = radius * cos(lat)

		planet_orbit.append([planet.name,planet_x,planet_y, planet_z])

		local.date = local.date + planet_interval

	local.date = ephem.now()

	planet = ephem.Jupiter()
	planet_interval = 4331.572/orbit_resolution
	for degrees in range(0,orbit_resolution,1):
		planet.compute(local)
		long = planet.hlong
		lat = planet.hlat
		lat = (pi*0.5)-planet.hlat
		radius = planet.sun_distance
		planet_x = radius * cos(long) * sin(lat)
		planet_y = radius * sin(long) * sin(lat)
		planet_z = radius * cos(lat)

		planet_orbit.append([planet.name,planet_x,planet_y, planet_z])

		local.date = local.date + planet_interval

	local.date = ephem.now()

	planet = ephem.Saturn()
	planet_interval = 10832.327/orbit_resolution
	for degrees in range(0,orbit_resolution,1):
		planet.compute(local)
		long = planet.hlong
		lat = planet.hlat
		lat = (pi*0.5)-planet.hlat
		radius = planet.sun_distance
		planet_x = radius * cos(long) * sin(lat)
		planet_y = radius * sin(long) * sin(lat)
		planet_z = radius * cos(lat)

		planet_orbit.append([planet.name,planet_x,planet_y, planet_z])

		local.date = local.date + planet_interval

	local.date = ephem.now()

	planet = ephem.Uranus()
	planet_interval = 30799.095/orbit_resolution
	for degrees in range(0,orbit_resolution,1):
		planet.compute(local)
		long = planet.hlong
		lat = planet.hlat
		lat = (pi*0.5)-planet.hlat
		radius = planet.sun_distance
		planet_x = radius * cos(long) * sin(lat)
		planet_y = radius * sin(long) * sin(lat)
		planet_z = radius * cos(lat)

		planet_orbit.append([planet.name,planet_x,planet_y, planet_z])

		local.date = local.date + planet_interval

	local.date = ephem.now()

	planet = ephem.Neptune()
	planet_interval = 60190/orbit_resolution
	for degrees in range(0,orbit_resolution,1):
		planet.compute(local)
		long = planet.hlong
		lat = planet.hlat
		lat = (pi*0.5)-planet.hlat
		radius = planet.sun_distance
		planet_x = radius * cos(long) * sin(lat)
		planet_y = radius * sin(long) * sin(lat)
		planet_z = radius * cos(lat)

		planet_orbit.append([planet.name,planet_x,planet_y, planet_z])

		local.date = local.date + planet_interval

	local.date = ephem.now()

	planet = ephem.Pluto()
	planet_interval = 90613.3055/orbit_resolution
	for degrees in range(0,orbit_resolution,1):
		planet.compute(local)
		long = planet.hlong
		lat = (pi*0.5)-planet.hlat
		radius = planet.sun_distance
		planet_x = radius * cos(long) * sin(lat)
		planet_y = radius * sin(long) * sin(lat)
		planet_z = radius * cos(lat)

		planet_orbit.append([planet.name,planet_x,planet_y, planet_z])

		local.date = local.date + planet_interval

	local.date = ephem.now()

	if asteroid_show_flag == True:
		run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
		data_dir = os.path.join(run_dir, 'data')
		asteroid_file = os.path.join(data_dir, 'asteroid.dat')
		try:
			f = open(asteroid_file,"rb")
		except:
			on_show_note("clicked",window,"Asteroid data file missing -- download it from web window","OK")
			print "Asteroid data (satellite elements) are missing!  Ignoring request!"
			asteroid_show_flag = False
		else:
			while 1:
				t = f.readline()
				if t == '':
					break
				if '\n' in t:
					t = t[:-1]
				if '\r' in t:
					t = t[:-1]
				a = t.split()
				line_type = a[0]
				if (line_type != "#"):
					b = t.split(",")
					c = b[0].split()
					asteroid_name = c[1]
					line1 = t
					t = f.readline()
					if t == '':
						break
					if '\n' in t:
						t = t[:-1]
					if '\r' in t:
						t = t[:-1]
					asteroid = ephem.readdb(line1)
					asteroid.compute(local)
					long = asteroid.hlong
					lat = (pi*0.5)-asteroid.hlat
					radius = asteroid.sun_distance

					if asteroid_read_flag == False:
						piter = asteroid_treestore.append(None, [asteroid_name, asteroid.az, asteroid.alt, asteroid.ra, asteroid.dec, float(asteroid.az), float(asteroid.alt), float(asteroid.ra), float(asteroid.dec)])

					asteroid_x = radius * cos(long) * sin(lat)
					asteroid_y = radius * sin(long) * sin(lat)
					asteroid_z = radius * cos(lat)

					planet_orbit.append(["ASTEROID" + asteroid_name,asteroid_x,asteroid_y, asteroid_z])

			asteroid_read_flag = True
			f.close()

	if comet_show_flag == True:
		run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
		data_dir = os.path.join(run_dir, 'data')
		comet_file = os.path.join(data_dir, 'comet.dat')
		try:
			f = open(comet_file,"rb")
		except:
			on_show_note("clicked",window,"Comet data file missing -- download it from web window","OK")
			print "Comet data (satellite elements) are missing!  Ignoring request!"
			comet_show_flag = False
		else:
			while 1:
				t = f.readline()
				if t == '':
					break
				if '\n' in t:
					t = t[:-1]
				if '\r' in t:
					t = t[:-1]
				a = t.split()
				line_type = a[0]
				if (line_type != "#"):
					b = t.split(",")
					c = b[0].split("/")
					comet_name = c[1]
					line1 = t
					t = f.readline()
					if t == '':
						break
					if '\n' in t:
						t = t[:-1]
					if '\r' in t:
						t = t[:-1]
					comet = ephem.readdb(line1)
					comet.compute(local)
					long = comet.hlong
					lat = (pi*0.5)-comet.hlat
					radius = comet.sun_distance

					if comet_read_flag == False:
						piter = comet_treestore.append(None, [comet_name, comet.az, comet.alt, comet.ra, comet.dec, float(comet.az), float(comet.alt), float(comet.ra), float(comet.dec)])

					comet_x = radius * cos(long) * sin(lat)
					comet_y = radius * sin(long) * sin(lat)
					comet_z = radius * cos(lat)

					planet_orbit.append(["COMET" + comet_name, comet_x, comet_y, comet_z])

			comet_read_flag = True
			f.close()

	if neo_show_flag == True:
		run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
		data_dir = os.path.join(run_dir, 'data')
		neo_file = os.path.join(data_dir, 'neo.dat')
		try:
			f = open(neo_file,"rb")
		except:
			on_show_note("clicked",window,"Comet data file missing -- download it from web window","OK")
			print "Comet data (satellite elements) are missing!  Ignoring request!"
			neo_show_flag = False
		else:
			while 1:
				t = f.readline()
				if t == '':
					break
				if '\n' in t:
					t = t[:-1]
				if '\r' in t:
					t = t[:-1]
				a = t.split()
				line_type = a[0]
				if (line_type != "#"):
					b = t.split(",")
					c = b[0].split()
					neo_name = c[1]
					line1 = t
					t = f.readline()
					if t == '':
						break
					if '\n' in t:
						t = t[:-1]
					if '\r' in t:
						t = t[:-1]
					neo = ephem.readdb(line1)
					neo.compute(local)
					long = neo.hlong
					lat = (pi*0.5)-neo.hlat
					radius = neo.sun_distance
					neo_x = radius * cos(long) * sin(lat)
					neo_y = radius * sin(long) * sin(lat)
					neo_z = radius * cos(lat)

					planet_orbit.append(["NEO" + neo_name, neo_x, neo_y, neo_z])

					if neo_read_flag == False:
						piter = neo_treestore.append(None, [neo_name, neo.az, neo.alt, neo.ra, neo.dec, float(neo.az), float(neo.alt), float(neo.ra), float(neo.dec)])

			neo_read_flag = True
			f.close()

	local.date = save #still required for 3.7.3.2

	plot_solar_system_orbit("clicked",window,vbox,object,local,planet_orbit)

	return

class plot_solar_system_position:
    	def asteroid_show_callback(self, widget):
		global asteroid_show_flag 
		if widget.get_active():
			asteroid_show_flag = True
		else:
			asteroid_show_flag = False

    	def asteroid_name_show_callback(self, widget):
		global asteroid_name_show_flag 
		if widget.get_active():
			asteroid_name_show_flag = True
		else:
			asteroid_name_show_flag = False

    	def comet_show_callback(self, widget):
		global comet_show_flag 
		if widget.get_active():
			comet_show_flag = True
		else:
			comet_show_flag = False

    	def comet_name_show_callback(self, widget):
		global comet_name_show_flag 
		if widget.get_active():
			comet_name_show_flag = True
		else:
			comet_name_show_flag = False

    	def neo_show_callback(self, widget):
		global neo_show_flag 
		if widget.get_active():
			neo_show_flag = True
		else:
			neo_show_flag = False

    	def neo_name_show_callback(self, widget):
		global neo_name_show_flag 
		if widget.get_active():
			neo_name_show_flag = True
		else:
			neo_name_show_flag = False

        def asteroid_search_callback(self, widget):
		global object_search_flag
		global asteroid_treestore

		print len(asteroid_treestore)
		if asteroid_show_flag == True:
			if len(asteroid_treestore) == 0:
				print "ERROR: No asteroid objects have been plotted yet!"
				object_search_flag = False
			else:
				object_search_flag = True
				object_search("Asteroid")

        def comet_search_callback(self, widget):
		global object_search_flag
		global comet_treestore

		if comet_show_flag == True:
			if len(comet_treestore) == 0:
				print "ERROR: No comet objects have been plotted yet!"
				object_search_flag = False
			else:
				object_search_flag = True
				object_search("Comet")

        def neo_search_callback(self, widget):
		global object_search_flag
		global neo_treestore

		if neo_show_flag == True:
			if len(neo_treestore) == 0:
				print "ERROR: No NEO objects have been plotted yet!"
				object_search_flag = False
			else:
				object_search_flag = True
				object_search("NEO")

    	def planet_trail_show_callback(self, widget):
		global planet_trail
		global planet_trail_show_flag 
		if widget.get_active():
			planet_trail_show_flag = True
		else:
			planet_trail_show_flag = False
			planet_trail = []

    	def legend_show_callback(self, widget):
		global legend_show_flag 
		if widget.get_active():
			legend_show_flag = True
		else:
			legend_show_flag = False

	def __init__(self,widget,window,vbox,object,local):
		table = gtk.Table(3,2,False)
		table.set_col_spacings(0)
		table.set_row_spacings(1)

		table1 = gtk.Table(1,1,False)
		table1.set_col_spacings(1)
		table1.set_row_spacings(1)

		table3 = gtk.Table(1,1,False)
		table3.set_col_spacings(1)
		table3.set_row_spacings(1)

		plot = plot_solar_system(window,object,local)

		vbox.pack_start(plot,True)
		plot.show()

		if controls_show_flag == True:

			if date_time_controls_show_flag == True:
				column = 0
				row = 0

				if verbose_show_flag == True:
					button = gtk.Button("Column Remove")
				else:
					button = gtk.Button("Remove")
				date_time_controls_show_flag2 = 0	
				button.connect("clicked",toggle_solar_system_plot_date_time_controls,window,vbox,object,local,date_time_controls_show_flag2)
				table1.attach(button,column,column+3,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Year")
				else:
					button = gtk.Button("Yr")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -5
				button = gtk.Button("-")
				button.connect("clicked",change_date_solar_system_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 5
				button = gtk.Button("+")
				button.connect("clicked",change_date_solar_system_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Month")
				else:
					button = gtk.Button("Mo")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -4
				button = gtk.Button("-")
				button.connect("clicked",change_date_solar_system_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 4
				button = gtk.Button("+")
				button.connect("clicked",change_date_solar_system_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Day")
				else:
					button = gtk.Button("Dy")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -3
				button = gtk.Button("-")
				button.connect("clicked",change_date_solar_system_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 3
				button = gtk.Button("+")
				button.connect("clicked",change_date_solar_system_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				date_time_change_flag = -2
				if verbose_show_flag == True:
					button = gtk.Button("Hour")
				else:
					button = gtk.Button("Hr")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -2
				button = gtk.Button("-")
				button.connect("clicked",change_date_solar_system_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 2
				button = gtk.Button("+")
				button.connect("clicked",change_date_solar_system_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Minute")
				else:
					button = gtk.Button("Mi")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -1
				button = gtk.Button("-")
				button.connect("clicked",change_date_solar_system_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 1
				button = gtk.Button("+")
				button.connect("clicked",change_date_solar_system_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				date_time_change_flag = 0
				if verbose_show_flag == True:
					button = gtk.Button('Reset Date/Time')
				else:
					button = gtk.Button('Reset')
				button.connect("clicked",change_date_solar_system_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+3,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Text Size")
				else:
					button = gtk.Button("Ts")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				button = gtk.Button("-")
				text_size_change_flag = -1
				button.connect("clicked",set_font_size,text_size_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = column+1
				button = gtk.Button("+")
				text_size_change_flag = 1
				button.connect("clicked",set_font_size,text_size_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Asteroids")
				else:
					button = gtk.ToggleButton("As")
				button.set_active(asteroid_show_flag)
				button.connect("toggled", self.asteroid_show_callback)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Names")
				else:
					button = gtk.ToggleButton("AsN")
				button.set_active(asteroid_name_show_flag)
				button.connect("toggled", self.asteroid_name_show_callback)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.Button("Search")
				else:
					button = gtk.Button("AsS")
				button.connect("clicked", self.asteroid_search_callback)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Comets")
				else:
					button = gtk.ToggleButton("Cm")
				button.set_active(comet_show_flag)
				button.connect("toggled", self.comet_show_callback)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Names")
				else:
					button = gtk.ToggleButton("CmN")
				button.set_active(comet_name_show_flag)
				button.connect("toggled", self.comet_name_show_callback)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.Button("Search")
				else:
					button = gtk.Button("CoS")
				button.connect("clicked", self.comet_search_callback)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("NEO")
				else:
					button = gtk.ToggleButton("Ne")
				button.set_active(neo_show_flag)
				button.connect("toggled", self.neo_show_callback)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Names")
				else:
					button = gtk.ToggleButton("NeN")
				button.set_active(neo_name_show_flag)
				button.connect("toggled", self.neo_name_show_callback)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.Button("Search")
				else:
					button = gtk.Button("NeS")
				button.connect("clicked", self.neo_search_callback)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				row = 0
				column = column + 1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Planet Trails")
				else:
					button = gtk.ToggleButton("Pt")
				button.set_active(planet_trail_show_flag)
				button.connect("toggled", self.planet_trail_show_callback)
				table3.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")
				column = 0

				column = 0
				row = row + 1
				column = column + 1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Legend Text")
				else:
					button = gtk.ToggleButton("LgT")
				button.set_active(legend_show_flag)
				button.connect("toggled", self.legend_show_callback)
				table3.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")
				column = 0

				column = 0
				row = row + 1
				column = column + 1
				if verbose_show_flag == True:
					button = gtk.Button("Orbit Plot")
				else:
					button = gtk.Button("Op")
				button.connect("clicked",plot_solar_system_orbit_start,object,local)
				table3.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = 0

				column = 1
				table.attach(table1,column,column+1,0,6)
				table1.show()

				column = 1
				table.attach(table3,column,column+1,6,7)
				table3.show()

		window.add(vbox)
		vbox.pack_start(table,False)

		window.show_all()

		if window_in_fullscreen == True:
			window.fullscreen()

		window.show()

		return

def plot_solar_system_orbit(widget,window,vbox,object,local,planet_orbit):
	plot = plot_solar_orbit(window,object,local,planet_orbit)
	vbox.pack_start(plot,True)
	plot.show()

	vbox.show()
	window.add(vbox)
	window.show()

        if window_in_fullscreen == True:
                window.fullscreen()

	return

class plot_solar_system(gtk.DrawingArea):
        def __init__(self,window,object,local):
                gtk.DrawingArea.__init__(self)
                self.connect("expose_event", self.expose_plot_solar_system, window, object, local)
                self.connect('button_press_event', self.button_press_event_cb)
                self.set_events(gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.POINTER_MOTION_MASK)

                return

    	def button_press_event_cb(self, widget, event):
		global select_object

        	if event.button == 1:
			select_object = ""
					
			gtk.gdk.threads_enter
    			self.queue_draw()
			gtk.gdk.threads_leave

        	elif event.button == 3:
        		print event.x, event.y

        	return True

        def expose_plot_solar_system(self, widget, event, window, object, local):
                self.context = widget.window.cairo_create()

                # set a clip region for the expose event
                self.context.rectangle(event.area.x, event.area.y,
                               event.area.width, event.area.height)
                self.context.clip()

                self.draw_plot_solar_system(self.context, window, object, local)

                return False

	def draw_plot_solar_system_planet(self, context, local):
		global rect
		global planet_trail
		global rotate_angle
		global rotate_angle_x
		global rotate_angle_y
		global rotate_axis

		self.planet.compute(local)
		long = self.planet.hlong
		lat = (pi*0.5)-self.planet.hlat
		if self.planet.name == "Sun":
			radius = self.planet.earth_distance
		else:
			radius = self.planet.sun_distance
		self.planet_x = radius * cos(long) * sin(lat)
		self.planet_y = radius * sin(long) * sin(lat)
		self.planet_z = radius * cos(lat)

		x = ((rect.width/2) + (rect.height/self.sun_field_of_view * -self.planet_x) * planet_field_of_view)
		y = ((rect.height/2) - (rect.height/self.sun_field_of_view * -self.planet_y) * planet_field_of_view)
		
		if planet_trail_show_flag == True:
			planet_trail.append([self.planet_x, self.planet_y])

		context.set_source_rgb(self.plot_color[0],self.plot_color[1],self.plot_color[2]) # planet plot color
		context.arc(x, y, self.size*4, 0, 2 * math.pi)
		context.fill()
		context.stroke()

		context.set_source_rgb(1,1,1) # planet name color
		if object_show_name_flag == True:
			context.move_to(x-5, y+15)
			if self.planet.name == "Sun":
				context.text_path("Earth")
			else:
				context.text_path(self.planet.name)
			context.set_font_size(font_size)
			context.fill()
			context.stroke()
		
		return

    	def draw_plot_solar_system(self, context, window, object, local):
		global rect
		global planet_trail
		global object_show_name_flag
		global asteroid_show_flag
		global comet_show_flag
		global neo_show_flag
		global asteroid_read_flag
		global comet_read_flag
		global neo_read_flag

		if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
			local_time_zone_daylight_flag = 0
		else:
			local_time_zone_daylight_flag = time.daylight

		local_time_zone_name = time.tzname

		size = 5

        	rect = self.get_allocation()
        	start_radius = 100
		x = ((rect.width/2))
		y = ((rect.height/2))
 
        	context.rectangle(0, 0, rect.width, rect.height)
		context.set_source_rgba(0,0,0) # black rectangle
		context.fill()

		font = "Sans"
		context.select_font_face(font)
		context.set_font_size(font_size)

		self.sun_field_of_view = 90.0

		context.set_source_rgb(sun_plot_color[0],sun_plot_color[1],sun_plot_color[2]) # sun plot color
		context.move_to(x-5, y+15)
		context.text_path("Sun")
		context.set_line_width(1.0)
		context.fill()
		context.stroke()
		context.arc(x, y, size*2, 0, 2 * math.pi)
		context.fill()
		context.stroke()

		self.size = 1

		self.planet = ephem.Mercury()
		self.plot_color = mercury_plot_color
		self.draw_plot_solar_system_planet(context,local)

		self.planet = ephem.Venus()
		self.plot_color = venus_plot_color
		self.draw_plot_solar_system_planet(context,local)

		self.planet = ephem.Sun()
		self.plot_color = earth_plot_color
		self.draw_plot_solar_system_planet(context,local)

		self.planet = ephem.Mars()
		self.plot_color = mars_plot_color
		self.draw_plot_solar_system_planet(context,local)

		self.planet = ephem.Jupiter()
		self.plot_color = jupiter_plot_color
		self.draw_plot_solar_system_planet(context,local)

		self.planet = ephem.Saturn()
		self.plot_color = saturn_plot_color
		self.draw_plot_solar_system_planet(context,local)

		self.planet = ephem.Uranus()
		self.plot_color = uranus_plot_color
		self.draw_plot_solar_system_planet(context,local)

		self.planet = ephem.Neptune()
		self.plot_color = neptune_plot_color
		self.draw_plot_solar_system_planet(context,local)

		self.planet = ephem.Pluto()
		self.plot_color = pluto_plot_color
		self.draw_plot_solar_system_planet(context,local)

		if asteroid_show_flag == True:
			run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
			data_dir = os.path.join(run_dir, 'data')
			asteroid_file = os.path.join(data_dir, 'asteroid.dat')
			try:
				f = open(asteroid_file,"rb")
			except:
				on_show_note("clicked",window,"Asteroid data file missing -- download it from web window","OK")
				print "Asteroid data (satellite elements) are missing!  Ignoring request!"
				asteroid_show_flag = False
			else:
				while 1:
					t = f.readline()
					if t == '':
						break
					if '\n' in t:
						t = t[:-1]
					if '\r' in t:
						t = t[:-1]
					a = t.split()
					line_type = a[0]
					if (line_type != "#"):
						b = t.split(",")
						c = b[0].split()
						asteroid_name = c[1]
						line1 = t
						t = f.readline()
						if t == '':
							break
						if '\n' in t:
							t = t[:-1]
						if '\r' in t:
							t = t[:-1]
						asteroid = ephem.readdb(line1)
						asteroid.compute(local)
						long = asteroid.hlong
						lat = (pi*0.5)-asteroid.hlat
						radius = asteroid.sun_distance

						if asteroid_read_flag == False:
							piter = asteroid_treestore.append(None, [asteroid_name, asteroid.az, asteroid.alt, asteroid.ra, asteroid.dec, float(asteroid.az), float(asteroid.alt), float(asteroid.ra), float(asteroid.dec)])

						self.asteroid_x = radius * cos(long) * sin(lat)
						self.asteroid_y = radius * sin(long) * sin(lat)
						self.asteroid_z = radius * cos(lat)

						x = ((rect.width/2) + (rect.height/self.sun_field_of_view * -self.asteroid_x) * planet_field_of_view)
						y = ((rect.height/2) - (rect.height/self.sun_field_of_view * -self.asteroid_y) * planet_field_of_view)
					
						self.size = 0.5
	
						self.plot_color = asteroid_plot_color

						context.set_source_rgb(self.plot_color[0],self.plot_color[1],self.plot_color[2]) # planet plot color
						context.arc(x, y, self.size*4, 0, 2 * math.pi)
						context.fill()
						context.stroke()

						if asteroid_name == select_object:
							context.set_source_rgb(1,1,1)
							context.arc(x, y, self.size*16, 0, 2 * math.pi)
							context.stroke()

						if asteroid_name_show_flag == True:
							context.set_source_rgb(asteroid_plot_color[0], asteroid_plot_color[1], asteroid_plot_color[2])
							font = "Sans"
							context.select_font_face(font)
							context.set_font_size(font_size)
							context.move_to(x + 5, y + 5)
							context.text_path(asteroid_name)
							context.fill()
							context.stroke()

				asteroid_read_flag = True
				f.close()

		if comet_show_flag == True:
			run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
			data_dir = os.path.join(run_dir, 'data')
			comet_file = os.path.join(data_dir, 'comet.dat')
			try:
				f = open(comet_file,"rb")
			except:
				on_show_note("clicked",window,"Comet data file missing -- download it from web window","OK")
				print "Comet data (satellite elements) are missing!  Ignoring request!"
				comet_show_flag = False
			else:
				while 1:
					t = f.readline()
					if t == '':
						break
					if '\n' in t:
						t = t[:-1]
					if '\r' in t:
						t = t[:-1]
					a = t.split()
					line_type = a[0]
					if (line_type != "#"):
						b = t.split(",")
						c = b[0].split("/")
						comet_name = c[1]
						line1 = t
						t = f.readline()
						if t == '':
							break
						if '\n' in t:
							t = t[:-1]
						if '\r' in t:
							t = t[:-1]
						comet = ephem.readdb(line1)
						comet.compute(local)
						long = comet.hlong
						lat = (pi*0.5)-comet.hlat
						radius = comet.sun_distance

						if comet_read_flag == False:
							piter = comet_treestore.append(None, [comet_name, comet.az, comet.alt, comet.ra, comet.dec, float(comet.az), float(comet.alt), float(comet.ra), float(comet.dec)])

						self.comet_x = radius * cos(long) * sin(lat)
						self.comet_y = radius * sin(long) * sin(lat)
						self.comet_z = radius * cos(lat)

						x = ((rect.width/2) + (rect.height/self.sun_field_of_view * -self.comet_x) * planet_field_of_view)
						y = ((rect.height/2) - (rect.height/self.sun_field_of_view * -self.comet_y) * planet_field_of_view)
					
						self.size = 0.5
	
						self.plot_color = comet_plot_color

						context.set_source_rgb(self.plot_color[0],self.plot_color[1],self.plot_color[2]) # planet plot color
						context.arc(x, y, self.size*4, 0, 2 * math.pi)
						context.fill()
						context.stroke()

						if comet_name == select_object:
							context.set_source_rgb(1,1,1)
							context.arc(x, y, self.size*16, 0, 2 * math.pi)
							context.stroke()

						if comet_name_show_flag == True:
							context.set_source_rgb(comet_plot_color[0], comet_plot_color[1], comet_plot_color[2])
							font = "Sans"
							context.select_font_face(font)
							context.set_font_size(font_size)
							context.move_to(x + 5, y + 5)
							context.text_path(comet_name)
							context.fill()
							context.stroke()

				comet_read_flag = True
				f.close()

		if neo_show_flag == True:
			run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
			data_dir = os.path.join(run_dir, 'data')
			neo_file = os.path.join(data_dir, 'neo.dat')
			try:
				f = open(neo_file,"rb")
			except:
				on_show_note("clicked",window,"Comet data file missing -- download it from web window","OK")
				print "Comet data (satellite elements) are missing!  Ignoring request!"
				neo_show_flag = False
			else:
				while 1:
					t = f.readline()
					if t == '':
						break
					if '\n' in t:
						t = t[:-1]
					if '\r' in t:
						t = t[:-1]
					a = t.split()
					line_type = a[0]
					if (line_type != "#"):
						b = t.split(",")
						c = b[0].split()
						neo_name = c[1]
						line1 = t
						t = f.readline()
						if t == '':
							break
						if '\n' in t:
							t = t[:-1]
						if '\r' in t:
							t = t[:-1]
						neo = ephem.readdb(line1)
						neo.compute(local)
						long = neo.hlong
						lat = (pi*0.5)-neo.hlat
						radius = neo.sun_distance

						if neo_read_flag == False:
							piter = neo_treestore.append(None, [neo_name, neo.az, neo.alt, neo.ra, neo.dec, float(neo.az), float(neo.alt), float(neo.ra), float(neo.dec)])

						self.neo_x = radius * cos(long) * sin(lat)
						self.neo_y = radius * sin(long) * sin(lat)
						self.neo_z = radius * cos(lat)

						x = ((rect.width/2) + (rect.height/self.sun_field_of_view * -self.neo_x) * planet_field_of_view)
						y = ((rect.height/2) - (rect.height/self.sun_field_of_view * -self.neo_y) * planet_field_of_view)
					
						self.size = 0.5
	
						self.plot_color = neo_plot_color

						context.set_source_rgb(self.plot_color[0],self.plot_color[1],self.plot_color[2]) # neo plot color
						context.arc(x, y, self.size*4, 0, 2 * math.pi)
						context.fill()
						context.stroke()

						if neo_name == select_object:
							context.set_source_rgb(1,1,1)
							context.arc(x, y, self.size*16, 0, 2 * math.pi)
							context.stroke()

						if neo_name_show_flag == True:
							context.set_source_rgb(neo_plot_color[0], neo_plot_color[1], neo_plot_color[2])
							font = "Sans"
							context.select_font_face(font)
							context.set_font_size(font_size)
							context.move_to(x + 5, y + 5)
							context.text_path(neo_name)
							context.fill()
							context.stroke()

				neo_read_flag = True
				f.close()

		if planet_trail_show_flag == True:
			for planetb in planet_trail:
				planetx = planetb[0]
				planety = planetb[1]
				x = ((rect.width/2) + (rect.height/self.sun_field_of_view * -planetx) * planet_field_of_view)
				y = ((rect.height/2) - (rect.height/self.sun_field_of_view * -planety) * planet_field_of_view)
		
				context.arc(x, y, 1, 0, 2 * math.pi)
				context.set_source_rgb(1,1,1) # moon white color
				context.fill()
				context.stroke()

		if date_in_gmt == True:
			date_title = 'DATE/TIME: '+str(local.date)+' GMT'
		else:
			date_title = 'DATE/TIME: '+str(current_local_date)+' '+local_time_zone_name[local_time_zone_daylight_flag]

		y = 25
		context.set_source_rgb(1,0.65,0) # date orange color
		context.move_to(0,y)
		context.text_path(date_title)
		context.fill()
		context.stroke()

class plot_solar_orbit(gtk.DrawingArea):
        def __init__(self,window,object,local,planet_orbit):
                gtk.DrawingArea.__init__(self)
                self.connect("expose_event", self.expose_plot_solar_system, object, local, planet_orbit)
                self.connect('button_press_event', self.button_press_event_cb)
                self.set_events(gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.POINTER_MOTION_MASK)

                return

    	def button_press_event_cb(self, widget, event):
		global select_object

        	if event.button == 1:
			select_object = ""
					
			gtk.gdk.threads_enter
    			self.queue_draw()
			gtk.gdk.threads_leave

        	elif event.button == 3:
        		print event.x, event.y

        	return True

        def expose_plot_solar_system(self, widget, event, object, local, planet_orbit):
                self.context = widget.window.cairo_create()

                # set a clip region for the expose event
                self.context.rectangle(event.area.x, event.area.y,
                               event.area.width, event.area.height)
                self.context.clip()

                self.draw_plot_solar_system(self.context,object,local, planet_orbit)

                return False

    	def draw_plot_solar_system(self, context,object,local, planet_orbit):
		global rect

		current_local_date = ephem.Date(ephem.localtime(local.date))

		if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
			local_time_zone_daylight_flag = 0
		else:
			local_time_zone_daylight_flag = time.daylight

		local_time_zone_name = time.tzname

		size = 5

        	rect = self.get_allocation()
        	start_radius = 100
		x = ((rect.width/2))
		y = ((rect.height/2))
 
        	context.rectangle(0, 0, rect.width, rect.height)
		context.set_source_rgba(0,0,0) # black rectangle
		context.fill()

		font = "Sans"
		context.select_font_face(font)
		context.set_font_size(font_size)

		sun_field_of_view = 90.0

		context.set_source_rgb(sun_plot_color[0],sun_plot_color[1],sun_plot_color[2]) # sun plot color
		context.move_to(x-5, y+15)
		context.text_path("Sun")
		context.set_line_width(1.0)
		context.fill()
		context.stroke()
		context.arc(x, y, size*2, 0, 2 * math.pi)
		context.fill()
		context.stroke()

		prev_planet_name = ""
		for coord in planet_orbit:
			planet_name = coord[0]
			x = ((rect.width/2) + (rect.height/sun_field_of_view * -coord[1]) * planet_field_of_view)
			y = ((rect.height/2) - (rect.height/sun_field_of_view * -coord[2]) * planet_field_of_view)
			
			if planet_name != prev_planet_name:
				if planet_name == "Mercury":
					plot_color = mercury_plot_color
					object_show_name_flag = True
					size = 1
				if planet_name == "Venus":
					plot_color = venus_plot_color
					object_show_name_flag = True
					size = 1
				if planet_name == "Earth":
					plot_color = earth_plot_color
					object_show_name_flag = True
					size = 1
				if planet_name == "Mars":
					plot_color = mars_plot_color
					object_show_name_flag = True
					size = 1
				if planet_name == "Jupiter":
					plot_color = jupiter_plot_color
					object_show_name_flag = True
					size = 1
				if planet_name == "Saturn":
					plot_color = saturn_plot_color
					object_show_name_flag = True
					size = 1
				if planet_name == "Uranus":
					plot_color = uranus_plot_color
					object_show_name_flag = True
					size = 1
				if planet_name == "Neptune":
					plot_color = neptune_plot_color
					object_show_name_flag = True
					size = 1
				if planet_name == "Pluto":
					plot_color = pluto_plot_color
					object_show_name_flag = True
					size = 1
				if planet_name[:8] == "ASTEROID":
					plot_color = asteroid_plot_color
					planet_name = planet_name[8:]
					object_show_name_flag = asteroid_name_show_flag
					size = 0.5
				if planet_name[:5] == "COMET":
					plot_color = comet_plot_color
					planet_name = planet_name[5:]
					object_show_name_flag = comet_name_show_flag
					size = 0.5
				if planet_name[:3] == "NEO":
					plot_color = neo_plot_color
					planet_name = planet_name[3:]
					object_show_name_flag = neo_name_show_flag
					size = 0.5
				context.set_source_rgb(1,1,1) # planet name color
				if object_show_name_flag == True:
					context.move_to(x-5, y+15)
					context.text_path(planet_name)
					context.set_font_size(font_size)
					context.fill()
					context.stroke()
				context.arc(x, y, size*4, 0, 2 * math.pi)
			else:
				context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(plot_color[0],plot_color[1],plot_color[2]) # planet plot color
			context.fill()
			context.stroke()
			prev_planet_name = planet_name

		if date_in_gmt == True:
			date_title = 'DATE/TIME: '+str(local.date)+' GMT'
		else:
			date_title = 'DATE/TIME: '+str(current_local_date)+' '+local_time_zone_name[local_time_zone_daylight_flag]

		y = 25
		context.set_source_rgb(1,0.65,0) # date orange color
		context.move_to(0,y)
		context.text_path(date_title)
		context.fill()
		context.stroke()

def get_moon_image_start(widget):
        if hildon_flag == 1:
                window = hildon.Window()
        else:
                window = gtk.Window()

	vbox = gtk.HBox(False, 0)

	window.connect("window-state-event", on_window_state_change, window)
        window.connect("key-press-event", on_key_press_basic, window)

	window.set_title('Clementine Maps for Moon: BASE VIEW')

	window.set_size_request(800,480)

	get_moon_image("clicked",window,vbox)

	return

def get_moon_image(widget,window,vbox):
	plot = get_moon(window)
	vbox.pack_start(plot,True)

	vbox.show()
	window.add(vbox)
	window.show()

        if window_in_fullscreen == True:
                window.fullscreen()

	plot.show()

	return

class get_moon(gtk.DrawingArea):
        def __init__(self,window):
                gtk.DrawingArea.__init__(self)
                self.connect("expose_event", self.expose_get_moon)
		self.connect('button_press_event', self.button_press_event_cb)
                self.set_events(gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.POINTER_MOTION_MASK)

                return

    	def button_press_event_cb(self, widget, event):
		global rect
		global moon_x
		global moon_y

        	rect = self.get_allocation()
        	if event.button == 1:
			moon_x = int(event.x/rect.width * 337)
			moon_y = int(event.y/rect.height * (480.0 * 168 / 400))

			#print moon_x, moon_y
			#if moon_x < 168:
			#	long = (moon_x-168)/168.0*180
			#	print "long=", long, abs(long), "W"
			#elif moon_x > 168:
			#	long = 360-(moon_x-168)/168.0*180
			#	print "long=", long, 360-long, "E"
			#else:
			#	print "long=", 0
			#if moon_y > 84:
			#	lat = -(moon_y-84)/84.0*90.0
			#	print "lat=", lat, abs(lat),"S"
			#elif moon_y < 84:
			#	lat = -(moon_y-84)/84.0*90.0
			#	print "lat=", lat, lat,"N"
			#else:
			#	print "lat=", 0

			zoom_moon_start()

			gtk.gdk.threads_enter
    			self.queue_draw()
			gtk.gdk.threads_leave

        	elif event.button == 3:
        		print event.x, event.y

        	return True

        def expose_get_moon(self, widget, event):
                self.context = widget.window.cairo_create()
		self.context.set_operator(cairo.OPERATOR_OVER)
        	self.context.paint()

		window = widget
		screen = window.get_screen()
		screen_width = screen.get_width()
		screen_height = screen.get_height()
	
		run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
		data_dir = os.path.join(run_dir, 'data')
		#moon_file = os.path.join(data_dir, 'clementine.png')
		moon_file = os.path.join(data_dir, 'moon_clementine_flat.png')

		image = cairo.ImageSurface.create_from_png(moon_file)
		image_width = image.get_width()
		image_height = image.get_height()
		image_pattern = cairo.SurfacePattern(image)

		scaler = cairo.Matrix()
		if window_in_fullscreen == False:
			scaler.scale(1.0,1.0)
		else:
			ratio = 800.0/screen_width
			ratio = 480.0/screen_height
			scaler.scale(800.0/screen_width,480.0/screen_height)

		image_pattern.set_matrix(scaler)
		image_pattern.set_filter(cairo.FILTER_BEST)

        	self.context.set_source(image_pattern)

                # set a clip region for the expose event
                self.context.rectangle(event.area.x, event.area.y,
                               event.area.width, event.area.height)
                self.context.clip()

                self.draw_get_moon(self.context, window)

		#t = cairo.ImageSurface(cairo.FORMAT_ARGB32, 800, 480)
		#t.write_to_png("junk.png")

                return False

    	def draw_get_moon(self, context, window):
        	rect = self.get_allocation()
 
                context.set_line_width(1)

		screen = window.get_screen()
		screen_width = screen.get_width()
		screen_height = screen.get_height()

		if window_in_fullscreen == False:
			size = 200 
		else:
			size = 200 * screen_width/800
			size = 200 * screen_height/480

		context.paint()

		context.set_source_rgb(1,1,0)
		context.set_font_size(font_size+3)
		context.move_to(10,rect.height-10)
		context.text_path('Alternate Interface Clementine Lunar Image Browser (CLIB) at http://www.cmf.nrl.navy.mil/clementine/clib/')
		context.fill()
		context.stroke()

		if grid_show_flag == True:
			context.set_source_rgb(0,1,0)
			for longitude in range(-180, 180, grid_space):
				x = (longitude/180.0) * (rect.width/2.0) + (rect.width/2.0)
				y = 0
				context.move_to(x, y)
				y = rect.height * (400 / 480.0)
				context.line_to(x, y)
				context.stroke()
				context.move_to(x, y/2)
				if longitude > 0:
					context.text_path(str(longitude) + u'\xb0' + 'E')
				else:
					context.text_path(str(abs(longitude)) + u'\xb0' + 'W')
				context.fill()
				context.stroke()
			for latitude in range(-90, 90, grid_space):
				y = ((latitude/90.0) * (rect.height/2.0) + (rect.height/2.0)) * (400 / 480.0)
				x = 0
				context.move_to(x, y)
				x = rect.width
				context.line_to(x, y)
				context.stroke()
				context.move_to(x/2, y)
				if latitude > 0:
					context.text_path(str(latitude) + u'\xb0' + 'S')
				else:
					context.text_path(str(abs(latitude)) + u'\xb0' + 'N')
				context.fill()
				context.stroke()

		if moon_feature_show_flag == True:
			context.set_line_width(1.0)
			run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
			data_dir = os.path.join(run_dir, 'data')
			moon_features_file = os.path.join(data_dir, 'lunar100.dat')
			f = open(moon_features_file,"rb")
			while 1:
				t = f.readline()
				if t == '':
					break
				if '\n' in t:
					t = t[:-1]
				if '\r' in t:
					t = t[:-1]
				a = t.split("\t")
				num = int(a[0])
				if num > 3:
					moon_feature_name = a[1]
					latitude = a[2]
					longitude = a[3]
					size = a[4]
					if latitude[-1:] == "S":
						latitude = float(latitude[:-1])
					else:
						latitude = -float(latitude[:-1])
					if longitude[-1:] == "W":
						longitude = -float(longitude[:-1])
					else:
						longitude = float(longitude[:-1])
					x = (longitude/180.0) * (rect.width/2.0) + (rect.width/2.0)
					y = ((latitude/90.0) * (rect.height/2.0) + (rect.height/2.0)) * (400 / 480.0)
					context.set_source_rgb(1,0,0)
					context.arc(x, y, object_size+10, 0, 2 * math.pi)
					context.stroke()
					if moon_feature_name_show_flag == True:
						#context.set_source_rgb(moon_feature_plot_color[0], moon_feature_plot_color[1], moon_feature_plot_color[2])
						context.set_source_rgb(0,0,0)
						context.set_font_size(font_size)
						context.move_to(x, y)
						context.text_path(moon_feature_name)
						context.fill()
						context.stroke()
			f.close()

		context.fill()
		context.stroke()

def zoom_moon_start():
	if hildon_flag == 1:
		window = hildon.Window()
	else:
		window = gtk.Window()

	image = gtk.Image()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_get_moon, window, image)

	window.set_size_request(800,480)

	zoom_moon("clicked",window,image)

	return

def zoom_moon(widget, window, image):
	window.set_title('Clementine Maps for Moon: ZOOM VIEW LEVEL ' + str(moon_zoom))

	url = "http://www.cmf.nrl.navy.mil/cgi-bin/clementine/clib/multires.pl?click.x=" + str(moon_x) + "&click.y=" + str(moon_y) + "&clickres=5&ox=0&oy=0&res=" + str(moon_zoom) + "&size=512&latitude=0&longitude=0&sensor=UVVIS&filter=415_nm"
	print "url=",url

	for line in urllib.urlopen(url).readlines():
		if line.lower().find('generated_images') >= 0:
			pos1 = line.find('"') + 1
			pos2 = line.find('"', pos1)
			pic_url = line[pos1:pos2]
			break

	screen = window.get_screen()
	screen_width = screen.get_width()
	screen_height = screen.get_height()
	window_width = window.get_size()[0]
	window_height = window.get_size()[1]

	color_black = gtk.gdk.color_parse('black')
	window.modify_bg(gtk.STATE_NORMAL, color_black)

	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
	data_dir = os.path.join(run_dir, 'data')
	image_file = os.path.join(data_dir, 'clementine_moon.gif')

	# retrieve it and save to the cache directory
	moon_pic_file = data_dir+"/"+os.path.split(pic_url)[1]
	urllib.urlretrieve(pic_url, moon_pic_file)
	os.system("mv "+moon_pic_file+" "+data_dir+"/clementine_moon.gif")
	#print "moon_pic_file=",moon_pic_file

	image.set_from_file(image_file)

	pixbuf = gtk.gdk.pixbuf_new_from_file(image_file)

	aspect = True
	enlarge = True

	display_image(image,pixbuf,aspect,enlarge,screen_width,screen_height,window_width,window_height)

	#url_show = "dare"
	#table = gtk.Table(1,1, False)
	#window.add(table)
	#table.attach(image,0,1,0,1)
	#event_box = gtk.EventBox()
	#table.attach(event_box,0,1,1,2)
	#set_event_box_color("clicked",event_box,"black","yellow",url_show)
	#table.show()

	window.add(image)

	image.show()

	if window_in_fullscreen == True:
		window.fullscreen()

	window.show()

	return

def plot_moon_phase_start(widget,local):
        if hildon_flag == 1:
                window = hildon.Window()
        else:
                window = gtk.Window()

	vbox = gtk.HBox(False, 0)

	window.connect("window-state-event", on_window_state_change, window)
        window.connect("key-press-event", on_key_press_plot_moon, window, vbox, local)

	window.set_title('Planet Ephemeris for Moon: CURRENT PHASE')

	window.set_size_request(800,480)

	plot_moon_phase("clicked",window,vbox,local)

	return

class plot_moon_phase:
    	def moon_feature_show_callback(self, widget):
		global moon_feature_show_flag 
		if widget.get_active():
			moon_feature_show_flag = True
		else:
			moon_feature_show_flag = False

    	def moon_feature_name_show_callback(self, widget):
		global moon_feature_name_show_flag 
		if widget.get_active():
			moon_feature_name_show_flag = True
		else:
			moon_feature_name_show_flag = False

        def moon_feature_search_callback(self, widget):
		global object_search_flag
		global moon_feature_treestore

		if moon_feature_show_flag == True:
			if len(moon_feature_treestore) == 0:
				print "ERROR: No moon features have been plotted yet!"
				object_search_flag = False
			else:
				object_search_flag = True
				moon_feature_search("Moon")

    	def grid_show_callback(self, widget):
		global grid_show_flag 
		if widget.get_active():
			grid_show_flag = True
		else:
			grid_show_flag = False

    	def legend_show_callback(self, widget):
		global legend_show_flag 
		if widget.get_active():
			legend_show_flag = True
		else:
			legend_show_flag = False

	def __init__(self,widget,window,vbox,local):
		table = gtk.Table(3,2,False)
		table.set_col_spacings(0)
		table.set_row_spacings(1)

		table1 = gtk.Table(1,1,False)
		table1.set_col_spacings(1)
		table1.set_row_spacings(1)

		table3 = gtk.Table(1,1,False)
		table3.set_col_spacings(1)
		table3.set_row_spacings(1)

		plot = plot_moon(window,local)
		vbox.pack_start(plot,True)
		plot.show()

		if controls_show_flag == True:

			if date_time_controls_show_flag == True:
				column = 0
				row = 0

				if verbose_show_flag == True:
					button = gtk.Button("Column Remove")
				else:
					button = gtk.Button("Remove")
				date_time_controls_show_flag2 = 0	
				button.connect("clicked",toggle_moon_plot_date_time_controls,window,vbox,object,local,date_time_controls_show_flag2)
				table1.attach(button,column,column+3,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Year")
				else:
					button = gtk.Button("Yr")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -5
				button = gtk.Button("-")
				button.connect("clicked",change_date_moon_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 5
				button = gtk.Button("+")
				button.connect("clicked",change_date_moon_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Month")
				else:
					button = gtk.Button("Mo")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -4
				button = gtk.Button("-")
				button.connect("clicked",change_date_moon_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 4
				button = gtk.Button("+")
				button.connect("clicked",change_date_moon_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Day")
				else:
					button = gtk.Button("Dy")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -3
				button = gtk.Button("-")
				button.connect("clicked",change_date_moon_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 3
				button = gtk.Button("+")
				button.connect("clicked",change_date_moon_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				date_time_change_flag = -2
				if verbose_show_flag == True:
					button = gtk.Button("Hour")
				else:
					button = gtk.Button("Hr")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -2
				button = gtk.Button("-")
				button.connect("clicked",change_date_moon_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 2
				button = gtk.Button("+")
				button.connect("clicked",change_date_moon_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Minute")
				else:
					button = gtk.Button("Mi")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -1
				button = gtk.Button("-")
				button.connect("clicked",change_date_moon_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 1
				button = gtk.Button("+")
				button.connect("clicked",change_date_moon_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				date_time_change_flag = 0
				if verbose_show_flag == True:
					button = gtk.Button('Reset Date/Time')
				else:
					button = gtk.Button('Reset')
				button.connect("clicked",change_date_moon_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+3,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Grid Space")
				else:
					button = gtk.Button("Gs")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				button = gtk.Button("-")
				grid_space_change_flag = -1
				button.connect("clicked",set_grid_space,grid_space_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = column+1
				button = gtk.Button("+")
				grid_space_change_flag = 1
				button.connect("clicked",set_grid_space,grid_space_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Text Size")
				else:
					button = gtk.Button("Ts")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				button = gtk.Button("-")
				text_size_change_flag = -1
				button.connect("clicked",set_font_size,text_size_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = column+1
				button = gtk.Button("+")
				text_size_change_flag = 1
				button.connect("clicked",set_font_size,text_size_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Features")
				else:
					button = gtk.ToggleButton("Fe")
				button.set_active(moon_feature_show_flag)
				button.connect("toggled", self.moon_feature_show_callback)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Names")
				else:
					button = gtk.ToggleButton("FeN")
				button.set_active(moon_feature_name_show_flag)
				button.connect("toggled", self.moon_feature_name_show_callback)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.Button("Search")
				else:
					button = gtk.Button("FeS")
				button.connect("clicked", self.moon_feature_search_callback)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				row = 0
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Grid Lines")
				else:
					button = gtk.ToggleButton("GrL")
				button.set_active(grid_show_flag)
				button.connect("toggled", self.grid_show_callback)
				table3.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column + 1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Legend Text")
				else:
					button = gtk.ToggleButton("LgT")
				button.set_active(legend_show_flag)
				button.connect("toggled", self.legend_show_callback)
				table3.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")
				column = 0

				column = 1
				table.attach(table1,column,column+1,0,9)
				table1.show()

				column = 1
				table.attach(table3,column,column+1,9,10)
				table3.show()

		window.add(vbox)
		vbox.pack_start(table,False)

		window.show_all()

		if window_in_fullscreen == True:
			window.fullscreen()

		window.show()

		return

class plot_moon(gtk.DrawingArea):
        def __init__(self,window,local):
                gtk.DrawingArea.__init__(self)
                self.connect("expose_event", self.expose_plot_moon, local)
		self.connect('button_press_event', self.button_press_event_cb)
                self.set_events(gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.POINTER_MOTION_MASK)

                return

    	def button_press_event_cb(self, widget, event):
		global select_object

        	if event.button == 1:
			select_object = ""
					
			gtk.gdk.threads_enter
    			self.queue_draw()
			gtk.gdk.threads_leave

        	elif event.button == 3:
        		print event.x, event.y

        	return True

        def expose_plot_moon(self, widget, event, local):
		global moon_zenith_angle
                self.context = widget.window.cairo_create()
		self.context.set_operator(cairo.OPERATOR_OVER)
        	self.context.paint()

		window = widget
		screen = window.get_screen()
		screen_width = screen.get_width()
		screen_height = screen.get_height()
	
		sun = ephem.Sun()
		moon = ephem.Moon()
		object = ephem.Moon()

		moon.compute(local)
		sun.compute(local)

		y = sin(sun.az - moon.az) * cos(sun.alt)
		x = cos(moon.alt)* sin(sun.alt) - sin(moon.alt) * cos(sun.alt) * cos(sun.az - moon.az)
		bearing = atan2(y, x)
		db =  bearing*180/pi % 360
		moon_zenith_angle = db - 90

		k=180.0/pi

		lon1r = moon.az - pi
		lat1r = moon.alt
		lon2r = sun.az - pi
		lat2r = sun.alt

		dlong = lon2r - lon1r
		dlat = lat2r - lat1r
		arg1=sin(dlong) * cos(lat2r)
		arg2a = cos(lat1r) * sin(lat2r)
		arg2b = sin(lat1r) * cos(lat2r) * cos(dlong)
		arg2 =  arg2a -  arg2b
		bearingr = atan2(arg1, arg2)

		bearing = round(((360 + (bearingr * k)) % 360))
		if abs(dlong) > pi:
			bearing = 360 - bearing
		moon_zenith_angle = bearing - 90

		x = (cos(sun.alt)*sin(sun.az - moon.az))
		y = (cos(moon.alt)*sin(sun.alt) - sin(moon.alt)*cos(sun.alt)*cos(sun.az - moon.az))
		x2 = atan((cos(sun.alt)*sin(sun.az - moon.az))/(cos(moon.alt)*sin(sun.alt) - sin(moon.alt)*cos(sun.alt)*cos(sun.az - moon.az)))
		moon_zenith_angle0 = x2*180/pi - 90
		x3 = atan2(y,x) 
		moon_zenith_angle1 = x3*180/pi 

		x = (cos(sun.dec)*sin(sun.ra - moon.ra))
		y = (cos(moon.dec)*sin(sun.dec) - sin(moon.dec)*cos(sun.dec)*cos(sun.ra - moon.ra))
		x3 = atan2(y,x) 
		x3 = atan2(x,y)
		moon_zenith_angle3 = x3*180/pi 

		q = atan(sin(local.sidereal_time())/ (tan(local.lat)*cos(moon.dec)-sin(moon.dec)*cos(local.sidereal_time())))
		q = sin(moon.az)*cos(local.lat)/cos(moon.dec)
		p = moon_zenith_angle3 - q*180/pi
		moon_zenith_angle4 = p + 90

                save = local.date
                local.date = local.date-0.001
                object.compute(local)
                moon_earlier_az = float(object.az)
                moon_earlier_alt = float(object.alt)
                local.date = save

                local.date = save
                local.date = local.date+0.001
                object.compute(local)
                moon_later_az = float(object.az)
                moon_later_alt = float(object.alt)
                local.date = save

                moon_zenith_angle2 = -atan((moon_later_alt - moon_earlier_alt)/(moon_later_az - moon_earlier_az)) * 180/pi
		moon_zenith_angle = moon_zenith_angle2
	
                # set a clip region for the expose event
                self.context.rectangle(event.area.x, event.area.y,
                               event.area.width, event.area.height)
                self.context.clip()

		#self.context.restore()

                self.draw_plot_moon(self.context, local, window)

                return False

    	def draw_plot_moon(self, context, local, window):
        	rect = self.get_allocation()
 
                context.set_line_width(1)

		screen = window.get_screen()
		screen_width = screen.get_width()
		screen_height = screen.get_height()

		run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
		data_dir = os.path.join(run_dir, 'data')
		#moon_file = os.path.join(data_dir, 'moon.png')
		#moon_file = os.path.join(data_dir, 'moon_grid.png')
		moon_file = os.path.join(data_dir, 'moon_clementine_sphere.png')

		image = cairo.ImageSurface.create_from_png(moon_file)
		image_width = image.get_width()
		image_height = image.get_height()
		image_pattern = cairo.SurfacePattern(image)

		scaler = cairo.Matrix()

		# rotate the image of the moon
		scaler.translate(400,240)
		scaler.rotate(-moon_zenith_angle *pi/180)
		scaler.translate(-400,-240)

		if window_in_fullscreen == False:
			scaler.scale(1.0,1.0)
		else:
			screen_width_ratio = 800.0/screen_width
			screen_height_ratio = 480.0/screen_height
			scaler.scale(screen_width_ratio, screen_height_ratio) # good

        	self.context.set_source(image_pattern)

		image_pattern.set_matrix(scaler)
		image_pattern.set_filter(cairo.FILTER_BEST)

		if window_in_fullscreen == False:
			self.context.translate(400,240)
			self.context.rotate(moon_zenith_angle *pi/180)
			self.context.translate(-400,-240)
		else:
			self.context.translate(screen_width/2,screen_height/2) # good
			self.context.rotate(moon_zenith_angle *pi/180) # good
			self.context.translate(-screen_width/2,-screen_height/2) # good
		
		if window_in_fullscreen == False:
			size_height = 200 
			size_width = 200 
		else:
			size_height = 200.0 * screen_height/480.0
			size_width = 200.0 * screen_width/800.0

		sun = ephem.Sun()
		sun.compute(local)
		sun_ra = sun.ra
		sun_dec = sun.dec

		moon = ephem.Moon()
		moon.compute(local)

		object = ephem.Moon()
		object.compute(local)

		moon_mag = object.mag
		moon_elong = object.elong
		moon_radius = object.radius
		moon_size = object.size
		moon_sun_distance = object.sun_distance
		moon_earth_distance = object.earth_distance
		moon_phase = object.phase
		moon_ra = object.ra
		moon_dec = object.dec
		moon_az = object.az
		moon_alt = object.alt
		moon_libration_lat = object.libration_lat
		moon_libration_long = object.libration_long 
		moon_colong = object.colong
		moon_subsolar_lat = object.subsolar_lat

		moon_position_angle = atan((cos(sun_dec) * sin(sun_ra - moon_ra)) / (cos(moon_dec) * sin(sun_dec) - sin(moon_dec) * cos(sun_dec) * cos(sun_ra - moon_ra))) * 180/pi

		save = local.date

		moon_next_full = ephem.next_full_moon(local.date)
		moon_next_new = ephem.next_new_moon(local.date)

                if moon_next_full < moon_next_new:
			moon_phase_name = "Waxing"
                        view_az = 0
                        view_alt = 0
                        az_degrees = 90
                        az = az_degrees * pi/180
                        for alt_degrees in range(90,-90,-1):
                                alt = alt_degrees * pi/180
                                arc_x = size_width * cos(alt)*sin(az-view_az)
                                arc_y = size_height * (cos(view_alt)*sin(alt)-sin(view_alt)*cos(alt)*cos(az-view_az))
				if window_in_fullscreen == False:
					x = ((800/2) + arc_x)
				else:
					x = ((screen_width/2) + arc_x)
                                y = ((rect.height/2) - arc_y)
                                context.line_to(x,y)

                        view_az = 0
                        view_alt = 0
                        az_degrees = -asin((moon_phase - 50)/50) * 180/pi
			
                        az = az_degrees * pi/180
                        for alt_degrees in range(-90,90,1):
                                alt = alt_degrees * pi/180
                                arc_x = size_width * cos(alt)*sin(az-view_az)
                                arc_y = size_height * (cos(view_alt)*sin(alt)-sin(view_alt)*cos(alt)*cos(az-view_az))
				if window_in_fullscreen == False:
					x = ((800/2) + arc_x)
				else:
					x = ((screen_width/2) + arc_x)
                                y = ((rect.height/2) - arc_y)
                                context.line_to(x,y)
                else:
			moon_phase_name = "Waning"
                        view_az = 0
                        view_alt = 0
                        az_degrees = -90
                        az = az_degrees * pi/180
                        for alt_degrees in range(90,-90,-1):
                                alt = alt_degrees * pi/180
                                arc_x = size_width * cos(alt)*sin(az-view_az)
                                arc_y = size_height * (cos(view_alt)*sin(alt)-sin(view_alt)*cos(alt)*cos(az-view_az))
				if window_in_fullscreen == False:
					x = ((800/2) + arc_x)
				else:
					x = ((screen_width/2) + arc_x)
                                y = ((rect.height/2) - arc_y)
                                context.line_to(x,y)

                        view_az = 0
                        view_alt = 0
                        az_degrees = asin((moon_phase - 50)/50) * 180/pi
                        az = az_degrees * pi/180
                        for alt_degrees in range(-90,90,1):
                                alt = alt_degrees * pi/180
                                arc_x = size_width * cos(alt)*sin(az-view_az)
                                arc_y = size_height * (cos(view_alt)*sin(alt)-sin(view_alt)*cos(alt)*cos(az-view_az))
				if window_in_fullscreen == False:
					x = ((800/2) + arc_x)
				else:
					x = ((screen_width/2) + arc_x)
                                y = ((rect.height/2) - arc_y)
                                context.line_to(x,y)

		context.paint_with_alpha(0.3)
		context.fill()
		context.stroke()

		if moon_feature_show_flag == True:
			moon_feature_treestore.clear()

			context.set_line_width(1.0)
			run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
			data_dir = os.path.join(run_dir, 'data')
			moon_features_file = os.path.join(data_dir, 'lunar100.dat')
			f = open(moon_features_file,"rb")
			while 1:
				t = f.readline()
				if t == '':
					break
				if '\n' in t:
					t = t[:-1]
				if '\r' in t:
					t = t[:-1]
				a = t.split("\t")
				num = int(a[0])
				if num > 3:
					moon_feature_name = a[1]
					latitude = a[2]
					longitude = a[3]
					moon_feature_size = a[4]
					alt = float(latitude[:-1]) * pi/180
					az = float(longitude[:-1]) * pi/180
					if latitude[-1:] == "S":
						alt = -alt
					if longitude[-1:] == "W":
						az = -az
					piter = moon_feature_treestore.append(None, [moon_feature_name, str(ephem.degrees(az)), str(ephem.degrees(alt)), moon_feature_size])
					arc_x = size_width * cos(alt)*sin(az-view_az)
					arc_y = size_height * (cos(view_alt)*sin(alt)-sin(view_alt)*cos(alt)*cos(az-view_az))
					if window_in_fullscreen == False:
                                		x = ((800/2) + arc_x)
					else:
                                		x = ((screen_width/2) + arc_x)
					y = ((rect.height/2) - arc_y)
					if moon_feature_name == select_object:
						context.set_source_rgb(0,1,0)
						context.arc(x, y, object_size+10, 0, 2 * math.pi)
						context.fill()
						context.stroke()
					else:
						context.set_source_rgb(1,0,0)
						context.arc(x, y, object_size+10, 0, 2 * math.pi)
						context.stroke()
					if moon_feature_name_show_flag == True:
						context.set_source_rgb(moon_feature_plot_color[0], moon_feature_plot_color[1], moon_feature_plot_color[2])
						context.set_font_size(font_size)
						context.move_to(x, y)
						context.text_path(moon_feature_name)
						context.fill()
						context.stroke()
			f.close()

		if grid_show_flag == True:
			context.set_source_rgb(0,1,0)
			view_az = 0
			view_alt = 0
			for az_degrees in range(-90, 90, grid_space):
				az = az_degrees * pi/180
				for alt_degrees in range(-90, 90, grid_inc):
					alt = alt_degrees * pi/180
					arc_x = size_width * cos(alt)*sin(az-view_az)
					arc_y = size_height * (cos(view_alt)*sin(alt)-sin(view_alt)*cos(alt)*cos(az-view_az))
					if window_in_fullscreen == False:
                                		x = ((800/2) + arc_x)
					else:
						x = ((screen_width/2) + arc_x)
					y = ((rect.height/2) - arc_y)
					context.arc(x, y, 1, 0, 2 * math.pi)
					context.fill()
					context.stroke()
			for alt_degrees in range(-90, 90, grid_space):
				alt = alt_degrees * pi/180
				for az_degrees in range(-90, 90, grid_inc):
					az = az_degrees * pi/180
					arc_x = size_width * cos(alt)*sin(az-view_az)
					arc_y = size_height * (cos(view_alt)*sin(alt)-sin(view_alt)*cos(alt)*cos(az-view_az))
					if window_in_fullscreen == False:
                                		x = ((800/2) + arc_x)
					else:
						x = ((screen_width/2) + arc_x)
					y = ((rect.height/2) - arc_y)
					context.arc(x, y, 1, 0, 2 * math.pi)
					context.fill()
					context.stroke()
		
		self.context.save()

		if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
			local_time_zone_daylight_flag = 0
		else:
			local_time_zone_daylight_flag = time.daylight

		local_time_zone_name = time.tzname

		if window_in_fullscreen == False:
			context.translate(400,240)
			context.scale(1, 1)
			context.rotate(-moon_zenith_angle *pi/180)
		else:
			context.translate(screen_width/2,screen_height/2)
			context.rotate(-moon_zenith_angle *pi/180)
		context.set_font_size(font_size)
		context.set_line_width(1.0)
		context.set_source_rgb(1,1,1)
		current_local_date = ephem.Date(ephem.localtime(local.date))

		location_title = 'LOCATION: '+str(local.lat)+u'\xb0 '+str(local.long)+u'\xb0'
		if date_in_gmt == True:
			date_title = 'DATE/TIME: '+str(local.date)+' GMT'
		else:
			date_title = 'DATE/TIME: '+str(current_local_date)+' '+local_time_zone_name[local_time_zone_daylight_flag]
		moon_az_title = 'MOON AZ: '+str(moon_az)+u'\xb0'
		moon_alt_title = 'MOON ALT: '+str(moon_alt)+u'\xb0'
		moon_ra_title = 'MOON RA: '+str(moon_ra)+u'\xb0'
		moon_dec_title = 'MOON DEC: '+str(moon_dec)+u'\xb0'
		moon_radius_title = 'MOON RADIUS: '+str(moon_radius)+u'\xb0'
		moon_size_title = 'MOON SIZE: '+str(ephem.degrees(moon_size/3600 * pi/180))+u'\xb0'
		moon_sun_distance_title = 'MOON SUN DISTANCE: '+str(round(moon_sun_distance,3))+'AU'
		moon_earth_distance_title = 'MOON EARTH DISTANCE: '+str(round(moon_earth_distance,3))+'AU'
		moon_phase_title = 'MOON PHASE: '+str(round(moon_phase,1))+'%'+' '+moon_phase_name
		moon_zenith_angle_title = 'MOON ZENITH ANGLE: '+str(ephem.degrees(moon_zenith_angle * pi/180))+u'\xb0'
		moon_position_angle_title = 'MOON POSITION ANGLE: '+str(ephem.degrees(moon_position_angle * pi/180))+u'\xb0'
		moon_libration_lat_title = 'MOON LIBRATION LAT: '+str(moon_libration_lat)+u'\xb0'
		moon_libration_long_title = 'MOON LIBRATION LONG: '+str(moon_libration_long)+u'\xb0'
		moon_colong_title = 'MOON COLONG: '+str(moon_colong)+u'\xb0'
		moon_subsolar_lat_title = 'MOON SUBSOLAR LAT: '+str(moon_subsolar_lat)+u'\xb0'

		if window_in_fullscreen == False:
			x = -390
			y = -220
		else:
			x = -screen_width/2 + 10
			y = -screen_height/2 + 20

		context.set_line_width(1.0)
		context.set_source_rgb(1,0.65,0) # date orange color
		context.move_to(x, y)
		context.text_path(date_title)
		context.fill()
		context.stroke()

		if legend_show_flag == True:
			y = y+15

			context.move_to(x, y)
			context.text_path(location_title)
			context.fill()
			context.stroke()

			context.set_source_rgb(1,0,0) # legend red color

			y = y+15

			context.move_to(x, y)
			context.text_path(moon_phase_title)
			context.fill()
			context.stroke()

			y = y+15

			context.move_to(x, y)
			context.text_path(moon_az_title)
			context.fill()
			context.stroke()

			y = y+15

			context.move_to(x, y)
			context.text_path(moon_alt_title)
			context.fill()
			context.stroke()

			y = y+15

			context.move_to(x, y)
			context.text_path(moon_ra_title)
			context.fill()
			context.stroke()

			y = y+15

			context.move_to(x, y)
			context.text_path(moon_dec_title)
			context.fill()
			context.stroke()

			y = y+15

			context.move_to(x, y)
			context.text_path(moon_radius_title)
			context.fill()
			context.stroke()

			y = y+15

			context.move_to(x, y)
			context.text_path(moon_size_title)
			context.fill()
			context.stroke()

			y = y+15

			context.move_to(x, y)
			context.text_path(moon_sun_distance_title)
			context.fill()
			context.stroke()

			y = y+15

			context.move_to(x, y)
			context.text_path(moon_earth_distance_title)
			context.fill()
			context.stroke()

			y = y+15

			context.move_to(x, y)
			context.text_path(moon_zenith_angle_title)
			context.fill()
			context.stroke()

			y = y+15

			context.move_to(x, y)
			context.text_path(moon_position_angle_title)
			context.fill()
			context.stroke()

			y = y+15

			context.move_to(x, y)
			context.text_path(moon_libration_lat_title)
			context.fill()
			context.stroke()

			y = y+15

			context.move_to(x, y)
			context.text_path(moon_libration_long_title)
			context.fill()
			context.stroke()

			y = y+15

			context.move_to(x, y)
			context.text_path(moon_colong_title)
			context.fill()
			context.stroke()

			y = y+15

			context.move_to(x, y)
			context.text_path(moon_subsolar_lat_title)
			context.fill()
			context.stroke()

			self.context.restore()

		local.date = save

def plot_planet_moon_position_start(widget,object,local):
	global moon_trail

        if hildon_flag == 1:
                window = hildon.Window()
        else:
                window = gtk.Window()

	vbox = gtk.HBox(False, 0)

	window.connect("window-state-event", on_window_state_change, window)
        window.connect("key-press-event", on_key_press_plot_planet_moon, window, vbox, object, local)

	window.set_title('Planet Ephemeris for ' + object.name + ': MOONS')

	window.set_size_request(800,480)

	moon_trail = []

	plot_planet_moon_position("clicked",window,vbox,object,local)

	return

class plot_planet_moon_position:
    	def moon_trail_show_callback(self, widget):
		global moon_trail
		global moon_trail_show_flag 
		if widget.get_active():
			moon_trail_show_flag = True
		else:
			moon_trail_show_flag = False
			moon_trail = []

    	def legend_show_callback(self, widget):
		global legend_show_flag 
		if widget.get_active():
			legend_show_flag = True
		else:
			legend_show_flag = False

	def __init__(self,widget,window,vbox,object,local):
		table = gtk.Table(3,2,False)
		table.set_col_spacings(0)
		table.set_row_spacings(1)

		table1 = gtk.Table(1,1,False)
		table1.set_col_spacings(1)
		table1.set_row_spacings(1)

		table3 = gtk.Table(1,1,False)
		table3.set_col_spacings(1)
		table3.set_row_spacings(1)

		if object.name == "Mars":
			plot = plot_mars_moon(window,object,local)
		if object.name == "Jupiter":
			plot = plot_jupiter_moon(window,object,local)
		if object.name == "Saturn":
			plot = plot_saturn_moon(window,object,local)
		if object.name == "Uranus":
			plot = plot_uranus_moon(window,object,local)

		vbox.pack_start(plot,True)
		plot.show()

		if controls_show_flag == True:

			if date_time_controls_show_flag == True:
				column = 0
				row = 0

				if verbose_show_flag == True:
					button = gtk.Button("Column Remove")
				else:
					button = gtk.Button("Remove")
				date_time_controls_show_flag2 = 0	
				button.connect("clicked",toggle_mars_plot_date_time_controls,window,vbox,object,local,date_time_controls_show_flag2)
				table1.attach(button,column,column+3,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Year")
				else:
					button = gtk.Button("Yr")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -5
				button = gtk.Button("-")
				button.connect("clicked",change_date_planet_moon_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 5
				button = gtk.Button("+")
				button.connect("clicked",change_date_planet_moon_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Month")
				else:
					button = gtk.Button("Mo")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -4
				button = gtk.Button("-")
				button.connect("clicked",change_date_planet_moon_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 4
				button = gtk.Button("+")
				button.connect("clicked",change_date_planet_moon_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Day")
				else:
					button = gtk.Button("Dy")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -3
				button = gtk.Button("-")
				button.connect("clicked",change_date_planet_moon_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 3
				button = gtk.Button("+")
				button.connect("clicked",change_date_planet_moon_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				date_time_change_flag = -2
				if verbose_show_flag == True:
					button = gtk.Button("Hour")
				else:
					button = gtk.Button("Hr")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -2
				button = gtk.Button("-")
				button.connect("clicked",change_date_planet_moon_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 2
				button = gtk.Button("+")
				button.connect("clicked",change_date_planet_moon_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Minute")
				else:
					button = gtk.Button("Mi")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -1
				button = gtk.Button("-")
				button.connect("clicked",change_date_planet_moon_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 1
				button = gtk.Button("+")
				button.connect("clicked",change_date_planet_moon_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				date_time_change_flag = 0
				if verbose_show_flag == True:
					button = gtk.Button('Reset Date/Time')
				else:
					button = gtk.Button('Reset')
				button.connect("clicked",change_date_planet_moon_plot,window,vbox,object,local,date_time_change_flag)
				table1.attach(button,column,column+3,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Text Size")
				else:
					button = gtk.Button("Ts")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				button = gtk.Button("-")
				text_size_change_flag = -1
				button.connect("clicked",set_font_size,text_size_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = column+1
				button = gtk.Button("+")
				text_size_change_flag = 1
				button.connect("clicked",set_font_size,text_size_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = 0
				row = 0
				column = column + 1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Moon Trails")
				else:
					button = gtk.ToggleButton("Mt")
				button.set_active(moon_trail_show_flag)
				button.connect("toggled", self.moon_trail_show_callback)
				table3.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")
				column = 0

				column = 0
				row = row + 1
				column = column + 1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Legend Text")
				else:
					button = gtk.ToggleButton("LgT")
				button.set_active(legend_show_flag)
				button.connect("toggled", self.legend_show_callback)
				table3.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")
				column = 0

				column = 1
				table.attach(table1,column,column+1,0,6)
				table1.show()

				column = 1
				table.attach(table3,column,column+1,6,7)
				table3.show()

		window.add(vbox)
		vbox.pack_start(table,False)

		window.show_all()

		if window_in_fullscreen == True:
			window.fullscreen()

		window.show()

		return

class plot_mars_moon(gtk.DrawingArea):
        def __init__(self,window,object,local):
                gtk.DrawingArea.__init__(self)
                self.connect("expose_event", self.expose_plot_mars_moon, object, local)
                self.connect('button_press_event', self.button_press_event_cb)
                self.set_events(gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.POINTER_MOTION_MASK)

                return

    	def button_press_event_cb(self, widget, event):
		global select_object

        	if event.button == 1:
			select_object = ""
					
			gtk.gdk.threads_enter
    			self.queue_draw()
			gtk.gdk.threads_leave

        	elif event.button == 3:
        		print event.x, event.y

        	return True

        def expose_plot_mars_moon(self, widget, event, object, local):
                self.context = widget.window.cairo_create()

                # set a clip region for the expose event
                self.context.rectangle(event.area.x, event.area.y,
                               event.area.width, event.area.height)
                self.context.clip()

                self.draw_plot_mars_moon(self.context,object,local)

                return False

    	def draw_plot_mars_moon(self, context,object,local):
		global rect
		global moon_trail

		if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
			local_time_zone_daylight_flag = 0
		else:
			local_time_zone_daylight_flag = time.daylight

		local_time_zone_name = time.tzname

        	rect = self.get_allocation()
        	start_radius = 100
        	x = rect.x + rect.width / 2
        	y = rect.y + rect.height / 2
		x = (rect.width/2) - (((rect.width/2) - x) * moon_field_of_view) 
 
        	context.rectangle(0, 0, rect.width, rect.height)
		context.set_source_rgba(0,0,0) # black rectangle
		context.fill()

		font = "Sans"
		context.select_font_face(font)
		context.set_font_size(font_size)

		planet_field_of_view = 15.0

		size = rect.width/planet_field_of_view * 1.0 * moon_field_of_view

		plot_color = mars_plot_color

		context.save() # required since altering scale
		context.arc(x, y, size, 0, 2 * math.pi)
		context.set_source_rgb(plot_color[0],plot_color[1],plot_color[2]) # planet plot color
		context.fill()
		context.stroke()
		context.restore() # required since altering scale

		moon = ephem.Phobos()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.00327 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("I")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			phobos_title = "[I]    " + moon.name + " (VISIBLE)"
		else:
			phobos_title = "[I]    " + moon.name + " (NOT VISIBLE)"

		moon = ephem.Deimos()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.00183 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("II")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			deimos_title = "[II]   " + moon.name + " (VISIBLE)"
		else:
			deimos_title = "[II]   " + moon.name + " (NOT VISIBLE)"

		if moon_trail_show_flag == True:
			for moon in moon_trail:
				moonx = moon[0]
				moony = moon[1]
				x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moonx) * moon_field_of_view)
				y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moony) * moon_field_of_view)
				context.arc(x, y, 1, 0, 2 * math.pi)
				context.set_source_rgb(1,1,1) # moon white color
				context.fill()
				context.stroke()

		current_local_date = ephem.Date(ephem.localtime(local.date))

		if date_in_gmt == True:
			date_title = 'DATE/TIME: '+str(local.date)+' GMT'
		else:
			date_title = 'DATE/TIME: '+str(current_local_date)+' '+local_time_zone_name[local_time_zone_daylight_flag]

		y = 25
		context.set_line_width(1.0)
		context.set_source_rgb(1,0.65,0) # date orange color
		context.move_to(0,y)
		context.text_path(date_title)
		context.fill()
		context.stroke()

		if legend_show_flag == True:
			context.set_source_rgb(1,0,0) # legend red color

			y = y+15
			context.move_to(0,y)
			context.text_path(phobos_title)
			context.fill()
			context.stroke()

			y = y+15
			context.move_to(0,y)
			context.text_path(deimos_title)
			context.fill()
			context.stroke()

class plot_jupiter_moon(gtk.DrawingArea):
        def __init__(self,window,object,local):
                gtk.DrawingArea.__init__(self)
                self.connect("expose_event", self.expose_plot_jupiter_moon, object, local)
                self.connect('button_press_event', self.button_press_event_cb)
                self.set_events(gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.POINTER_MOTION_MASK)

                return

    	def button_press_event_cb(self, widget, event):
        	if event.button == 1:
			select_object = ""
					
			gtk.gdk.threads_enter
    			self.queue_draw()
			gtk.gdk.threads_leave

        	elif event.button == 3:
        		print event.x, event.y

        	return True

        def expose_plot_jupiter_moon(self, widget, event, object, local):
                self.context = widget.window.cairo_create()

                # set a clip region for the expose event
                self.context.rectangle(event.area.x, event.area.y,
                               event.area.width, event.area.height)
                self.context.clip()

                self.draw_plot_jupiter_moon(self.context,object,local)

                return False

    	def draw_plot_jupiter_moon(self, context,object,local):
		global rect
		global moon_trail

		if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
			local_time_zone_daylight_flag = 0
		else:
			local_time_zone_daylight_flag = time.daylight

		local_time_zone_name = time.tzname

        	rect = self.get_allocation()
        	start_radius = 100
        	x = rect.x + rect.width / 2
        	y = rect.y + rect.height / 2
		x = (rect.width/2) - (((rect.width/2) - x) * moon_field_of_view) 
 
        	context.rectangle(0, 0, rect.width, rect.height)
		context.set_source_rgba(0,0,0) # black rectangle
		context.fill()

		font = "Sans"
		context.select_font_face(font)
		context.set_font_size(font_size)

		planet_field_of_view = 30.0

		size = rect.width/planet_field_of_view * 1.0 * moon_field_of_view

		plot_color = jupiter_plot_color

		context.save() # required since altering scale
		context.translate(x,y) # dare
		context.scale(1,0.9351) # dare
		context.arc(0, 0, size, 0, 2 * math.pi) # dare
		context.set_source_rgb(plot_color[0],plot_color[1],plot_color[2]) # planet plot color
		context.fill()
		context.stroke()
		context.restore() # required since restoring scale

		moon = ephem.Io()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.02737 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("I")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			io_title = "[I]    " + moon.name + " (VISIBLE)"
		else:
			io_title = "[I]    " + moon.name + " (NOT VISIBLE)"

		moon = ephem.Europa()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.02335 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("II")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			europa_title = "[II]   " + moon.name + " (VISIBLE)"
		else:
			europa_title = "[II]   " + moon.name + " (NOT VISIBLE)"

		moon = ephem.Ganymede()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.03936 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("III")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			ganymede_title = "[III]  " + moon.name + " (VISIBLE)"
		else:
			ganymede_title = "[III]  " + moon.name + " (NOT VISIBLE)"

		moon = ephem.Callisto()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.03605 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("IV")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			callisto_title = "[IV]   " + moon.name + " (VISIBLE)"
		else:
			callisto_title = "[IV]   " + moon.name + " (NOT VISIBLE)"
	
		if moon_trail_show_flag == True:
			for moon in moon_trail:
				moonx = moon[0]
				moony = moon[1]
				x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moonx) * moon_field_of_view)
				y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moony) * moon_field_of_view)
				context.arc(x, y, 1, 0, 2 * math.pi)
				context.set_source_rgb(1,1,1) # moon white color
				context.fill()
				context.stroke()

		current_local_date = ephem.Date(ephem.localtime(local.date))

		if date_in_gmt == True:
			date_title = 'DATE/TIME: '+str(local.date)+' GMT'
		else:
			date_title = 'DATE/TIME: '+str(current_local_date)+' '+local_time_zone_name[local_time_zone_daylight_flag]

		y = 25
		context.set_line_width(1.0)
		context.set_source_rgb(1,0.65,0) # date orange color
		context.move_to(0,y)
		context.text_path(date_title)
		context.fill()
		context.stroke()

		if legend_show_flag == True:
			context.set_source_rgb(1,0,0) # legend red color

			y = y+15
			context.move_to(0,y)
			context.text_path(io_title)
			context.fill()
			context.stroke()
			
			y = y+15
			context.move_to(0,y)
			context.text_path(europa_title)
			context.fill()
			context.stroke()
		
			y = y+15
			context.move_to(0,y)
			context.text_path(ganymede_title)
			context.fill()
			context.stroke()
	
			y = y+15
			context.move_to(0,y)
			context.text_path(callisto_title)
			context.fill()
			context.stroke()

class plot_saturn_moon(gtk.DrawingArea):
        def __init__(self,window,object,local):
                gtk.DrawingArea.__init__(self)
                self.connect("expose_event", self.expose_plot_saturn_moon, object, local)
                self.connect('button_press_event', self.button_press_event_cb)
                self.set_events(gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.POINTER_MOTION_MASK)

                return

    	def button_press_event_cb(self, widget, event):
        	if event.button == 1:
			select_object = ""
					
			gtk.gdk.threads_enter
    			self.queue_draw()
			gtk.gdk.threads_leave

        	elif event.button == 3:
        		print event.x, event.y

        	return True

        def expose_plot_saturn_moon(self, widget, event, object, local):
                self.context = widget.window.cairo_create()

                # set a clip region for the expose event
                self.context.rectangle(event.area.x, event.area.y,
                               event.area.width, event.area.height)
                self.context.clip()

                self.draw_plot_saturn_moon(self.context,object,local)

                return False

    	def draw_plot_saturn_moon(self, context,object,local):
		global rect
		global moon_trail

		if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
			local_time_zone_daylight_flag = 0
		else:
			local_time_zone_daylight_flag = time.daylight

		local_time_zone_name = time.tzname

        	rect = self.get_allocation()
        	start_radius = 100
        	x = rect.x + rect.width / 2
        	y = rect.y + rect.height / 2
		x = (rect.width/2) - (((rect.width/2) - x) * moon_field_of_view) 
 
        	context.rectangle(0, 0, rect.width, rect.height)
		context.set_source_rgba(0,0,0) # black rectangle
		context.fill()

		font = "Sans"
		context.select_font_face(font)
		context.set_font_size(font_size)

		planet_field_of_view = 110.0

		plot_color = saturn_plot_color

		context.save() # required since altering scale
		context.translate(x,y) # dare
		context.scale(1,0.9020) # dare
		context.arc(0, 0, rect.width/planet_field_of_view * moon_field_of_view, 0, 2 * math.pi) # dare
		context.set_source_rgb(plot_color[0],plot_color[1],plot_color[2]) # planet plot color
		context.fill()
		context.stroke()
		context.restore() #required since restoring scale

		object.compute(local)
		ring_tilt_scale = sin(object.earth_tilt)

		context.save() # required since altering scale
		context.translate(x,y) # dare
		context.scale(1,ring_tilt_scale) # dare
		context.arc(0, 0, rect.width/planet_field_of_view * moon_field_of_view * 2* 1.0138, 0, 2 * math.pi) # dare
		context.set_source_rgb(0.8,0.8,0) # saturn redish color
		context.stroke()
		context.restore() #required since restoring scale

		context.save() # required since altering scale
		context.translate(x,y) # dare
		context.scale(1,ring_tilt_scale) # dare
		context.arc(0, 0, rect.width/planet_field_of_view * moon_field_of_view * 2* 1.1349, 0, 2 * math.pi) # dare
		context.set_source_rgb(0.8,0.8,0) # saturn redish color
		context.stroke()
		context.restore() #required since restoring scale

		context.save() # required since altering scale
		context.translate(x,y) # dare
		context.scale(1,ring_tilt_scale) # dare
		context.arc(0, 0, rect.width/planet_field_of_view * moon_field_of_view * 2* 1.0743, 0, 2 * math.pi) # dare
		ring_line_size = rect.width/planet_field_of_view * moon_field_of_view * 2 * 0.1211
		context.set_line_width(ring_line_size)
		context.set_source_rgba(0.8,0.8,0,0.3) # saturn redish color
		context.stroke()
		context.restore() #required since restoring scale

		context.save() # required since altering scale
		context.translate(x,y) # dare
		context.scale(1,ring_tilt_scale) # dare
		context.arc(0, 0, rect.width/planet_field_of_view * moon_field_of_view * 2* 0.7633, 0, 2 * math.pi) # dare
		context.set_source_rgb(0.8,0.8,0) # saturn redish color
		context.stroke()
		context.restore() #required since restoring scale

		context.save() # required since altering scale
		context.translate(x,y) # dare
		context.scale(1,ring_tilt_scale) # dare
		context.arc(0, 0, rect.width/planet_field_of_view * moon_field_of_view * 2* 0.8690, 0, 2 * math.pi) # dare
		ring_line_size = rect.width/planet_field_of_view * moon_field_of_view * 2 * 0.2115
		context.set_line_width(ring_line_size)
		context.set_source_rgba(0.8,0.8,0,0.3) # saturn redish color
		context.stroke()
		context.restore() #required since restoring scale

		context.save() # required since altering scale
		context.translate(x,y) # dare
		context.scale(1,ring_tilt_scale) # dare
		context.arc(0, 0, rect.width/planet_field_of_view * moon_field_of_view * 2* 0.9748, 0, 2 * math.pi) # dare
		ring_line_size = rect.width/planet_field_of_view * moon_field_of_view * 2 * 0.4230
		ring_line_size = rect.width/planet_field_of_view * moon_field_of_view * 2 * 0.2
		context.set_source_rgb(0.8,0.8,0) # saturn redish color
		context.stroke()
		context.restore() #required since restoring scale

		moon = ephem.Mimas()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.00332 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("I")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			mimas_title = "[I]    " + moon.name + " (VISIBLE)"
		else:
			mimas_title = "[I]    " + moon.name + " (NOT VISIBLE)"

		moon = ephem.Enceladus()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.00415 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("II")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			enceladus_title = "[II]   " + moon.name + " (VISIBLE)"
		else:
			enceladus_title = "[II]   " + moon.name + " (NOT VISIBLE)"

		moon = ephem.Tethys()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.00879 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("III")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			tethys_title = "[III]  " + moon.name + " (VISIBLE)"
		else:
			tethys_title = "[III]  " + moon.name + " (NOT VISIBLE)"

		moon = ephem.Dione()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.00929 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("IV")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			dione_title = "[IV]   " + moon.name + " (VISIBLE)"
		else:
			dione_title = "[IV]   " + moon.name + " (NOT VISIBLE)"

		moon = ephem.Rhea()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.01269 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("V")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			rhea_title = "[V]    " + moon.name + " (VISIBLE)"
		else:
			rhea_title = "[V]    " + moon.name + " (NOT VISIBLE)"

		moon = ephem.Titan()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.04273 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("VI")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			titan_title = "[VI]   " + moon.name + " (VISIBLE)"
		else:
			titan_title = "[VI]   " + moon.name + " (NOT VISIBLE)"

		moon = ephem.Hyperion()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.00242 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("VII")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			hyperion_title = "[VII]  " + moon.name + " (VISIBLE)"
		else:
			hyperion_title = "[VII]  " + moon.name + " (NOT VISIBLE)"

		moon = ephem.Iapetus()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.01195*moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("VIII")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			iapetus_title = "[VIII] " + moon.name + " (VISIBLE)"
		else:
			iapetus_title = "[VIII] " + moon.name + " (NOT VISIBLE)"

		if moon_trail_show_flag == True:
			for moon in moon_trail:
				moonx = moon[0]
				moony = moon[1]
				x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moonx) * moon_field_of_view)
				y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moony) * moon_field_of_view)
				context.arc(x, y, 1, 0, 2 * math.pi)
				context.set_source_rgb(1,1,1) # moon white color
				context.fill()
				context.stroke()
		current_local_date = ephem.Date(ephem.localtime(local.date))

		if date_in_gmt == True:
			date_title = 'DATE/TIME: '+str(local.date)+' GMT'
		else:
			date_title = 'DATE/TIME: '+str(current_local_date)+' '+local_time_zone_name[local_time_zone_daylight_flag]

		y = 25
		context.set_line_width(1.0)
		context.set_source_rgb(1,0.65,0) # date orange color
		context.move_to(0,y)
		context.text_path(date_title)
		context.fill()
		context.stroke()

		if legend_show_flag == True:
			context.set_source_rgb(1,0,0) # legend red color

			y = y+15
			context.move_to(0,y)
			context.text_path(mimas_title)
			context.fill()
			context.stroke()
			
			y = y+15
			context.move_to(0,y)
			context.text_path(enceladus_title)
			context.fill()
			context.stroke()
		
			y = y+15
			context.move_to(0,y)
			context.text_path(tethys_title)
			context.fill()
			context.stroke()
		
			y = y+15
			context.move_to(0,y)
			context.text_path(dione_title)
			context.fill()
			context.stroke()
		
			y = y+15
			context.move_to(0,y)
			context.text_path(rhea_title)
			context.fill()
			context.stroke()
		
			y = y+15
			context.move_to(0,y)
			context.text_path(titan_title)
			context.fill()
			context.stroke()
		
			y = y+15
			context.move_to(0,y)
			context.text_path(hyperion_title)
			context.fill()
			context.stroke()
		
			y = y+15
			context.move_to(0,y)
			context.text_path(iapetus_title)
			context.fill()
			context.stroke()

class plot_uranus_moon(gtk.DrawingArea):
        def __init__(self,window,object,local):
                gtk.DrawingArea.__init__(self)
                self.connect("expose_event", self.expose_plot_uranus_moon, object, local)
                self.connect('button_press_event', self.button_press_event_cb)
                self.set_events(gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.POINTER_MOTION_MASK)

                return

    	def button_press_event_cb(self, widget, event):
        	if event.button == 1:
			select_object = ""
					
			gtk.gdk.threads_enter
    			self.queue_draw()
			gtk.gdk.threads_leave

        	elif event.button == 3:
        		print event.x, event.y

        	return True

        def expose_plot_uranus_moon(self, widget, event, object, local):
                self.context = widget.window.cairo_create()

                # set a clip region for the expose event
                self.context.rectangle(event.area.x, event.area.y,
                               event.area.width, event.area.height)
                self.context.clip()

                self.draw_plot_uranus_moon(self.context,object,local)

                return False

    	def draw_plot_uranus_moon(self, context,object,local):
		global rect
		global moon_trail

		if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
			local_time_zone_daylight_flag = 0
		else:
			local_time_zone_daylight_flag = time.daylight

		local_time_zone_name = time.tzname

        	rect = self.get_allocation()
        	start_radius = 100
        	x = rect.x + rect.width / 2
        	y = rect.y + rect.height / 2
		x = (rect.width/2) - (((rect.width/2) - x) * moon_field_of_view) 
 
        	context.rectangle(0, 0, rect.width, rect.height)
		context.set_source_rgba(0,0,0) # black rectangle
		context.fill()

		font = "Sans"
		context.select_font_face(font)
		context.set_font_size(font_size)

		planet_field_of_view = 50.0

		size = rect.width/planet_field_of_view * 1.0 * moon_field_of_view

		plot_color = uranus_plot_color

		context.arc(x, y, size, 0, 2 * math.pi)
		context.set_source_rgb(plot_color[0],plot_color[1],plot_color[2]) # planet plot color
		context.fill()
		context.stroke()

		moon = ephem.Ariel()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.04530 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("I")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			ariel_title = "[I]    " + moon.name + " (VISIBLE)"
		else:
			ariel_title = "[I]    " + moon.name + " (NOT VISIBLE)"

		moon = ephem.Umbriel()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.04575 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("II")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			umbriel_title = "[II]   " + moon.name + " (VISIBLE)"
		else:
			umbriel_title = "[II]   " + moon.name + " (NOT VISIBLE)"

		moon = ephem.Titania()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.06173 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("III")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			titania_title = "[III]  " + moon.name + " (VISIBLE)"
		else:
			titania_title = "[III]  " + moon.name + " (NOT VISIBLE)"

		moon = ephem.Oberon()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.06001 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("IV")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			oberon_title = "[IV]   " + moon.name + " (VISIBLE)"
		else:
			oberon_title = "[IV]   " + moon.name + " (NOT VISIBLE)"

		moon = ephem.Miranda()
		moon.compute(local)

		size = rect.width/planet_field_of_view * 0.01845 * moon_field_of_view
		if size < 1.0:
			size = 1.0

		if moon.earth_visible == 1.0:
			x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moon.x) * moon_field_of_view)
			y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moon.y) * moon_field_of_view)

			if moon_trail_show_flag == True:
				moon_trail.append([moon.x, moon.y])

			context.set_source_rgb(1,1,1) # moon white color
			context.move_to(x-5, y+15)
			context.text_path("V")
			context.set_line_width(1.0)
			context.fill()
			context.stroke()
			context.arc(x, y, size, 0, 2 * math.pi)
			context.set_source_rgb(1,1,1) # moon white color
			context.fill()
			context.stroke()
			miranda_title = "[V]    " + moon.name + " (VISIBLE)"
		else:
			miranda_title = "[V]    " + moon.name + " (NOT VISIBLE)"

		if moon_trail_show_flag == True:
			for moon in moon_trail:
				moonx = moon[0]
				moony = moon[1]
				x = ((rect.width/2) + (rect.height/planet_field_of_view * 2.0 * -moonx) * moon_field_of_view)
				y = ((rect.height/2) - (rect.height/planet_field_of_view * 2.0 * -moony) * moon_field_of_view)
				context.arc(x, y, 1, 0, 2 * math.pi)
				context.set_source_rgb(1,1,1) # moon white color
				context.fill()
				context.stroke()

		current_local_date = ephem.Date(ephem.localtime(local.date))

		if date_in_gmt == True:
			date_title = 'DATE/TIME: '+str(local.date)+' GMT'
		else:
			date_title = 'DATE/TIME: '+str(current_local_date)+' '+local_time_zone_name[local_time_zone_daylight_flag]

		y = 25
		context.set_line_width(1.0)
		context.set_source_rgb(1,0.65,0) # date orange color
		context.move_to(0,y)
		context.text_path(date_title)
		context.fill()
		context.stroke()
		if legend_show_flag == True:
			context.set_source_rgb(1,0,0) # legend red color

			y = y+15
			context.move_to(0,y)
			context.text_path(ariel_title)
			context.fill()
			context.stroke()

			y = y+15
			context.move_to(0,y)
			context.text_path(umbriel_title)
			context.fill()
			context.stroke()
			y = y+15

			context.move_to(0,y)
			context.text_path(titania_title)
			context.fill()
			context.stroke()
			y = y+15

			context.move_to(0,y)
			context.text_path(oberon_title)
			context.fill()
			context.stroke()
			y = y+15

			context.move_to(0,y)
			context.text_path(miranda_title)
			context.fill()
			context.stroke()

def plot_object_rise_set_graph_start(widget,object,local):
        if hildon_flag == 1:
                window = hildon.Window()
        else:
                window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
        window.connect("key-press-event", on_key_press_plot_object_rise_set_graph, window, object, local)

	if rise_set_plot_flag == True:
		window.set_title('Planet Ephemeris for ' + object.name + ' in ' + str(local.date.tuple()[0]) + ': RISE/SET PLOT')
	else:
		window.set_title('Planet Ephemeris for ' + object.name + ' in ' + str(local.date.tuple()[0]) + ' at 12:00 noon: ANALEMMA')

	window.set_size_request(800,480)

	if hildon_flag == True and rise_set_flag == True:
		on_show_note(widget,window,"Please wait while lengthy rise/set calculations are made","OK")

	vbox = gtk.HBox(False, 0)

	plot = plot_object_rise_set_graph(window,object,local)
	vbox.pack_start(plot,True)
	plot.show()

	vbox.show()
	window.add(vbox)
	window.show()

        if window_in_fullscreen == True:
                window.fullscreen()

	return

class plot_object_rise_set_graph(gtk.DrawingArea):
        def __init__(self,window,object,local):
                gtk.DrawingArea.__init__(self)
                self.connect("expose_event", self.expose_plot_graph, object, local)
                self.set_events(gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.POINTER_MOTION_MASK)

                return

        def expose_plot_graph(self, widget, event, object, local):
                self.context = widget.window.cairo_create()

                # set a clip region for the expose event
                self.context.rectangle(event.area.x, event.area.y,
                               event.area.width, event.area.height)
                self.context.clip()

                self.draw_plot_object_rise_set_graph(self.context,object,local)

                return False

    	def draw_plot_object_rise_set_graph(self, context,object,local):
		global rect

		if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
			local_time_zone_daylight_flag = 0
		else:
			local_time_zone_daylight_flag = time.daylight

		local_time_zone_name = time.tzname

        	rect = self.get_allocation()

        	x = rect.x + rect.width / 2
        	y = rect.y + rect.height / 2
    
                context.rectangle(0, 0, rect.width, rect.height)

		y_axis_margin = 30
		x_axis_margin = 15

		save = local.date #still required for 3.7.3.2

		local.date = save
	
		current_local_date = ephem.date(ephem.localtime(local.date)).tuple()
		current_local_year = current_local_date[0]
		current_local_month = current_local_date[1]
		current_local_day = current_local_date[2]
		current_local_hour = current_local_date[3]
		current_local_minute = current_local_date[4]
		current_local_second = current_local_date[5]
		current_local_time = current_local_hour + (current_local_minute/60.0) + (current_local_second/60.0/60.0)

		if rise_set_plot_flag == False:
			start_of_year = ephem.date(str(local.date.tuple()[0])+'/01/01 12:00') #reset to start of year GMT at noon
			local.date = start_of_year
			timezone_offset = ephem.date(local.date)-ephem.date(ephem.localtime(local.date))
			local.date = local.date+timezone_offset # compensate for differing time zone

			object_min_alt = 90.0
			object_max_alt = -90.0
			object_min_az = 360.0
			object_max_az = 0.0
		else:
			solar_time_zone = ((float(local.long)*180/pi)/15) - int((float(local.long)*180/pi)/15)

			start_of_year = ephem.date(str(local.date.tuple()[0])+'/01/01 00:00') #reset to start of year GMT at midnight
			local.date = start_of_year
			timezone_offset = ephem.date(local.date)-ephem.date(ephem.localtime(local.date))
			local.date = local.date+timezone_offset # compensate for differing time zone

			context.set_source_rgba(1,1,1) # white rectangle
			context.fill()

			context.set_source_rgb(0,0,1) # azimuth grid blue color
			font = "Sans"
			context.select_font_face(font)
			context.set_font_size(font_size)

			for i in range(24):
				y = (rect.height - x_axis_margin) - ((rect.height - x_axis_margin)/24.0 * i)
				context.move_to(0, y+4)
				context.text_path(str(i).zfill(2)+":00")
				context.set_line_width(1.0)
				context.fill()
				context.stroke()
				context.move_to(y_axis_margin, y)
				context.line_to(rect.width, y)
				context.set_line_width(0.5)
				context.stroke()

			save3 = local.date #still required for 3.7.3.2

			for i in range(12):
				day_of_year = int(float(local.date) - float(start_of_year)) + 1
				x = ((rect.width - y_axis_margin)/366.0 * day_of_year) + y_axis_margin
				y = rect.height 
				context.set_source_rgb(0,0,1) # legend blue color
				context.move_to(x-4, y)
				context.text_path(calendar.month_name[local.date.triple()[1]])
				context.set_line_width(1.0)
				context.fill()
				context.stroke()
				y = rect.height - x_axis_margin
				context.move_to(x, y)
				context.line_to(x, 0)
				context.set_line_width(0.5)
				context.stroke()
				current_date = (local.date.triple()[0],local.date.triple()[1]+1,local.date.triple()[2])
				local.date = current_date	

			local.date = save3

		for i in range(1, 366, 2):
			day_of_year = int(float(local.date) - float(start_of_year)) + 1

			object.compute(local)

			if rise_set_plot_flag == False:
				if object.alt < object_min_alt:
					object_min_alt = object.alt
				if object.alt > object_max_alt:
					object_max_alt = object.alt
				if object.az < object_min_az:
					object_min_az = object.az
				if object.az > object_max_az:
					object_max_az = object.az
			else:		
				save2 = local.date #still required for 3.7.3.2
		
				object_next_antitransit = local.next_antitransit(object)
				local.date = save2
				object_next_rising = local.next_rising(object)
				local.date = save2
				object_next_transit = local.next_transit(object)
				local.date = save2
				object_next_setting = local.next_setting(object)
				local.date = save2

				object_local_datetime = str(ephem.localtime(local.date))[:19]
				object_local_next_antitransit = str(ephem.localtime(object_next_antitransit))[:19]
				object_local_next_rising = str(ephem.localtime(object_next_rising))[:19]
				object_local_next_transit = str(ephem.localtime(object_next_transit))[:19]
				object_local_next_setting = str(ephem.localtime(object_next_setting))[:19]

				if date_in_gmt == True:
					object_next_antitransit_show = str(object_next_antitransit)[:19]+' GMT'
					object_next_rising_show = str(object_next_rising)[:19]+' GMT'
					object_next_setting_show = str(object_next_setting)[:19]+' GMT'
					object_next_transit_show = str(object_next_transit)[:19]+' GMT'
				else:
					object_next_antitransit_show = str(ephem.localtime(object_next_antitransit))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
					object_next_rising_show = str(ephem.localtime(object_next_rising))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
					object_next_setting_show = str(ephem.localtime(object_next_setting))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
					object_next_transit_show = str(ephem.localtime(object_next_transit))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]

				day_length = (object_next_setting-object_next_rising)*24
				if day_length < 0:
					day_length = 24.0 + day_length

				object_next_antitransit_hour = ephem.Date(object_local_next_antitransit).triple()[2] - int(ephem.Date(object_local_next_antitransit).triple()[2])
				object_next_rising_hour = ephem.Date(object_local_next_rising).triple()[2] - int(ephem.Date(object_local_next_rising).triple()[2])
				object_next_setting_hour = ephem.Date(object_local_next_setting).triple()[2] - int(ephem.Date(object_local_next_setting).triple()[2])
				object_next_transit_hour = ephem.Date(object_local_next_transit).triple()[2] - int(ephem.Date(object_local_next_transit).triple()[2])

				context.set_line_width(1)

				y = (rect.height - x_axis_margin) - ((rect.height - x_axis_margin)/24.0 * day_length) 
				x = ((rect.width - y_axis_margin)/366.0 * day_of_year) + y_axis_margin

				context.arc(x, y, 1, 0, 2 * math.pi)
				context.set_source_rgb(1,0,1) # magenta daylength color
				context.fill()
				context.stroke()

				y = (rect.height - x_axis_margin) - ((rect.height - x_axis_margin) * object_next_antitransit_hour)

				context.arc(x, y, 1, 0, 2 * math.pi)
				context.set_source_rgb(0,1,1) # cyan antitransit color
				context.fill()
				context.stroke()

				y = (rect.height - x_axis_margin) - ((rect.height - x_axis_margin) * object_next_rising_hour)

				context.arc(x, y, 1, 0, 2 * math.pi)
				context.set_source_rgb(1,0,0) # red rise color
				context.fill()
				context.stroke()

				y = (rect.height - x_axis_margin) - ((rect.height - x_axis_margin) * object_next_setting_hour)

				context.arc(x, y, 1, 0, 2 * math.pi)
				context.set_source_rgb(0,0,1) # blue set color
				context.fill()
				context.stroke()

				y = (rect.height - x_axis_margin) - ((rect.height - x_axis_margin) * object_next_transit_hour)

				context.arc(x, y, 1, 0, 2 * math.pi)
				context.set_source_rgb(1,1,0) # yellow transit color
				context.fill()
				context.stroke()

				loop_current_local_date = local.date.tuple()
				loop_current_local_year = loop_current_local_date[0]
				loop_current_local_month = loop_current_local_date[1]
				loop_current_local_day = loop_current_local_date[2]

			local.date = local.date + 2

		local.date = save #still required for 3.7.3.2

		if rise_set_plot_flag == False:
			y_axis_margin = 20
			x_axis_margin = 15

			context.set_source_rgba(1,1,1) # white rectangle
			context.fill()

			context.set_source_rgb(0,0,1) # alt,az blue color
			font = "Sans"
			context.select_font_face(font)
			context.set_font_size(font_size)

			object_min_alt = float(object_min_alt)*180/pi
			object_max_alt = float(object_max_alt)*180/pi
			object_min_az = float(object_min_az)*180/pi
			object_max_az = float(object_max_az)*180/pi

			alt_range = int(object_max_alt-object_min_alt+1)
			az_range = int(object_max_az-object_min_az+1)

			for i in range(alt_range):
				context.set_line_width(1.0)
				y = (rect.height - x_axis_margin) - ((rect.height - x_axis_margin)/float(alt_range) * i)
				context.move_to(0, y+4)
				context.text_path(str(i+int(object_min_alt))+u'\xb0')
				context.set_line_width(0.5)
				context.fill()
				context.stroke()
				context.move_to(y_axis_margin, y)
				context.line_to(rect.width, y)
				context.stroke()

			for i in range(az_range):
				context.set_line_width(1.0)
				x = ((rect.width - y_axis_margin)/float(az_range) * i) + y_axis_margin
				y = rect.height 
				context.move_to(x-4, y)
				context.text_path(str(i+int(object_min_az))+u'\xb0')
				context.set_line_width(0.5)
				context.fill()
				context.stroke()
				y = rect.height - x_axis_margin
				context.move_to(x, y)
				context.line_to(x, 0)
				context.stroke()

			context.set_line_width(1)

			start_of_year = ephem.date(str(local.date.tuple()[0])+'/01/01 12:00') #reset to start of year GMT
			local.date = start_of_year
			timezone_offset = ephem.date(local.date)-ephem.date(ephem.localtime(local.date))
			local.date = local.date+timezone_offset # compensate for differing time zone

			spring = ephem.next_equinox(ephem.date(str(local.date.tuple()[0])))
			summer = ephem.next_solstice(spring)
			fall = ephem.next_equinox(summer)
			winter = ephem.next_solstice(fall)

			spring_day_of_year = int(float(spring) - float(start_of_year)) +1
			summer_day_of_year = int(float(summer) - float(start_of_year)) +1
			fall_day_of_year = int(float(fall) - float(start_of_year)) +1
			winter_day_of_year = int(float(winter) - float(start_of_year)) +1

			local_spring = str(ephem.localtime(spring))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
			local_summer = str(ephem.localtime(summer))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
			local_fall = str(ephem.localtime(fall))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
			local_winter = str(ephem.localtime(winter))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]

			for i in range(1, 366):
				day_of_year = int(float(local.date) - float(start_of_year)) + 1

				object.compute(local)

				object_alt = float(object.alt)*180/pi
				object_az = float(object.az)*180/pi

				y = (rect.height - x_axis_margin) - ((rect.height - x_axis_margin)/float(alt_range) * ((object_alt - object_min_alt)))
				x = ((rect.width - y_axis_margin)/float(az_range) * ((object_az - object_min_az))) + y_axis_margin

				context.arc(x, y, 1, 0, 2 * math.pi)
				context.set_source_rgb(1,0,0) # red analemma color
				context.fill()
				context.stroke()

				if day_of_year == spring_day_of_year:
					context.move_to(x+4, y+4)
					context.text_path("SPRING: " + local_spring)
					context.set_line_width(1.0)
					context.set_source_rgb(0,0,0) # black text color
					context.fill()
					context.stroke()
					context.arc(x, y, 3, 0, 2 * math.pi)
					context.set_source_rgb(0,1,0) # green plot color
					context.fill()
					context.stroke()
				if day_of_year == summer_day_of_year:
					context.move_to(x+4, y+4)
					context.text_path("SUMMER: " + local_summer)
					context.set_line_width(1.0)
					context.set_source_rgb(0,0,0) # black text color
					context.fill()
					context.stroke()
					context.arc(x, y, 3, 0, 2 * math.pi)
					context.set_source_rgb(0,1,0) # green plot color
					context.fill()
					context.stroke()
				if day_of_year == fall_day_of_year:
					context.move_to(x+4, y+4)
					context.text_path("FALL: " + local_fall)
					context.set_line_width(1.0)
					context.set_source_rgb(0,0,0) # black text color
					context.fill()
					context.stroke()
					context.arc(x, y, 3, 0, 2 * math.pi)
					context.set_source_rgb(0,1,0) # green plot color
					context.fill()
					context.stroke()
				if day_of_year == winter_day_of_year:
					context.move_to(x+4, y+4)
					context.text_path("WINTER: " + local_winter)
					context.set_line_width(1.0)
					context.set_source_rgb(0,0,0) # black text color
					context.fill()
					context.stroke()
					context.arc(x, y, 3, 0, 2 * math.pi)
					context.set_source_rgb(0,1,0) # green plot color
					context.fill()
					context.stroke()

				if local.date.tuple()[2] == 1.0:
					context.move_to(x+4, y+4)
					#context.text_path(str(local.date.triple()[1]))
					context.text_path(calendar.month_name[local.date.triple()[1]])
					context.set_line_width(1.0)
					context.set_source_rgb(0,0,0) # black text color
					context.fill()
					context.stroke()
					context.arc(x, y, 3, 0, 2 * math.pi)
					context.set_source_rgb(0,0,1) # blue plot color
					context.fill()
					context.stroke()

				loop_current_local_date = local.date.tuple()
				loop_current_local_year = loop_current_local_date[0]
				loop_current_local_month = loop_current_local_date[1]
				loop_current_local_day = loop_current_local_date[2]

				local.date = local.date + 1

			local.date = save #still required for 3.7.3.2
			day_of_year = int(float(local.date) - float(start_of_year)) + 1
			start_of_year = ephem.date(str(local.date.tuple()[0])+'/01/01 12:00') #reset to start of year GMT at noon
			local.date = start_of_year
			timezone_offset = ephem.date(local.date)-ephem.date(ephem.localtime(local.date))
			local.date = local.date+timezone_offset # compensate for differing time zone
			local.date = local.date + day_of_year

			object.compute(local)
			
			object_alt = float(object.alt)*180/pi
			object_az = float(object.az)*180/pi

			y = (rect.height - x_axis_margin) - ((rect.height - x_axis_margin)/float(alt_range) * ((object_alt - object_min_alt)))
			x = ((rect.width - y_axis_margin)/float(az_range) * ((object_az - object_min_az))) + y_axis_margin

			context.move_to(x+4, y+4)
			context.text_path("TODAY")
			context.set_line_width(1.0)
			context.set_source_rgb(0,0,0) # black text color
			context.fill()
			context.stroke()
			context.arc(x, y, 5, 0, 2 * math.pi)
			context.set_source_rgb(1,0,1) # magenta dot plot color
			context.fill()
			context.stroke()
		else:
			local.date = save
			timezone_offset = ephem.date(local.date)-ephem.date(ephem.localtime(local.date))
			local.date = local.date-timezone_offset # compensate for differing time zone

			day_of_year = int(float(local.date) - float(start_of_year)) + 1
			start_of_year = ephem.date(str(local.date.tuple()[0])+'/01/01 00:00') #reset to start of year GMT at midnight
			local.date = start_of_year
			timezone_offset = ephem.date(local.date)-ephem.date(ephem.localtime(local.date))
			local.date = local.date+timezone_offset # compensate for differing time zone
			local.date = local.date + day_of_year - 1

			object.compute(local)

			object_next_antitransit = local.next_antitransit(object)
			object_next_rising = local.next_rising(object)
			object_next_transit = local.next_transit(object)
			object_next_setting = local.next_setting(object)

			object_local_datetime = str(ephem.localtime(local.date))[:19]
			object_local_next_antitransit = str(ephem.localtime(object_next_antitransit))[:19]
			object_local_next_rising = str(ephem.localtime(object_next_rising))[:19]
			object_local_next_transit = str(ephem.localtime(object_next_transit))[:19]
			object_local_next_setting = str(ephem.localtime(object_next_setting))[:19]

			if date_in_gmt == True:
				object_next_antitransit_show = str(object_next_antitransit)[:19]+' GMT'
				object_next_rising_show = str(object_next_rising)[:19]+' GMT'
				object_next_setting_show = str(object_next_setting)[:19]+' GMT'
				object_next_transit_show = str(object_next_transit)[:19]+' GMT'
			else:
				object_next_antitransit_show = str(ephem.localtime(object_next_antitransit))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
				object_next_rising_show = str(ephem.localtime(object_next_rising))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
				object_next_setting_show = str(ephem.localtime(object_next_setting))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
				object_next_transit_show = str(ephem.localtime(object_next_transit))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]

			day_length = (object_next_setting-object_next_rising)*24
			if day_length < 0:
				day_length = 24.0 + day_length

			object_next_antitransit_hour = ephem.Date(object_local_next_antitransit).triple()[2] - int(ephem.Date(object_local_next_antitransit).triple()[2])
			object_next_rising_hour = ephem.Date(object_local_next_rising).triple()[2] - int(ephem.Date(object_local_next_rising).triple()[2])
			object_next_setting_hour = ephem.Date(object_local_next_setting).triple()[2] - int(ephem.Date(object_local_next_setting).triple()[2])
			object_next_transit_hour = ephem.Date(object_local_next_transit).triple()[2] - int(ephem.Date(object_local_next_transit).triple()[2])

			x = ((rect.width - y_axis_margin)/366.0 * day_of_year) + y_axis_margin
			y = (rect.height - x_axis_margin) - ((rect.height - x_axis_margin)/24.0 * current_local_time)

			context.move_to(x-35, y+4)
			context.text_path("NOW")
			context.set_line_width(1.0)
			context.set_source_rgb(0,0,0) # black current time color
			context.fill()
			context.stroke()
			context.arc(x, y, 5, 0, 2 * math.pi)
			context.set_source_rgb(1,0,1) # magenta dot plot color
			context.fill()
			context.stroke()

			y2 = rect.height - x_axis_margin
			context.move_to(x, y2)
			context.line_to(x, 0)
			context.stroke()

			x2 = rect.width - y_axis_margin
			context.move_to(y_axis_margin, y)
			context.line_to(rect.width, y)
			context.stroke()

			if x > (rect.width-150):
				print "too close to edge!", x
				x = x-150

			y = (rect.height - x_axis_margin) - ((rect.height - x_axis_margin) * object_next_antitransit_hour)

			context.move_to(x+4, y+4)
			context.text_path("ANTITRANSIT: " + object_next_antitransit_show)
			context.set_line_width(1.0)
			context.set_source_rgb(0,0,0) # black current time color
			context.fill()
			context.stroke()
			context.arc(x, y, 3, 0, 2 * math.pi)
			context.set_source_rgb(0,1,0) # blue plot color
			context.fill()
			context.stroke()

			y = (rect.height - x_axis_margin) - ((rect.height - x_axis_margin) * object_next_rising_hour)

			context.move_to(x+4, y)
			context.text_path("RISE: " + object_next_rising_show)
			context.set_line_width(1.0)
			context.set_source_rgb(0,0,0) # black current time color
			context.fill()
			context.stroke()
			context.arc(x, y, 3, 0, 2 * math.pi)
			context.set_source_rgb(0,1,0) # blue plot color
			context.fill()
			context.stroke()

			y = (rect.height - x_axis_margin) - ((rect.height - x_axis_margin) * object_next_transit_hour)

			context.move_to(x+4, y+4)
			context.text_path("TRANSIT: " + object_next_transit_show)
			context.set_line_width(1.0)
			context.set_source_rgb(0,0,0) # black current time color
			context.fill()
			context.stroke()
			context.arc(x, y, 3, 0, 2 * math.pi)
			context.set_source_rgb(0,1,0) # blue plot color
			context.fill()
			context.stroke()

			y = (rect.height - x_axis_margin) - ((rect.height - x_axis_margin) * object_next_setting_hour)

			context.move_to(x+4, y+4)
			context.text_path("SET: " + object_next_setting_show)
			context.set_line_width(1.0)
			context.set_source_rgb(0,0,0) # black current time color
			context.fill()
			context.stroke()
			context.arc(x, y, 3, 0, 2 * math.pi)
			context.set_source_rgb(0,1,0) # blue plot color
			context.fill()
			context.stroke()

		local.date = save #still required for 3.7.3.2

def plot_sun_angle_graph_start(widget,object,local):
        if hildon_flag == 1:
                window = hildon.Window()
        else:
                window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
        window.connect("key-press-event", on_key_press_plot_sun_angle_graph, window, object, local)

	window.set_title('Planet Ephemeris for ' + object.name + ' in ' + str(local.date.tuple()[0]) + ': ANGLE PLOT')

	window.set_size_request(800,480)

	#if hildon_flag == True:
	#	on_show_note(widget,window,"Please wait while lengthy angle calculations are made","OK")

	vbox = gtk.HBox(False, 0)

	plot = plot_sun_angle_graph(window,object,local)
	vbox.pack_start(plot,True)
	plot.show()

	vbox.show()
	window.add(vbox)
	window.show()

        if window_in_fullscreen == True:
                window.fullscreen()

	return

class plot_sun_angle_graph(gtk.DrawingArea):
        def __init__(self,window,object,local):
                gtk.DrawingArea.__init__(self)
                self.connect("expose_event", self.expose_plot_sun_angle_graph, object, local)
                self.set_events(gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.POINTER_MOTION_MASK)

                return

        def expose_plot_sun_angle_graph(self, widget, event, object, local):
                self.context = widget.window.cairo_create()

                # set a clip region for the expose event
                self.context.rectangle(event.area.x, event.area.y,
                               event.area.width, event.area.height)
                self.context.clip()

                self.draw_plot_sun_angle_graph(self.context,object,local)

                return False

    	def draw_plot_sun_angle_graph(self, context,object,local):
		global rect

		if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
			local_time_zone_daylight_flag = 0
		else:
			local_time_zone_daylight_flag = time.daylight

		local_time_zone_name = time.tzname

        	rect = self.get_allocation()

        	x = rect.x + rect.width / 2
        	y = rect.y + rect.height / 2
     
                context.rectangle(0, 0, rect.width, rect.height)

		y_axis_margin = 30
		x_axis_margin = 15

		save = local.date #still required for 3.7.3.2

		timezone_offset = ephem.date(local.date)-ephem.date(ephem.localtime(local.date))

		start_of_year = ephem.date(str(local.date.tuple()[0])+'/01/01 12:00') #reset to start of year GMT
		local.date = start_of_year

		# calculate season dates
		spring = ephem.next_equinox(ephem.date(str(local.date.tuple()[0])))
		summer = ephem.next_solstice(spring)
		fall = ephem.next_equinox(summer)
		winter = ephem.next_solstice(fall)

		# create nice season labels
		local_spring_day = str(ephem.localtime(spring))[:10]
		local_summer_day = str(ephem.localtime(summer))[:10]
		local_fall_day = str(ephem.localtime(fall))[:10]
		local_winter_day = str(ephem.localtime(winter))[:10]

		# set to longest day of the year and compensate for reversed seasons south of equator
		if local.lat > 0:
			local.date = summer
		else:
			local.date = winter

		# set time to midnight to get next sunrise, transit, and sunset in proper order
		start_of_day = ephem.date(str(local.date.tuple()[0])+'/'+str(local.date.tuple()[1])+'/'+str(local.date.tuple()[2])+' 00:00') #reset to start of day GMT
		local.date = start_of_day

		# calculate sunrise, transit, and sunset on longest day of the year
		self.object_next_rising = local.next_rising(object)
		self.object_next_transit = local.next_transit(object)
		self.object_next_setting = local.next_setting(object)

		# calculate minimum azimuth of sun and compensate for reversed orientation south of equator
		local.date = self.object_next_rising 
		object.compute(local)
		if local.lat >= 0:
			self.object_min_az = object.az * 180/pi
		else:
			self.object_max_az = object.az * 180/pi 
		
		# calculate maximum altitude of sun
		local.date = self.object_next_transit 
		object.compute(local)
		self.object_min_alt = 0.0
		self.object_max_alt = object.alt * 180/pi

		# calculate maximum azimuth of sun and compensate for reversed orientation south of equator
		local.date = self.object_next_setting
		object.compute(local)
		if local.lat >= 0:
			self.object_max_az = object.az * 180/pi
		else:
			self.object_min_az = object.az * 180/pi 
		
		self.y_axis_margin = 20
		self.x_axis_margin = 15

		context.set_source_rgba(1,1,1) # white rectangle
		context.fill()

		# calculate max altitude and azimuth ranges and compensate for reversed orientation south of equator
		self.alt_range = int(self.object_max_alt-self.object_min_alt+5)
		self.az_range = int(self.object_max_az-self.object_min_az)
		if local.lat >= 0:
			self.az_range = int(self.object_max_az-self.object_min_az)
		else:
			self.az_range = int(self.object_max_az+(360-self.object_min_az))

		# set up text and grid color
		context.set_source_rgb(0,0,1) # alt,az blue color
		font = "Sans"
		context.select_font_face(font)
		context.set_font_size(font_size)

		# plot altitude axis grid
		for i in range(0, self.alt_range, 5):
			context.set_line_width(1.0)
			y = (rect.height - self.x_axis_margin) - ((rect.height - self.x_axis_margin)/float(self.alt_range) * i)
			context.move_to(0, y+4)
			context.text_path(str(i+int(self.object_min_alt))+u'\xb0')
			context.set_line_width(0.5)
			context.fill()
			context.stroke()
			context.move_to(self.y_axis_margin, y)
			context.line_to(rect.width, y)
			context.stroke()

		# plot azimuth axis grid
		for i in range(0, self.az_range, 10):
			context.set_line_width(1.0)
			x = ((rect.width - self.y_axis_margin)/float(self.az_range) * i) + self.y_axis_margin
			y = rect.height 
			context.move_to(x-4, y)
			context.text_path(str(i+int(self.object_min_az))+u'\xb0')
			context.set_line_width(0.5)
			context.fill()
			context.stroke()
			y = rect.height - self.x_axis_margin
			context.move_to(x, y)
			context.line_to(x, 0)
			context.stroke()

		# calculate and plot sun angles for first day of summer
		if local.lat > 0:
			local.date = summer
			self.label = "Summer: " + local_summer_day
		else:
			local.date = winter
			self.label = "Summer: " + local_winter_day

		self.plot_color = [1,0,0]
		self.plot_sun_angle_day(context, object, local)

		# calculate and plot sun angles for first day of spring
		local.date = spring
		if local.lat >= 0:
			self.label = "Spring: " + local_spring_day + " / Fall: " + local_fall_day
		else:
			self.label = "Spring: " + local_fall_day + " / Fall: " + local_spring_day

		self.plot_color = [0,0.6,0]
		self.plot_sun_angle_day(context, object, local)

		# calculate sun angles for first day of winter 
		if local.lat >= 0:
			local.date = winter
			self.label = "Winter: " + local_winter_day
		else:
			local.date = summer
			self.label = "Winter: " + local_summer_day

		self.plot_color = [0,0,0.6]
		self.plot_sun_angle_day(context, object, local)

		# calculate and plot sun angles for current date
		local.date = save 
		if date_in_gmt == True:
			local_date_show = str(local.date)[:19]+' GMT'
		else:
			local_date_show = str(ephem.localtime(local.date))[:19]+' '+local_time_zone_name[local_time_zone_daylight_flag]
		self.label = "Now: " + str(local_date_show)

		self.plot_color = [0,0,0]
		self.plot_sun_angle_day(context, object, local)

		# calculate and plot sun angle for current time
		local.date = save 
		object.compute(local)

		object_alt = float(object.alt)*180/pi
		object_az = float(object.az)*180/pi

		y = (rect.height - self.x_axis_margin) - ((rect.height - self.x_axis_margin)/float(self.alt_range) * ((object_alt - self.object_min_alt)))
		if local.lat >= 0:	
			x = ((rect.width - self.y_axis_margin)/float(self.az_range) * ((object_az - self.object_min_az))) +self.y_axis_margin
		elif object_az > 180:
			x = ((rect.width - self.y_axis_margin)/float(self.az_range) * ((object_az - self.object_min_az))) +self.y_axis_margin
		else:
			x = ((rect.width - self.y_axis_margin)/float(self.az_range) * ((object_az + 360 - self.object_min_az))) +self.y_axis_margin

		context.arc(x, y, 10, 0, 2 * math.pi)
		context.set_source_rgb(sun_plot_color[0], sun_plot_color[1], sun_plot_color[2])
		context.fill()
		context.stroke()

		# calculate and plot moon angle for current time
		local.date = save 
	
		object = ephem.Moon()
		object.compute(local)

		object_alt = float(object.alt)*180/pi
		object_az = float(object.az)*180/pi

		y = (rect.height - self.x_axis_margin) - ((rect.height - self.x_axis_margin)/float(self.alt_range) * ((object_alt - self.object_min_alt)))
		if local.lat >= 0:	
			x = ((rect.width - self.y_axis_margin)/float(self.az_range) * ((object_az - self.object_min_az))) +self.y_axis_margin
		elif object_az > 180:
			x = ((rect.width - self.y_axis_margin)/float(self.az_range) * ((object_az - self.object_min_az))) +self.y_axis_margin
		else:
			x = ((rect.width - self.y_axis_margin)/float(self.az_range) * ((object_az + 360 - self.object_min_az))) +self.y_axis_margin

		context.arc(x, y, 10, 0, 2 * math.pi)
		context.set_source_rgb(moon_plot_color[0], moon_plot_color[1], moon_plot_color[2])
		context.fill()
		context.stroke()

		local.date = save #still required for 3.7.3.2

	def plot_sun_angle_day(self, context, object, local):
		# set time to midnight to get next sunrise, transit, and sunset in proper order
		start_of_day = ephem.date(str(local.date.tuple()[0])+'/'+str(local.date.tuple()[1])+'/'+str(local.date.tuple()[2])+' 00:00') #reset to start of day GMT
		local.date = start_of_day

		# calculate sunrise, transit, and sunset on current date
		self.object_next_rising = local.next_rising(object)
		self.object_next_transit = local.next_transit(object)
		self.object_next_setting = local.next_setting(object)

		local.date = self.object_next_rising
		#for hr in range (24*60):
		for hr in range (24*10):
		#for hr in range (24):
			object.compute(local)

			object_alt = float(object.alt)*180/pi
			object_az = float(object.az)*180/pi

			y = (rect.height - self.x_axis_margin) - ((rect.height - self.x_axis_margin)/float(self.alt_range) * ((object_alt - self.object_min_alt)))
			if local.lat >= 0:	
				x = ((rect.width - self.y_axis_margin)/float(self.az_range) * ((object_az - self.object_min_az))) +self.y_axis_margin
			elif object_az > 180:
				x = ((rect.width - self.y_axis_margin)/float(self.az_range) * ((object_az - self.object_min_az))) +self.y_axis_margin
			else:
				x = ((rect.width - self.y_axis_margin)/float(self.az_range) * ((object_az + 360 - self.object_min_az))) +self.y_axis_margin

			context.arc(x, y, 1, 0, 2 * math.pi)
			context.set_source_rgb(self.plot_color[0], self.plot_color[1], self.plot_color[2])
			context.fill()
			context.stroke()

			if self.label != "":	
				if local.date > self.object_next_transit:
					if self.label[:6] == "Spring":
						context.move_to(x-90, y-5)
					else:
						context.move_to(x-60, y-5)
					context.text_path(self.label)
					context.fill()
					context.stroke()
					self.label = ""

			#local.date = local.date + (1/24.0/60)
			local.date = local.date + (1/24.0/10)
			#local.date = local.date + (1/24.0)

			if local.date > self.object_next_setting:
				break

def on_window_state_change(widget, event, window, *args):
	global window_in_fullscreen
        if event.new_window_state & gtk.gdk.WINDOW_STATE_FULLSCREEN:
            	window_in_fullscreen = True

        else:
            	window_in_fullscreen = False

	return

def on_key_press_basic(widget, event, window, *args):
	global window_in_fullscreen

        if event.keyval == gtk.keysyms.F6:
            	# The "Full screen" hardware key has been pressed
            	if window_in_fullscreen:
			window_in_fullscreen = False
                	window.unfullscreen ()
            	else:
                	window.fullscreen ()
			window_in_fullscreen = True

	if event.keyval == gtk.keysyms.Escape:
		window.destroy()

	return

def on_key_press_google(widget, event, window, image, local, *args):
	global window_in_fullscreen
	global google_view
	global google_map_type

        if event.keyval == gtk.keysyms.F6:
            	# The "Full screen" hardware key has been pressed
            	if window_in_fullscreen:
			window_in_fullscreen = False
                	window.unfullscreen ()
            	else:
                	window.fullscreen ()
			window_in_fullscreen = True

        if event.keyval == gtk.keysyms.F7:
            	# The "Zoom In" hardware key has been pressed
		google_view = google_view / 2
		image.destroy()
        	get_google_map("clicked",window,image,local,local)

        if event.keyval == gtk.keysyms.F8:
            	# The "Zoom Out" hardware key has been pressed
		google_view = google_view * 2
		image.destroy()
        	get_google_map("clicked",window,image,local,local)

        #if event.keyval == gtk.keysyms.Escape:
            	# The "Escape" hardware key has been pressed

        if event.keyval == gtk.keysyms.Return:
            	# The "Return" hardware key has been pressed
		if google_map_type == 'roadmap':
			google_map_type = 'hybrid'
		else:
			if google_map_type == 'hybrid':
				google_map_type = 'terrain'
			else:
				if google_map_type == 'terrain':
					google_map_type = 'satellite'
				else:
					if google_map_type == 'satellite':
						google_map_type = 'roadmap'
		image.destroy()
        	get_google_map("clicked",window,image,local,local)

	if event.keyval == gtk.keysyms.Escape:
		window.destroy()

	return

def on_key_press_plot_sky(widget, event, window, vbox, local, *args):
	global window_in_fullscreen
	global polar_field_of_view
	global equi_field_of_view
	global altitude_center
	global azimuth_center
	global controls_show_flag
	global verbose_show_flag
	global on_off_controls_show_flag
	global date_time_controls_show_flag
	global polar_sky_flag
	global horizon_grid_flag
	global horizon_proj_flag
	global sun_compass_flag
	global x_shift_degrees
	global y_shift_degrees

        if event.keyval == gtk.keysyms.F6:
            	# The "Full screen" hardware key has been pressed
            	if window_in_fullscreen:
			window_in_fullscreen = False
                	window.unfullscreen ()
            	else:
			window_in_fullscreen = True
                	window.fullscreen ()

        if event.keyval == gtk.keysyms.F7:
            	# The "Zoom In" hardware key has been pressed
		polar_field_of_view = polar_field_of_view / 1.10
		equi_field_of_view = equi_field_of_view / 1.10

        if event.keyval == gtk.keysyms.F8:
            	# The "Zoom Out" hardware key has been pressed
		polar_field_of_view = polar_field_of_view * 1.10
		equi_field_of_view = equi_field_of_view * 1.10

        if event.keyval == gtk.keysyms.Up:
            	# The "Arrow Up" hardware key has been pressed
		altitude_center = altitude_center + 5
		y_shift_degrees = y_shift_degrees - 5

        if event.keyval == gtk.keysyms.Down:
            	# The "Arrow Down" hardware key has been pressed
		altitude_center = altitude_center - 5
		y_shift_degrees = y_shift_degrees + 5

        if event.keyval == gtk.keysyms.Left:
            	# The "Arrow Left" hardware key has been pressed
		azimuth_center = azimuth_center - 5
		x_shift_degrees = x_shift_degrees - 5

        if event.keyval == gtk.keysyms.Right:
            	# The "Arrow Right" hardware key has been pressed
		azimuth_center = azimuth_center + 5
		x_shift_degrees = x_shift_degrees + 5

	if event.keyval == gtk.keysyms.Escape:
            	# The "Escape" hardware key has been pressed
		window.destroy()

        if event.keyval == gtk.keysyms.Return:
            	# The "Return" hardware key has been pressed
		if horizon_proj_flag == True:
			horizon_proj_flag = False
		else:
			horizon_proj_flag = True

        if event.keyval == gtk.keysyms.F4:
            	# The "Open Menu" hardware key has been pressed
		if controls_show_flag == True:
			if verbose_show_flag == True:
				verbose_show_flag = False
			else:
				#print "Turning off controls..."
				controls_show_flag = False
		else:
			controls_show_flag = True
			verbose_show_flag = True
			on_off_controls_show_flag = True
			date_time_controls_show_flag = True

	vbox.destroy()

	if event.keyval != gtk.keysyms.Escape:
		if sun_compass_flag == False:
			plot_sky("clicked",window,vbox,local)
		else:
			plot_compass("clicked",window,vbox,local)
	else:
		print "now destroying sky plot/compass plot window..."
		window.destroy()
	
	return

def on_key_press_get_moon(widget, event, window, image, *args):
	global window_in_fullscreen
	global moon_zoom
	global moon_x
	global moon_y

        if event.keyval == gtk.keysyms.F6:
            	# The "Full screen" hardware key has been pressed
            	if window_in_fullscreen:
			window_in_fullscreen = False
                	window.unfullscreen ()
            	else:
			window_in_fullscreen = True
                	window.fullscreen ()

        if event.keyval == gtk.keysyms.F7:
            	# The "Zoom In" hardware key has been pressed
		moon_zoom = moon_zoom - 1
		if moon_zoom < 0:
			moon_zoom = 0

        if event.keyval == gtk.keysyms.F8:
            	# The "Zoom Out" hardware key has been pressed
		moon_zoom = moon_zoom + 1
		if moon_zoom > 5:
			moon_zoom = 5

        if event.keyval == gtk.keysyms.Left:
            	# The "Arrow Left" hardware key has been pressed
		moon_x = moon_x - 10

        if event.keyval == gtk.keysyms.Right:
            	# The "Arrow Right" hardware key has been pressed
		moon_x = moon_x + 10

        if event.keyval == gtk.keysyms.Up:
            	# The "Arrow Up" hardware key has been pressed
		moon_y = moon_y - 10

        if event.keyval == gtk.keysyms.Down:
            	# The "Arrow Down" hardware key has been pressed
		moon_y = moon_y + 10

	if event.keyval != gtk.keysyms.Escape:
		image.destroy()
		zoom_moon("clicked",window,image)
	else:
		print "now destroying moon zoom window..."
		window.destroy()

	return

def on_key_press_plot_moon(widget, event, window, vbox, local, *args):
	global window_in_fullscreen
	global controls_show_flag
	global on_off_controls_show_flag
	global date_time_controls_show_flag
	global verbose_show_flag

        if event.keyval == gtk.keysyms.F6:
            	# The "Full screen" hardware key has been pressed
            	if window_in_fullscreen:
			window_in_fullscreen = False
                	window.unfullscreen ()
            	else:
			window_in_fullscreen = True
                	window.fullscreen ()

        if event.keyval == gtk.keysyms.F4:
            	# The "Open Menu" hardware key has been pressed
		if controls_show_flag == True:
			if verbose_show_flag == True:
				verbose_show_flag = False
			else:
				controls_show_flag = False
		else:
			controls_show_flag = True
			verbose_show_flag = True
			on_off_controls_show_flag = True
			date_time_controls_show_flag = True

        if event.keyval == gtk.keysyms.F7:
                # The "Zoom In" hardware key has been pressed
		current_date = (local.date.triple()[0],local.date.triple()[1],local.date.triple()[2]+1)
		local.date=ephem.date(current_date)

        if event.keyval == gtk.keysyms.F8:
                # The "Zoom Out" hardware key has been pressed
		current_date = (local.date.triple()[0],local.date.triple()[1],local.date.triple()[2]-1)
		local.date=ephem.date(current_date)

        if event.keyval == gtk.keysyms.Up:
            	# The "Arrow Up" hardware key has been pressed
		current_date = (local.date.triple()[0],local.date.triple()[1]+1,local.date.triple()[2])
		local.date=ephem.date(current_date)

        if event.keyval == gtk.keysyms.Down:
            	# The "Arrow Down" hardware key has been pressed
		current_date = (local.date.triple()[0],local.date.triple()[1]-1,local.date.triple()[2])
		local.date=ephem.date(current_date)

        if event.keyval == gtk.keysyms.Left:
            	# The "Arrow Left" hardware key has been pressed
		current_date = (local.date.triple()[0],local.date.triple()[1],local.date.triple()[2]-(1/24.0))
		local.date=ephem.date(current_date)

        if event.keyval == gtk.keysyms.Right:
            	# The "Arrow Right" hardware key has been pressed
		current_date = (local.date.triple()[0],local.date.triple()[1],local.date.triple()[2]+(1/24.0))
		local.date=ephem.date(current_date)

	vbox.destroy()

	if event.keyval != gtk.keysyms.Escape:
		plot_moon_phase("clicked",window,vbox,local)
	else:
		print "now destroying moon phase window..."
		window.destroy()
	
	return

def rotate_3dpoint(p, angle, axis):
        """Rotate a 3D point around given axis."""

        ret = [p[0], 0.0, 0.0, 0.0]
        angle = angle*pi/180
        cosang = cos(angle)
        sinang = sin(angle)
        ret[1] += (cosang+(1-cosang)*axis[0]*axis[0])*p[1]
        ret[1] += ((1-cosang)*axis[0]*axis[1]-axis[2]*sinang)*p[2]
        ret[1] += ((1-cosang)*axis[0]*axis[2]+axis[1]*sinang)*p[3]
        ret[2] += ((1-cosang)*axis[0]*axis[1]+axis[2]*sinang)*p[1]
        ret[2] += (cosang+(1-cosang)*axis[1]*axis[1])*p[2]
        ret[2] += ((1-cosang)*axis[1]*axis[2]-axis[0]*sinang)*p[3]
        ret[3] += ((1-cosang)*axis[0]*axis[2]-axis[1]*sinang)*p[1]
        ret[3] += ((1-cosang)*axis[1]*axis[2]+axis[0]*sinang)*p[2]
        ret[3] += (cosang+(1-cosang)*axis[2]*axis[2])*p[3]

        return ret

def rotate_object(obj, angle, axis):
        """Rotate an object around given axis."""

        for i in range(len(obj)):
                obj[i] = rotate_3dpoint(obj[i], angle, axis)

def on_key_press_plot_solar_system(widget, event, window, vbox, object, local, *args):
        global window_in_fullscreen
        global planet_field_of_view
	global controls_show_flag
	global date_time_controls_show_flag
	global verbose_show_flag
	global planet_trail

        if event.keyval == gtk.keysyms.F4:
            	# The "Open Menu" hardware key has been pressed
		if controls_show_flag == True:
			if verbose_show_flag == True:
				verbose_show_flag = False
			else:
				controls_show_flag = False
		else:
			controls_show_flag = True
			verbose_show_flag = True
			date_time_controls_show_flag = True

        if event.keyval == gtk.keysyms.F6:
                # The "Full screen" hardware key has been pressed
                if window_in_fullscreen:
                        window_in_fullscreen = False
                        window.unfullscreen ()
                else:
                        window_in_fullscreen = True
                        window.fullscreen ()

        if event.keyval == gtk.keysyms.F7:
                # The "Zoom In" hardware key has been pressed
                planet_field_of_view = planet_field_of_view * 2.0

        if event.keyval == gtk.keysyms.F8:
                # The "Zoom Out" hardware key has been pressed
                planet_field_of_view = planet_field_of_view / 2.0

        if event.keyval == gtk.keysyms.Up:
            	# The "Arrow Up" hardware key has been pressed
		current_date = (local.date.triple()[0],local.date.triple()[1]+1,local.date.triple()[2])
		local.date=ephem.date(current_date)

        if event.keyval == gtk.keysyms.Down:
            	# The "Arrow Down" hardware key has been pressed
		current_date = (local.date.triple()[0],local.date.triple()[1]-1,local.date.triple()[2])
		local.date=ephem.date(current_date)

        if event.keyval == gtk.keysyms.Left:
            	# The "Arrow Left" hardware key has been pressed
		current_date = (local.date.triple()[0],local.date.triple()[1],local.date.triple()[2]-(1/24.0))
		local.date=ephem.date(current_date)

        if event.keyval == gtk.keysyms.Right:
            	# The "Arrow Right" hardware key has been pressed
		current_date = (local.date.triple()[0],local.date.triple()[1],local.date.triple()[2]+(1/24.0))
		local.date=ephem.date(current_date)

        vbox.destroy()

        if event.keyval != gtk.keysyms.Escape:
                plot_solar_system_position("clicked",window,vbox,object,local)
	else:
		print "now destroying solar system plot window..."
		window.destroy()

        return

def on_key_press_plot_solar_orbit(widget, event, window, vbox, object, local, planet_orbit, *args):
        global window_in_fullscreen
        global planet_field_of_view
	global controls_show_flag
	global date_time_controls_show_flag
	global verbose_show_flag

        if event.keyval == gtk.keysyms.Up:
                # The "Arrow Up" hardware key has been pressed
                rotate_object(planet_orbit, 5.0, (1,0,0))

        if event.keyval == gtk.keysyms.Down:
                # The "Arrow Down" hardware key has been pressed
                rotate_object(planet_orbit, -5.0, (1,0,0))

        if event.keyval == gtk.keysyms.Right:
                # The "Arrow Up" hardware key has been pressed
                rotate_object(planet_orbit, 5.0, (0,1,0))

        if event.keyval == gtk.keysyms.Left:
                # The "Arrow Down" hardware key has been pressed
                rotate_object(planet_orbit, -5.0, (0,1,0))

        if event.keyval == gtk.keysyms.F4:
            	# The "Open Menu" hardware key has been pressed
		if controls_show_flag == True:
			if verbose_show_flag == True:
				verbose_show_flag = False
			else:
				controls_show_flag = False
		else:
			controls_show_flag = True
			verbose_show_flag = True
			date_time_controls_show_flag = True

        if event.keyval == gtk.keysyms.F6:
                # The "Full screen" hardware key has been pressed
                if window_in_fullscreen:
                        window_in_fullscreen = False
                        window.unfullscreen ()
                else:
                        window_in_fullscreen = True
                        window.fullscreen ()

        if event.keyval == gtk.keysyms.F7:
                # The "Zoom In" hardware key has been pressed
                planet_field_of_view = planet_field_of_view * 2.0

        if event.keyval == gtk.keysyms.F8:
                # The "Zoom Out" hardware key has been pressed
                planet_field_of_view = planet_field_of_view / 2.0

        #if event.keyval == gtk.keysyms.Return:
                # The "Return" hardware key has been pressed

        vbox.destroy()

        if event.keyval != gtk.keysyms.Escape:
                plot_solar_system_orbit("clicked",window,vbox,object,local,planet_orbit)
	else:
		print "now destroying solar system plot window..."
		window.destroy()

        return

def on_key_press_plot_planet_moon(widget, event, window, vbox, object, local, *args):
	global window_in_fullscreen
	global moon_field_of_view
	global controls_show_flag
	global on_off_controls_show_flag
	global date_time_controls_show_flag
	global verbose_show_flag

        if event.keyval == gtk.keysyms.F4:
            	# The "Open Menu" hardware key has been pressed
		if controls_show_flag == True:
			if verbose_show_flag == True:
				verbose_show_flag = False
			else:
				controls_show_flag = False
		else:
			controls_show_flag = True
			verbose_show_flag = True
			on_off_controls_show_flag = True
			date_time_controls_show_flag = True

        if event.keyval == gtk.keysyms.F6:
            	# The "Full screen" hardware key has been pressed
            	if window_in_fullscreen:
			window_in_fullscreen = False
                	window.unfullscreen ()
            	else:
			window_in_fullscreen = True
                	window.fullscreen ()

        if event.keyval == gtk.keysyms.F7:
            	# The "Zoom In" hardware key has been pressed
		moon_field_of_view = moon_field_of_view * 2.0

        if event.keyval == gtk.keysyms.F8:
            	# The "Zoom Out" hardware key has been pressed
		moon_field_of_view = moon_field_of_view / 2.0

        if event.keyval == gtk.keysyms.Up:
            	# The "Arrow Up" hardware key has been pressed
		current_date = (local.date.triple()[0],local.date.triple()[1]+1,local.date.triple()[2])
		local.date=ephem.date(current_date)

        if event.keyval == gtk.keysyms.Down:
            	# The "Arrow Down" hardware key has been pressed
		current_date = (local.date.triple()[0],local.date.triple()[1]-1,local.date.triple()[2])
		local.date=ephem.date(current_date)

        if event.keyval == gtk.keysyms.Left:
            	# The "Arrow Left" hardware key has been pressed
		current_date = (local.date.triple()[0],local.date.triple()[1],local.date.triple()[2]-(1/24.0))
		local.date=ephem.date(current_date)

        if event.keyval == gtk.keysyms.Right:
            	# The "Arrow Right" hardware key has been pressed
		current_date = (local.date.triple()[0],local.date.triple()[1],local.date.triple()[2]+(1/24.0))
		local.date=ephem.date(current_date)

	vbox.destroy()

	if event.keyval != gtk.keysyms.Escape:
		plot_planet_moon_position("clicked",window,vbox,object,local)
	else:
		print "now destroying mars moon plot window..."
		window.destroy()
	
	return

def on_key_press_plot_object_rise_set_graph(widget, event, window, object, local, *args):
	global window_in_fullscreen
	global rise_set_plot_flag

        if event.keyval == gtk.keysyms.F6:
            	# The "Full screen" hardware key has been pressed
            	if window_in_fullscreen:
			window_in_fullscreen = False
                	window.unfullscreen ()
            	else:
			window_in_fullscreen = True
                	window.fullscreen ()

        if event.keyval == gtk.keysyms.Return:
            	# The "Return" hardware key has been pressed
		if rise_set_plot_flag == True:
			rise_set_plot_flag = False
		else:
			rise_set_plot_flag = True

	window.destroy()

	if event.keyval != gtk.keysyms.Escape:
		plot_object_rise_set_graph_start("clicked",object,local)
	
	return

def on_key_press_plot_sun_angle_graph(widget, event, window, object, local, *args):
	global window_in_fullscreen

        if event.keyval == gtk.keysyms.F6:
            	# The "Full screen" hardware key has been pressed
            	if window_in_fullscreen:
			window_in_fullscreen = False
                	window.unfullscreen ()
            	else:
			window_in_fullscreen = True
                	window.fullscreen ()

	window.destroy()

	if event.keyval != gtk.keysyms.Escape:
		plot_sun_angle_graph_start("clicked",object,local)
	
	return

def get_ngc_image(widget,window,ngc_id):
        if hildon_flag == 1:
                window = hildon.Window()
        else:
                window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title('Planet Ephemeris: WEB NGC IMAGE')

	window.set_size_request(800,480)

	screen = window.get_screen()
	screen_width = screen.get_width()
	screen_height = screen.get_height()
	window_width = window.get_size()[0]
	window_height = window.get_size()[1]

        color_black = gtk.gdk.color_parse('black')
	window.modify_bg(gtk.STATE_NORMAL, color_black)

	ngc_group = int(ngc_id/1000)

	url = "http://www.ngcic.org/dss/n/" + str(ngc_group) + "/n" + str(int(ngc_id)).zfill(4) + ".jpg"

	try:
		file = urllib.urlopen(url)
		print "url=",url
		ngc_pic = file.read()
		run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
		data_dir = os.path.join(run_dir, 'data')
		image_file = os.path.join(data_dir, 'ngc.jpg')
		temp_file = open(image_file,"wb")
		temp_file.write(ngc_pic)
		temp_file.close()
		print "image_file=",image_file
	except Exception:
		print "web connection timeout"

	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
	data_dir = os.path.join(run_dir, 'data')
	image_file = os.path.join(data_dir, 'ngc.jpg')
	image = gtk.Image()
	image.set_from_file(image_file)

	pixbuf = gtk.gdk.pixbuf_new_from_file(image_file)

	aspect = True
	enlarge = True

	display_image(image,pixbuf,aspect,enlarge,screen_width,screen_height,window_width,window_height)

	table = gtk.Table(1,1, False)
	window.add(table)
	table.attach(image,0,1,0,1)

	event_box = gtk.EventBox()
	table.attach(event_box,0,1,1,2)
	set_event_box_color("clicked",event_box,"black","yellow",url)
	table.show()

	image.show()

	if window_in_fullscreen == True:
		window.fullscreen()

	window.show()

	return

def select_ngc_image(widget, window):
	dialog = custom_set_ngc_id(window)
	dialog.run()

	ngc_id = dialog.value[0]
	print ngc_id

	dialog.destroy()

	get_ngc_image("clicked",window,ngc_id)

	return

class custom_set_ngc_id(gtk.Dialog) :   
        value = [0] * 1

        def hildon_set_value(self, widget, data=None, index=0):
       	 	self.value[index] = widget.get_value()
           	return

        def set_value(self,widget,index):
           	self.value[index] = widget.get_value()
           	return

        def __init__(self,window):
		ngc_id = 1

                super(custom_set_ngc_id, self).__init__("Set NGC ID", window,
                                gtk.DIALOG_DESTROY_WITH_PARENT | gtk.DIALOG_MODAL,
                                (gtk.STOCK_OK, gtk.RESPONSE_OK))

		label = gtk.Label('NGC Object ID:')
		self.vbox.pack_start(label)
		label.show()

        	if hildon_flag == 1:
			number_editor = hildon.NumberEditor(1,6999)
			number_editor.connect("notify::value",self.hildon_set_value,0)
			number_editor.set_value(ngc_id)
			self.vbox.pack_start(number_editor)
			self.value[0] = ngc_id
			number_editor.show()
		else:
			adjustment = gtk.Adjustment(ngc_id, 1, 6999, 1, 1, 0)
			spinbutton = gtk.SpinButton(adjustment,0,0)
			spinbutton.connect("changed",self.set_value, 0)
			self.vbox.pack_start(spinbutton)
			self.value[0] = ngc_id
			spinbutton.show()		

value = ""

def set_location(widget, window, table, vbox, local):
	dialog = custom_set_location(window,local)
	dialog.run()

	if dialog.value[0] > 0:
		local.lat = str(dialog.value[0]+dialog.value[1]/60.0+dialog.value[2]/3600.0)
	else:
		local.lat = str(-(abs(dialog.value[0])+dialog.value[1]/60.0+dialog.value[2]/3600.0))
	if dialog.value[3] > 0:
		local.long = str(dialog.value[3]+dialog.value[4]/60.0+dialog.value[5]/3600.0)
	else:
		local.long = str(-(abs(dialog.value[3])+dialog.value[4]/60.0+dialog.value[5]/3600.0))

	local.temp = dialog.value[6]
	local.pressure = dialog.value[7]
	local.elevation = dialog.value[8]

	print "new local.temp=", local.temp
	print "new local.pressure=", local.pressure
	print "new local.elevation=", local.elevation

	dialog.destroy()

	table.destroy()
	vbox.destroy()

	show_object_info_basic("clicked",window,local)

	return

class custom_set_location(gtk.Dialog) :   
        value = [0] * 9

        def hildon_set_value(self, widget, data=None, index=0):
       	 	self.value[index] = widget.get_value()
           	return

        def set_value(self,widget,index):
           	self.value[index] = widget.get_value()
           	return

        def __init__(self,window,local):
		local_lat = str(local.lat)
		local_long = str(local.long)
		local_temp = local.temp
		local_pressure = local.pressure
		local_elevation = local.elevation

        	if hildon_flag == 1:
			lat_degrees = int(round(float(local_lat.split(":")[0])))
			lat_minutes = int(round(float(local_lat.split(":")[1])))
			lat_seconds = int(round(float(local_lat.split(":")[2])))
			long_degrees = int(round(float(local_long.split(":")[0])))
			long_minutes = int(round(float(local_long.split(":")[1])))
			long_seconds = int(round(float(local_long.split(":")[2])))
		else:
			lat_degrees = float(local_lat.split(":")[0])
			lat_minutes = float(local_lat.split(":")[1])
			lat_seconds = float(local_lat.split(":")[2])
			long_degrees = float(local_long.split(":")[0])
			long_minutes = float(local_long.split(":")[1])
			long_seconds = float(local_long.split(":")[2])

                super(custom_set_location, self).__init__("Set Location", window,
                                gtk.DIALOG_DESTROY_WITH_PARENT | gtk.DIALOG_MODAL,
                                (gtk.STOCK_OK, gtk.RESPONSE_OK))

		label = gtk.Label('Latitude Degrees:')
		self.vbox.pack_start(label)
		label.show()

        	if hildon_flag == 1:
			number_editor = hildon.NumberEditor(-90,90)
			number_editor.connect("notify::value",self.hildon_set_value,0)
			number_editor.set_value(lat_degrees)
			self.vbox.pack_start(number_editor)
			self.value[0] = lat_degrees
			number_editor.show()
		else:
			adjustment = gtk.Adjustment(lat_degrees, -90, 90, 1, 1, 0)
			spinbutton = gtk.SpinButton(adjustment,0,0)
			spinbutton.connect("changed",self.set_value, 0)
			self.vbox.pack_start(spinbutton)
			self.value[0] = lat_degrees
			spinbutton.show()		

		label = gtk.Label('Latitude Minutes:')
		self.vbox.pack_start(label)
		label.show()
		
        	if hildon_flag == 1:
			number_editor = hildon.NumberEditor(0,60)
			number_editor.connect("notify::value",self.hildon_set_value,1)
			number_editor.set_value(lat_minutes)
			self.vbox.pack_start(number_editor)
			self.value[1] = lat_minutes
			number_editor.show()
		else:
			adjustment = gtk.Adjustment(lat_minutes, 0, 60, 1, 1, 0)
			spinbutton = gtk.SpinButton(adjustment,0,0)
			spinbutton.connect("changed",self.set_value, 1)
			self.vbox.pack_start(spinbutton)
			self.value[1] = lat_minutes
			spinbutton.show()		

		label = gtk.Label('Latitude Seconds:')
		self.vbox.pack_start(label)
		label.show()
		
        	if hildon_flag == 1:
			number_editor = hildon.NumberEditor(0,60)
			number_editor.connect("notify::value",self.hildon_set_value,2)
			number_editor.set_value(lat_seconds)
			self.vbox.pack_start(number_editor)
			self.value[2] = lat_seconds
			number_editor.show()
		else:
			adjustment = gtk.Adjustment(lat_seconds, 0, 60, 0.1, 1, 0)
			spinbutton = gtk.SpinButton(adjustment,0,1)
			spinbutton.connect("changed",self.set_value, 2)
			self.vbox.pack_start(spinbutton)
			self.value[2] = lat_seconds
			spinbutton.show()		

		label = gtk.Label('Longitude Degrees:')
		self.vbox.pack_start(label)
		label.show()
		
        	if hildon_flag == 1:
			number_editor = hildon.NumberEditor(-180,180)
			number_editor.connect("notify::value",self.hildon_set_value,3)
			number_editor.set_value(long_degrees)
			self.vbox.pack_start(number_editor)
			self.value[3] = long_degrees
			number_editor.show()
		else:
			adjustment = gtk.Adjustment(long_degrees, -180, 180, 1, 1, 0)
			spinbutton = gtk.SpinButton(adjustment,0,0)
			spinbutton.connect("changed",self.set_value, 3)
			self.vbox.pack_start(spinbutton)
			self.value[3] = long_degrees
			spinbutton.show()		

		label = gtk.Label('Longitude Minutes:')
		self.vbox.pack_start(label)
		label.show()
		
        	if hildon_flag == 1:
			number_editor = hildon.NumberEditor(0,60)
			number_editor.connect("notify::value",self.hildon_set_value,4)
			number_editor.set_value(long_minutes)
			self.vbox.pack_start(number_editor)
			self.value[4] = long_minutes
			number_editor.show()
		else:
			adjustment = gtk.Adjustment(long_minutes, 0, 60, 1, 1, 0)
			spinbutton = gtk.SpinButton(adjustment,0,0)
			spinbutton.connect("changed",self.set_value, 4)
			self.vbox.pack_start(spinbutton)
			self.value[4] = long_minutes
			spinbutton.show()		

		label = gtk.Label('Longitude Seconds:')
		self.vbox.pack_start(label)
		label.show()
		
        	if hildon_flag == 1:
			number_editor = hildon.NumberEditor(0,60)
			number_editor.connect("notify::value",self.hildon_set_value,5)
			number_editor.set_value(long_seconds)
			self.vbox.pack_start(number_editor)
			self.value[5] = long_seconds
			number_editor.show()
		else:
			adjustment = gtk.Adjustment(long_seconds, 0, 60, 0.1, 1, 0)
			spinbutton = gtk.SpinButton(adjustment,0,1)
			spinbutton.connect("changed",self.set_value, 5)
			self.vbox.pack_start(spinbutton)
			self.value[5] = long_seconds
			spinbutton.show()		

		label = gtk.Label('Temperature Celsius:')
		self.vbox.pack_start(label)
		label.show()
		
        	if hildon_flag == 1:
			number_editor = hildon.NumberEditor(-90,60)
			number_editor.connect("notify::value",self.hildon_set_value,6)
			number_editor.set_value(local_temp)
			self.vbox.pack_start(number_editor)
			self.value[6] = local_temp
			number_editor.show()
		else:
			adjustment = gtk.Adjustment(local_temp, -90, 60, 0.1, 1, 0)
			spinbutton = gtk.SpinButton(adjustment,0,1)
			spinbutton.connect("changed",self.set_value, 6)
			self.vbox.pack_start(spinbutton)
			self.value[6] = local_temp
			spinbutton.show()		

		label = gtk.Label('Pressure Millibars:')
		self.vbox.pack_start(label)
		label.show()
		
        	if hildon_flag == 1:
			number_editor = hildon.NumberEditor(800,1100)
			number_editor.connect("notify::value",self.hildon_set_value,7)
			number_editor.set_value(local_pressure)
			self.vbox.pack_start(number_editor)
			self.value[7] = local_pressure
			number_editor.show()
		else:
			adjustment = gtk.Adjustment(local_pressure, 800, 1100, 0.1, 1, 0)
			spinbutton = gtk.SpinButton(adjustment,0,1)
			spinbutton.connect("changed",self.set_value, 7)
			self.vbox.pack_start(spinbutton)
			self.value[7] = local_pressure
			spinbutton.show()		

		label = gtk.Label('Elevation Meters:')
		self.vbox.pack_start(label)
		label.show()
		
        	if hildon_flag == 1:
			number_editor = hildon.NumberEditor(-500,10000)
			number_editor.connect("notify::value",self.hildon_set_value,8)
			number_editor.set_value(local_elevation)
			self.vbox.pack_start(number_editor)
			self.value[8] = local_elevation
			number_editor.show()
		else:
			adjustment = gtk.Adjustment(local_elevation, -500, 10000, 1, 1, 0)
			spinbutton = gtk.SpinButton(adjustment,0,1)
			spinbutton.connect("changed",self.set_value, 8)
			self.vbox.pack_start(spinbutton)
			self.value[8] = local_elevation
			spinbutton.show()		

                return

def set_date(widget,window,button,table,vbox,local,save):
	if hildon_flag == 1:
		y = local.date.tuple()
    		dialog = hildon.CalendarPopup (window,y[0],y[1],y[2])
    		dialog.run()
    		date_tuple = dialog.get_date()
    		dialog.destroy()
        	x = str(date_tuple[0])+'/'+str(date_tuple[1])+'/'+str(date_tuple[2])+' '+str(y[3])+':'+str(y[4])+':'+str(y[5])
        	local.date = ephem.date(x)
		save = local.date
		y = local.date.triple()
    		button.set_label(str(y[0])+'/'+str(y[1]).zfill(2)+'/'+str(int(y[2])).zfill(2))
		set_button_color("clicked",button,"black","black","black","red","green","red")
		table.destroy()
		vbox.destroy()
		show_object_info_basic("clicked",window,local)
	else:
		y = local.date.tuple()
		dialog = gtk_set_date(window,local)
		dialog.run()
		dialog.destroy()
        	x = str(dialog.value[0])+'/'+str(dialog.value[1])+'/'+str(dialog.value[2])+' '+str(y[3])+':'+str(y[4])+':'+str(y[5])
        	local.date = ephem.date(x)
		save = local.date
		y = local.date.triple()
    		button.set_label(str(y[0])+'/'+str(y[1]).zfill(2)+'/'+str(int(y[2])).zfill(2))
		table.destroy()
		vbox.destroy()
		show_object_info_basic("clicked",window,local)
    
    	return

class gtk_set_date(gtk.Dialog) :   
        value = [0] * 3

        def set_value(self,widget,index):
           	self.value[index] = widget.get_value_as_int()
           	return

        def __init__(self,window,local):
		date_year = local.date.tuple()[0]
		date_month = local.date.tuple()[1]
		date_day = local.date.tuple()[2]
                super(gtk_set_date, self).__init__("Set Date", window,
                                gtk.DIALOG_DESTROY_WITH_PARENT | gtk.DIALOG_MODAL,
                                (gtk.STOCK_OK, gtk.RESPONSE_OK))
		label = gtk.Label('Date Degrees:')
		self.vbox.pack_start(label)
		label.show()

		adjustment = gtk.Adjustment(date_year, 0, 3000, 1, 1, 0)
		spinbutton = gtk.SpinButton(adjustment,0,0)
		spinbutton.connect("changed",self.set_value, 0)
		self.vbox.pack_start(spinbutton)
		self.value[0] = date_year
		spinbutton.show()		

		label = gtk.Label('Date Month:')
		self.vbox.pack_start(label)
		label.show()
		
		adjustment = gtk.Adjustment(date_month, 1, 12, 1, 1, 0)
		spinbutton = gtk.SpinButton(adjustment,0,0)
		spinbutton.connect("changed",self.set_value, 1)
		self.vbox.pack_start(spinbutton)
		self.value[1] = date_month
		spinbutton.show()		

		label = gtk.Label('Date Day:')
		self.vbox.pack_start(label)
		label.show()
		
		adjustment = gtk.Adjustment(date_day, 1, 31, 1, 1, 0)
		spinbutton = gtk.SpinButton(adjustment,0,0)
		spinbutton.connect("changed",self.set_value, 2)
		self.vbox.pack_start(spinbutton)
		self.value[2] = date_day
		spinbutton.show()		

                return

def set_time(widget,window,button,table,vbox,local,save):
	if hildon_flag == 1:
		time_picker = hildon.TimePicker(window)

		if date_in_gmt == True:
			on_show_note(widget,window,"Set time for GMT","OK")
			y = local.date.tuple()
		else:
			on_show_note(widget,window,"Set time for "+time.tzname[time.daylight],"OK")
			y = ephem.Date(local.date-ephem.hour*(time.timezone/60/60-time.daylight)).tuple()

		time_picker.set_time(y[3],y[4])
    		response = time_picker.run()                                                                                                                                                   
    		time_picker.hide()                                                                                                                                                             
		z = time_picker.get_time()
		y = local.date.tuple()
        	x = str(y[0])+'/'+str(y[1])+'/'+str(y[2])+' '+str(z[0])+':'+str(z[1])
        	local.date = ephem.date(x)

		if date_in_gmt == False:
			local.date = ephem.Date(local.date+ephem.hour*(time.timezone/60/60-time.daylight))

		save = local.date
		button.set_label(str(z[0]).zfill(2)+':'+str(z[1]).zfill(2))
		table.destroy()
		vbox.destroy()
		show_object_info_basic("clicked",window,local)
	else:
		if date_in_gmt == True:
			on_show_note(widget,window,"Set time for GMT","OK")
			y = local.date.tuple()
		else:
			on_show_note(widget,window,"Set time for "+time.tzname[time.daylight],"OK")
			y = ephem.Date(local.date-ephem.hour*(time.timezone/60/60-time.daylight)).tuple()

		dialog = gtk_set_time(window,y)
		dialog.run()
		dialog.destroy()
        	x = str(y[0])+'/'+str(y[1])+'/'+str(y[2])+' '+str(dialog.value[0])+':'+str(dialog.value[1])+':'+str(dialog.value[2])
        	local.date = ephem.date(x)

		if date_in_gmt == False:
			local.date = ephem.Date(local.date+ephem.hour*(time.timezone/60/60-time.daylight))

		save = local.date
		y = local.date.triple()
    		button.set_label(str(y[0])+'/'+str(y[1]).zfill(2)+'/'+str(int(y[2])).zfill(2))
		table.destroy()
		vbox.destroy()
		show_object_info_basic("clicked",window,local)

	return

class gtk_set_time(gtk.Dialog) :   
        value = [0] * 3

        def set_value(self,widget,index):
           	self.value[index] = widget.get_value_as_int()
           	return

        def __init__(self,window,local):
		time_hour = local[3]
		time_minute = local[4]
		time_second = local[5]
                super(gtk_set_time, self).__init__("Set Time", window,
                                gtk.DIALOG_DESTROY_WITH_PARENT | gtk.DIALOG_MODAL,
                                (gtk.STOCK_OK, gtk.RESPONSE_OK))
		label = gtk.Label('Time Hour:')
		self.vbox.pack_start(label)
		label.show()

		adjustment = gtk.Adjustment(time_hour, 0, 24, 1, 1, 0)
		spinbutton = gtk.SpinButton(adjustment,0,0)
		spinbutton.connect("changed",self.set_value, 0)
		self.vbox.pack_start(spinbutton)
		self.value[0] = time_hour
		spinbutton.show()		

		label = gtk.Label('Time Minute:')
		self.vbox.pack_start(label)
		label.show()
		
		adjustment = gtk.Adjustment(time_minute, 0, 60, 1, 1, 0)
		spinbutton = gtk.SpinButton(adjustment,0,0)
		spinbutton.connect("changed",self.set_value, 1)
		self.vbox.pack_start(spinbutton)
		self.value[1] = time_minute
		spinbutton.show()		

		label = gtk.Label('Time Second:')
		self.vbox.pack_start(label)
		label.show()
		
		adjustment = gtk.Adjustment(time_second, 0, 60, 1, 1, 0)
		spinbutton = gtk.SpinButton(adjustment,0,0)
		spinbutton.connect("changed",self.set_value, 2)
		self.vbox.pack_start(spinbutton)
		self.value[2] = time_second
		spinbutton.show()		

                return

def set_object_size(widget,object_size_change_flag):
	global object_size
	if object_size_change_flag == -1:
		object_size = object_size-1
	if object_size_change_flag == 1:
		object_size = object_size+1

	return

def set_object_bright(widget,object_bright_change_flag):
	global object_bright
	if object_bright_change_flag == -1:
		object_bright = object_bright+0.1
	if object_bright_change_flag == 1:
		object_bright = object_bright-0.1

	return

def set_grid_space(widget,grid_space_change_flag):
	global grid_space
	if grid_space_change_flag == -1:
		grid_space = grid_space - 1
	if grid_space_change_flag == 1:
		grid_space = grid_space + 1
	if grid_space > 90:
		grid_space = 90
	if grid_space < 1:
		grid_space = 1

	return

def set_field_of_view(widget,field_of_view_size_change_flag):
	global polar_field_of_view
	global equi_field_of_view
	if field_of_view_size_change_flag == -1:
		polar_field_of_view = polar_field_of_view / 1.10
		equi_field_of_view = equi_field_of_view / 1.10
	if field_of_view_size_change_flag == 1:
		polar_field_of_view = polar_field_of_view * 1.10
		equi_field_of_view = equi_field_of_view * 1.10
	if polar_field_of_view <= 0:
		polar_field_of_view = 5.0
	if equi_field_of_view <= 0:
		equi_field_of_view = 5.0

	return

def set_magnitude(widget,magnitude_change_flag):
    	global magnitude_limit
	if magnitude_change_flag == -1:
		magnitude_limit = magnitude_limit - 1
	if magnitude_change_flag == 1:
		magnitude_limit = magnitude_limit + 1
	if magnitude_limit < -0.51:
		magnitude_limit = -0.51
	if magnitude_limit > 12.49:
		magnitude_limit = 12.49

	return

def set_font_size(widget,text_size_change_flag):
    	global font_size
	if text_size_change_flag == -1:
		font_size = font_size - 1	
	if text_size_change_flag == 1:
		font_size = font_size + 1	
	if font_size < 1:
		font_size = 1

	return

def az_alt_to_polar(polar):
	global horizon_proj_flag

	view_az = (-90*pi/180) - (x_shift_degrees*pi/180)
	view_alt = (90*pi/180) + (y_shift_degrees*pi/180)
	az = polar[0]
	if horizon_proj_flag == False:
		az = (2*pi)-az
	alt = polar[1]	
	try:
		r = (100)/(1+sin(view_alt)*sin(alt)+cos(view_alt)*cos(alt)*cos(az-view_az))
	except ZeroDivisionError:
     		print "You can't divide by zero, you silly."
		r = 1.0
	x = r * cos(alt)*sin(az-view_az)
	y = r * (cos(view_alt)*sin(alt)-sin(view_alt)*cos(alt)*cos(az-view_az))
	x = r * cos(alt)*sin(az-view_az)

        return (x,y)

def convert_coordinates(x,y,local):
	global horizon_proj_flag

	if horizon_proj_flag == True:
		star = ephem.FixedBody()
		star._ra = x 
		star._dec = y
		star._epoch = 2000
		star.compute(local)
		az = star.az
		alt = star.alt
	else:
		az = ephem.degrees(ephem.hours(x))
		alt = ephem.degrees(y)

	return (az, alt)

def toggle_on_off_controls(widget,window,vbox,local,on_off_controls_show_flag2):
        global on_off_controls_show_flag

        if on_off_controls_show_flag == True:
                on_off_controls_show_flag = False
        else:
                on_off_controls_show_flag = True

        if on_off_controls_show_flag2 == 0:
                on_off_controls_show_flag = False
        else:
                on_off_controls_show_flag = True

	vbox.destroy()

	plot_sky("clicked",window,vbox,local)

        return

def toggle_solar_system_plot_date_time_controls(widget,window,vbox,object,local,planet_orbit,date_time_controls_show_flag2):
	print "in toggle_mars_plot_date_time_controls"
        global date_time_controls_show_flag

        if date_time_controls_show_flag == True:
                date_time_controls_show_flag = False
        else:
                date_time_controls_show_flag = True

        if date_time_controls_show_flag2 == 0:
                date_time_controls_show_flag = False
        else:
                date_time_controls_show_flag = True

	vbox.destroy()

	plot_solar_system_position("clicked",window,vbox,object,local,planet_orbit)

        return

def toggle_moon_plot_date_time_controls(widget,window,vbox,object,local,date_time_controls_show_flag2):
        global date_time_controls_show_flag

        if date_time_controls_show_flag == True:
                date_time_controls_show_flag = False
        else:
                date_time_controls_show_flag = True

        if date_time_controls_show_flag2 == 0:
                date_time_controls_show_flag = False
        else:
                date_time_controls_show_flag = True

	vbox.destroy()

	plot_moon_phase("clicked",window,vbox,local)

        return

def toggle_sky_plot_date_time_controls(widget,window,vbox,local,date_time_controls_show_flag2):
        global date_time_controls_show_flag

        if date_time_controls_show_flag == True:
                date_time_controls_show_flag = False
        else:
                date_time_controls_show_flag = True

        if date_time_controls_show_flag2 == 0:
                date_time_controls_show_flag = False
        else:
                date_time_controls_show_flag = True

	vbox.destroy()

	plot_sky("clicked",window,vbox,local)

        return

def toggle_mars_plot_date_time_controls(widget,window,vbox,object,local,date_time_controls_show_flag2):
	print "in toggle_mars_plot_date_time_controls"
        global date_time_controls_show_flag

        if date_time_controls_show_flag == True:
                date_time_controls_show_flag = False
        else:
                date_time_controls_show_flag = True

        if date_time_controls_show_flag2 == 0:
                date_time_controls_show_flag = False
        else:
                date_time_controls_show_flag = True

	vbox.destroy()

	plot_planet_moon_position("clicked",window,vbox,object,local)

        return

def myfrange(start, stop, n):
    	l = [0.0] * n
    	nm1 = n - 1
    	nm1inv = 1.0 / nm1
    	for i in range(n):
        	l[i] = nm1inv * (start*(nm1 - i) + stop*i)
    	return l 

class plot_dome(gtk.DrawingArea):
    	global magnitude_limit

    	def __init__(self,window,local):
        	gtk.DrawingArea.__init__(self)
        	self.connect("expose_event", self.expose_plot_dome, window, local)
		self.connect('button_press_event', self.button_press_event_cb, local)
		self.set_events(gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.POINTER_MOTION_MASK)	

		return
      
    	def button_press_event_cb(self, widget, event, local):
		global polar_sky_flag
		global azimuth_center
		global altitude_center
		global polar_field_of_view
		global equi_field_of_view
		global x_shift_degrees
		global y_shift_degrees
		global star_show_flag
		global search_alt_dec
		global search_az_ra
		global object_search_flag
		global select_object

        	if event.button == 1:
			if polar_sky_flag == False:
				if object_search_flag == True:
					azimuth_center = search_az_ra * 180/pi
					altitude_center = search_alt_dec * 180/pi
					object_search_flag = False
				else:
					azimuth_center = int(azimuth_center + ((event.x - min(rect.width/2,rect.height/2)) / (min(rect.width/2,rect.height/2))) * equi_field_of_view/2)
					altitude_center = int(altitude_center + ((min(rect.width/2,rect.height/2) - event.y) / (min(rect.width/2,rect.height/2))) * equi_field_of_view/2)

					select_object = "" # needed?

				if azimuth_center > 360:
					azimuth_center = 360
				if azimuth_center < -360:
					azimuth_center = -360
				if altitude_center > 90:
					altitude_center = 90
				if altitude_center < -90:
					altitude_center = -90
			else:
				if object_search_flag == True:
					print search_az_ra, search_alt_dec
					x_shift_degrees = -(180 - search_az_ra * 180/pi) 
					y_shift_degrees =  90 - (search_alt_dec * 180/pi) 
					object_search_flag = False
				else:
					x = rect.x + rect.width / 2
					y = rect.y + rect.height / 2
					minxy = min(x,y)
					rel_x = (event.x-x)/minxy
					rel_y = (event.y-y)/minxy

					x_shift_degrees = x_shift_degrees + (rel_x * 90) # good
					y_shift_degrees = y_shift_degrees + (rel_y * 90) # good

					select_object = ""
					
			gtk.gdk.threads_enter
    			self.queue_draw()
			gtk.gdk.threads_leave

        	elif event.button == 3:
        		print event.x, event.y

        	return True

    	def expose_plot_dome(self, widget, event, window, local):
        	self.context = widget.window.cairo_create()
        
        	# set a clip region for the expose event
        	self.context.rectangle(event.area.x, event.area.y,
                               event.area.width, event.area.height)
        	self.context.clip()
        
        	self.draw_plot_dome(self.context,window,local)
        
        	return False
    
    	def draw_plot_dome(self, context,window,local):
        	global magnitude_limit
		global rect
		global sun_compass_flag
		global horizon_proj_flag
		global constellation_show_flag
		global messier_show_flag
		global milkyway_show_flag
		global controls_show_flag
		global controls_show_flag
		global inverse_show_flag
		global planet_show_flag
		global grid_show_flag
		global polar_sky_flag
		global star_show_flag
		global satellite_show_flag
		global ngc_show_flag
		global asteroid_show_flag
		global comet_show_flag
		global neo_show_flag
		global planet_treestore
		global star_treestore
		global messier_treestore
		global ngc_treestore
		global satellite_treestore
		global asteroid_treestore
		global comet_treestore
		global neo_treestore
		global x_shift_degrees
		global y_shift_degrees
		global asteroid_read_flag
		global comet_read_flag
		global neo_read_flag
		global daylight_show_flag

        	rect = self.get_allocation()
        	start_radius = 100
        	x = rect.x + rect.width / 2
        	y = rect.y + rect.height / 2

		if sun_compass_flag == True:      
			horizon_proj_flag_save = horizon_proj_flag
			constellation_show_flag_save = constellation_show_flag
			messier_show_flag_save = messier_show_flag
			milkyway_show_flag_save = milkyway_show_flag
			controls_show_flag_save = controls_show_flag
			inverse_show_flag_save = inverse_show_flag
			planet_show_flag_save = planet_show_flag
			grid_show_flag_save = grid_show_flag
			polar_sky_flag_save = polar_sky_flag
			star_show_flag_save = star_show_flag
			asteroid_show_flag_save = asteroid_show_flag
			comet_show_flag_save = comet_show_flag
			neo_show_flag_save = neo_show_flag
			x_shift_degrees_save = x_shift_degrees
			y_shift_degrees_save = y_shift_degrees
			horizon_proj_flag = True
			constellation_show_flag = False
			messier_show_flag = False
			milkyway_show_flag = False
			controls_show_flag = False
			inverse_show_flag = True
			planet_show_flag = True
			grid_show_flag = True
			polar_sky_flag = True
			star_show_flag = False
			asteroid_show_flag = False
			comet_show_flag = False
			neo_show_flag = False
			x_shift_degrees = 0
			y_shift_degrees = 0

        	context.rectangle(0, 0, rect.width, rect.height)

		if inverse_show_flag == False:
			if daylight_show_flag == False:
        			context.set_source_rgba(0.1,0.1,0.1) # dark grey rectangle
			else:
				sun = ephem.Sun()
				sun.compute(local)
				sun_alt = sun.alt * 180/pi
				if sun_alt >= 0:
					#context.set_source_rgb(daylight_plot_color[0], daylight_plot_color[1], daylight_plot_color[2])
					context.set_source_rgb(daylight_plot_color[0] * 0.9, daylight_plot_color[1] * 0.9, daylight_plot_color[2] * 0.9)
				elif (sun_alt > -18) and (sun_alt < 0):
					#context.set_source_rgb((1+(sun_alt/18)) * daylight_plot_color[0], (1+(sun_alt/18)) * daylight_plot_color[1], (1+(sun_alt/18)) * daylight_plot_color[2])
					context.set_source_rgb((1+(sun_alt/18)) * daylight_plot_color[0] * 0.9, (1+(sun_alt/18)) * daylight_plot_color[1] * 0.9, (1+(sun_alt/18)) * daylight_plot_color[2] * 0.9)
				else:
        				context.set_source_rgba(0.1,0.1,0.1) # dark grey rectangle
		else:
        		context.set_source_rgba(0.9,0.9,0.9) # light grey rectangle

        	context.fill()

		if polar_sky_flag == True:
        		radius = min(rect.width / 2, rect.height / 2) * (90/polar_field_of_view)
        		context.arc(x, y, radius, 0, 2 * math.pi)
        		mult_radius = float(radius) / start_radius
		else:
			alt = 90
			az = 0
			y1 = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
			x1 = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
			alt = -90
			az = 360
			y2 = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
			x2 = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
			context.rectangle(x1,y1,x2-x1,y2-y1)

		if inverse_show_flag == False:
			if daylight_show_flag == False:
        			context.set_source_rgb(0,0,0) # black sky color
			else:
				sun = ephem.Sun()
				sun.compute(local)
				sun_alt = sun.alt * 180/pi
				if sun_alt >= 0:
					context.set_source_rgb(daylight_plot_color[0], daylight_plot_color[1], daylight_plot_color[2])
				elif (sun_alt > -18) and (sun_alt < 0):
					context.set_source_rgb((1+(sun_alt/18)) * daylight_plot_color[0], (1+(sun_alt/18)) * daylight_plot_color[1], (1+(sun_alt/18)) * daylight_plot_color[2])
				else:
        				context.set_source_rgb(0,0,0) # black sky color
        		context.fill_preserve()
			context.fill()
        		context.stroke()
        		context.set_source_rgb(1,1,1) # return to white color
		else:
			if sun_compass_flag == False:
				context.set_source_rgb(1,1,1) # white sky color
				context.fill_preserve()
				context.fill()
				context.stroke()
				context.set_source_rgb(0,0,0) # return to black color
			else:
				context.set_source_rgb(daylight_plot_color[0], daylight_plot_color[1], daylight_plot_color[2])
				context.fill_preserve()
				context.fill()
				context.stroke()
				context.set_source_rgb(0,0,0) # return to black color

		current_local_date = ephem.Date(ephem.localtime(local.date))

		if ((ephem.date(local.date)-ephem.date(ephem.localtime(local.date)))*24) == (time.timezone/60/60):
			local_time_zone_daylight_flag = 0
		else:
			local_time_zone_daylight_flag = time.daylight

		local_time_zone_name = time.tzname

		if date_in_gmt == True:
			date_title = 'DATE/TIME: '+str(local.date)+' GMT'
		else:
			date_title = 'DATE/TIME: '+str(current_local_date)+' '+local_time_zone_name[local_time_zone_daylight_flag]

		if horizon_proj_flag == True:
			coordinates_title = 'COORD SYSTEM: '+'Horizon (Azimuth/Altitude)'
		else:
			coordinates_title = 'COORD SYSTEM: '+'Equatorial (RA/DEC)'
		location_title = 'LOCATION: '+str(local.lat)+u'\xb0 '+str(local.long)+u'\xb0'
		azimuth_title = 'AZIMUTH CENTER: '+str(azimuth_center)+u'\xb0'

		if polar_sky_flag == True:
			field_of_view_title = 'POLAR FIELD OF VIEW: '+str(polar_field_of_view)+u'\xb0'
			if horizon_proj_flag == True:
				x_shift_title = 'AZIMUTH CENTER: '+str(ephem.degrees((180 + x_shift_degrees) * pi/180))+u'\xb0'
				y_shift_title = 'ALTITUDE CENTER: '+str(ephem.degrees((90 - y_shift_degrees) * pi/180))+u'\xb0'
			else:
				x_shift_title = 'RA CENTER: '+str(ephem.hours(-x_shift_degrees * pi/180))+'h'
				y_shift_title = 'DEC CENTER: '+str(ephem.degrees((90 - y_shift_degrees) * pi/180))+u'\xb0'
		else:
			field_of_view_title = 'EQUI FIELD OF VIEW: '+str(equi_field_of_view)+u'\xb0'

		altitude_title = 'ALTITUDE CENTER: '+str(altitude_center)+u'\xb0'
		magnitude_title = 'MAGNITUDE LIMIT: '+str(magnitude_limit)
		size_title = 'OBJECT SIZE: '+str(object_size)
		bright_title = 'OBJECT BRIGHT: '+str(object_bright)
		grid_title = 'GRID SPACING: '+str(grid_space)+u'\xb0'
		text_title = 'TEXT SIZE: '+str(font_size)+u'\xb0'

		context.set_source_rgb(1,0.65,0) # date orange color
		font = "Sans"
		context.select_font_face(font)
		context.set_font_size(font_size)
		context.move_to(0,10)
		context.text_path(date_title)
		context.set_line_width(1.0)
		context.fill()
		context.stroke()

		if star_show_flag == True:
			star_treestore.clear()

			context.set_line_width(1.0)
			font = "Sans"
			context.select_font_face(font)
			context.set_font_size(font_size)

			run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
			data_dir = os.path.join(run_dir, 'data')
			stars_file = os.path.join(data_dir, 'stars.dat')
			f = open(stars_file,"rb")
			while 1:
				t = f.readline()
				if t == '':
					break
				if '\n' in t:
					t = t[:-1]
				if '\r' in t:
					t = t[:-1]
				a = t.split()
				ra = a[1]
				dec = a[2]
				b = ra.split(":")
				c = dec.split(":")				

				mag = a[3]
				desc = a[4]
				if a[5] == "NAME":
					blank = " "
					star_name = blank.join(a[6:])
				else:
					star_name = ""
				if float(mag) > magnitude_limit:
					break
				azalt = convert_coordinates(ra, dec, local)
				az = azalt[0]
				alt = azalt[1]
				if star_name != "":
					piter = star_treestore.append(None, [star_name, az, alt, ra, dec, float(az), float(alt), float(b[0]) + float(b[1])/60, float(c[0]) + float(c[1])/60])
					if star_name == select_object:
						if horizon_proj_flag == True:
							search_az_ra = az 
							search_alt_dec = alt
						else:
							search_az_ra = pi - ra
							search_alt_dec = dec
						x_shift_degrees = (search_az_ra * 180/pi) # good
						y_shift_degrees = (search_alt_dec * 180/pi) - 90 # good
				if polar_sky_flag == True:
					az = float(2*pi/4*3-az)
					alt = float(alt)
					az_alt = (az,alt)
					xy = az_alt_to_polar(az_alt)
					x = xy[0] * mult_radius + rect.width / 2
					y = xy[1] * mult_radius + rect.height / 2
					mult_radius = float(radius) / start_radius
					plot_size=7-float(mag)+object_size
					if plot_size < 1:
						plot_size = 1
					show_mag = float(mag)/7 + object_bright
					if show_mag > 1:
						show_mag = 1
					if show_mag < 0:
						show_mag = 0
					if inverse_show_flag == False:
						show_mag=1-float(show_mag)
					else:
						show_mag=float(show_mag)
					context.arc(x, y, plot_size, 0, 2 * math.pi)
					context.set_source_rgb(show_mag,show_mag,show_mag) # star shades of grey color
					context.fill()
					context.stroke()

					if star_name_show_flag == True:
						context.set_source_rgb(star_plot_color[0], star_plot_color[1], star_plot_color[2])
						context.move_to(x + 5, y + 5)
						context.text_path(star_name)
						context.fill()
						context.stroke()
				else:
					if horizon_proj_flag == False:
						az = (2*pi)-az
					az = az/(2*pi)*360
					alt = alt/(2*pi)*360
					y = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
					x = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
					plot_size=7-float(mag)+object_size
					if plot_size < 1:
						plot_size = 1
					show_mag = float(mag)/7 + object_bright
					if show_mag > 1:
						show_mag = 1
					if show_mag < 0:
						show_mag = 0
					if inverse_show_flag == False:
						show_mag=1-float(show_mag)
					else:
						show_mag=float(show_mag)
					context.arc(x, y, plot_size, 0, 2 * math.pi)
					context.set_source_rgb(show_mag,show_mag,show_mag) # star shades of grey color
					context.fill()
					context.stroke()

					if star_name_show_flag == True:
						context.set_source_rgb(star_plot_color[0], star_plot_color[1], star_plot_color[2])
						context.move_to(x + 5, y + 5)
						context.text_path(star_name)
						context.fill()
						context.stroke()

			f.close()

		if grid_show_flag == True:
			context.set_line_width(0.5)
			if polar_sky_flag == True: 
				for az in range(0,361,grid_space):
					for alt in range(-81,81,grid_inc):
						az2 = az*pi/180
						if sun_compass_flag == False:
							az3 = float(2*pi/4*3-az2)
						else:
							az3 = float(az2-2*pi/4)
						alt2 = alt*pi/180
						az_alt = (az3,alt2)
						xy = az_alt_to_polar(az_alt)
						x1 = xy[0] * mult_radius + rect.width / 2
						y1 = xy[1] * mult_radius + rect.height / 2
						context.move_to(x1, y1)
						context.set_line_width(0.5)
						context.arc(x1, y1, 0.2, 0, 2 * math.pi)
						if sun_compass_flag == False:
							context.set_source_rgb(0,1,0) # azimuth value green color
						else:
							context.set_source_rgb(0,0,0) # azimuth value black color
						context.stroke()

						if (alt % 90) == 0:
							context.move_to(x1-10, y1+5)
							if horizon_proj_flag == True:
								context.text_path(str(az)+u'\xb0')
							else:
								ra_show = str(ephem.hours(az*pi/180))
								ra_split = ra_show.split(":")
								context.text_path(ra_split[0]+":"+ra_split[1]+'h')
							context.fill()
							context.stroke()

						if sun_compass_flag == True:
							if alt == 0:
								if (az == 0):
									context.set_font_size(font_size+20)
									context.move_to(x1-10, y1+30)
									context.set_source_rgb(1,1,1) # horizon north white color
									context.set_line_width(1.0)
									context.text_path("N")
									context.fill()
									context.stroke()
									context.set_font_size(font_size)
								if (az == 90):
									context.set_font_size(font_size+20)
									context.move_to(x1-25, y1+10)
									context.set_source_rgb(1,1,1) # horizon north white color
									context.set_line_width(1.0)
									context.text_path("E")
									context.fill()
									context.stroke()
									context.set_font_size(font_size)
								if (az == 180):
									context.set_font_size(font_size+20)
									context.move_to(x1-10, y1-10)
									context.set_source_rgb(1,1,1) # horizon north white color
									context.set_line_width(1.0)
									context.text_path("S")
									context.fill()
									context.stroke()
									context.set_font_size(font_size)
								if (az == 270):
									context.set_font_size(font_size+20)
									context.move_to(x1+5, y1+10)
									context.set_source_rgb(1,1,1) # horizon north white color
									context.set_line_width(1.0)
									context.text_path("W")
									context.fill()
									context.stroke()
									context.set_font_size(font_size)

				for alt in range(-90,91,grid_space):
					for az in range(0,361,grid_inc):
						az2 = az*pi/180
						alt2 = alt*pi/180
						az_alt = (az2,alt2)
						xy = az_alt_to_polar(az_alt)
						x1 = xy[0] * mult_radius + rect.width / 2
						y1 = xy[1] * mult_radius + rect.height / 2
						context.move_to(x1, y1)
						context.set_line_width(0.5)
						context.arc(x1, y1, 0.2, 0, 2 * math.pi)
						if sun_compass_flag == False:
							context.set_source_rgb(0,1,0) # azimuth value green color
						else:
							context.set_source_rgb(0,0,0) # azimuth value black color
						context.stroke()
						if (az % 90) == 0 and alt != 0:
							context.move_to(x1-10, y1+5)
							context.text_path(str(alt)+u'\xb0')
							context.fill()
							context.stroke()

			else:	
				for az in range(0,361,grid_space):
					alt = 0
					if horizon_proj_flag == True:
						x = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
					else:
						x = (360-az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
					y = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)

					context.move_to(x-10, y+5)
					if horizon_proj_flag == True:
						context.text_path(str(az)+u'\xb0')
					else:
						ra_show = str(ephem.hours(az*pi/180))
						ra_split = ra_show.split(":")
						context.text_path(ra_split[0]+":"+ra_split[1]+'h')
					context.set_source_rgb(0,1,0) # azimuth value green color
					context.fill()
					context.stroke()

				for alt in range(-90,91,grid_space):
					az = 0
					x = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
					y = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)

					context.move_to(x-25, y+5)
					context.text_path(str(alt)+u'\xb0')
					context.set_source_rgb(0,1,0) # altitude value green color
					context.fill()
					context.stroke()

				for alt in range(-90,91,grid_space):
					az = 360 
					x = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
					y = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)

					context.move_to(x+5, y)
					context.text_path(str(alt)+u'\xb0')
					context.set_source_rgb(0,1,0) # altitude value green color
					context.fill()
					context.stroke()
				
				for az in range(0,361,grid_space):
					alt = 90
					y1 = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
					if horizon_proj_flag == True:
						x1 = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
					else:
						x1 = (360-az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
					alt = -90
					y2 = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
					context.set_dash([1],1)
					context.move_to(x1, y1)
					context.line_to(x1, y2)
					context.set_source_rgb(0,1,0) # azimuth grid green color
					context.stroke()

					context.move_to(x1-10, y1+5)
					if horizon_proj_flag == True:
						context.text_path(str(az)+u'\xb0')
					else:
						ra_show = str(ephem.hours(az*pi/180))
						ra_split = ra_show.split(":")
						context.text_path(ra_split[0]+":"+ra_split[1]+'h')
					context.set_source_rgb(0,1,0) # azimuth value green color
					context.fill()
					context.stroke()
				
					context.move_to(x1-10, y2+5)
					if horizon_proj_flag == True:
						context.text_path(str(az)+u'\xb0')
					else:
						ra_show = str(ephem.hours(az*pi/180))
						ra_split = ra_show.split(":")
						context.text_path(ra_split[0]+":"+ra_split[1]+'h')
					context.set_source_rgb(0,1,0) # azimuth value green color
					context.fill()
					context.stroke()
				
				for alt in range(-90,89,grid_space):
					az = 0
					y1 = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
					x1 = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
					az = 360
					y2 = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
					x2 = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
					context.move_to(x1, y1)
					context.line_to(x2, y2)
					context.set_source_rgb(0,1,0) # altitude grid green color
					context.stroke()
				
				alt = 90
				az = 0
				y1 = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
				x1 = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
				az = 360
				y2 = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
				x2 = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
				context.set_line_width(0.5)
				context.move_to(x1, y1)
				context.line_to(x2, y1)
				context.set_source_rgb(0,1,0) # azimuth grid top green color
				context.stroke()

				alt = -90
				az = 0
				y1 = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
				x1 = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
				az = 360
				y2 = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
				x2 = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
				context.set_line_width(0.5)
				context.move_to(x1, y1)
				context.line_to(x2, y1)
				context.set_source_rgb(0,1,0) # azimuth grid bottom green color
				context.stroke()
				
				alt = 90
				az = 0
				y1 = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
				x1 = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
				alt = -90
				y2 = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
				x2 = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
				context.set_line_width(0.5)
				context.move_to(x1, y1)
				context.line_to(x2, y2)
				context.set_source_rgb(0,1,0) # altitude grid left green color
				context.stroke()
				
				alt = 90
				az = 360
				y1 = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
				x1 = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
				alt = -90
				y2 = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
				x2 = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
				context.set_line_width(0.5)
				context.move_to(x1, y1)
				context.line_to(x2, y2)
				context.set_source_rgb(0,1,0) # altitude grid right green color
				context.stroke()
				
		if milkyway_show_flag == True:
                	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
                	data_dir = os.path.join(run_dir, 'data')
                	milkyway_file = os.path.join(data_dir, 'milkyway.dat')
			f = open(milkyway_file,"rb")
			prev_az1 = -1
			good_az1 = 1
			milkyway_list = ([])
			if inverse_show_flag == False:
				context.set_source_rgba(1,1,1,0.1) # milkyway paint transparent white color
			else:
				context.set_source_rgba(0,0,1,0.1) # milkyway paint transparent blue color
			while 1:
				t = f.readline()
				if t == '':
					break
				if '\n' in t:
					t = t[:-1]
				if '\r' in t:
					t = t[:-1]
				a = t.split()
				ra1 = str(float(a[1]))
				dec1 = str(float(a[2]))
				azalt = convert_coordinates(ra1, dec1, local)
				az1 = azalt[0]
				alt1 = azalt[1]
				if polar_sky_flag == True:
					az1 = float(2*pi/4*3-az1)
					alt1 = float(alt1)
					az_alt1 = (az1,alt1)
					xy = az_alt_to_polar(az_alt1)
					x1 = xy[0] * mult_radius + rect.width / 2
					y1 = xy[1] * mult_radius + rect.height / 2
					mult_radius = float(radius) / start_radius
					if a[0] == "M":
						context.fill()
						context.stroke()
						context.close_path()
						context.move_to(x1, y1)
					else:
						context.line_to(x1, y1)
				else:
					if horizon_proj_flag == False:
						az1 = (2*pi)-az1
						az1 = az1/(2*pi)*360
						alt1 = alt1/(2*pi)*360
						y1 = min(rect.height,rect.width)/2 - (alt1-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
						x1 = (az1-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
						if a[0] == "M":
							context.fill()
							context.stroke()
							context.close_path()
							context.move_to(x1, y1)
						else:
							context.line_to(x1, y1)
					else:
						az1 = az1/(2*pi)*360
						alt1 = alt1/(2*pi)*360
						y1 = min(rect.height,rect.width)/2 - (alt1-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
						x1 = (az1-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
						if a[0] == "M":
							if good_az1 == 1:
								for coord in milkyway_list:
									x2 = coord[0]
									y2 = coord[1]
									context.line_to(x2, y2)
								context.fill()
								context.stroke()
								context.close_path()
							context.move_to(x1, y1)
							good_az1 = 1
							milkyway_list = ([])
						else:
							if abs(prev_az1 - az1) < 50:
								milkyway_list.append([x1,y1])
							else:
								good_az1 = -1
								milkway_list = ([])
						prev_az1 = az1

			if horizon_proj_flag == True:
				if good_az1 == 1:
					for coord in milkyway_list:
						x2 = coord[0]
						y2 = coord[1]
						context.line_to(x2, y2)
			context.fill()
			context.stroke()
			context.close_path()

			f.close()

		if constellation_show_flag == True:
			constellation_treestore.clear()

			context.set_line_width(1.0)

			run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
			data_dir = os.path.join(run_dir, 'data')
			constellation_file = os.path.join(data_dir, 'names.dat')
			f = open(constellation_file,"rb")
			while 1:
				t = f.readline()
				if t == '':
					break
				if '\n' in t:
					t = t[:-1]
				if '\r' in t:
					t = t[:-1]
				a = t.split(",")
				ra = str(float(a[0])/1000.0)
				dec = str(float(a[1])/100.0)
				constellation_name = a[2]
				azalt = convert_coordinates(ra, dec, local)
				az = azalt[0]
				alt = azalt[1]
				piter = constellation_treestore.append(None, [constellation_name, az, alt, ra, dec, float(az), float(alt), float(ra), float(dec)])
				if constellation_name == select_object:
					if horizon_proj_flag == True:
						search_az_ra = az 
						search_alt_dec = alt
					else:
						search_az_ra = pi - ra
						search_alt_dec = dec
					x_shift_degrees = (search_az_ra * 180/pi) # good
					y_shift_degrees = (search_alt_dec * 180/pi) - 90 # good

				if constellation_name_show_flag == True: # special case where we need to read contellation names file ALL the time but not always plot the names
					if polar_sky_flag == True:
						az = float(2*pi/4*3-az)
						alt = float(alt)
						az_alt = (az,alt)
						xy = az_alt_to_polar(az_alt)
						x = xy[0] * mult_radius + rect.width / 2
						y = xy[1] * mult_radius + rect.height / 2
						mult_radius = float(radius) / start_radius
						context.set_source_rgb(0.3,0.3,1) # constellation lines light blue color
						context.move_to(x, y)
						context.text_path(constellation_name)
						context.fill()
						context.stroke()
					else:
						if horizon_proj_flag == False:
							az = (2*pi)-az
						az = az/(2*pi)*360
						alt = alt/(2*pi)*360
						y = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
						x = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
						context.set_source_rgb(0.3,0.3,1) # constellation lines light blue color
						context.move_to(x, y)
						context.text_path(constellation_name)
						context.fill()
						context.stroke()

                	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
                	data_dir = os.path.join(run_dir, 'data')
                	constellation_file = os.path.join(data_dir, 'constellations.dat')
			f = open(constellation_file,"rb")
			while 1:
				t = f.readline()
				if t == '':
					break
				if '\n' in t:
					t = t[:-1]
				if '\r' in t:
					t = t[:-1]
				a = t.split(",")
				ra1 = str(float(a[1])/1000.0)
				dec1 = str(float(a[2])/100.0)
				ra2 = str(float(a[3])/1000.0)
				dec2 = str(float(a[4])/100.0)
				azalt = convert_coordinates(ra1, dec1, local)
				az1 = azalt[0]
				alt1 = azalt[1]
				azalt = convert_coordinates(ra2, dec2, local)
				az2 = azalt[0]
				alt2 = azalt[1]
				if polar_sky_flag == True:
					az1 = float(2*pi/4*3-az1)
					alt1 = float(alt1)
					az_alt1 = (az1,alt1)
					az2 = float(2*pi/4*3-az2)
					alt2 = float(alt2)
					az_alt2 = (az2,alt2)
					xy = az_alt_to_polar(az_alt1)
					x1 = xy[0] * mult_radius + rect.width / 2
					y1 = xy[1] * mult_radius + rect.height / 2
					xy = az_alt_to_polar(az_alt2)
					x2 = xy[0] * mult_radius + rect.width / 2
					y2 = xy[1] * mult_radius + rect.height / 2
					mult_radius = float(radius) / start_radius
					context.set_source_rgb(0.3,0.3,1) # constellation lines light blue color
					context.move_to(x1, y1)
					context.line_to(x2, y2)
					context.stroke()
				else:
					if horizon_proj_flag == False:
						az1 = (2*pi)-az1
						az2 = (2*pi)-az2
					az1 = az1/(2*pi)*360
					alt1 = alt1/(2*pi)*360
					az2 = az2/(2*pi)*360
					alt2 = alt2/(2*pi)*360
					if az1 > az2:
						az3 = az1
						az1 = az2
						az2 = az3
						alt3 = alt1
						alt1 = alt2
						alt2 = alt3
					if abs(az1 - az2) > 100:
						az2 = az2 - 360
						y1 = min(rect.height,rect.width)/2 - (alt1-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
						x1 = (az1-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
						y2 = min(rect.height,rect.width)/2 - (alt2-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
						x2 = (az2-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
						context.set_source_rgb(0.3,0.3,1) # constellation lines light blue color
						context.move_to(x1, y1)
						context.line_to(x2, y2)
						context.stroke()
						
						az2 = az2 + 360
						az1 = az1 + 360
						y1 = min(rect.height,rect.width)/2 - (alt1-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
						x1 = (az1-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
						y2 = min(rect.height,rect.width)/2 - (alt2-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
						x2 = (az2-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
						context.set_source_rgb(0.3,0.3,1) # constellation lines light blue color
						context.move_to(x1, y1)
						context.line_to(x2, y2)
						context.stroke()
					else:
						y1 = min(rect.height,rect.width)/2 - (alt1-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
						x1 = (az1-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
						y2 = min(rect.height,rect.width)/2 - (alt2-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
						x2 = (az2-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
						context.set_source_rgb(0.3,0.3,1) # constellation lines light blue color
						context.move_to(x1, y1)
						context.line_to(x2, y2)
						context.stroke()

			f.close()

		if ecliptic_show_flag == True:
			context.set_line_width(1.0)
			if inverse_show_flag == False:
				context.set_source_rgb(1,1,0) # ecliptic circle yellow color
			else:
				context.set_source_rgb(1,0,0) # ecliptic circle red color
			for i in range (0,360,1):
				star = ephem.FixedBody()
				ecliptic_x = ephem.Ecliptic(i*pi/180,0)
				ecliptic_y = ephem.Equatorial(ecliptic_x)
				if horizon_proj_flag == True:
					star._ra = ecliptic_y.ra
					star._dec = ecliptic_y.dec
					star._epoch = 2000
					star.compute(local)
					az = star.az 
					alt = star.alt
				else:
					az = ecliptic_y.ra
					alt = ecliptic_y.dec
				if polar_sky_flag == True:
					az = float(2*pi/4*3-az)
					alt = float(alt)
					az_alt = (az,alt)
					xy = az_alt_to_polar(az_alt)
					mult_radius = float(radius) / start_radius
					x = xy[0] * mult_radius + rect.width / 2
					y = xy[1] * mult_radius + rect.height / 2
					context.arc(x, y, 0.5, 0, 2 * math.pi)
					context.fill()
					context.stroke()
					if (i == 0):
						context.arc(x, y, 5, 0, 2 * math.pi)
						#context.fill()
						context.stroke()
				else:
					if horizon_proj_flag == False:
						az = (2*pi)-az
					az = az/(2*pi)*360
					alt = alt/(2*pi)*360
					y = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
					x = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
					context.arc(x, y, 0.5, 0, 2 * math.pi)
					context.fill()
					context.stroke()
					if (i == 0):
						context.arc(x, y, 5, 0, 2 * math.pi)
						#context.fill()
						context.stroke()

		if galactic_show_flag == True:
			context.set_line_width(1.0)
			for i in range (0,360,1):
				star = ephem.FixedBody()
				ecliptic_x = ephem.Galactic(i*pi/180,0)
				ecliptic_y = ephem.Equatorial(ecliptic_x)
				if horizon_proj_flag == True:
					star._ra = ecliptic_y.ra
					star._dec = ecliptic_y.dec
					star._epoch = 2000
					star.compute(local)
					az = star.az 
					alt = star.alt
				else:
					az = ecliptic_y.ra
					alt = ecliptic_y.dec
				if polar_sky_flag == True:
					az = float(2*pi/4*3-az)
					alt = float(alt)
					az_alt = (az,alt)
					xy = az_alt_to_polar(az_alt)
					mult_radius = float(radius) / start_radius
					x = xy[0] * mult_radius + rect.width / 2
					y = xy[1] * mult_radius + rect.height / 2
					context.arc(x, y, 0.5, 0, 2 * math.pi)
					context.set_source_rgb(0.5,0.5,1) # galactic line light blue color
					context.fill()
					context.stroke()
					if (i == 0):
						context.arc(x, y, 5, 0, 2 * math.pi)
						context.set_source_rgb(0.5,0.5,1) # galactic center light blue color
						#context.fill()
						context.stroke()
				else:
					if horizon_proj_flag == False:
						az = (2*pi)-az
					az = az/(2*pi)*360
					alt = alt/(2*pi)*360
					y = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
					x = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
					context.arc(x, y, 0.5, 0, 2 * math.pi)
					context.set_source_rgb(0.5,0.5,1) # galactic line light blue color
					context.fill()
					context.stroke()
					if (i == 0):
						context.arc(x, y, 5, 0, 2 * math.pi)
						context.set_source_rgb(0.5,0.5,1) # galactic center light blue color
						#context.fill()
						context.stroke()

		if horizon_show_flag == True:
			context.set_line_width(1.0)
			for i in range (0,360,1):
				if horizon_proj_flag == True:
					az = i
					az2 = az*pi/180
					alt = 0
					alt2 = alt*pi/180
				else:
					az2 = local.radec_of(i*pi/180,0)[0]
					alt2 = local.radec_of(i*pi/180,0)[1]

				if polar_sky_flag == True:
					az3 = float(2*pi/4*3-az2)
					az_alt = (az3,alt2)
					xy = az_alt_to_polar(az_alt)
					mult_radius = float(radius) / start_radius
					x = xy[0] * mult_radius + rect.width / 2
					y = xy[1] * mult_radius + rect.height / 2
					context.arc(x, y, 0.5, 0, 2 * math.pi)
					context.set_source_rgb(1,0.65,0) # horizon line orange color
					context.fill()
					context.stroke()
					if (i == 0):
						context.move_to(x, y)
						context.move_to(x-5, y+15)
						context.set_source_rgb(1,0.65,0) # horizon north orange color
						context.text_path("N")
						context.fill()
						context.stroke()
					if (i == 90):
						context.move_to(x, y)
						context.move_to(x+3, y+5)
						context.set_source_rgb(1,0.65,0) # horizon east orange color
						context.text_path("E")
						context.fill()
						context.stroke()
					if (i == 180):
						context.move_to(x, y)
						context.move_to(x-3, y-10)
						context.set_source_rgb(1,0.65,0) # horizon south orange color
						context.text_path("S")
						context.fill()
						context.stroke()
					if (i == 270):
						context.move_to(x, y)
						context.move_to(x-15, y+5)
						context.set_source_rgb(1,0.65,0) # horizon west orange color
						context.text_path("W")
						context.fill()
						context.stroke()
				else:
					if horizon_proj_flag == False:
						az = (2*pi)-az2
					az = az2/(2*pi)*360
					alt = alt2/(2*pi)*360
					y = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
					x = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
					context.arc(x, y, 0.5, 0, 2 * math.pi)
					context.set_source_rgb(1,0.65,0) # horizon line orange color
					context.fill()
					context.stroke()
					if (i == 0):
						context.move_to(x, y)
						context.set_source_rgb(1,0.65,0) # horizon north orange color
						context.text_path("N")
						context.fill()
						context.stroke()
					if (i == 90):
						context.move_to(x, y)
						context.set_source_rgb(1,0.65,0) # horizon east orange color
						context.text_path("E")
						context.fill()
						context.stroke()
					if (i == 180):
						context.move_to(x, y)
						context.set_source_rgb(1,0.65,0) # horizon south orange color
						context.text_path("S")
						context.fill()
						context.stroke()
					if (i == 270):
						context.move_to(x, y)
						context.set_source_rgb(1,0.65,0) # horizon west orange color
						context.text_path("W")
						context.fill()
						context.stroke()

		if messier_show_flag == True:
			messier_treestore.clear()

			context.set_line_width(1.0)
                	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
                	data_dir = os.path.join(run_dir, 'data')
                	messier_file = os.path.join(data_dir, 'messier.dat')
			f = open(messier_file,"rb")
			while 1:
				t = f.readline()
				if t == '':
					break
				if '\n' in t:
					t = t[:-1]
				if '\r' in t:
					t = t[:-1]
				a = t.split()
				messier_name = "M"+a[0]
				ra = a[3]+":"+a[4]
				dec = a[5]+":"+a[6]
				mag = a[7]
				if float(mag) > magnitude_limit:
					break
				azalt = convert_coordinates(ra, dec, local)
				az = azalt[0]
				alt = azalt[1]
				piter = messier_treestore.append(None, [messier_name, az, alt, ra, dec, float(az), float(alt), float(a[3])+ float(a[4])/60, float(a[5]) + float(a[6])/60])
				if messier_name == select_object:
					if horizon_proj_flag == True:
						search_az_ra = az 
						search_alt_dec = alt
					else:
						search_az_ra = pi - ra
						search_alt_dec = dec
					x_shift_degrees = (search_az_ra * 180/pi) # good
					y_shift_degrees = (search_alt_dec * 180/pi) - 90 # good
				if polar_sky_flag == True:
					az = float(2*pi/4*3-az)
					alt = float(alt)
					az_alt = (az,alt)
					xy = az_alt_to_polar(az_alt)
					x = xy[0] * mult_radius + rect.width / 2
					y = xy[1] * mult_radius + rect.height / 2
					mult_radius = float(radius) / start_radius
					context.save() # required since altering scale
					context.translate(x,y) # dare
					context.scale(1,0.5) # dare
					plot_size=7+object_size
					if plot_size < 1:
						plot_size = 1
					context.arc(0, 0, plot_size, 0, 2 * math.pi) # dare
					context.set_source_rgb(messier_plot_color[0], messier_plot_color[1], messier_plot_color[2],)
					context.stroke()
					context.restore() # required since restoring scale

					if messier_name_show_flag == True:
						context.set_source_rgb(messier_plot_color[0], messier_plot_color[1], messier_plot_color[2],)
						font = "Sans"
						context.select_font_face(font)
						context.set_font_size(font_size)
						context.move_to(x + 5, y + 5)
						context.text_path(messier_name)
						context.fill()
						context.stroke()
				else:
					if horizon_proj_flag == False:
						az = (2*pi)-az
					az = az/(2*pi)*360
					alt = alt/(2*pi)*360
					y = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
					x = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
					plot_size=7+object_size
					if plot_size < 1:
						plot_size = 1
					context.save() # required since altering scale
					context.translate(x,y) # dare
					context.scale(1,0.5) # dare
					context.arc(0, 0, plot_size, 0, 2 * math.pi) # dare
					context.set_source_rgb(messier_plot_color[0], messier_plot_color[1], messier_plot_color[2],)
					context.stroke()
					context.restore() # required since restoring scale

					if messier_name_show_flag == True:
						context.set_source_rgb(messier_plot_color[0], messier_plot_color[1], messier_plot_color[2],)
						font = "Sans"
						context.select_font_face(font)
						context.set_font_size(font_size)
						context.move_to(x + 5, y + 5)
						context.text_path(messier_name)
						context.fill()
						context.stroke()

			f.close()

		if ngc_show_flag == True:		
			ngc_treestore.clear()

			context.set_line_width(1.0)
                	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
                	data_dir = os.path.join(run_dir, 'data')
                	ngc_file = os.path.join(data_dir, 'ngc.dat')
			try:
				f = open(ngc_file,"rb")
			except:
				on_show_note("clicked",window,"NGC data file missing -- download it from web window before trying to use it!","OK")
				print "NGC data is missing!  Ignoring request!"
				ngc_show_flag = False
			else:
				while 1:
					t = f.readline()
					if t == '':
						break
					if '\n' in t:
						t = t[:-1]
					if '\r' in t:
						t = t[:-1]
					a = t.split()
					ngc_name = a[0]
					ra = a[1]+":"+a[2]
					dec = a[3]+":"+a[4]
					mag = a[5]
					if float(mag) > magnitude_limit:
						break
					azalt = convert_coordinates(ra, dec, local)
					az = azalt[0]
					alt = azalt[1]
					piter = ngc_treestore.append(None, [ngc_name, az, alt, ra, dec, float(az), float(alt), float(a[1])+ float(a[2])/60, float(a[3]) + float(a[4])/60])
					if ngc_name == select_object:
						if horizon_proj_flag == True:
							search_az_ra = az 
							search_alt_dec = alt
						else:
							search_az_ra = pi - ra
							search_alt_dec = dec
						x_shift_degrees = (search_az_ra * 180/pi) # good
						y_shift_degrees = (search_alt_dec * 180/pi) - 90 # good
					if polar_sky_flag == True:
						az = float(2*pi/4*3-az)
						alt = float(alt)
						az_alt = (az,alt)
						xy = az_alt_to_polar(az_alt)
						x = xy[0] * mult_radius + rect.width / 2
						y = xy[1] * mult_radius + rect.height / 2
						plot_size=7+object_size
						if plot_size < 1:
							plot_size = 1
						context.save() # required since altering scale
						context.translate(x,y) # dare
						context.scale(1,0.5) # dare
						context.arc(0, 0, plot_size, 0, 2 * math.pi) # dare
						context.set_source_rgb(ngc_plot_color[0], ngc_plot_color[1], ngc_plot_color[2])
						context.stroke()
						context.restore() # requires since restoring scale

						if ngc_name_show_flag == True:
							context.set_source_rgb(ngc_plot_color[0], ngc_plot_color[1], ngc_plot_color[2])
							font = "Sans"
							context.select_font_face(font)
							context.set_font_size(font_size)
							context.move_to(x + 5, y + 5)
							context.text_path(ngc_name)
							context.fill()
							context.stroke()
					else:
						if horizon_proj_flag == False:
							az = (2*pi)-az
						az = az/(2*pi)*360
						alt = alt/(2*pi)*360
						y = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
						x = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
						plot_size=7+object_size
						if plot_size < 1:
							plot_size = 1
						context.save() # required since altering scale
						context.translate(x,y) # dare
						context.scale(1,0.5) # dare
						context.arc(0, 0, plot_size, 0, 2 * math.pi) # dare
						context.set_source_rgb(ngc_plot_color[0], ngc_plot_color[1], ngc_plot_color[2])
						context.stroke()
						context.restore() # requires since restoring scale

						if ngc_name_show_flag == True:
							context.set_source_rgb(ngc_plot_color[0], ngc_plot_color[1], ngc_plot_color[2])
							font = "Sans"
							context.select_font_face(font)
							context.set_font_size(font_size)
							context.move_to(x + 5, y + 5)
							context.text_path(ngc_name)
							context.fill()
							context.stroke()

				f.close()

		if satellite_show_flag == True:
			satellite_treestore.clear()

			context.set_line_width(1.0)
                	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
                	data_dir = os.path.join(run_dir, 'data')
                	tle_file = os.path.join(data_dir, 'tle.dat')
			try:
				f = open(tle_file,"rb")
			except:
				on_show_note("clicked",window,"TLE data file missing -- download it from web window","OK")
				print "TLE data (satellite elements) are missing!  Ignoring request!"
				satellite_show_flag = False
			else:
				while 1:
					t = f.readline()
					if t == '':
						break
					if '\n' in t:
						t = t[:-1]
					if '\r' in t:
						t = t[:-1]
					a = t.split()
					tle_type = a[0]
					if (tle_type != "1") and (tle_type != "2"):
						satellite_name = t
						line1 = t
						t = f.readline()
						if t == '':
							break
						if '\n' in t:
							t = t[:-1]
						if '\r' in t:
							t = t[:-1]
						line2 = t
						t = f.readline()
						if t == '':
							break
						if '\n' in t:
							t = t[:-1]
						if '\r' in t:
							t = t[:-1]
						line3 = t
					sat = ephem.readtle(line1, line2, line3)
					sat.compute(local)
					ra = sat.ra
					dec = sat.dec
					azalt = convert_coordinates(ra, dec, local)
					az = azalt[0]
					alt = azalt[1]
					piter = satellite_treestore.append(None, [satellite_name, az, alt, ra, dec, float(az), float(alt), float(ra), float(dec)])
					if satellite_name == select_object:
						if horizon_proj_flag == True:
							search_az_ra = az 
							search_alt_dec = alt
						else:
							search_az_ra = pi - ra
							search_alt_dec = dec
						x_shift_degrees = (search_az_ra * 180/pi) # good
						y_shift_degrees = (search_alt_dec * 180/pi) - 90 # good
					if polar_sky_flag == True:
						az = float(2*pi/4*3-az)
						alt = float(alt)
						az_alt = (az,alt)
						xy = az_alt_to_polar(az_alt)
						x = xy[0] * mult_radius + rect.width / 2
						y = xy[1] * mult_radius + rect.height / 2
						plot_size=6+object_size
						if plot_size < 1:
							plot_size = 1
						context.arc(x, y, plot_size, 0, 2 * math.pi) # dare
						context.set_source_rgb(satellite_plot_color[0], satellite_plot_color[1], satellite_plot_color[2])
						context.stroke()

						if satellite_name_show_flag == True:
							context.set_source_rgb(satellite_plot_color[0], satellite_plot_color[1], satellite_plot_color[2])
							font = "Sans"
							context.select_font_face(font)
							context.set_font_size(font_size)
							context.move_to(x + 5, y + 5)
							context.text_path(satellite_name)
							context.fill()
							context.stroke()
					else:
						if horizon_proj_flag == False:
							az = (2*pi)-az
						az = az/(2*pi)*360
						alt = alt/(2*pi)*360
						y = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
						x = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
						plot_size=6+object_size
						if plot_size < 1:
							plot_size = 1
						context.arc(x, y, plot_size, 0, 2 * math.pi) # dare
						context.set_source_rgb(satellite_plot_color[0], satellite_plot_color[1], satellite_plot_color[2])
						context.stroke()

						if satellite_name_show_flag == True:
							context.set_source_rgb(satellite_plot_color[0], satellite_plot_color[1], satellite_plot_color[2])
							font = "Sans"
							context.select_font_face(font)
							context.set_font_size(font_size)
							context.move_to(x + 5, y + 5)
							context.text_path(satellite_name)
							context.fill()
							context.stroke()

				f.close()

		if asteroid_show_flag == True:
			context.set_line_width(1.0)
                	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
                	data_dir = os.path.join(run_dir, 'data')
                	asteroid_file = os.path.join(data_dir, 'asteroid.dat')
			try:
				f = open(asteroid_file,"rb")
			except:
				on_show_note("clicked",window,"Asteroid data file missing -- download it from web window","OK")
				print "Asteroid data (orbital elements) are missing!  Ignoring request!"
				asteroid_show_flag = False
			else:
				while 1:
					t = f.readline()
					if t == '':
						break
					if '\n' in t:
						t = t[:-1]
					if '\r' in t:
						t = t[:-1]
					a = t.split()
					line_type = a[0]
					if (line_type != "#"):
						b = t.split(",")
						c = b[0].split()
						asteroid_name = c[1]
						line1 = t
						t = f.readline()
						if t == '':
							break
						if '\n' in t:
							t = t[:-1]
						if '\r' in t:
							t = t[:-1]
						asteroid = ephem.readdb(line1)
						asteroid.compute(local)
						ra = asteroid.ra
						dec = asteroid.dec
						az = asteroid.az
						alt = asteroid.alt

						if asteroid_read_flag == False:
							piter = asteroid_treestore.append(None, [asteroid_name, asteroid.az, asteroid.alt, asteroid.ra, asteroid.dec, float(asteroid.az), float(asteroid.alt), float(asteroid.ra), float(asteroid.dec)])
							#print "filling asteroid_treestore in draw_plot_dome"

						if asteroid_name == select_object:
							if horizon_proj_flag == True:
								search_az_ra = az 
								search_alt_dec = alt
							else:
								search_az_ra = pi - ra
								search_alt_dec = dec
							x_shift_degrees = (search_az_ra * 180/pi) # good
							y_shift_degrees = (search_alt_dec * 180/pi) - 90 # good
						if polar_sky_flag == True:
							az = float(2*pi/4*3-az)
							alt = float(alt)
							az_alt = (az,alt)
							xy = az_alt_to_polar(az_alt)
							x = xy[0] * mult_radius + rect.width / 2
							y = xy[1] * mult_radius + rect.height / 2
							plot_size=5+object_size
							if plot_size < 1:
								plot_size = 1
							context.arc(x, y, plot_size, 0, 2 * math.pi)
							context.set_source_rgb(asteroid_plot_color[0], asteroid_plot_color[1], asteroid_plot_color[2])
							context.stroke()

							if asteroid_name_show_flag == True:
								context.set_source_rgb(asteroid_plot_color[0], asteroid_plot_color[1], asteroid_plot_color[2])
								font = "Sans"
								context.select_font_face(font)
								context.set_font_size(font_size)
								context.move_to(x + 5, y + 5)
								context.text_path(asteroid_name)
								context.fill()
								context.stroke()
						else:
							if horizon_proj_flag == False:
								az = (2*pi)-az
							az = az/(2*pi)*360
							alt = alt/(2*pi)*360
							y = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
							x = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
							plot_size=5+object_size
							if plot_size < 1:
								plot_size = 1
							context.arc(x, y, plot_size, 0, 2 * math.pi)
							context.set_source_rgb(asteroid_plot_color[0], asteroid_plot_color[1], asteroid_plot_color[2])
							context.stroke()

							if asteroid_name_show_flag == True:
								context.set_source_rgb(asteroid_plot_color[0], asteroid_plot_color[1], asteroid_plot_color[2])
								font = "Sans"
								context.select_font_face(font)
								context.set_font_size(font_size)
								context.move_to(x + 5, y + 5)
								context.text_path(asteroid_name)
								context.fill()
								context.stroke()

				asteroid_read_flag = True
				f.close()

		if comet_show_flag == True:
			context.set_line_width(1.0)
                	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
                	data_dir = os.path.join(run_dir, 'data')
                	comet_file = os.path.join(data_dir, 'comet.dat')
			try:
				f = open(comet_file,"rb")
			except:
				on_show_note("clicked",window,"Comet data file missing -- download it from web window","OK")
				print "Comet data (orbital elements) are missing!  Ignoring request!"
				comet_show_flag = False
			else:
				while 1:
					t = f.readline()
					if t == '':
						break
					if '\n' in t:
						t = t[:-1]
					if '\r' in t:
						t = t[:-1]
					a = t.split()
					line_type = a[0]
					if (line_type != "#"):
						b = t.split(",")
						c = b[0].split("/")
						comet_name = c[1]
						line1 = t
						t = f.readline()
						if t == '':
							break
						if '\n' in t:
							t = t[:-1]
						if '\r' in t:
							t = t[:-1]
						comet = ephem.readdb(line1)
						comet.compute(local)
						ra = comet.ra
						dec = comet.dec
						az = comet.az
						alt = comet.alt

						if comet_read_flag == False:
							piter = comet_treestore.append(None, [comet_name, comet.az, comet.alt, comet.ra, comet.dec, float(comet.az), float(comet.alt), float(comet.ra), float(comet.dec)])

						if comet_name == select_object:
							if horizon_proj_flag == True:
								search_az_ra = az 
								search_alt_dec = alt
							else:
								search_az_ra = pi - ra
								search_alt_dec = dec
							x_shift_degrees = (search_az_ra * 180/pi) # good
							y_shift_degrees = (search_alt_dec * 180/pi) - 90 # good
						if polar_sky_flag == True:
							az = float(2*pi/4*3-az)
							alt = float(alt)
							az_alt = (az,alt)
							xy = az_alt_to_polar(az_alt)
							x = xy[0] * mult_radius + rect.width / 2
							y = xy[1] * mult_radius + rect.height / 2
							plot_size = 10 + object_size
							if plot_size < 1:
								plot_size = 1
							context.save() # required since altering scale
							context.translate(x,y) # dare
							context.scale(1, 0.1) # dare
							context.arc(0, 0, plot_size, 0, 2 * math.pi) # dare
							context.set_source_rgb(comet_plot_color[0], comet_plot_color[1], comet_plot_color[2])
							context.fill()
							context.stroke()
							context.restore() # required since restoring scale

							if comet_name_show_flag == True:
								context.set_source_rgb(comet_plot_color[0], comet_plot_color[1], comet_plot_color[2])
								font = "Sans"
								context.select_font_face(font)
								context.set_font_size(font_size)
								context.move_to(x + 5, y + 5)
								context.text_path(comet_name)
								context.fill()
								context.stroke()
						else:
							if horizon_proj_flag == False:
								az = (2*pi)-az
							az = az/(2*pi)*360
							alt = alt/(2*pi)*360
							y = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
							x = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
							plot_size= 10 + object_size
							if plot_size < 1:
								plot_size = 1
							context.save() # required since altering scale
							context.translate(x,y) # dare
							context.scale(1, 0.1) # dare
							context.arc(0, 0, plot_size, 0, 2 * math.pi) # dare
							context.set_source_rgb(comet_plot_color[0], comet_plot_color[1], comet_plot_color[2])
							context.fill()
							context.stroke()
							context.restore() # required since restoring scale

							if comet_name_show_flag == True:
								context.set_source_rgb(comet_plot_color[0], comet_plot_color[1], comet_plot_color[2])
								font = "Sans"
								context.select_font_face(font)
								context.set_font_size(font_size)
								context.move_to(x + 5, y + 5)
								context.text_path(comet_name)
								context.fill()
								context.stroke()

				comet_read_flag = True
				f.close()

		if neo_show_flag == True:
			context.set_line_width(1.0)
                	run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
                	data_dir = os.path.join(run_dir, 'data')
                	neo_file = os.path.join(data_dir, 'neo.dat')
			try:
				f = open(neo_file,"rb")
			except:
				on_show_note("clicked",window,"NEO data file missing -- download it from web window","OK")
				print "NEO data (orbital elements) are missing!  Ignoring request!"
				neo_show_flag = False
			else:
				while 1:
					t = f.readline()
					if t == '':
						break
					if '\n' in t:
						t = t[:-1]
					if '\r' in t:
						t = t[:-1]
					a = t.split()
					line_type = a[0]
					if (line_type != "#"):
						b = t.split(",")
						c = b[0].split()
						neo_name = c[1]
						line1 = t
						t = f.readline()
						if t == '':
							break
						if '\n' in t:
							t = t[:-1]
						if '\r' in t:
							t = t[:-1]
						neo = ephem.readdb(line1)
						neo.compute(local)
						ra = neo.ra
						dec = neo.dec
						az = neo.az
						alt = neo.alt

						if neo_read_flag == False:
							piter = neo_treestore.append(None, [neo_name, neo.az, neo.alt, neo.ra, neo.dec, float(neo.az), float(neo.alt), float(neo.ra), float(neo.dec)])

						if neo_name == select_object:
							if horizon_proj_flag == True:
								search_az_ra = az 
								search_alt_dec = alt
							else:
								search_az_ra = pi - ra
								search_alt_dec = dec
							x_shift_degrees = (search_az_ra * 180/pi) # good
							y_shift_degrees = (search_alt_dec * 180/pi) - 90 # good
						if polar_sky_flag == True:
							az = float(2*pi/4*3-az)
							alt = float(alt)
							az_alt = (az,alt)
							xy = az_alt_to_polar(az_alt)
							x = xy[0] * mult_radius + rect.width / 2
							y = xy[1] * mult_radius + rect.height / 2
							plot_size=5+object_size
							if plot_size < 1:
								plot_size = 1
							context.arc(x, y, plot_size, 0, 2 * math.pi)
							context.set_source_rgb(neo_plot_color[0], neo_plot_color[1], neo_plot_color[2])
							context.stroke()

							if neo_name_show_flag == True:
								context.set_source_rgb(neo_plot_color[0], neo_plot_color[1], neo_plot_color[2])
								font = "Sans"
								context.select_font_face(font)
								context.set_font_size(font_size)
								context.move_to(x + 5, y + 5)
								context.text_path(neo_name)
								context.fill()
								context.stroke()
						else:
							if horizon_proj_flag == False:
								az = (2*pi)-az
							az = az/(2*pi)*360
							alt = alt/(2*pi)*360
							y = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
							x = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
							plot_size=5+object_size
							if plot_size < 1:
								plot_size = 1
							context.arc(x, y, plot_size, 0, 2 * math.pi)
							context.set_source_rgb(neo_plot_color[0], neo_plot_color[1], neo_plot_color[2])
							context.stroke()

							if comet_name_show_flag == True:
								context.set_source_rgb(neo_plot_color[0], neo_plot_color[1], neo_plot_color[2])
								font = "Sans"
								context.select_font_face(font)
								context.set_font_size(font_size)
								context.move_to(x + 5, y + 5)
								context.text_path(neo_name)
								context.fill()
								context.stroke()

				neo_read_flag = True
				f.close()

		if planet_show_flag == True:

			planet_treestore.clear()

			planet = ephem.Sun()
			self.draw_planet(context,rect,planet,local)

			planet = ephem.Mercury()
			self.draw_planet(context,rect,planet,local)

			planet = ephem.Venus()
			self.draw_planet(context,rect,planet,local)

			planet = ephem.Moon()
			self.draw_planet(context,rect,planet,local)

			planet = ephem.Mars()
			self.draw_planet(context,rect,planet,local)

			planet = ephem.Jupiter()
			self.draw_planet(context,rect,planet,local)

			planet = ephem.Saturn()
			self.draw_planet(context,rect,planet,local)

			planet = ephem.Uranus()
			self.draw_planet(context,rect,planet,local)

			planet = ephem.Neptune()
			self.draw_planet(context,rect,planet,local)

			planet = ephem.Pluto()
			self.draw_planet(context,rect,planet,local)

		if legend_show_flag == True:
			context.set_line_width(0.5)
			y = 25
			context.set_source_rgb(1,0,0) # legend red color
			context.move_to(0,y)
			context.text_path(coordinates_title)
			context.fill()
			context.stroke()
			y = y+15
			context.move_to(0,y)
			context.text_path(location_title)
			context.fill()
			context.stroke()
			y = y+15
			if polar_sky_flag == True:
				context.move_to(0,y)
				context.text_path(x_shift_title)
				context.fill()
				context.stroke()
				y = y+15
				context.move_to(0,y)
				context.text_path(y_shift_title)
				context.fill()
				context.stroke()
				y = y+15
			else:
				context.move_to(0,y)
				context.text_path(azimuth_title)
				context.fill()
				context.stroke()
				y = y+15
				context.move_to(0,y)
				context.text_path(altitude_title)
				context.fill()
				context.stroke()
				y = y+15
			context.move_to(0,y)
			context.text_path(field_of_view_title)
			context.fill()
			context.stroke()
			y = y+15
			context.move_to(0,y)
			context.text_path(magnitude_title)
			context.fill()
			context.stroke()
			y = y+15
			context.move_to(0,y)
			context.text_path(size_title)
			context.fill()
			context.stroke()
			y = y+15
			context.move_to(0,y)
			context.text_path(bright_title)
			context.fill()
			context.stroke()
			y = y+15
			context.move_to(0,y)
			context.text_path(grid_title)
			context.fill()
			context.stroke()
			y = y+15
			context.move_to(0,y)
			context.text_path(text_title)
			context.fill()
			context.stroke()

		if sun_compass_flag == True:
			horizon_proj_flag = horizon_proj_flag_save
			constellation_show_flag = constellation_show_flag_save
			messier_show_flag = messier_show_flag_save
			milkyway_show_flag = milkyway_show_flag_save
			controls_show_flag = controls_show_flag_save
			inverse_show_flag = inverse_show_flag_save
			planet_show_flag = planet_show_flag_save
			grid_show_flag = grid_show_flag_save
			polar_sky_flag = polar_sky_flag_save
			star_show_flag = star_show_flag_save
			asteroid_show_flag = asteroid_show_flag_save 
			comet_show_flag = comet_show_flag_save
			neo_show_flag = neo_show_flag_save
			x_shift_degrees = x_shift_degrees_save
			y_shift_degrees = y_shift_degrees_save

	def draw_planet(self,context,rect,planet,local):
		global planet_treestore
		global x_shift_degrees
		global y_shift_degrees

		radius = min(rect.width / 2, rect.height / 2)
		start_radius = 100
		mult_radius = float(radius) / start_radius

		planet.compute(local)
		planet_az = planet.az
		planet_alt = planet.alt
		planet_ra = planet.ra
		planet_dec = planet.dec
		planet_mag = planet.mag
		planet_name = planet.name

		if horizon_proj_flag == True:
			az = planet.az
			alt = planet.alt
		else:
			az = planet.ra
			alt = planet.dec

		if planet.name == "Sun":
			if sun_compass_flag == False:
				plot_size = float(planet.radius)*60 / polar_field_of_view * min(rect.width, rect.height)
				if plot_size < 10:
					plot_size = 10
			else:
				plot_size = 30
			plot_color = sun_plot_color
		elif planet.name == "Moon":
			if sun_compass_flag == False:
				plot_size = float(planet.radius)*60 / polar_field_of_view * min(rect.width, rect.height)
				if plot_size < 10:
					plot_size = 10
			else:
				plot_size = 30
			plot_color = moon_plot_color
		elif planet_name == "Mercury":
			plot_size = float(planet.radius)*60 / polar_field_of_view * min(rect.width, rect.height)
			if plot_size < 7:
				plot_size = 7
			plot_color = mercury_plot_color
		elif planet_name == "Venus":
			plot_size = float(planet.radius)*60 / polar_field_of_view * min(rect.width, rect.height)
			if plot_size < 7:
				plot_size = 7
			plot_color = venus_plot_color
		elif planet_name == "Mars":
			plot_size = float(planet.radius)*60 / polar_field_of_view * min(rect.width, rect.height)
			if plot_size < 7:
				plot_size = 7
			plot_color = mars_plot_color
		elif planet_name == "Jupiter":
			plot_size = float(planet.radius)*60 / polar_field_of_view * min(rect.width, rect.height)
			if plot_size < 7:
				plot_size = 7
			plot_color = jupiter_plot_color
		elif planet_name == "Saturn":
			plot_size = float(planet.radius)*60 / polar_field_of_view * min(rect.width, rect.height)
			if plot_size < 7:
				plot_size = 7
			plot_color = saturn_plot_color
		elif planet_name == "Uranus":
			plot_size = float(planet.radius)*60 / polar_field_of_view * min(rect.width, rect.height)
			if plot_size < 7:
				plot_size = 7
			plot_color = uranus_plot_color
		elif planet_name == "Neptune":
			plot_size = float(planet.radius)*60 / polar_field_of_view * min(rect.width, rect.height)
			if plot_size < 7:
				plot_size = 7
			plot_color = neptune_plot_color
		elif planet_name == "Pluto":
			plot_size = float(planet.radius)*60 / polar_field_of_view * min(rect.width, rect.height)
			if plot_size < 7:
				plot_size = 7
			plot_color = pluto_plot_color

		plot_size = plot_size + object_size
		if plot_size < 1:
			plot_size = 1

		piter = planet_treestore.append(None, [planet_name, planet_az, planet_alt, planet_ra, planet_dec, float(planet_az), float(planet_alt), float(planet_ra), float(planet_dec)])
		if planet_name == select_object:
			if horizon_proj_flag == True:
				search_az_ra = planet_az 
				search_alt_dec = planet_alt
			else:
				search_az_ra = pi - planet_ra
				search_alt_dec = planet_dec
			x_shift_degrees = (search_az_ra * 180/pi) # good
			y_shift_degrees = (search_alt_dec * 180/pi) - 90 # good

		if (polar_sky_flag == True):
			context.set_line_width(1.0)
			if sun_compass_flag == False:
				az = float(2*pi/4*3-az)
			else:
				az = float(az-2*pi/4)
			alt = float(alt)
			az_alt = (az,alt)

			xy = az_alt_to_polar(az_alt)
			radius = min(rect.width / 2, rect.height / 2) * (90/polar_field_of_view)
			mult_radius = float(radius) / start_radius
			mag=7-float(planet_mag)
			x = xy[0] * mult_radius + rect.width / 2
			y = xy[1] * mult_radius + rect.height / 2
			context.arc(x, y, plot_size, 0, 2 * math.pi)
			context.set_source_rgb(plot_color[0],plot_color[1],plot_color[2]) # planet circle unique color
			context.fill()
			context.stroke()
			if planet.name == "Saturn":
				ring_tilt_scale = sin(planet.earth_tilt)
				context.save() # required since altering scale
				context.translate(x,y) # dare
				context.scale(1,ring_tilt_scale) # dare
				context.arc(0, 0, plot_size * 2 * 1.1349, 0, 2 * math.pi) # dare
				context.set_source_rgb(plot_color[0],plot_color[1],plot_color[2]) # planet circle unique color
				context.stroke()
				context.restore() #required since restoring scale

			if planet_name_show_flag == True:
				context.set_source_rgba(0,1,0) # planet name green color
				font = "Sans"
				context.select_font_face(font)
				context.set_font_size(font_size)
				context.move_to(x + 5, y + 5)
				context.text_path(planet_name)
				context.fill()
				context.stroke()
		else:
			context.set_line_width(1.0)

			if horizon_proj_flag == False:
				az = (2*pi)-az
			az = az/(2*pi)*360
			alt = alt/(2*pi)*360
			y = min(rect.height,rect.width)/2 - (alt-altitude_center)/equi_field_of_view * min(rect.height,rect.width)
			x = (az-azimuth_center)/equi_field_of_view * min(rect.height,rect.width)  + min(rect.height,rect.width)/2
			context.arc(x, y, plot_size, 0, 2 * math.pi)
			context.set_source_rgb(plot_color[0],plot_color[1],plot_color[2]) # planet circle unique color
			context.fill()
			context.stroke()
			if planet.name == "Saturn":
				ring_tilt_scale = sin(planet.earth_tilt)
				context.save() # required since altering scale
				context.translate(x,y) # dare
				context.scale(1,ring_tilt_scale) # dare
				context.arc(0, 0, plot_size * 2 * 1.1349, 0, 2 * math.pi) # dare
				context.set_source_rgb(plot_color[0],plot_color[1],plot_color[2]) # planet circle unique color
				context.stroke()
				context.restore() #required since restoring scale

			if planet_name_show_flag == True:
				context.set_source_rgba(0,1,0) # planet name green color
				font = "Sans"
				context.select_font_face(font)
				context.set_font_size(font_size)
				context.move_to(x + 5, y + 5)
				context.text_path(planet_name)
				context.fill()
				context.stroke()

def plot_compass_start(widget,local):
        if hildon_flag == 1:
                window = hildon.Window()
        else:
                window = gtk.Window()

	vbox = gtk.HBox(False, 0)

        window.connect("window-state-event", on_window_state_change, window)
        window.connect("key-press-event", on_key_press_plot_sky, window, vbox, local)

	window.set_title('Planet Ephemeris: SUN COMPASS')

	window.set_size_request(800,480)

	plot_compass("clicked",window,vbox,local)

	return

def plot_compass(widget,window,vbox,local):
	global sun_compass_flag

	sun_compass_flag = True

	plot = plot_dome(window,local)
	vbox.pack_start(plot,True)
	plot.show()

	window.add(vbox)

	window.show_all()

	if window_in_fullscreen == True:
		window.fullscreen()

        window.show()

	return

def plot_sky_start(widget,local):
        if hildon_flag == 1:
                window = hildon.Window()
        else:
                window = gtk.Window()

	vbox = gtk.HBox(False, 0)

        window.connect("window-state-event", on_window_state_change, window)
        window.connect("key-press-event", on_key_press_plot_sky, window, vbox, local)

	window.set_title('Planet Ephemeris: SKY PLOT')

	window.set_size_request(800,480)

	plot_sky("clicked",window,vbox,local)

	return

class plot_sky:
    	def star_show_callback(self, widget):
		global star_show_flag 
		if widget.get_active():
			star_show_flag = True
		else:
			star_show_flag = False

    	def star_name_show_callback(self, widget):
		global star_name_show_flag 
		if widget.get_active():
			star_name_show_flag = True
		else:
			star_name_show_flag = False

    	def planet_show_callback(self, widget):
		global planet_show_flag 
		if widget.get_active():
			planet_show_flag = True
		else:
			planet_show_flag = False

    	def planet_name_show_callback(self, widget):
		global planet_name_show_flag 
		if widget.get_active():
			planet_name_show_flag = True
		else:
			planet_name_show_flag = False

    	def constellation_show_callback(self, widget):
		global constellation_show_flag 
		if widget.get_active():
			constellation_show_flag = True
		else:
			constellation_show_flag = False

    	def constellation_name_show_callback(self, widget):
		global constellation_name_show_flag 
		if widget.get_active():
			constellation_name_show_flag = True
		else:
			constellation_name_show_flag = False

    	def ngc_show_callback(self, widget):
		global ngc_show_flag 
		if widget.get_active():
			ngc_show_flag = True
		else:
			ngc_show_flag = False

    	def ngc_name_show_callback(self, widget):
		global ngc_name_show_flag 
		if widget.get_active():
			ngc_name_show_flag = True
		else:
			ngc_name_show_flag = False

    	def messier_show_callback(self, widget):
		global messier_show_flag 
		if widget.get_active():
			messier_show_flag = True
		else:
			messier_show_flag = False

    	def messier_name_show_callback(self, widget):
		global messier_name_show_flag 
		if widget.get_active():
			messier_name_show_flag = True
		else:
			messier_name_show_flag = False

    	def satellite_show_callback(self, widget):
		global satellite_show_flag 
		if widget.get_active():
			satellite_show_flag = True
		else:
			satellite_show_flag = False

    	def satellite_name_show_callback(self, widget):
		global satellite_name_show_flag 
		if widget.get_active():
			satellite_name_show_flag = True
		else:
			satellite_name_show_flag = False

    	def asteroid_show_callback(self, widget):
		global asteroid_show_flag 
		if widget.get_active():
			asteroid_show_flag = True
		else:
			asteroid_show_flag = False

    	def asteroid_name_show_callback(self, widget):
		global asteroid_name_show_flag 
		if widget.get_active():
			asteroid_name_show_flag = True
		else:
			asteroid_name_show_flag = False

    	def comet_show_callback(self, widget):
		global comet_show_flag 
		if widget.get_active():
			comet_show_flag = True
		else:
			comet_show_flag = False

    	def comet_name_show_callback(self, widget):
		global comet_name_show_flag 
		if widget.get_active():
			comet_name_show_flag = True
		else:
			comet_name_show_flag = False

    	def neo_show_callback(self, widget):
		global neo_show_flag 
		if widget.get_active():
			neo_show_flag = True
		else:
			neo_show_flag = False

    	def neo_name_show_callback(self, widget):
		global neo_name_show_flag 
		if widget.get_active():
			neo_name_show_flag = True
		else:
			neo_name_show_flag = False

    	def milkyway_show_callback(self, widget):
		global milkyway_show_flag 
		if widget.get_active():
			milkyway_show_flag = True
		else:
			milkyway_show_flag = False

    	def inverse_show_callback(self, widget):
		global inverse_show_flag 
		if widget.get_active():
			inverse_show_flag = True
		else:
			inverse_show_flag = False

    	def grid_show_callback(self, widget):
		global grid_show_flag 
		if widget.get_active():
			grid_show_flag = True
		else:
			grid_show_flag = False

    	def legend_show_callback(self, widget):
		global legend_show_flag 
		if widget.get_active():
			legend_show_flag = True
		else:
			legend_show_flag = False

    	def daylight_show_callback(self, widget):
		global daylight_show_flag 
		if widget.get_active():
			daylight_show_flag = True
		else:
			daylight_show_flag = False

    	def horizon_show_callback(self, widget):
		global horizon_show_flag 
		if widget.get_active():
			horizon_show_flag = True
		else:
			horizon_show_flag = False

    	def ecliptic_show_callback(self, widget):
		global ecliptic_show_flag 
		if widget.get_active():
			ecliptic_show_flag = True
		else:
			ecliptic_show_flag = False

    	def galactic_show_callback(self, widget):
		global galactic_show_flag 
		if widget.get_active():
			galactic_show_flag = True
		else:
			galactic_show_flag = False

    	def equirectangular_projection_callback(self, widget):
		global polar_sky_flag 
		if widget.get_active():
			polar_sky_flag = False
		else:
			polar_sky_flag = True

    	def polar_projection_callback(self, widget):
		global polar_sky_flag 
		if widget.get_active():
			polar_sky_flag = True
		else:
			polar_sky_flag = False

    	def equatorial_projection_callback(self, widget):
		global horizon_proj_flag 
		if widget.get_active():
			horizon_proj_flag = False
		else:
			horizon_proj_flag = True

    	def horizon_projection_callback(self, widget):
		global horizon_proj_flag 
		if widget.get_active():
			horizon_proj_flag = True
		else:
			horizon_proj_flag = False

        def planet_search_callback(self, widget):
		global object_search_flag
		global star_treestore

		if planet_show_flag == True:
			if len(planet_treestore) == 0:
				print "ERROR: No Planet objects have been plotted yet!"
				object_search_flag = False
			else:
				object_search_flag = True
				object_search("Planet")

        def star_search_callback(self, widget):
		global object_search_flag
		global star_treestore

		if star_show_flag == True:
			if len(star_treestore) == 0:
				print "ERROR: No Star objects have been plotted yet!"
				object_search_flag = False
			else:
				object_search_flag = True
				object_search("Star")

        def messier_search_callback(self, widget):
		global object_search_flag
		global messier_treestore

		if messier_show_flag == True:
			if len(messier_treestore) == 0:
				print "ERROR: No Messier objects have been plotted yet!"
				object_search_flag = False
			else:
				object_search_flag = True
				object_search("Messier")

        def ngc_search_callback(self, widget):
		global object_search_flag
		global ngc_treestore

		if ngc_show_flag == True:
			if len(ngc_treestore) == 0:
				print "ERROR: No NGC objects have been plotted yet!"
				object_search_flag = False
			else:
				object_search_flag = True
				object_search("NGC")

        def satellite_search_callback(self, widget):
		global object_search_flag
		global satellite_treestore

		if satellite_show_flag == True:
			if len(satellite_treestore) == 0:
				print "ERROR: No Satellite objects have been plotted yet!"
				object_search_flag = False
			else:
				object_search_flag = True
				object_search("Satellite")

        def asteroid_search_callback(self, widget):
		global object_search_flag
		global asteroid_treestore

		if asteroid_show_flag == True:
			if len(asteroid_treestore) == 0:
				print "ERROR: No asteroid objects have been plotted yet!"
				object_search_flag = False
			else:
				object_search_flag = True
				object_search("Asteroid")

        def comet_search_callback(self, widget):
		global object_search_flag
		global comet_treestore

		if comet_show_flag == True:
			if len(comet_treestore) == 0:
				print "ERROR: No comet objects have been plotted yet!"
				object_search_flag = False
			else:
				object_search_flag = True
				object_search("Comet")

        def neo_search_callback(self, widget):
		global object_search_flag
		global neo_treestore

		if neo_show_flag == True:
			if len(neo_treestore) == 0:
				print "ERROR: No NEO objects have been plotted yet!"
				object_search_flag = False
			else:
				object_search_flag = True
				object_search("NEO")

        def constellation_search_callback(self, widget):
		global object_search_flag
		global constellation_treestore

		if constellation_show_flag == True:
			if len(constellation_treestore) == 0:
				print "ERROR: No Constellation objects have been plotted yet!"
				object_search_flag = False
			else:
				object_search_flag = True
				object_search("Constellation")

	def __init__(self,widget,window,vbox,local):
		global sun_compass_flag

		table = gtk.Table(3,2,False)
		table.set_col_spacings(0)
		table.set_row_spacings(1)

		table1 = gtk.Table(1,1,False)
		table1.set_col_spacings(1)
		table1.set_row_spacings(1)

		table2 = gtk.Table(1,1,False)
		table2.set_col_spacings(1)
		table2.set_row_spacings(1)

		table3 = gtk.Table(1,1,False)
		table3.set_col_spacings(1)
		table3.set_row_spacings(1)

		table4 = gtk.Table(1,1,False)
		table4.set_col_spacings(1)
		table4.set_row_spacings(1)

		sun_compass_flag = False

		plot = plot_dome(window,local)
		vbox.pack_start(plot,True)
		plot.show()

		if controls_show_flag == True:

			if on_off_controls_show_flag == True:
				column = 0
				row = 0

				if verbose_show_flag == True:
					button = gtk.Button("Column Remove")
				else:
					button = gtk.Button("Remove")
				on_off_controls_show_flag2 = 0	
				button.connect("clicked",toggle_on_off_controls,window,vbox,local,on_off_controls_show_flag2)
				table2.attach(button,column,column+5,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Stars")
				else:
					button = gtk.ToggleButton("St")
				button.set_active(star_show_flag)
				button.connect("toggled", self.star_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Names")
				else:
					button = gtk.ToggleButton("StN")
				button.set_active(star_name_show_flag)
				button.connect("toggled", self.star_name_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.Button("Search")
				else:
					button = gtk.Button("StS")
				button.connect("clicked", self.star_search_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Planets")
				else:
					button = gtk.ToggleButton("Pl")
				button.set_active(planet_show_flag)
				button.connect("toggled", self.planet_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Names")
				else:
					button = gtk.ToggleButton("PlN")
				button.set_active(planet_name_show_flag)
				button.connect("toggled", self.planet_name_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.Button("Search")
				else:
					button = gtk.Button("PlS")
				button.connect("clicked", self.planet_search_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Constellations")
				else:
					button = gtk.ToggleButton("Co")
				button.set_active(constellation_show_flag)
				button.connect("toggled", self.constellation_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Names")
				else:
					button = gtk.ToggleButton("CoN")
				button.set_active(constellation_name_show_flag)
				button.connect("toggled", self.constellation_name_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.Button("Search")
				else:
					button = gtk.Button("CoS")
				button.connect("clicked", self.constellation_search_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("NGC")
				else:
					button = gtk.ToggleButton("Ng")
				button.set_active(ngc_show_flag)
				button.connect("toggled", self.ngc_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Names")
				else:
					button = gtk.ToggleButton("NgN")
				button.set_active(ngc_name_show_flag)
				button.connect("toggled", self.ngc_name_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.Button("Search")
				else:
					button = gtk.Button("NgS")
				button.connect("clicked", self.ngc_search_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Messier")
				else:
					button = gtk.ToggleButton("Me")
				button.set_active(messier_show_flag)
				button.connect("toggled", self.messier_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Names")
				else:
					button = gtk.ToggleButton("MeN")
				button.set_active(messier_name_show_flag)
				button.connect("toggled", self.messier_name_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.Button("Search")
				else:
					button = gtk.Button("MeS")
				button.connect("clicked", self.messier_search_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Satellites")
				else:
					button = gtk.ToggleButton("Sa")
				button.set_active(satellite_show_flag)
				button.connect("toggled", self.satellite_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Names")
				else:
					button = gtk.ToggleButton("SaN")
				button.set_active(satellite_name_show_flag)
				button.connect("toggled", self.satellite_name_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.Button("Search")
				else:
					button = gtk.Button("SaS")
				button.connect("clicked", self.satellite_search_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Asteroids")
				else:
					button = gtk.ToggleButton("As")
				button.set_active(asteroid_show_flag)
				button.connect("toggled", self.asteroid_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Names")
				else:
					button = gtk.ToggleButton("AsN")
				button.set_active(asteroid_name_show_flag)
				button.connect("toggled", self.asteroid_name_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.Button("Search")
				else:
					button = gtk.Button("AsS")
				button.connect("clicked", self.asteroid_search_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Comets")
				else:
					button = gtk.ToggleButton("Cm")
				button.set_active(comet_show_flag)
				button.connect("toggled", self.comet_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Names")
				else:
					button = gtk.ToggleButton("CmN")
				button.set_active(comet_name_show_flag)
				button.connect("toggled", self.comet_name_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.Button("Search")
				else:
					button = gtk.Button("CoS")
				button.connect("clicked", self.comet_search_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("NEO")
				else:
					button = gtk.ToggleButton("Ne")
				button.set_active(neo_show_flag)
				button.connect("toggled", self.neo_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Names")
				else:
					button = gtk.ToggleButton("NeN")
				button.set_active(neo_name_show_flag)
				button.connect("toggled", self.neo_name_show_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.Button("Search")
				else:
					button = gtk.Button("NeS")
				button.connect("clicked", self.neo_search_callback)
				table2.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				row = 0
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Legend Text")
				else:
					button = gtk.ToggleButton("LgT")
				button.set_active(legend_show_flag)
				button.connect("toggled", self.legend_show_callback)
				table4.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column + 1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Milkyway")
				else:
					button = gtk.ToggleButton("MlW")
				button.set_active(milkyway_show_flag)
				button.connect("toggled", self.milkyway_show_callback)
				table4.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Grid Lines")
				else:
					button = gtk.ToggleButton("GrL")
				button.set_active(grid_show_flag)
				button.connect("toggled", self.grid_show_callback)
				table4.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column + 1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Horizon Line")
				else:
					button = gtk.ToggleButton("HzL")
				button.set_active(horizon_show_flag)
				button.connect("toggled", self.horizon_show_callback)
				table4.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Ecliptic Line")
				else:
					button = gtk.ToggleButton("EcL")
				button.set_active(ecliptic_show_flag)
				button.connect("toggled", self.ecliptic_show_callback)
				table4.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column + 1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Galactic Line")
				else:
					button = gtk.ToggleButton("GaL")
				button.set_active(galactic_show_flag)
				button.connect("toggled", self.galactic_show_callback)
				table4.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				row = row + 1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Daylight Mode")
				else:
					button = gtk.ToggleButton("DyM")
				button.set_active(daylight_show_flag)
				button.connect("toggled", self.daylight_show_callback)
				table4.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = column + 1
				if verbose_show_flag == True:
					button = gtk.ToggleButton("Inverse Mode")
				else:
					button = gtk.ToggleButton("IvM")
				button.set_active(inverse_show_flag)
				button.connect("toggled", self.inverse_show_callback)
				table4.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"red","green","black","light blue","black","red")

				column = 0
				table.attach(table2,column,column+1,0,10)
				table2.show()

				column = 0
				table.attach(table4,column,column+1,10,16)
				table4.show()

			if date_time_controls_show_flag == True:
				column = 0
				row = 0

				if verbose_show_flag == True:
					button = gtk.Button("Column Remove")
				else:
					button = gtk.Button("Remove")
				date_time_controls_show_flag2 = 0	
				button.connect("clicked",toggle_sky_plot_date_time_controls,window,vbox,local,date_time_controls_show_flag2)
				table1.attach(button,column,column+3,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Year")
				else:
					button = gtk.Button("Yr")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -5
				button = gtk.Button("-")
				button.connect("clicked",change_date_sky_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 5
				button = gtk.Button("+")
				button.connect("clicked",change_date_sky_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Month")
				else:
					button = gtk.Button("Mo")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -4
				button = gtk.Button("-")
				button.connect("clicked",change_date_sky_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 4
				button = gtk.Button("+")
				button.connect("clicked",change_date_sky_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Day")
				else:
					button = gtk.Button("Dy")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -3
				button = gtk.Button("-")
				button.connect("clicked",change_date_sky_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 3
				button = gtk.Button("+")
				button.connect("clicked",change_date_sky_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Hour")
				else:
					button = gtk.Button("Hr")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -2
				button = gtk.Button("-")
				button.connect("clicked",change_date_sky_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 2
				button = gtk.Button("+")
				button.connect("clicked",change_date_sky_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Minute")
				else:
					button = gtk.Button("Mi")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				date_time_change_flag = -1
				button = gtk.Button("-")
				button.connect("clicked",change_date_sky_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = column+1
				date_time_change_flag = 1
				button = gtk.Button("+")
				button.connect("clicked",change_date_sky_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				date_time_change_flag = 0
				if verbose_show_flag == True:
					button = gtk.Button('Reset Date/Time')
				else:
					button = gtk.Button('Reset')
				button.connect("clicked",change_date_sky_plot,window,vbox,local,date_time_change_flag)
				table1.attach(button,column,column+3,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","red","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Size")
				else:
					button = gtk.Button("Sz")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				button = gtk.Button("-")
				object_size_change_flag = -1
				button.connect("clicked",set_object_size,object_size_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = column+1
				button = gtk.Button("+")
				object_size_change_flag = 1
				button.connect("clicked",set_object_size,object_size_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Brightness")
				else:
					button = gtk.Button("Br")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				button = gtk.Button("-")
				object_bright_change_flag = -1
				button.connect("clicked",set_object_bright,object_bright_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = column+1
				button = gtk.Button("+")
				object_bright_change_flag = 1
				button.connect("clicked",set_object_bright,object_bright_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Field of View")
				else:
					button = gtk.Button("Fv")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				button = gtk.Button("-")
				field_of_view_size_change_flag = -1
				button.connect("clicked",set_field_of_view,field_of_view_size_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = column+1
				button = gtk.Button("+")
				field_of_view_size_change_flag = 1
				button.connect("clicked",set_field_of_view,field_of_view_size_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Grid Space")
				else:
					button = gtk.Button("Gs")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				button = gtk.Button("-")
				grid_space_change_flag = -1
				button.connect("clicked",set_grid_space,grid_space_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = column+1
				button = gtk.Button("+")
				grid_space_change_flag = 1
				button.connect("clicked",set_grid_space,grid_space_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Magnitude")
				else:
					button = gtk.Button("Mg")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				button = gtk.Button("-")
				magnitude_change_flag = -1
				button.connect("clicked",set_magnitude,magnitude_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = column+1
				button = gtk.Button("+")
				magnitude_change_flag = 1
				button.connect("clicked",set_magnitude,magnitude_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = 0
				row = row+1
				if verbose_show_flag == True:
					button = gtk.Button("Text Size")
				else:
					button = gtk.Button("Ts")
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				button = gtk.Button("-")
				text_size_change_flag = -1
				button.connect("clicked",set_font_size,text_size_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = column+1
				button = gtk.Button("+")
				text_size_change_flag = 1
				button.connect("clicked",set_font_size,text_size_change_flag)
				table1.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","light blue","green","red")

				column = 0 
				row = 0
				if verbose_show_flag == True:
					button = gtk.Button("Projection")
				else:
					button = gtk.Button("Pr")
				table3.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = column+1
				if verbose_show_flag == True:
					button = gtk.Button("Coordinates")
				else:
					button = gtk.Button("Co")
				table3.attach(button,column,column+1,row,row+1)
				button.show()                            
				set_button_color("clicked",button,"black","black","black","green","green","green")

				column = 0
				row = row+1

				if verbose_show_flag == True:
        				button = gtk.RadioButton(None, "Rectangular")
				else:
					button = gtk.RadioButton(None, "R")
				button.set_active(not polar_sky_flag)
        			button.connect("toggled", self.equirectangular_projection_callback)
				button.show()                            
				eb = gtk.EventBox()
				eb.add(button)
				eb.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse("gray"))
				label = button.child
				label.modify_font(pango.FontDescription("sans 12"))
				table3.attach(eb,column,column+1,row,row+1)

				row = row+1

				if verbose_show_flag == True:
					button = gtk.RadioButton(button,"Polar")
				else:
					button = gtk.RadioButton(button,"P")
				button.set_active(polar_sky_flag)
        			button.connect("toggled", self.polar_projection_callback)
				button.show()                            
				eb = gtk.EventBox()
				eb.add(button)
				eb.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse("gray"))
				label = button.child
				label.modify_font(pango.FontDescription("sans 12"))
				table3.attach(eb,column,column+1,row,row+1)

				column = column+1
				row = row-1

				if verbose_show_flag == True:
					button = gtk.RadioButton(None, "Equatorial")
				else:
					button = gtk.RadioButton(None, "E")
				button.set_active(not horizon_proj_flag)
        			button.connect("toggled", self.equatorial_projection_callback)
				button.show()                            
				eb = gtk.EventBox()
				eb.add(button)
				eb.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse("gray"))
				label = button.child
				label.modify_font(pango.FontDescription("sans 12"))
				table3.attach(eb,column,column+1,row,row+1)

				row = row+1

				if verbose_show_flag == True:
					button = gtk.RadioButton(button, "Horizon")
				else:
					button = gtk.RadioButton(button, "H")
				button.set_active(horizon_proj_flag)
        			button.connect("toggled", self.horizon_projection_callback)
				button.show()                            
				eb = gtk.EventBox()
				eb.add(button)
				eb.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse("gray"))
				label = button.child
				label.modify_font(pango.FontDescription("sans 12"))
				table3.attach(eb,column,column+1,row,row+1)

				column = 1
				table.attach(table1,column,column+1,0,13)
				table1.show()

				column = 1
				table.attach(table3,column,column+1,13,16)
				table3.show()

		window.add(vbox)
		vbox.pack_start(table,False)

		window.show_all()

		if window_in_fullscreen == True:
			window.fullscreen()

		window.show()

		return

def set_column_style(column):
	label = gtk.Label(column.get_title())
	label.modify_font(pango.FontDescription("sans " + str(font_size)))
	label.show()

	fg_normal_color = "yellow"
	bg_normal_color = "black"
	
	label.set_markup('<span background="' + bg_normal_color + '" foreground="' + fg_normal_color + '">' + str(label.get_text()) + '</span>')
	label.set_use_markup(True)

	column.set_resizable(True)
	column.set_widget(label)

	return

def set_cell_style(cell):
	font = pango.FontDescription('sans ' + str(font_size))
	cell.set_property('font-desc', font)
	cell.set_property("cell-background", 'black')
	cell.set_property("foreground", 'green')
	cell.set_property("background", 'black')

def moon_feature_search(object_type):
	global moon_feature_treestore

	# create the ini TreeView using specific object treestore
	object_treeview = gtk.TreeView(moon_feature_treestore)

	object_treeview.set_headers_visible(True)              # needed for Maemo
	ini_selection = object_treeview.get_selection()
	ini_selection.set_mode(gtk.SELECTION_SINGLE)
	object_treeview.set_property("show-expanders", True)   # needed for Maemo
	object_treeview.set_property("enable-grid-lines", True)
	object_treeview.set_property("enable-tree-lines", True)

        color_black = gtk.gdk.color_parse('black')
	object_treeview.modify_base(gtk.STATE_NORMAL, color_black)

	# create the TreeViewColumns to display the data
	object_name = gtk.TreeViewColumn(object_type + ' Name')
	set_column_style(object_name)
	object_longitude_string = gtk.TreeViewColumn(object_type + ' Longitude')
	set_column_style(object_longitude_string)
	object_latitude_string = gtk.TreeViewColumn(object_type + ' Latitude')
	set_column_style(object_latitude_string)
	object_size_string = gtk.TreeViewColumn(object_type + ' Size')
	set_column_style(object_size_string)

	# add columns to ini treeview
	object_treeview.append_column(object_name)
	object_treeview.append_column(object_longitude_string)
	object_treeview.append_column(object_latitude_string)
	object_treeview.append_column(object_size_string)

	# create CellRendererText to render the data
	object_name_cell = gtk.CellRendererText()
	set_cell_style(object_name_cell)
	object_longitude_string_cell = gtk.CellRendererText()
	set_cell_style(object_longitude_string_cell)
	object_latitude_string_cell = gtk.CellRendererText()
	set_cell_style(object_latitude_string_cell)
	object_size_string_cell = gtk.CellRendererText()
	set_cell_style(object_size_string_cell)

	object_treeview.connect('cursor-changed', select_moon_feature_cb)

	# add the ini cells to the ini columns and allow them to expand
	object_name.pack_start(object_name_cell, True)
	object_longitude_string.pack_start(object_longitude_string_cell, True)
	object_latitude_string.pack_start(object_latitude_string_cell, True)
	object_size_string.pack_start(object_size_string_cell, True)

	# set the ini cells "text" attribute to matching column - retrieve text from that column in ini treestore
	column = 0
	object_name.add_attribute(object_name_cell, 'text', column)
	object_name.set_sort_column_id(column)
	column = column + 1
	object_longitude_string.add_attribute(object_longitude_string_cell, 'text', column)
	object_longitude_string.set_sort_column_id(column)
	column = column + 1
	object_latitude_string.add_attribute(object_latitude_string_cell, 'text', column)
	object_latitude_string.set_sort_column_id(column)
	column = column + 1
	object_size_string.add_attribute(object_size_string_cell, 'text', column)
	object_size_string.set_sort_column_id(column)

	# Create a new window
	if hildon_flag == 1:
		window = hildon.Window()
	else:
		window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title(object_type + " Search")
	window.set_size_request(800, 480)

	# make ui layout
	vbox = gtk.VBox()

	# add scrolled window
	scrolledwindow = gtk.ScrolledWindow()
	bbox = gtk.HButtonBox()
	vbox.pack_start(scrolledwindow)
	vbox.pack_start(bbox, False)

	scrolledwindow.add(object_treeview)

	if window_in_fullscreen == True:
		window.fullscreen()

	window.add(vbox)
	window.show_all()

	return

def select_moon_feature_cb(object_treeview):
	global object_treestore
	global object_iter
	global select_object

	path, column = object_treeview.get_cursor()
	if not column:
		return
	colname = column.get_title()
	object_treestore, object_iter = object_treeview.get_selection().get_selected()

	select_object = object_treestore.get_value(object_iter, 0)

	return

def object_search(object_type):
	global planet_treestore
	global star_treestore
	global messier_treestore
	global ngc_treestore
	global satellite_treestore
	global asteroid_treestore
	global comet_treestore
	global neo_treestore
	global constellation_treestore

	# create the ini TreeView using specific object treestore
	if object_type == "Planet":
		object_treeview = gtk.TreeView(planet_treestore)
	if object_type == "Star":
		object_treeview = gtk.TreeView(star_treestore)
	if object_type == "Messier":
		object_treeview = gtk.TreeView(messier_treestore)
	if object_type == "NGC":
		object_treeview = gtk.TreeView(ngc_treestore)
	if object_type == "Satellite":
		object_treeview = gtk.TreeView(satellite_treestore)
	if object_type == "Asteroid":
		object_treeview = gtk.TreeView(asteroid_treestore)
	if object_type == "Comet":
		object_treeview = gtk.TreeView(comet_treestore)
	if object_type == "NEO":
		object_treeview = gtk.TreeView(neo_treestore)
	if object_type == "Constellation":
		object_treeview = gtk.TreeView(constellation_treestore)

	object_treeview.set_headers_visible(True)              # needed for Maemo
	ini_selection = object_treeview.get_selection()
	ini_selection.set_mode(gtk.SELECTION_SINGLE)
	object_treeview.set_property("show-expanders", True)   # needed for Maemo
	object_treeview.set_property("enable-grid-lines", True)
	object_treeview.set_property("enable-tree-lines", True)

        color_black = gtk.gdk.color_parse('black')
	object_treeview.modify_base(gtk.STATE_NORMAL, color_black)

	# create the TreeViewColumns to display the data
	object_name = gtk.TreeViewColumn(object_type + ' Name')
	set_column_style(object_name)
	object_azimuth_string = gtk.TreeViewColumn(object_type + ' Azimuth')
	set_column_style(object_azimuth_string)
	object_altitude_string = gtk.TreeViewColumn(object_type + ' Altitude')
	set_column_style(object_altitude_string)
	object_ra_string = gtk.TreeViewColumn(object_type + ' RA')
	set_column_style(object_ra_string)
	object_dec_string = gtk.TreeViewColumn(object_type + ' Dec')
	set_column_style(object_dec_string)
	object_azimuth_value = gtk.TreeViewColumn(object_type + ' Azimuth')
	set_column_style(object_azimuth_value)
	object_altitude_value = gtk.TreeViewColumn(object_type + ' Altitude')
	set_column_style(object_altitude_value)
	object_ra_value = gtk.TreeViewColumn(object_type + ' RA')
	set_column_style(object_ra_value)
	object_dec_value = gtk.TreeViewColumn(object_type + ' Dec')
	set_column_style(object_dec_value)

	# set some treeview columns invisible
	object_azimuth_value.set_visible(False)
	object_altitude_value.set_visible(False)
	object_ra_value.set_visible(False)
	object_dec_value.set_visible(False)

	# add columns to ini treeview
	object_treeview.append_column(object_name)
	object_treeview.append_column(object_azimuth_string)
	object_treeview.append_column(object_altitude_string)
	object_treeview.append_column(object_ra_string)
	object_treeview.append_column(object_dec_string)
	object_treeview.append_column(object_azimuth_value)
	object_treeview.append_column(object_altitude_value)
	object_treeview.append_column(object_ra_value)
	object_treeview.append_column(object_dec_value)

	# create CellRendererText to render the data
	object_name_cell = gtk.CellRendererText()
	set_cell_style(object_name_cell)
	object_azimuth_string_cell = gtk.CellRendererText()
	set_cell_style(object_azimuth_string_cell)
	object_altitude_string_cell = gtk.CellRendererText()
	set_cell_style(object_altitude_string_cell)
	object_ra_string_cell = gtk.CellRendererText()
	set_cell_style(object_ra_string_cell)
	object_dec_string_cell = gtk.CellRendererText()
	set_cell_style(object_dec_string_cell)
	object_azimuth_value_cell = gtk.CellRendererText()
	set_cell_style(object_azimuth_value_cell)
	object_altitude_value_cell = gtk.CellRendererText()
	set_cell_style(object_altitude_value_cell)
	object_ra_value_cell = gtk.CellRendererText()
	set_cell_style(object_ra_value_cell)
	object_dec_value_cell = gtk.CellRendererText()
	set_cell_style(object_dec_value_cell)

	object_treeview.connect('cursor-changed', select_generic_cb)

	# add the ini cells to the ini columns and allow them to expand
	object_name.pack_start(object_name_cell, True)
	object_azimuth_string.pack_start(object_azimuth_string_cell, True)
	object_altitude_string.pack_start(object_altitude_string_cell, True)
	object_ra_string.pack_start(object_ra_string_cell, True)
	object_dec_string.pack_start(object_dec_string_cell, True)
	object_azimuth_value.pack_start(object_azimuth_value_cell, True)
	object_altitude_value.pack_start(object_altitude_value_cell, True)
	object_ra_value.pack_start(object_ra_value_cell, True)
	object_dec_value.pack_start(object_dec_value_cell, True)

	# set the ini cells "text" attribute to matching column - retrieve text from that column in ini treestore
	column = 0
	object_name.add_attribute(object_name_cell, 'text', column)
	object_name.set_sort_column_id(column)
	column = column + 1
	object_azimuth_string.add_attribute(object_azimuth_string_cell, 'text', column)
	object_azimuth_string.set_sort_column_id(column)
	column = column + 1
	object_altitude_string.add_attribute(object_altitude_string_cell, 'text', column)
	object_altitude_string.set_sort_column_id(column)
	column = column + 1
	object_ra_string.add_attribute(object_ra_string_cell, 'text', column)
	object_ra_string.set_sort_column_id(column)
	column = column + 1
	object_dec_string.add_attribute(object_dec_string_cell, 'text', column)
	object_dec_string.set_sort_column_id(column)
	column = column + 1
	object_azimuth_value.add_attribute(object_azimuth_value_cell, 'text', column)
	object_azimuth_value.set_sort_column_id(column)
	column = column + 1
	object_altitude_value.add_attribute(object_altitude_value_cell, 'text', column)
	object_altitude_value.set_sort_column_id(column)
	column = column + 1
	object_ra_value.add_attribute(object_ra_value_cell, 'text', column)
	object_ra_value.set_sort_column_id(column)
	column = column + 1
	object_dec_value.add_attribute(object_dec_value_cell, 'text', column)
	object_dec_value.set_sort_column_id(column)

	# Create a new window
	if hildon_flag == 1:
		window = hildon.Window()
	else:
		window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title(object_type + " Search")
	window.set_size_request(800, 480)

	# make ui layout
	vbox = gtk.VBox()

	# add scrolled window
	scrolledwindow = gtk.ScrolledWindow()
	bbox = gtk.HButtonBox()
	vbox.pack_start(scrolledwindow)
	vbox.pack_start(bbox, False)

	scrolledwindow.add(object_treeview)

	if window_in_fullscreen == True:
		window.fullscreen()

	window.add(vbox)
	window.show_all()

	return

def select_generic_cb(object_treeview):
	global search_az_ra
	global search_alt_dec
	global horizon_proj_flag
	global object_treestore
	global object_iter
	global select_object

	path, column = object_treeview.get_cursor()
	if not column:
		return
	colname = column.get_title()
	object_treestore, object_iter = object_treeview.get_selection().get_selected()

	select_object = object_treestore.get_value(object_iter, 0)

	if horizon_proj_flag == True:
		search_az_ra = object_treestore.get_value(object_iter, 5)
		search_alt_dec = object_treestore.get_value(object_iter, 6)
	else:
		search_az_ra = pi - object_treestore.get_value(object_iter, 7)
		search_alt_dec = object_treestore.get_value(object_iter, 8)

	return

def edit_messier_url(widget):
	global messier_image_treestore

	edit_url(messier_image_treestore)

	return

def edit_misc_url(widget):
	global misc_image_treestore

	edit_url(misc_image_treestore)

	return

def edit_url(url_treestore):
	# create the url TreeView using url treestore
	url_treeview = gtk.TreeView(url_treestore)
	url_treeview.set_headers_visible(True)              # needed for Maemo
	url_selection = url_treeview.get_selection()
	url_selection.set_mode(gtk.SELECTION_SINGLE)
	url_treeview.set_property("show-expanders", True)   # needed for Maemo
	url_treeview.set_property("enable-grid-lines", True)
	url_treeview.set_property("enable-tree-lines", True)

        color_black = gtk.gdk.color_parse('black')
	url_treeview.modify_base(gtk.STATE_NORMAL, color_black)

	# create the url TreeViewColumns to display the data
	url_object_name = gtk.TreeViewColumn('Object Name')
	set_column_style(url_object_name)
	url_object_url = gtk.TreeViewColumn('Object URL')
	set_column_style(url_object_url)

	# add columns to url treeview
	url_treeview.append_column(url_object_name)
	url_treeview.append_column(url_object_url)

	# create ini CellRendererText to render the data
	url_object_name_cell = gtk.CellRendererText()
	set_cell_style(url_object_name_cell)
	url_object_name_cell.connect('edited', url_name_edited_cb, url_treestore)
	url_object_name_cell.set_property('editable', True)
	url_object_url_cell = gtk.CellRendererText()
	set_cell_style(url_object_url_cell)
	url_object_url_cell.connect('edited', url_url_edited_cb, url_treestore)
	url_object_url_cell.set_property('editable', True)

	# add the url cells to the url columns and allow them to expand
	url_object_name.pack_start(url_object_name_cell, True)
	url_object_url.pack_start(url_object_url_cell, True)

	# set the ini cells "text" attribute to matching column - retrieve text from that column in url treestore
	column = 0
	url_object_name.add_attribute(url_object_name_cell, 'text', column)
	column = column + 1
	url_object_url.add_attribute(url_object_url_cell, 'text', column)

	# Create a new window
	if hildon_flag == 1:
		window = hildon.Window()
	else:
		window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title("Edit Misc URLs")
	window.set_size_request(800, 480)

	# make ui layout
	vbox = gtk.VBox()

	# add scrolled window
	scrolledwindow = gtk.ScrolledWindow()
	bbox = gtk.HButtonBox()
	vbox.pack_start(scrolledwindow)
	vbox.pack_start(bbox, False)

	scrolledwindow.add(url_treeview)

	if window_in_fullscreen == True:
		window.fullscreen()

	window.add(vbox)
	window.show_all()

	return

def url_name_edited_cb(cell, path, new_name, url_treestore):
	"""
	Called when a text cell is edited.  It puts the new text
	in the model so that it is displayed properly.
	"""
	object_name = url_treestore[path][0]
	object_url = url_treestore[path][1]

	if len(new_name.split()) > 1:
		print "ERROR: Object name contains blanks!"
		return

	url_treestore[path][0] = new_name

	return

def url_url_edited_cb(cell, path, new_url, url_treestore):
	"""
	Called when a text cell is edited.  It puts the new text
	in the model so that it is displayed properly.
	"""
	object_name = url_treestore[path][0]
	object_url = url_treestore[path][1]

	if not re.search("(http|https)://([a-zA-Z0-9.]|%[0-9A-Za-z]|/|:[0-9]?)*", new_url):
		print "ERROR: Invalid URL entered!"
		return

	url_treestore[path][1] = new_url

	return

def save_messier_url(widget):
	global messir_treestore

	file_name = "data/messier_url.ini"
	save_url(messier_image_treestore, file_name)

	return

def save_misc_url(widget):
	global misc_image_treestore

	file_name = "data/misc_url.ini"
	save_url(misc_image_treestore, file_name)

	return

def save_url(url_treestore, file_name):
	file = open(file_name, "w")

	for row in url_treestore:
		file.write(row[0] + " " + row[1] + "\n")

	file.close()

	return

def select_ini_cb(ini_treeview):
	global observer

	path, column = ini_treeview.get_cursor()
	if not column:
		return
	colname = column.get_title()
	ini_treestore, iter = ini_treeview.get_selection().get_selected()

	ini_name = ini_treestore.get_value(iter, 0)
	ini_value = ini_treestore.get_value(iter, 1)

	if ini_name == "latitude:":
		print "changing!"
		ini_treestore.set_value(iter, 1, observer.lat*180/pi)

	if ini_name == "longitude:":
		print "changing!"
		ini_treestore.set_value(iter, 1, observer.long*180/pi)

	if ini_name == "pressure:":
		print "changing!"
		ini_treestore.set_value(iter, 1, observer.pressure)

	if ini_name == "temperature:":
		print "changing!"
		ini_treestore.set_value(iter, 1, observer.temp)

	if ini_name == "elevation:":
		print "changing!"
		ini_treestore.set_value(iter, 1, observer.elevation)

	return

def edit_ini(widget):
	global ini_treestore

	# create the ini TreeView using ini treestore
	ini_treeview = gtk.TreeView(ini_treestore)
	ini_treeview.set_headers_visible(True)              # needed for Maemo
	ini_selection = ini_treeview.get_selection()
	ini_selection.set_mode(gtk.SELECTION_SINGLE)
	ini_treeview.set_property("show-expanders", True)   # needed for Maemo
	ini_treeview.set_property("enable-grid-lines", True)
	ini_treeview.set_property("enable-tree-lines", True)

        color_black = gtk.gdk.color_parse('black')
	ini_treeview.modify_base(gtk.STATE_NORMAL, color_black)

	# create the ini TreeViewColumns to display the data
	ini_setting_name = gtk.TreeViewColumn('Setting Name')
	set_column_style(ini_setting_name)
	ini_setting_value = gtk.TreeViewColumn('Setting Value')
	set_column_style(ini_setting_value)

	# add columns to ini treeview
	ini_treeview.append_column(ini_setting_name)
	ini_treeview.append_column(ini_setting_value)

	# create ini CellRendererText to render the data
	ini_setting_name_cell = gtk.CellRendererText()
	set_cell_style(ini_setting_name_cell)
	ini_setting_value_cell = gtk.CellRendererText()
	set_cell_style(ini_setting_value_cell)
	ini_setting_value_cell.connect('edited', ini_value_edited_cb, ini_treestore)
	ini_setting_value_cell.set_property('editable', True)
	ini_treeview.connect('cursor-changed', select_ini_cb)

	# add the ini cells to the ini columns and allow them to expand
	ini_setting_name.pack_start(ini_setting_name_cell, True)
	ini_setting_value.pack_start(ini_setting_value_cell, True)

	# set the ini cells "text" attribute to matching column - retrieve text from that column in ini treestore
	column = 0
	ini_setting_name.add_attribute(ini_setting_name_cell, 'text', column)
	column = column + 1
	ini_setting_value.add_attribute(ini_setting_value_cell, 'text', column)

	# Create a new window
	if hildon_flag == 1:
		window = hildon.Window()
	else:
		window = gtk.Window()

	window.connect("window-state-event", on_window_state_change, window)
	window.connect("key-press-event", on_key_press_basic, window)

	window.set_title("Edit Ini Paramaters")
	window.set_size_request(800, 480)

	# make ui layout
	vbox = gtk.VBox()

	# add scrolled window
	scrolledwindow = gtk.ScrolledWindow()
	bbox = gtk.HButtonBox()
	vbox.pack_start(scrolledwindow)
	vbox.pack_start(bbox, False)

	scrolledwindow.add(ini_treeview)

	if window_in_fullscreen == True:
		window.fullscreen()

	window.add(vbox)
	window.show_all()

	return

def save_ini(widget):
	global ini_treestore

	file_name = "mephemeris.ini"
	file = open(file_name, "w")

	for row in ini_treestore:
		file.write(row[0] + " " + row[1] + "\n")
		for subrow in row.iterchildren():
			file.write(subrow[0] + " " + subrow[1] + "\n")

	file.close()

	return

def ini_value_edited_cb(cell, path, new_value, ini_treestore):
	"""
	Called when a text cell is edited.  It puts the new text
	in the model so that it is displayed properly.
	"""
	ini_name = ini_treestore[path][0]
	old_value = ini_treestore[path][1]

	if ini_name == "latitude:":
		try:
		    	float(new_value) if '.' in new_value else int(new_value)
		except ValueError:
			print "ERROR: Invalid decimal number entered!"
		    	return

	if ini_name == "longitude:":
		try:
		    	float(new_value) if '.' in new_value else int(new_value)
		except ValueError:
			print "ERROR: Invalid decimal number entered!"
		    	return

	if ini_name == "temperature:":
		try:
		    	float(new_value) if '.' in new_value else int(new_value)
		except ValueError:
			print "ERROR: Invalid decimal number entered!"
		    	return

	if ini_name == "pressure:":
		try:
		    	float(new_value) if '.' in new_value else int(new_value)
		except ValueError:
			print "ERROR: Invalid decimal number entered!"
		    	return

	if ini_name == "elevation:":
		try:
		    	float(new_value) if '.' in new_value else int(new_value)
		except ValueError:
			print "ERROR: Invalid decimal number entered!"
		    	return

	if ini_name == "yahoo:":
		new_value = string.upper(new_value)

	if ini_name == "wunderground:":
		new_value = string.upper(new_value)

	if ini_name == "number:":
                try:
                        stripped = str(int(new_value))
                except:
			print "ERROR: Invalid integer number entered!"
                        return
		
	if ini_name == "timeout:":
                try:
                        stripped = str(int(new_value))
                except:
			print "ERROR: Invalid integer number entered!"
                        return

	if ini_name == "mode:":	
		if new_value not in ["gpsd", "gpsbt"]:
			print "ERROR: Invalid GPS source entered!"
			return

	if ini_name == "host:":
		try:
			inet_aton(str(new_value))
		except:
			print "ERROR: Invalid IP address entered!"
			return
	
	if ini_name == "port:":
               	try:
                        stripped = str(int(new_value))
                except:
			print "ERROR: Invalid integer number entered!"
                        return
	
	if ini_name == "resolution:":
                try:
                        stripped = str(int(new_value))
                except:
			print "ERROR: Invalid integer number entered!"
                        return

	if ini_name in ["source:", "group"]:
		if not re.search("(http|https)://([a-zA-Z0-9.]|%[0-9A-Za-z]|/|:[0-9]?)*", new_value):
			print "ERROR: Invalid URL entered!"
			return

	ini_treestore[path][1] = new_value

	return

def populate_url(url_treestore, file_name):
	try:
		run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
		data_dir = os.path.join(run_dir, 'data')
		url_file = os.path.join(data_dir, file_name)
		f = open(url_file, "rb")
	except:
		on_show_note("clicked",window,"url data file missing","OK")
		print "URL data (photo links) are missing!  Ignoring request!"
	else:
		while 1:
			t = f.readline()
			if t == '':
				break
			if '\n' in t:
				t = t[:-1]
			if '\r' in t:
				t = t[:-1]
			a = t.split()
			object_name = a[0]
			object_url = a[1]

			piter = url_treestore.append(None, [object_name, object_url])

	f.close()

	return(url_treestore)

class main_window:
	def __init__(self):
		global initial_lat
		global initial_long
		global initial_temp
		global initial_pressure
		global initial_elevation
		global yahoo_location
		global wunderground_location
		global tle_number
		global tle_group
		global gps_mode
		global gps_timeout
		global gpsd_host
		global gpsd_port
		global ngc_source
		global asteroid_source
		global comet_source
		global neo_source
		global orbit_resolution
		global ini_treestore
		global messier_image_treestore
		global misc_image_treestore
		global star_treestore
		global planet_treestore
		global messier_treestore
		global ngc_treestore
		global satellite_treestore
		global asteroid_treestore
		global comet_treestore
		global neo_treestore
		global constellation_treestore
		global moon_feature_treestore

		config = ConfigParser.ConfigParser()
                run_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
		ini_file = os.path.join(run_dir, 'mephemeris.ini')
		config.read(ini_file)

		longitude = config.get("location","longitude")
		latitude = config.get("location","latitude")
		temperature = config.get("location","temperature")
		pressure = config.get("location","pressure")
		elevation = config.get("location","elevation")
		yahoo_location = config.get("location","yahoo")
		wunderground_location = config.get("location","wunderground")
		tle_number = config.get("tle","number")
		tle_group = config.get("tle","group")
		gps_mode = config.get("gps","mode")
		gps_timeout = config.get("gps","timeout")
		gpsd_host = config.get("gps","host")
		gpsd_port = config.get("gps","port")
		ngc_source = config.get("ngc","source")
		asteroid_source = config.get("asteroid","source")
		comet_source = config.get("comet","source")
		neo_source = config.get("neo","source")
		orbit_resolution = float(config.get("plot","resolution"))
		orbit_resolution = int(orbit_resolution)

		local = ephem.Observer()

		local.long = str(longitude)
		local.lat = str(latitude)
		local.temp = float(temperature)
		local.pressure = float(pressure)
		local.elevation = float(elevation)

		initial_lat = local.lat
		initial_long = local.long
		initial_temp = local.temp
		initial_pressure = local.pressure
		initial_elevation = local.elevation

		print "local.lat=", local.lat
		print "local.long=", local.long
		print "local.temp=", local.temp
		print "local.pressure=", local.pressure
		print "local.elevation=", local.elevation

		print "yahoo location=", yahoo_location
		print "wunderground location=", wunderground_location

		print "tle number=", tle_number
		print "tle_group=", tle_group

		print "gps mode=", gps_mode
		print "gps timeout=", gps_timeout
		print "gpsd host=", gpsd_host
		print "gpsd port=", gpsd_port

		print "ngc source=", ngc_source
		print "asteroid_source=", asteroid_source
		print "comet_source=", comet_source
		print "neo_source=", neo_source

		now = ephem.now()
		save = now
		local.date = ephem.now()
		local.date = now

		if hildon_flag == 1:
			window = hildon.Window()
		else:
			window = gtk.Window(gtk.WINDOW_TOPLEVEL)

		window.connect("destroy", gtk.main_quit)
		window.connect("window-state-event", on_window_state_change, window)
		window.connect("key-press-event", on_key_press_basic, window)

		self.window_in_fullscreen = False

		window.set_title('MEphemeris ' + str(VERSION))

		window.set_size_request(800,480)

		path = os.path.dirname(os.path.abspath(sys.argv[0]))
		image_file = os.path.join(path, "mephemeris.jpg")
		image = gtk.Image()
		image.set_from_file(image_file)
		image.show()

		button = gtk.Button()
		window.add(button)
		button.connect("clicked",show_object_info_basic_start,window,button,local)
		button.set_image(image)
		button.show()

		window.show()

		if window_in_fullscreen == True:
			window.fullscreen()

		window.show_all()

                # create a ini TreeStore
                ini_treestore = gtk.TreeStore(str, str)

		try:
			f = open(ini_file, "rb")
		except:
			on_show_note("clicked",window,"ini data file missing -- download it from web window","OK")
			print "ini data (initialization parameters) are missing!  Ignoring request!"
		else:
			while 1:
				t = f.readline()
				if t == '':
					break
				if '\n' in t:
					t = t[:-1]
				if '\r' in t:
					t = t[:-1]
				a = t.split()
				ini_name = a[0]
				if len(a) > 1:
					ini_value = a[1]
				else:
					ini_value = ""

				if ini_name[:1] == "[":
					piter = ini_treestore.append(None, [ini_name, ini_value])
				else:
					citer = ini_treestore.append(piter, [ini_name, ini_value])

		f.close()

                # create a misc photos TreeStore
                misc_image_treestore = gtk.TreeStore(str, str)
		file_name = 'misc_url.dat'
		populate_url(misc_image_treestore, file_name) 

                # create a messier photos TreeStore
                messier_image_treestore = gtk.TreeStore(str, str)
		file_name = 'messier_url.dat'
		populate_url(messier_image_treestore, file_name) 

		# create a star TreeStore
		star_treestore = gtk.TreeStore(str, str, str, str, str, float, float, float, float)

                # create a planet TreeStore
                planet_treestore = gtk.TreeStore(str, str, str, str, str, float, float, float, float)

		# create a messier TreeStore
		messier_treestore = gtk.TreeStore(str, str, str, str, str, float, float, float, float)

		# create a NGC TreeStore
		ngc_treestore = gtk.TreeStore(str, str, str, str, str, float, float, float, float)

		# create a satellite TreeStore
		satellite_treestore = gtk.TreeStore(str, str, str, str, str, float, float, float, float)

		# create an asteroid TreeStore
		asteroid_treestore = gtk.TreeStore(str, str, str, str, str, float, float, float, float)

		# create a comet TreeStore
		comet_treestore = gtk.TreeStore(str, str, str, str, str, float, float, float, float)

		# create a NEO TreeStore
		neo_treestore = gtk.TreeStore(str, str, str, str, str, float, float, float, float)

		# create a constellation TreeStore
		constellation_treestore = gtk.TreeStore(str, str, str, str, str, float, float, float, float)

		# create a moon feature TreeStore
		moon_feature_treestore = gtk.TreeStore(str, str, str, str)

        def main(self):
           	gtk.main()
           	return

if __name__ == '__main__': 
        main = main_window()
        main.main()


