/**
 * maemo-barcode - barcode detection/recognition application
 *
 * Copyright (c) 2008 Simon Pickering <S.G.Pickering@bath.ac.uk>
 *
 * Various parts of barcode recognition and GStreamer manipulation code written by:
 *       Timothy Terriberry
 *       Adam Harwell
 *       Jonas Hurrelmann
 *
 * Original GStreamer code based on the maemo-examples package
 * Copyright (c) 2007-2008 Nokia Corporation. All rights reserved.
 * Copyright (c) 2006 INdT.
 * @author Talita Menezes <talita.menezes@indt.org.br>
 * @author Cidorvan Leite <cidorvan.leite@indt.org.br>
 * @author Jami Pekkanen <jami.pekkanen@nokia.com>
 */

#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gst/gst.h>
#include <gst/gstbin.h>
#include <gst/interfaces/xoverlay.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-program.h>
#include <assert.h>
#include <gdk/gdkx.h>

#include <time.h>
#include <dmtx.h>

#include <pthread.h>
#include <sched.h>

#include "common.h"
#ifdef N900
#include "camera_n900.h"
#else
#include "camera_n8x0.h"
#endif /* N900 */

#ifndef USE_ZBAR
#include "batoo.h"
#include "image.h"
#include "binarize.h"
#include "qrcode.h"
#else /* USE_ZBAR */
#include <zbar.h>
#endif /* USE_ZBAR */

#include "maemo-barcode.h"
#include "web.h"
#include "decoder.h"



extern AppData *appdata;

//static unsigned char buffer[WIDTH * HEIGHT];
//static unsigned char buffer_out[WIDTH * HEIGHT];

extern int scanning_status;
extern int processing;

char globalEAN[14];

pthread_t dmtx_thread=NULL;
pthread_mutex_t thread_run_mutex     = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t condition_mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t  condition_cond  = PTHREAD_COND_INITIALIZER;
int thread_run = 0;
unsigned char *thread_buffer = NULL;
unsigned char *thread_output = NULL;
int thread_width = 0;
int thread_height = 0;
int thread_return = 0;
//time_t thread_last_run_time;
int thread_fourcc;

#ifndef USE_ZBAR
clock_t QR1D_timer;
#endif /* USE_ZBAR */

clock_t dmtx_timer;



#ifndef USE_ZBAR

// define run intervals in s (annoyingly the fn returns a double)
#define QR1D_RUN_INTERVAL (int)(0.200 * CLOCKS_PER_SEC)
#define DMTX_RUN_INTERVAL (int)(1.000 * CLOCKS_PER_SEC)

/* Analyse the image data */
int analyse_image(unsigned char *buffer, unsigned char **output, unsigned char *fourcc) 
{
    gulong *microseconds;
    //output[0] = 0;
    
    if(!thread_run){
        // libdmtx thread is not running, no need for a mutex...
        // check and see if it has decoded something
        if(thread_return){
            fprintf(stdout, "analyse_image(): found datamatrix\n");
            // it has decoded something)
            *output = thread_output; // make output a pointer to the thread shared output buffer
            thread_return = 0;
            return BARCODE_TYPE_DMTX;
        }else if (clock()-dmtx_timer>=DMTX_RUN_INTERVAL){
            // not decoded or not been run
            // so start it running
            memcpy(thread_buffer, buffer, thread_width*thread_height*sizeof(char)); // should be the right amount of data....
            
            
            //pthread_mutex_lock( &thread_run_mutex );
            thread_run = 1; // tell the code (this and the decoder thread) that it should be running
            //pthread_mutex_unlock( &thread_run_mutex );
           
            pthread_mutex_lock( &condition_mutex );
            pthread_cond_signal( &condition_cond ); // signal the thread to start running again
            pthread_mutex_unlock( &condition_mutex );
        }
    }
    // otherwise it must be running, so leave it be
    
    
    // limit how many times this code is called too
    if (clock()-QR1D_timer<QR1D_RUN_INTERVAL)
        return BARCODE_TYPE_NONE;
    else
        QR1D_timer = clock(); // otherwise reset it
    
    if(batoo_analyse_image(buffer, output, appdata->frame_width, appdata->frame_height)){
        fprintf(stdout, "Batoo successful: %s\n", *output);
        return BARCODE_TYPE_1D;
    }
    else if(derfQR_analyse_image(buffer, output, appdata->frame_width, appdata->frame_height)){
        fprintf(stdout, "DerfQR successful: %s\n", *output);
        return BARCODE_TYPE_QRCODE;
    }
    //else if(libdmtx_analyse_image(buffer, output, appdata->frame_width, appdata->frame_height)){
    //    fprintf(stdout, "libdmtx successful: %s\n", output);
    //    return BARCODE_TYPE_DMTX;
    //}
    else{
        return BARCODE_TYPE_NONE;
    }
}



/* Analyse the image data - try 1D barcode */
int batoo_analyse_image(unsigned char *buffer, unsigned char **output, int width, int height) // output is malloc'd in here
{
   //static unsigned char single_col[height];
   int *resultlist[MAX_VOTES];
   char *temp_output;
   int row;
   
   int votes[MAX_VOTES];
   int last_vote=0, current_vote=-1;
   int i;

   fprintf(stdout, "Entered analyse_1dbarcode_image()\n");
  
   for (row=0; row < height; row=row+4){
      int *result = analyse_row (width, (unsigned char *)(buffer + width * row));
      int detected=0;

      if (result != NULL){
         // see if this one has already been detected
         temp_output = number_to_string(result);
         fprintf(stdout, "found something: %s\n", temp_output);
         free(temp_output);
         detected=0;
         for(i=0;i<last_vote;i++){
            if (compare_numbers (resultlist[i], result)){
               detected=1;
               votes[i]++;
               if (votes[i]>WINNING_VOTES){
                  //appdata->label_text = number_to_string (result);
                  *output = number_to_string(result);
                  fprintf(stdout, "Detected EAN: %s\n", *output);
                  //appdata->scan_in_progress=0; // stop the scan now then
                  return TRUE;
               }
            }
         }
         
         // check that it was found
         if(!detected){
            // and if not we add it to the end
            if(last_vote<MAX_VOTES){ // check there's room
               resultlist[last_vote++] = result;
            }else{
               // we have already filled the voting slots
               // push out the oldest entry and keep going
               // if we've already filled in the array, we start counting from the start again, and fill it up once more
               current_vote++;
               if(current_vote>MAX_VOTES)
                  current_vote=0;
               resultlist[current_vote] = result;
            }
         }
      }
      

      // if we've been cancelled then return immediately
      if (!scanning_status){
         fprintf(stdout, "User requested scan to stop\n");
         return FALSE;
      }
   }

   //printf("Scan timed out and found no matches\n");

   /* So presumably nothing won, we could return whichever one came top */
   return FALSE;
}




int derfQR_analyse_image(unsigned char *img, unsigned char **output, int width, int height) // output malloc'd in here
{
    qr_reader          *reader;
    //qr_code_data_list   qrlist;
    char              **text;
    //int                 i;
    int                 ntext;
  
    fprintf(stdout, "Entered derfQR_analyse_image()\n");
    
    qr_binarize(img,width,height);
    reader=qr_reader_alloc();
    ntext=qr_reader_extract_text(reader,img,width,height,&text,1);
    qr_reader_free(reader);
    fprintf(stdout, "derfQR_analyse_image(): ntext=%d\n", ntext);
    
    if(ntext){
        fprintf(stdout, "derfQR_analyse_image(): %s\n", text[0]);
        *output = (char *) malloc (strlen(text[0]) * sizeof (char)); // only handle the first string
        strcpy(*output, text[0]);
        qr_text_list_free(text,ntext);
        return TRUE;
    }else{
        qr_text_list_free(text,ntext);
        return FALSE;  
    }
}






#else /* USE_ZBAR */







// define run intervals in s (annoyingly the fn returns a double)
#define DMTX_RUN_INTERVAL (int)(1.000 * CLOCKS_PER_SEC)

/* Analyse the image data */
int analyse_image(unsigned char *buffer, unsigned char **output, unsigned char *fourcc, int bpp) 
{
    gulong *microseconds;
    unsigned char *out = output;

    //output[0] = 0;
    
    if(!thread_run){
        fprintf(appdata->fid, "analyse_image(): no dmtx thread running\n");
        fflush(appdata->fid);
        // libdmtx thread is not running, no need for a mutex...
        // check and see if it has decoded something
        if(thread_return){
            fprintf(appdata->fid, "analyse_image(): found datamatrix\n");
            fflush(appdata->fid);
            // it has decoded something)
            *output = thread_output; // make output a pointer to the thread shared output buffer
            thread_return = 0;
            return ZBAR_QRCODE; // QR is also 2D so achieves the same effect //BARCODE_TYPE_DMTX;
        }else if (clock()-dmtx_timer>=DMTX_RUN_INTERVAL){
            fprintf(appdata->fid, "analyse_image(): start dmtx thread running\n");
            fflush(appdata->fid);
            // not decoded or not been run
            // so start it running
            memcpy(thread_buffer, buffer, thread_width*thread_height*sizeof(char)*bpp/8); // should be the right amount of data....
            if(strstr(fourcc, "RGB")){
                fprintf(appdata->fid, "is RGB, %d\n", thread_fourcc); // DEBUG
                fflush(appdata->fid);
                if(bpp==16)
                    thread_fourcc=DmtxPack16bppRGB;
                else if (bpp==24)
                    thread_fourcc=DmtxPack24bppRGB;
                else if (bpp==32)
                    thread_fourcc=DmtxPack32bppRGBX;                    
            }else if (bpp==8)
                thread_fourcc=DmtxPack8bppK;
            
            //pthread_mutex_lock( &thread_run_mutex );
            thread_run = 1; // tell the code (this and the decoder thread) that it should be running
            //pthread_mutex_unlock( &thread_run_mutex );
           
            pthread_mutex_lock( &condition_mutex );
            pthread_cond_signal( &condition_cond ); // signal the thread to start running again
            pthread_mutex_unlock( &condition_mutex );
        }
    }
    // otherwise it must be running, so leave it be
    
    //printf("about to zbar_image_create\n");
    //image = zbar_image_create();
    //printf("called zbar_image_create\n");
    zbar_image_set_format(appdata->image, *(int*)fourcc); //*(int*)"Y800");
    fprintf(appdata->fid, "set image format\n");
    fflush(appdata->fid);
    zbar_image_set_size(appdata->image, appdata->frame_width, appdata->frame_height);
    fprintf(appdata->fid, "set image size\n");
    fflush(appdata->fid);
    zbar_image_set_data(appdata->image, buffer, appdata->frame_width * appdata->frame_height, NULL); //zbar_image_free_data
    fprintf(appdata->fid, "set image data\n");
    fflush(appdata->fid);

    /* scan the image for barcodes */
    int n = zbar_scan_image(appdata->scanner, appdata->image);
    if(n){
        /* extract results */
        const zbar_symbol_t *symbol = zbar_image_first_symbol(appdata->image); // only use the first symbol
        zbar_symbol_type_t typ = zbar_symbol_get_type(symbol);
        printf("Extracting symbols :)\n");
        *output = zbar_symbol_get_data(symbol);
        printf("decoded %s symbol \"%s\"\n", zbar_get_symbol_name(typ), *output);
        
        /* clean up */
        //zbar_image_destroy(image);
        /*
        if(typ==ZBAR_QRCODE)
            return BARCODE_TYPE_QRCODE;
        else if(typ==ZBAR_ISBN13 || typ==ZBAR_EAN13 || typ==ZBAR_UPCA || typ==ZBAR_ISBN10 || typ==ZBAR_UPCE || typ==ZBAR_EAN8)
            return BARCODE_TYPE_1D;
        */
        return typ;
    }else{
        //zbar_image_destroy(image);
        return ZBAR_NONE; //BARCODE_TYPE_NONE;
    }
}

#endif /* USE_ZBAR */

void libdmtx_thread_main()
{
    while(1){
        pthread_mutex_lock( &condition_mutex );
        while( thread_run==0 )
        {
            pthread_cond_wait( &condition_cond, &condition_mutex );
        }
        pthread_mutex_unlock( &condition_mutex );

        //pthread_mutex_lock( &thread_run_mutex );
        // copy data from global int array - might not be needed actually
        
        // call our function
        
        thread_return = libdmtx_analyse_image(thread_buffer, &thread_output, thread_width, thread_height);
        
        // return data in our global vars
        
        // reset our timer, hopefully this should be threadsafe, etc...
        dmtx_timer = clock();

        // stop ourselves running
        thread_run = 0;
        //pthread_mutex_unlock( &thread_run_mutex );        
        
    }
}

/* Analyse the image data */
int libdmtx_analyse_image(unsigned char *data, unsigned char **output, int width, int height) // output malloc'd in here
{
    //GdkPixbuf *pixbuf = NULL;
    //GError *error = NULL;
    //unsigned int bpp;
    //const char *directory;
    //GString *filename;
    //unsigned int base_len, i;
    //struct stat statbuf;
    //int ret,ii;
    //int ean[14];
    //DmtxEncode     *enc;
    DmtxImage      *img;
    DmtxDecode     *dec;
    DmtxRegion     *reg;
    DmtxMessage    *msg;
    DmtxTime        timeout;
    
    fprintf(stdout, "Entered libdmtx_analyse_image()\n");

    img = dmtxImageCreate(data, width, height, thread_fourcc); // DmtxPack8bppK
    assert(img != NULL);
    fprintf(stdout, "libdmtx_analyse_image(): done dmtxImageCreate\n");

    dec = dmtxDecodeCreate(img, 1);
    assert(dec != NULL);
    fprintf(stdout, "libdmtx_analyse_image(): done dmtxDecodeCreate\n");
    
    timeout = dmtxTimeAdd(dmtxTimeNow(), 1000); // give it 100 ms to analyse...
    reg = dmtxRegionFindNext(dec, &timeout);
    
    fprintf(stdout, "libdmtx_analyse_image(): done dmtxRegionFindNext\n");
    if(reg != NULL) {
        msg = dmtxDecodeMatrixRegion(dec, reg, DmtxUndefined);
        fprintf(stdout, "libdmtx_analyse_image(): done dmtxDecodeMatrixRegion\n");
        if(msg != NULL) {
            fprintf(stdout, "Found datamatrix barcode!\n");
            //fputs("output: \"", stdout);
            //fwrite(msg->output, sizeof(unsigned char), msg->outputIdx, stdout);
            //fputs("\"\n", stdout);
            *output = (char *) malloc((msg->outputIdx+1) * sizeof (char));
            memcpy(*output, msg->output, msg->outputIdx);
            *(*output + msg->outputIdx +1) = 0;
            
            dmtxMessageDestroy(&msg);
            dmtxRegionDestroy(&reg);
            dmtxDecodeDestroy(&dec);
            dmtxImageDestroy(&img);
            return TRUE;
        }
        dmtxRegionDestroy(&reg);
    }

    dmtxDecodeDestroy(&dec);
    dmtxImageDestroy(&img);
    return FALSE;
}





void init_dmtx_thread()
{
    pthread_attr_t tattr;
    int ret;
    struct sched_param param;
    
    /* initialized with default attributes */
    ret = pthread_attr_init (&tattr);
    
    /* safe to get existing scheduling param */
    ret = pthread_attr_getschedparam (&tattr, &param);

    /* set the priority; others are unchanged */
    param.sched_priority = param.sched_priority + 10;

    /* setting the new scheduling param */
    ret = pthread_attr_setschedparam (&tattr, &param);

    // let's tack on the thread init stuff here too.
    ret = pthread_create( &dmtx_thread, &tattr, libdmtx_thread_main, NULL);
    
    thread_width = appdata->frame_width; // not really necessary as these won't change during the program
    thread_height = appdata->frame_height;
    thread_buffer = (char*)malloc(thread_width*thread_height*sizeof(char));
}
 


