/*****************************************************************************
 *** Mauku - Jaiku client for Maemo devices 
 ***
 *** User interface for sending
 ***
 *** Copyright (c) 2007 Henrik Hedberg <hhedberg@innologies.fi>
 ***
 *** Licensed under the Apache License, Version 2.0 (the "License");
 *** you may not use this file except in compliance with the License.
 *** You may obtain a copy of the License at
 ***
 ***     http://www.apache.org/licenses/LICENSE-2.0
 ***
 *** Unless required by applicable law or agreed to in writing, software
 *** distributed under the License is distributed on an "AS IS" BASIS,
 *** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *** See the License for the specific language governing permissions and
 *** limitations under the License.
 ***
 *****************************************************************************/

#include "config.h"
 
#include <glib.h>
#include <gtk/gtk.h>
#include <gconf/gconf-client.h>
#include <json.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <string.h>
#include <stdlib.h>

#if HILDON == 1

#include <hildon/hildon.h>

#else

#include <hildon-widgets/hildon-banner.h>
#include <hildon-widgets/hildon-note.h>

#endif

#include <miaouw/miaouwwindow.h>
#include <miaouw/miaouwscrolledwindow.h>

#include "mauku.h"

typedef struct {
	View* view;
	gchar* channel; /* or NULL (Jaiku channel) */
	gchar* nick; /* or NULL (Twitter nick) */
	gint icon_id;
	GtkWidget* vbox;
	GtkWidget* hbox;
	GtkWidget* icon;
	GtkWidget* ok_button;
	GtkWidget* icon_button;
	GtkWidget* chars_label;
	GtkTextBuffer* text_buffer;
	GtkWidget* jaiku_check_button;
	GtkWidget* twitter_check_button;
	GtkTextMark* start_mark;
	GtkTextMark* end_mark;
} SendingData;

typedef struct {
	gint id;
	gchar* name;
	GdkPixbuf* pixbuf;
} Icon;

typedef struct {
	Icon* icon;
	GtkWidget* event_box;
	GtkWidget* scrolled_window;
	GtkWidget* dialog;
} IconData;

static gboolean clear_entry(GtkTextView* text_view, GdkEventFocus* event, SendingData* sending_data);  
void destroy_sending_data(SendingData* sending_data);
static void on_add_comment_clicked(GtkButton* button, SendingData* sending_data);
static void on_add_icon_clicked(GtkButton* not_used, SendingData* sending_data);
static gboolean on_icon_button_press(GtkWidget* widget, GdkEventButton* event, IconData* data);
static gboolean on_icon_motion_notify(GtkWidget* widget, GdkEventMotion* event, IconData* data);
static gboolean on_icon_button_release(GtkWidget* widget, GdkEventButton* event, IconData* data);
static void on_send_ok_clicked(SendingData* sending_data);
static void remove_top_bottom_widget(SendingData* sending_data);
static void update_labels(SendingData* sending_data);

static Icon icons[] = {
	{ 300, "http://www.jaiku.com/themes/classic/icons/web-speechbubble.gif", NULL },
	{ 301, "http://www.jaiku.com/themes/classic/icons/web-car.gif", NULL },
	{ 302, "http://www.jaiku.com/themes/classic/icons/web-alarmclock.gif", NULL },
	{ 303, "http://www.jaiku.com/themes/classic/icons/web-loudspeaker.gif", NULL },
	{ 304, "http://www.jaiku.com/themes/classic/icons/web-tram.gif", NULL },
	{ 305, "http://www.jaiku.com/themes/classic/icons/web-casette.gif", NULL },
	{ 306, "http://www.jaiku.com/themes/classic/icons/web-underware.gif", NULL },
	{ 307, "http://www.jaiku.com/themes/classic/icons/web-rollerblade.gif", NULL },
	{ 308, "http://www.jaiku.com/themes/classic/icons/web-uzi.gif", NULL },
	{ 309, "http://www.jaiku.com/themes/classic/icons/web-scoop.gif", NULL },
	{ 310, "http://www.jaiku.com/themes/classic/icons/web-bomb.gif", NULL },
	{ 311, "http://www.jaiku.com/themes/classic/icons/web-bra.gif", NULL },
	{ 312, "http://www.jaiku.com/themes/classic/icons/web-videotape.gif", NULL },
	{ 313, "http://www.jaiku.com/themes/classic/icons/web-cigarettes.gif", NULL },
	{ 314, "http://www.jaiku.com/themes/classic/icons/web-vinyl.gif", NULL },
	{ 315, "http://www.jaiku.com/themes/classic/icons/web-champaign.gif", NULL },
	{ 316, "http://www.jaiku.com/themes/classic/icons/web-airplain.gif", NULL },
	{ 317, "http://www.jaiku.com/themes/classic/icons/web-bus.gif", NULL },
	{ 318, "http://www.jaiku.com/themes/classic/icons/web-grumpy.gif", NULL },
	{ 319, "http://www.jaiku.com/themes/classic/icons/web-coffee.gif", NULL },
	{ 320, "http://www.jaiku.com/themes/classic/icons/web-camera.gif", NULL },
	{ 321, "http://www.jaiku.com/themes/classic/icons/web-basketball.gif", NULL },
	{ 322, "http://www.jaiku.com/themes/classic/icons/web-beer.gif", NULL },
	{ 323, "http://www.jaiku.com/themes/classic/icons/web-binoculars.gif", NULL },
	{ 324, "http://www.jaiku.com/themes/classic/icons/web-boiler.gif", NULL },
	{ 325, "http://www.jaiku.com/themes/classic/icons/web-walk.gif", NULL },
	{ 326, "http://www.jaiku.com/themes/classic/icons/web-wallclock.gif", NULL },
	{ 327, "http://www.jaiku.com/themes/classic/icons/web-trashcan.gif", NULL },
	{ 328, "http://www.jaiku.com/themes/classic/icons/web-tv.gif", NULL },
	{ 329, "http://www.jaiku.com/themes/classic/icons/web-computing.gif", NULL },
	{ 330, "http://www.jaiku.com/themes/classic/icons/web-videocamera.gif", NULL },
	{ 331, "http://www.jaiku.com/themes/classic/icons/web-game.gif", NULL },
	{ 332, "http://www.jaiku.com/themes/classic/icons/web-cone.gif", NULL },
	{ 333, "http://www.jaiku.com/themes/classic/icons/web-driller.gif", NULL },
	{ 334, "http://www.jaiku.com/themes/classic/icons/web-popcorn.gif", NULL },
	{ 335, "http://www.jaiku.com/themes/classic/icons/web-playshirt.gif", NULL },
	{ 336, "http://www.jaiku.com/themes/classic/icons/web-disc.gif", NULL },
	{ 337, "http://www.jaiku.com/themes/classic/icons/web-event.gif", NULL },
	{ 338, "http://www.jaiku.com/themes/classic/icons/web-exclamationmark.gif", NULL },
	{ 339, "http://www.jaiku.com/themes/classic/icons/web-football.gif", NULL },
/*	{ 340, "http://www.jaiku.com/themes/classic/icons/web-footballshoe.gif", NULL }, */
	{ 341, "http://www.jaiku.com/themes/classic/icons/web-eat.gif", NULL },
	{ 342, "http://www.jaiku.com/themes/classic/icons/web-gameboy.gif", NULL },
	{ 343, "http://www.jaiku.com/themes/classic/icons/web-grenade.gif", NULL },
	{ 344, "http://www.jaiku.com/themes/classic/icons/web-hand.gif", NULL },
	{ 345, "http://www.jaiku.com/themes/classic/icons/web-hanger.gif", NULL },
	{ 346, "http://www.jaiku.com/themes/classic/icons/web-hearingprotector.gif", NULL },
	{ 347, "http://www.jaiku.com/themes/classic/icons/web-love.gif", NULL },
	{ 348, "http://www.jaiku.com/themes/classic/icons/web-balloons.gif", NULL },
	{ 349, "http://www.jaiku.com/themes/classic/icons/web-clock.gif", NULL },
	{ 350, "http://www.jaiku.com/themes/classic/icons/web-barrier.gif", NULL },
/*	{ 351, "http://www.jaiku.com/themes/classic/icons/web-laptop.gif", NULL }, */
	{ 352, "http://www.jaiku.com/themes/classic/icons/web-megaphone.gif", NULL },
	{ 353, "http://www.jaiku.com/themes/classic/icons/web-microwave.gif", NULL },
	{ 354, "http://www.jaiku.com/themes/classic/icons/web-book.gif", NULL },
	{ 355, "http://www.jaiku.com/themes/classic/icons/web-middlefinger.gif", NULL },
	{ 356, "http://www.jaiku.com/themes/classic/icons/web-notes.gif", NULL },
	{ 357, "http://www.jaiku.com/themes/classic/icons/web-question.gif", NULL },
	{ 358, "http://www.jaiku.com/themes/classic/icons/web-rollator.gif", NULL },
	{ 359, "http://www.jaiku.com/themes/classic/icons/web-shuttlecock.gif", NULL },
	{ 360, "http://www.jaiku.com/themes/classic/icons/web-salt.gif", NULL },
	{ 361, "http://www.jaiku.com/themes/classic/icons/web-scull.gif", NULL },
	{ 362, "http://www.jaiku.com/themes/classic/icons/web-sk8.gif", NULL },
	{ 363, "http://www.jaiku.com/themes/classic/icons/web-sleep.gif", NULL },
	{ 364, "http://www.jaiku.com/themes/classic/icons/web-snorkeling.gif", NULL },
	{ 365, "http://www.jaiku.com/themes/classic/icons/web-snowflake.gif", NULL },
	{ 366, "http://www.jaiku.com/themes/classic/icons/web-soda.gif", NULL },
	{ 367, "http://www.jaiku.com/themes/classic/icons/web-song.gif", NULL },
	{ 368, "http://www.jaiku.com/themes/classic/icons/web-spraycan.gif", NULL },
	{ 369, "http://www.jaiku.com/themes/classic/icons/web-sticks.gif", NULL },
	{ 370, "http://www.jaiku.com/themes/classic/icons/web-storm.gif", NULL },
	{ 371, "http://www.jaiku.com/themes/classic/icons/web-straitjacket.gif", NULL },
	{ 372, "http://www.jaiku.com/themes/classic/icons/web-metro.gif", NULL },
	{ 373, "http://www.jaiku.com/themes/classic/icons/web-luggage.gif", NULL },
	{ 374, "http://www.jaiku.com/themes/classic/icons/web-sun.gif", NULL },
	{ 375, "http://www.jaiku.com/themes/classic/icons/web-taxi.gif", NULL },
	{ 376, "http://www.jaiku.com/themes/classic/icons/web-technics.gif", NULL },
	{ 377, "http://www.jaiku.com/themes/classic/icons/web-toaster.gif", NULL },
	{ 378, "http://www.jaiku.com/themes/classic/icons/web-train.gif", NULL },
	{ 379, "http://www.jaiku.com/themes/classic/icons/web-wheelchair.gif", NULL },
	{ 380, "http://www.jaiku.com/themes/classic/icons/web-zippo.gif", NULL },
	{ 381, "http://www.jaiku.com/themes/classic/icons/web-icecream.gif", NULL },
	{ 382, "http://www.jaiku.com/themes/classic/icons/web-movie.gif", NULL },
	{ 383, "http://www.jaiku.com/themes/classic/icons/web-makeup.gif", NULL },
	{ 384, "http://www.jaiku.com/themes/classic/icons/web-bandaid.gif", NULL },
	{ 385, "http://www.jaiku.com/themes/classic/icons/web-wine.gif", NULL },
	{ 386, "http://www.jaiku.com/themes/classic/icons/web-clean.gif", NULL },
	{ 387, "http://www.jaiku.com/themes/classic/icons/web-blading.gif", NULL },
	{ 388, "http://www.jaiku.com/themes/classic/icons/web-bike.gif", NULL },
	{ 389, "http://www.jaiku.com/themes/classic/icons/web-pils.gif", NULL },
	{ 390, "http://www.jaiku.com/themes/classic/icons/web-picnic.gif", NULL },
	{ 391, "http://www.jaiku.com/themes/classic/icons/web-lifejacket.gif", NULL },
	{ 392, "http://www.jaiku.com/themes/classic/icons/web-home.gif", NULL },
	{ 393, "http://www.jaiku.com/themes/classic/icons/web-happy.gif", NULL },
	{ 394, "http://www.jaiku.com/themes/classic/icons/web-toiletpaper.gif", NULL },
	{ 395, "http://www.jaiku.com/themes/classic/icons/web-theatre.gif", NULL },
	{ 396, "http://www.jaiku.com/themes/classic/icons/web-shop.gif", NULL },
	{ 397, "http://www.jaiku.com/themes/classic/icons/web-search.gif", NULL },
	{ 398, "http://www.jaiku.com/themes/classic/icons/web-cloudy.gif", NULL },
	{ 399, "http://www.jaiku.com/themes/classic/icons/web-hurry.gif", NULL },
	{ 400, "http://www.jaiku.com/themes/classic/icons/web-morning.gif", NULL },
/*	{ 401, "http://www.jaiku.com/themes/classic/icons/web-car.gif", NULL }, */
	{ 402, "http://www.jaiku.com/themes/classic/icons/web-baby-boy.gif", NULL },
	
	{ 0, NULL, NULL }
};

static gint id;

void sending_show_add_comment_top_bottom_widget(View* view) {
	SendingData* sending_data;
	GtkWidget* note;
	GtkWidget* hbox;
	GtkWidget* button_hbox;
	GtkWidget* cancel_button;
	GtkWidget* text_label;
	GtkWidget* text_view;
	GtkWidget* scrolled;
	gchar* s;
		
	sending_data = g_malloc0(sizeof(SendingData));
	sending_data->view = view;
	sending_data->vbox = gtk_vbox_new(FALSE, 0);
	g_signal_connect_swapped(sending_data->vbox, "destroy", G_CALLBACK(g_free), sending_data);
	gtk_box_pack_start(GTK_BOX(sending_data->vbox), gtk_hseparator_new(), FALSE, FALSE, 0);
	
	text_view = gtk_text_view_new();
	gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(text_view), GTK_WRAP_WORD);
	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled), GTK_SHADOW_ETCHED_IN);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_widget_set_size_request(text_view, 480, 92);
	gtk_container_add(GTK_CONTAINER(scrolled), text_view);
	gtk_box_pack_start(GTK_BOX(sending_data->vbox), scrolled, FALSE, FALSE, 2);

	sending_data->text_buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text_view));
	g_signal_connect_swapped(sending_data->text_buffer, "changed", G_CALLBACK(update_labels), sending_data);
	
	hbox = gtk_hbox_new(FALSE, 4);
	gtk_box_pack_start(GTK_BOX(sending_data->vbox), hbox, FALSE, FALSE, 0);
	text_label = gtk_label_new("Add a comment.");
	gtk_box_pack_start(GTK_BOX(hbox), text_label, TRUE, TRUE, 0);

	button_hbox = gtk_hbox_new(TRUE, 4);
	gtk_box_pack_start(GTK_BOX(hbox), button_hbox, FALSE, FALSE, 0);
	sending_data->ok_button = gtk_button_new_from_stock(GTK_STOCK_OK);
	gtk_widget_set_sensitive(sending_data->ok_button, FALSE);
	g_signal_connect(sending_data->ok_button, "clicked", G_CALLBACK(on_add_comment_clicked), sending_data);
	gtk_box_pack_start(GTK_BOX(button_hbox), sending_data->ok_button, TRUE, TRUE, 0);
	cancel_button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
	g_signal_connect_swapped(cancel_button, "clicked", G_CALLBACK(remove_top_bottom_widget), sending_data);
	gtk_box_pack_start(GTK_BOX(button_hbox), cancel_button, TRUE, TRUE, 0);

	view_set_top_bottom_widget(view, sending_data->vbox, FALSE);
	gtk_widget_show_all(sending_data->vbox);
	gtk_window_present(GTK_WINDOW(view_get_window(view)));

	gtk_widget_grab_focus(text_view);
}

void sending_show_send_top_bottom_widget(View* view) {
	SendingData* sending_data;
	gchar* s;
	GtkWidget* note;
	GtkWidget* hbox;
	GtkWidget* button_hbox;
	GtkWidget* cancel_button;
	GtkWidget* text_label;
	GtkWidget* text_view;
	GtkWidget* scrolled;
	GtkTextIter start_iter;
	GtkTextIter end_iter;
	
	sending_data = g_malloc0(sizeof(SendingData));
	sending_data->view = view;
	if (g_str_has_prefix(view_get_unique_id(view), "http://jaiku.com/channel/") && (s = strchr(view_get_unique_id(view) + 25, '/'))) {
		sending_data->channel = g_strndup(view_get_unique_id(view) + 25, s - (view_get_unique_id(view) + 25));
	}
	if (view_get_user_data(view, "twitter:nick")) {
		sending_data->nick = g_strconcat("@", (gchar*)view_get_user_data(view, "twitter:nick"), " ", NULL);
	}
	sending_data->vbox = gtk_vbox_new(FALSE, 0);
	g_signal_connect_swapped(sending_data->vbox, "destroy", G_CALLBACK(destroy_sending_data), sending_data);
	gtk_box_pack_start(GTK_BOX(sending_data->vbox), gtk_hseparator_new(), FALSE, FALSE, 0);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_box_pack_start(GTK_BOX(sending_data->vbox), hbox, FALSE, FALSE, 2);
	if (sending_data->nick) {
		s = g_strconcat("Reply to ", sending_data->nick + 1, NULL);
	} else if (sending_data->channel) {
		s = g_strconcat("Send to the channel #", sending_data->channel, NULL);
	} else {
		s = g_strdup("Send to ");
	}
	text_label = gtk_label_new(s);
	g_free(s);
	gtk_misc_set_alignment(GTK_MISC(text_label), 0.5, 1.0);
	gtk_box_pack_start(GTK_BOX(hbox), text_label, FALSE, FALSE, 2);
	if (!sending_data->channel && !sending_data->nick) {
		sending_data->jaiku_check_button = gtk_check_button_new_with_label("Jaiku");
		gtk_box_pack_start(GTK_BOX(hbox), sending_data->jaiku_check_button, FALSE, FALSE, 2);
		if (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_JAIKU_ENABLED, NULL)) {
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(sending_data->jaiku_check_button), TRUE);
		} else {
			gtk_widget_set_sensitive(sending_data->jaiku_check_button, FALSE);
		}
		g_signal_connect_swapped(sending_data->jaiku_check_button, "toggled", G_CALLBACK(update_labels), sending_data);
		sending_data->twitter_check_button = gtk_check_button_new_with_label("Twitter");
		gtk_box_pack_start(GTK_BOX(hbox), sending_data->twitter_check_button, FALSE, FALSE, 2);
		if (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_TWITTER_ENABLED, NULL)) {
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(sending_data->twitter_check_button), TRUE);
		} else {
			gtk_widget_set_sensitive(sending_data->twitter_check_button, FALSE);
		}
		g_signal_connect_swapped(sending_data->twitter_check_button, "toggled", G_CALLBACK(update_labels), sending_data);
	}

	sending_data->hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(sending_data->vbox), sending_data->hbox, FALSE, FALSE, 2);

	text_view = gtk_text_view_new();
	sending_data->text_buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text_view));
	gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(text_view), GTK_WRAP_WORD);
	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled), GTK_SHADOW_ETCHED_IN);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_widget_set_size_request(text_view, 480, 60);
	gtk_container_add(GTK_CONTAINER(scrolled), text_view);
	gtk_box_pack_end(GTK_BOX(sending_data->hbox), scrolled, TRUE, TRUE, 4);
	if (sending_data->nick) {
		gtk_text_buffer_set_text(sending_data->text_buffer, sending_data->nick, -1);
		gtk_text_buffer_get_bounds(sending_data->text_buffer, &start_iter, &end_iter);
		sending_data->start_mark = gtk_text_buffer_create_mark(sending_data->text_buffer, NULL, &start_iter, TRUE);
		sending_data->end_mark = gtk_text_buffer_create_mark(sending_data->text_buffer, NULL, &end_iter, TRUE);
	}

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_box_pack_start(GTK_BOX(sending_data->vbox), hbox, FALSE, FALSE, 0);
	s = g_strdup_printf("%d characters left.", (sending_data->nick ? 140 - strlen(sending_data->nick) : 140));
	sending_data->chars_label = gtk_label_new(s);
	g_free(s);
	gtk_box_pack_start(GTK_BOX(hbox), sending_data->chars_label, TRUE, TRUE, 0);
	
	g_signal_connect_swapped(sending_data->text_buffer, "changed", G_CALLBACK(update_labels), sending_data);

	button_hbox = gtk_hbox_new(TRUE, 4);
	gtk_box_pack_start(GTK_BOX(hbox), button_hbox, FALSE, FALSE, 0);
	sending_data->ok_button = gtk_button_new_from_stock(GTK_STOCK_OK);
	gtk_widget_set_sensitive(sending_data->ok_button, FALSE);
	g_signal_connect_swapped(sending_data->ok_button, "clicked", G_CALLBACK(on_send_ok_clicked), sending_data);
	gtk_box_pack_start(GTK_BOX(button_hbox), sending_data->ok_button, TRUE, TRUE, 0);
	if (!sending_data->nick && gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_JAIKU_ENABLED, NULL)) {
		sending_data->icon_button = gtk_button_new_with_label("Add Icon");
		g_signal_connect(sending_data->icon_button, "clicked", G_CALLBACK(on_add_icon_clicked), sending_data);
		gtk_box_pack_start(GTK_BOX(button_hbox), sending_data->icon_button, TRUE, TRUE, 0);
	}
	cancel_button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
	g_signal_connect_swapped(cancel_button, "clicked", G_CALLBACK(remove_top_bottom_widget), sending_data);
	gtk_box_pack_start(GTK_BOX(button_hbox), cancel_button, TRUE, TRUE, 0);

	view_set_top_bottom_widget(view, sending_data->vbox, FALSE);
	gtk_widget_show_all(sending_data->vbox);
	gtk_window_present(GTK_WINDOW(view_get_window(view)));

	gtk_widget_grab_focus(text_view);
}

void destroy_sending_data(SendingData* sending_data) {
	g_free(sending_data->nick);
	g_free(sending_data->channel);
	g_free(sending_data);
}

static void on_add_icon_clicked(GtkButton* not_used, SendingData* sending_data) {
	GtkWidget* dialog;
	GtkWidget* table;
	MicrofeedHttp* http;
	GtkWidget* image;
	GtkWidget* event_box;
	int i;
	char buffer[1024];
	GtkWidget* scrolled_window;
	GArray* icon_data_array;
	IconData icon_data;

	if (!network_make_connected() || !(http = microfeed_http_new())) {
		return;
	}
	dialog = gtk_dialog_new_with_buttons("Select an icon for a Jaiku", NULL, GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT,
	                                     "No icon", GTK_RESPONSE_REJECT, NULL);
	scrolled_window = miaouw_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled_window), GTK_SHADOW_ETCHED_IN);
	gtk_widget_set_size_request(scrolled_window, 633, 250);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), scrolled_window, FALSE, FALSE, 16);
	table = gtk_table_new(10, 10, TRUE);
	icon_data_array = g_array_new(FALSE, FALSE, sizeof(IconData));
	icon_data.dialog = dialog;
	icon_data.scrolled_window = scrolled_window;
	for (i = 0; icons[i].id; i++) {
		if (!icons[i].pixbuf) {
			snprintf(buffer, 1024, "%s/.mauku/icons/%d.png", (getenv("HOME") ? getenv("HOME") : "/home/user"), icons[i].id);
			if (!(icons[i].pixbuf = gdk_pixbuf_new_from_file(buffer, NULL))) {
				icons[i].pixbuf = image_cache_load_image(image_cache, http, icons[i].name);
				if (icons[i].pixbuf) {
					gdk_pixbuf_save(icons[i].pixbuf, buffer, "png", NULL, NULL);
				}
			}
		}
		if (icons[i].pixbuf) {
			icon_data.icon = &icons[i];
			g_array_append_val(icon_data_array, icon_data);
		}
	}
	for (i = 0; i < icon_data_array->len; i++) {
		image = gtk_image_new_from_pixbuf((&g_array_index(icon_data_array, IconData, i))->icon->pixbuf);
		gtk_misc_set_padding(GTK_MISC(image), 10, 10);
		event_box = gtk_event_box_new();
		(&g_array_index(icon_data_array, IconData, i))->event_box = event_box;
		gtk_container_add(GTK_CONTAINER(event_box), image);
		g_signal_connect(event_box, "button-release-event", G_CALLBACK(on_icon_button_release), &g_array_index(icon_data_array, IconData, i));
		g_signal_connect(event_box, "button-press-event", G_CALLBACK(on_icon_button_press), &g_array_index(icon_data_array, IconData, i));
		g_signal_connect(event_box, "motion_notify_event", G_CALLBACK(on_icon_motion_notify), &g_array_index(icon_data_array, IconData, i));
		gtk_table_attach(GTK_TABLE(table), event_box, i % 10, (i % 10) + 1, i / 10, (i / 10) + 1, 0, 0, 0, 0);
		miaouw_scrolled_window_activate_scrolling(MIAOUW_SCROLLED_WINDOW(scrolled_window), event_box);
	}
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), table);
	microfeed_http_free(http);

	id = 0;
	gtk_widget_show_all(dialog);
	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_REJECT) {
		gtk_widget_destroy(dialog);
	}

	if (sending_data->icon) {
		gtk_widget_destroy(sending_data->icon);
		sending_data->icon = NULL;
	}
	if (id) {
		for (i = 0; i < icon_data_array->len; i++) {
			if ((&g_array_index(icon_data_array, IconData, i))->icon->id == id) {
				sending_data->icon = gtk_image_new_from_pixbuf((&g_array_index(icon_data_array, IconData, i))->icon->pixbuf);
				gtk_box_pack_start(GTK_BOX(sending_data->hbox), sending_data->icon, FALSE, FALSE, 0);
				gtk_widget_show(sending_data->icon);
				break;
			}
		}
	}
	
	g_array_free(icon_data_array, TRUE);

	sending_data->icon_id = id;		
}

static void on_add_comment_clicked(GtkButton* button, SendingData* sending_data) {
	GtkTextIter start_iter;
	GtkTextIter end_iter;
	const gchar* text;
	GtkWidget* note;
	gchar* s;
	
	gtk_text_buffer_get_start_iter(sending_data->text_buffer, &start_iter);
	gtk_text_buffer_get_end_iter(sending_data->text_buffer, &end_iter);
	text = gtk_text_buffer_get_text(sending_data->text_buffer, &start_iter, &end_iter, FALSE);	
	if (strlen(text) > 0) {
		if (!jaiku_send_comment(view_get_unique_id(sending_data->view), text)) {
			note = hildon_note_new_information(NULL, "Could not send a comment.");
			gtk_dialog_run(GTK_DIALOG(note));
			gtk_widget_destroy(note);
		} else {
			view_start_update(sending_data->view, FALSE);
		}
	}
	
	view_set_top_bottom_widget(sending_data->view, NULL, FALSE);
}


static gboolean on_icon_button_press(GtkWidget* widget, GdkEventButton* event, IconData* data) {
	GdkColor color;

	color.red = 0xffff;
	color.green = color.blue = 0xeeee;
	gdk_colormap_alloc_color(gdk_colormap_get_system(), &color, FALSE, TRUE);
	gtk_widget_modify_bg(data->event_box, GTK_STATE_NORMAL, &color);

	return FALSE;
}

static gboolean on_icon_motion_notify(GtkWidget* widget, GdkEventMotion* event, IconData* data) {
	GdkColor color;

	if (miaouw_scrolled_window_is_dragged(MIAOUW_SCROLLED_WINDOW(data->scrolled_window))) {
		color.red = color.green = color.blue = 0xffff;
		gdk_colormap_alloc_color(gdk_colormap_get_system(), &color, FALSE, TRUE);
		gtk_widget_modify_bg(data->event_box, GTK_STATE_NORMAL, &color);
	}

	return FALSE;
}

static gboolean on_icon_button_release(GtkWidget* widget, GdkEventButton* event, IconData* data) {
	if (!miaouw_scrolled_window_is_dragged(MIAOUW_SCROLLED_WINDOW(data->scrolled_window))) {
		id = data->icon->id;
		gtk_widget_destroy(data->dialog);
	}
	
	return FALSE;
}

static void on_send_ok_clicked(SendingData* sending_data) {
	gboolean should_update = FALSE;
	GtkTextIter start_iter;
	GtkTextIter end_iter;
	GString* string;
	const gchar* text;
	MicrofeedHttp* http;
	GtkWidget* note;
	gchar* s;
	gchar buffer[1024];
	gchar* nonce;
	gchar* location;
	JsonObject* object;
	
	gtk_text_buffer_get_start_iter(sending_data->text_buffer, &start_iter);
	gtk_text_buffer_get_end_iter(sending_data->text_buffer, &end_iter);
	text = gtk_text_buffer_get_text(sending_data->text_buffer, &start_iter, &end_iter, FALSE);
	if (strlen(text) > 0) {
		if (sending_data->nick) {
			if (!twitter_send(text)) {
				note = hildon_note_new_information(NULL, "Could not send a new Tweet.");
				gtk_dialog_run(GTK_DIALOG(note));
				gtk_widget_destroy(note);
			} else {
				should_update = FALSE;
			}						
		} else if (sending_data->channel) {
			if (!jaiku_send_jaiku(sending_data->channel, text, sending_data->icon_id)) {
				note = hildon_note_new_information(NULL, "Could not send a new Jaiku.");
				gtk_dialog_run(GTK_DIALOG(note));
				gtk_widget_destroy(note);
			} else {
				should_update = TRUE;
			}
		} else {
			if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sending_data->jaiku_check_button))) {
				if (!jaiku_send_jaiku(NULL, text, sending_data->icon_id)) {
					note = hildon_note_new_information(NULL, "Could not send a new Jaiku.");
					gtk_dialog_run(GTK_DIALOG(note));
					gtk_widget_destroy(note);
				} else {
					should_update = TRUE;
				}
			}
			if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sending_data->twitter_check_button))) {
				if (!twitter_send(text)) {
					note = hildon_note_new_information(NULL, "Could not send a new Tweet.");
					gtk_dialog_run(GTK_DIALOG(note));
					gtk_widget_destroy(note);
				} else {
					should_update = TRUE;
				}			
			}
		}
	}
	
	if (should_update) {
		view_start_update(sending_data->view, FALSE);
	}
	view_set_top_bottom_widget(sending_data->view, NULL, FALSE);
}

static void remove_top_bottom_widget(SendingData* sending_data) {
	view_set_top_bottom_widget(sending_data->view, NULL, FALSE);
}

static void update_labels(SendingData* sending_data) {
	char buffer[1024];
	gint count;
	GtkTextIter start_iter;
	GtkTextIter end_iter;
	gchar* nick;
	
	count = gtk_text_buffer_get_char_count(sending_data->text_buffer);
	if (sending_data->chars_label) {
		snprintf(buffer, 1024, "%d characters left.", 140 - count);
		gtk_label_set_text(GTK_LABEL(sending_data->chars_label), buffer);
		if (sending_data->jaiku_check_button) {
			gtk_widget_set_sensitive(sending_data->ok_button,
		                        	 count > 0 && count <= 140 &&
						 (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sending_data->jaiku_check_button)) ||
						  gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sending_data->twitter_check_button))));
		} else {
			gtk_widget_set_sensitive(sending_data->ok_button,
		                        	 count > 0 && count <= 140);
		}
	} else {
		gtk_widget_set_sensitive(sending_data->ok_button, count > 0);
	}
	
	if (sending_data->jaiku_check_button) {
		gtk_widget_set_sensitive(sending_data->icon_button,
		                         gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sending_data->jaiku_check_button)));
	}
	
	if (sending_data->nick) {
		gtk_text_buffer_get_iter_at_mark(sending_data->text_buffer, &start_iter, sending_data->start_mark);
		gtk_text_buffer_get_iter_at_mark(sending_data->text_buffer, &end_iter, sending_data->end_mark);
		if (strcmp(gtk_text_buffer_get_text(sending_data->text_buffer, &start_iter, &end_iter, FALSE), sending_data->nick)) {
			nick = sending_data->nick;
			sending_data->nick = NULL;
			gtk_text_buffer_delete(sending_data->text_buffer, &start_iter, &end_iter);
			gtk_text_buffer_get_iter_at_mark(sending_data->text_buffer, &start_iter, sending_data->start_mark);
			gtk_text_buffer_get_iter_at_mark(sending_data->text_buffer, &end_iter, sending_data->end_mark);
			gtk_text_buffer_insert(sending_data->text_buffer, &start_iter, nick, -1);
			gtk_text_buffer_delete_mark(sending_data->text_buffer, sending_data->end_mark);
			sending_data->end_mark = gtk_text_buffer_create_mark(sending_data->text_buffer, NULL, &start_iter, TRUE);
			sending_data->nick = nick;
		}
	}
}
