/*****************************************************************************
 *** Mauku - Jaiku client for Maemo devices 
 ***
 *** GeoGlue support
 ***
 *** Copyright (c) 2007 Jussi Kukkonen <jhkukkon@cc.hut.fi>
 ***
 *** Licensed under the Apache License, Version 2.0 (the "License");
 *** you may not use this file except in compliance with the License.
 *** You may obtain a copy of the License at
 ***
 ***     http://www.apache.org/licenses/LICENSE-2.0
 ***
 *** Unless required by applicable law or agreed to in writing, software
 *** distributed under the License is distributed on an "AS IS" BASIS,
 *** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *** See the License for the specific language governing permissions and
 *** limitations under the License.
 ***
 *****************************************************************************/

#include "config.h"

#ifndef HAVE_GEOCLUE
/* stubs for compiling without geoclue */
void geoclue_init() {}
void geoclue_cleanup() {}
#else

#include <string.h> /* for strlen */

#include <geoclue/position.h>

#include "mauku.h" 

static position_provider *geoclue; /* current geoclue position provider */

/* use the hostip backend */
#define GEOCLUE_BACKEND_SERVICE "org.freedesktop.geoclue.position.hostip"
#define GEOCLUE_BACKEND_PATH "/org/freedesktop/geoclue/position/hostip"

/* these files will be checked to see if geoclue is available on runtime */
#define GEOCLUE_LIB "/usr/lib/libgeoclue.so.0"
#define GEOCLUE_HOSTIP_BIN "/usr/bin/geoclue-position-server-hostip"


/* handler for geoclue civic location changed -signal */
static void geoclue_location_callback (void* userdata, char* country, char* region, char* locality, char* area, char* postalcode,
                                       char* street, char* building, char* floor, char* room, char* description, char* text) {
	char *location = NULL;
	gboolean use_country, use_locality;
	
	use_country = country && (strlen (country) > 0);
	use_locality = locality && (strlen (locality) > 0);

	if (use_country && use_locality) {
		location = g_strdup_printf ("%s, %s", locality, country);
	} else if (use_country) {
		location = g_strdup_printf ("%s", country);
	} else if (use_locality) {
		location = g_strdup_printf ("%s", locality);
	}

	/* if no location is available let the old one stay */
	if (location) {
		fprintf(stdout, "Geoclue: setting new location: %s\n", location);
		jaiku_set_location (location);
	}

	g_free (location);
}

void geoclue_init() {
	/* check for libgeoclue and geoclue hostip backend existence */
	if (!g_file_test(GEOCLUE_LIB, G_FILE_TEST_EXISTS) ||
	    !g_file_test(GEOCLUE_HOSTIP_BIN, G_FILE_TEST_EXISTS)) {
		fprintf(stdout, "libgeoclue0 or hostip backend not found: Geoclue is not in use.\n");
		return;
	}

	/* Initialize backend */
	geoclue = g_malloc(sizeof(position_provider));
	geoclue->service = GEOCLUE_BACKEND_SERVICE;
	geoclue->path = GEOCLUE_BACKEND_PATH;
	geoclue->connection = NULL;
	geoclue->proxy = NULL;
	geoclue->ref = 1;
	
	if (geoclue_position_init(geoclue) != GEOCLUE_POSITION_SUCCESS){
		fprintf(stderr, "Geoclue init failed\n");
		return;
	}
	if (geoclue_position_add_civic_callback(geoclue, 
	                                        (civic_callback)geoclue_location_callback, 
	                                        NULL) != GEOCLUE_POSITION_SUCCESS) {
		fprintf(stderr, "Geoclue callback-set failed\n");
		return;
	}
	

	/* Get civic location from geoclue and pretend a location changed signal 
	   (from the backends POV current location is not new so it does not emit signal, 
	   but from ours it probably is). This is a result of not having a proper master 
	   server in geoclue */
	
	char *country = NULL;
	char *region = NULL;
	char *locality = NULL;
	char *area = NULL;
	char *postalcode = NULL;
	char *street = NULL;
	char *building = NULL;
	char *floor = NULL;
	char *room = NULL;
	char *description = NULL;
	char *text = NULL;

	/* FIXME: This is a synchronous call (possibly doing a http fetch -- may take a while) */
	if (geoclue_position_civic_location(geoclue,
	                                    &country, &region, &locality, &area, &postalcode, &street,
	                                    &building, &floor, &description, &room, &text) == GEOCLUE_POSITION_SUCCESS) {
		/* call the callback manually */
		geoclue_location_callback(NULL, 
		                          country, region, locality, area, postalcode,
		                          street, building,floor, room, description, text);
	}
	g_free (country);
	g_free (region);
	g_free (locality);
	g_free (area);
	g_free (postalcode);
	g_free (street);
	g_free (building);
	g_free (floor);
	g_free (room);
	g_free (description);
	g_free (text);

	fprintf(stdout, "Geoclue Initialized\n");
}

void geoclue_cleanup() {
	if (geoclue) {
		if (geoclue_position_remove_civic_callback(geoclue, 
		                                           (civic_callback)geoclue_location_callback, 
		                                           NULL) != GEOCLUE_POSITION_SUCCESS) {
			fprintf(stderr, "Geoclue callback remove failed\n");
		}
		if (geoclue_position_close(geoclue) != GEOCLUE_POSITION_SUCCESS) {
			fprintf(stderr, "Geoclue close failed\n");
		}
	}
}

#endif
