/*
  MathJinni - A simple formular calculator
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "UnitConversion.h"

#include <Lum/Base/String.h>

#include <Lum/Combo.h>
#include <Lum/Label.h>
#include <Lum/String.h>
#include <Lum/TextValue.h>

#include "Util.h"

UnitConversion::UnitConversion()
 : type(new Lum::Model::SizeT()),
   types(new Lum::Model::StringTable()),
   src(new Lum::Model::SizeT()),
   srcUnits(new Lum::Model::StringTable()),
   dest(new Lum::Model::SizeT()),
   destUnits(new Lum::Model::StringTable()),
   value(new Lum::Model::String(L"")),
   result(new Lum::Model::String(L""))
{
  Observe(type);
  Observe(src);
  Observe(dest);
  Observe(value);
}

void UnitConversion::CalcSize()
{
  Lum::Combo               *combo;
  Lum::Label               *label;
  Lum::String              *string;
  Lum::TextValue           *textValue;

  label=new Lum::Label();

  combo=new Lum::IndexCombo();
  combo->SetFlex(true,false);
  combo->SetModel(type);
  combo->SetTableModel(types);
  label->AddLabel(L"Type:",combo);

  combo=new Lum::IndexCombo();
  combo->SetFlex(true,false);
  combo->SetModel(src);
  combo->SetTableModel(srcUnits);
  label->AddLabel(L"Source Unit:",combo);

  combo=new Lum::IndexCombo();
  combo->SetFlex(true,false);
  combo->SetModel(dest);
  combo->SetTableModel(destUnits);
  label->AddLabel(L"Destination Unit:",combo);

  string=new Lum::String();
  string->SetFlex(true,false);
  string->SetAlignment(Lum::String::right);
  string->SetInputHelper(new Lum::InputHelperNumber(Lum::InputHelperNumber::typeFloat));
  string->SetModel(value);
  label->AddLabel(L"Value:",string);

  textValue=new Lum::TextValue();
  textValue->SetFlex(true,false);
  textValue->SetAlignment(Lum::TextValue::right);
  textValue->SetModel(result);
  label->AddLabel(L"Result:",textValue);

  container=label;

  Component::CalcSize();
}

void UnitConversion::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==type && !type->IsNull()) {
    srcUnits->Off();
    srcUnits->Clear();
    for (size_t i=0; i<units[type->Get()-1]->GetAlternativeCount(); i++) {
      srcUnits->Append(units[type->Get()-1]->GetAlternative(i)->GetName());
    }
    srcUnits->On();

    destUnits->Off();
    destUnits->Clear();
    for (size_t i=0; i<units[type->Get()-1]->GetAlternativeCount(); i++) {
      destUnits->Append(units[type->Get()-1]->GetAlternative(i)->GetName());
    }
    destUnits->On();

    src->Set(1);
    dest->Set(1);

    Convert();
  }
  else if (model==value || model==src || model==dest) {
    Convert();
  }
}

void UnitConversion::LoadConfig(Lum::Config::Node *top)
{
  for (size_t i=0; i<units.size(); i++) {
    types->Append(units[i]->GetType());
  }

  if (units.size()>0) {
    type->Set(1);
  }
}

void UnitConversion::StoreConfig(Lum::Config::Node *top)
{
  // no code
}

void UnitConversion::Convert()
{
  if (value->IsNull() || value->Empty()) {
    result->Set(L"");
    return;
  }

  Parser::Expression *expression;
  Parser::ReturnCode returnCode;
  size_t             pos;
  Alternative        *srcAlt=units[type->Get()-1]->GetAlternative(src->Get()-1);
  Alternative        *destAlt=units[type->Get()-1]->GetAlternative(dest->Get()-1);
  double             srcFactor;
  double             destFactor;
  double             res;

  expression=parser.Parse(Lum::Base::WStringToString(srcAlt->GetFactor()),returnCode,pos,false);
  if (expression==NULL) {
    result->Set(L"Internal error: src factor has parse error (code "+Lum::Base::NumberToWString((size_t)returnCode)+L")");

    return;
  }

  if (!expression->Calculate(srcFactor,returnCode)) {
    result->Set(L"Internal error: src factor has calculation error (code "+Lum::Base::NumberToWString((size_t)returnCode)+L")");
    delete expression;
    return;
  }

  delete expression;

  expression=parser.Parse(Lum::Base::WStringToString(destAlt->GetFactor()),returnCode,pos,false);
  if (expression==NULL) {
    result->Set(L"Internal error: dest factor has parse error (code "+Lum::Base::NumberToWString((size_t)returnCode)+L")");
    return;
  }

  if (!expression->Calculate(destFactor,returnCode)) {
    result->Set(L"Internal error: dest factor has calculation error (code "+Lum::Base::NumberToWString((size_t)returnCode)+L")");
    delete expression;
    return;
  }

  delete expression;

  expression=parser.Parse(Lum::Base::WStringToString(value->Get()),returnCode,pos);
  if (expression==NULL) {
    result->Set(L"<ERROR>");
    return;
  }

  if (!expression->Calculate(res,returnCode)) {
    result->Set(L"<ERROR>");
    delete expression;
    return;
  }

  res=(destFactor*res)/srcFactor;

  result->Set(DoubleToWStringFloat(res));

  delete expression;
}
