/*
  MathJinni - A simple formular calculator
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Parser.h"

#include <locale.h>
#include <iostream>
/**
  The mathematical functions of the parser. They directly work on the stack.
*/

void PushVar(Parser::Stack& stack, Parser::Variable *variable)
{
  stack.pos++;
  stack.stack[stack.pos]=variable->value;
}

void Add(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.pos--;
  stack.stack[stack.pos]=stack.stack[stack.pos]+stack.stack[stack.pos+1];
}

void Min(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.pos--;
  stack.stack[stack.pos]=stack.stack[stack.pos]-stack.stack[stack.pos+1];
}

void Mul(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.pos--;
  stack.stack[stack.pos]=stack.stack[stack.pos]*stack.stack[stack.pos+1];
}

void Div(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.pos--;

  if (stack.stack[stack.pos+1]==0) {
    stack.error=Parser::DIVBYZERO;
  }

  stack.stack[stack.pos]=stack.stack[stack.pos]/stack.stack[stack.pos+1];
}

void Neg(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.stack[stack.pos]=-stack.stack[stack.pos];
}

void Abs(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.stack[stack.pos]=fabs(stack.stack[stack.pos]);
}

void Pow(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.pos--;
  stack.stack[stack.pos]=pow(stack.stack[stack.pos],stack.stack[stack.pos+1]);
}

void Sin(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.stack[stack.pos]=sin(stack.stack[stack.pos]);
}

void ASin(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.stack[stack.pos]=asin(stack.stack[stack.pos]);
}

void Cos(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.stack[stack.pos]=cos(stack.stack[stack.pos]);
}

void ACos(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.stack[stack.pos]=acos(stack.stack[stack.pos]);
}

void Tan(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.stack[stack.pos]=tan(stack.stack[stack.pos]);
}

void ATan(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.stack[stack.pos]=atan(stack.stack[stack.pos]);
}

void Sinh(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.stack[stack.pos]=sinh(stack.stack[stack.pos]);
}

void ASinh(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.stack[stack.pos]=asinh(stack.stack[stack.pos]);
}

void Cosh(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.stack[stack.pos]=cosh(stack.stack[stack.pos]);
}

void ACosh(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.stack[stack.pos]=acosh(stack.stack[stack.pos]);
}

void Tanh(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.stack[stack.pos]=tanh(stack.stack[stack.pos]);
}

void ATanh(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.stack[stack.pos]=atanh(stack.stack[stack.pos]);
}

void Etox(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  stack.stack[stack.pos]=exp(stack.stack[stack.pos]);
}

/*
(*  PROCEDURE Log(VAR stack : Stack; data : Data);

  BEGIN
    IF stack.stack[stack.pos]>=0 THEN
      stack.stack[stack.pos]:=m.log(stack.stack[stack.pos],10);
    ELSE
      stack.error:=NEGVAL;
    END;
  END Log;*)
*/

void Ln(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  if (stack.stack[stack.pos]<0) {
    stack.error=Parser::NEGVAL;
  }

  stack.stack[stack.pos]=log(stack.stack[stack.pos]);
}

void Lg(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  if (stack.stack[stack.pos]<0) {
    stack.error=Parser::NEGVAL;
  }

  stack.stack[stack.pos]=log10(stack.stack[stack.pos]);
}

void Sqrt(Parser::Stack& stack, Parser::Variable* /* variable */)
{
  if (stack.stack[stack.pos]<0) {
    stack.error=Parser::NEGVAL;
  }

  stack.stack[stack.pos]=sqrt(stack.stack[stack.pos]);
}

/**
 Evaluates the given string, preparsed by the parser, and returs possible errors.
 */
bool Parser::Expression::Calculate(double& value, Parser::ReturnCode& result)
{
  // TODO: check stack size
  size_t x;

  stack.error=OK;
  stack.pos=-1;

  x=0;
  while (x<code.size() && stack.error==OK) {
    code[x].function(stack,code[x].data);

    x++;
  }

  value=stack.stack[0];
  result=stack.error;

  return result==Parser::OK;
}

/**
  The parser creates using 'AddFunktion' and 'AddPushVar' the code and at the
  same time evaluates the size of the stack.
*/
void Parser::Expression::AddFunction(Function *function)
{
  Operation operation;

  operation.function=function->function;
  operation.data=NULL;

  code.push_back(operation);
}

void Parser::Expression::AddPushVar(Variable* variable)
{
  Operation operation;

  operation.function=PushVar;
  operation.data=variable;

  //stack.stackSize++;

  code.push_back(operation);
}

Parser::Stack::Stack()
: pos(0),error(OK)
{
  // no code
}


Parser::Variable::Variable(double value)
 : value(value)
{
  // no code
}

Parser::Function::Function(MathFunct function, TokenType type)
 : function(function),type(type)
{
  // no code
}

Parser::Parser()
{
  struct lconv *l;

  l=localeconv();
  if (l!=NULL && l->decimal_point!=NULL) {
    decimalPoint=l->decimal_point[0];
  }
  else {
    decimalPoint='.';
  }

  AddFunction("+",Add,OP1);           // do not delete!
  AddFunction("-",Min,OP1);

  AddFunction("*",Mul,OP2);           // do not delete!
  AddFunction("/",Div,OP2);

  AddFunction("^",Pow,OP3);

  AddFunction("neg",Neg,OP4);         // do not delete!
  AddFunction("abs",Abs,OP4);
  AddFunction("sin",Sin,OP4);
  AddFunction("asin",ASin,OP4);
  AddFunction("cos",Cos,OP4);
  AddFunction("acos",ACos,OP4);
  AddFunction("tan",Tan,OP4);
  AddFunction("atan",ATan,OP4);
  AddFunction("sinh",Sinh,OP4);
  AddFunction("cosh",Cosh,OP4);
  AddFunction("tanh",Tanh,OP4);
  AddFunction("atanh",ATanh,OP4);
  AddFunction("exp",Etox,OP4);
//  AddFunction("log",Log,OP4);
  AddFunction("ln",Ln,OP4);
  AddFunction("lg",Lg,OP4);
//  AddFunction("sqr",Sqr,OP4);
  AddFunction("sqrt",Sqrt,OP4);

  AddConstant("pi",M_PI);
  AddConstant("e",M_E);

  plus=new Function(Add,OP1);
  minus=new Function(Min,OP1);
  times=new Function(Mul,OP2);
  negate=new Function(Neg,OP4);
  pushVar=new Function(PushVar,OP4);
}

Parser::~Parser()
{
  for (std::map<std::string,Data*>::iterator iter=tokens.begin(); iter!=tokens.end(); ++iter) {
    delete iter->second;
  }
}

Parser::Variable* Parser::MakeNumber(double value)
{
  return new Variable(value);
}

/**
  Add a custom variable to the table of nown identifiers.
*/
void Parser::AddConstant(const std::string& name, double value)
{
  tokens[name]=new Variable(value);
}

/**
  Add a custom variable to the table of nown identifiers.
*/
Parser::Variable* Parser::AddVariable(const std::string& name, double value)
{
  Variable *variable=new Variable(value);

  tokens[name]=variable;

  return variable;
}

/**
  The user can get a pointer to a variable. Changing the value
  (the member "value") he can quickly modify the value of a variable.
*/

Parser::Variable* Parser::GetVariable(const std::string& name)
{
  std::map<std::string,Data*>::iterator iter;

  iter=tokens.find(name);

  if (iter==tokens.end()) {
    return NULL;
  }

  return dynamic_cast<Variable*>(iter->second);
}

/**
  The scanner can get information about a token.
*/

Parser::Data* Parser::GetToken(const std::string& name)
{
  std::map<std::string,Data*>::iterator iter;

  iter=tokens.find(name);

  if (iter==tokens.end()) {
    return NULL;
  }

  return iter->second;
}

/**
  Using this method you can add your own function of the types OP1-OP4.
*/
void Parser::AddFunction(const std::string& name, MathFunct function, TokenType type)
{
  tokens[name]=new Function(function,type);
}

/**
  The scanner.
  Letter = "a".."Z"
  Indent = Letter @{Letter@}
  Digit  = "0".."9"
  Number = @{Digit@} ["." Digit @{Digit@}] ["E" [("+"|"-")] Digit @{Digit@}]
*/
Parser::TokenType Parser::GetToken(const std::string& string,
                                   size_t& pos,
                                   Parser::Data*& data,
                                   Parser::ReturnCode& result,
                                   bool localized)
{
  size_t       state;
  size_t       bufferPos;
  std::string  buffer;
  double       value;

  state=0;
  bufferPos=0;
  data=NULL;

  while (true) {
    switch (state) {
    case 0:
      if (pos>=string.length()) {
        return FINISHED;
      }
      else if (string[pos]=='(') {
        pos++;
        return LBRAC;
      }
      else if (string[pos]==')') {
        pos++;
        return RBRAC;
      }
      else if (string[pos]=='+') {
        pos++;
        data=plus;
        return PLUS;
      }
      else if (string[pos]=='-') {
        pos++;
        data=minus;
        return MINUS;
      }
      else if (string[pos]>='0' && string[pos]<='9') {
        state=1;
      }
      else if ((localized && string[pos]==decimalPoint) || (!localized && string[pos]=='.')) {
        buffer.append(1,decimalPoint);
        pos++;
        state=2;
      }
      else if ((string[pos]>='a' && string[pos]<='z') || (string[pos]>='A' && string[pos]<='Z')) {
        buffer.clear();
        while ((string[pos]>='a' && string[pos]<='z') || (string[pos]>='A' && string[pos]<='Z')) {
          buffer.append(1,string[pos]);
          pos++;
        }

        data=GetToken(buffer);
        if (data==NULL) {
          result=UNKNOWNIDENT;
          return tokenError;
        }
        else if (dynamic_cast<Variable*>(data)!=NULL) {
          return NUMBER;
        }
        else if (dynamic_cast<Function*>(data)!=NULL) {
          return dynamic_cast<Function*>(data)->type;
        }
        else {
          result=UNKNOWN;
          return tokenError;
        }
      }
      else if (string[pos]==' ') {
        pos++;
      }
      else {
        buffer.clear();
        buffer.append(1,string[pos]);
        data=GetToken(buffer);
        pos++;
        if (data==NULL) {
          result=UNKNOWNIDENT;
          return tokenError;
        }
        else if (dynamic_cast<Variable*>(data)) {
          return NUMBER;
        }
        else if (dynamic_cast<Function*>(data)) {
          return dynamic_cast<Function*>(data)->type;
        }
        else {
          result=UNKNOWN;
          return tokenError;
        }
      }
      break;
    // Number
    case 1:
      buffer.clear();
     while (string[pos]>='0' && string[pos]<='9') {
       buffer.append(1,string[pos]);
       pos++;
      }
      if ((localized && string[pos]==decimalPoint) || (!localized && string[pos]=='.')) {
        buffer.append(1,decimalPoint);
        pos++;
        state=2;
      }
      else {
        state=3;
      }
      break;
    // Fraction
    case 2:
     if  (string[pos]>='0' && string[pos]<='9') {
       while (string[pos]>='0' && string[pos]<='9') {
         buffer.append(1,string[pos]);
         pos++;
       }
       state=3;
      }
      else {
        result=NODIGIT;
        return tokenError;
      }
      break;
    // Exponent
    case 3:
      if (string[pos]=='e' || string[pos]=='E') {
        buffer.append(1,'e');
        pos++;
        if (string[pos]=='+' || string[pos]=='-') {
          buffer.append(1,string[pos]);
          pos++;
        }
        if (string[pos]>='0' && string[pos]<='9') {
          while (string[pos]>='0' && string[pos]<='9') {
            buffer.append(1,string[pos]);
            pos++;
          }
        }
        else {
          result=NODIGIT;
          return tokenError;
        }
      }
      else if ((localized && string[pos]==decimalPoint) || (!localized && string[pos]=='.')) {
        result=NODIGIT;
        return tokenError;
      }

      if (sscanf(buffer.c_str(),"%lf",&value)!=1) {
        result=NODIGIT;
        return tokenError;
      }
      data=MakeNumber(value);
      return NUMBER;
      break;
    }
  }
}

void Parser::ParseFactor(const std::string& string,
                         size_t& pos,
                         Data*& data,
                         Parser::ReturnCode& result,
                         Expression *expression,
                         TokenType& token,
                         bool localized)
{
  Data *oldFakt;

  switch (token) {
  case NUMBER:
    expression->AddPushVar(dynamic_cast<Variable*>(data));
    token=GetToken(string,pos,data,result,localized);
    break;
  case LBRAC:
    token=GetToken(string,pos,data,result,localized);
    if (token==tokenError) {
      return;
    }

    ParseExpression(string,pos,data,result,expression,token,localized);

    if (result!=OK) {
      return;
    }

    if (token==RBRAC) {
      token=GetToken(string,pos,data,result,localized);
    }
    else {
      result=NORBRAC;
    }
    break;
  case OP4:
    oldFakt=data;

    token=GetToken(string,pos,data,result,localized);
    if (token==tokenError) {
      return;
    }

    ParseFactor(string,pos,data,result,expression,token,localized);

    if (result!=OK) {
      return;
    }

    expression->AddFunction(dynamic_cast<Function*>(oldFakt));
    break;
  case RBRAC:
    result=WRONGBRAC;
    break;
  case tokenError:
    // Use result from parser
    break;
  default:
    result=UNKNOWN;
    break;
  }
}

void Parser::ParseBigTerm(const std::string& string,
                          size_t& pos,
                          Data*& data,
                          Parser::ReturnCode& result,
                          Expression *expression,
                          TokenType& token,
                          bool localized)
{
  Data *oldBig;

  ParseFactor(string,pos,data,result,expression,token,localized);

  if (result!=OK) {
    return;
  }

  while (token==OP3) {
    oldBig=data;

    token=GetToken(string,pos,data,result,localized);
    if (token==tokenError) {
      return;
    }

    ParseBigTerm(string,pos,data,result,expression,token,localized);

    if (result!=OK) {
      return;
    }

    expression->AddFunction(dynamic_cast<Function*>(oldBig));
  }
}

void Parser::ParseTerm(const std::string& string,
                       size_t& pos,
                       Data*& data,
                       Parser::ReturnCode& result,
                       Expression *expression,
                       TokenType& token,
                       bool localized)
{
  TokenType old;
  Data      *oldTerm;

  ParseBigTerm(string,pos,data,result,expression,token,localized);

  if (result!=OK) {
    return;
  }

  while (token==OP2 || token==NUMBER) {
    old=token;
    oldTerm=data;
    if (token==OP2) {
      token=GetToken(string,pos,data,result,localized);
      if (token==tokenError) {
        return;
      }
    }

    ParseBigTerm(string,pos,data,result,expression,token,localized);

    if (result!=OK) {
      return;
    }

    if (old==OP2) {
      expression->AddFunction(dynamic_cast<Function*>(oldTerm));
    }
    else {
      // Add "*" manually
      expression->AddFunction(times);
    }
  }
}

void Parser::ParseExpression(const std::string& string,
                             size_t& pos,
                             Data*& data,
                             Parser::ReturnCode& result,
                             Expression *expression,
                             TokenType& token,
                             bool localized)
{
  bool   neg;
  size_t old;
  Data*  oldExp;

  neg=false;
  oldExp=NULL;

  if (token==PLUS || token==MINUS) {
    if (token==MINUS) {
      neg=true;
    }
    token=GetToken(string,pos,data,result,localized);
    if (token==tokenError) {
      return;
    }
  }

  ParseTerm(string,pos,data,result,expression,token,localized);

  if (result!=OK) {
    return;
  }

  if (neg) {
    expression->AddFunction(negate);
  }

  while (token==PLUS || token==MINUS || token==OP1) {
    old=token;
    oldExp=data;

    token=GetToken(string,pos,data,result,localized);
    if (token==tokenError) {
      return;
    }

    ParseTerm(string,pos,data,result,expression,token,localized);

    if (result!=OK) {
      return;
    }

    if(old==PLUS) {
      expression->AddFunction(plus);
    }
    else if (old==MINUS) {
      expression->AddFunction(minus);
    }
    else {
      expression->AddFunction(dynamic_cast<Function*>(oldExp));
    }
  }
}

/**
  EBNF:

  Expression  = [ "+" | "-" ] Term [ AddOperator Term ].

  AddOperator = "+" | "-" | OP1.

  Term        = BigTerm @{ [MulOperator] BigTerm @}.

  MulOperator = "*" | "/" | OP2.

  BigTerm     = Factor @{ PotOperator BigTerm @}.

  PotOperator = "^" | OP3.

  Factor      = Number | "(" Expression ")" | Designator [ "(" BigTerm ")" ].


  - OP1-OP4 can be a operator declared by user.

  - The multiplication operator can be left out. A * will be automatically inserted
    where approximate. But you must write X X and not XX!
*/
Parser::Expression* Parser::Parse(const std::string& string,
                                  Parser::ReturnCode& result,
                                  size_t& pos,
                                  bool localized)
{
  Expression *expression;
  TokenType  token;
  Data       *data;

  pos=0;
  result=OK;

  expression=new Expression();

  token=GetToken(string,pos,data,result,localized);

  if (token==FINISHED) {
    result=NOOP;
  }
  else if (token!=tokenError) {
    ParseExpression(string,pos,data,result,expression,token,localized);

    if (result==OK && token!=FINISHED) {
      if (token==RBRAC) {
        result=WRONGBRAC;
      }
      else {
        result=UNKNOWN;
      }
    }
  }

  if (result==OK) {
    return expression;
  }
  else {
    delete expression;
    return NULL;
  }
}
