/*
  MathJinni - A simple formular calculator
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "FunctionInput.h"

#include <Lum/Base/String.h>

#include <Lum/Button.h>
#include <Lum/Grid.h>
#include <Lum/Panel.h>
#include <Lum/Tab.h>
#include <Lum/Table.h>

FunctionInput::FunctionInput()
 : constants(new ShortcutsModel(new ShortcutsDataProvider())),
   constantSelection(new Lum::Model::SingleLineSelection()),
   constantAction(new Lum::Model::Action()),
   functions(new ShortcutsModel(new ShortcutsDataProvider())),
   functionSelection(new Lum::Model::SingleLineSelection()),
   functionAction(new Lum::Model::Action()),
   trigometrics(new ShortcutsModel(new ShortcutsDataProvider())),
   trigometricSelection(new Lum::Model::SingleLineSelection()),
   trigometricAction(new Lum::Model::Action()),
   backspaceAction(new Lum::Model::Action()),
   clearAction(new Lum::Model::Action())
{
  constants->Append(Shortcut(L"\x03c0"/*L"pi"*/,L"pi"));
  constants->Append(Shortcut(L"e"));

  functions->Append(Shortcut(L"^"));
  functions->Append(Shortcut(L"ln",L"ln("));
  functions->Append(Shortcut(L"lg",L"lg("));

  functions->Append(Shortcut(L"\x221a"/*L"sqrt"*/,L"sqrt("));
  functions->Append(Shortcut(L"abs",L"abs("));

  trigometrics->Append(Shortcut(L"sin",L"sin("));
  trigometrics->Append(Shortcut(L"cos",L"cos("));
  trigometrics->Append(Shortcut(L"tan",L"tan("));

  trigometrics->Append(Shortcut(L"asin",L"asin("));
  trigometrics->Append(Shortcut(L"acos",L"acos("));
  trigometrics->Append(Shortcut(L"atan",L"atan("));

  trigometrics->Append(Shortcut(L"sinh",L"sinh("));
  trigometrics->Append(Shortcut(L"cosh",L"cosh"));
  trigometrics->Append(Shortcut(L"tanh",L"tanh"));

  trigometrics->Append(Shortcut(L"asinh",L"asinh("));
  trigometrics->Append(Shortcut(L"acosh",L"acosh("));
  trigometrics->Append(Shortcut(L"atanh",L"atanh("));

  Observe(constantAction);
  Observe(functionAction);
  Observe(trigometricAction);
  Observe(backspaceAction);
  Observe(clearAction);
}

void FunctionInput::SetString(Lum::String *string)
{
  this->string=string;
}

void FunctionInput::SetCalculateAction(Lum::Model::Action* action)
{
  calculateAction=action;
}

Lum::Object* FunctionInput::CreateButton(const std::wstring& label, Lum::Model::Action *action) const
{
  return Lum::Button::Create(label,action,true,true);
}

Lum::Object* FunctionInput::CreateTextButton(const std::wstring& label, const std::wstring& text)
{
  Lum::Model::ActionRef action;

  action=new Lum::Model::Action();

  Observe(action);

  if (text.empty()) {
    actionTextMap[action.Get()]=label;
  }
  else {
    actionTextMap[action.Get()]=text;
  }

  return CreateButton(label,action);
}

void FunctionInput::CalcSize()
{
  Lum::Grid   *grid;
  Lum::HPanel *panel;
  Lum::Tab    *tab;
  Lum::Table  *table;

  panel=Lum::HPanel::Create(true,true);

  tab=Lum::Tab::Create(true,true);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetMinWidth(Lum::Base::Size::stdCharWidth,20);
  table->GetTableView()->SetMinHeight(Lum::Base::Size::stdCharHeight,6);
  table->SetModel(constants);
  table->SetSelection(constantSelection);
  table->SetDoubleClickAction(constantAction);
  tab->Add(L"Const.",table);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetMinWidth(Lum::Base::Size::stdCharWidth,20);
  table->GetTableView()->SetMinHeight(Lum::Base::Size::stdCharHeight,6);
  table->SetModel(functions);
  table->SetSelection(functionSelection);
  table->SetDoubleClickAction(functionAction);
  tab->Add(L"F(x)",table);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetMinWidth(Lum::Base::Size::stdCharWidth,20);
  table->GetTableView()->SetMinHeight(Lum::Base::Size::stdCharHeight,6);
  table->SetModel(trigometrics);
  table->SetSelection(trigometricSelection);
  table->SetDoubleClickAction(trigometricAction);
  tab->Add(L"Trig.",table);

  panel->Add(tab);

  panel->AddSpace(true);

  grid=new Lum::Grid();
  grid->SetFlex(true,true);
  grid->SetSize(5,4);
  grid->SetEqualDimensions(true,true);
  grid->SetSpace(true,true);

  grid->SetObject(0,0,CreateTextButton(L"7"));
  grid->SetObject(1,0,CreateTextButton(L"8"));
  grid->SetObject(2,0,CreateTextButton(L"9"));
  grid->SetObject(3,0,CreateTextButton(L"/"));
  grid->SetObject(4,0,CreateButton(L"<-",backspaceAction));

  grid->SetObject(0,1,CreateTextButton(L"4"));
  grid->SetObject(1,1,CreateTextButton(L"5"));
  grid->SetObject(2,1,CreateTextButton(L"6"));
  grid->SetObject(3,1,CreateTextButton(L"*"));
  grid->SetObject(4,1,CreateButton(L"Clr",clearAction));

  grid->SetObject(0,2,CreateTextButton(L"1"));
  grid->SetObject(1,2,CreateTextButton(L"2"));
  grid->SetObject(2,2,CreateTextButton(L"3"));
  grid->SetObject(3,2,CreateTextButton(L"\x2212",L"-"));
  grid->SetObject(4,2,CreateTextButton(L"("));

  grid->SetObject(0,3,CreateTextButton(L"0"));
  grid->SetObject(1,3,CreateTextButton(Lum::Base::GetDecimalPointWString()));
  if (calculateAction.Valid()) {
    grid->SetObject(2,3,CreateButton(L"=",calculateAction));
  }
  else {
    grid->SetObject(2,3,CreateTextButton(L"x"));
  }
  grid->SetObject(3,3,CreateTextButton(L"+"));
  grid->SetObject(4,3,CreateTextButton(L")"));

  panel->Add(grid);

  container=panel;

  Component::CalcSize();
}

void FunctionInput::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==constantAction &&
      constantAction->IsFinished() &&
      constantSelection->HasSelection()) {
    string->ActionInsertString(constants->GetEntry(constantSelection->GetLine()).value);
  }
  else if (model==functionAction &&
           functionAction->IsFinished() &&
           functionSelection->HasSelection()) {
    string->ActionInsertString(functions->GetEntry(functionSelection->GetLine()).value);
  }
  else if (model==trigometricAction &&
           trigometricAction->IsFinished() &&
           trigometricSelection->HasSelection()) {
    string->ActionInsertString(trigometrics->GetEntry(trigometricSelection->GetLine()).value);
  }
  else if (model==backspaceAction && backspaceAction->IsFinished()) {
    string->ActionBackspace();
  }
  else if (model==clearAction && clearAction->IsFinished()) {
    string->ActionEraseAll();
  }
  else if (dynamic_cast<Lum::Model::Action*>(model)!=NULL &&
           dynamic_cast<Lum::Model::Action*>(model)->IsFinished()) {
    std::map<Lum::Model::Action*,std::wstring>::const_iterator iter;

    iter=actionTextMap.find(dynamic_cast<Lum::Model::Action*>(model));
    if (iter!=actionTextMap.end()) {
      string->ActionInsertString(iter->second);
    }
  }

  Component::Resync(model,msg);
}

