/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 * Copyright (C) 2006-2007 John Costigan.
 *
 * POI and GPS-Info code originally written by Cezary Jackiewicz.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>
#include <gtk/gtk.h>
#include <libgnomevfs/gnome-vfs.h>

#include "hildon-mapper.h"

#include "path.h"
#include "utils.h"
#include "mapper-types.h"
#include "settings.h"
#include "ui-common.h"
#include "hildon-wrappers.h"
#include "file.h"

/**
 * Open a file for reading.
 */
gboolean
file_open(gchar **dir, gchar **new_file)
{
GtkWidget *dialog;
gboolean success=FALSE;
GtkFileFilter *filter;

#ifdef WITH_HILDON
dialog = hildon_file_chooser_dialog_new(GTK_WINDOW(mapp.mainwindow), GTK_FILE_CHOOSER_ACTION_OPEN);
#else
dialog = gtk_file_chooser_dialog_new("Open...", GTK_WINDOW(mapp.mainwindow), GTK_FILE_CHOOSER_ACTION_OPEN,
						GTK_STOCK_CANCEL,
						GTK_RESPONSE_CANCEL,
						GTK_STOCK_OPEN,
						GTK_RESPONSE_OK, NULL);
#endif

/* Show .gpx files only */
filter=gtk_file_filter_new();
gtk_file_filter_add_pattern(filter, "*.gpx");
gtk_file_filter_set_name(filter, "GPX");
gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog), filter);

if (dir && *dir) {
	gtk_file_chooser_set_current_folder_uri(GTK_FILE_CHOOSER(dialog), *dir);
} else {
	const gchar *home;

	home=g_get_home_dir();
	if (home)
		gtk_file_chooser_set_current_folder_uri(GTK_FILE_CHOOSER(dialog), home);
}

*new_file=NULL;

if (gtk_dialog_run(GTK_DIALOG(dialog))==GTK_RESPONSE_OK) {
		/* Get the selected filename. */
		*new_file = gtk_file_chooser_get_uri(GTK_FILE_CHOOSER(dialog));
		success=TRUE;
}

gtk_widget_destroy(dialog);
return success;
}

/**
 * Open a file for writing. Return a GnomeVFSHandle to the file for writing.
 */
gboolean
file_save(gchar **dir, gchar **file, GnomeVFSHandle **vfs_handle)
{
GtkWidget *dialog;
gboolean success=FALSE;

#ifdef WITH_HILDON
dialog = hildon_file_chooser_dialog_new(GTK_WINDOW(mapp.mainwindow), GTK_FILE_CHOOSER_ACTION_SAVE);
#else
dialog = gtk_file_chooser_dialog_new("Save...", GTK_WINDOW(mapp.mainwindow), GTK_FILE_CHOOSER_ACTION_SAVE,
						     GTK_STOCK_CANCEL,
						     GTK_RESPONSE_CANCEL,
						     GTK_STOCK_SAVE,
						     GTK_RESPONSE_OK, NULL);
gtk_file_chooser_set_do_overwrite_confirmation(GTK_FILE_CHOOSER(dialog), TRUE);
#endif

if (dir && *dir) {
	gtk_file_chooser_set_current_folder_uri(GTK_FILE_CHOOSER(dialog), *dir);
} else {
	const gchar *home;

	home=g_get_home_dir();
	if (home)
		gtk_file_chooser_set_current_folder_uri(GTK_FILE_CHOOSER(dialog), home);
}

if (file && *file) {
	gtk_file_chooser_set_uri(GTK_FILE_CHOOSER(dialog), *file);
	if (strlen(*file)==0)
		gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), "track.gpx");
} else
	gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), "track.gpx");

if (gtk_dialog_run(GTK_DIALOG(dialog))==GTK_RESPONSE_OK) {
	gchar *file_uri_str;
	GnomeVFSResult vfs_result;

	file_uri_str = gtk_file_chooser_get_uri(GTK_FILE_CHOOSER(dialog));
	vfs_result = gnome_vfs_create(vfs_handle, file_uri_str, GNOME_VFS_OPEN_WRITE, FALSE, 0664);
	g_free(file_uri_str);

	if (vfs_result!=GNOME_VFS_OK) {
		success=FALSE;
	} else {
		if (dir) {
			g_free(*dir);
			*dir = gtk_file_chooser_get_current_folder_uri(GTK_FILE_CHOOSER(dialog));
		}
		if (file) {
			g_free(*file);
			*file = gtk_file_chooser_get_uri(GTK_FILE_CHOOSER(dialog));
		}
		success=TRUE;
	}
}

gtk_widget_destroy(dialog);
return success;
}

gboolean
file_open_get_contents(gchar **dir, gchar **buffer, guint *bytes)
{
gchar *file_uri_str;
GnomeVFSResult vfs_result;

if (!file_open(dir, &file_uri_str)) {
	buffer=NULL;
	return FALSE;
}

vfs_result=gnome_vfs_read_entire_file(file_uri_str, bytes, buffer);
if (vfs_result!=GNOME_VFS_OK)
	return FALSE;

return TRUE;
}
