/*
 * This file is part of mapper
 *
 * Copyright (C) 2007-2009 Kaj-Michael Lang
 * Copyright (C) 2006-2007 John Costigan.
 *
 * POI and GPS-Info code originally written by Cezary Jackiewicz.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <locale.h>
#include <errno.h>
#include <sys/wait.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <fcntl.h>
#include <libintl.h>
#include <locale.h>

#include "hildon-mapper.h"

#include "utils.h"
#include "poi.h"
#include "path.h"
#include "route.h"
#include "track.h"
#include "settings.h"
#include "gps.h"
#include "map.h"
#include "map-download.h"
#include "mapper-types.h"
#include "ui-common.h"
#include "db.h"
#include "latlon.h"
#include "cb.h"
#include "poi-gui.h"
#include "gps-panels.h"
#include "gps-conn.h"
#include "search.h"
#include "dialogs.h"
#include "filter-gui.h"
#include "import-gui.h"
#include "map-repo.h"
#include "config-gconf.h"
#include "settings-gui.h"

static gint hb_action=0;

static gboolean
path_tree_view_update_store(GtkWidget *tree_view, Path *path)
{
gboolean r;
g_return_val_if_fail(tree_view, FALSE);
g_return_val_if_fail(path, FALSE);
g_return_val_if_fail(path->store, FALSE);

if ((r=path_update_store(path)))
	gtk_tree_view_set_model(GTK_TREE_VIEW(tree_view), GTK_TREE_MODEL(path->store));
else
	gtk_tree_view_set_model(GTK_TREE_VIEW(tree_view), NULL);
return r;
}

/***********************
  Route menu callbacks 
 ***********************/
gboolean 
menu_cb_route_download(GtkAction * action)
{
route_download();
route_menu_set_sensitive(path_tree_view_update_store(mapp.route_tree_view, _route));
return TRUE;
}

gboolean 
menu_cb_route_open(GtkAction *action)
{
route_open_file();
route_menu_set_sensitive(path_tree_view_update_store(mapp.route_tree_view, _route));
return TRUE;
}

gboolean 
menu_cb_route_save(GtkAction * action)
{
route_save();
return TRUE;
}

gboolean 
menu_cb_route_distnext(GtkAction * action)
{
route_show_distance_to_next();
return TRUE;
}

gboolean 
menu_cb_route_distlast(GtkAction * action)
{
route_show_distance_to_last();
return TRUE;
}

gboolean 
menu_cb_route_reset(GtkAction * action)
{
route_find_nearest_point();
map_force_redraw();
MACRO_QUEUE_DRAW_AREA();
return TRUE;
}

gboolean 
menu_cb_route_clear(GtkAction * action)
{
route_clear();
gtk_tree_view_set_model(GTK_TREE_VIEW(mapp.route_tree_view), NULL);
route_menu_set_sensitive(FALSE);
return TRUE;
}

/***********************
  Track menu callbacks 
 ***********************/
gboolean 
menu_cb_track_open(GtkAction * action)
{
track_open();
path_tree_view_update_store(mapp.track_tree_view, _track);
return TRUE;
}

gboolean 
menu_cb_track_save(GtkAction * action)
{
track_save(_track);
path_tree_view_update_store(mapp.track_tree_view, _track);
return TRUE;
}

gboolean 
menu_cb_track_insert_break(GtkAction * action)
{
track_insert_break(_track);
path_tree_view_update_store(mapp.track_tree_view, _track);
return TRUE;
}

gboolean 
menu_cb_track_insert_mark(GtkAction * action)
{
if (track_insert_mark(_track)) {
	map_render_paths();
	MACRO_QUEUE_DRAW_AREA();
	path_tree_view_update_store(mapp.track_tree_view, _track);
}
return TRUE;
}

gboolean 
menu_cb_track_distlast(GtkAction * action)
{
track_show_distance_from_last(_track);
return TRUE;
}

gboolean 
menu_cb_track_distfirst(GtkAction * action)
{
track_show_distance_from_first(_track);
return TRUE;
}

gboolean 
menu_cb_track_clear(GtkAction * action)
{
track_clear(_track);
path_tree_view_update_store(mapp.track_tree_view, _track);
return TRUE;
}

gboolean 
menu_cb_track_filter(GtkAction * action)
{
filter_dialog(mapp.mainwindow);
return TRUE;
}

/***********************
  Other menu callbacks 
 ***********************/
gboolean 
menu_cb_show_tracks(GtkAction *action)
{
_show_tracks ^= TRACKS_MASK;
if (gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action))) {
	_show_tracks |= TRACKS_MASK;
	map_render_paths();
	MACRO_QUEUE_DRAW_AREA();
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Tracks are now shown"));
} else {
	_show_tracks &= ~TRACKS_MASK;
	map_force_redraw();
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Tracks are now hidden"));
}
return TRUE;
}

gboolean 
menu_cb_show_scale(GtkAction * action)
{
_show_scale=gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));
MACRO_QUEUE_DRAW_AREA();
return TRUE;
}

gboolean 
menu_cb_show_routes(GtkAction * action)
{
if (gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action))) {
	_show_tracks |= ROUTES_MASK;
	map_render_paths();
	MACRO_QUEUE_DRAW_AREA();
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Routes are now shown"));
} else {
	_show_tracks &= ~ROUTES_MASK;
	map_force_redraw();
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Routes are now hidden"));
}
return TRUE;
}

gboolean 
menu_cb_show_velvec(GtkAction * action)
{
_show_velvec=gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));
map_move_mark();
return TRUE;
}

gboolean 
menu_cb_show_poi(GtkAction * action)
{
_show_poi=gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));
map_force_redraw();
return TRUE;
}

gboolean
menu_cb_autocenter(GtkAction *action, GtkRadioAction *current)
{
guint new_center_unitx, new_center_unity;
gint value=gtk_radio_action_get_current_value (GTK_RADIO_ACTION (current));

switch (value) {
	case CENTER_LEAD:
		_center_mode=CENTER_LEAD;
		MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Auto-Center Mode: Lead"));
	break;
	case CENTER_LATLON:
		_center_mode=CENTER_LATLON;
		MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Auto-Center Mode: Lat/Lon"));
	break;
	case CENTER_MANUAL:
	default:
		_center_mode=-_center_mode;
		MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Auto-Center Off"));
		return TRUE;
	break;
}

MACRO_RECALC_CENTER(_gps->data, new_center_unitx, new_center_unity);
map_center_unit(new_center_unitx, new_center_unity);
return TRUE;
}

gboolean 
menu_cb_goto_latlon(GtkAction * action)
{
map_dialog_goto_latlon();
return TRUE;
}

gboolean 
menu_cb_goto_home(GtkAction *action)
{
if (map_goto_position(&_home)==FALSE) {
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Home not set."));
} else {
	if (map_get_zoom()>3)
		map_set_zoom(3);
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("At home location"));
}
return TRUE;
}

gboolean 
menu_cb_goto_destination(GtkAction *action)
{
if (map_goto_position(&_dest)==FALSE) {
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Destination not set."));
} else {
	if (map_get_zoom()>3)
		map_set_zoom(3);
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("At destination"));
}
return TRUE;
}

gboolean 
menu_cb_goto_gps(GtkAction *action)
{
if (_gps->data.fix==FIX_NOFIX) {
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("GPS does not have fix."));
	return TRUE;
}
_center_mode=CENTER_LATLON;
map_center_latlon(_gps->data.lat, _gps->data.lon);
map_update_location_from_center();
if (map_get_zoom()>3)
	map_set_zoom(3);
MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("At GPS coordinates."));
return TRUE;
}

gboolean 
menu_cb_goto_nextway(GtkAction * action)
{
if (_next_way && _next_way->point->unity) {
	if (_center_mode > 0)
		set_action_activate("autocenter_none", TRUE);

	map_center_unit(_next_way->point->unitx, _next_way->point->unity);
} else {
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("There is no next waypoint."));
}

return TRUE;
}

gboolean
cmenu_cb_goto_poi(GtkAction *action, gpointer data)
{
guint32 id;
guint unitx, unity;
poi_info *poi;

id=*(guint32 *)data;

poi=poi_get_by_id(id);
g_return_val_if_fail(poi, TRUE);

map_center_latlon(poi->lat, poi->lon);
map_update_location_from_center();
return TRUE;
}

gboolean 
menu_cb_goto_nearpoi(GtkAction * action)
{
gdouble lat, lon;
poi_info *p;

if (_center_mode > 0) {
	/* Auto-Center is enabled - use the GPS position. */
	lat=_gps->data.lat;
	lon=_gps->data.lon;
} else {
	/* Auto-Center is disabled - use the view center. */
	unit2latlon(_center.unitx, _center.unity, lat, lon);
}

p=poi_find_nearest(lat, lon);

if (p) {
	gchar *banner;

	banner=g_strdup_printf("%s (%s)", p->label, p->cat_label);
	g_printf("%s\n", banner);
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, banner);
	g_free(banner);
	poi_free(p);

	if (_center_mode > 0)
		set_action_activate("autocenter_none", TRUE);

	map_center_latlon(p->lat, p->lon);
	map_update_location_from_center();
} else {
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("No POIs found."));
}

return TRUE;
}

gboolean 
menu_cb_maps_select(GtkAction * action, gpointer new_repo)
{
repo_set_curr(new_repo);
map_force_redraw();
return TRUE;
}

gboolean 
cb_zoom_auto(GtkAction * action)
{
map_set_autozoom(TRUE, _gps->data.speed);
return TRUE;
}

gboolean 
cb_zoom_base(GtkAction * action)
{
map_set_autozoom(FALSE, 0);
map_set_zoom(3);
return TRUE;
}

gboolean 
cb_zoomin(GtkAction * action)
{
map_set_autozoom(FALSE, 0);
g_idle_add((GSourceFunc)map_zoom_in, NULL);
return TRUE;
}

gboolean 
cb_zoomout(GtkAction * action)
{
map_set_autozoom(FALSE, 0);
g_idle_add((GSourceFunc)map_zoom_out, NULL);
return TRUE;
}

gboolean 
cb_fullscreen(GtkAction * action)
{
if ((_fullscreen=gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action)))) {
	gtk_window_fullscreen(GTK_WINDOW(mapp.mainwindow));
} else {
	gtk_window_unfullscreen(GTK_WINDOW(mapp.mainwindow));
}
gtk_idle_add((GSourceFunc)mapper_window_present, NULL);
return TRUE;
}

gboolean
menu_cb_view_toolbar(GtkAction *action)
{
if (gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action)))
	gtk_widget_show(mapp.toolbar);
else
	gtk_widget_hide(mapp.toolbar);

return TRUE;
}

gboolean
menu_cb_audio_toggle(GtkAction *action)
{
if (gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action)))
	hb_action=1;
else
	hb_action=0;
return TRUE;
}

gboolean 
menu_cb_enable_gps(GtkAction * action)
{
if ((_enable_gps=gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action)))) {
	if (!_gps->io.address && gps_type_needs_config(_gps->io.type)) {
		MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Cannot enable GPS until a GPS Receiver has been configured in the GPS Settings dialog."));
		set_action_activate("gps_enabled", FALSE);
		return TRUE;
	}
	gps_conn_set_state(_gps, RCVR_DOWN);
	gps_connect_now(_gps);
} else {
	if (_gps->io.conn > RCVR_OFF)
		gps_conn_set_state(_gps, RCVR_OFF);
	gps_disconnect(_gps);
	track_add(NULL);
	_speed_excess=FALSE;
}

set_action_sensitive("goto_gps", _enable_gps);
set_action_sensitive("autocenter_latlon", _enable_gps);
set_action_sensitive("autocenter_lead", _enable_gps);

if (_enable_gps==FALSE)
	set_action_activate("autocenter_none", TRUE);

map_move_mark();

return TRUE;
}

gboolean 
menu_cb_auto_download(GtkAction * action)
{
if ((_auto_download=gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action)) )) {
	if (_curr_repo->url == REPOTYPE_NONE) {
		popup_error(mapp.mainwindow, _("NOTE: You must set a Map URI in the current repository in order to download maps."));
		/* set_action_activate("", FALSE); */
	}
	map_force_redraw();
} else {
	map_download_stop();
}

return TRUE;
}

gboolean 
menu_cb_settings(GtkAction * action)
{
if (settings_dialog())
	map_force_redraw();
return TRUE;
}

gboolean 
menu_cb_settings_gps(GtkAction * action)
{
if (settings_dialog_gps(_gps)) {
	/* Settings have changed - reconnect to receiver. */
	if (_enable_gps) {
		gps_conn_set_state(_gps, RCVR_DOWN);
		gps_disconnect(_gps);
		gps_connect_now(_gps);
	}
}

return TRUE;
}

gboolean 
menu_cb_settings_colors(GtkAction * action)
{
settings_dialog_colors();
map_force_redraw();
return TRUE;
}

gboolean 
menu_cb_settings_osm(GtkAction * action)
{
settings_dialog_osm();
return TRUE;
}

gboolean 
menu_cb_settings_keys(GtkAction * action)
{
settings_dialog_hardkeys();
return TRUE;
}

gboolean 
menu_cb_import_osm(GtkAction * action)
{
osm_import_dialog(mapp.mainwindow);
return TRUE;
}

gboolean 
menu_cb_about(GtkAction * action)
{
mapper_about_dialog();
return TRUE;
}

gboolean 
window_cb_key_press(GtkWidget * widget, GdkEventKey * event)
{
CustomKey custom_key;

switch (event->keyval) {
	case HILDON_HARDKEY_UP:
		custom_key=CUSTOM_KEY_UP;
	break;
	case HILDON_HARDKEY_DOWN:
		custom_key=CUSTOM_KEY_DOWN;
	break;
	case HILDON_HARDKEY_LEFT:
		custom_key=CUSTOM_KEY_LEFT;
	break;
	case HILDON_HARDKEY_RIGHT:
		custom_key=CUSTOM_KEY_RIGHT;
	break;
	case HILDON_HARDKEY_SELECT:
		custom_key=CUSTOM_KEY_SELECT;
	break;
	case HILDON_HARDKEY_INCREASE:
		custom_key=CUSTOM_KEY_INCREASE;
	break;
	case HILDON_HARDKEY_DECREASE:
		custom_key=CUSTOM_KEY_DECREASE;
	break;
	case HILDON_HARDKEY_FULLSCREEN:
		custom_key=CUSTOM_KEY_FULLSCREEN;
	break;
	case HILDON_HARDKEY_ESC:
		custom_key=CUSTOM_KEY_ESC;
	break;
	default:
		return FALSE;
}

	switch (_action[custom_key]) {
	case CUSTOM_ACTION_PAN_NORTH:
		map_pan(0, -PAN_UNITS);
	break;
	case CUSTOM_ACTION_PAN_WEST:
		map_pan(-PAN_UNITS, 0);
	break;
	case CUSTOM_ACTION_PAN_SOUTH:
		map_pan(0, PAN_UNITS);
	break;
	case CUSTOM_ACTION_PAN_EAST:
		map_pan(PAN_UNITS, 0);
	break;
	case CUSTOM_ACTION_TOGGLE_AUTOCENTER:
		switch (_center_mode) {
		case CENTER_LATLON:
		case CENTER_WAS_LEAD:
			set_action_activate("autocenter_lead", TRUE);
		break;
		case CENTER_LEAD:
		case CENTER_WAS_LATLON:
			set_action_activate("autocenter_latlon", TRUE);
		break;
		default:
			set_action_activate("autocenter_latlon", TRUE);
		break;
		}
	break;
	case CUSTOM_ACTION_ZOOM_IN:
	case CUSTOM_ACTION_ZOOM_OUT:
		if (!_key_zoom_timeout_sid) {
			_key_zoom_new=_zoom + (_action[custom_key] == CUSTOM_ACTION_ZOOM_IN ? -_curr_repo->view_zoom_steps : _curr_repo->view_zoom_steps);
			/* Remember, _key_zoom_new is unsigned. */
			if (_key_zoom_new < MAX_ZOOM) {
				_key_zoom_timeout_sid=g_timeout_add(400, map_key_zoom_timeout, NULL);
			}
		}
	break;
	case CUSTOM_ACTION_TOGGLE_FULLSCREEN:
		set_action_activate("view_fullscreen", !_fullscreen);
	break;
	case CUSTOM_ACTION_TOGGLE_TRACKS:
		switch (_show_tracks) {
		case 0:
			/* Nothing shown, nothing saved; just set both. */
			_show_tracks=TRACKS_MASK | ROUTES_MASK;
			break;
		case TRACKS_MASK << 16:
		case ROUTES_MASK << 16:
		case (ROUTES_MASK | TRACKS_MASK) << 16:
			/* Something was saved and nothing changed since.
			 * Restore saved. */
			_show_tracks=_show_tracks >> 16;
			break;
		default:
			/* There is no history, or they changed something
			 * since the last historical change. Save and
			 * clear. */
			_show_tracks=_show_tracks << 16;
		}
		set_action_activate("view_route", _show_tracks & ROUTES_MASK);
		set_action_activate("view_track", _show_tracks & TRACKS_MASK);
	break;
	case CUSTOM_ACTION_TOGGLE_SCALE:
		set_action_activate("view_scale", _show_scale);
	break;
	case CUSTOM_ACTION_TOGGLE_POI:
		set_action_activate("view_poi", _show_poi);
	break;
	case CUSTOM_ACTION_CHANGE_REPO: {
			GList *curr=g_list_find(_repo_list, _curr_repo);
			if (!curr)
				break;

			/* Loop until we reach a next-able repo, or until we get
			 * back to the current repo. */
			while ((curr=(curr->next ? curr->next : _repo_list)) && !((RepoData *) curr->data)->nextable && curr->data != _curr_repo) {
			}

			if (curr->data != _curr_repo) {
				repo_set_curr(curr->data);
				gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(_curr_repo->menu_item), TRUE);
			} else {
				popup_error(mapp.mainwindow, _("There are no other next-able repositories."));
			}
			break;
		}
	break;
	case CUSTOM_ACTION_ROUTE_DISTNEXT:
		route_show_distance_to_next();
	break;
	case CUSTOM_ACTION_ROUTE_DISTLAST:
		route_show_distance_to_last();
	break;
	case CUSTOM_ACTION_TRACK_BREAK:
		track_insert_break(_track);
	break;
	case CUSTOM_ACTION_TRACK_DISTLAST:
		track_show_distance_from_last(_track);
	break;
	case CUSTOM_ACTION_TRACK_DISTFIRST:
		track_show_distance_from_first(_track);
	break;
	case CUSTOM_ACTION_TOGGLE_GPS:
		set_action_activate("gps_enable", !_enable_gps);
	break;
	case CUSTOM_ACTION_TOGGLE_GPSINFO:
		set_action_activate("gps_info", !_gps_info);
	break;
	case CUSTOM_ACTION_TOGGLE_SPEEDLIMIT:
		_speed_on ^= 1;
	break;
	default:
		return FALSE;
}
return TRUE;
}

gboolean 
window_cb_key_release(GtkWidget * widget, GdkEventKey * event)
{
switch (event->keyval) {
	case HILDON_HARDKEY_INCREASE:
	case HILDON_HARDKEY_DECREASE:
		if (_key_zoom_timeout_sid) {
			g_source_remove(_key_zoom_timeout_sid);
			_key_zoom_timeout_sid=0;
			map_set_zoom(_key_zoom_new);
		}
	return TRUE;
	break;
	default:
		return FALSE;
}
}

void 
cmenu_show_latlon(guint unitx, guint unity)
{
gdouble lat, lon;
gchar buffer[80], tmp1[16], tmp2[16];

unit2latlon(unitx, unity, lat, lon);
lat_format(_degformat, lat, tmp1);
lon_format(_degformat, lon, tmp2);

g_snprintf(buffer, sizeof(buffer),
	 "%s: %s\n"
	 "%s: %s", _("Latitude"), tmp1, _("Longitude"), tmp2);

MACRO_BANNER_SHOW_INFO(mapp.mainwindow, buffer);
}

static void 
cmenu_clip_latlon(guint unitx, guint unity)
{
gchar buffer[80];
gdouble lat, lon;

unit2latlon(unitx, unity, lat, lon);
g_snprintf(buffer, sizeof(buffer), "%.06f,%.06f", lat, lon);

gtk_clipboard_set_text(gtk_clipboard_get(GDK_SELECTION_CLIPBOARD), buffer, -1);
}

static void 
cmenu_route_to(guint unitx, guint unity)
{
gdouble lat,lon;

unit2latlon(unitx, unity, lat, lon);

_dest.valid=TRUE;
_dest.lat=lat;
_dest.lon=lon;
route_download();
route_menu_set_sensitive(path_tree_view_update_store(mapp.route_tree_view, _route));
}

void
cmenu_distance_to(guint unitx, guint unity)
{
gchar buffer[80], b2[32];
gdouble lat, lon;

unit2latlon(unitx, unity, lat, lon);
distance_str(b2, sizeof(b2), _gps->data.lat, _gps->data.lon, lat, lon, _units);
g_snprintf(buffer, sizeof(buffer), "%s: %s", _("Distance"), b2);
MACRO_BANNER_SHOW_INFO(mapp.mainwindow, buffer);
}

void 
cmenu_add_route(guint unitx, guint unity)
{
MACRO_PATH_INCREMENT_TAIL(*_route);
_route->tail->unitx=x2unit(_cmenu_position_x);
_route->tail->unity=y2unit(_cmenu_position_y);
route_find_nearest_point();
map_force_redraw();
}

void cmenu_route_add_way(guint unitx, guint unity)
{
gdouble lat, lon;
gchar tmp1[16], tmp2[16], *p_latlon;
GtkWidget *dialog;
GtkWidget *table;
GtkWidget *label;
GtkWidget *txt_scroll;
GtkWidget *txt_desc;

dialog=gtk_dialog_new_with_buttons(_("Add Waypoint"),
			     GTK_WINDOW(mapp.mainwindow),
			     GTK_DIALOG_MODAL, GTK_STOCK_OK,
			     GTK_RESPONSE_ACCEPT,
			     GTK_STOCK_CANCEL,
			     GTK_RESPONSE_REJECT, NULL);

gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), table=gtk_table_new(2, 2, FALSE), TRUE, TRUE, 0);
gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(_("Lat, Lon")), 0, 1, 0, 1, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);

unit2latlon(unitx, unity, lat, lon);
lat_format(_degformat, lat, tmp1);
lon_format(_degformat, lon, tmp2);
p_latlon=g_strdup_printf("%s, %s", tmp1, tmp2);
gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(p_latlon), 1, 2, 0, 1, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 0.0f, 0.5f);
g_free(p_latlon);

gtk_table_attach(GTK_TABLE(table), label=gtk_label_new(_("Description")), 0, 1, 1, 2, GTK_FILL, 0, 2, 4);
gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);

txt_scroll=gtk_scrolled_window_new(NULL, NULL);
gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(txt_scroll), GTK_SHADOW_IN);
gtk_table_attach(GTK_TABLE(table), txt_scroll, 1, 2, 1, 2, GTK_EXPAND | GTK_FILL, 0, 2, 4);

gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(txt_scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

txt_desc=gtk_text_view_new();
gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(txt_desc), GTK_WRAP_WORD);

gtk_container_add(GTK_CONTAINER(txt_scroll), txt_desc);
gtk_widget_set_size_request(GTK_WIDGET(txt_scroll), 400, 60);

gtk_widget_show_all(dialog);

while (GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(dialog))) {
	GtkTextBuffer *tbuf;
	GtkTextIter ti1, ti2;
	gchar *desc;

	tbuf=gtk_text_view_get_buffer(GTK_TEXT_VIEW(txt_desc));
	gtk_text_buffer_get_iter_at_offset(tbuf, &ti1, 0);
	gtk_text_buffer_get_end_iter(tbuf, &ti2);
	desc=gtk_text_buffer_get_text(tbuf, &ti1, &ti2, TRUE);

	if (*desc) {
		/* There's a description.  Add a waypoint. */
		MACRO_PATH_INCREMENT_TAIL(*_route);
		_route->tail->unitx=unitx;
		_route->tail->unity=unity;
		_route->tail->time=0;
		_route->tail->altitude=NAN;

		MACRO_PATH_INCREMENT_WTAIL(*_route);
		_route->wtail->point=_route->tail;
		_route->wtail->desc
		   =gtk_text_buffer_get_text(tbuf, &ti1, &ti2, TRUE);
	} else {
		GtkWidget *confirm;

		g_free(desc);

		confirm=hildon_note_new_confirmation(GTK_WINDOW(dialog),
					 _("Creating a \"waypoint\" with no description actually "
					  "adds a break point.  Is that what you want?"));
		if (GTK_RESPONSE_OK == gtk_dialog_run(GTK_DIALOG(confirm))) {
			/* There's no description.  Add a break by adding a (0, 0)
			 * point (if necessary), and then the ordinary route point. */
			if (_route->tail->unity) {
				MACRO_PATH_INCREMENT_TAIL(*_route);
				*_route->tail=_point_null;
			}

			MACRO_PATH_INCREMENT_TAIL(*_route);
			_route->tail->unitx=unitx;
			_route->tail->unity=unity;
			_route->tail->time=0;
			_route->tail->altitude=NAN;

			gtk_widget_destroy(confirm);
		} else {
			gtk_widget_destroy(confirm);
			continue;
		}
	}
	route_find_nearest_point();
	map_render_paths();
	MACRO_QUEUE_DRAW_AREA();
	break;
}
route_menu_set_sensitive(path_tree_view_update_store(mapp.route_tree_view, _route));
gtk_widget_destroy(dialog);
}

gboolean 
cmenu_cb_loc_show_latlon(GtkAction * action)
{
cmenu_show_latlon(x2unit(_cmenu_position_x), y2unit(_cmenu_position_y));
return TRUE;
}

gboolean 
cmenu_cb_loc_clip_latlon(GtkAction * action)
{
cmenu_clip_latlon(x2unit(_cmenu_position_x), y2unit(_cmenu_position_y));
return TRUE;
}

gboolean 
cmenu_cb_loc_route_to(GtkAction * action)
{
cmenu_route_to(x2unit(_cmenu_position_x), y2unit(_cmenu_position_y));
return TRUE;
}

gboolean 
cmenu_cb_loc_distance_to(GtkAction * action)
{
cmenu_distance_to(x2unit(_cmenu_position_x), y2unit(_cmenu_position_y));
return TRUE;
}

gboolean 
cmenu_cb_loc_add_route(GtkAction * action)
{
cmenu_add_route(x2unit(_cmenu_position_x), y2unit(_cmenu_position_y));
return TRUE;
}

gboolean 
cmenu_cb_loc_add_way(GtkAction * action)
{
cmenu_route_add_way(x2unit(_cmenu_position_x), y2unit(_cmenu_position_y));
return TRUE;
}

gboolean 
cmenu_cb_loc_add_poi(GtkAction * action)
{
guint ux, uy;
poi_info *poi;

poi=poi_new();
ux=x2unit(_cmenu_position_x);
uy=y2unit(_cmenu_position_y);
unit2latlon(ux, uy, poi->lat, poi->lon);
poi_edit_dialog(ACTION_ADD_POI, poi);

return TRUE;
}

static gboolean 
cb_display_search(gpointer data)
{
gdouble lat, lon;

if (_center_mode>0) {
	lat=_gps->data.lat;
	lon=_gps->data.lon;
} else {
	unit2latlon(_center.unitx, _center.unity, lat, lon);
}
mapper_search_dialog(SEARCH_TYPE_POI, lat, lon);
return FALSE;
}

gboolean
cb_poi_search(GtkAction *action)
{
MACRO_BANNER_SHOW_INFO(mapp.mainwindow, _("Loading search..."));
g_idle_add((GSourceFunc)cb_display_search, NULL);
return TRUE;
}

gboolean 
cb_poi_add(GtkAction *action)
{
gdouble lat,lon;
const gchar *name=gtk_action_get_name(action);
poi_info *p;

if (_center_mode>0) {
	lat=_gps->data.lat;
	lon=_gps->data.lon;
} else {
	unit2latlon(_center.unitx, _center.unity, lat, lon);
}

p=poi_new();
p->lat=lat;
p->lon=lon;
poi_edit_dialog(ACTION_ADD_POI, p);

return TRUE;
}

gboolean
menu_cb_search_address(GtkAction *action)
{
gdouble lat, lon;

if (_center_mode>0) {
	lat=_gps->data.lat;
	lon=_gps->data.lon;
} else {
	unit2latlon(_center.unitx, _center.unity, lat, lon);
}
mapper_search_dialog(SEARCH_TYPE_WAY, lat, lon);
return TRUE;
}

gboolean 
cmenu_cb_loc_set_home(GtkAction * action)
{
guint unitx, unity;

unitx=x2unit(_cmenu_position_x);
unity=y2unit(_cmenu_position_y);
unit2latlon(unitx, unity, _home.lat, _home.lon);
_home.valid=TRUE;

if (!config_save_position(&_home, GCONF_KEY_POSITION_HOME))
	popup_error(mapp.mainwindow, _("Failed to save home position."));

map_update_destination(_home.lat, _home.lon);
map_force_redraw();
return TRUE;
}

gboolean 
cmenu_cb_loc_set_destination(GtkAction *action)
{
guint unitx, unity;

unitx=x2unit(_cmenu_position_x);
unity=y2unit(_cmenu_position_y);
unit2latlon(unitx, unity, _dest.lat, _dest.lon);
_dest.valid=TRUE;

if (!config_save_position(&_dest, GCONF_KEY_POSITION_DEST))
	popup_error(mapp.mainwindow, _("Failed to save destination."));

map_update_destination(_gps->data.lat, _gps->data.lon);
map_force_redraw();
return TRUE;
}

gboolean 
cmenu_cb_loc_set_gps(GtkAction * action)
{
_gps->data.unitx=x2unit(_cmenu_position_x);
_gps->data.unity=y2unit(_cmenu_position_y);
unit2latlon(_gps->data.unitx, _gps->data.unity, _gps->data.lat, _gps->data.lon);

/* Move mark to new location. */
_gps->data.time=time(NULL);
gps_location_update(_gps);

return TRUE;
}

gboolean 
cmenu_cb_way_show_latlon(GtkAction * action)
{
WayPoint *way;

if ((way=route_find_nearest_waypoint(x2unit(_cmenu_position_x), y2unit(_cmenu_position_y))))
	cmenu_show_latlon(way->point->unitx, way->point->unity);

return TRUE;
}

gboolean 
cmenu_cb_way_show_desc(GtkAction * action)
{
WayPoint *way;

if ((way=route_find_nearest_waypoint(x2unit(_cmenu_position_x), y2unit(_cmenu_position_y)))) {
	MACRO_BANNER_SHOW_INFO(mapp.mainwindow, way->desc);
}

return TRUE;
}

gboolean 
cmenu_cb_way_clip_latlon(GtkAction * action)
{
WayPoint *way;

if ((way=route_find_nearest_waypoint(x2unit(_cmenu_position_x), y2unit(_cmenu_position_y))))
	cmenu_clip_latlon(way->point->unitx, way->point->unity);

return TRUE;
}

gboolean 
cmenu_cb_way_clip_desc(GtkAction * action)
{
WayPoint *way;

if ((way=route_find_nearest_waypoint(x2unit(_cmenu_position_x), y2unit(_cmenu_position_y))))
	gtk_clipboard_set_text(gtk_clipboard_get(GDK_SELECTION_CLIPBOARD), way->desc, -1);

return TRUE;
}

gboolean 
cmenu_cb_way_route_to(GtkAction * action)
{
WayPoint *way;

if ((way=route_find_nearest_waypoint(x2unit(_cmenu_position_x), y2unit(_cmenu_position_y))))
	cmenu_route_to(way->point->unitx, way->point->unity);

return TRUE;
}

gboolean 
cmenu_cb_way_distance_to(GtkAction * action)
{
WayPoint *way;

if ((way=route_find_nearest_waypoint(x2unit(_cmenu_position_x), y2unit(_cmenu_position_y))))
	route_show_distance_to(way->point);

return TRUE;
}

gboolean 
cmenu_cb_way_delete(GtkAction * action)
{
WayPoint *way;

if ((way=route_find_nearest_waypoint(x2unit(_cmenu_position_x), y2unit(_cmenu_position_y)))) {
	gchar buffer[BUFFER_SIZE];
	GtkWidget *confirm;

	g_snprintf(buffer, sizeof(buffer), "%s:\n%s\n", _("Confirm delete of waypoint"), way->desc);
	confirm=hildon_note_new_confirmation(GTK_WINDOW(mapp.mainwindow), buffer);

	if (GTK_RESPONSE_OK == gtk_dialog_run(GTK_DIALOG(confirm))) {
		Point *pdel_min, *pdel_max, *pdel_start, *pdel_end;
		guint num_del;

		/* Delete surrounding route data, too. */
		if (way == _route->whead)
			pdel_min=_route->head;
		else
			pdel_min=way[-1].point;

		if (way == _route->wtail)
			pdel_max=_route->tail;
		else
			pdel_max=way[1].point;

		/* Find largest continuous segment around the waypoint, EXCLUDING
		 * pdel_min and pdel_max. */
		for (pdel_start=way->point - 1; pdel_start->unity
		     && pdel_start > pdel_min; pdel_start--) {
		}
		for (pdel_end=way->point + 1; pdel_end->unity
		     && pdel_end < pdel_max; pdel_end++) {
		}

		/* If pdel_end is set to _route->tail, and if _route->tail is a
		 * non-zero point, then delete _route->tail. */
		if (pdel_end == _route->tail && pdel_end->unity)
			pdel_end++;	/* delete _route->tail too */
		/* else, if *both* endpoints are zero points, delete one. */
		else if (!pdel_start->unity && !pdel_end->unity)
			pdel_start--;

		/* Delete BETWEEN pdel_start and pdel_end, exclusive. */
		num_del=pdel_end - pdel_start - 1;

		memmove(pdel_start + 1, pdel_end,(_route->tail - pdel_end + 1) * sizeof(Point));
		_route->tail -= num_del;

		/* Remove waypoint and move/adjust subsequent waypoints. */
		g_free(way->desc);
		while (way++ != _route->wtail) {
			way[-1]=*way;
			way[-1].point -= num_del;
		}
		_route->wtail--;

		route_find_nearest_point();
		map_force_redraw();
	}
	gtk_widget_destroy(confirm);
}

return TRUE;
}

gboolean
menu_cb_category(GtkAction * action)
{
if (poi_category_list())
	map_force_redraw();

return TRUE;
}

gboolean 
cmenu_cb_way_add_poi(GtkAction * action)
{
WayPoint *way;

if ((way=route_find_nearest_waypoint(x2unit(_cmenu_position_x), y2unit(_cmenu_position_y)))) {
	poi_info *p;

	p=poi_new();
	unit2latlon(way->point->unitx, way->point->unity, p->lat, p->lon);
	poi_edit_dialog(ACTION_ADD_POI, p);
}
return TRUE;
}

gboolean 
cmenu_cb_poi_route_to(GtkAction* action, gpointer data)
{
guint32 id;
guint unitx, unity;
poi_info *poi;

id=*(guint32 *)data;

poi=poi_get_by_id(id);
g_return_val_if_fail(poi, TRUE);

latlon2unit(poi->lat, poi->lon, unitx, unity);
cmenu_route_to(unitx, unity);

return TRUE;
}

gboolean 
cmenu_cb_poi_distance_to(GtkAction *action, gpointer data)
{
guint32 id;
guint unitx, unity;
poi_info *poi;

id=*(guint32 *)data;

poi=poi_get_by_id(id);
g_return_val_if_fail(poi, TRUE);

latlon2unit(poi->lat, poi->lon, unitx, unity);
cmenu_distance_to(unitx, unity);

return TRUE;
}

gboolean 
cmenu_cb_poi_add_route(GtkAction *action, gpointer data)
{
guint32 id;
guint unitx, unity;
poi_info *poi;

id=*(guint32 *)data;

poi=poi_get_by_id(id);
g_return_val_if_fail(poi, TRUE);

latlon2unit(poi->lat, poi->lon, unitx, unity);
cmenu_add_route(unitx, unity);

return TRUE;
}

gboolean 
cmenu_cb_poi_add_way(GtkAction *action, gpointer data)
{
guint32 id;
guint unitx, unity;
poi_info *poi;

id=*(guint32 *)data;

poi=poi_get_by_id(id);
g_return_val_if_fail(poi, TRUE);

latlon2unit(poi->lat, poi->lon, unitx, unity);
cmenu_route_add_way(unitx, unity);

poi_free(poi);
return TRUE;
}

gboolean
cmenu_cb_poi_show_poi(GtkAction *action, gpointer data)
{
guint32 id;

id=*(guint32 *)data;

if (poi_info_dialog(mapp.mainwindow, id)==FALSE)
        g_warning("Huh? Failed to display info dialog\n");

return TRUE;
}

gboolean 
cmenu_cb_poi_edit_poi(GtkAction *action, gpointer data)
{
guint32 id;

id=*(guint32 *)data;
poi_edit_dialog(ACTION_EDIT_POI, poi_get_by_id(id));
return TRUE;
}

/***/

gboolean 
headphone_button_cb(gpointer data)
{
Dicto *d;
gchar *f;
static guint m=0;

g_debug("Headset btn");
switch (hb_action) {
case 1:
	d=dicto_widget_get_dicto(DICTO_WIDGET(mapp.audio_ui));
	f=dicto_filename_time_new(d, 0);
	dicto_toggle_record(d, f);
	g_free(f);
break;
case 0:
default:
	path_insert_mark_text(_track, g_strdup_printf("BM %d", m++));
	path_tree_view_update_store(mapp.track_tree_view, _track);
	hildon_banner_show_information(mapp.mainwindow, NULL, _("Mark added"));
break;
}
return FALSE;
} 

gboolean
dicto_recording_cb(Dicto *d, gpointer data)
{
static guint r=0;

path_insert_mark_text(_track, g_strdup_printf("RS: %d", r++));
path_tree_view_update_store(mapp.track_tree_view, _track);
return FALSE;
}

void
path_waypoint_activated_cb(GtkTreeView *tree_view, GtkTreePath *path, GtkTreeViewColumn *column, gpointer data)
{
GtkTreeIter iter;
GtkTreeModel *model;
gdouble lat, lon;

if (!gtk_tree_selection_get_selected(gtk_tree_view_get_selection(GTK_TREE_VIEW(tree_view)), NULL, &iter))
	return;

model=gtk_tree_view_get_model(GTK_TREE_VIEW(tree_view));
if (!model)
	return;

gtk_tree_model_get(model, &iter, ROUTE_LAT, &lat, -1);
gtk_tree_model_get(model, &iter, ROUTE_LON, &lon, -1);

map_center_latlon(lat, lon);
g_idle_add((GSourceFunc)map_update_location_from_center, NULL);
}

