/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 * Copyright (C) 2006-2007 John Costigan.
 *
 * POI and GPS-Info code originally written by Cezary Jackiewicz.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <libintl.h>
#include <locale.h>
#include <math.h>
#include <errno.h>
#include <glib/gstdio.h>
#include <fcntl.h>

#include "gps.h"
#include "gps-conn.h"
#include "gps-nmea-parse.h"

#define DELIM ","

#define NMEA_RMC "$GPRMC"
#define NMEA_GGA "$GPGGA"
#define NMEA_GSA "$GPGSA"
#define NMEA_GSV "$GPGSV"
#define NMEA_PRLEN (6)

#define MACRO_PARSE_INT(tofill, str) { \
    gchar *error_check; \
    (tofill) = strtol((str), &error_check, 10); \
    if(error_check == (str)) \
    { \
        g_printerr("Line %d: Failed to parse string as int: %s\n", __LINE__, str); \
        return; \
    } \
}
#define MACRO_PARSE_FLOAT(tofill, str) { \
    gchar *error_check; \
    (tofill) = g_ascii_strtod((str), &error_check); \
    if(error_check == (str)) \
    { \
        g_printerr("Failed to parse string as float: %s\n", str); \
        return; \
    } \
}

static void
gps_nmea_parse_rmc(Gps *gps, gchar *sentence)
{
	/* Recommended Minimum Navigation Information C
	 *  1) UTC Time
	 *  2) Status, V=Navigation receiver warning A=Valid
	 *  3) Latitude
	 *  4) N or S
	 *  5) Longitude
	 *  6) E or W
	 *  7) Speed over ground, knots
	 *  8) Track made good, degrees true
	 *  9) Date, ddmmyy
	 * 10) Magnetic Variation, degrees
	 * 11) E or W
	 * 12) FAA mode indicator (NMEA 2.3 and later)
	 * 13) Checksum
	 */
	gchar *token, *dpoint, *gpsdate = NULL;
	gdouble tmpd = 0.f;
	guint tmpi = 0;

	/* Parse time. */
	token = strsep(&sentence, DELIM);
	if (token && *token)
		gpsdate = token;

	token = strsep(&sentence, DELIM);
	/* Token is now Status. */
	if (token && *token == 'A') {
		/* Data is valid. */
		if (gps->io.conn != RCVR_FIXED) {
			gps->data.newly_fixed = TRUE;
			gps_conn_set_state(gps, RCVR_FIXED);
		} else {
			gps->data.newly_fixed = FALSE;
		}
	} else {
		/* Data is invalid - not enough satellites?. */
		if (gps->io.conn != RCVR_UP) {
			gps_conn_set_state(gps, RCVR_UP);
			/* track_add(NULL, FALSE); */
		}
	}

	/* Parse the latitude. */
	token = strsep(&sentence, DELIM);
	if (token && *token) {
		dpoint = strchr(token, '.');
		if (!dpoint)	/* handle buggy NMEA */
			dpoint = token + strlen(token);
		MACRO_PARSE_FLOAT(tmpd, dpoint - 2);
		dpoint[-2] = '\0';
		MACRO_PARSE_INT(tmpi, token);
		gps->data.lat = tmpi + (tmpd * (1.0 / 60.0));
	}

	/* Parse N or S. */
	token = strsep(&sentence, DELIM);
	if (token && *token == 'S')
		gps->data.lat = -gps->data.lat;

	/* Parse the longitude. */
	token = strsep(&sentence, DELIM);
	if (token && *token) {
		dpoint = strchr(token, '.');
		if (!dpoint)	/* handle buggy NMEA */
			dpoint = token + strlen(token);
		MACRO_PARSE_FLOAT(tmpd, dpoint - 2);
		dpoint[-2] = '\0';
		MACRO_PARSE_INT(tmpi, token);
		gps->data.lon = tmpi + (tmpd * (1.0 / 60.0));
	}

	/* Parse E or W. */
	token = strsep(&sentence, DELIM);
	if (token && *token == 'W')
		gps->data.lon = -gps->data.lon;

	/* Parse speed over ground, knots. */
	token = strsep(&sentence, DELIM);
	if (token && *token) {
		MACRO_PARSE_FLOAT(gps->data.speed, token);
		if (gps->data.fix > FIX_NOFIX) {
			gps->data.maxspeed=MAX(gps->data.maxspeed, gps->data.speed);
		}
	}

	/* Parse heading, degrees from true north. */
	token = strsep(&sentence, DELIM);
	if (token && *token) {
		MACRO_PARSE_FLOAT(gps->data.heading, token);
	}

	/* Parse date. */
	token = strsep(&sentence, DELIM);
	if (token && *token && gpsdate) {
		struct tm time;
		gpsdate[6] = '\0';	/* Make sure time is 6 chars long. */
		strcat(gpsdate, token);
		strptime(gpsdate, "%H%M%S%d%m%y", &time);
		gps->data.time = mktime(&time) + _gmtoffset;
	} else {
		gps->data.time = time(NULL);
	}

	gps_data_integerize(&gps->data);
	gps->data.lheading=gps->data.heading;
}

static void
gps_nmea_parse_gga(Gps *gps, gchar * sentence)
{
	/* GGA          Global Positioning System Fix Data
	   1. Fix Time
	   2. Latitude
	   3. N or S
	   4. Longitude
	   5. E or W
	   6. Fix quality
	   0 = invalid
	   1 = GPS fix (SPS)
	   2 = DGPS fix
	   3 = PPS fix
	   4 = Real Time Kinematic
	   5 = Float RTK
	   6 = estimated (dead reckoning) (2.3 feature)
	   7 = Manual input mode
	   8 = Simulation mode
	   7. Number of satellites being tracked
	   8. Horizontal dilution of position
	   9. Altitude, Meters, above mean sea level
	   10. Alt unit (meters)
	   11. Height of geoid (mean sea level) above WGS84 ellipsoid
	   12. unit
	   13. (empty field) time in seconds since last DGPS update
	   14. (empty field) DGPS station ID number
	   15. the checksum data
	 */
	gchar *token;

	/* Skip Fix time */
	token = strsep(&sentence, DELIM);
	/* Skip latitude */
	token = strsep(&sentence, DELIM);
	/* Skip N or S */
	token = strsep(&sentence, DELIM);
	/* Skip longitude */
	token = strsep(&sentence, DELIM);
	/* Skip S or W */
	token = strsep(&sentence, DELIM);

	/* Parse Fix quality */
	token = strsep(&sentence, DELIM);
	if (token && *token)
		MACRO_PARSE_INT(gps->data.fixquality, token);

	/* Skip number of satellites */
	token = strsep(&sentence, DELIM);

	/* Parse Horizontal dilution of position */
	token = strsep(&sentence, DELIM);
	if (token && *token)
		MACRO_PARSE_INT(gps->data.hdop, token);

	/* Altitude */
	token = strsep(&sentence, DELIM);
	if (token && *token) {
		MACRO_PARSE_FLOAT(gps->data.altitude, token);
	} else
		gps->data.altitude = NAN;
}

static void
gps_nmea_parse_gsa(Gps *gps, gchar * sentence)
{
	/* GPS DOP and active satellites
	 *  1) Auto selection of 2D or 3D fix (M = manual)
	 *  2) 3D fix - values include: 1 = no fix, 2 = 2D, 3 = 2D
	 *  3) PRNs of satellites used for fix
	 *     (space for 12)
	 *  4) PDOP (dilution of precision)
	 *  5) Horizontal dilution of precision (HDOP)
	 *  6) Vertical dilution of precision (VDOP)
	 *  7) Checksum
	 */
	gchar *token;
	guint i,si;
	gint satforfix[12];

	/* Skip Auto selection. */
	token = strsep(&sentence, DELIM);

	/* 3D fix. */
	token = strsep(&sentence, DELIM);
	if (token && *token)
		MACRO_PARSE_INT(gps->data.fix, token);

	gps->data.satinuse = 0;
	for (i = 0; i < 12; i++) {
		gint fprn;
		token = strsep(&sentence, DELIM);
		if (token && *token)
			fprn=atoi(token);
		else
			fprn=-1;
		satforfix[i]=fprn;
		gps->data.sat[i].fix=FALSE;
	}

	for (i=0;i<12;i++)
		for (si=0;si<12;si++) {
			if (gps->data.sat[i].prn==satforfix[si])
				gps->data.sat[i].fix=TRUE;
	}

	/* PDOP */
	token = strsep(&sentence, DELIM);
	if (token && *token)
		MACRO_PARSE_FLOAT(gps->data.pdop, token);

	/* HDOP */
	token = strsep(&sentence, DELIM);
	if (token && *token)
		MACRO_PARSE_FLOAT(gps->data.hdop, token);

	/* VDOP */
	token = strsep(&sentence, DELIM);
	if (token && *token)
		MACRO_PARSE_FLOAT(gps->data.vdop, token);
}

static void
gps_nmea_parse_gsv(Gps *gps, gchar * sentence)
{
	/* Must be GSV - Satellites in view
	 *  1) total number of messages
	 *  2) message number
	 *  3) satellites in view
	 *  4) satellite number
	 *  5) elevation in degrees (0-90)
	 *  6) azimuth in degrees to true north (0-359)
	 *  7) SNR in dB (0-99)
	 *  more satellite infos like 4)-7)
	 *  n) checksum
	 */
	gchar *token;
	guint msgcnt = 0, nummsgs = 0;
	static guint running_total = 0;
	static guint num_sats_used = 0;
	static guint satcnt = 0;
	/* g_printf("%s(): %s\n", __PRETTY_FUNCTION__, sentence); */

	/* Parse number of messages. */
	token = strsep(&sentence, DELIM);
	if (token && *token)
		MACRO_PARSE_INT(nummsgs, token);

	/* Parse message number. */
	token = strsep(&sentence, DELIM);
	if (token && *token)
		MACRO_PARSE_INT(msgcnt, token);

	/* Parse number of satellites in view. */
	token = strsep(&sentence, DELIM);
	if (token && *token) {
		MACRO_PARSE_INT(gps->data.satinview, token);
		if (gps->data.satinview > 12)	/* Handle buggy NMEA. */
			gps->data.satinview = 12;
	}

	/* Loop until there are no more satellites to parse. */
	while (sentence && satcnt < 12) {
		/* Get token for Satellite Number. */
		token = strsep(&sentence, DELIM);
		if (token && *token)
			gps->data.sat[satcnt].prn = atoi(token);

		/* Get token for elevation in degrees (0-90). */
		token = strsep(&sentence, DELIM);
		if (token && *token)
			gps->data.sat[satcnt].elevation = atoi(token);

		/* Get token for azimuth in degrees to true north (0-359). */
		token = strsep(&sentence, DELIM);
		if (token && *token)
			gps->data.sat[satcnt].azimuth = atoi(token);

		/* Get token for SNR. */
		token = strsep(&sentence, DELIM);
		if (token && *token && (gps->data.sat[satcnt].snr = atoi(token))) {
			/* SNR is non-zero - add to total and count as used. */
			running_total += gps->data.sat[satcnt].snr;
			num_sats_used++;
		}
		satcnt++;
	}

	if (msgcnt == nummsgs) {
		/*  This is the last message. Calculate signal strength. */
		if (num_sats_used) {
			if (gps->io.conn==RCVR_UP) {
				gdouble fraction = running_total * sqrt(num_sats_used) / num_sats_used / 100.0;
				fraction=CLAMP(fraction, 0.0, 1.0);
				if (gps->connection_progress!=NULL)
					gps->connection_progress(gps, fraction);
			}
			running_total=0;
			num_sats_used=0;
		}
		satcnt = 0;
	}
}

gboolean
gps_nmea_parse(Gps *gps)
{
g_assert(gps);

if (!gps->io.nmea)
	return FALSE;

if (strlen(gps->io.nmea)<NMEA_PRLEN+1)
	return FALSE;

if (!strncmp(gps->io.nmea, NMEA_GSV, NMEA_PRLEN)) {
	gps_nmea_parse_gsv(gps, gps->io.nmea + NMEA_PRLEN+1);
	if (gps->update_satellite)
		g_idle_add_full(G_PRIORITY_DEFAULT, (GSourceFunc)gps->update_satellite, gps, NULL);

} else if (!strncmp(gps->io.nmea, NMEA_RMC, NMEA_PRLEN)) {
	gps_nmea_parse_rmc(gps, gps->io.nmea + NMEA_PRLEN+1);
	if ((gps->io.conn==RCVR_FIXED) && (gps->update_location!=NULL))
		gps->update_location(gps);

} else if (!strncmp(gps->io.nmea, NMEA_GGA, NMEA_PRLEN)) {
	gps_nmea_parse_gga(gps, gps->io.nmea + NMEA_PRLEN+1);
	if (gps->update_info)
		g_idle_add_full(G_PRIORITY_DEFAULT, (GSourceFunc)gps->update_info, gps, NULL);

} else if (!strncmp(gps->io.nmea, NMEA_GSA, NMEA_PRLEN)) {
	gps_nmea_parse_gsa(gps, gps->io.nmea + NMEA_PRLEN+1);
	if (gps->update_satellite)
		g_idle_add_full(G_PRIORITY_DEFAULT, (GSourceFunc)gps->update_satellite, gps, NULL);
	if (gps->update_info)
		g_idle_add_full(G_PRIORITY_DEFAULT, (GSourceFunc)gps->update_info, gps, NULL);

} else g_printerr("Unknown NMEA: [%s]\n", gps->io.nmea);

return FALSE;
}
