/*
 * Copyright (C) 2009 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of libzeemote.
 *
 * libzeemote is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * libzeemote is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libzeemote.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>

#include "zeemote.h"

const char *state_names[] = {
  "unknown          ",
  "connecting       ",
  "connection failed",
  "connected        ",
  "connection lost  ",
  "disconnected     "
};

static const char *get_name(int idx) {
  static const char *type_str[] = {
    "Zeemote JS1",
    ""
  };

  static const char unknown[] = "<unknown>";

  if(idx >= 0 && idx < 1) return type_str[idx];

  return unknown;
}

int main(int argc, char **argv) {
  char addr[18];
  bdaddr_t bdaddr;

  printf("zeemote_test - (c) 2009 by Till Harbaum <till@harbaum.org>\n");

  if(argc <= 1) {
    int i;

    printf("No device address given, trying to autodetect device ...\n");
    zeemote_scan_result_t *result = zeemote_scan();
    if(!result->number_of_devices) {
      printf("No zeemote devices found\n");
      free(result);
      exit(0);
    }

    printf("Found %d zeemote devices:\n", result->number_of_devices);
    for(i=0;i<result->number_of_devices;i++) {
      ba2str(&result->device[i].bdaddr, addr);
      printf("%d: %s @ %s\n", i, get_name(result->device[i].type), addr);
    }

    /* use first device found */
    bdaddr = result->device[0].bdaddr;
  } else
    baswap(&bdaddr, strtoba(argv[1]));
 
  ba2str(&bdaddr, addr);
  printf("Connecting to %s\n", addr);

  zeemote_t *zeemote = zeemote_connect(&bdaddr);
  if(!zeemote) {
    printf("Unable to connect to zeemote\n");
    return 1;
  }

  printf("Press buttons A and B simultaneously to quit\n");

  zeemote_state_t *state = NULL;
  do {
    int i;
    zeemote_info_t *info = zeemote_get_info(zeemote);

    state = zeemote_get_state(zeemote);
    
    printf("\r%s ", state_names[state->state]);
    for(i=0;i<info->num_axes;i++)
      printf("X%d:%6d ", i, state->axis[i]);

    for(i=0;i<info->num_buttons;i++)
      printf("%c:%s ", 'A'+i, (state->buttons & (1<<i))?"on ":"off");

    fflush(stdout);

    usleep(10000);
  } while((state->state != ZEEMOTE_STATE_CONNECTION_LOST) &&
	  (state->state != ZEEMOTE_STATE_CONNECTION_FAILED) &&
	  (state->state != ZEEMOTE_STATE_DISCONNECTED) &&
	  (state->buttons & (ZEEMOTE_BUTTON_A | ZEEMOTE_BUTTON_B)) != 
	  (ZEEMOTE_BUTTON_A | ZEEMOTE_BUTTON_B));

  zeemote_disconnect(zeemote);

  printf("\nquit!\n");

  return 0;
}
