/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of libobd.
 *
 *  libobd is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  libobd is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with libobd.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>

#include "libobd.h"
#include "obd-chan.h"
#include "obd-conn.h"
#include "obd-log.h"

#define CHECK_BYTE '\xFA'

struct obd {
	char check;
	struct obdcon conn;
	struct obdchan chan;
};

static int obd_check_param(void *obd)
{
	struct obd* pobd;

	if(obd == NULL)
		return -1;

	pobd = obd;

	if(pobd->check != CHECK_BYTE) {
		LOG_ERROR("obd pointer does not point to a obd struct");
		return -1;
	}

	return 0;
}

void* obd_alloc()
{
	struct obd* pobd;
	pobd = malloc(sizeof(struct obd));

	if(pobd == NULL)
		return NULL;

	obdcon_init(&pobd->conn);
	obdchan_init(&pobd->chan);

	pobd->check = CHECK_BYTE;
	return pobd;
}

int obd_open_dev(void* obd, const char *devname)
{
	struct obd* pobd;

	if (obd_check_param(obd))
		return -1;

	pobd = obd;

	if(pobd->conn.status != OBDCON_NOT_CONNECTED)
		return -1;

	if(pobd->chan.type != OBDCHAN_UNDEFINED)
		return -1;

	return obdcon_opendev(&pobd->conn, devname);
}

int obd_open_simulator(void* obd)
{
	
	struct obd* pobd;

	if(obd_check_param(obd)) {
		LOG_ERROR("Wrong obd pointer (buffer overflow?)");
		return -1;
	}

	pobd = obd;

	if(pobd->conn.status != OBDCON_NOT_CONNECTED) {
		LOG_ERROR("OBD connection already established: %d", pobd->conn.status);
		return -1;
	}

	return obdchan_setup_simulator(&pobd->chan);
}

int obd_open_file(void* obd, char* filename)
{
	struct obd* pobd;

	if(obd_check_param(obd)) {
		LOG_ERROR("Wrong obd pointer (buffer overflow?)");
		return -1;
	}

	if(filename == NULL) {
		LOG_ERROR("Filename param is NULL");
		return -1;
	}

	pobd = obd;

	if(pobd->conn.status != OBDCON_NOT_CONNECTED) {
		LOG_ERROR("OBD connection already established: %d", pobd->conn.status);
		return -1;
	}

	return obdchan_setup_file(&pobd->chan, filename);

}

int obd_channel_setup(void* obd)
{
	struct obd* pobd;

	if(obd_check_param(obd)) {
		LOG_ERROR("Wrong obd pointer (buffer overflow?)");
		return -1;
	}

	pobd = obd;
	if(pobd->conn.status != OBDCON_CONNECTED) {
		LOG_ERROR("Channel setup called but no serial is connected!");
		return -1;
	}

	return obdchan_setup_channel(&pobd->chan, &pobd->conn);
}

int obd_close(void* obd)
{
	struct obd* pobd;

	if(obd_check_param(obd)) {
		LOG_ERROR("Wrong obd pointer (buffer overflow?)");
		return -1;
	}

	pobd = obd;

	if(pobd->chan.type == OBDCHAN_UNDEFINED)
		LOG_ERROR("Trying to close an not opened obd device");
	else
		obdchan_close(&pobd->chan);

	if(pobd->conn.status != OBDCON_NOT_CONNECTED)
		obdcon_closedev(&pobd->conn);

	return 0;
}

int obd_dealloc(void* obd)
{
	struct obd* pobd;

	if(obd_check_param(obd)) {
		LOG_ERROR("Wrong obd pointer (buffer overflow?)");
		return -1;
	}

	pobd = obd;

	if(pobd->conn.status != OBDCON_NOT_CONNECTED)
		obdcon_closedev(&pobd->conn);

	if(pobd->chan.type != OBDCHAN_UNDEFINED)
		obdchan_close(&pobd->chan);

	free(obd);

	return 0;
}

int obd_sendpid (void* obd, struct obd_sensor_data* sensor_data, unsigned int timeout)
{
	struct obd* pobd;

	if(obd_check_param(obd))
		return -1;

	pobd = obd;

	if(pobd->chan.type == OBDCHAN_UNDEFINED) {
		LOG_ERROR("Channel type is undefined? %d", pobd->chan.type);
		return -1;
	}

	sensor_data->valid = 0;

	return obdchan_sendpid(&pobd->chan, sensor_data, timeout);
}

int obd_senddtc (void* obd, struct obd_dtc_data* dtc_data, unsigned int timeout)
{
	struct obd* pobd;

	if(obd_check_param(obd))
		return -1;

	pobd = obd;

	if(pobd->chan.type == OBDCHAN_UNDEFINED) {
		LOG_ERROR("Channel type is undefined? %d", pobd->chan.type);
		return -1;
	}

	return obdchan_senddtc(&pobd->chan, dtc_data, timeout);

}

int obd_channel_name(void* obd, char* buffer, unsigned int size)
{
	struct obd* pobd;

	if(obd_check_param(obd))
		return -1;

	if(buffer == NULL || size == 0)
		return -1;

	pobd = obd;

	if(pobd->chan.type == OBDCHAN_UNDEFINED) {
		LOG_ERROR("Channel type is undefined? %d", pobd->chan.type);
		return -1;
	}

	return obdchan_name(&pobd->chan, buffer, size);
}

int obd_log_setup(FILE* file, int level)
{
	int lev;

	switch(level) {
		case OBDLOG_ERROR:
		case OBDLOG_WARNING:
		case OBDLOG_INFO:
		case OBDLOG_DEBUG:
			lev = level;
		default:
			lev = OBDLOG_ERROR;
	}

	obdlog_setup(file, level);
	return 0;
}
