
/* pkmodule.c - part of libgcrypt-py
 * Copyright 2004 (C) Nat Tuck
 * Licenced under the GNU LGPL 2.1+, see COPYING.txt for details
 */

// TODO: De-jank this. Apparently error codes aren't in
//      gcrypt.h ...
const int INVALID_SIGNATURE = 16382; // Whatever.

#include <Python.h>
#include <gcrypt.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

inline void process_error(gcry_error_t error, const char * msg) {
    fprintf(stderr, "ERROR: libgcrypt-py / pkmodule.c / %s\n", msg);
    fprintf(stderr, "    source: %s, error: %s\n", 
        gcry_strsource (error), gcry_strerror (error));
    exit(4);
}

void free_sexp(void * sexp) {
    printf("Something actually called free_sexp\n");
    gcry_sexp_release(*(gcry_sexp_t *)sexp);
    free(sexp);
}

static PyObject * pk_quick_random(PyObject * self, PyObject * args) {
    gcry_control (GCRYCTL_ENABLE_QUICK_RANDOM, 0);
    fprintf(stderr, "WARNING: Insecure random numbers enabled,\n");
    fprintf(stderr, "   there is no way to turn them off.\n");

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject * pk_load_sexp(PyObject * self, PyObject * args) {
    gcry_error_t err = 0;
    
    char * sexp;
    int len;
    if(!PyArg_ParseTuple(args, "s#", &sexp, &len))
        return NULL;

    gcry_sexp_t * new_sexp = malloc(sizeof(gcry_sexp_t));
    err = gcry_sexp_new(new_sexp, (void *) sexp, len, 1);
    if(err) process_error(err, "gcry_sexp_new (px_load_sexp)");
    
    return Py_BuildValue("O",
        PyCObject_FromVoidPtr(new_sexp, free_sexp));
}

static PyObject * pk_dump_sexp(PyObject * self, PyObject * args) {
    gcry_error_t err = 0;
   
    void * py_sexp;
    if(!PyArg_ParseTuple(args, "O", &py_sexp))
        return NULL;

    gcry_sexp_t * sexp;
    sexp = (gcry_sexp_t *) PyCObject_AsVoidPtr(py_sexp);

    int size = gcry_sexp_sprint(*sexp, GCRYSEXP_FMT_ADVANCED, NULL, 0);
    unsigned char buffer[size];
    gcry_sexp_sprint(*sexp, GCRYSEXP_FMT_ADVANCED, buffer, size);

    return Py_BuildValue("s#", buffer, size);
}

static PyObject * pk_gen_key_pair(PyObject * self, PyObject * args) {
    gcry_error_t err = 0;
    const char * key_parm = "(genkey (%s (nbits %d)))";

    char * algo = 0;
    int keylen = 0;
    if(!PyArg_ParseTuple(args, "si", &algo, &keylen))
        return NULL;
    
    gcry_sexp_t arg_sexp;
    err = gcry_sexp_build(&arg_sexp, 0, key_parm, algo, keylen);
    if(err) process_error(err, "gcry_sexp_build (pk_gen_key_pair)");
    
    gcry_sexp_t keypair;
    err = gcry_pk_genkey(&keypair, arg_sexp);
    if(err) process_error(err, "gcry_pk_keygen (pk_gen_key_pair)");
    gcry_sexp_release(arg_sexp);
    
    gcry_sexp_t * public = malloc(sizeof(gcry_sexp_t));
    gcry_sexp_t * private = malloc(sizeof(gcry_sexp_t));

    *public = gcry_sexp_find_token(keypair, "public-key", 0);
    *private = gcry_sexp_find_token(keypair, "private-key", 0);
    // TODO: Error check the above 2 statements.
    gcry_sexp_release(keypair);
    
    return Py_BuildValue("(OO)",
        PyCObject_FromVoidPtr(public, free_sexp),
        PyCObject_FromVoidPtr(private, free_sexp));
}

static PyObject * pk_encrypt (PyObject * self, PyObject * args) {
    gcry_error_t err = 0;
    
    char * plaintext;
    int len;
    void * py_key;
    if(!PyArg_ParseTuple(args, "s#O", &plaintext, &len, &py_key))
        return NULL;
    
    gcry_sexp_t * key;
    key = (gcry_sexp_t *) PyCObject_AsVoidPtr(py_key);
    
    gcry_mpi_t data_mpi;
    gcry_mpi_scan(&data_mpi, GCRYMPI_FMT_STD, plaintext, len, 0);
    
    gcry_sexp_t data_sexp;
    err = gcry_sexp_build(&data_sexp, 0,
        "(data (flags pkcs1) (value %m))", data_mpi);
    if(err) process_error(err, "gcry_sexp_build (pk_encrypt)");

    gcry_mpi_release(data_mpi);
 
    gcry_sexp_t ciphertext;
    err = gcry_pk_encrypt(&ciphertext, data_sexp, *key);
    if(err) process_error(err, "gcry_pk_encrypt (pk_encrypt)");

    gcry_sexp_release(data_sexp);
    
    int size = gcry_sexp_sprint(ciphertext, GCRYSEXP_FMT_ADVANCED, NULL, 0);
    unsigned char buffer[size];
    gcry_sexp_sprint(ciphertext, GCRYSEXP_FMT_ADVANCED, buffer, size);

    gcry_sexp_release(ciphertext);
    
    return Py_BuildValue("s#", buffer, size); 
}

static PyObject * pk_decrypt (PyObject * self, PyObject * args) {
    gcry_error_t err = 0;

    unsigned char * ciphertext;
    int ct_len;
    void * py_key;
    if(!PyArg_ParseTuple(args, "s#O", &ciphertext, &ct_len, &py_key))
        return NULL;

    gcry_sexp_t * key;
    key = (gcry_sexp_t *) PyCObject_AsVoidPtr(py_key);

    gcry_sexp_t ct_sexp;
    err = gcry_sexp_new(&ct_sexp, (void *) ciphertext, ct_len, 1);
    if(err) process_error(err, "gcry_sexp_new (pk_decrypt)");
    
    gcry_sexp_t pt_sexp; 
    err = gcry_pk_decrypt(&pt_sexp, ct_sexp, *key);
    if(err) process_error(err, "gcry_decrypt (pk_decrypt)");
    gcry_sexp_release(ct_sexp);

    gcry_mpi_t pt_mpi;
    pt_mpi = gcry_sexp_nth_mpi(pt_sexp, 0, GCRYMPI_FMT_STD);
    gcry_sexp_release(pt_sexp);

    unsigned char * pt_data;
    size_t pt_size;
    gcry_mpi_aprint(GCRYMPI_FMT_STD, &pt_data, &pt_size, pt_mpi);
    gcry_mpi_release(pt_mpi);

    PyObject * plaintext = Py_BuildValue("s#", pt_data, pt_size);
    free(pt_data);

    return plaintext;
}

static PyObject * pk_sign (PyObject * self, PyObject * args) {
    gcry_error_t err = 0; 

    unsigned char * hash;
    unsigned char * algo;
    int hash_len;
    void * py_key;
    if(!PyArg_ParseTuple(args, "s#Os", &hash, &hash_len, &py_key, &algo))
        return NULL;

    gcry_sexp_t * key;
    key = (gcry_sexp_t *) PyCObject_AsVoidPtr(py_key);

    gcry_sexp_t data_sexp;
    err = gcry_sexp_build(&data_sexp, 0,
        "(data (flags pkcs1) (hash %s %b))", algo, hash_len, hash);
    if(err) process_error(err, "gcry_sexp_build (pk_sign)");

    gcry_sexp_t sig_sexp;
    err = gcry_pk_sign(&sig_sexp, data_sexp, *key);
    if(err) process_error(err, "gcry_sexp_build (pk_sign)");
    gcry_sexp_release(data_sexp);

    int size = gcry_sexp_sprint(sig_sexp, GCRYSEXP_FMT_ADVANCED, NULL, 0);
    unsigned char buffer[size];
    gcry_sexp_sprint(sig_sexp, GCRYSEXP_FMT_ADVANCED, buffer, size);
    gcry_sexp_release(sig_sexp); 

    return Py_BuildValue("s#", buffer, size);
}

static PyObject * pk_verify (PyObject * self, PyObject * args) {
    gcry_error_t err = 0; 

    char *sig, *hash, *algo;
    int sig_len, hash_len;
    void * py_key;
    if(!PyArg_ParseTuple(args, "s#s#Os", &sig, &sig_len, &hash, 
          &hash_len, &py_key, &algo))
        return NULL;

    gcry_sexp_t data_sexp;
    err = gcry_sexp_build(&data_sexp, 0,
        "(data (flags pkcs1) (hash %s %b))", algo, hash_len, hash);
    if(err) process_error(err, "gcry_sexp_build (pk_verify)");

    gcry_sexp_t sig_sexp;
    err = gcry_sexp_new(&sig_sexp, sig, sig_len, 1);
    if(err) process_error(err, "gcry_sexp_new (pk_verify)");

    gcry_sexp_t * key;
    key = (gcry_sexp_t *) PyCObject_AsVoidPtr(py_key);

    err = gcry_pk_verify(sig_sexp, data_sexp, *key);
    gcry_sexp_release(sig_sexp);
    gcry_sexp_release(data_sexp);
    if(err)
        if (gcry_err_code_to_errno(err) == INVALID_SIGNATURE)
            return Py_BuildValue("i", 0);
        else 
            process_error(err, "gcry_pk_verify (pk_verify)");

    // If we get to here, the signature should be valid.
    return Py_BuildValue("i", 1);
}

static PyMethodDef PkMethods [] = {
    {"gen_key_pair", pk_gen_key_pair, METH_VARARGS, "Make a cipher object."},
    {"use_quick_random", pk_quick_random, METH_VARARGS, "Cheater random #s."},
    {"load_sexp", pk_load_sexp, METH_VARARGS, "Load a sexp from a string."},
    {"dump_sexp", pk_dump_sexp, METH_VARARGS, "Dump a sexp to a string."},
    {"encrypt", pk_encrypt, METH_VARARGS, "Encrypt some plaintext."},
    {"decrypt", pk_decrypt, METH_VARARGS, "Decrypt some ciphertext."},
    {"sign", pk_sign, METH_VARARGS, "Sign some message."},
    {"verify", pk_verify, METH_VARARGS, "Verify some signature."},
    {NULL, NULL, 0, NULL}
};

PyMODINIT_FUNC init_Pk(void) {
    (void) Py_InitModule("_Pk", PkMethods);
    (void) gcry_check_version("1.2.0");
    gcry_control (GCRYCTL_DISABLE_SECMEM, 0);
}
