/* hashmodule.c - part of libgcrypt-py
 * Copyright 2004 (C) Nat Tuck
  * Licenced under the GNU LGPL 2.1+, see COPYING.txt for details
   */

#include <Python.h>
#include <gcrypt.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

void hash_del(void * hobj) {
    gcry_md_close(* (gcry_md_hd_t *) hobj);
    free(hobj);
}

static PyObject * hash_algo_list (PyObject * self, PyObject * args) {
    return Py_BuildValue("{s:i,s:i,s:i,s:i}",
        "SHA1", GCRY_MD_SHA1, "RMD160", GCRY_MD_RMD160,
        "MD5", GCRY_MD_MD5, "TIGER", GCRY_MD_TIGER);
}

static PyObject * hash_init(PyObject * self, PyObject * args) {
    gcry_md_hd_t * hobj;
    unsigned int algo;

    if (!PyArg_ParseTuple(args, "i", &algo))
        return NULL;
    
    hobj = malloc(sizeof(gcry_md_hd_t));
    gcry_md_open(hobj, algo, 0);
    if (hobj == 0) {
        fprintf(stderr, "POORLY DONE ERROR: _Hash: hobj == 0\n");
        return NULL;
    }

    return PyCObject_FromVoidPtr((void *) hobj, hash_del);
}

static PyObject * hash_add(PyObject * self, PyObject * args) {
    PyObject * py_hobj;
    char * data;
    unsigned int dlen;

    if (!PyArg_ParseTuple(args, "Os#", &py_hobj, &data, &dlen))
        return NULL;

    gcry_md_hd_t * hobj;
    hobj = (gcry_md_hd_t *) PyCObject_AsVoidPtr(py_hobj);

    gcry_md_write(*hobj, data, dlen);

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject * hash_hash(PyObject * self, PyObject * args) {
    PyObject * py_hobj;

    if (!PyArg_ParseTuple(args, "O", &py_hobj))
        return NULL;

    gcry_md_hd_t * hobj;
    hobj = (gcry_md_hd_t *) PyCObject_AsVoidPtr(py_hobj);

    unsigned char * hash;
    
    hash = gcry_md_read(*hobj, 0);
    
    return Py_BuildValue("s#", hash, 
        gcry_md_get_algo_dlen(gcry_md_get_algo(*hobj)));
}


static PyMethodDef HashMethods [] = {
    {"algo_list", hash_algo_list, METH_VARARGS, "Map algo names to IDs"},
    {"init", hash_init, METH_VARARGS, "Make a hobj"},
    {"add", hash_add, METH_VARARGS, "Add some data to the hobj"},
    {"hash", hash_hash, METH_VARARGS, "Calculate and return the hash"},
    {NULL, NULL, 0, NULL}
};

PyMODINIT_FUNC init_Hash(void) {
    (void) Py_InitModule("_Hash", HashMethods);
}
