
/* ciphermodule.c - part of libgcrypt-py
 * Copyright 2004 (C) Nat Tuck
 * Licenced under the GNU LGPL 2.1+, see COPYING.txt for details
 */

#include <Python.h>
#include <gcrypt.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#define MODE_CBC_CTS 299

void cipher_del(void * cobj) {
    gcry_cipher_close(* (gcry_cipher_hd_t *) cobj);
    free(cobj);
}

static PyObject * cipher_algo_list (PyObject * self, PyObject * args) {
    return Py_BuildValue("{s:i,s:i,s:i,s:i,s:i,s:i,s:i}",
        "3DES", GCRY_CIPHER_3DES, "BLOWFISH-128", GCRY_CIPHER_BLOWFISH,
        "AES-128", GCRY_CIPHER_AES128, "AES-192", GCRY_CIPHER_AES192,
        "AES-256", GCRY_CIPHER_AES256, "TWOFISH-128", GCRY_CIPHER_TWOFISH128,
        "TWOFISH-256", GCRY_CIPHER_TWOFISH); 
}

static PyObject * cipher_mode_list (PyObject * self, PyObject * args) {
    return Py_BuildValue("{s:i,s:i,s:i}",
        "ECB", GCRY_CIPHER_MODE_ECB, "CBC", GCRY_CIPHER_MODE_CBC,
        "CBC-CTS", MODE_CBC_CTS);
}

static PyObject * cipher_get_info (PyObject * self, PyObject * args) {
    unsigned int algo;
    size_t blocksize, keysize;

    if(!PyArg_ParseTuple(args, "I", &algo))
        return NULL;

    gcry_cipher_algo_info(algo, GCRYCTL_GET_BLKLEN, 0, &blocksize);
    gcry_cipher_algo_info(algo, GCRYCTL_GET_KEYLEN, 0, &keysize);

    return Py_BuildValue("{s:i,s:i}",
        "blocksize", (int) blocksize, "keysize", (int) keysize);
}

static PyObject * cipher_init(PyObject * self, PyObject * args) {
    unsigned int algo, mode;
    unsigned char *key, *iv;
    unsigned int klen, ilen;

    if(!PyArg_ParseTuple(args, "IIs#z#", 
            &algo, &mode, &key, &klen, &iv, &ilen))
        return NULL;
    
    gcry_cipher_hd_t * cobj;
    cobj = malloc(sizeof(gcry_cipher_hd_t));
    
    unsigned int flags = 0;
    if(mode == MODE_CBC_CTS) {
        flags = GCRY_CIPHER_CBC_CTS;
        mode = GCRY_CIPHER_MODE_CBC;    
    }

    gcry_cipher_open(cobj, algo, mode, flags);
    gcry_cipher_setkey(*cobj, (void *) key, (size_t) klen);
    if(iv != 0)
        gcry_cipher_setiv(*cobj, (void *) iv, (size_t) ilen);

    return PyCObject_FromVoidPtr((void *) cobj, cipher_del);
}

static PyObject * cipher_encrypt(PyObject * self, PyObject * args) {
    void * py_cobj;
    unsigned char * data;
    unsigned int dlen;

    if(!PyArg_ParseTuple(args, "Os#", &py_cobj, &data, &dlen))
        return NULL;

    gcry_cipher_hd_t * cobj;
    cobj = (gcry_cipher_hd_t *) PyCObject_AsVoidPtr(py_cobj);

    unsigned char ciphertext[dlen];
    int error = 0;
    
    error = gcry_cipher_encrypt(*cobj, (unsigned char *) &ciphertext, 
        dlen, data, dlen);
    if(error) fprintf(stderr, "Encrypt error: %d\n", error);

    return Py_BuildValue("s#", &ciphertext, dlen); 
}

static PyObject * cipher_decrypt(PyObject * self, PyObject * args) {
    void * py_cobj;
    unsigned char * data;
    unsigned int dlen;

    if(!PyArg_ParseTuple(args, "Os#", &py_cobj, &data, &dlen))
        return NULL;

    gcry_cipher_hd_t * cobj;
    cobj = (gcry_cipher_hd_t *) PyCObject_AsVoidPtr(py_cobj);

    unsigned char plaintext[dlen];
    int error = 0;

    error = gcry_cipher_decrypt(*cobj, (unsigned char *) &plaintext, 
        dlen, data, dlen);
    if(error) fprintf(stderr, "Decrypt error: %d\n", error);

    return Py_BuildValue("s#", &plaintext, dlen); 
}

static PyMethodDef CipherMethods [] = {
    {"algo_list", cipher_algo_list, METH_VARARGS, "Map algo names to IDs"},
    {"mode_list", cipher_mode_list, METH_VARARGS, "Map mode names to IDs"},
    {"get_info", cipher_get_info, METH_VARARGS, "Get info about a cipher."},
    {"encrypt", cipher_encrypt, METH_VARARGS, "Encrypt some plaintext."},
    {"decrypt", cipher_decrypt, METH_VARARGS, "Decrypt some ciphertext."},
    {"init", cipher_init, METH_VARARGS, "Make a cipher object."},
    {NULL, NULL, 0, NULL}
};

PyMODINIT_FUNC init_Cipher(void) {
    (void) Py_InitModule("_Cipher", CipherMethods);
    (void) gcry_check_version("1.2.0");
}
