# Gcrypt.py - part of libgcrypt-py
# Copyright 2004 (C) Nat Tuck
# Licenced under the GNU LGPL 2.1+, see COPYING.txt for details.

import _Gcrypt._Random

def GenRandom(length, strength = 0):
    if(strength == 0):
        return _Gcrypt._Random.gen_weak(length)
    if(strength == 1):
        return _Gcrypt._Random.gen_medium(length)
    if(strength == 2):
        return _Gcrypt._Random.gen_strong(length)
    raise "The second argument to GenRandom must be one of 0, 1, 2"

import _Gcrypt._Hash

class Hash:
    def __init__(self, algo):
        self._hdic = _Gcrypt._Hash.algo_list()
        self._hobj = _Gcrypt._Hash.init(self._hdic[algo])

    def Add(self, data):
        _Gcrypt._Hash.add(self._hobj, data)

    def Hash(self):
        return _Gcrypt._Hash.hash(self._hobj)

    def HexHash(self):
        from binascii import hexlify
        return hexlify(self.Hash())

import _Gcrypt._Cipher

class Cipher:
    def __init__(self, algo, mode, key, iv = None):
        self._adic = _Gcrypt._Cipher.algo_list()
        self._mdic = _Gcrypt._Cipher.mode_list()
        self._info = _Gcrypt._Cipher.get_info(self._adic[algo])
        self._mode = mode
        self._what = None
        
        if(iv == None and mode != 'ECB'):
            iv = GenRandom(self._info['blocksize'], 0)
        if(iv != None and len(iv) != self._info['blocksize']):
            raise "Invalid length for IV. length != blocksize"
        if(len(key) != self._info['keysize']):
            raise "Invalid length for key. length != keysize"
        
        self._iv = iv
        self._cipher = _Gcrypt._Cipher.init(
            self._adic[algo], self._mdic[mode], key, iv)

    def _pad(self, data):
        bs = self._info['blocksize']
        padding = bs - (len(data) % bs)
        data += chr(padding) * padding
        if(len(data) % bs != 0):
            raise "Padding didn't work."
        return data

    def _unpad(self, data):
        padding = ord(data[-1:])
        data = data[:-padding]
        return data

    def Encrypt(self, pt):
        if(self._mode != 'ECB'):
            if(self._what == "Dec"):
                raise "You need two handles to enc/dec with this mode."
            else:
                self._what = "Enc"
        if(self._mode != 'CBC-CTS'):
            pt = self._pad(pt)
        return _Gcrypt._Cipher.encrypt(self._cipher, pt)

    def Decrypt(self, ct):
        if(self._mode != 'ECB'):
            if(self._what == "Enc"):
                raise "You need two handles to enc/dec with this mode."
            else:
                self._what = "Dec"
        pt = _Gcrypt._Cipher.decrypt(self._cipher, ct)
        if(self._mode != 'CBC-CTS'):
            pt = self._unpad(pt)
        return pt

import _Gcrypt._Pk

class PkPublic:
    def __init__(self, sexp, cobj = 0):
        if(sexp):
            self._key = _Gcrypt._Pk.load_sexp(sexp)
        elif(cobj):
            self._key = cobj
        else:
            raise "Yea, that's not gonna work"
    
    def Encrypt(self, msg):
        return _Gcrypt._Pk.encrypt(msg, self._key)

    def VerifyHash(self, sig, hash, algo):
        return _Gcrypt._Pk.verify(sig, hash, self._key, algo.lower())

    def Verify(self, sig, msg, algo):
        hash = Hash(algo)
        hash.Add(msg)
        return self.VerifyHash(sig, hash.Hash(), algo)
    
    def Dump(self):
        return _Gcrypt._Pk.dump_sexp(self._key)

class PkPrivate:
    def __init__(self, sexp, cobj = 0):
        if(sexp):
            self._key = _Gcrypt._Pk.load_sexp(sexp)
        elif(cobj):
            self._key = cobj
        else:
            raise "Yea, that's not gonna work"

    def _unpad(self, data):
        # Removes pkcs1 #2 padding.
        # i.e. Remove everything up to and including the
        #      second null (\x00) in the string.
        # Huh? Apparently there's one null less than I thought,
        #      ... whatever.
        a = data.index("\x00")
        return data[a+1:]

    def Decrypt(self, msg):
        pt = _Gcrypt._Pk.decrypt(msg, self._key)
        return self._unpad(pt)

    def SignHash(self, hash, algo):
        return _Gcrypt._Pk.sign(hash, self._key, algo.lower())

    def Sign(self, msg, algo):
        hash = Hash(algo)
        hash.Add(msg)
        return self.SignHash(hash.Hash(), algo)

    def Dump(self):
        return _Gcrypt._Pk.dump_sexp(self._key)

def PkGenerate(algo, bits):
    # Remove the next line when done with testing.
    # _Gcrypt._Pk.use_quick_random()

    if(algo.upper() == "ELGAMAL"):
        algo = "ELG"
    
    (pub, prv) = _Gcrypt._Pk.gen_key_pair(algo, bits) 
    return {"public": PkPublic(0, pub), "private": PkPrivate(0, prv)};
