/*
 * This file is part of libdicto
 *
 * Copyright (C) 2007-2010 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef _DICTO_H
#define _DICTO_H

#include <gst/gst.h>

G_BEGIN_DECLS

#define DICTO_TYPE				(dicto_get_type ())
#define DICTO(obj)				(G_TYPE_CHECK_INSTANCE_CAST ((obj), DICTO_TYPE, Dicto))
#define DICTO_CLASS(klass)		(G_TYPE_CHECK_CLASS_CAST ((klass), DICTO_TYPE, DictoClass))
#define IS_DICTO(obj)			(G_TYPE_CHECK_INSTANCE_TYPE ((obj), DICTO_TYPE))
#define IS_DICTO_CLASS(klass)	(G_TYPE_CHECK_CLASS_TYPE ((klass), DICTO_TYPE))
#define DICTO_GET_CLASS(obj)		(G_TYPE_INSTANCE_GET_CLASS ((obj), DICTO_TYPE, DictoClass))

/* Format description structure */
typedef struct _DictoFormat DictoFormat;
struct _DictoFormat {
	guint id;
	const gchar *desc;
	const gchar *ext;
	const gchar *filter;
	const gchar *muxer;
	const gchar *encoder;
	const gchar *mime;
	guint channels;
	guint rate;
	guint bits;
};

typedef enum {
	DICTO_STATE_UNKNOWN=0,
	DICTO_STATE_STOPPED,
	DICTO_STATE_READY,
	DICTO_STATE_PLAYING,
	DICTO_STATE_RECORDING,
	DICTO_STATE_PAUSED,
	DICTO_STATE_ERROR,
} DictoState;

#include "dicto/dicto-formats-enum.h"

#define	FORMAT_DEFAULT FORMAT_WAV_8K_16B_M

typedef struct _Dicto Dicto;
struct _Dicto {
	GObject parent;

	/* Private */
	gpointer priv;
};

typedef struct _DictoClass DictoClass;
struct _DictoClass {
	GObjectClass parent;

	void (*started_play) (Dicto *dicto, gpointer data);
	void (*stopped_play) (Dicto *dicto, gpointer data);
	void (*started_record) (Dicto *dicto, gpointer data);
	void (*stopped_record) (Dicto *dicto, gpointer data);
	void (*error) (Dicto *dicto, gpointer data);
	void (*position_update) (Dicto *dicto, gdouble pos, gpointer data);
	void (*refresh) (Dicto *dicto, gpointer data);
	void (*ready) (Dicto *dicto, gpointer data);
	void (*level) (Dicto *dicto, gdouble level, gpointer data);
	void (*paused_play) (Dicto *dicto, gpointer data);
};

GType dicto_get_type(void);

Dicto *dicto_new(gchar *basedir);
Dicto *dicto_new_full(gchar *basedir, DictoFormats f, gchar *sink, gchar *source);

gboolean dicto_set_basedir(Dicto *d, const gchar *basedir);

/* Recording */
gboolean dicto_record(Dicto *d, const gchar *file);
gboolean dicto_record_timeout(Dicto *d, const gchar *file, gdouble secs);
gboolean dicto_toggle_record(Dicto *d, const gchar *file);
gboolean dicto_pause_record(Dicto *d);

/* Playback */
gboolean dicto_prepare(Dicto *d, const gchar *file);
gboolean dicto_play(Dicto *d, const gchar *file);
gboolean dicto_pause(Dicto *d);

gboolean dicto_stop(Dicto *d);
gboolean dicto_seek(Dicto *d, gdouble pos);

/* Management */
gboolean dicto_delete_full(Dicto *d, const gchar *file, GError **err);
gboolean dicto_delete(Dicto *d, const gchar *file);

DictoState dicto_get_state(Dicto *d);

gchar *dicto_filename_time_new(Dicto *d, time_t t);

gboolean dicto_refresh_list(Dicto *d);
GHashTable *dicto_notes_get_list(Dicto *d);
gboolean dicto_remove(Dicto *d, const gchar *file);

gboolean dicto_set_format(Dicto *d, DictoFormats nf);

gdouble dicto_normalize_dB(gdouble dB);

const DictoFormat *dicto_get_format(Dicto *d, DictoFormats f);

gboolean dicto_get_position(Dicto *d, gdouble *length, gdouble *position);

gchar *dicto_get_tag(Dicto *d, const gchar *tag);
GstTagSetter *dicto_get_tag_setter(Dicto *d);

gboolean dicto_recording_format_add(Dicto *d, guint f_id, DictoFormat *f);
gboolean dicto_recording_format_remove(Dicto *d, guint f_id);

G_END_DECLS

#endif
