/*
 * tileset.c
 */

/* Includes */

#include <gtk/gtk.h>
#include <sys/stat.h>
#include <string.h>

#include "tileset.h"

const char *image_paths[] = {
  "/usr/share/" APPNAME "/tilesets/stones/",   // final installation path 
  "tilesets/stones/",                          // local path for testing
  ""
};

#define EMPTY  "empty.png"
#define BOARD  "board.png"
#define TILE   "tile%02ds.png"

/* draw a tile from the tileset onto the board */
void tileset_draw_tile(tileset_t *tileset, int tile, int xc, int yc) {
  GdkPixbuf *board_pixbuf = gtk_image_get_pixbuf(tileset->board);
  GdkPixbuf *tile_pixbuf = gtk_image_get_pixbuf(tileset->tile[tile]);

  // check board image parameters
  int board_channels = gdk_pixbuf_get_n_channels(board_pixbuf);
  int board_rowstride = gdk_pixbuf_get_rowstride(board_pixbuf);
  int board_width = gdk_pixbuf_get_width(board_pixbuf);
  int board_height = gdk_pixbuf_get_height(board_pixbuf);
  unsigned char *board_pixels = gdk_pixbuf_get_pixels(board_pixbuf);

  // check board image parameters
  g_assert (gdk_pixbuf_get_colorspace(board_pixbuf) == GDK_COLORSPACE_RGB);
  g_assert (gdk_pixbuf_get_bits_per_sample(board_pixbuf) == 8);
  // no need to do this check as either cases are working
  //  g_assert (gdk_pixbuf_get_has_alpha(board_pixbuf));
  //  g_assert (board_channels == 4);

  // check tile image parameters
  int tile_channels = gdk_pixbuf_get_n_channels(tile_pixbuf);
  int tile_rowstride = gdk_pixbuf_get_rowstride(tile_pixbuf);
  int tile_width = gdk_pixbuf_get_width(tile_pixbuf);
  int tile_height = gdk_pixbuf_get_height(tile_pixbuf);
  unsigned char *tile_pixels = gdk_pixbuf_get_pixels(tile_pixbuf);

  // the tile has alpha channel
  g_assert (gdk_pixbuf_get_colorspace(tile_pixbuf) == GDK_COLORSPACE_RGB);
  g_assert (gdk_pixbuf_get_bits_per_sample(tile_pixbuf) == 8);
  g_assert (gdk_pixbuf_get_has_alpha (tile_pixbuf));
  g_assert (tile_channels == 4);

  // adjust position into pixel offset
  xc = (board_width - tile_width) / 2 - 
    (tileset->offset * 3 / 2) + xc * tileset->offset;
  yc = (board_height - tile_height) / 2 - 
    (tileset->offset * 3 / 2) + yc * tileset->offset;

#if 0 // http://developer.gnome.org/doc/books/WGA/graphics-gdk-pixbuf.html
  // hmm, imho this should work, but it doesn't give any visible result ...
  gdk_pixbuf_composite(tile_pixbuf, board_pixbuf,
		       xc, yc, tile_width, tile_height,
		       0.0, 0.0, 1.0, 1.0, GDK_INTERP_BILINEAR,
		       255);
#else
  int x, y, b;
  for(y = 0; y < tile_height;y++) {
    for(x = 0; x < tile_width;x++) {
      unsigned int alpha = tile_pixels[y*tile_rowstride+x*tile_channels+3];

      for(b=0;b<3;b++) {
#define COMPOSITE(a,b,v) ((guchar) (((guint16) (a) * (unsigned)(v) +\
                                     (guint16) (b) * (255 - (unsigned)(v)))\
                                    >> 8))

	board_pixels[(yc+y)*board_rowstride+
		     ((xc+x)*board_channels)+b] = 
	COMPOSITE(tile_pixels[y*tile_rowstride+x*tile_channels+b],
		  board_pixels[(yc+y)*board_rowstride+
			       ((xc+x)*board_channels)+b], alpha);
      }
    }
  }
#endif  

  gtk_widget_queue_draw(GTK_WIDGET(tileset->board));
}

static gchar*
tileset_file_exists(const gchar *dir, const gchar *file) {
  gchar *fullname;
  struct stat s;
  gint status;

  fullname = (gchar*)g_malloc(strlen(dir)+1+strlen(file)+1);
  strcpy(fullname, dir);
  strcat(fullname, G_DIR_SEPARATOR_S);
  strcat(fullname, file);

  status = stat(fullname, &s);
  if(status == 0 && S_ISREG(s.st_mode))
    return fullname;

  g_free(fullname);
  return NULL;
}


static GtkImage *tileset_load_image(const char *name) {
  gint i=0;
  gchar *fullname;
  GtkImage *img;

  do {
    if((fullname = tileset_file_exists(image_paths[i], name))) {
      img = GTK_IMAGE(gtk_image_new_from_file(fullname));
      g_free(fullname);
      return img;
    }

    i++;
  } while(image_paths[i][0]);

  return NULL;
}

void tileset_load(tileset_t *tileset) {
  int i;

  /* TODO: get these from config files */
  strcpy(tileset->name, "Stones");
  tileset->offset = 65;
  
  for(i=0;i<16;i++) {
    gchar str[16];
    sprintf(str, TILE, i);
    tileset->tile[i] = tileset_load_image(str);
    g_assert(tileset->tile[i] != NULL);
  }
  
  tileset->board = tileset_load_image(BOARD);
  g_assert(tileset->board != NULL);

  tileset->empty = tileset_load_image(EMPTY);
  g_assert(tileset->empty != NULL);

  //  gtk_misc_set_padding(GTK_MISC(tileset->board), 0, 0);
  //  gtk_misc_set_alignment(GTK_MISC(tileset->board), 0, 0);
}

void tileset_reload(tileset_t *tileset) {
  GtkWidget *board_eventbox;
  int i;

  /* unload all graphics and reload them */
  g_assert(tileset->board != NULL);
  board_eventbox = GTK_WIDGET(tileset->board)->parent;
  gtk_widget_destroy(GTK_WIDGET(tileset->board));
  tileset->board = NULL;

  g_assert(tileset->empty != NULL);
  gtk_widget_destroy(GTK_WIDGET(tileset->empty));
  tileset->empty = NULL;

  for(i=0;i<16;i++) {
    if(tileset->tile[i]) {
      gtk_widget_destroy(GTK_WIDGET(tileset->tile[i]));
      tileset->tile[i] = NULL;
    }
  }

  tileset_load(tileset);

  /* re-attach board image to parent event box */
  gtk_container_add(GTK_CONTAINER(board_eventbox), 
		    GTK_WIDGET(tileset->board));

  gtk_widget_show_all(GTK_WIDGET(tileset->board));
}

void tileset_get_slot(tileset_t *tileset, int mx, int my, int *sx, int *sy) {

  GdkPixbuf *board_pixbuf = gtk_image_get_pixbuf(tileset->board);

  *sx = *sy = -1;   // asume click was outside slot

  /* search slot */

  mx -= gdk_pixbuf_get_width(board_pixbuf)/2;
  mx += 2*tileset->offset;
  if((mx >= 0)&&(mx < 4*tileset->offset))
    *sx = mx/tileset->offset;

  my -= gdk_pixbuf_get_height(board_pixbuf)/2;
  my += 2*tileset->offset;
  if((my >= 0)&&(my < 4*tileset->offset))
    *sy = my/tileset->offset;
}
