/*
 * stack.c
 */

#include <hildon/hildon-banner.h>
#include <gtk/gtk.h>

#include "stack.h"

/* move a tile from the stack into the "selected" frame */
void stack2selected(stack_t *stack, int item) {
  GdkPixbuf *pixbuf;
  GtkWidget *parent = stack->item[item]->parent;
 
  /* make game engine aware of changes ... */
  game_stack_remove(stack->game, item);
  
  /* ... and change the gui accordingly */
  
  /* remove tile from stack ... */
  gtk_widget_destroy(stack->item[item]);
  
  /* ... and replace with empty place */
  pixbuf = gtk_image_get_pixbuf(stack->tileset->empty);
  
  gtk_container_add(GTK_CONTAINER(parent), 
	    (stack->item[item] = gtk_image_new_from_pixbuf(pixbuf)));
  
  gtk_widget_show_all(GTK_WIDGET(stack->item[item]));
  
  /* add tile to "selected" gui */
  selected_add(stack->selected, item);
}

static gboolean
stack_select_callback(GtkWidget *widget, GdkEventButton *event, 
		      gpointer data) {  
  stack_t *stack = (stack_t*)data;
  int i, item = -1;

  if(stack->game->state == STATE_IDLE) {
    hildon_banner_show_information(GTK_WIDGET(stack->window), NULL, 
				   "No game running. Please select "
				   "\"New\" from the menu!");
    return TRUE;
  }

  if(stack->game->state != STATE_HUMAN_SELECTS) {
    hildon_banner_show_information(GTK_WIDGET(stack->window), 
				   NULL, "Not your turn to select!");
    return TRUE;
  }

  /* figure out which stack item is in the event box */
  /* the user just clicked on */
  for(i=0;i<16;i++)
    if(stack->item[i]->parent == widget)
      item = i;

  g_assert(item != -1);

  if(stack->selected->tile == LOC_EMPTY) {

    if(!game_tile_on_stack(stack->game, item))
      hildon_banner_show_information(GTK_WIDGET(stack->window), 
				     NULL, "Already taken");
    else {
      stack2selected(stack, item);

      stack->game->state = STATE_CPU_PLACES;
      g_signal_emit_by_name(G_OBJECT(stack->window), 
			    "statusbar_message_state");

      /* send signal to main window ... */
      g_signal_emit_by_name(G_OBJECT(stack->window), "cpu_places");
    }
  } else
    hildon_banner_show_information(GTK_WIDGET(stack->window), 
				   NULL, "Already one selected");


  return TRUE;
}

GtkWidget *stack_create_gui(stack_t *stack, tileset_t *tileset, 
			    game_state_t *game, HildonWindow *window,
			    selected_t *selected) {
  int x, y;
  GtkWidget *frame, *stackbox, *eventbox;
  GdkPixbuf *pixbuf;

  stack->tileset = tileset;
  stack->game = game;
  stack->window = window;
  stack->selected = selected;
  

  /* tile stack */
  frame = gtk_frame_new(" Remaining ");

  stackbox = gtk_table_new(STACK_HEIGHT, STACK_WIDTH, TRUE);
  gtk_container_set_border_width(GTK_CONTAINER(stackbox), 0);
  gtk_container_add(GTK_CONTAINER(frame), stackbox);

  for(y=0;y<STACK_HEIGHT;y++) {
    for(x=0;x<STACK_WIDTH;x++) {
      eventbox = gtk_event_box_new();
      g_signal_connect(G_OBJECT(eventbox), 
		       "button_press_event",
		       G_CALLBACK (stack_select_callback),
		       stack);
      
      gtk_table_attach(GTK_TABLE(stackbox),
		       eventbox,
		       x,x+1,y,y+1,
		       (GtkAttachOptions)(GTK_EXPAND),
		       (GtkAttachOptions)(GTK_FILL),
		       0, 0);

      if(game_tile_on_stack(stack->game, STACK_WIDTH*y+x)) 
	pixbuf = gtk_image_get_pixbuf(stack->tileset->tile[STACK_WIDTH*y+x]);
      else 
	pixbuf = gtk_image_get_pixbuf(stack->tileset->empty);

      gtk_container_add(GTK_CONTAINER(eventbox), 
	(stack->item[STACK_WIDTH*y+x] = gtk_image_new_from_pixbuf(pixbuf)));
    }
  }

  return frame;
}

/* update graphic represenation of the entire stack (e.g. after */
/* new game or tileset change) */
void stack_update(stack_t *stack) {
  int x, y;

  for(y=0;y<STACK_HEIGHT;y++) {
    for(x=0;x<STACK_WIDTH;x++) {
      GtkWidget *parent = stack->item[STACK_WIDTH*y+x]->parent;
      GdkPixbuf *pixbuf;
      
      /* remove tile from stack ... */
      gtk_widget_destroy(stack->item[STACK_WIDTH*y+x]);
      
      /* and replace with new image */
      if(game_tile_on_stack(stack->game, STACK_WIDTH*y+x)) 
	pixbuf = gtk_image_get_pixbuf(stack->tileset->tile[STACK_WIDTH*y+x]);
      else 
	pixbuf = gtk_image_get_pixbuf(stack->tileset->empty);

      gtk_container_add(GTK_CONTAINER(parent), 
	(stack->item[STACK_WIDTH*y+x] = gtk_image_new_from_pixbuf(pixbuf)));

      gtk_widget_show_all(GTK_WIDGET(parent));
    }
  }
}
