/*
 * game.c
 */

#include <hildon/hildon-banner.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>   // for sleep
#include <string.h>   // for memcpy
#include <glib.h>

#include "game.h"

#define MAX_DEPTH 3

#define PLACE 0
#define GIVE  1

int game_get_tile_at(game_state_t *game, int x, int y) {
  return game->loc[y][x];
}

gboolean game_tile_on_stack(game_state_t *game, int tile) {
  return game->stack & (1<<tile);
}

void game_place_tile(game_state_t *game, int tile, int x, int y) {
  game->loc[y][x] = tile;
}

void game_stack_remove(game_state_t *game, int tile) {
  game->stack &= ~(1<<tile);
}

int game_placed(game_state_t *game) {
  int s,x,y;

  for(s=y=0;y<4;y++) 
    for(x=0;x<4;x++)
      if(game->loc[y][x] != LOC_EMPTY)
	s++;

  return s;
}

/* check board and return true if it is a winning situation */
int game_check(game_state_t *game) {
  int mask_0, mask_1;
  int x,y;

  /* vertical check */
  for(x=0;x<4;x++) {
    mask_0 = mask_1 = 0;

    for(y=0;y<4;y++) {
      mask_0 |=          game->loc[y][x];
      mask_1 |=  0x0f & ~game->loc[y][x];
    }

    /* all four tiles present (no empty one found?) and mask contains */
    /* a bit set/unset in all entries? */
    if( (!(mask_0 & 0x80)) && (((mask_0 != 0x0f) || (mask_1 != 0x0f))))
      return GAME_WON;
  }

  /* horizontal check */
  for(y=0;y<4;y++) {
    mask_0 = mask_1 = 0;

    for(x=0;x<4;x++) {
      mask_0 |=          game->loc[y][x];
      mask_1 |=  0x0f & ~game->loc[y][x];
    }

    /* all four tiles present (no empty one found?) and mask contains */
    /* a bit set/unset in all entries? */
    if( (!(mask_0 & 0x80)) && (((mask_0 != 0x0f) || (mask_1 != 0x0f))))
      return GAME_WON;
  }


  /* diagonal check */
  for(y=0;y<2;y++) {
    mask_0 = mask_1 = 0;

    for(x=0;x<4;x++) {
      int ty = (1-y)*3+((2*y-1)*x);

      mask_0 |=          game->loc[ty][x];
      mask_1 |=  0x0f & ~game->loc[ty][x];
    }

    /* all four tiles present (no empty one found?) and mask contains */
    /* a bit set/unset in all entries? */
    if( (!(mask_0 & 0x80)) && (((mask_0 != 0x0f) || (mask_1 != 0x0f))))
      return GAME_WON;
  }

  /* check if entire board filled up */
  if(game_placed(game) == 16)
    return GAME_DRAW;

  /* neither win, nor draw -> game continues */
  return GAME_RUN;
}

/* return number of tiles still in this stack */
int stack_tiles(stack_state_t stack) {
  int i,t;
  for(t=i=0;i<16;i++)
    if(stack & (1<<i))
      t++;

  return t;
}

float place_any(game_state_t game, int depth, int give, int *);

float place_one(game_state_t *game, int tile, int depth, 
		int give, int *placing) {
  int x,y;
  float result;
  float avg_result = 0.0;
  float best_result = -1.0;

  /* try all possible places */
  for(y=0;y<4;y++) {
    for(x=0;x<4;x++) {
      if(game->loc[y][x] == LOC_EMPTY) {
	/* move tile from stack here */
	game->loc[y][x] = tile;         /* place on game */
	
	/* we've placed one more tile on the board. */
	/* check if this is a win situation */
	if(game_check(game) == GAME_WON) {
	  //	  printf("wins at %d %d/%d\n", depth, x,y);
	  
	  if(give) {
	    //	    printf("-");

	    /* if this happens while we are in a "give" cycle, just */
	    /* stop thinking about this tile as this is max bad */
	    game->loc[y][x] = LOC_EMPTY; /* remove from board */
	    return 0.0;
	  } else {
	    //	    printf("+");

	    /* if this happens while we are in the "put" cycle it's */
	    /* good as we can win with this tile */

	    if(!depth) {
	      printf("i can win!!\n");
	      *placing = 4*y+x;
	      /* -> return this place */
	    }

	    game->loc[y][x] = LOC_EMPTY; /* remove from board */
	    return 1.0;
	  }

	} else {
	  /* this tile could just be placed without immediate win. */
	  /* so see what happens if further tiles are added */

	  result = place_any(*game, depth+1, !give, NULL);
	  avg_result += result;

	  if(placing) {
	    if(result > best_result) {
	      best_result = result;
	      //	      printf("x=%d, y=%d\n", x,y);
	      *placing = 4*y+x;
	    }
	  }
	}

	game->loc[y][x] = LOC_EMPTY; /* remove from board */
      }
    }
  }

  /* divide the result by the number of stack items (add one, since the */
  /* one we are currently trying to place has already been removed, so */
  /* the result ranges from 1.0 (max good) to 0.0 (max bad) */
  return avg_result/(stack_tiles(game->stack)+1);
}
  
/* try to place any of the stack items on the board */
/* this is creating by purpose a local copy if the board */
/* (game is not a pointer!) */
float place_any(game_state_t game, int depth, int give, int *val) {
  int t;

  /* all 16 items on the stack will get a rating */
  float cur_rating, avg_rating = 0.0, best_rating = -1.0;

  /* we don't have an answer as we aren't supposed to search deeper. */
  /* so just return a neutral 0.5 score */
  if(depth == MAX_DEPTH) {
    //    printf(".");
    return 0.5;
  }

  /* stack is empty, end of recursion ... */
  if(!game.stack) {
    printf("all tiles used up -> draw game\n");
    //    printf(".");
    return 0.5;
  }

  if(!depth) 
    printf("       ");
  
  /* try all tiles in stack */
  for(t=0;t<16;t++) {

    /* is this tile still on the stack? */
    if(game.stack & (1<<t)) {
      //      printf("trying to place %s\n", tile_ascii(t));

      game.stack &= ~(1<<t);       /* remove from stack */
      cur_rating = place_one(&game, t, depth, give, NULL);
      game.stack |= (1<<t);        /* add to stack */

      avg_rating += cur_rating;

      if(!depth) {
	/* remember the best tile we found */
	if(cur_rating > best_rating) {
	  best_rating = cur_rating;
	  
	  if(val) *val = t;
	}
	
	printf("%.2f ", cur_rating);
      }
    }
  }

  if(!depth) 
    printf("\n");

  return avg_rating/stack_tiles(game.stack);
}

static gboolean
give_thread_complete_idle(gpointer data) {
  game_state_t *game = (game_state_t*)data;

  /* game thread isn't running anymore */
  game->thread = NULL;

  /* close the prograss bar */
  gtk_widget_destroy(game->animation);

  g_signal_emit_by_name(G_OBJECT(game->window), "cpu_has_selected");

  return FALSE;
}

gpointer give_thread(gpointer data) {
  game_state_t *game = (game_state_t*)data;

  /* just return random tile if only 1 or 2 placed yet */
  if(game_placed(game) < 3)
    game->async_result = rand()%16;
  else
    /* start game analysis */
    place_any(*game, 0, GIVE, &game->async_result);

  g_idle_add((GSourceFunc)give_thread_complete_idle, data);
  return NULL;
}

void game_give(game_state_t *game) {

  /* there must no engine be running already */
  g_assert(game->thread == NULL);

  /* create a prograss dialog */
  game->animation = hildon_banner_show_animation(GTK_WIDGET(game->window), 
						 NULL, "Selecting ..."); 

  game->thread = g_thread_create(give_thread, game, FALSE, NULL);
}

static gboolean
place_thread_complete_idle(gpointer data) {
  game_state_t *game = (game_state_t*)data;

  /* game thread isn't running anymore */
  game->thread = NULL;

  /* close the prograss bar */
  gtk_widget_destroy(game->animation);

  g_signal_emit_by_name(G_OBJECT(game->window), "cpu_has_placed");

  return FALSE;
}

gpointer place_thread(gpointer data) {
  game_state_t *game = (game_state_t*)data;

  if(!game_placed(game))
    game->async_result = rand()%16;
  else
    /* start game analysis */
    place_one(game, game->tile, 0, PLACE, &game->async_result);

  g_idle_add((GSourceFunc)place_thread_complete_idle, data);
  return NULL;
}

void game_place(game_state_t *game, int tile) {

  /* there must no engine be running already */
  g_assert(game->thread == NULL);

  /* create a prograss dialog */
  game->tile = tile;
  game->animation = hildon_banner_show_animation(GTK_WIDGET(game->window), 
						 NULL, "Placing ..."); 

  game->thread = g_thread_create(place_thread, game, FALSE, NULL);
}


void game_restart(game_state_t *game) {
  int x,y;

  game->state = STATE_IDLE;

  /* stack is full and ... */
  game->stack = 0xffff;

  /* ... board is empty */
  for(y=0;y<4;y++)
    for(x=0;x<4;x++)
      game->loc[y][x] = LOC_EMPTY;
  
}

void game_init(game_state_t *game) {
  int x,y;

  game->thread = NULL;

  /* setup stack by removing used tiles */
  for(game->stack=0xffff,y=0;y<4;y++)
    for(x=0;x<4;x++)
      if(game->loc[y][x] != LOC_EMPTY)
	game->stack &= ~(1<<game->loc[y][x]);
}
