using Gtk;
using Gdk;

public class Discovery
{
	private HashTable<string,string> connection;
	private bool got_body = false;
	private Knots knots;
	private string[] servers;
	
	public Discovery(Knots knots)
	{
		this.knots = knots;
	}
	
	public int server_count()
	{
		return servers.length;	
	}
	
	public void select_server(int selected)
	{
		if (selected < server_count())
			load_connection(servers[selected], true, false);
	}
	
	public string serverlist_xml()
	{
		string xml = "<?xml version=\"1.0\" encoding=\"UTF-8\" ?><root><items>";
		int index = 0;
		foreach(string server in servers)
		{
			string[] tokens = server.split("/");
			if (tokens[0].strip() == "")
				tokens[0] = tokens[3];
			xml += "<item><server>" + tokens[0] + "</server><id>" + index.to_string() + "</id><type>server</type></item>";
			index += 1;
		}
		return xml + "</items><pages><current>1</current><total>1</total></pages></root>";
	}
	
	public bool load_connection(string? address, bool save, bool test)
	{
		if (address != null && address.str("/") != null)
		{
			string[] tokens = address.split("/");
			string[] keys = {"name", "ssl", "port", "address"};
			if (tokens.length == 4)
			{
				connection = new HashTable<string,string>(str_hash, str_equal);
				for (int i = 0; i < 4; i++)
				{
					connection.insert(keys[i], tokens[i]);
				}
				connection.insert("user", "guest");
				connection.insert("password", "guest");
				if (!test || test_connection())
				{
					if (save)
						knots.get_settings().set_value("last_server", address);
					return true;
				}
				else
				{
					connection = null;
					return false;
				}
			}
		}
		return false;
	}
	
	public void add_server(string server)
	{
		if (server != null)
		{
			foreach(string s in servers)
			{
				if (s.split("/")[2] == server.split("/")[2] && s.split("/")[3] == server.split("/")[3])
					return;
			}
			servers += server;
		}
	}
	
	public void discover_server_address()
	{
		this.connection = null;
		servers = new string[0];
		string[] address = new string[0];
		string discover = null;
		try
		{
			GLib.Process.spawn_command_line_sync("/usr/share/knots/discover" + (knots.get_settings().get_value("discover_wait") != null ? " " + knots.get_settings().get_value("discover_wait") : ""), out discover);
		}
		catch (GLib.SpawnError se)
		{
			print("ERROR: %s\n", se.message);
		}
		if (discover != null)
		{
			string[] serverlist = discover.strip().split("\n");
			foreach(string server in serverlist)
			{
				if (server.str("/") != null)
				{
					address += server;
				}
			}
		}
		if (knots.get_settings().get_value("last_server") != null)
			address += knots.get_settings().get_value("last_server");
		foreach(string server in address)
		{
			if (load_connection(server, false, true))
			{
				add_server(server);
			}	
		}
		if (server_count() == 0)
		{
			string server = show_address_dialog();
			if (server != null && load_connection(server, false, true))
			{
				add_server(server);
			}
		}
		if (server_count() == 1)
		{
			knots.get_settings().set_value("last_server", servers[0]);
			load_connection(servers[0], true, false);
			knots.update_transcode_profiles();
			
		}
	}
	
	public string? show_address_dialog()
	{
		MessageDialog dialog = new MessageDialog.with_markup(null, 0, Gtk.MessageType.QUESTION, Gtk.ButtonsType.OK_CANCEL, "Please enter server address:");
		Gtk.Entry server_address = new Gtk.Entry();
		server_address.set_text("192.168.0.1:1978");
		CheckButton cb = new CheckButton.with_label("Use ssl?");
		dialog.vbox.pack_start(server_address, false, true, 1);
		dialog.vbox.pack_start(cb, false, true, 1);
		server_address.show();
		var response = dialog.run();
		if (response == ResponseType.OK)
		{
			string[] address_port = server_address.get_text().split(":");
			if (address_port.length == 2)
			{
				string address = "/0/" + address_port[1] + "/" + address_port[0];
				dialog.destroy();
				return address;
			}
		}
		dialog.destroy();
		return null;
	}
	
	public bool test_connection()
	{
		if (connected())
		{
			string test = fetch("/external/test_connection");
			if (test != null && test.str("KNOTS_OK") != null)
			{
				return true;
			}
		}
		return false;
	}
	
	public void disconnect()
	{
		this.connection = null;
		knots.disable_transcode_menu(true);
	}
	
	public string fetch(string url)
	{
		return fetch_from_address(url, address());
	}
	
	public string fetch_from_address(string url, string? server_address)
	{
		
		got_body = false;
		var session = new Soup.SessionSync ();
		session.timeout = 30;
		string uri = (uses_ssl() ? "https" : "http") + "://" + server_address + url; 
		var message = new Soup.Message ("GET", uri);
		session.authenticate +=  (sess, type, realm, user, pass) => {
			stdout.printf("Auth required\n");
		};
		message.got_body += (message) => {
			got_body = true;
		};
		message.finished += () => {
			got_body = true;
		};
		session.send_message (message);
		while (!got_body)
			GLib.Thread.usleep(50);
		if (message.status_code == 200)
		{
			string body = message.response.body;
			if (body != null)
				return xml_cleanup(message.response.body);
		}
		return "";
	}
	
	public string xml_cleanup(string data)
	{
		if (data != null && data.str("<?xml") != null)
		{
			return data.split("</root>")[0] + "</root>";
		}
		else
		if (data != null && data.str("END_CONTENT") != null)
		{
			return data.split("END_CONTENT")[0];
		}
		return data;
	}
	
	public bool connected()
	{
		return connection != null;
	}
	
	public bool uses_ssl()
	{
		if (connection != null)
		{
			return ((string)connection.lookup("ssl")).to_int() == 1;
		}
		return false;
	}
	
	public string? username()
	{
		if (connected())
			return (string)connection.lookup("user");
		return null;
	}
	
	public string? password()
	{
		if (connected())
			return (string)connection.lookup("password");
		return null;
	}
	
	public string? ip()
	{
		if (connected())
			return (string)connection.lookup("address");
		return null;
	}
	
	public int port()
	{
		if (connected())
			return ((string)connection.lookup("port")).to_int();
		return -1;
	}
	
	public string? address()
	{
		if (connected())
			return ip() + ":" + port().to_string();
		return null;
	}
	
	public string? server_name()
	{
		if (connected())
			return (string)connection.lookup("name");
		return null;
	}
	
	
}
