using GLib;
using Gtk;
using Hildon;
using Soup;
using Xml;
using Osso;

public class Player : PageWidget
{
	string player_id;
	private string address;
	private string fetch_address;
	private int playing;
	private int last_played;
	private HashTable status;
	private Hildon.Seekbar seekbar;
	private int seekable = -1;
	private bool automatic_seek;
	private Gtk.Label info;
	private GLib.TimeVal last_seek;
	public int ITEM = 0;
	public int ARTIST = 1;
	public int ALBUM = 2;
	public int PLAYLIST = 3;
	public int RETURN_PAGE;
	Pid? player_pid;
	GLib.IOChannel io;
	private bool fullscreen;
	Scroller scroller;
	private int[] restart;
	private bool updating;
	private bool seeking;
	private bool stopping;
	private int rotation_before_playback;
	Gtk.HBox timedisplay;
	Gtk.HBox hb;
	Label current_position;
	Scroller now_playing;
	bool waiting_for_response;
	
	construct {
		playing = 0;
		last_played = -1;
		restart = new int[2];
		stopping = false;
	}
	
	public override void add_signals()
	{
		get_toolbar_button(0).clicked += () => {
			go_back();
		};
		get_toolbar_button(1).clicked += () => {
			on_stop();
		};
		get_toolbar_button(2).clicked += () => {
			fetch_lyrics();
		};
		get_toolbar_button(3).clicked += () => {
			show_info();
		};
		get_toolbar_button(4).clicked += () => {
			loop(true);
		};
		get_toolbar_button(5).clicked += () => {
			loop(false);
		};
		get_toolbar_button(6).clicked += () => {
			rate();
		};
		get_toolbar_button(7).clicked += () => {
			previous_item();
		};
		get_toolbar_button(8).clicked += () => {
			next_item();
		};
		seekbar.value_changed += () => {
			seek(null);
		};
		seekbar.move_slider += () => {
			seeking = true;
			update_timedisplay((int)(seekbar.get_position() * (double)seekbar.get_total_time()));
		};
	}
	
	public void rate()
	{
		int now_playing = this.playing;
		if (!is_playing_music())
		{
			get_knots().debug("Rating video so pausing it. Will prolly stop.\n", null);
			mplayer_key('p');
			queue_draw();
		}
		MessageDialog dialog = new MessageDialog.with_markup(null, 0, Gtk.MessageType.QUESTION, Gtk.ButtonsType.OK_CANCEL, "Rate");
		dialog.set_image(new Gtk.Image.from_file("/usr/share/knots/knots_button_rate.png"));
		Hildon.NumberEditor ne = new Hildon.NumberEditor(0, 10);
		ne.value = 5;
		dialog.vbox.pack_start(ne, true, true, 1);
		dialog.vbox.show_all();
		var response = dialog.run();
		if (response == ResponseType.OK)
		{
			get_knots().get_connection().set_url_cache(false);
			get_knots().get_connection().fetch_from_address("/external/rate?id=" + now_playing.to_string() + "&rating=" + ne.value.to_string(), fetch_address);
			if (is_playing_music())
			{
				if (get_toolbar_button(2).visible)
					show_info();
			}
			else
			{
				get_knots().get_info().show_info(null);
			}
		}
		dialog.destroy();
		if (!is_playing_music())
			mplayer_key('p');
	}
	
	public void change_aspect(Gtk.CheckMenuItem toggled)
	{
		int index = 0;
		string[] fields = {"none", "16:9", "4:3"};
		var children = ((Gtk.Container)toggled.get_parent()).get_children();
		foreach(Gtk.Widget w in children)
		{
			Gtk.CheckMenuItem cmi = (Gtk.CheckMenuItem)w;
			if (cmi == toggled)
			{
				cmi.active = true;
				get_knots().get_settings().set_value("aspect", fields[index]);
			}
			else
			{
				// Use draw_as_radio as bool to prevent infinite loop
				cmi.draw_as_radio = false;
				cmi.active = false;
				cmi.draw_as_radio = true;
			}
			index += 1;
		}
	}
	
	public void go_back()
	{
		this.get_knots().get_notebook().set_current_page(RETURN_PAGE);
		check_update_need();
	}
	
	public override void orientation_change()
	{
		if (get_toolbar() != null)
		{
			if (get_knots().ORIENTATION == get_knots().LANDSCAPE)
			{
				reorder_child(get_toolbar(), 2);
				((Gtk.VBox)get_content()).reorder_child(timedisplay, 0);
				((Gtk.VBox)get_content()).remove(now_playing);
				now_playing.width_request = 180;
				now_playing.height_request = -1;
				hb.pack_start(now_playing, false, true, 0);
				now_playing.set_scroll_policy(true, false);
			}
			else
			{
				reorder_child(get_toolbar(), 0);
				hb.remove(now_playing);
				now_playing.height_request = 180;
				now_playing.width_request = -1;
				((Gtk.VBox)get_content()).pack_start(now_playing, false, true, 0);
				((Gtk.VBox)get_content()).reorder_child(timedisplay, 2);
				now_playing.set_scroll_policy(false, true);
			}
			reorder_now_playing();
		}
	}
	
	public override Gtk.Widget create_content()
	{
		last_seek = TimeVal();
		seekbar = new Hildon.Seekbar();
		seekbar.height_request = 40;
		Gtk.VBox vbox = new Gtk.VBox(false, 0);
		info = new Gtk.Label("");
		info.set_line_wrap(true);
		get_knots().change_foreground_color(info, "#fff");
		Gtk.Fixed f = new Gtk.Fixed();
		f.put(info, 10, 10);
		scroller = new Scroller(f);
		timedisplay = new Gtk.HBox(false, 0);
		current_position = new Gtk.Label("");
		get_knots().change_foreground_color(current_position, "#fff");
		current_position.width_request = 200;
		timedisplay.pack_start(seekbar, true, true, 0);
		timedisplay.pack_start(current_position, false, false, 0);
		vbox.pack_start(timedisplay, false, true, 0);
		scroller.set_scroll_policy(false, true);
		hb = new Gtk.HBox(false, 0);
		hb.pack_start(scroller, true, true, 0);
		Gtk.Fixed ff = new Gtk.Fixed();
		now_playing = new Scroller(ff);
		now_playing.enable_clicking(false);
		now_playing.hide();
		now_playing.set_scroll_policy(true, false);
		if (get_knots().ORIENTATION == get_knots().LANDSCAPE)
		{
			now_playing.width_request = 180;
			now_playing.height_request = -1;
			hb.pack_start(now_playing, false, true, 0);
		}
		else
		{
			now_playing.height_request = 180;
			now_playing.width_request = -1;
			vbox.pack_start(now_playing, false, true, 0);
		}
		vbox.pack_start(hb, true, true, 0);
		return vbox;
	}
	
	public void loop(bool looped)
	{
		if (looped)
		{
			get_toolbar_button(4).hide();
			get_toolbar_button(5).show();
		}
		else
		{
			get_toolbar_button(4).show();
			get_toolbar_button(5).hide();
		}
		get_knots().get_connection().set_url_cache(false);
		get_knots().get_connection().fetch_from_address("/external/loop?player_id=" + player_id + "&loop=" + looped.to_string(), fetch_address);
	}
	
	public override string[] get_toolbar_items()
	{
		return {"back", "stop", "lyrics", "info", "repeat", "no_repeat", "rate", "prev", "next"};
	}
	
	public void seek(double? new_position)
	{
		if (!automatic_seek && seekable == 1)
		{
			var newtime = TimeVal();
			newtime.get_current_time();
			if ((newtime.tv_sec - last_seek.tv_sec) > 0)
			{
				last_seek = newtime;
				double total_time = (double)seekbar.get_total_time();
				if (total_time > 0)
				{
					double position = (double)seekbar.get_position() / total_time;
					if (new_position != null)
						position = new_position;
					if (position > 1.0)
						position = 1.0;
					else
					if (position < 0)
						position = 0;
					get_knots().get_connection().set_url_cache(false);
					get_knots().get_connection().fetch_from_address("/external/seek?player_id=" + player_id + "&position=" + position.to_string(), fetch_address);
				}
			}
		}
		seeking = false;
	}
	
	public void next_item()
	{
		get_knots().get_connection().set_url_cache(false);
		get_knots().get_connection().fetch_from_address("/external/next_playlist_item?player_id=" + player_id, fetch_address);
		update_status();
		handle_media_change();
	}
	
	public void previous_item()
	{
		get_knots().get_connection().set_url_cache(false);
		get_knots().get_connection().fetch_from_address("/external/previous_playlist_item?player_id=" + player_id, fetch_address);
		update_status();
		handle_media_change();
	}
	
	public void fetch_lyrics()
	{
		string lyrics = get_knots().get_connection().fetch_from_address("/external/lyrics?id=" + playing.to_string(), fetch_address);
		if (lyrics != null && lyrics.strip() != "")
		{
			set_info(lyrics);
			get_toolbar_button(3).show();
			get_toolbar_button(2).hide();
		}
		else
		{
			get_knots().alert("No lyrics available.");
		}
	}
	
	public void set_info(string info)
	{
		this.info.set_markup(info);
		scroller.scroll_to_top();
	}

	public void on_stop()
	{
		if (!stopping)
		{
			mplayer_key('q');
			stopping = true;
		}
		else
		{
			try
			{
				string output;
				GLib.Process.spawn_command_line_sync("killall -9 mplayer", out output);
			}
			catch (GLib.SpawnError se)
			{
				print("ERROR: %s\n", se.message);
			}
		}
	}
	
	public bool update_status()
	{
		get_knots().debug("Updating status\n", null);
		get_knots().get_connection().set_url_cache(false);
		string xml = get_knots().get_connection().fetch_from_address("/external/player_properties?player_id=" + player_id, fetch_address);
		XMList list = new XMList(xml);
		if (list != null && list.itemcount() == 1)
		{
			status = list.items()[0];
			get_knots().debug("Status OK\n", null);
			return status != null;
		}
		return false;
	}
	
	public void set_automatic_position(int position)
	{
		automatic_seek = true;
		seekbar.set_position(position);
		automatic_seek = false;
	}
	
	public void return_to_player()
	{
		this.RETURN_PAGE = 0;
		this.get_knots().get_player().reveal();
	}
	
	public void handle_media_change()
	{
		get_toolbar_button(2).show();
		get_toolbar_button(3).hide();
		this.playing = (get_player_value("media_id")).to_int();
		int index = (get_player_value("playlistindex")).to_int();
		int playlist_length = (get_player_value("playlist_length")).to_int();
		seekable = 0;
		if (get_player_value("seekable") == "true")
		{
			get_knots().debug("Stream not seekable\n", null);
			seekbar.set_total_time((get_player_value("duration")).to_int());
			timedisplay.show();
			update_timedisplay(null);
			update_seekbar();
			seekable = 1;
		}
		if (get_player_value("looped") == "false")
		{
			get_knots().debug("Not looped\n", null);
			get_toolbar_button(4).show();
			get_toolbar_button(5).hide();
		}
		else
		{
			get_knots().debug("Looped\n", null);
			get_toolbar_button(4).hide();
			get_toolbar_button(5).show();
		}
		set_info("");
		if (is_playing_music())
		{
			get_knots().debug("Playing music\n", null);
			get_toolbar_button(7).hide();
			get_toolbar_button(8).hide();
			get_toolbar_button(0).show();
			get_toolbar_button(2).show();
			get_knots().get_connection().set_url_cache(false);
			set_info(get_knots().get_connection().fetch_from_address("/external/info?id=" + this.playing.to_string(), fetch_address));
			if (get_knots().get_settings().get_value("disable_taskswitcher_change") == null)
				generate_app_switcher_icon();
			revert_rotation();
			if (playlist_length > 1)
			{
				highlight_item(index - 1);
			}
		}
		else
		{
			get_knots().debug("Playing video\n", null);
			get_knots().get_window().set_icon(get_knots().get_common().get_app_icon());
			if (get_knots().get_settings().get_value_default_to("landscape_video", "false") == "true")
			{
				rotation_before_playback = get_knots().get_rotation();
				if (rotation_before_playback == 1 || rotation_before_playback == 3)
				{
					get_knots().set_rotation(0);
				}
			}
			get_toolbar_button(0).hide();
			get_toolbar_button(2).hide();
			get_toolbar_button(7).show();
			get_toolbar_button(8).show();
			get_toolbar_button(8).set_sensitive(index < playlist_length);
			get_toolbar_button(7).set_sensitive(index > 1);
		}
		window.set_title("Knots - " + get_player_value("title"));
		scroller.scroll_to_top();
		check_update_need();
		get_knots().debug("Handle media done\n", null);
	}
	
	public void generate_app_switcher_icon()
	{
		if (get_knots().get_common().get_app_icon() != null)
		{
			var session = new Soup.SessionSync ();
			var message = new Soup.Message ("GET", "http://" + fetch_address + "/root/resource_file?type=screenshot&file=" + get_player_value("media_id") + "&mediatype=1");
			message.got_body += (message) => {
				try
				{
					var filename = "/tmp/knots" + get_player_value("media_id");
					FileUtils.set_contents (filename, message.response.body, message.response.length);
					Gtk.Image img2 = new Gtk.Image.from_file(filename);
					FileUtils.unlink(filename);
					if (img2 != null)
					{
						Gdk.Pixbuf knots_icon = get_knots().get_common().get_app_icon().add_alpha(false, 255, 255, 255);
						Gdk.Pixbuf shot = img2.get_pixbuf().scale_simple(44, 44, Gdk.InterpType.BILINEAR); 
						//shot.copy_area (0, 0, 44, 44, knots_icon, 0, 0);
						shot.composite(knots_icon, 10, 5, 44, 44, 10, 5, 1, 1, Gdk.InterpType.BILINEAR, 255);
						get_knots().get_window().set_icon(knots_icon);
					}
				}
				catch (GLib.Error e)
				{
				
				}
			};
			session.send_message (message);
		}
	}
	
	public void show_info()
	{
		get_toolbar_button(2).show();
		get_toolbar_button(3).hide();
		get_knots().get_connection().set_url_cache(false);
		set_info(get_knots().get_connection().fetch_from_address("/external/info?id=" + this.playing.to_string(), fetch_address));
	}
	
	public int get_position_sec()
	{
		return (int)(seekbar.get_position() * seekbar.get_total_time());
	}
	
	public bool play(int id, int playback_type)
	{
		if (!this.is_playing())
		{
			this.waiting_for_response = false;
			fetch_address = this.get_knots().get_connection().address();
			get_toolbar_button(2).show();
			get_toolbar_button(3).hide();
			get_toolbar_button(4).hide();
			get_toolbar_button(5).hide();
			this.rotation_before_playback = -1;
			this.seeking = false;
			this.updating = false;
			this.fullscreen = false;
			this.get_knots().get_player().reveal();
			get_knots().get_browser().get_toolbar_button(4).hide();
			get_knots().get_browser().get_toolbar_button(5).show();
			timedisplay.hide();
			set_automatic_position(0);
			player_id = null;
			string profile = get_knots().get_settings().get_value_default_to("transcoding_profile", "2");
			get_knots().get_connection().set_url_cache(false);
			if (playback_type == ITEM)
			{
				player_id = get_knots().get_connection().fetch_from_address("/external/play?profile=" + profile + "&id=" + id.to_string(), fetch_address).split("\n")[0];
				this.RETURN_PAGE = 1;
			}
			else
			if (playback_type == ARTIST)
			{
				player_id = get_knots().get_connection().fetch_from_address("/external/play_artist?profile=" + profile + "&id=" + id.to_string(), fetch_address).split("\n")[0];
				this.RETURN_PAGE = 1;
			}
			else
			if (playback_type == ALBUM)
			{
				player_id = get_knots().get_connection().fetch_from_address("/external/play_album?profile=" + profile + "&id=" + id.to_string(), fetch_address).split("\n")[0];
				this.RETURN_PAGE = 1;
			}
			else
			if (playback_type == PLAYLIST)
			{
				player_id = get_knots().get_connection().fetch_from_address("/external/play_playlist?profile=" + profile + "&id=" + id.to_string(), fetch_address).split("\n")[0];
				this.RETURN_PAGE = 3;
			}
			get_knots().debug("Player ID is %s\n", player_id);
			string? password = null;
			if (player_id != null)
			{
				password = player_id.split(":")[1];
				player_id = player_id.split(":")[0];
			}
			if (player_id != null && update_status())
			{
				handle_media_change();
				address = "http://" + player_id + ":" + password + "@" + this.get_knots().get_connection().ip() + ":" + get_player_value("port") + "/" + get_player_value("stream");
				GLib.Timeout.add(100, start_playback);
			}
			else
			{
				get_knots().alert("Stream failed to start");
				this.get_knots().get_notebook().set_current_page(RETURN_PAGE);
			}
		}
		else
		{
			get_knots().alert("Restarting playback");
			this.playing = -1;
			this.restart = {id, playback_type};
			on_stop();
		}
		return false;
	}
	
	public bool start_playback()
	{
		get_knots().debug("Making sure Window is rotated\n", null);
		// Make sure window is rotated before starting playback.
		if (is_playing() && !is_playing_music() && get_knots().get_settings().get_value_default_to("landscape_video", "false") == "true" && (window.get_screen().get_width() < window.get_screen().get_height() || get_toolbar().allocation.width !=  window.get_screen().get_width()))
		{
			get_knots().debug("Not yet rotated\n", null);
			return true;
		}
		else
		{
			if (!is_playing_music() && !get_knots().is_fullscreen())
			{
				get_knots().debug("Forcing fullscreen\n", null);
				get_knots().toggle_fullscreen();
			}
			try
			{
				get_knots().debug("Starting player\n", null);
				Thread.create(start_player, true);
			}
			catch (ThreadError e)
			{
			}
			return false;
		}
	}
	
	public void check_update_need()
	{
		bool should_update = this.is_playing() && this.is_seekable() && this.is_showing() && get_knots().is_visible() && !this.is_restarting() && !this.fullscreen;
		if (!this.updating && should_update)
		{
			this.updating = true;
			update_status();
			GLib.Timeout.add(1000, update_progress);
		}
		else
		if (this.updating && !should_update)
		{
			this.updating = false;
		}
	}
	
	public override void display_state_changed(Osso.DisplayState state)
	{
		check_update_need();
	}
	
	public override void window_state_changed(bool covered)
	{
		check_update_need();
	}
	
	public bool update_progress()
	{
		if (this.updating && !this.seeking)
		{
			GLib.TimeVal time = TimeVal();
			time.get_current_time();
			if (time.tv_sec % 5 == 0 && !waiting_for_response)
			{
				waiting_for_response = true;
				if (!busy())
					update_status();
				else
					get_knots().debug("Skipping update\n", null);
				waiting_for_response = false;
				if (!fullscreen)
					update_seekbar();
				if ((get_player_value("media_id")).to_int() != playing)
					handle_media_change();
			}
			int position = (int)(parse_double(get_player_value("position")) * (double)seekbar.get_total_time()) + (int)time.tv_sec % 5;
			if (position > seekbar.get_total_time())
				position = seekbar.get_total_time();
			update_timedisplay(position);
			
		}
		return this.updating;
	}
	
	public bool busy()
	{
		return scroller.busy() || now_playing.busy();
	}
	
	public void update_timedisplay(int? position)
	{
		int total = seekbar.get_total_time();
		if (position == null)
			position = (int)(parse_double(get_player_value("position")) * (double)total);
		current_position.set_label(get_knots().get_info().humanize_duration(position) + " / " + get_knots().get_info().humanize_duration(total)); 
	}
	
	public bool is_showing()
	{
		return this.get_knots().get_notebook().get_current_page() == 2;
	}
	
	public bool is_seekable()
	{
		return seekable == 1;
	}
	
	public void update_seekbar()
	{
		int position = (int)(parse_double(get_player_value("position")) * (double)seekbar.get_total_time());
		if (!seeking)
			set_automatic_position(position);
	}
	
	public double parse_double(string ds)
	{
		string str = ds;
		if (str == null)
			str = "0.0";
		string[] tokens = str.split(".");
		if (tokens.length == 2)
		{
			string s;
			double d = (tokens[0] + "," + tokens[1]).to_double(out s);
			if (s != null && s != "")
				d = (tokens[0] + "." + tokens[1]).to_double();
			return d;
		}
		else
			return 1;
	}
	
	public bool create_playing_list()
	{
		if (is_playing_music())
		{
			get_knots().get_connection().set_url_cache(false);
			string xml = get_knots().get_connection().fetch_from_address("/external/currently_playing_playlist?player_id=" + player_id, fetch_address);
			XMList list = new XMList(xml);
			if (list.itemcount() > 1)
			{
				var children = now_playing.get_container().get_children();
				foreach (Widget child in children)
				{
					now_playing.get_container().remove(child);
				}
				now_playing.show();
				int index = 0;
				foreach(HashTable ht in list.items())
				{
					Item item = new Item(get_knots(), ht, true);
					item.create();
					if (index == 0)
						item.highlight();
					item.button_release_event += select_item;
					((Gtk.Fixed)now_playing.get_container()).put(item, 0, 0);
					index += 1;
				}
				reorder_now_playing();
				now_playing.show_all();
			}
			else
			{
				now_playing.hide();
			}
		}
		else
		{
			now_playing.hide();
		}
		return false;
	}
	
	public void reorder_now_playing()
	{
		int x = 0;
		int y = 0;
		var children = now_playing.get_container().get_children();
		foreach (Widget child in children)
		{
			Item i = (Item)child;
			((Gtk.Fixed)now_playing.get_container()).move(i, x, y);
			if (get_knots().ORIENTATION == get_knots().LANDSCAPE)
				y += i.height();
			else
				x += i.width();
		}
		if (get_player_value("playlistindex") != null)
			highlight_item(get_player_value("playlistindex").to_int() - 1);
	}
	
	public bool select_item(Gtk.Widget w, Gdk.EventButton e)
	{
		if (!now_playing.window_scrolled())
		{
			Item i = (Item)w;
			int old_index = (get_player_value("playlistindex")).to_int();
			int new_index = get_knots().ORIENTATION == get_knots().LANDSCAPE ? (int)(i.allocation.y + e.y) / i.height() + 1 : (int)(i.allocation.x + e.x) / i.width() + 1;
			if (old_index != new_index)
			{
				get_knots().get_connection().set_url_cache(false);
				get_knots().get_connection().fetch_from_address("/external/select_playlist_item?player_id=" + player_id + "&index=" + new_index.to_string(), fetch_address);
				update_status();
				handle_media_change();
			}
			
		}
		return false;
	}
	
	public void highlight_item(int index)
	{
		var children = now_playing.get_container().get_children();
		int count = 0;
		foreach (Widget child in children)
		{
			Item i = (Item)child;
			if (count == index)
			{
				i.highlight();
				if (get_knots().ORIENTATION == get_knots().LANDSCAPE)
					now_playing.vadjustment.value = i.allocation.y - (get_knots().get_screen_height() - 90) / 2 + i.height() / 2;
				else
					now_playing.hadjustment.value = i.allocation.x - get_knots().get_screen_width() / 2 + i.width() / 2;

			}
			else
				i.lowlight();
			count += 1;
		}
	}
	
	public void* start_player()
	{
		try
		{
			get_knots().debug("Starting playback\n", null);
			GLib.Idle.add(create_playing_list);
			get_knots().debug("Playing list created\n", null);
			int ww,wh;
			window.get_size(out ww, out wh);
			double wwr = ((double)(window.get_screen().get_width() - 20) / (get_player_value("video_width")).to_double());
			double whr = ((double)(window.get_screen().get_height() - 120) / (get_player_value("video_height")).to_double());
			if (whr < wwr)
				wwr = whr;
			int w = (((get_player_value("video_width")).to_double() * wwr) / 16 * 16).to_string().to_int();
			int h = (((get_player_value("video_height")).to_double() * wwr) / 16 * 16).to_string().to_int();
			int x = (ww - w) / 2;
			int y = 42;
			int stdin;
			string[] params = {"/usr/bin/mplayer", "-nofs", "-quiet", "-vo", "omapfb:fb_overlay_only:x=" + x.to_string() + ":y=" + y.to_string() + ":w=" + w.to_string() +":h=" + h.to_string(), "-cache",  get_knots().get_settings().get_value_default_to("cache", "128")};
			string aspect = get_knots().get_settings().get_value_default_to("aspect", "none");
			if (aspect == "none")
			{
				params += "-noaspect";
			}
			else
			{
				params += "-aspect";
				params += aspect;
			}
			params += address;
			Process.spawn_async_with_pipes (null, params, null, SpawnFlags.DO_NOT_REAP_CHILD | SpawnFlags.LEAVE_DESCRIPTORS_OPEN | SpawnFlags.STDOUT_TO_DEV_NULL | SpawnFlags.STDERR_TO_DEV_NULL, null, out player_pid, out stdin, null, null);
			GLib.ChildWatch.add(player_pid, player_stopped);
			io = new GLib.IOChannel.unix_new(stdin);
			io.init();
			get_knots().debug("MPlayer started\n", null);
			if (!is_playing_music() && parse_double(get_player_value("position")) > 0.0)
			{
				int position = (int)(parse_double(get_player_value("position")) * (double)seekbar.get_total_time());
				if (position > seekbar.get_total_time())
					position = seekbar.get_total_time();
			}
		}
		catch (GLib.SpawnError se)
		{
			GLib.warning ("Error spawning build process: %s", se.message);
			on_stop();
		}
		return null;
	}
	
	public void player_stopped(GLib.Pid pid, int status)
	{
		GLib.Process.close_pid(pid);
		stop();
	}
	
	public void stop()
	{
		get_toolbar().show();
		this.updating = false;
		get_knots().get_connection().set_url_cache(false);
		get_knots().get_connection().fetch_from_address("/external/stop?client_id=" + player_id, fetch_address);
		if (!this.is_restarting())
		{
			if (this.is_showing())
				this.get_knots().get_notebook().set_current_page(RETURN_PAGE);
			else
				get_knots().alert("Playback stopped");
			if (RETURN_PAGE == 1)
				get_knots().get_info().update_info();
			this.playing = 0;
		}
		timedisplay.hide();
		get_knots().get_browser().get_toolbar_button(4).show();
		get_knots().get_browser().get_toolbar_button(5).hide();
		this.last_played = this.playing;
		try
		{
			if (io != null)
				io.shutdown(false);
		}
		catch (GLib.IOChannelError ce)
		{
			
		}
		io = null;
		window.set_title("Knots - ");
		get_knots().get_window().set_icon(get_knots().get_common().get_app_icon());
		GLib.Timeout.add(1000, redraw);
		stopping = false;
	}
	
	public bool redraw()
	{
		get_knots().get_window().queue_draw();
		if (this.is_restarting())
		{
			this.playing = 0;
			play(restart[0], restart[1]);
		}
		else
		{
			if (!this.is_playing())
				revert_rotation();
		}
		return false;
	}
	
	public bool is_playing_music()
	{
		return get_player_value("mediatype") != null && get_player_value("mediatype").to_int() == 1;
	}
	
	public void revert_rotation()
	{
		if (rotation_before_playback == 1 || rotation_before_playback == 3)
		{
			int current_rotation = get_knots().get_rotation();
			if (current_rotation != rotation_before_playback)
				get_knots().set_rotation(rotation_before_playback);
		}
	}
	
	public bool is_playing()
	{
		return playing != 0;
	}
	
	public bool is_restarting()
	{
		return playing == -1;
	}
	
	public void toggle_fullscreen()
	{
		fullscreen = !fullscreen;
		mplayer_key('f');
		if (!fullscreen)
		{
			timedisplay.show();
			get_toolbar().show();
			GLib.Timeout.add(100, redraw);
		}
		else
		{	
			timedisplay.hide();
			get_toolbar().hide();
		}
		check_update_need();
	}
	
	public void mplayer_key(char key)
	{
		if (io != null)
		{
			char[] c = {key};
			try
			{
				try
				{
					size_t written;
					io.write_chars(c, out written);
					io.flush();
				}
				catch (GLib.IOChannelError e)
				{
					GLib.warning ("Error: %s", e.message);
				}
			}
			catch (GLib.ConvertError ee)
			{
				GLib.warning ("Error: %s", ee.message);
			}
		}
	}
	
	public string get_player_value(string key)
	{
		if (status != null)
			return (string)status.lookup(key);
		else
		{
			if (this.is_playing())
				this.on_stop();
			return "";
		}
	}
	
	public bool key_event(Gdk.EventKey ev)
	{
		switch (ev.keyval)
		{
			case Gdk.KeySyms.F4:
				break;
			case Gdk.KeySyms.F6: case Gdk.KeySyms.F:
				if (is_playing_music())
				{
					get_knots().toggle_fullscreen();
				}
				else
				if (io != null)
				{
					toggle_fullscreen();
				}
				break;
			case Gdk.KeySyms.F7:
				mplayer_key('0');
				break;
			case Gdk.KeySyms.F8:
				mplayer_key('9');
				break;
			case Gdk.KeySyms.Up:
				get_knots().change_volume(true);
				break;
			case Gdk.KeySyms.Down:
				get_knots().change_volume(false);
				break;
			case Gdk.KeySyms.Return:
				break;
			case Gdk.KeySyms.Escape:
				this.on_stop();
				break;
			case Gdk.KeySyms.space:
				break;
			default:
				mplayer_key((char)ev.keyval);
			break;
		}
		return false;
	}
	
	public override void reveal()
	{
		base.reveal();
		check_update_need();
	}
	
	public override void scroll_up()
	{
		scroller.scroll_up(); 
	}
	
	public override void scroll_down()
	{
		scroller.scroll_down();
	}
}

