using GLib;
using Gtk;
using Hildon;
using Soup;
using Xml;
using Osso;

public class Player : PageWidget
{
	string player_id;
	private string address;
	private int playing;
	private int last_played;
	private HashTable status;
	private Hildon.Seekbar seekbar;
	private int seekable = -1;
	private bool automatic_seek;
	private Gtk.Label info;
	private GLib.TimeVal last_seek;
	public int ITEM = 0;
	public int ALBUM = 1;
	public int PLAYLIST = 2;
	public int RETURN_PAGE;
	Pid? player_pid;
	GLib.IOChannel io;
	private bool fullscreen;
	Scroller scroller;
	private int[] restart;
	private bool updating;
	private bool seeking;
	private int rotation_before_playback;
	Gtk.HBox timedisplay;
	Label current_position;
	Gtk.ToolButton loop_button;
	Gtk.ToolButton unloop_button;
	
	construct {
		playing = 0;
		last_played = -1;
		restart = new int[2];
	}
	
	public override void add_signals()
	{
		get_toolbar_button(0).clicked += () => {
			go_back();
		};
		get_toolbar_button(1).clicked += () => {
			on_stop();
		};
		get_toolbar_button(2).clicked += () => {
			fetch_lyrics();
		};
		get_toolbar_button(3).clicked += () => {
			show_info();
		};
		get_toolbar_button(4).clicked += () => {
			previous_item();
		};
		get_toolbar_button(5).clicked += () => {
			next_item();
		};
		seekbar.value_changed += () => {
			seek(null);
		};
		seekbar.move_slider += () => {
			seeking = true;
			update_timedisplay((int)(seekbar.get_position() * (double)seekbar.get_total_time()));
		};
	}
	
	public void change_aspect(Gtk.CheckMenuItem toggled)
	{
		int index = 0;
		string[] fields = {"none", "16:9", "4:3"};
		var children = ((Gtk.Container)toggled.get_parent()).get_children();
		foreach(Gtk.Widget w in children)
		{
			Gtk.CheckMenuItem cmi = (Gtk.CheckMenuItem)w;
			if (cmi == toggled)
			{
				cmi.active = true;
				get_knots().get_settings().set_value("aspect", fields[index]);
			}
			else
			{
				// Use draw_as_radio as bool to prevent infinite loop
				cmi.draw_as_radio = false;
				cmi.active = false;
				cmi.draw_as_radio = true;
			}
			index += 1;
		}
	}
	
	public void go_back()
	{
		this.get_knots().get_notebook().set_current_page(RETURN_PAGE);
		check_update_need();
	}
	
	public override void orientation_change()
	{
		if (get_toolbar() != null)
		{
			if (get_knots().ORIENTATION == get_knots().LANDSCAPE)
			{
				reorder_child(get_toolbar(), 1);
				((Gtk.VBox)get_content()).reorder_child(timedisplay, 0);
			}
			else
			{
				reorder_child(get_toolbar(), 0);
				((Gtk.VBox)get_content()).reorder_child(timedisplay, 1);
			}
		}
	}
	
	public override Gtk.Widget create_content()
	{
		last_seek = TimeVal();
		seekbar = new Hildon.Seekbar();
		seekbar.height_request = 40;
		Gtk.VBox vbox = new Gtk.VBox(false, 0);
		info = new Gtk.Label("");
		info.set_line_wrap(true);
		get_knots().change_foreground_color(info, "#fff");
		Gtk.Fixed f = new Gtk.Fixed();
		f.put(info, 10, 10);
		scroller = new Scroller(f);
		timedisplay = new Gtk.HBox(false, 0);
		current_position = new Gtk.Label("");
		get_knots().change_foreground_color(current_position, "#fff");
		current_position.width_request = 200;
		
		loop_button = new Gtk.ToolButton(new Gtk.Image.from_file("/usr/share/knots/knots_button_repeat.png"), "");
		loop_button.clicked += () => {
			loop(true);
		};
		unloop_button = new Gtk.ToolButton(new Gtk.Image.from_file("/usr/share/knots/knots_button_no_repeat.png"), "");
		unloop_button.clicked += () => {
			loop(false);
		};
		loop_button.show();
		unloop_button.hide();
		timedisplay.pack_start(seekbar, true, true, 0);
		timedisplay.pack_start(loop_button, false, false, 0);
		timedisplay.pack_start(unloop_button, false, false, 0);
		timedisplay.pack_start(current_position, false, false, 0);
		scroller.set_scroll_policy(false, true);
		vbox.pack_start(scroller, true, true, 0);
		vbox.pack_start(timedisplay, false, true, 0);
		return vbox;
	}
	
	public void loop(bool looped)
	{
		if (looped)
		{
			loop_button.hide();
			unloop_button.show();
		}
		else
		{
			unloop_button.hide();
			loop_button.show();
		}
		get_knots().get_connection().fetch("/external/loop?player_id=" + player_id + "&loop=" + looped.to_string());
	}
	
	public override string[] get_toolbar_items()
	{
		return {"back", "stop", "lyrics", "info", "prev", "next"};
	}
	
	public void seek(double? new_position)
	{
		if (!automatic_seek && seekable == 1)
		{
			var newtime = TimeVal();
			newtime.get_current_time();
			if ((newtime.tv_sec - last_seek.tv_sec) > 0)
			{
				last_seek = newtime;
				double total_time = (double)seekbar.get_total_time();
				if (total_time > 0)
				{
					double position = (double)seekbar.get_position() / total_time;
					if (new_position != null)
						position = new_position;
					if (position > 1.0)
						position = 1.0;
					else
					if (position < 0)
						position = 0;
					get_knots().get_connection().fetch("/external/seek?player_id=" + player_id + "&position=" + position.to_string());
				}
			}
		}
		seeking = false;
	}
	
	public void next_item()
	{
		get_knots().get_connection().fetch("/external/next_playlist_item?player_id=" + player_id);
		update_status();
		handle_media_change();
	}
	
	public void previous_item()
	{
		get_knots().get_connection().fetch("/external/previous_playlist_item?player_id=" + player_id);
		update_status();
		handle_media_change();
	}
	
	public void fetch_lyrics()
	{
		string lyrics = get_knots().get_connection().fetch("/external/lyrics?id=" + playing.to_string());
		if (lyrics != null && lyrics.strip() != "")
		{
			set_info(lyrics);
			get_toolbar_button(3).show();
			get_toolbar_button(2).hide();
		}
		else
		{
			get_knots().alert("No lyrics available.");
		}
	}
	
	public void set_info(string info)
	{
		this.info.set_markup(info);
	}

	public void on_stop()
	{
		mplayer_key('q');
	}
	
	public bool update_status()
	{
		string xml = get_knots().get_connection().fetch("/external/player_properties?player_id=" + player_id);
		XMList list = new XMList(xml);
		if (list.itemcount() == 1)
		{
			status = list.items()[0];
			return status != null;
		}
		return false;
	}
	
	public void set_automatic_position(int position)
	{
		automatic_seek = true;
		seekbar.set_position(position);
		automatic_seek = false;
	}
	
	public void return_to_player()
	{
		this.RETURN_PAGE = 0;
		this.get_knots().get_player().reveal();
	}
	
	public void handle_media_change()
	{
		get_toolbar_button(2).show();
		get_toolbar_button(3).hide();
		this.playing = (get_player_value("media_id")).to_int();
		int index = (get_player_value("playlistindex")).to_int();
		int playlist_length = (get_player_value("playlist_length")).to_int();
		seekable = 0;
		if (get_player_value("seekable") == "true")
		{
			seekbar.set_total_time((get_player_value("duration")).to_int());
			timedisplay.show();
			update_timedisplay(null);
			update_seekbar();
			seekable = 1;
		}
		if (get_player_value("looped") == "false")
		{
			loop_button.show();
			unloop_button.hide();
		}
		else
		{
			unloop_button.show();
			loop_button.hide();
		}
		set_info("");
		if (index < playlist_length)
			get_toolbar_button(5).set_sensitive(true);
		else
			get_toolbar_button(5).set_sensitive(false);
		if (index > 1)
			get_toolbar_button(4).set_sensitive(true);
		else
			get_toolbar_button(4).set_sensitive(false);
		if (get_player_value("mediatype").to_int() == 1)
		{
			get_toolbar_button(0).set_sensitive(true);
			get_toolbar_button(2).set_sensitive(true);
			set_info(get_knots().get_connection().fetch("/external/info?id=" + this.playing.to_string()));
			revert_rotation();
		}
		else
		{
			if (get_knots().get_settings().get_value_default_to("landscape_video", "false") == "true")
			{
				rotation_before_playback = get_knots().get_rotation();
				if (rotation_before_playback == 1 || rotation_before_playback == 3)
				{
					get_knots().set_rotation(0);
				}
			}
			get_toolbar_button(0).set_sensitive(false);
			get_toolbar_button(2).set_sensitive(false);
		}
		scroller.scroll_to_top();
		check_update_need();
	}
	
	public void show_info()
	{
		get_toolbar_button(2).show();
		get_toolbar_button(3).hide();
		set_info(get_knots().get_connection().fetch("/external/info?id=" + this.playing.to_string()));
	}
	
	public int get_position_sec()
	{
		return (int)(seekbar.get_position() * seekbar.get_total_time());
	}
	
	public bool play(int id, int playback_type)
	{
		if (!this.is_playing())
		{
			get_toolbar_button(2).show();
			get_toolbar_button(3).hide();
			loop_button.hide();
			unloop_button.hide();
			this.rotation_before_playback = -1;
			this.seeking = false;
			this.updating = false;
			this.fullscreen = false;
			this.get_knots().get_player().reveal();
			get_knots().get_browser().get_toolbar_button(4).hide();
			get_knots().get_browser().get_toolbar_button(5).show();
			timedisplay.hide();
			set_automatic_position(0);
			player_id = null;
			if (playback_type == ITEM)
			{
				player_id = get_knots().get_connection().fetch("/external/play?profile=2&id=" + id.to_string()).split("\n")[0];
				this.RETURN_PAGE = 1;
			}
			else
			if (playback_type == ALBUM)
			{
				player_id = get_knots().get_connection().fetch("/external/play_album?profile=2&id=" + id.to_string()).split("\n")[0];
				this.RETURN_PAGE = 1;
			}
			else
			if (playback_type == PLAYLIST)
			{
				player_id = get_knots().get_connection().fetch("/external/play_playlist?profile=2&id=" + id.to_string()).split("\n")[0];
				this.RETURN_PAGE = 3;
			}
			if (update_status())
			{
				handle_media_change();
				address = "http://" + this.get_knots().get_connection().ip() + ":" + get_player_value("port") + "/" + get_player_value("stream");
				GLib.Timeout.add(100, start_playback);
			}
			else
			{
				get_knots().alert("Stream failed to start");
				this.get_knots().get_notebook().set_current_page(RETURN_PAGE);
			}
		}
		else
		{
			get_knots().alert("Restarting playback");
			this.playing = -1;
			this.restart = {id, playback_type};
			on_stop();
		}
		return false;
	}
	
	public bool start_playback()
	{
		// Make sure window is rotated before starting playback.
		if (is_playing() && get_player_value("mediatype").to_int() != 1 && get_knots().get_settings().get_value_default_to("landscape_video", "false") == "true" && (window.get_screen().get_width() < window.get_screen().get_height() || get_toolbar().allocation.width !=  window.get_screen().get_width()))
			return true;
		else
		{
			try
			{
				Thread.create(start_player, true);
			}
			catch (ThreadError e)
			{
			}
			return false;
		}
	}
	
	public void check_update_need()
	{
		bool should_update = this.is_playing() && this.is_seekable() && this.is_showing() && get_knots().is_visible() && !this.is_restarting() && !this.fullscreen;
		if (!this.updating && should_update)
		{
			this.updating = true;
			update_status();
			GLib.Timeout.add(1000, update_progress);
		}
		else
		if (this.updating && !should_update)
		{
			this.updating = false;
		}
	}
	
	public override void display_state_changed(Osso.DisplayState state)
	{
		check_update_need();
	}
	
	public override void window_state_changed(bool covered)
	{
		check_update_need();
	}
	
	public bool update_progress()
	{
		if (this.updating && !this.seeking)
		{
			GLib.TimeVal time = TimeVal();
			time.get_current_time();
			if (time.tv_sec % 5 == 0)
			{
				update_status();
				if (!fullscreen)
					update_seekbar();
				if ((get_player_value("media_id")).to_int() != playing)
					handle_media_change();
			}
			int position = (int)(parse_double(get_player_value("position")) * (double)seekbar.get_total_time()) + (int)time.tv_sec % 5;
			if (position > seekbar.get_total_time())
				position = seekbar.get_total_time();
			update_timedisplay(position);
		}
		return this.updating;
	}
	
	public void update_timedisplay(int? position)
	{
		int total = seekbar.get_total_time();
		if (position == null)
			position = (int)(parse_double(get_player_value("position")) * (double)total);
		current_position.set_label(get_knots().get_info().humanize_duration(position) + " / " + get_knots().get_info().humanize_duration(total)); 
	}
	
	public bool is_showing()
	{
		return this.get_knots().get_notebook().get_current_page() == 2;
	}
	
	public bool is_seekable()
	{
		return seekable == 1;
	}
	
	public void update_seekbar()
	{
		int position = (int)(parse_double(get_player_value("position")) * (double)seekbar.get_total_time());
		if (!seeking)
			set_automatic_position(position);
	}
	
	public double parse_double(string str)
	{
		string[] tokens = str.split(".");
		if (tokens.length == 2)
		{
			string s;
			double d = (tokens[0] + "," + tokens[1]).to_double(out s);
			if (s != null && s != "")
				d = (tokens[0] + "." + tokens[1]).to_double();
			return d;
		}
		else
			return 1;
	}
	
	public void* start_player()
	{
		try
		{
			int ww,wh;
			window.get_size(out ww, out wh);
			double wwr = ((double)(window.get_screen().get_width() - 20) / (get_player_value("video_width")).to_double());
			double whr = ((double)(window.get_screen().get_height() - 100) / (get_player_value("video_height")).to_double());
			if (whr < wwr)
				wwr = whr;
			int w = (((get_player_value("video_width")).to_double() * wwr) / 16 * 16).to_string().to_int();
			int h = (((get_player_value("video_height")).to_double() * wwr) / 16 * 16).to_string().to_int();
			int x = (ww - w) / 2;
			int y = 50;
			int stdin;
			string[] params = {"/usr/bin/mplayer", "-nofs", "-quiet", "-vo", "omapfb:fb_overlay_only:x=" + x.to_string() + ":y=" + y.to_string() + ":w=" + w.to_string() +":h=" + h.to_string(), "-cache",  get_knots().get_settings().get_value_default_to("cache", "128")};
			string aspect = get_knots().get_settings().get_value_default_to("aspect", "none");
			if (aspect == "none")
			{
				params += "-noaspect";
			}
			else
			{
				params += "-aspect";
				params += aspect;
			}
			params += address;
			Process.spawn_async_with_pipes (null, params, null, SpawnFlags.DO_NOT_REAP_CHILD | SpawnFlags.LEAVE_DESCRIPTORS_OPEN | SpawnFlags.STDOUT_TO_DEV_NULL | SpawnFlags.STDERR_TO_DEV_NULL, null, out player_pid, out stdin, null, null);
			GLib.ChildWatch.add(player_pid, player_stopped);
			io = new GLib.IOChannel.unix_new(stdin);
			io.init();
		}
		catch (GLib.SpawnError se)
		{
			GLib.warning ("Error spawning build process: %s", se.message);
			on_stop();
		}
		return null;
	}
	
	public void player_stopped(GLib.Pid pid, int status)
	{
		GLib.Process.close_pid(pid);
		stop();
	}
	
	public void stop()
	{
		get_toolbar().show();
		this.updating = false;
		get_knots().get_connection().fetch("/external/stop?client_id=" + player_id);
		if (!this.is_restarting())
		{
			if (this.is_showing())
				this.get_knots().get_notebook().set_current_page(RETURN_PAGE);
			else
				get_knots().alert("Playback stopped");
			if (RETURN_PAGE == 1)
				get_knots().get_info().update_info();
			this.playing = 0;
		}
		timedisplay.hide();
		get_knots().get_browser().get_toolbar_button(4).show();
		get_knots().get_browser().get_toolbar_button(5).hide();
		this.last_played = this.playing;
		try
		{
			if (io != null)
				io.shutdown(false);
		}
		catch (GLib.IOChannelError ce)
		{
			
		}
		io = null;
		GLib.Timeout.add(1000, redraw);
	}
	
	public bool redraw()
	{
		get_knots().get_window().queue_draw();
		if (this.is_restarting())
		{
			this.playing = 0;
			play(restart[0], restart[1]);
		}
		else
		{
			if (!this.is_playing())
				revert_rotation();
		}
		return false;
	}
	
	public void revert_rotation()
	{
		if (rotation_before_playback == 1 || rotation_before_playback == 3)
		{
			int current_rotation = get_knots().get_rotation();
			if (current_rotation != rotation_before_playback)
				get_knots().set_rotation(rotation_before_playback);
		}
	}
	
	public bool is_playing()
	{
		return playing != 0;
	}
	
	public bool is_restarting()
	{
		return playing == -1;
	}
	
	public void toggle_fullscreen()
	{
		fullscreen = !fullscreen;
		mplayer_key('f');
		if (!fullscreen)
		{
			timedisplay.show();
			get_toolbar().show();
			GLib.Timeout.add(100, redraw);
		}
		else
		{	
			timedisplay.hide();
			get_toolbar().hide();
		}
		check_update_need();
	}
	
	public void mplayer_key(char key)
	{
		char[] c = {key};
		try
		{
			try
			{
				size_t written;
				io.write_chars(c, out written);
				io.flush();
			}
			catch (GLib.IOChannelError e)
			{
				GLib.warning ("Error: %s", e.message);
			}
		}
		catch (GLib.ConvertError ee)
		{
			GLib.warning ("Error: %s", ee.message);
		}
	}
	
	public string get_player_value(string key)
	{
		if (status != null)
			return (string)status.lookup(key);
		else
		{
			this.on_stop();
			return "";
		}
	}
	
	public bool key_event(Gdk.EventKey ev)
	{
		switch (ev.keyval)
		{
			case Gdk.KeySyms.F4:
				break;
			case Gdk.KeySyms.F6:
				if (get_player_value("mediatype").to_int() == 1)
				{
					get_knots().toggle_fullscreen();
				}
				else
				{
					toggle_fullscreen();
				}
				break;
			case Gdk.KeySyms.F7:
				mplayer_key('0');
				break;
			case Gdk.KeySyms.F8:
				mplayer_key('9');
				break;
			case Gdk.KeySyms.Up:
				get_knots().change_volume(true);
				break;
			case Gdk.KeySyms.Down:
				get_knots().change_volume(false);
				break;
			case Gdk.KeySyms.Return:
				break;
			case Gdk.KeySyms.Escape:
				get_knots().clean_up();
				break;
		}
		return false;
	}
	
	public override void reveal()
	{
		base.reveal();
		check_update_need();
	}
	
	public override void scroll_up()
	{
		scroller.scroll_up(); 
	}
	
	public override void scroll_down()
	{
		scroller.scroll_down();
	}
}

