/*
 * * Copyright (C) 2007 Tuomas Kulve <tuomas@kulve.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "kilikali.h"
#include "kilikali-vfs.h"
#include "playlist.h"

#include <gtk/gtk.h>

#include <string.h>                   /* strlen */


#define MIN_VALID_URI_LEN             5
#define DEFAULT_PLAYLIST_LEN          (100 * 1024)


static void validate_and_add_uri(GtkTreeModel *model, const gchar *uri);
static gboolean is_invalid(const gchar *uri);
static gboolean is_local_file(const gchar *uri);
static gboolean is_uri(const gchar *uri);
static gboolean is_http_uri(const gchar *uri);
static void _add_http_uri(GtkTreeModel *model, const gchar *uri);


void playlist_load(GtkTreeModel *model, const gchar *uri)
{
    int size;
    int len;
    gchar *list = NULL;
    gchar *newline = NULL;
    gchar *list_uri = NULL;

    g_debug("Loading playlist: %s", uri);

    if (is_http_uri(uri)) {
        _add_http_uri (model, uri);
        return;
    }

    if (vfs_load_content(uri, &size, &list) == FALSE) {
        return;
    }
    
    g_debug("Loaded %d bytes", size);

    list_uri = list;
    len = size;
    while ((newline = g_strstr_len(list_uri, len, "\n")) != NULL) {

        *newline = '\0';

        /* remove possible carriage return */
        if (newline -1 > list_uri && *(newline -1) == '\r') {
            *(newline - 1) = '\0';
        }
        
        g_debug("Checking for: %s", list_uri);
        validate_and_add_uri(model, list_uri);

        len -= strlen(list_uri + 1);
        list_uri = newline + 1;
    }

    g_debug("All files added");

    g_free(list);
}



void playlist_save(GtkTreeModel *model, const gchar *uri)
{
    GString *list;
    GtkTreeIter iter;

    g_debug("in playlist_save");

    list = g_string_sized_new(DEFAULT_PLAYLIST_LEN);

    /* false, if list is empty */
    if (gtk_tree_model_get_iter_first(model, &iter)) {

        /* add all uris to g_string */
        do {
            gchar *fullpath = NULL;

            gtk_tree_model_get(model, &iter, COL_FULLPATH, &fullpath, -1);

            list = g_string_append(list, fullpath);
            list = g_string_append_c(list, '\n');

        } while (gtk_tree_model_iter_next(model, &iter));

        /* save g_string to playlist file */
        vfs_save_content(uri, list->len, list->str);
    } else {
        /* Save an empty list */
        vfs_save_content(uri, list->len, list->str);
    }

    g_string_free(list, TRUE);

}



void playlist_auto_load(GtkTreeModel *model)
{
    gchar *uri;

    g_debug("Auto loading playlist");

#ifdef USE_HILDON
    uri = g_strdup_printf("%s/MyDocs/.sounds/kilikali-auto.m3u", 
                          g_getenv("HOME"));
#else
    uri = g_strdup_printf("%s/.kilikali-auto.m3u", 
                          g_getenv("HOME"));
#endif
    playlist_load(model, uri);
    
    g_free(uri);
}



void playlist_auto_save(GtkTreeModel *model)
{
    gchar *uri;

    g_debug("Auto saving playlist");

#ifdef USE_HILDON
    uri = g_strdup_printf("%s/MyDocs/.sounds/kilikali-auto.m3u", 
                          g_getenv("HOME"));
#else
    uri = g_strdup_printf("%s/.kilikali-auto.m3u", 
                          g_getenv("HOME"));
#endif
    playlist_save(model, uri);
    
    g_free(uri);
}



/*
 * static functions 
 */


static void validate_and_add_uri(GtkTreeModel *model, const gchar *uri)
{

    g_debug("in validate_and_add_uri");

    if (is_invalid(uri)) {
        return;
    }

    if (is_http_uri(uri)) {
        _add_http_uri (model, uri);
        return;
    }

    if (is_uri(uri)) {
        g_idle_add(vfs_add_file_from_idle, vfs_modeluri(model, uri));
        return;
    }


    if (is_local_file(uri)) {
        gchar *fileuri;

        fileuri = g_strdup_printf("file://%s", uri);
        g_idle_add(vfs_add_file_from_idle, vfs_modeluri(model, fileuri));
        g_free(fileuri);
        return;
    }

    g_warning("Unexpected line in playlist: %s", uri);
}



/* Some guessing here.. */
static gboolean is_invalid(const gchar *uri)
{

    g_debug("in is_invalid");

    if (strlen(uri) < MIN_VALID_URI_LEN) {
        return TRUE;
    }
    
    if (uri[0] == '#'  || uri[0] == ' '  || uri[0] == '\0' ||
        uri[0] == '\t' || uri[0] == '\n') {
        return TRUE;
    }

    return FALSE;
}



/* more guessing.. */
static gboolean is_local_file(const gchar *uri)
{
    g_debug("in is_local_file");

    if (uri[0] == '/'  || (uri[0] == '.'  && uri[1] == '/')) {
        return TRUE;
    }

    return FALSE;
}



/* and even more guessing.. */
static gboolean is_uri(const gchar *uri)
{
    g_debug("in is_uri");

    if (g_strrstr(uri, "://") != NULL) {
        return TRUE;
    }

    return FALSE;
}

static gboolean is_http_uri(const gchar *uri)
{
    g_debug("in %s", __FUNCTION__);

    if (g_strrstr(uri, "http://") != NULL) {
        return TRUE;
    }

    return FALSE;
}

static void _add_http_uri(GtkTreeModel *model, const gchar *uri)
{
    GtkTreeIter iter;
    gtk_list_store_append (GTK_LIST_STORE(model), &iter);
    gtk_list_store_set (GTK_LIST_STORE(model), &iter,
                        COL_SONG, g_strdup(uri),
                        COL_FULLPATH, g_strdup(uri),
                        -1);
}
/* Emacs indentatation information
   Local Variables:
   indent-tabs-mode:nil
   tab-width:4
   c-set-offset:4
   c-basic-offset:4
   End: 
*/
