
#include <gdk/gdkx.h>
#include <gtk/gtk.h>

#include "kilikali-video-widget.h"

#define THUMBNAIL_SIZE 64
#define FRAME_BORDER 4

G_DEFINE_TYPE(KilikaliVideoWidget, kilikali_video_widget, GTK_TYPE_BIN)

/*
enum
{
  ,
  LAST_SIGNAL
};

static guint kilikali_video_signals[LAST_SIGNAL] = { 0 };
*/

enum
{
    PROP_THUMBNAIL_MODE = 1,
    PROP_THUMBNAIL_VISIBLE,
    PROP_STREAM_POSITION
};

typedef struct _KilikaliVideoWidgetPrivate KilikaliVideoWidgetPrivate;
struct _KilikaliVideoWidgetPrivate
{
    GdkWindow *frame_window;
    GdkWindow *controls_window;
    GdkWindow *video_window;
    gboolean thumbnail_mode;
    gboolean thumbnail_visible;
    double stream_position;
    gboolean pressed;
    gint old_x;
    gint press_position;
    guint input_timer;
};

#define KILIKALI_VIDEO_WIDGET_GET_PRIVATE(o)  \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), KILIKALI_TYPE_VIDEO_WIDGET, KilikaliVideoWidgetPrivate))

#ifdef DEBUG
#define METHOD_START \
  KilikaliVideoWidgetPrivate *priv; \
  g_return_if_fail(videowidget != NULL); \
  priv = KILIKALI_VIDEO_WIDGET_GET_PRIVATE (videowidget); \
  g_return_if_fail(priv != NULL); \
  g_debug("%s", __FUNCTION__);
#else
#define METHOD_START \
  KilikaliVideoWidgetPrivate *priv = KILIKALI_VIDEO_WIDGET_GET_PRIVATE (videowidget);
#endif

/* Method declarations */
static void
kilikali_video_widget_finalize(GObject *widget);
static void
kilikali_video_widget_set_property ( GObject * object, 
                                     guint prop_id,
                                     const GValue * value, 
                                     GParamSpec * pspec);
static void
kilikali_video_widget_get_property (GObject * object, 
                                    guint prop_id,
                                    GValue * value, 
                                    GParamSpec * pspec);

static void
kilikali_video_widget_realize (GtkWidget *widget);
void
kilikali_video_widget_map (GtkWidget *widget);
void
kilikali_video_widget_unmap (GtkWidget *widget);

static void
kilikali_video_widget_size_request (GtkWidget     *widget,
                                    GtkRequisition *requisition);
static void
kilikali_video_widget_size_allocate (GtkWidget     *widget,
                            GtkAllocation *allocation);
static gboolean
kilikali_video_widget_expose (GtkWidget *widget,
                     GdkEventExpose *expose);
static gboolean
kilikali_video_widget_button_press (GtkWidget *widget,
                                    GdkEventButton *button);
static gboolean
kilikali_video_widget_motion_notify (GtkWidget *widget,
                                     GdkEventMotion *motion);
static gboolean
kilikali_video_widget_button_release (GtkWidget *widget,
                                      GdkEventButton *expose);
static gboolean
kilikali_video_widget_key_release (GtkWidget *widget,
                                   GdkEventKey *expose);

static void
_resize_subwindows(GtkWidget *widget);

/*** Private methods */

static void
kilikali_video_widget_class_init(KilikaliVideoWidgetClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);

  gobject_class->finalize = kilikali_video_widget_finalize;
  gobject_class->set_property = kilikali_video_widget_set_property;
  gobject_class->get_property = kilikali_video_widget_get_property;

  widget_class->realize = kilikali_video_widget_realize;
  widget_class->map = kilikali_video_widget_map;
  widget_class->unmap = kilikali_video_widget_unmap;
//  widget_class->configure_event = kilikali_video_widget_configure;
  widget_class->size_allocate = kilikali_video_widget_size_allocate;
  widget_class->size_request = kilikali_video_widget_size_request;
  widget_class->expose_event = kilikali_video_widget_expose;
  widget_class->button_press_event = kilikali_video_widget_button_press;
  widget_class->motion_notify_event = kilikali_video_widget_motion_notify;
  widget_class->button_release_event = kilikali_video_widget_button_release;
  widget_class->key_release_event = kilikali_video_widget_key_release;

  g_object_class_install_property (gobject_class,
    PROP_THUMBNAIL_MODE,
    g_param_spec_boolean ("thumbnail-mode",
                          "Thumbnail mode",
                          "Whether the thumbnail mode is active",
                          TRUE,
                          G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
    PROP_THUMBNAIL_VISIBLE,
    g_param_spec_boolean ("thumbnail-visible",
                          "Video thumbnail visibility",
                          "Whether the video thumbnail is shown",
                          FALSE,
                          G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class,
    PROP_STREAM_POSITION,
    g_param_spec_double ("stream-position",
                         "The relative position in the video stream",
                         "Sets or gets the relative position in the video stream",
                         0.0, 1.0, 0.0,
                         G_PARAM_READWRITE));
  
  g_type_class_add_private (klass, sizeof (KilikaliVideoWidgetPrivate));
}

static void
kilikali_video_widget_init(KilikaliVideoWidget *videowidget)
{
    KilikaliVideoWidgetPrivate *priv = KILIKALI_VIDEO_WIDGET_GET_PRIVATE (videowidget);


    GTK_WIDGET_SET_FLAGS(videowidget, GTK_APP_PAINTABLE);
    GTK_WIDGET_SET_FLAGS(videowidget, GTK_CAN_FOCUS);
    GTK_WIDGET_UNSET_FLAGS(videowidget, GTK_DOUBLE_BUFFERED);
    GTK_WIDGET_UNSET_FLAGS(videowidget, GTK_NO_WINDOW);

    priv->thumbnail_mode = TRUE;
    videowidget->ignore_resizes = FALSE;
}

static void
kilikali_video_widget_finalize(GObject *videowidget)
{
/*
  KilikaliVideoWidget *videowidget = KILIKALI_VIDEO_WIDGET(widget);
  METHOD_START
*/


}

static void
kilikali_video_widget_set_property ( GObject * object, 
                                     guint prop_id,
                                     const GValue * value, 
                                     GParamSpec * pspec)
{
    KilikaliVideoWidget *videowidget = KILIKALI_VIDEO_WIDGET(object);
    KilikaliVideoWidgetPrivate *priv = KILIKALI_VIDEO_WIDGET_GET_PRIVATE (videowidget);

    switch (prop_id) {
        case PROP_THUMBNAIL_MODE:
            priv->thumbnail_mode = g_value_get_boolean(value);
            _resize_subwindows(GTK_WIDGET(object));
            break;
        case PROP_THUMBNAIL_VISIBLE:
            priv->thumbnail_visible = g_value_get_boolean(value);
            _resize_subwindows(GTK_WIDGET(object));
            break;
        case PROP_STREAM_POSITION:
            if (!priv->pressed)
            {
                priv->stream_position = g_value_get_double(value);
            }
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
kilikali_video_widget_get_property (GObject * object, 
                                    guint prop_id,
                                    GValue * value, 
                                    GParamSpec * pspec)
{
    KilikaliVideoWidget *videowidget = KILIKALI_VIDEO_WIDGET(object);
    KilikaliVideoWidgetPrivate *priv = KILIKALI_VIDEO_WIDGET_GET_PRIVATE (videowidget);

    switch (prop_id) {
        case PROP_THUMBNAIL_MODE:
            g_value_set_boolean(value, priv->thumbnail_mode);
            break;
        case PROP_THUMBNAIL_VISIBLE:
            g_value_set_boolean(value, priv->thumbnail_visible);
            break;
        case PROP_STREAM_POSITION:
            g_value_set_double(value, priv->stream_position);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}


static void
kilikali_video_widget_send_configure (KilikaliVideoWidget *videowidget)
{
  GtkWidget *widget;
  GdkEvent *event = gdk_event_new (GDK_CONFIGURE);

  widget = GTK_WIDGET (videowidget);

  event->configure.window = g_object_ref (widget->window);
  event->configure.send_event = TRUE;
  event->configure.x = widget->allocation.x;
  event->configure.y = widget->allocation.y;
  event->configure.width = widget->allocation.width;
  event->configure.height = widget->allocation.height;
  
  gtk_widget_event (widget, event);
  gdk_event_free (event);
}


static void
kilikali_video_widget_realize (GtkWidget *widget)
{
  GdkWindowAttr attributes;
  gint attributes_mask;

  KilikaliVideoWidget *videowidget = KILIKALI_VIDEO_WIDGET(widget);
  METHOD_START

  g_return_if_fail (KILIKALI_IS_VIDEO_WIDGET (widget));

  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes.event_mask = gtk_widget_get_events (widget) | GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK;

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

  widget->window = gdk_window_new (gtk_widget_get_parent_window (widget),
                                   &attributes, attributes_mask);
  gdk_window_set_user_data(widget->window, widget);

  /* Create our thumbnail frame window */
  attributes.x = widget->allocation.width - THUMBNAIL_SIZE - FRAME_BORDER*2;
  attributes.y = widget->allocation.height / 2;
  attributes.width = THUMBNAIL_SIZE + FRAME_BORDER*2;
  attributes.height = THUMBNAIL_SIZE + FRAME_BORDER*2;
  priv->frame_window = gdk_window_new (widget->window,
                                       &attributes, attributes_mask);
  gdk_window_set_user_data(priv->frame_window, widget);

  /* Create our video controls window */
  attributes.x = FRAME_BORDER;
  attributes.y = widget->allocation.y + widget->allocation.height - 48;
  attributes.width = widget->allocation.width;
  attributes.height = 48;
  attributes.event_mask = gtk_widget_get_events (widget) | GDK_EXPOSURE_MASK;
  priv->controls_window = gdk_window_new (priv->frame_window,
                                       &attributes, attributes_mask);
  gdk_window_set_user_data(priv->controls_window, widget);

  /* Create our video window */
  attributes.x = FRAME_BORDER;
  attributes.y = FRAME_BORDER;
  attributes.width = THUMBNAIL_SIZE;
  attributes.height = THUMBNAIL_SIZE;
  attributes.event_mask = gtk_widget_get_events (widget) | GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK;
  priv->video_window = gdk_window_new (priv->frame_window,
                                       &attributes, attributes_mask);
  gdk_window_set_user_data(priv->video_window, widget);
  
  widget->style = gtk_style_attach (widget->style, widget->window);
  gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);

  kilikali_video_widget_send_configure(KILIKALI_VIDEO_WIDGET(widget));
  
}

void
kilikali_video_widget_map (GtkWidget *widget)
{
    KilikaliVideoWidget *videowidget = KILIKALI_VIDEO_WIDGET(widget);
    KilikaliVideoWidgetPrivate *priv = KILIKALI_VIDEO_WIDGET_GET_PRIVATE (videowidget);

    GTK_WIDGET_CLASS(kilikali_video_widget_parent_class)->map(widget);
    if (priv->thumbnail_visible)
    {
        gdk_window_show(priv->video_window);
        gdk_window_show(priv->controls_window);
        gdk_window_show(priv->frame_window);
        gdk_window_raise(priv->video_window);
    }
}

void
kilikali_video_widget_unmap (GtkWidget *widget)
{
    KilikaliVideoWidget *videowidget = KILIKALI_VIDEO_WIDGET(widget);
    KilikaliVideoWidgetPrivate *priv = KILIKALI_VIDEO_WIDGET_GET_PRIVATE (videowidget);

    GTK_WIDGET_CLASS(kilikali_video_widget_parent_class)->unmap(widget);
    gdk_window_hide(priv->frame_window);
    gdk_window_hide(priv->video_window);
    gdk_window_hide(priv->controls_window);
}

static void
kilikali_video_widget_size_request (GtkWidget     *widget,
                                    GtkRequisition *requisition)
{
    guint border;
    g_return_if_fail (requisition != NULL);

    requisition->width = 0;
    requisition->height = 0;

    if (GTK_BIN(widget)->child != NULL)
    {
        border = gtk_container_get_border_width (GTK_CONTAINER(widget));
        
        gtk_widget_size_request(GTK_BIN(widget)->child, requisition);
        requisition->width += border*2;
        requisition->height += border*2;
    }

    requisition->width += THUMBNAIL_SIZE + FRAME_BORDER*2;
    requisition->height += THUMBNAIL_SIZE + FRAME_BORDER*2;

}

static void
_resize_subwindows(GtkWidget *widget)
{
    KilikaliVideoWidget *videowidget = KILIKALI_VIDEO_WIDGET(widget);
    KilikaliVideoWidgetPrivate *priv = KILIKALI_VIDEO_WIDGET_GET_PRIVATE (videowidget);

    if (videowidget->ignore_resizes)
    {
        //g_debug("Ignoring resize");
        return;
    }

gdk_error_trap_push();

    if (!priv->thumbnail_visible)
    {
        gdk_window_hide(priv->video_window);
        gdk_window_hide(priv->frame_window);
        gdk_window_hide(priv->controls_window);
        return;
    }

    if (priv->thumbnail_mode)
    {
        int offset = 0;
        

        /* Special case: we don't want to obscure the scrollbar */
        if (GTK_IS_SCROLLED_WINDOW(GTK_BIN(widget)->child))
        {
            if (GTK_BIN(GTK_BIN(widget)->child)->child != NULL)
            {
                offset = 
                    GTK_BIN(GTK_BIN(widget)->child)->child->allocation.width
                    - widget->allocation.width;
            }
        }
        
        gdk_window_move_resize(priv->frame_window,
                                widget->allocation.width + offset
                                - THUMBNAIL_SIZE
                                - FRAME_BORDER * 2,
                                (widget->allocation.height / 2)
                                 - THUMBNAIL_SIZE / 2 - FRAME_BORDER,
                                THUMBNAIL_SIZE + FRAME_BORDER * 2,
                                THUMBNAIL_SIZE + FRAME_BORDER * 2);
        gdk_window_move_resize(priv->video_window,
                                FRAME_BORDER, FRAME_BORDER,
                                THUMBNAIL_SIZE,
                                THUMBNAIL_SIZE);
#ifdef DEBUG
        g_debug("Resizing to %i,%i %ix%i",
                                widget->allocation.width + offset
                                - THUMBNAIL_SIZE
                                - FRAME_BORDER * 2,
                                (widget->allocation.height / 2)
                                 - THUMBNAIL_SIZE / 2 - FRAME_BORDER,
                                THUMBNAIL_SIZE,
                                THUMBNAIL_SIZE);
#endif
    } else {

#ifdef DEBUG
        g_debug("Resizing to %ix%i",
                                widget->allocation.width,
                                widget->allocation.height);
#endif
        gdk_window_move_resize(priv->frame_window,
                                0, 0,
                                widget->allocation.width,
                                widget->allocation.height);
        gdk_window_move_resize(priv->controls_window,
                               widget->allocation.x,
                               widget->allocation.y + widget->allocation.height - 48,
                               widget->allocation.width, 48);
        gdk_window_move_resize(priv->video_window,
                                0, 0,
                                widget->allocation.width,
                                widget->allocation.height);
    }

    gdk_window_show(priv->frame_window);
    gdk_window_show(priv->controls_window);
    gdk_window_show(priv->video_window);

    /* Make sure the child widgets are refreshed */
    gtk_widget_queue_draw(widget);

gdk_flush ();
if (gdk_error_trap_pop ())
    {
        g_debug("X error!");
    }


}

static void
kilikali_video_widget_size_allocate (GtkWidget     *widget,
                                     GtkAllocation *allocation)
{
    guint border;
    g_return_if_fail (allocation != NULL);

#ifdef DEBUG
    g_debug("%s: %i,%i %ix%i", __FUNCTION__,
            allocation->x, allocation->y,
            allocation->width, allocation->height);
#endif

    widget->allocation = *allocation;

    if (GTK_WIDGET_REALIZED (widget))
    {
        _resize_subwindows(widget);

        gdk_window_move_resize (widget->window,
        	      allocation->x, allocation->y,
        	      allocation->width, allocation->height);

        kilikali_video_widget_send_configure(KILIKALI_VIDEO_WIDGET(widget));

    }

    if (GTK_BIN(widget)->child != NULL)
    {
        border = gtk_container_get_border_width (GTK_CONTAINER(widget));

        allocation->x = border;
        allocation->y = border;
        allocation->width -= border*2;
        allocation->height -= border*2;

        gtk_widget_size_allocate(GTK_BIN(widget)->child, allocation);
    }


}

static gboolean
kilikali_video_widget_expose (GtkWidget *widget,
                              GdkEventExpose *expose)
{
  KilikaliVideoWidget *videowidget = KILIKALI_VIDEO_WIDGET(widget);
  KilikaliVideoWidgetPrivate *priv = KILIKALI_VIDEO_WIDGET_GET_PRIVATE (videowidget);

  if (expose->count > 0)
  {
    return FALSE;
  }

  if (expose->window == priv->frame_window) {
      /* Frame Window */

      gdk_draw_rectangle(priv->frame_window,
                         widget->style->black_gc,
                         TRUE,
                         expose->area.x, expose->area.y,
                         expose->area.width, expose->area.height);
      gdk_draw_rectangle(priv->frame_window,
                         widget->style->white_gc,
                         TRUE,
                         expose->area.x + 2, expose->area.y + 2,
                         expose->area.width - 4, expose->area.height - 4);

  } else if (expose->window == priv->video_window) {
      /* Video Window */

      gdk_draw_rectangle(priv->video_window,
                         widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                         TRUE,
                         expose->area.x, expose->area.y,
                         expose->area.width, expose->area.height);

  } else if (expose->window == priv->controls_window) {
      /* On-screen controls */
      int fill_size = (int)widget->allocation.width*priv->stream_position;
      
      gdk_draw_rectangle(priv->controls_window,
                         widget->style->bg_gc[GTK_WIDGET_STATE(widget)],
                         TRUE,
                         widget->allocation.x, widget->allocation.y,
                         fill_size, widget->allocation.height);
      gdk_draw_rectangle(priv->controls_window,
                         widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                         TRUE,
                         widget->allocation.x + fill_size, widget->allocation.y,
                         widget->allocation.width - fill_size,
                         widget->allocation.height);
        
  } else {
      gdk_window_clear(widget->window);
      gtk_container_propagate_expose(GTK_CONTAINER(widget),
                                     GTK_BIN(widget)->child,
                                     expose);
  }  
  return FALSE;
}

static gboolean
kilikali_video_widget_button_press (GtkWidget *widget,
                                    GdkEventButton *button)
{
    KilikaliVideoWidget *videowidget = KILIKALI_VIDEO_WIDGET(widget);
    KilikaliVideoWidgetPrivate *priv = KILIKALI_VIDEO_WIDGET_GET_PRIVATE (videowidget);

    if (button->window == priv->video_window)
    {
        if (priv->pressed == TRUE)
        {
            return TRUE;
        }
        if (button->y > widget->allocation.y + widget->allocation.height - 48)
        {
            priv->pressed = TRUE;
            priv->old_x = button->x;
            priv->press_position = priv->stream_position;
            gdk_window_raise (priv->controls_window);
        }
    }
    return TRUE;
}

static gboolean
kilikali_video_widget_motion_notify (GtkWidget *widget,
                                     GdkEventMotion *motion)
{
    KilikaliVideoWidget *videowidget = KILIKALI_VIDEO_WIDGET(widget);
    KilikaliVideoWidgetPrivate *priv = KILIKALI_VIDEO_WIDGET_GET_PRIVATE (videowidget);

    if (priv->pressed && motion->window == priv->video_window)
    {
        int delta;
        delta = motion->x - priv->old_x;
        priv->stream_position += (double)delta / (double)(widget->allocation.width*0.5);
        priv->stream_position = CLAMP(priv->stream_position, 0.0, 1.0);
        priv->old_x = motion->x;
        gdk_window_invalidate_rect(priv->controls_window, NULL, FALSE);
    }
    return TRUE;
}

gboolean input_reset(gpointer user_data);
gboolean input_reset(gpointer user_data)
{
    GtkWidget *widget = GTK_WIDGET(user_data);
    KilikaliVideoWidget *videowidget = KILIKALI_VIDEO_WIDGET(user_data);
    KilikaliVideoWidgetPrivate *priv = KILIKALI_VIDEO_WIDGET_GET_PRIVATE (videowidget);

    if (!priv->pressed)
    {
#ifndef MAEMO_CHANGES
        priv->thumbnail_mode = !priv->thumbnail_mode;
        g_object_notify(G_OBJECT(widget), "thumbnail-mode");
#endif
    } else {
        gdk_window_raise (priv->video_window);
        g_object_notify(G_OBJECT(widget), "stream-position");
    }

    priv->pressed = FALSE;
    priv->input_timer = 0;
    return FALSE;
}

static gboolean
kilikali_video_widget_button_release (GtkWidget *widget,
                                      GdkEventButton *button)
{

    KilikaliVideoWidget *videowidget = KILIKALI_VIDEO_WIDGET(widget);
    KilikaliVideoWidgetPrivate *priv = KILIKALI_VIDEO_WIDGET_GET_PRIVATE (videowidget);

    if (button->window != priv->video_window)
    {
        return TRUE;
    }

    if (priv->input_timer > 0)
    {
        g_source_remove(priv->input_timer);
    }

    if (!priv->pressed)
    {
        int delta;
        delta = button->x - priv->old_x;
        priv->stream_position += (double)delta / (double)(widget->allocation.width*0.5);
        priv->stream_position = CLAMP(priv->stream_position, 0.0, 0.9);
    }

  priv->input_timer = g_timeout_add(250, input_reset, (gpointer)widget);

  return TRUE;
}

static gboolean
kilikali_video_widget_key_release (GtkWidget *widget,
                                   GdkEventKey *key)
{
/*
  KilikaliVideoWidget *videowidget = KILIKALI_VIDEO_WIDGET(widget);
  METHOD_START
*/

  return FALSE;
}

/*** Public methods */

GtkWidget *kilikali_video_widget_new (void)
{
    return GTK_WIDGET(g_object_new(KILIKALI_TYPE_VIDEO_WIDGET, NULL));
}

gulong kilikali_video_widget_get_video_xid (KilikaliVideoWidget *videowidget)
{
    KilikaliVideoWidgetPrivate *priv = KILIKALI_VIDEO_WIDGET_GET_PRIVATE (videowidget);

    if (priv->video_window != NULL)
    {
        return GDK_WINDOW_XID(priv->video_window);
    }
    
    return 0;
}

/* Emacs indentatation information
   Local Variables:
   indent-tabs-mode:nil
   tab-width:4
   c-set-offset:4
   c-basic-offset:4
   End: 
*/
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

