/*
 * * Copyright (C) 2007 Tuomas Kulve <tuomas@kulve.fi>
 *                      Kalle Vahlman <zuh@iki.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "kilikali.h"
#include "kilikali-vfs.h"

#include <glib.h>
#include <libgnomevfs/gnome-vfs.h>
#include <gtk/gtk.h>

#define MAX_CONTENT_SIZE                 1024*1024

static void
kilikali_vfs_class_init (KilikaliVFSClass *klass);
static void
kilikali_vfs_instance_init ( GTypeInstance *instance,
                             gpointer g_class );
static void
kilikali_vfs_finalize (GObject *obj);

enum
{
    PROP_MODEL = 1,
    PROP_URI
};

static void
kilikali_vfs_set_property ( GObject * object, 
                            guint prop_id,
                            const GValue * value, 
                            GParamSpec * pspec);
static void
kilikali_vfs_get_property (GObject * object, 
                           guint prop_id,
                           GValue * value, 
                           GParamSpec * pspec);

static
void kilikali_vfs_scan_dir(KilikaliVFS *vfs);

static void _handle_file_uri_cb(GnomeVFSAsyncHandle *handle,
                               GnomeVFSResult result,
                               GList *list,
                               guint entries_read,
                               gpointer callback_data);
static void _add_file(GtkListStore *store,
                      GnomeVFSFileInfo *info,
                      GnomeVFSURI *dir);

/* Invisible to outside */
static GObjectClass *parent_class = NULL;

GType kilikali_vfs_get_type (void)
{
  static GType type = 0;
  if (type == 0) {
    static const GTypeInfo info = {
      sizeof (KilikaliVFSClass),
      NULL,   /* base_init */
      NULL,   /* base_finalize */
      (GClassInitFunc)kilikali_vfs_class_init,   /*   class_init */
      NULL,   /* class_finalize */
      NULL,   /* class_data */
      sizeof (KilikaliVFS),
      0,      /* n_preallocs */
      kilikali_vfs_instance_init    /* instance_init */
    };

    type = g_type_register_static (G_TYPE_OBJECT,
                                   "KilikaliVFSType",
                                   &info, 0);
  }

  return type;
}

struct _KilikaliVFSPrivate
{ 
    GtkTreeModel *model;
    gchar *uri;

    GnomeVFSURI *dir;
    GnomeVFSAsyncHandle *handle;
};

static void
kilikali_vfs_class_init (KilikaliVFSClass *klass)
{
    GObjectClass *gobject_class;

    gnome_vfs_init();

    gobject_class = (GObjectClass *) klass;

    parent_class = g_type_class_ref (G_TYPE_OBJECT);

    gobject_class->finalize = kilikali_vfs_finalize;
    gobject_class->set_property = kilikali_vfs_set_property;
    gobject_class->get_property = kilikali_vfs_get_property;

    g_object_class_install_property (gobject_class,
        PROP_MODEL,
        g_param_spec_object ("model",
                             "Model",
                             "The model to which the songs are pushed",
                             GTK_TYPE_TREE_MODEL,
                             G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

    g_object_class_install_property (gobject_class,
        PROP_URI,
        g_param_spec_string ("uri",
                             "URI",
                             "URI used as base of the currently active scan, "
                             "NULL if not scanning",
                              NULL,
                              G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

    klass->finished_signal_id = g_signal_new ("finished",
        G_TYPE_FROM_CLASS (gobject_class),
        G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
        0 /* class closure */,
        NULL /* accumulator */,
        NULL /* accu_data */,
        g_cclosure_marshal_VOID__VOID,
        G_TYPE_NONE /* return_type */,
        0     /* n_params */,
        NULL  /* param_types */);

}

static void
kilikali_vfs_instance_init ( GTypeInstance *instance,
                             gpointer g_class )
{
    KilikaliVFS *vfs = (KilikaliVFS *)instance;
    KilikaliVFSPrivate *priv;

    vfs->private = g_new0 (KilikaliVFSPrivate, 1);
    priv = vfs->private;
}

static void
kilikali_vfs_finalize (GObject *object)
{
    KilikaliVFS *vfs;

    /* it's not null if we got it, but it might not be ours */
    g_return_if_fail (KILIKALI_IS_VFS(object));

    vfs = KILIKALI_VFS(object);

    if (vfs->private->model != NULL)
    {
        g_object_ref(vfs->private->model);
        vfs->private->model = NULL;
    }
    if (vfs->private->uri != NULL)
    {
        g_free(vfs->private->uri);
        vfs->private->uri = NULL;
    }
    g_free (vfs->private);
    vfs->private = NULL;
}

static void
kilikali_vfs_set_property ( GObject * object, 
                            guint prop_id,
                            const GValue * value, 
                            GParamSpec * pspec)
{
  KilikaliVFS *vfs;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (KILIKALI_IS_VFS(object));

  vfs = KILIKALI_VFS(object);

  switch (prop_id) {
    case PROP_MODEL:
      vfs->private->model = g_value_get_object(value);
      g_object_ref(vfs->private->model);
      break;
    case PROP_URI:
      vfs->private->uri = g_strdup(g_value_get_string(value));
      kilikali_vfs_scan_dir(vfs);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
kilikali_vfs_get_property (GObject * object, 
                           guint prop_id,
                           GValue * value, 
                           GParamSpec * pspec)
{
  KilikaliVFS *vfs;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (KILIKALI_IS_VFS(object));

  vfs = KILIKALI_VFS(object);

  switch (prop_id) {
    case PROP_MODEL:
      g_value_set_object(value, vfs->private->model);
      break;
    case PROP_URI:
      g_value_set_string(value, vfs->private->uri);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}


static void
kilikali_vfs_scan_dir(KilikaliVFS *vfs)
{
    if (vfs->private->uri == NULL || vfs->private->model == NULL)
    {
        return;
    }

    g_debug("Opening dir: %s", vfs->private->uri);
    vfs->private->dir = gnome_vfs_uri_new(vfs->private->uri);
    gnome_vfs_uri_ref(vfs->private->dir);
    gnome_vfs_async_load_directory_uri(&vfs->private->handle,
                                       vfs->private->dir,
                                       GNOME_VFS_FILE_INFO_DEFAULT,
                                       1,
                                       GNOME_VFS_PRIORITY_MIN,
                                       _handle_file_uri_cb, (gpointer)vfs);
                                       
    gnome_vfs_uri_unref(vfs->private->dir);
}


/***********************/
/* Public sync VFS API */
/***********************/


gpointer vfs_modeluri(GtkTreeModel *model, const gchar *uri)
{
    struct modeluri *mu;

    mu = g_new0(struct modeluri, 1);

    mu->model = model;
    mu->uri = g_strdup(uri);

    return mu;
}

gboolean vfs_add_file_from_idle(gpointer data)
{
    
    vfs_add_file((struct modeluri *)data);

    return FALSE;
}


void vfs_add_file(struct modeluri *mu)
{
    GnomeVFSFileInfo *info;
    GnomeVFSResult   result;
    GnomeVFSURI      *vfsuri;
    
    g_assert(mu != NULL);
    
    vfsuri = gnome_vfs_uri_new(mu->uri);
    
    info = gnome_vfs_file_info_new();
   
    result = gnome_vfs_get_file_info_uri(vfsuri, info, 
                                         GNOME_VFS_FILE_INFO_DEFAULT); 
    if (result != GNOME_VFS_OK) {
        g_warning("vfs_add_file: failed to get file info for %s: %s", 
                  mu->uri,
                  gnome_vfs_result_to_string(result));
        gnome_vfs_uri_unref(vfsuri);
        gnome_vfs_file_info_unref(info);
        g_free(mu->uri);
        g_free(mu);
        return;
    }
    
    _add_file(GTK_LIST_STORE(mu->model), info, vfsuri);
    gnome_vfs_file_info_unref(info);
    g_free(mu->uri);
    g_free(mu);
    return;
}



gboolean vfs_load_content(const gchar *uri, gint *size, gchar **content)
{
    GnomeVFSResult result;
    GnomeVFSFileInfo *info;

    g_debug("Loading content for: %s", uri);

    info = gnome_vfs_file_info_new();

    /* get info */
    result = gnome_vfs_get_file_info(uri, info, GNOME_VFS_FILE_INFO_DEFAULT);
    if (result != GNOME_VFS_OK) {
        g_warning("Failed to get file info for %s: %s", 
                  uri, gnome_vfs_result_to_string(result));
        gnome_vfs_file_info_unref(info);
        return FALSE;
    }
    
    /* check size */
    if (info->size > MAX_CONTENT_SIZE) {
        g_warning("File %s is too big to be loaded in the memory: %lld", 
                  uri, (long long int)info->size);

        gnome_vfs_file_info_unref(info);
        return FALSE;
    }

    gnome_vfs_file_info_unref(info);

    /* load into memory */
    result = gnome_vfs_read_entire_file(uri, (int *)size, (char **)content);
    if (result != GNOME_VFS_OK) {
        g_warning("Failed to read %s into memory: %s", 
                  uri, gnome_vfs_result_to_string(result));
        return FALSE;
    }

    return TRUE;
}



gboolean vfs_save_content(const gchar *uri, gint size, gchar *content)
{
    GnomeVFSResult result;
    GnomeVFSHandle *handle = NULL;
    GnomeVFSFileSize written, total_written;
    
    /* open file for writing */
    result = gnome_vfs_create(&handle, uri, 
                              GNOME_VFS_OPEN_WRITE | GNOME_VFS_OPEN_TRUNCATE,
                              FALSE /* not exclusive */,
                              0644 /* perm */);
    if (result != GNOME_VFS_OK) {
        g_warning("Failed to open %s for writing: %s",
                  uri, gnome_vfs_result_to_string(result));
        return FALSE;
    }

    
    /* write in loop until all is written */
    total_written = 0;
    while (total_written < size) {
        result = gnome_vfs_write(handle,
                                 content + total_written,
                                 size - total_written,
                                 &written);
        
        if (result != GNOME_VFS_OK) {
            g_warning("Failed to write to %s: %s",
                      uri, gnome_vfs_result_to_string(result));
            gnome_vfs_close(handle);
            return FALSE;
        }

        total_written += written;
    }

    gnome_vfs_close(handle);

    return TRUE;
}



/*
 * Static functions
 */


/*
 * Callback from gnome_vfs_async_load_directory_uri. Handle one file.
 */
static void _handle_file_uri_cb(GnomeVFSAsyncHandle *handle,
                                GnomeVFSResult result,
                                GList *list,
                                guint entries_read,
                                gpointer callback_data)
{
    KilikaliVFS *vfs = KILIKALI_VFS(callback_data);
    GnomeVFSFileInfo *info;

    if (result == GNOME_VFS_ERROR_EOF) {
        g_debug("All files read");
        g_signal_emit (G_OBJECT(vfs),
                       KILIKALI_VFS_GET_CLASS (vfs)->finished_signal_id,
                       0 /* details */, 
                       NULL);
        return;
    }

    if (result != GNOME_VFS_OK) {
        g_warning("handle_file_uri_cb: error reading file");
        return;
    }

    g_assert(entries_read == 1);
    g_assert(list != NULL);
    g_assert(list->data != NULL);

    info = list->data;

    if (info->type & GNOME_VFS_FILE_TYPE_REGULAR) {
        g_debug("%s, size: %lld", 
                info->name, (long long int)info->size);
    } else {
        g_debug("%s is not a regular file", info->name);
        return;
    }
    
    _add_file(GTK_LIST_STORE(vfs->private->model),
              info,
              gnome_vfs_uri_append_file_name(vfs->private->dir, info->name));
}



/*
 * Add "approved" file to treeview. Function will unref uri and info
 */
static void
_add_file(GtkListStore *store, GnomeVFSFileInfo *info, GnomeVFSURI *uri)
{
    gchar *fullpath;
    GtkTreeIter iter;

    fullpath = gnome_vfs_uri_to_string(uri, GNOME_VFS_URI_HIDE_NONE);
    gnome_vfs_uri_unref(uri);
    g_assert(fullpath);

    gtk_list_store_append (GTK_LIST_STORE(store), &iter);
    gtk_list_store_set (GTK_LIST_STORE(store), &iter,
                        COL_SONG, g_markup_escape_text(info->name, -1),
                        COL_FULLPATH, fullpath,
                        -1);

}



/* Emacs indentatation information
   Local Variables:
   indent-tabs-mode:nil
   tab-width:4
   c-set-offset:4
   c-basic-offset:4
   End: 
*/
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

