/*
 * * Copyright (C) 2007 Kalle Vahlman <zuh@iki.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gdk/gdkkeysyms.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>

#ifdef USE_HILDON
#ifdef USE_IT2008OS
#include <hildon/hildon-program.h>
#include <hildon/hildon-file-chooser-dialog.h>
#include <hildon/hildon-controlbar.h>
#else
#include <hildon-widgets/hildon-program.h>
#include <hildon-widgets/hildon-file-chooser-dialog.h>
#include <hildon-widgets/hildon-controlbar.h>
#endif
#include "kilikali-plugin.h"
#endif 

#include "kilikali.h"
#include "kilikali-ui.h"
#include "kilikali-switchbox.h"
#include "kilikali-video-widget.h"
#include "kilikali-audio.h"
#include "kilikali-vfs.h"
#include "kilikali-device.h"
#include "playlist.h"
#include "progressbox.h"

#include "moko-finger-scroll.h"

enum
{
    PROP_MODEL = 1,
    PROP_AUDIO
};

struct _KilikaliUIPrivate
{ 
    GtkTreeModel *model;
    GtkWidget *treeview;
    GtkUIManager *uimanager;
    GtkActionGroup *actions;
    
    gchar *song_length;

    GtkLabel *now_playing_label;
    GtkLabel *time_label;
    
    GtkAdjustment *volume_adj;
    GtkRange *time_range;
    gboolean seek_lock;
    gulong position;

    GList *playlist;
    GtkTreeRowReference *current_song;
    
    KilikaliAudio *audio;
    KilikaliVFS *vfs;

    GtkWidget *list_widget;
    GtkWidget *video_widget;
    
    gboolean fullscreen;
    gboolean fullscreen_window;
    gboolean playing_video;
    gboolean previously_video;
};

struct _PlaylistEntryData
{
    /* Used to get back at the UI from the button callback */
    KilikaliUI *ui;

    GtkTreeRowReference *row;

    GtkWidget *hbox;
    GtkWidget *image;
    GtkWidget *title_label;
    GtkWidget *time_label;
    GtkWidget *progress;
};

/* TODO: Make this static void _ */
void add_action (GtkAction *action,
                 KilikaliUI *ui);
static void load_playlist_action (GtkAction *action,
                                  KilikaliUI *ui);
static void load_playlist_location_action (GtkAction *action,
                                           KilikaliUI *ui);
static void save_playlist_action (GtkAction *action,
                                  KilikaliUI *ui);
void remove_action (GtkAction *action,
                    KilikaliUI *ui);
void clear_action (GtkAction *action,
                   KilikaliUI *ui);
void quit_action (GtkAction *action,
                  KilikaliUI *ui);
void play_action (GtkAction *action,
                  KilikaliUI *ui);
void pause_action (GtkAction *action,
                   KilikaliUI *ui);
void stop_action (GtkAction *action,
                  KilikaliUI *ui);
void next_action (GtkAction *action,
                  KilikaliUI *ui);
void prev_action (GtkAction *action,
                  KilikaliUI *ui);
void about_action (GtkAction *action,
                   KilikaliUI *ui);

static void
_vfs_finished (GObject *object, KilikaliUI *ui);

static void
on_model_row_inserted (GtkTreeModel *model,
                       GtkTreePath *path,
                       GtkTreeIter *iter,
                       KilikaliUI *ui);
static void
on_model_row_changed (GtkTreeModel *model,
                      GtkTreePath *path,
                      GtkTreeIter *iter,
                      gpointer user_data);
static void
on_model_row_deleted (GtkTreeModel *model,
                      GtkTreePath *path,
                      KilikaliUI *);

static void
on_remove_clicked (GtkButton *button,
                   struct _PlaylistEntryData *entry);

static void
seek_current_song (KilikaliUI *ui,
                   gint offset);
static void 
on_stream_pos_changed (KilikaliAudio *audio,
                       KilikaliUI *ui);
static void 
on_got_stream_length (KilikaliAudio *audio,
                      KilikaliUI *ui);
static void 
on_radio_name_setted (KilikaliAudio *audio,
                      gpointer data,
                      KilikaliUI *ui);

static void
on_end_of_stream (KilikaliAudio *audio,
                  KilikaliUI *ui);
static void
on_sid_playing (KilikaliAudio *audio,
                KilikaliUI *ui);
static void
on_tag_found (KilikaliAudio *audio,
                KilikaliUI *ui);

static void
on_video_playing (KilikaliAudio *audio,
                  KilikaliUI *ui);
static void 
on_type_unknown (KilikaliAudio *audio,
                 KilikaliUI *ui);

static void
on_volume_change (GtkRange *range,
                  KilikaliUI *ui);

gboolean
on_time_range_press (GtkWidget *widget,
                     GdkEventButton *event,
                     KilikaliUI *ui);
static void
on_position_change (GtkRange *range,
                    KilikaliUI *ui);
gboolean
on_time_range_release (GtkWidget *widget,
                       GdkEventButton *event,
                       KilikaliUI *ui);
static void
on_thumbnail_mode_toggle (GObject    *gobject,
                          GParamSpec *pspec,
                          KilikaliUI *ui);
static void
on_video_stream_position_changed (GObject    *gobject,
                                  GParamSpec *pspec,
                                  KilikaliUI *ui);

/* Menuitems */
static const GtkActionEntry entries[] = {
    { "FileMenu", NULL, "_File" },
    { "Add", GTK_STOCK_ADD, "_Add songs...", "<control>A",
      "Add media from a directory", G_CALLBACK(add_action) },
    { "Load playlist", GTK_STOCK_ADD, "_Load playlist...", "<control>L",
      "Load a playlist from a file", G_CALLBACK(load_playlist_action) },
    { "Load playlist location", GTK_STOCK_ADD,
      "_Load playlist from a location...", "<control>N",
      "Load a playlist from location", 
      G_CALLBACK(load_playlist_location_action) },
    { "Save playlist", GTK_STOCK_SAVE, "_Save playlist...", "<control>S",
      "Save current list as a playlist", G_CALLBACK(save_playlist_action) },
    { "Remove", GTK_STOCK_REMOVE, "_Remove selected", "<control>R",
      "Remove selected song(s)", G_CALLBACK(remove_action) },
    { "Clear", GTK_STOCK_CLEAR, "_Clear list", "<control>C",
      "Clear the list of songs", G_CALLBACK(clear_action) },
    { "ControlMenu", NULL, "_Control" },
    { "Play", GTK_STOCK_MEDIA_PLAY, "_Play", "<control>P",
#ifdef USE_HILDON
      NULL,
#else
      "Play the selected song",
#endif
      G_CALLBACK(play_action) },
    { "Pause", GTK_STOCK_MEDIA_PAUSE, "_Pause", "<control>U",
#ifdef USE_HILDON
      NULL,
#else
      "Pause the playing song",
#endif
       G_CALLBACK(pause_action) },
    { "Stop", GTK_STOCK_MEDIA_STOP, "_Stop", "<control>S",
#ifdef USE_HILDON
      NULL,
#else
      "Stop the playback",
#endif
       G_CALLBACK(stop_action) },
    { "Next", GTK_STOCK_MEDIA_NEXT, "_Next", "<control>N",
#ifdef USE_HILDON
      NULL,
#else
      "Jump to next song",
#endif
       G_CALLBACK(next_action) },
    { "Previous", GTK_STOCK_MEDIA_PREVIOUS, "_Previous", "<control>B",
#ifdef USE_HILDON
      NULL,
#else
      "Jump to previous song",
#endif
       G_CALLBACK(prev_action) },
    { "Quit", GTK_STOCK_QUIT, "_Quit", "<control>Q",
#ifdef USE_HILDON
      NULL,
#else
      "Quit the program",
#endif
      G_CALLBACK(quit_action) },
    { "HelpMenu", NULL, "_Help" },
    { "About", GTK_STOCK_ABOUT, "_About", NULL,
      "Information about the program", G_CALLBACK(about_action) },
};

static void
kilikali_ui_class_init (KilikaliUIClass *klass);
static void
kilikali_ui_instance_init ( GTypeInstance *instance,
                                gpointer g_class );

static void
kilikali_ui_set_property ( GObject * object, 
                               guint prop_id,
                               const GValue * value, 
                               GParamSpec * pspec);
static void
kilikali_ui_get_property (GObject * object, 
                              guint prop_id,
                              GValue * value, 
                              GParamSpec * pspec);

/* Internal functions */
static GtkWidget *
kilikali_ui_create_list_widget (KilikaliUI *ui);

static gboolean
on_toplevel_key_press (GtkWidget *widget,
                       GdkEventKey *event,
                       KilikaliUI *ui);
static gboolean
on_video_widget_key_press (GtkWidget *widget,
                           GdkEventKey *event,
                           KilikaliUI *ui);
static void on_toplevel_destroy (GtkWidget *widget,
                                 KilikaliUI *ui);

/*static gboolean
on_song_activated (GtkTreeView *treeview,
				 GtkTreePath *path,
				 GtkTreeViewColumn *col,
				 gpointer userdata);*/
static void
on_song_activated (GtkButton *button,
                   struct _PlaylistEntryData *entry);

/* Invisible to outside */
static GObjectClass *parent_class = NULL;

GType kilikali_ui_get_type (void)
{
  static GType type = 0;
  if (type == 0) {
    static const GTypeInfo info = {
      sizeof (KilikaliUIClass),
      NULL,   /* base_init */
      NULL,   /* base_finalize */
      (GClassInitFunc)kilikali_ui_class_init,   /*   class_init */
      NULL,   /* class_finalize */
      NULL,   /* class_data */
      sizeof (KilikaliUI),
      0,      /* n_preallocs */
      kilikali_ui_instance_init    /* instance_init */
    };

    type = g_type_register_static (G_TYPE_OBJECT,
                                   "KilikaliUIType",
                                   &info, 0);
  }

  return type;
}

static void
kilikali_ui_instance_init ( GTypeInstance *instance,
                                gpointer g_class )
{
#ifdef USE_HILDON
	HildonProgram *program;
#else
    GtkWidget *menubar;
#endif
    KilikaliUI *ui = (KilikaliUI *)instance;
    KilikaliUIPrivate *priv;
    GtkWidget *widget;
    GtkWidget *vbox;
    GtkWidget *hbox;
    /*GtkWidget *list;
    GtkWidget *sbox;*/
    GtkWidget *vport;
    GtkWidget *swnd;
    GtkWidget *align;
    GtkIconTheme *theme;

    /* Check if there's a refresh icon in Tanco icon theme. If found,
       set Tango as the default icon theme. */
    theme = gtk_icon_theme_new();
    gtk_icon_theme_set_custom_theme(theme, "Tango");

    if (gtk_icon_theme_has_icon(theme, GTK_STOCK_REFRESH)) {
        GtkSettings *settings;
        
        settings = gtk_settings_get_default();
        gtk_settings_set_string_property(settings,
                                         "gtk-icon-theme-name",
                                         "Tango",
                                         "theme");
    }

    ui->private = g_new0 (KilikaliUIPrivate, 1);
    priv = ui->private;
    
    priv->song_length = NULL;
    priv->previously_video = FALSE;
    
    priv->uimanager = gtk_ui_manager_new ();

    priv->actions = gtk_action_group_new ("MenuActions");
    gtk_action_group_add_actions (priv->actions, entries,
                                  G_N_ELEMENTS (entries), ui);
    gtk_ui_manager_insert_action_group(priv->uimanager,
                                       priv->actions, -1);

    if (gtk_ui_manager_add_ui_from_file(priv->uimanager,
                                        PACKAGE_DATA_DIR "/actions.xml",
                                        NULL) == 0) {
        g_error("Failed to load actions files %s/%s",
                PACKAGE_DATA_DIR, "actions.xml");
    }

    g_set_application_name ("kilikali");

#ifdef USE_HILDON
# ifdef USE_IT2008OS
    program = HILDON_PROGRAM (hildon_program_get_instance());
# else
    program = HILDON_PROGRAM (hildon_program_new());
# endif
    ui->toplevel = GTK_WIDGET (hildon_window_new ());
    hildon_program_add_window (program, HILDON_WINDOW(ui->toplevel));
#else
    ui->toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_widget_set_size_request(ui->toplevel, 800, 480);
#endif

    g_signal_connect (G_OBJECT(ui->toplevel), "destroy",
                    G_CALLBACK(on_toplevel_destroy), ui);
    g_signal_connect (G_OBJECT(ui->toplevel), "key-press-event",
                    G_CALLBACK(on_toplevel_key_press), ui);

    /* Packing */
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_widget_show(hbox);

    /* Menu */
#ifdef USE_HILDON
    gtk_container_add(GTK_CONTAINER(ui->toplevel), hbox);
    hildon_program_set_common_menu(program,
        GTK_MENU(gtk_ui_manager_get_widget(priv->uimanager, "/ui/MaemoMenu")));
#else
    vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(ui->toplevel), vbox);
    gtk_widget_show(vbox);
    menubar = gtk_ui_manager_get_widget(priv->uimanager, "/ui/MenuBar");
    gtk_box_pack_start(GTK_BOX(vbox), menubar, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
#endif

    gtk_widget_show(gtk_ui_manager_get_widget(priv->uimanager, "/ui/MaemoMenu"));

#if 0
    /* Video widget */
    priv->video_widget = kilikali_video_widget_new();
    g_assert (priv->video_widget != NULL);
    gtk_box_pack_start(GTK_BOX(vbox), priv->video_widget, TRUE, TRUE, 0);
    gtk_widget_show(priv->video_widget);

    g_signal_connect(G_OBJECT(priv->video_widget), "notify::thumbnail-mode",
                     G_CALLBACK(on_thumbnail_mode_toggle), ui);
    g_signal_connect(G_OBJECT(priv->video_widget), "notify::stream-position",
                     G_CALLBACK(on_video_stream_position_changed), ui);
    g_signal_connect (G_OBJECT(priv->video_widget), "key-press-event",
                    G_CALLBACK(on_video_widget_key_press), ui);

    /* List widget */
    priv->list_widget = kilikali_ui_create_list_widget(ui);
    g_assert (priv->list_widget != NULL);
    gtk_container_add(GTK_CONTAINER(priv->video_widget), priv->list_widget);
    gtk_widget_show(priv->list_widget);


    /* Switchable controls area */
    sbox = kilikali_switchbox_new();
    gtk_box_pack_start(GTK_BOX(vbox), sbox, FALSE, TRUE, 6);
    gtk_widget_show(sbox);
    
    /* Control bar (play, pause, etc) */
    list = gtk_ui_manager_get_widget(priv->uimanager, "/ui/ControlBar");
    gtk_container_add(GTK_CONTAINER(sbox), list);
    gtk_widget_show(list);


    /* Volume setting */
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 6);
    gtk_container_add(GTK_CONTAINER(sbox), hbox);

    list = gtk_label_new("Volume:");
    gtk_box_pack_start(GTK_BOX(hbox), list, FALSE, TRUE, 6);
    gtk_widget_show(list);

#ifdef USE_HILDON
    list = hildon_controlbar_new();
    hildon_controlbar_set_range (HILDON_CONTROLBAR(list), 0, 10);
#else
    list = gtk_hscale_new_with_range(0.0, 1.0, 0.1);
    gtk_scale_set_draw_value(GTK_SCALE(list), FALSE);
#endif
    g_signal_connect (G_OBJECT(list), "value-changed",
                      G_CALLBACK(on_volume_change), ui);
    gtk_box_pack_start(GTK_BOX(hbox), list, TRUE, TRUE, 0);
    gtk_widget_show(list);
    priv->volume_adj = gtk_range_get_adjustment(GTK_RANGE(list));

    /* Time & Now playing labels */
    hbox = gtk_hbox_new(FALSE, 6);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
    gtk_widget_show(hbox);

    list = gtk_label_new("");
    gtk_box_pack_start(GTK_BOX(hbox), list, FALSE, TRUE, 0);
    gtk_misc_set_padding(GTK_MISC(list), 0.1, 0.5);
    gtk_widget_show(list);
    priv->time_label = GTK_LABEL(list);

    list = gtk_label_new("");
    gtk_box_pack_start(GTK_BOX(hbox), list, FALSE, TRUE, 0);
    gtk_misc_set_padding(GTK_MISC(list), 0.1, 0.5);
    gtk_widget_show(list);
    priv->now_playing_label = GTK_LABEL(list);

    /* Timeline */
    hbox = gtk_hbox_new(FALSE, 6);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 6);
    gtk_box_pack_start(GTK_BOX(sbox), hbox, FALSE, TRUE, 0);

    list = gtk_label_new("Time:");
    gtk_box_pack_start(GTK_BOX(hbox), list, FALSE, TRUE, 0);
    gtk_widget_show(list);

    list = gtk_hscale_new_with_range(0.0, 1.0, 0.1);
    gtk_scale_set_draw_value(GTK_SCALE(list), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), list, TRUE, TRUE, 0);
    gtk_widget_show(list);
    priv->time_range = GTK_RANGE(list);
    g_signal_connect (G_OBJECT(list), "button-press-event",
                      G_CALLBACK(on_time_range_press), ui);
    g_signal_connect (G_OBJECT(list), "value-changed",
                      G_CALLBACK(on_position_change), ui);
    g_signal_connect (G_OBJECT(list), "button-release-event",
                      G_CALLBACK(on_time_range_release), ui);
#endif    

    /* Cover / volume control */
    vbox = gtk_vbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 7);

    widget = gtk_image_new_from_file(PACKAGE_DATA_DIR "/cover-placeholder.png");
    gtk_box_pack_start(GTK_BOX(vbox), widget, FALSE, FALSE, 7);

    align = gtk_alignment_new(0.5, 0.0, 0.0, 1.0);
    gtk_box_pack_start(GTK_BOX(vbox), align, TRUE, TRUE, 10);

    widget = gtk_vscale_new_with_range(0.0, 1.0, 0.1);
    gtk_scale_set_draw_value(GTK_SCALE(widget), FALSE);
    gtk_range_set_inverted(GTK_RANGE(widget), TRUE);
    g_signal_connect (G_OBJECT(widget), "value-changed",
                      G_CALLBACK(on_volume_change), ui);
    gtk_container_add(GTK_CONTAINER(align), widget);
    priv->volume_adj = gtk_range_get_adjustment(GTK_RANGE(widget));

    /* Playlist */
#ifdef USE_SCROLLED_WINDOW
    swnd = gtk_scrolled_window_new(NULL, NULL);
    gtk_box_pack_start(GTK_BOX(hbox), swnd, TRUE, TRUE, 0);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swnd), GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);

    vport = gtk_viewport_new(NULL, NULL);
    gtk_viewport_set_shadow_type(GTK_VIEWPORT(vport), GTK_SHADOW_NONE);
    gtk_container_add(GTK_CONTAINER(swnd), vport);

    priv->list_widget = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(vport), priv->list_widget);
#else
    swnd = moko_finger_scroll_new ();
    gtk_box_pack_start(GTK_BOX(hbox), swnd, TRUE, TRUE, 0);

    priv->list_widget = gtk_vbox_new(FALSE, 0);
    moko_finger_scroll_add_with_viewport (MOKO_FINGER_SCROLL (swnd), priv->list_widget);
#endif
    gtk_widget_set_name(priv->list_widget, "playlist");

    gtk_widget_show_all(hbox);
    gtk_widget_show_all(swnd);


    gtk_action_set_visible(gtk_action_group_get_action(ui->private->actions,
                                                         "Pause"),
                             FALSE);
    gtk_action_set_sensitive(gtk_action_group_get_action(ui->private->actions,
                                                         "Next"),
                             FALSE);
    gtk_action_set_sensitive(gtk_action_group_get_action(ui->private->actions,
                                                         "Previous"),
                             FALSE);
    gtk_action_set_sensitive(gtk_action_group_get_action(ui->private->actions,
                                                         "Stop"),
                             FALSE);
    gtk_action_set_sensitive(gtk_action_group_get_action(ui->private->actions,
                                                         "Pause"),
                             FALSE);
}

static void
kilikali_ui_finalize (GObject *obj)
{
}

static void
kilikali_ui_class_init(KilikaliUIClass * klass)
{
  GObjectClass *gobject_class;

  gobject_class = (GObjectClass *) klass;

  parent_class = g_type_class_ref (G_TYPE_OBJECT);

  gobject_class->finalize = kilikali_ui_finalize;
  gobject_class->set_property = kilikali_ui_set_property;
  gobject_class->get_property = kilikali_ui_get_property;

  g_object_class_install_property (gobject_class,
    PROP_MODEL,
    g_param_spec_object ("model",
                         "Model",
                         "The model from which the view gets it data",
                         GTK_TYPE_TREE_MODEL,
                         G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
    PROP_AUDIO,
    g_param_spec_object ("audio",
                         "KilikaliAudio",
                         "The audio interface controller for the UI",
                         KILIKALI_TYPE_AUDIO,
                         G_PARAM_READWRITE));

}

static void
kilikali_ui_set_property ( GObject * object, 
                               guint prop_id,
                               const GValue * value, 
                               GParamSpec * pspec)
{
  KilikaliUI *ui;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (KILIKALI_IS_UI(object));

  ui = KILIKALI_UI(object);

  switch (prop_id) {
    case PROP_MODEL:
      /*g_object_set (ui->private->treeview,
                    "model", g_value_get_object(value),
                    NULL);*/
      ui->private->model = GTK_TREE_MODEL(g_value_get_object(value));
      g_signal_connect (G_OBJECT(ui->private->model), "row-inserted",
                        G_CALLBACK(on_model_row_inserted), ui);
      g_signal_connect (G_OBJECT(ui->private->model), "row-changed",
                        G_CALLBACK(on_model_row_changed), ui);
      g_signal_connect (G_OBJECT(ui->private->model), "row-deleted",
                        G_CALLBACK(on_model_row_deleted), ui);
      break;
    case PROP_AUDIO:
      ui->private->audio = g_value_get_object(value);
      if (ui->private->audio != NULL)
      {
        gdouble volume = 0.0;
        g_signal_connect(G_OBJECT(ui->private->audio), "audio-eos",
                         G_CALLBACK(on_end_of_stream), ui);
        g_signal_connect(G_OBJECT(ui->private->audio), "sid-playing",
                         G_CALLBACK(on_sid_playing), ui);
        g_signal_connect(G_OBJECT(ui->private->audio), "video-playing",
                         G_CALLBACK(on_video_playing), ui);
        g_signal_connect(G_OBJECT(ui->private->audio), "type-unknown",
                         G_CALLBACK(on_type_unknown), ui);
        g_signal_connect(G_OBJECT(ui->private->audio), "stream-pos",
                         G_CALLBACK(on_stream_pos_changed), ui);
        g_signal_connect(G_OBJECT(ui->private->audio), "stream-len",
                         G_CALLBACK(on_got_stream_length), ui);
        g_signal_connect(G_OBJECT(ui->private->audio), "notify::iradio-name", 
                         G_CALLBACK(on_radio_name_setted), ui);
        g_signal_connect(G_OBJECT(ui->private->audio), "tag-found",
                         G_CALLBACK(on_tag_found), ui);

        /*if (!GTK_WIDGET_REALIZED(ui->private->video_widget))
            gtk_widget_realize(ui->private->video_widget);
        g_object_set(G_OBJECT(ui->private->audio),
                     "xid", kilikali_video_widget_get_video_xid(KILIKALI_VIDEO_WIDGET(ui->private->video_widget)),
                     NULL);*/

        g_object_get(G_OBJECT(ui->private->audio),
                     "volume", &volume,
                     NULL);

        g_debug ("Initial volume: %0.2f", volume);
        gtk_adjustment_set_value(ui->private->volume_adj, volume);
      }
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
static void
kilikali_ui_get_property (GObject * object, 
                              guint prop_id,
                              GValue * value, 
                              GParamSpec * pspec)
{
  KilikaliUI *ui;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (KILIKALI_IS_UI(object));

  ui = KILIKALI_UI(object);

  switch (prop_id) {
    case PROP_MODEL:
      {
        GtkTreeModel *model;
        g_object_get (ui->private->treeview,
                      "model", &model,
                      NULL);
        g_value_set_object(value, model);
      }
      break;
    case PROP_AUDIO:
      {
        g_value_set_object(value, ui->private->audio);
      }
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

KilikaliUI *
kilikali_ui_new ( void )
{
  KilikaliUI *ui = g_object_new (KILIKALI_TYPE_UI, NULL);
  return ui;
}

#ifdef USE_HILDON 
void
kilikali_ui_action(KilikaliUI *ui, kilikali_plugin_t plugin_action)
{
    GtkAction *action;
    gchar *a_str;

    switch (plugin_action) {
    case KILIKALI_PLUGIN_ACTION_NEXT:
        a_str = "Next";
        break;
    case KILIKALI_PLUGIN_ACTION_PREV:
        a_str = "Previous";
        break;
    case KILIKALI_PLUGIN_ACTION_PLAY:
        a_str = "Play";
        break;
    case KILIKALI_PLUGIN_ACTION_PAUSE:
        a_str = "Pause";
        break;
    case KILIKALI_PLUGIN_ACTION_STOP:
        a_str = "Stop";
        break;
    default:
        g_warning("Invalid action: %d", plugin_action);
        return;
    }
        
    /* perform the action */
    action = gtk_action_group_get_action(ui->private->actions, a_str);
    gtk_action_activate(action);
}
#endif

static void
on_selection_changed (GtkTreeSelection *selection,
                      KilikaliUI *ui)
{
    gboolean playstate;
    
    if (gtk_tree_selection_count_selected_rows(selection) > 0)
    {
        playstate = TRUE;
    } else {
        playstate = FALSE;
    }

    gtk_action_set_sensitive(gtk_action_group_get_action(
                                ui->private->actions,
                                "Play"),
                                playstate);
}

static GtkWidget *
kilikali_ui_create_list_widget (KilikaliUI *ui)
{
    GtkTreeSelection *selection;
    GtkWidget *swin;
    GtkCellRenderer *renderer;

    g_assert(ui != NULL);
    g_assert(KILIKALI_IS_UI(ui));

    swin = GTK_WIDGET(g_object_new(GTK_TYPE_SCROLLED_WINDOW,
                                    "hscrollbar-policy", GTK_POLICY_AUTOMATIC,
                                    "vscrollbar-policy", GTK_POLICY_AUTOMATIC,
                                    NULL));

    ui->private->treeview = GTK_WIDGET(g_object_new(GTK_TYPE_TREE_VIEW,
                                                    "headers-visible", FALSE,
                                                    "fixed-height-mode", FALSE,
                                                    NULL));
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(ui->private->treeview));
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);

    g_signal_connect(G_OBJECT(selection), "changed",
                    G_CALLBACK(on_selection_changed), ui);
    g_signal_connect(G_OBJECT(ui->private->treeview), "row-activated",
                    G_CALLBACK(on_song_activated), ui);

    gtk_container_add(GTK_CONTAINER(swin), ui->private->treeview);
    gtk_widget_show(ui->private->treeview);

    /* Set up the renderers */
    renderer = gtk_cell_renderer_text_new();
    gtk_tree_view_insert_column_with_attributes(
        GTK_TREE_VIEW(ui->private->treeview),
        -1,
        "Song", renderer,
        "markup", COL_SONG,
        NULL);

    return swin;
}

static gboolean
on_toplevel_key_press (GtkWidget *widget,
                       GdkEventKey *event,
                       KilikaliUI *ui)
{
    GtkAction *action;
    g_assert(ui != NULL);
    g_assert(KILIKALI_IS_UI(ui));

    switch (event->keyval)
    {
      case GDK_Escape:
#ifndef MAEMO_CHANGES
        if (ui->private->fullscreen)
        {
            gtk_window_unfullscreen(GTK_WINDOW(ui->toplevel));
            ui->private->fullscreen = FALSE;
        }
#endif
        g_object_set(G_OBJECT(ui->private->video_widget),
                     "thumbnail-visible", FALSE,
                     "thumbnail-mode", TRUE,
                     NULL);

        /* Stop the music */
        g_debug ("Stopping video");
        action = gtk_action_group_get_action(ui->private->actions, "Stop");
#ifndef MAEMO_CHANGES
        if (ui->private->previously_video) {
#endif
            ui->private->previously_video = FALSE;
            gtk_action_activate(action);
            return TRUE;
#ifndef MAEMO_CHANGES
        }
#endif
        break;
      case GDK_f:
      case GDK_F6:
      case GDK_F11:
        /* FIXME: use window-state-event signal to track fullscreen state */
        if (ui->private->fullscreen && 
#ifndef MAEMO_CHANGES
            ui->private->playing_video == TRUE)
#else
            ui->private->playing_video == FALSE)
#endif
        {
            gtk_window_unfullscreen(GTK_WINDOW(ui->toplevel));
            ui->private->fullscreen = FALSE;
#ifndef MAEMO_CHANGES
            g_object_set(G_OBJECT(ui->private->video_widget),
                                  "thumbnail-mode", TRUE,
                                  NULL);
#endif
        } else {
            gtk_window_fullscreen(GTK_WINDOW(ui->toplevel));
            ui->private->fullscreen = TRUE;
#ifndef MAEMO_CHANGES
            g_object_set(G_OBJECT(ui->private->video_widget),
                                  "thumbnail-mode", FALSE,
                                  NULL);
#endif
        }
        return TRUE;
        break;
      case GDK_Left:
        g_debug("Seek backward");
        seek_current_song(ui, -KILIKALI_SEEK_STEP_SMALL);
        return TRUE;
        break;
      case GDK_Right:
        g_debug("Seek forward");
        seek_current_song(ui, KILIKALI_SEEK_STEP_SMALL);
        return TRUE;
        break;
      default:
        /* Unhandled key */
        break;
    }

    return FALSE;
}

static gboolean
on_video_widget_key_press (GtkWidget *widget,
                           GdkEventKey *event,
                           KilikaliUI *ui)
{
    GtkAction *action;
    kilikali_audio_states state;
    g_assert(ui != NULL);
    g_assert(KILIKALI_IS_UI(ui));


    switch (event->keyval)
    {
      case GDK_Return:
        g_object_get(G_OBJECT(ui->private->audio), "state", &state, NULL);
        switch (state)
        {
            case AUDIO_STATE_PAUSED:
                action = gtk_action_group_get_action(ui->private->actions, "Play");
                break;
            case AUDIO_STATE_PLAYING:
                action = gtk_action_group_get_action(ui->private->actions, "Pause");
                break;
            default:
                action = NULL;
                break;
        }
        if (action != NULL)
        {
            gtk_action_activate(action);
            return TRUE;
        }

        break;
      default:
        /* Unhandled key, pass it to the toplevel. A bit hackish approach... */
        on_toplevel_key_press (widget, event, ui);
        break;
    }

    return FALSE;
}


static void on_toplevel_destroy (GtkWidget *widget,
                                 KilikaliUI *ui)
{
    g_assert(ui != NULL);
    g_assert(KILIKALI_IS_UI(ui));
    
    gtk_action_activate(gtk_action_group_get_action(
                            ui->private->actions,
                            "Quit"));

}

/*static gboolean
on_song_activated (GtkTreeView *treeview,
                   GtkTreePath *path,
                   GtkTreeViewColumn *col,
                   gpointer userdata)*/
static void
on_song_activated (GtkButton *button,
                   struct _PlaylistEntryData *entry)
{
    KilikaliUI *ui = entry->ui;
    GtkTreeModel *model;
    GtkTreeIter   iter;
    gchar *filename = NULL;
    gchar *song     = NULL;

    g_debug(__FUNCTION__);

    /*g_object_set(G_OBJECT(ui->private->video_widget),
                          "thumbnail-visible", FALSE,
                          NULL);*/

    model = gtk_tree_row_reference_get_model(entry->row);

    if (ui->private->current_song &&
        gtk_tree_path_compare(gtk_tree_row_reference_get_path(ui->private->current_song),
                              gtk_tree_row_reference_get_path(entry->row)) == 0)
    {
        gint state;
        g_object_get(G_OBJECT(ui->private->audio), "state", &state, NULL);
        if (state == AUDIO_STATE_PLAYING)
        {
            kilikali_audio_pause(KILIKALI_AUDIO(ui->private->audio));
            gtk_image_set_from_stock(GTK_IMAGE(entry->image), "kilikali-play", icon_size);
        }
        else
        {
            kilikali_audio_play(KILIKALI_AUDIO(ui->private->audio));
            gtk_image_set_from_stock(GTK_IMAGE(entry->image), "kilikali-pause", icon_size);
        }
    }
    else if (gtk_tree_model_get_iter(model, &iter, gtk_tree_row_reference_get_path(entry->row)))
    {
        gtk_tree_model_get (model, &iter,
                            COL_SONG, &song,
                            COL_FULLPATH, &filename,
                            -1);
        if (filename == NULL || song == NULL)
        {
            return /*FALSE*/;
        }
        
        if(ui->private->current_song)
        {
            GtkTreeIter iter2;
            struct _PlaylistEntryData *entry2;

            gtk_tree_model_get_iter(model, &iter2, gtk_tree_row_reference_get_path(ui->private->current_song));
            gtk_tree_model_get(model, &iter2, COL_UIDATA, &entry2, -1);
            gtk_label_set_text(GTK_LABEL(entry2->time_label), "--:--");
            gtk_widget_hide(entry2->time_label);
            gtk_image_set_from_stock(GTK_IMAGE(entry2->image), "kilikali-music", icon_size);
            progressbox_set_progress(PROGRESSBOX(entry2->progress), -1);
            gtk_widget_set_state(entry2->title_label, GTK_STATE_NORMAL);
            gtk_widget_set_state(entry2->time_label, GTK_STATE_NORMAL);
        }

        /* May be NULL, no need to check */
        gtk_tree_row_reference_free(ui->private->current_song);
        ui->private->current_song = gtk_tree_row_reference_copy(entry->row);

        kilikali_audio_stop(KILIKALI_AUDIO(ui->private->audio));
        if (ui->private->song_length != NULL) {
            g_free (ui->private->song_length);
            ui->private->song_length = NULL;
        }

        g_object_set (KILIKALI_AUDIO(ui->private->audio),
                      "uri", filename,
                      NULL);
        /*gtk_label_set_text(ui->private->now_playing_label, song);
        gtk_label_set_text(GTK_LABEL(ui->private->time_label), "??:?? / ??:??");*/
        kilikali_audio_play(KILIKALI_AUDIO(ui->private->audio));

        gtk_widget_show(entry->time_label);
        gtk_image_set_from_stock(GTK_IMAGE(entry->image), "kilikali-pause", icon_size);
        gtk_widget_set_state(entry->title_label, GTK_STATE_ACTIVE);
        gtk_widget_set_state(entry->time_label, GTK_STATE_ACTIVE);
        progressbox_set_progress(PROGRESSBOX(entry->progress), 0.0);

        gtk_action_set_sensitive(gtk_action_group_get_action(
                                    ui->private->actions,
                                    "Next"),
                                  TRUE);
        gtk_action_set_sensitive(gtk_action_group_get_action(
                                    ui->private->actions,
                                    "Previous"),
                                  TRUE);
        gtk_action_set_visible(gtk_action_group_get_action(
                                    ui->private->actions,
                                    "Play"),
                                  FALSE);
        gtk_action_set_visible(gtk_action_group_get_action(
                                    ui->private->actions,
                                    "Pause"),
                                  TRUE);
        gtk_action_set_sensitive(gtk_action_group_get_action(
                                    ui->private->actions,
                                    "Pause"),
                                  TRUE);
        gtk_action_set_sensitive(gtk_action_group_get_action(
                                    ui->private->actions,
                                    "Stop"),
                                  TRUE);

        g_free(filename);
        g_free(song);
    }

    return /*FALSE*/;
}

void add_action (GtkAction *action,
                 KilikaliUI *ui)
{
    gint response;
    GtkWindow *toplevel;
    GtkWidget *chooser;
    gchar *path = NULL;
    GObject *vfs;

    toplevel = GTK_WINDOW(ui->toplevel);

#ifdef USE_HILDON
    chooser = 
    hildon_file_chooser_dialog_new(toplevel,
    						   GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
#else
    chooser = gtk_file_chooser_dialog_new("Choose a directory to scan",
                                    toplevel,
                                    GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
                                    GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                    GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                                    NULL);
#endif

    gtk_dialog_set_default_response(GTK_DIALOG(chooser), GTK_RESPONSE_ACCEPT);

    response = gtk_dialog_run (GTK_DIALOG (chooser));

    if (response == GTK_RESPONSE_ACCEPT ||
        response == GTK_RESPONSE_OK)
    {
        path = gtk_file_chooser_get_uri (GTK_FILE_CHOOSER(chooser));
    }

    gtk_widget_destroy(chooser);
    chooser = NULL;

    if (path == NULL)
        return;

    /* Start scan, the pointer to the object is lost here, but it is unreffed
     * in the callback(s).
     * TODO: offer a way to cancel the scan
     */
    vfs = g_object_new(KILIKALI_TYPE_VFS,
                       "model", ui->private->model,
                       "uri", path,
                       NULL);

    g_signal_connect (vfs, "finished",
                      G_CALLBACK(_vfs_finished), ui);

    g_free(path);
}



void load_playlist_action (GtkAction *action,
                           KilikaliUI *ui)
{
    gint response;
    GtkWindow *toplevel;
    GtkWidget *chooser;
    gchar *uri = NULL;
    GtkTreeModel *model;

    toplevel = GTK_WINDOW(ui->toplevel);

#ifdef USE_HILDON
    chooser = 
    hildon_file_chooser_dialog_new(toplevel, GTK_FILE_CHOOSER_ACTION_OPEN);
#else
    chooser = gtk_file_chooser_dialog_new("Load playlist",
                                    toplevel,
                                    GTK_FILE_CHOOSER_ACTION_OPEN,
                                    GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                    GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                                    NULL);
#endif

    gtk_dialog_set_default_response(GTK_DIALOG(chooser), GTK_RESPONSE_ACCEPT);

    response = gtk_dialog_run (GTK_DIALOG (chooser));

    if (response == GTK_RESPONSE_ACCEPT ||
        response == GTK_RESPONSE_OK)
    {
        uri = gtk_file_chooser_get_uri (GTK_FILE_CHOOSER(chooser));
    }

    gtk_widget_destroy(chooser);
    chooser = NULL;

    if (uri == NULL)
        return;

    model = gtk_tree_view_get_model(GTK_TREE_VIEW(ui->private->treeview));

    playlist_load(model, uri);

    g_free(uri);
}


void load_playlist_location_action (GtkAction *action,
                                    KilikaliUI *ui)
{
    gint response;
    GtkWindow *toplevel;
    GtkWidget *dialog;
    GtkWidget *entry;
    gchar *uri = NULL;
    GtkTreeModel *model;

    toplevel = GTK_WINDOW(ui->toplevel);

    
    dialog =  gtk_dialog_new_with_buttons("Open playlist location",
                                          toplevel,
                                          GTK_DIALOG_MODAL | 
                                          GTK_DIALOG_DESTROY_WITH_PARENT,
                                          GTK_STOCK_OK,
                                          GTK_RESPONSE_ACCEPT,
                                          GTK_STOCK_CANCEL,
                                          GTK_RESPONSE_REJECT,
                                          NULL);

    
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_ACCEPT);

    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), "http://");
    
    gtk_container_add(GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),
                      entry);
    gtk_widget_show(entry);

    response = gtk_dialog_run(GTK_DIALOG(dialog));

    if (response == GTK_RESPONSE_ACCEPT)
    {
        uri = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));
    }

    gtk_widget_destroy(dialog);
    dialog = NULL;
    
    if (uri == NULL)
        return;

    model = gtk_tree_view_get_model(GTK_TREE_VIEW(ui->private->treeview));
    playlist_load(model, uri);

    g_free(uri);
}



void save_playlist_action (GtkAction *action,
                           KilikaliUI *ui)
{
    gint response;
    GtkWindow *toplevel;
    GtkWidget *chooser;
    gchar *uri = NULL;
    GtkTreeModel *model;

    toplevel = GTK_WINDOW(ui->toplevel);

#ifdef USE_HILDON
    chooser = 
    hildon_file_chooser_dialog_new(toplevel, GTK_FILE_CHOOSER_ACTION_SAVE);
#else
    chooser = gtk_file_chooser_dialog_new("Save playlist",
                                    toplevel,
                                    GTK_FILE_CHOOSER_ACTION_SAVE,
                                    GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                    GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                                    NULL);
#endif

    gtk_dialog_set_default_response(GTK_DIALOG(chooser), GTK_RESPONSE_ACCEPT);

    response = gtk_dialog_run (GTK_DIALOG (chooser));

    if (response == GTK_RESPONSE_ACCEPT ||
        response == GTK_RESPONSE_OK)
    {
        uri = gtk_file_chooser_get_uri (GTK_FILE_CHOOSER(chooser));
    }

    gtk_widget_destroy(chooser);
    chooser = NULL;

    if (uri == NULL)
        return;

    model = gtk_tree_view_get_model(GTK_TREE_VIEW(ui->private->treeview));

    playlist_save(model, uri);

    g_free(uri);
}


static void
_vfs_finished (GObject *object, KilikaliUI *ui)
{
    g_object_unref(object);
}

void remove_action (GtkAction *action,
                    KilikaliUI *ui)
{
    GList *rows;
    GList *iter;
    GtkTreeModel *model;
    GtkTreeSelection *selection;

    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(ui->private->treeview));
    rows = g_list_reverse(gtk_tree_selection_get_selected_rows(selection, &model));

    /* We'll traverse the list backwards since that way the paths will be
     * valid even if we remove the row it's pointing to
     */
    iter = rows;
    while (iter != NULL)
    {
        GtkTreeIter treeiter;
        GtkTreePath *path = (GtkTreePath *)iter->data;
        
        gtk_tree_model_get_iter(model, &treeiter, path);
        gtk_list_store_remove(GTK_LIST_STORE(model), &treeiter);
        
        /* Optimization, free the paths here so we don't need to traverse the
         * list again to go through them for freeing
         */
        gtk_tree_path_free(path);
        iter = iter->next;
    }
    g_list_free (rows);
}

void clear_action (GtkAction *action,
                   KilikaliUI *ui)
{
    /*g_object_set(G_OBJECT(ui->private->video_widget),
                          "thumbnail-visible", FALSE,
                          NULL);*/

    /* May be NULL, no need to check */
    gtk_tree_row_reference_free(ui->private->current_song);
    ui->private->current_song = NULL;

	gtk_list_store_clear(GTK_LIST_STORE(ui->private->model));
}

void quit_action (GtkAction *action,
                  KilikaliUI *ui)
{
    /* TODO: Save settings etc */

    kilikali_audio_stop (ui->private->audio);

    gtk_main_quit();
}

void play_action (GtkAction *action,
                  KilikaliUI *ui)
{
    ui->private->previously_video = FALSE;

    if  (ui->private->current_song == NULL)
    {
        gchar *filename = NULL;
        gchar *song     = NULL;
        GList *rows;
        GtkTreeIter iter;
        GtkTreeModel *model;
        GtkTreeSelection *selection;
        GtkTreePath *path;

        selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(ui->private->treeview));
        rows = gtk_tree_selection_get_selected_rows(selection, &model);

        if (rows == NULL)
        {
            return;
        }

        path = (GtkTreePath *)rows->data;
        if(!gtk_tree_model_get_iter(model, &iter, path))
        {
            g_list_foreach (rows, (GFunc)gtk_tree_path_free, NULL);
            g_list_free (rows);
            return;
        }
        gtk_tree_model_get (model, &iter,
                            COL_SONG, &song,
                            COL_FULLPATH, &filename,
                            -1);
        if (filename == NULL || song == NULL)
        {
            g_list_foreach (rows, (GFunc)gtk_tree_path_free, NULL);
            g_list_free (rows);
            return;
        }
        
        gtk_tree_row_reference_free(ui->private->current_song);
        ui->private->current_song = gtk_tree_row_reference_new(model, path);

        gtk_label_set_text(ui->private->now_playing_label, song);
        gtk_label_set_text(GTK_LABEL(ui->private->time_label), "??:?? / ??:??");
        g_object_set (KILIKALI_AUDIO(ui->private->audio),
                      "uri", filename,
                      NULL);

        g_list_foreach (rows, (GFunc)gtk_tree_path_free, NULL);
        g_list_free (rows);
        g_free(filename);
        g_free(song);
    }
    
    while (gtk_events_pending())
        gtk_main_iteration();

    kilikali_audio_play(ui->private->audio);
    gtk_action_set_visible(gtk_action_group_get_action(
                                ui->private->actions,
                                "Play"),
                                FALSE);
    gtk_action_set_visible(gtk_action_group_get_action(
                                ui->private->actions,
                                "Pause"),
                                TRUE);
    gtk_action_set_sensitive(gtk_action_group_get_action(
                                ui->private->actions,
                                "Pause"),
                                TRUE);
    gtk_action_set_sensitive(gtk_action_group_get_action(
                                ui->private->actions,
                                "Stop"),
                                TRUE);
    gtk_action_set_sensitive(gtk_action_group_get_action(
                                ui->private->actions,
                                "Next"),
                                TRUE);
    gtk_action_set_sensitive(gtk_action_group_get_action(
                                ui->private->actions,
                                "Previous"),
                                TRUE);

    if (ui->private->playing_video)
    {
        ui->private->previously_video = TRUE;
        kilikali_device_inhibit_blanking();
    }
}

void pause_action (GtkAction *action,
                   KilikaliUI *ui)
{
    while (gtk_events_pending())
        gtk_main_iteration();

    kilikali_audio_pause(ui->private->audio);

    gtk_action_set_visible(gtk_action_group_get_action(
                                ui->private->actions,
                                "Play"),
                                TRUE);
    gtk_action_set_visible(gtk_action_group_get_action(
                                ui->private->actions,
                                "Pause"),
                                FALSE);

    if (ui->private->playing_video)
    {
        kilikali_device_allow_blanking();
    }
}

void stop_action (GtkAction *action,
                  KilikaliUI *ui)
{
    while (gtk_events_pending())
        gtk_main_iteration();

    g_object_set(G_OBJECT(ui->private->video_widget),
                          "thumbnail-visible", FALSE,
                          NULL);
    if (ui->private->playing_video)
    {
        ui->private->playing_video = FALSE;
        kilikali_device_allow_blanking();
        gtk_widget_grab_focus(ui->private->list_widget);
    }

    gtk_label_set_text(ui->private->now_playing_label, "");
    gtk_label_set_text(GTK_LABEL(ui->private->time_label), "");
    gtk_tree_row_reference_free(ui->private->current_song);
    ui->private->current_song = NULL;
    kilikali_audio_stop(ui->private->audio);
    gtk_action_set_sensitive(gtk_action_group_get_action(
                                ui->private->actions,
                                "Next"),
                                FALSE);
    gtk_action_set_sensitive(gtk_action_group_get_action(
                                ui->private->actions,
                                "Previous"),
                                FALSE);
    gtk_action_set_sensitive(gtk_action_group_get_action(
                                ui->private->actions,
                                "Stop"),
                                FALSE);
    gtk_action_set_visible(gtk_action_group_get_action(
                                ui->private->actions,
                                "Play"),
                                TRUE);
    gtk_action_set_visible(gtk_action_group_get_action(
                                ui->private->actions,
                                "Pause"),
                                FALSE);
    if (ui->private->song_length != NULL) {
        g_free (ui->private->song_length);
        ui->private->song_length = NULL;
    }

}

void next_action (GtkAction *action,
                  KilikaliUI *ui)
{
    gchar *filename = NULL;
    gchar *song     = NULL;
    GtkTreePath *path;
    GtkTreeIter iter;
    GtkTreeModel *model;
    GtkTreeSelection *selection;

    if (ui->private->current_song == NULL
        || !gtk_tree_row_reference_valid(ui->private->current_song))
    {
        return;
    }

    g_object_set(G_OBJECT(ui->private->video_widget),
                          "thumbnail-visible", FALSE,
                          NULL);

    model = gtk_tree_view_get_model(GTK_TREE_VIEW(ui->private->treeview));
    
    if (model == NULL
        || gtk_tree_model_iter_n_children(model, NULL) == 0)
    {
        return;
    }
    
    path = gtk_tree_row_reference_get_path(ui->private->current_song);
    gtk_tree_path_next(path);

    /* If the path leads to invalid row, fall back to the first one
     * This implies global repeat btw, which should be user-controllable
     */
    if (!gtk_tree_model_get_iter(model, &iter, path))
    {
        gtk_tree_path_free(path);
        path = gtk_tree_path_new_first();
        /* This should always succeed since we exit on empty models earlier,
         * but let's make sure...
         */
        if (!gtk_tree_model_get_iter(model, &iter, path))
        {
            gtk_tree_path_free(path);
            return;
        }
    }

    gtk_tree_model_get (model, &iter,
                        COL_SONG, &song,
                        COL_FULLPATH, &filename,
                        -1);

    if (filename == NULL || song == NULL)
    {
        gtk_tree_path_free(path);
        return;
    }
    
    /* Update UI
     * FIXME: This is not nice, since if you had a selection, it'll clear it.
     * Again, a fresh widget to show the music would be better.
     */
    selection =
        gtk_tree_view_get_selection(GTK_TREE_VIEW(ui->private->treeview));
    g_assert(selection != NULL);
    gtk_tree_selection_unselect_all(selection);
    gtk_tree_selection_select_path(selection, path);

    /* May be NULL, no need to check */
    gtk_tree_row_reference_free(ui->private->current_song);
    ui->private->current_song = gtk_tree_row_reference_new(model, path);

    while (gtk_events_pending())
        gtk_main_iteration();

    kilikali_audio_stop(KILIKALI_AUDIO(ui->private->audio));
    if (ui->private->song_length != NULL) {
        g_free (ui->private->song_length);
        ui->private->song_length = NULL;
    }

    g_object_set (KILIKALI_AUDIO(ui->private->audio),
                    "uri", filename,
                    NULL);
    gtk_label_set_text(ui->private->now_playing_label, song);
    gtk_label_set_text(GTK_LABEL(ui->private->time_label), "??:?? / ??:??");
    kilikali_audio_play(KILIKALI_AUDIO(ui->private->audio));

    g_free(filename);
    g_free(song);
}

void prev_action (GtkAction *action,
                  KilikaliUI *ui)
{
    gchar *filename = NULL;
    gchar *song     = NULL;
    GtkTreePath *path;
    GtkTreeIter iter;
    GtkTreeModel *model;
    GtkTreeSelection *selection;

    if (ui->private->current_song == NULL
        || !gtk_tree_row_reference_valid(ui->private->current_song))
    {
        return;
    }

    g_object_set(G_OBJECT(ui->private->video_widget),
                          "thumbnail-visible", FALSE,
                          NULL);

    model = gtk_tree_view_get_model(GTK_TREE_VIEW(ui->private->treeview));
    
    if (model == NULL
        || gtk_tree_model_iter_n_children(model, NULL) == 0)
    {
        return;
    }
    
    path = gtk_tree_row_reference_get_path(ui->private->current_song);
    gtk_tree_path_prev(path);

    /* If the path leads to invalid row, fall back to the first one
     * This implies global repeat btw, which should be user-controllable
     */
    if (!gtk_tree_model_get_iter(model, &iter, path))
    {
        gtk_tree_path_free(path);
        path = gtk_tree_path_new_first();
        /* This should always succeed since we exit on empty models earlier,
         * but let's make sure...
         */
        if (!gtk_tree_model_get_iter(model, &iter, path))
        {
            gtk_tree_path_free(path);
            return;
        }
    }

    gtk_tree_model_get (model, &iter,
                        COL_SONG, &song,
                        COL_FULLPATH, &filename,
                        -1);

    if (filename == NULL || song == NULL)
    {
        gtk_tree_path_free(path);
        return;
    }
    
    /* Update UI
     * FIXME: This is not nice, since if you had a selection, it'll clear it.
     * Again, a fresh widget to show the music would be better.
     */
    selection =
        gtk_tree_view_get_selection(GTK_TREE_VIEW(ui->private->treeview));
    g_assert(selection != NULL);
    gtk_tree_selection_unselect_all(selection);
    gtk_tree_selection_select_path(selection, path);
    
    /* May be NULL, no need to check */
    gtk_tree_row_reference_free(ui->private->current_song);
    ui->private->current_song = gtk_tree_row_reference_new(model, path);

    kilikali_audio_stop(KILIKALI_AUDIO(ui->private->audio));
    if (ui->private->song_length != NULL) {
        g_free (ui->private->song_length);
        ui->private->song_length = NULL;
    }

    g_object_set (KILIKALI_AUDIO(ui->private->audio),
                    "uri", filename,
                    NULL);

    gtk_label_set_text(ui->private->now_playing_label, song);
    gtk_label_set_text(GTK_LABEL(ui->private->time_label), "??:?? / ??:??");
    kilikali_audio_play(KILIKALI_AUDIO(ui->private->audio));

    g_free(filename);
}

void about_action (GtkAction *action,
                   KilikaliUI *ui)
{
    GtkAboutDialog *aboutdialog;
    const gchar *authors[] =
    {
        "Joni Valtanen, <jvaltane@kapsi.fi>",
        "Tuomas Kulve, <tuomas@kulve.fi>",
        "Kalle Vahlman, <zuh@iki.fi>",
        "Jari Havanto, <jari.havanto@gmail.com>",
        NULL
    };
    
    /* TRANSLATORS: Replace this string with your names, one name per line. */
    /*  gchar *translators = _("translator-credits");*/

    aboutdialog = GTK_ABOUT_DIALOG(gtk_about_dialog_new ());

    gtk_about_dialog_set_version (aboutdialog, VERSION);
    gtk_about_dialog_set_name (aboutdialog, _("kilikali"));
    gtk_about_dialog_set_copyright (aboutdialog,
        _("Copyright 2007-2008 Joni Valtanen\n"));
    gtk_about_dialog_set_website(aboutdialog,
        "http://kilikali.garage.maemo.org");
    gtk_about_dialog_set_website_label(aboutdialog,
        _("Kilikali website"));
    gtk_about_dialog_set_license (aboutdialog,
        "GNU General Public License version 2");

    gtk_about_dialog_set_authors (aboutdialog, authors);

    gtk_dialog_run(GTK_DIALOG(aboutdialog));
    gtk_widget_destroy(GTK_WIDGET(aboutdialog));
}

static gchar *
msecs_to_string(guint msecs)
{
    /* Support hours:minutes:seconds or minutes:seconds */
    guint hours = (msecs/3600000);
    guint mins = (msecs-(hours*3600000))/60000;
    guint secs = (msecs-(mins*60000)-(hours*3600000))/1000;
    gchar string[10] = { 0, };

    if (hours > 0) {
        g_snprintf (string, 9, "%02u:%02u:%02u", hours, mins, secs );
    } else {
        g_snprintf (string, 9, "%02u:%02u", mins, secs );
    }

    return g_strdup(string);
}

static void
on_model_row_inserted (GtkTreeModel *model,
                       GtkTreePath *path,
                       GtkTreeIter *iter,
                       KilikaliUI *ui)
{
    GtkWidget *button;
    GtkWidget *align;
    gchar *song;

    g_assert (ui != NULL);
    g_assert (KILIKALI_IS_UI(ui));

    if (gtk_tree_model_iter_n_children(ui->private->model, NULL)>1)
    {
        GtkWidget *sep;
        sep = gtk_hseparator_new();
        gtk_box_pack_start(GTK_BOX(ui->private->list_widget), sep, FALSE, FALSE, 0);
        gtk_widget_show(sep);
    }

    struct _PlaylistEntryData *entry=g_malloc(sizeof(struct _PlaylistEntryData));
    entry->ui=ui;
    entry->row=gtk_tree_row_reference_new(model, path);

    entry->hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(ui->private->list_widget), entry->hbox, FALSE, FALSE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(entry->hbox), 6);
    gtk_widget_show(GTK_WIDGET(entry->hbox));

    button = gtk_button_new();
    gtk_box_pack_start(GTK_BOX(entry->hbox), button, FALSE, FALSE, 0);
    entry->image = gtk_image_new_from_stock("kilikali-music", icon_size);
    gtk_container_add(GTK_CONTAINER(button), entry->image);
    gtk_widget_show_all(GTK_WIDGET(button));

    g_signal_connect(G_OBJECT(button), "clicked",
                 G_CALLBACK(on_song_activated), entry);

    entry->progress = progressbox_new(0, 5);
    gtk_container_set_border_width(GTK_CONTAINER(entry->progress), 10);
    gtk_box_pack_start(GTK_BOX(entry->hbox), entry->progress, TRUE, TRUE, 5);
    progressbox_set_progress(PROGRESSBOX(entry->progress), -1);
    gtk_widget_show(GTK_WIDGET(entry->progress));

    gtk_tree_model_get(model, iter, COL_SONG, &song, -1);
    entry->title_label=gtk_label_new(song);
    gtk_box_pack_start(GTK_BOX(entry->progress), entry->title_label, 1, 1, 0);
    gtk_label_set_ellipsize(GTK_LABEL(entry->title_label), PANGO_ELLIPSIZE_END);
    gtk_widget_show(GTK_WIDGET(entry->title_label));

    entry->time_label=gtk_label_new("--:--");
    gtk_box_pack_start(GTK_BOX(entry->progress), entry->time_label, 0, 0, 5);

    align = gtk_alignment_new(0.5, 0.0, 0.0, 0.0);
    gtk_box_pack_end(GTK_BOX(entry->hbox), align, FALSE, FALSE, 0);

    button = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(align), button);
    gtk_container_add(GTK_CONTAINER(button), gtk_image_new_from_stock("kilikali-remove", GTK_ICON_SIZE_BUTTON));
    gtk_widget_show_all(GTK_WIDGET(align));

    g_signal_connect(G_OBJECT(button), "clicked",
                 G_CALLBACK(on_remove_clicked), entry);

    gtk_list_store_set(GTK_LIST_STORE(model), iter, COL_UIDATA, entry, -1);
    ui->private->playlist = g_list_prepend(ui->private->playlist, entry);
}

static void
on_model_row_changed (GtkTreeModel *model,
                      GtkTreePath *path,
                      GtkTreeIter *iter,
                      gpointer user_data)
{
    struct _PlaylistEntryData *data;
    gchar *song;

    gtk_tree_model_get(model, iter,
                       COL_UIDATA, &data,
                       COL_SONG, &song,
                       -1);

    gtk_label_set_text(GTK_LABEL(data->title_label), song);
}

static void
purge_separators_callback (GtkWidget *child,
                           gboolean *status)
{
    if (GTK_IS_SEPARATOR(child))
    {
        if(*status)
            gtk_object_destroy(GTK_OBJECT(child));
        *status = TRUE;
    }
    else
        *status = FALSE;
}

static void
on_model_row_deleted (GtkTreeModel *model,
                      GtkTreePath *path,
                      KilikaliUI *ui)
{
    GList *iter;
    gboolean status;

    for (iter = ui->private->playlist; iter;)
    {
        GList *next = iter->next;
        struct _PlaylistEntryData *entry = (struct _PlaylistEntryData *)iter->data;
        if (!gtk_tree_row_reference_valid(entry->row))
        {
            gtk_object_destroy(GTK_OBJECT(entry->hbox));
            gtk_tree_row_reference_free(entry->row);
            g_free(entry);
            ui->private->playlist = g_list_delete_link(ui->private->playlist, iter);
        }
        iter = next;
    }

    status = TRUE;
    gtk_container_foreach(GTK_CONTAINER(ui->private->list_widget), (GtkCallback)purge_separators_callback, &status);
}

static void
on_remove_clicked (GtkButton *button,
                   struct _PlaylistEntryData *entry)
{
    KilikaliUI *ui = entry->ui;
    GtkTreeIter iter;

    gtk_tree_model_get_iter(ui->private->model, &iter, gtk_tree_row_reference_get_path(entry->row));
    gtk_list_store_remove(GTK_LIST_STORE(ui->private->model), &iter);
}

static void
seek_current_song (KilikaliUI *ui,
                   gint offset)
{
    GtkTreeModel *model;
    GtkTreeIter iter;
    struct _PlaylistEntryData *entry;
    guint length = 0;

    if (!ui->private->current_song || !gtk_tree_row_reference_valid(ui->private->current_song))
        return;

    g_object_get(ui->private->audio, "stream-length", &length, NULL);

    /* What problems this can cause? Should audio-iface 
       to be fixed instead? */
    if ((offset < 0) && (ui->private->position + offset > length)) {
        ui->private->position = 10 - offset;
    } else if ((offset > 0) && (ui->private->position + offset > length)) {
        ui->private->position = length - offset - 1000;
    }

    model = ui->private->model;

    gtk_tree_model_get_iter(model, &iter, gtk_tree_row_reference_get_path(ui->private->current_song));
    gtk_tree_model_get(model, &iter, COL_UIDATA, &entry, -1);

    ui->private->position += offset;
    kilikali_audio_absolute(ui->private->audio, ui->private->position);

    progressbox_set_progress(PROGRESSBOX(entry->progress), ui->private->position*1.0/length);
}


static void 
on_stream_pos_changed (KilikaliAudio *audio,
                       KilikaliUI *ui)
{
    //GtkAdjustment *adj;
    /* position in msecs */
    guint length = 0;
    guint position = 0;
    gchar *pos_string = NULL;
    gchar *time_string = NULL;
    GtkTreeIter iter;
    struct _PlaylistEntryData *entry;

    if (ui->private->seek_lock)
    {
        return;
    }
    
    g_object_get (audio,
                  "stream_length", &length,
                  "stream-position", &position,
                  NULL);

    ui->private->position = position;
    pos_string = msecs_to_string(length-position);
    if (pos_string == NULL) {
        return;
    }

    time_string=g_strconcat("-", pos_string, NULL);
    g_free(pos_string);

    gtk_tree_model_get_iter(ui->private->model, &iter, gtk_tree_row_reference_get_path(ui->private->current_song));
    gtk_tree_model_get(ui->private->model, &iter, COL_UIDATA, &entry, -1);
    gtk_label_set_text(GTK_LABEL(entry->time_label), time_string);
    g_free(time_string);

    progressbox_set_progress(PROGRESSBOX(entry->progress), position*1.0/length);

#if 0
    /* update only if we have length */
    if (ui->private->song_length != NULL) {
        time_string = g_strconcat(pos_string, " / ", ui->private->song_length, NULL);
        g_free (pos_string);
    } else {
        time_string = pos_string;
    }
    gtk_label_set_text(GTK_LABEL(ui->private->time_label), time_string);
    g_free (time_string);

    gtk_range_set_value(ui->private->time_range, position);
    adj = gtk_range_get_adjustment(ui->private->time_range);
    ui->private->seek_lock = TRUE;
    g_object_set(G_OBJECT(ui->private->video_widget),
                 "stream-position", adj->value / adj->upper, 
                 NULL);
#endif
}

static void 
on_got_stream_length (KilikaliAudio *audio,
                      KilikaliUI *ui)
{
    /* length in msecs */
    guint len = 0;

    g_assert (ui != NULL);
    g_assert (KILIKALI_IS_UI(ui));

    g_object_get (audio, "stream-length", &len, NULL);
    if (ui->private->song_length != NULL) {
        g_free (ui->private->song_length);
        ui->private->song_length = NULL;
    }
    if (len == 0) {
        return;
    }
    ui->private->song_length = msecs_to_string(len);
    
    /*gtk_range_set_range(ui->private->time_range, 0, len);
    gtk_range_set_increments(ui->private->time_range, len/100, len/20);
    gtk_range_set_value(ui->private->time_range, 0);*/

#ifdef MAEMO_CHANGES
    /* yet one more hack for volume problem with maemo */
    {
        gdouble volume = 0.0;
        g_object_get (audio, "volume", &volume, NULL);
        g_object_set(G_OBJECT(audio), "volume", volume, NULL);
    }
#endif
    
}

static void 
on_radio_name_setted (KilikaliAudio *audio,
                      gpointer data,
                      KilikaliUI *ui)
{
    gchar *radioname = NULL;

    g_debug (__FUNCTION__);

    g_assert (ui != NULL);
    g_assert (KILIKALI_IS_UI(ui));
    g_assert (GTK_IS_LABEL (ui->private->now_playing_label));

    g_object_get (audio, "iradio-name", &radioname, NULL);
    if (radioname == NULL) {
        g_debug ("Couldn't get radio name");      
        return;
    }
    gtk_label_set_text(ui->private->now_playing_label, radioname);

}

static void 
on_end_of_stream (KilikaliAudio *audio,
                  KilikaliUI *ui)
{
    g_object_set(G_OBJECT(ui->private->video_widget),
                          "thumbnail-mode", TRUE,
                          NULL);
    g_object_set(G_OBJECT(ui->private->video_widget),
                          "thumbnail-visible", FALSE,
                          NULL);

    ui->private->playing_video = FALSE;

    if (ui->private->current_song
        && gtk_tree_row_reference_valid(ui->private->current_song))
    {
        GtkTreePath *path;
        GtkTreeIter iter;

        path = gtk_tree_row_reference_get_path(ui->private->current_song);
        gtk_tree_path_next(path);

        if (gtk_tree_model_get_iter(ui->private->model, &iter, path))
        {
            struct _PlaylistEntryData *entry;
            gtk_tree_model_get(ui->private->model, &iter, COL_UIDATA, &entry, -1);

            on_song_activated(0, entry);
        }
    }
}

static void 
on_sid_playing (KilikaliAudio *audio,
                KilikaliUI *ui)
{
    g_debug(__FUNCTION__);

#if 0
    /* This is only for testing with multrack sids */
    g_object_set (audio, "sid-tune", 1, NULL );
#endif

}

static void 
on_tag_found (KilikaliAudio *audio,
                KilikaliUI *ui)
{
    GtkTreeModel *model;
    GtkTreePath *path;
    GtkTreeIter iter;
    KilikaliAudioTags *tags = NULL;
    gchar *eartist;
    gchar *ealbum;
    gchar *etitle;
    gchar *finalstring;

    if (ui->private->current_song == NULL
        || !gtk_tree_row_reference_valid(ui->private->current_song))
    {
        return;
    }

    model = ui->private->model;
    
    if (model == NULL
        || gtk_tree_model_iter_n_children(model, NULL) == 0)
    {
        return;
    }
    
    path = gtk_tree_row_reference_get_path(ui->private->current_song);
    gtk_tree_model_get_iter(model, &iter, path);
    gtk_tree_path_free(path);
    
    g_object_get (audio, "tags", &tags, NULL);

    if (tags->title != NULL) {
        etitle = g_markup_escape_text(tags->title, -1);
    } else {
        etitle = g_markup_escape_text("Unknown", -1);
    }
    if (tags->album != NULL) {
        ealbum = g_markup_escape_text(tags->album, -1);
    }
    else {
        ealbum = g_markup_escape_text("Unknown", -1);
    }
    if (tags->artist != NULL) {
        eartist = g_markup_escape_text(tags->artist, -1);
    } else {
        eartist = g_markup_escape_text("Unknown", -1);
    }

#if 0
    gtk_list_store_set (GTK_LIST_STORE(model), &iter,
                        COL_SONG, etitle,
                        -1);
#endif
    gtk_list_store_set (GTK_LIST_STORE(model), &iter,
                        COL_ARTIST, eartist,
                        -1);
    gtk_list_store_set (GTK_LIST_STORE(model), &iter,
                        COL_ALBUM, ealbum,
                        -1);

    if (tags->artist != NULL) {
        finalstring = g_strconcat (eartist, " - ", etitle, NULL);
        gtk_list_store_set (GTK_LIST_STORE(model), &iter,
                            COL_SONG, finalstring,
                            -1);
        g_free (finalstring);
    }
    g_free(etitle);
    g_free(eartist);
    g_free(ealbum);
}

static void
on_video_playing (KilikaliAudio *audio,
                  KilikaliUI *ui)
{
    
    g_assert (ui != NULL);
    g_assert (KILIKALI_IS_UI(ui));

    gtk_widget_grab_focus(ui->private->video_widget);

    g_object_set(G_OBJECT(ui->private->video_widget),
                          "thumbnail-visible", TRUE,
#ifdef MAEMO_CHANGES
                          "thumbnail-mode", FALSE,
#endif
                          NULL);

    ui->private->playing_video = TRUE;
    kilikali_device_inhibit_blanking();
}

static void 
on_type_unknown (KilikaliAudio *audio,
                 KilikaliUI *ui)
{
    g_debug(__FUNCTION__);
    
}

static void
on_volume_change (GtkRange *range,
                  KilikaliUI *ui)
{
    GtkAdjustment *adj;
    
    g_assert (ui != NULL);
    g_assert (KILIKALI_IS_UI(ui));

    adj = gtk_range_get_adjustment(range);

    g_object_set(G_OBJECT(ui->private->audio),
                 "volume", adj->value,
                 NULL);
}

gboolean
on_time_range_press (GtkWidget *widget,
                     GdkEventButton *event,
                     KilikaliUI *ui)
{
    g_assert (ui != NULL);
    g_assert (KILIKALI_IS_UI(ui));

    ui->private->seek_lock = TRUE;

    return FALSE;
}

static void
on_position_change (GtkRange *range,
                    KilikaliUI *ui)
{
    GtkAdjustment *adj;
    
    g_assert (ui != NULL);
    g_assert (KILIKALI_IS_UI(ui));

    if (ui->private->seek_lock)
    {
        return;
    }
    
    adj = gtk_range_get_adjustment(range);

}

gboolean
on_time_range_release (GtkWidget *widget,
                       GdkEventButton *event,
                       KilikaliUI *ui)
{
    GtkAdjustment *adj;

    g_assert (ui != NULL);
    g_assert (KILIKALI_IS_UI(ui));

    ui->private->seek_lock = FALSE;

    adj = gtk_range_get_adjustment(ui->private->time_range);
    kilikali_audio_absolute (ui->private->audio, (gulong)adj->value);

    return FALSE;
}

static void
on_thumbnail_mode_toggle (GObject    *gobject,
                          GParamSpec *pspec,
                          KilikaliUI *ui)
{
    GList *children;
    GList *iter;
    gboolean thumbnail_mode = FALSE;
    gboolean thumbnail_visible = FALSE;
    
    if (ui->private->current_song == FALSE)
    {
        return;
    }

    children = gtk_container_get_children(GTK_CONTAINER(
                                            GTK_BIN(ui->toplevel)->child));

    g_object_get(gobject, 
                 "thumbnail-mode", &thumbnail_mode, 
                 "thumbnail-visible", &thumbnail_visible, 
                 NULL);

#ifdef MAEMO_CHANGES
    KILIKALI_VIDEO_WIDGET(gobject)->ignore_resizes = TRUE;
    gdk_window_freeze_updates (ui->toplevel->window);
#endif

    for (iter = children; iter != NULL; iter = iter->next)
    {
        GtkWidget *child = GTK_WIDGET(iter->data);
        
        if (child != ui->private->video_widget)
        {
            if (thumbnail_mode)
            {
                gtk_widget_show(child);
            } else {
                gtk_widget_hide(child);
            }
        }
    }

    while(gtk_events_pending())
    {
        gtk_main_iteration ();
    }

#ifdef MAEMO_CHANGES
    KILIKALI_VIDEO_WIDGET(gobject)->ignore_resizes = FALSE;
    gdk_window_thaw_updates (ui->toplevel->window);

    if (thumbnail_mode)
    {
        if (!ui->private->fullscreen) {
            gtk_window_unfullscreen(GTK_WINDOW(ui->toplevel));
        }
    } else {
        gtk_window_fullscreen(GTK_WINDOW(ui->toplevel));
    }
#else
    if (thumbnail_mode && thumbnail_visible && ui->private->fullscreen)
    {
        gtk_window_unfullscreen(GTK_WINDOW(ui->toplevel));
        ui->private->fullscreen = FALSE;

    }
#endif

    g_list_free(children);
}

static void
on_video_stream_position_changed (GObject    *gobject,
                                  GParamSpec *pspec,
                                  KilikaliUI *ui)
{
    GtkAdjustment *adj;
    gdouble rel = 0;
    
    if (ui->private->seek_lock)
    {
        ui->private->seek_lock = FALSE;
        return;
    }
    
    g_object_get(gobject, "stream-position", &rel, NULL);
    
    adj = gtk_range_get_adjustment(ui->private->time_range);
    kilikali_audio_absolute (ui->private->audio, (gulong)(adj->upper*rel));
}

/* Emacs indentatation information
   Local Variables:
   indent-tabs-mode:nil
   tab-width:4
   c-set-offset:4
   c-basic-offset:4
   End: 
*/
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

