/*
 * * Copyright (C) 2007 Kalle Vahlman <zuh@iki.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

#include <gtk/gtkicontheme.h>
#include <gtk/gtkstock.h>

#include "kilikali-switchbox.h"

enum
{
    PROP_ANIMATE = 1
};

struct _KilikaliSwitchBoxPrivate
{ 
    GtkWidget *visible_child;
    GdkWindow *event_window;

    GdkPixbuf *switch_icon; /* FIXME: make this per-child */

    gboolean animate;
};

static void
kilikali_switchbox_class_init (KilikaliSwitchBoxClass *klass);

static void
kilikali_switchbox_set_property (GObject * object, 
                                 guint prop_id,
                                 const GValue * value, 
                                 GParamSpec * pspec);
static void
kilikali_switchbox_get_property (GObject * object, 
                                 guint prop_id,
                                 GValue * value, 
                                 GParamSpec * pspec);

static void
kilikali_switchbox_add (GtkContainer   *container,
                        GtkWidget      *widget,
                        gpointer       user_data);

static void
kilikali_switchbox_remove (GtkContainer   *container,
                           GtkWidget      *widget,
                           gpointer       user_data);


static void 
kilikali_switchbox_size_request  (GtkWidget      *widget,
                                  GtkRequisition *requisition);
static void
kilikali_switchbox_size_allocate (GtkWidget      *widget,
                                  GtkAllocation  *allocation);

static void
kilikali_switchbox_realize (GtkWidget   *widget);
static void
kilikali_switchbox_map (GtkWidget *widget);
static void
kilikali_switchbox_unmap (GtkWidget *widget);

static gboolean
kilikali_switchbox_expose_event (GtkWidget *widget,
                                 GdkEventExpose *event);
static gboolean
kilikali_switchbox_button_press_event (GtkWidget *widget,
                                       GdkEventButton *event);
static gboolean
kilikali_switchbox_button_release_event (GtkWidget *widget,
                                         GdkEventButton *event);

G_DEFINE_TYPE (KilikaliSwitchBox, kilikali_switchbox, GTK_TYPE_HBOX)

static void
kilikali_switchbox_init ( KilikaliSwitchBox *sb)
{
    KilikaliSwitchBoxPrivate *priv;

    sb->private = g_new0 (KilikaliSwitchBoxPrivate, 1);
    priv = sb->private;

    g_signal_connect_after(G_OBJECT(sb), "add",
                           G_CALLBACK(kilikali_switchbox_add), NULL);
    g_signal_connect_after(G_OBJECT(sb), "remove",
                           G_CALLBACK(kilikali_switchbox_remove), NULL);
}

#if 0
static void
kilikali_switchbox_finalize (GObject *obj)
{
}
#endif 

static void
kilikali_switchbox_class_init(KilikaliSwitchBoxClass * klass)
{
    GObjectClass *gobject_class;
    GtkWidgetClass *widget_class;

    gobject_class = (GObjectClass *) klass;

    gobject_class->set_property = kilikali_switchbox_set_property;
    gobject_class->get_property = kilikali_switchbox_get_property;

    widget_class = (GtkWidgetClass *) klass;

    widget_class->size_request = kilikali_switchbox_size_request;
    widget_class->size_allocate = kilikali_switchbox_size_allocate;
    widget_class->realize = kilikali_switchbox_realize;
    widget_class->map = kilikali_switchbox_map;
    widget_class->unmap = kilikali_switchbox_unmap;
    widget_class->expose_event = kilikali_switchbox_expose_event;
    widget_class->button_press_event = kilikali_switchbox_button_press_event;
    widget_class->button_release_event = kilikali_switchbox_button_release_event;

    g_object_class_install_property (gobject_class,
    PROP_ANIMATE,
    g_param_spec_boolean ("animate",
                            "Enable animations",
                            "Animate the child widget switches",
                            TRUE,
                            G_PARAM_READWRITE));

}

static void
kilikali_switchbox_set_property ( GObject * object, 
                               guint prop_id,
                               const GValue * value, 
                               GParamSpec * pspec)
{
  KilikaliSwitchBox *sb;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (KILIKALI_IS_SWITCHBOX(object));

  sb = KILIKALI_SWITCHBOX(object);

  switch (prop_id) {
    case PROP_ANIMATE:
      sb->private->animate = g_value_get_boolean(value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
kilikali_switchbox_get_property (GObject * object, 
                              guint prop_id,
                              GValue * value, 
                              GParamSpec * pspec)
{
  KilikaliSwitchBox *sb;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (KILIKALI_IS_SWITCHBOX(object));

  sb = KILIKALI_SWITCHBOX(object);

  switch (prop_id) {
    case PROP_ANIMATE:
      g_value_set_boolean(value, sb->private->animate);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

GtkWidget *
kilikali_switchbox_new ( void )
{
  return GTK_WIDGET(g_object_new (KILIKALI_TYPE_SWITCHBOX, NULL));
}

static void
kilikali_switchbox_realize (GtkWidget   *widget)
{
    GdkWindowAttr attributes;
    gint attributes_mask;
    gint border_width;
    KilikaliSwitchBox *sb;
    KilikaliSwitchBoxPrivate *priv;

    sb = KILIKALI_SWITCHBOX(widget);
    priv = sb->private;

    GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
    border_width = GTK_CONTAINER (widget)->border_width;

    attributes.window_type = GDK_WINDOW_CHILD;
    attributes.x = widget->allocation.x;
    attributes.y = widget->allocation.y;
    attributes.width = widget->allocation.width;
    attributes.height = widget->allocation.height;
    attributes.wclass = GDK_INPUT_OUTPUT;
    attributes.event_mask = GDK_ALL_EVENTS_MASK;
    /*tk_widget_get_events (widget) | GDK_EXPOSURE_MASK;*/
    
    attributes_mask = GDK_WA_X | GDK_WA_Y;

    widget->window = gdk_window_new (gtk_widget_get_parent_window (widget),
                                     &attributes, attributes_mask);
    gdk_window_set_user_data(widget->window, widget);
    
    widget->style = gtk_style_attach (widget->style, widget->window);
}

static void
kilikali_switchbox_map (GtkWidget *widget)
{
    KilikaliSwitchBox *sb;
    KilikaliSwitchBoxPrivate *priv;

    sb = KILIKALI_SWITCHBOX(widget);
    priv = sb->private;
  
    GTK_WIDGET_CLASS (kilikali_switchbox_parent_class)->map (widget);

    gdk_window_show (widget->window);
}

static void
kilikali_switchbox_unmap (GtkWidget *widget)
{
    KilikaliSwitchBox *sb;
    KilikaliSwitchBoxPrivate *priv;

    sb = KILIKALI_SWITCHBOX(widget);
    priv = sb->private;
    
    gdk_window_hide (widget->window);

    GTK_WIDGET_CLASS (kilikali_switchbox_parent_class)->unmap (widget);
}

static void
kilikali_switchbox_add (GtkContainer   *container,
                        GtkWidget      *widget,
                        gpointer        user_data)
{
    KilikaliSwitchBox *sb;
    KilikaliSwitchBoxPrivate *priv;
    GtkBoxChild *child;
    GList *child_node;

    sb = KILIKALI_SWITCHBOX(container);
    priv = sb->private;

    child_node = g_list_last(GTK_BOX(container)->children);
    if (child_node != NULL)
    {
        child = (GtkBoxChild *) child_node->data;
        if (priv->visible_child != NULL)
            gtk_widget_hide(priv->visible_child);
        priv->visible_child = child->widget;
        gtk_widget_show(priv->visible_child);
    }
}

static void
kilikali_switchbox_remove (GtkContainer   *container,
                           GtkWidget      *widget,
                           gpointer        user_data)
{
    KilikaliSwitchBox *sb;
    KilikaliSwitchBoxPrivate *priv;
    GList *child_node;

    sb = KILIKALI_SWITCHBOX(container);
    priv = sb->private;

    child_node = GTK_BOX(sb)->children;
    
    while (child_node != NULL)
    {
        GtkBoxChild *child = (GtkBoxChild *) child_node->data;
        if (priv->visible_child == child->widget)
        {
            if (child_node->next != NULL)
            {
                GtkBoxChild *new = (GtkBoxChild *) child_node->next->data;
                gtk_widget_hide(priv->visible_child);
                priv->visible_child = new->widget;
                gtk_widget_show(priv->visible_child);
            } else {
                /* Will be NULL if list is empty */
                child_node = g_list_first(GTK_BOX(sb)->children);
                if (child_node != NULL)
                {
                    GtkBoxChild *new = (GtkBoxChild *) child_node->data;
                    gtk_widget_hide(priv->visible_child);
                    priv->visible_child = new->widget;
                    gtk_widget_show(priv->visible_child);
                }
            }
            gtk_widget_queue_draw(widget);
            break;
        }
        child_node = child_node->next;
    }
}

static void 
kilikali_switchbox_size_request  (GtkWidget      *widget,
                                  GtkRequisition *requisition)
{

    KilikaliSwitchBox *sb;
    KilikaliSwitchBoxPrivate *priv;
    GList *child_node;
    gint border;

    sb = KILIKALI_SWITCHBOX(widget);
    priv = sb->private;

    border = GTK_CONTAINER (widget)->border_width;
    requisition->width = border;
    requisition->height = border;

    /* Collect child requisitions */
    for (child_node = GTK_BOX(widget)->children;
         child_node != NULL;
         child_node = child_node->next)
    {
        GtkRequisition child_req;
        GtkBoxChild *child = (GtkBoxChild *) child_node->data;
        gtk_widget_size_request(child->widget, &child_req);
        requisition->width += child_req.width + child->padding;

        /* Request according to tallest child */
        if (requisition->height < child_req.height + border)
        {
            requisition->height = child_req.height + border;
        }
    }

    /* Add our switcher and borders to it */
    /* FIXME: Wonder where this should come from? */
    requisition->width += 48 + border;
    requisition->height += border;

    /* Make sure we always request space at least for the icon */
    requisition->width = MAX(48 + 2*border, requisition->width);
    requisition->height = MAX(48 + 2*border, requisition->height);
}

static void
kilikali_switchbox_size_allocate (GtkWidget      *widget,
                                  GtkAllocation  *allocation)
{
    KilikaliSwitchBox *sb;
    KilikaliSwitchBoxPrivate *priv;
    GList *child_node;
    GtkAllocation child_alloc;
    gint border;

    sb = KILIKALI_SWITCHBOX(widget);
    priv = sb->private;

    border = GTK_CONTAINER (widget)->border_width;

    /* Every child gets allocated our width minus the switcher
     * and allocation height.
     */

    widget->allocation = *allocation;

    if (GTK_WIDGET_REALIZED (widget))
    {
        gdk_window_move_resize (widget->window,
        		    widget->allocation.x,
        		    widget->allocation.y,
        		    widget->allocation.width,
        		    widget->allocation.height);
    }
    
    child_alloc = *allocation;

    child_alloc.x = border;
    child_alloc.y = border;
    child_alloc.width -= 48 + border * 2;
    child_alloc.height -= border * 2;

    for (child_node = GTK_BOX(widget)->children;
         child_node != NULL;
         child_node = child_node->next)
    {
        GtkBoxChild *child = (GtkBoxChild *) child_node->data;
        gtk_widget_size_allocate(child->widget, &child_alloc);
    }

    gtk_widget_queue_draw(widget);
}


static gboolean
kilikali_switchbox_expose_event (GtkWidget *widget,
                                 GdkEventExpose *event)
{
    KilikaliSwitchBox *sb;
    KilikaliSwitchBoxPrivate *priv;
    gint border;

    sb = KILIKALI_SWITCHBOX(widget);
    priv = sb->private;

    border = GTK_CONTAINER (widget)->border_width;

    if (priv->switch_icon == NULL)
    {
        GtkIconTheme *theme;

        theme = gtk_icon_theme_get_default();
        priv->switch_icon = gtk_icon_theme_load_icon(theme, GTK_STOCK_REFRESH,
                                                     32, 0, NULL);
    }

    if (priv->visible_child != NULL
        && GTK_WIDGET_NO_WINDOW(priv->visible_child))
    {
        /* Clear bg (this should happen automatically, but doesn't?) */
        gdk_draw_rectangle(widget->window,
                           widget->style->base_gc[GTK_WIDGET_STATE(widget)],
                           TRUE,
                           0, 0, 
                           widget->allocation.width, widget->allocation.height);
    }
    
    if (priv->visible_child != NULL)
    {
        gtk_container_propagate_expose (GTK_CONTAINER(widget),
                                        priv->visible_child,
                                        event);
    }

    if (priv->switch_icon != NULL)
    {
        gint icon_x, icon_y;
        
        icon_x = border;
        icon_x += (widget->allocation.width - border*2) - 48;
        icon_x += 4;
        
        icon_y = border;
        icon_y += (widget->allocation.height - border*2) / 2;
        icon_y -= 20;
        
        gdk_draw_pixbuf(widget->window,
                        widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                        priv->switch_icon,
                        0, 0,
                        icon_x, icon_y,
                        32, 32,
                        GDK_RGB_DITHER_NONE, 0, 0);
    }

    return FALSE;
}

static gboolean
kilikali_switchbox_button_press_event (GtkWidget *widget,
                                       GdkEventButton *event)
{
    return FALSE;
}

static gboolean
kilikali_switchbox_button_release_event (GtkWidget *widget,
                                         GdkEventButton *event)
{
    gint border;
    GList *child_node;
    KilikaliSwitchBox *sb;
    KilikaliSwitchBoxPrivate *priv;

    sb = KILIKALI_SWITCHBOX(widget);
    priv = sb->private;
    
    border = GTK_CONTAINER (widget)->border_width;

    if (event->x < (widget->allocation.width - border) - 48)
    {
        return FALSE;
    }
    
    child_node = GTK_BOX(widget)->children;
    
    while (child_node != NULL)
    {
        GdkRectangle rect;
        
        GtkBoxChild *child = (GtkBoxChild *) child_node->data;
        if (priv->visible_child == child->widget)
        {
            if (child_node->next != NULL)
            {
                GtkBoxChild *new = (GtkBoxChild *) child_node->next->data;
                gtk_widget_hide(priv->visible_child);
                priv->visible_child = new->widget;
            } else {
                /* Will be NULL if list is empty */
                child_node = g_list_first(GTK_BOX(widget)->children);
                if (child_node != NULL)
                {
                    GtkBoxChild *new = (GtkBoxChild *) child_node->data;
                    gtk_widget_hide(priv->visible_child);
                    priv->visible_child = new->widget;
                }
            }
            gtk_widget_show(priv->visible_child);
            rect.x = 0;
            rect.y = 0;
            rect.width = widget->allocation.width;
            rect.height = widget->allocation.height;
            gdk_window_invalidate_rect (widget->window, &rect, TRUE);
            break;
        }
        child_node = child_node->next;
    }

    return TRUE;
}

/* Emacs indentatation information
   Local Variables:
   indent-tabs-mode:nil
   tab-width:4
   c-set-offset:4
   c-basic-offset:4
   End: 
*/
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

