/*
 * * Copyright (C) 2007 Tuomas Kulve <tuomas@kulve.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "kilikali-plugin.h"

/* Definition of HildonStatusBarPluginFn_st */
#include <hildon-status-bar-lib/hildon-status-bar-item.h>
#include <gtk/gtk.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>


struct kilikali_plugin {
    HildonStatusBarItem *item;
    GtkWidget *button;
    GtkUIManager *uimanager;
    GtkActionGroup *actions;

    osso_context_t *osso;
};

/* definition for status bar */
void kilikalisb_entry(HildonStatusBarPluginFn_st *fn);

void *kilikalisb_initialize(HildonStatusBarItem *item, GtkWidget **button);
void kilikalisb_destroy(void *data);
int kilikalisb_get_priority(void *data);
void kilikalisb_set_conditional(void *data, gboolean visible);

static void kilikalisb_clicked(GtkButton *button, gpointer user_data);
static DBusHandlerResult signal_filter(DBusConnection *connection, 
                                       DBusMessage *message,
                                       void *user_data);
static void popup_place(GtkMenu *menu, 
                        gint *x, gint *y,
                        gboolean *push_in,
                        gpointer data);
static void play_action(GtkAction *action, struct kilikali_plugin *kilikali);
static void pause_action(GtkAction *action, struct kilikali_plugin *kilikali);
static void stop_action(GtkAction *action, struct kilikali_plugin *kilikali);
static void prev_action(GtkAction *action, struct kilikali_plugin *kilikali);
static void next_action(GtkAction *action, struct kilikali_plugin *kilikali);




/* Menuitems */
static const GtkActionEntry entries[] = {
    { "Play", GTK_STOCK_MEDIA_PLAY, "_Play", "<control>P",
      NULL,
      G_CALLBACK(play_action) },
    { "Pause", GTK_STOCK_MEDIA_PAUSE, "_Pause", "<control>U",
      NULL,
       G_CALLBACK(pause_action) },
    { "Stop", GTK_STOCK_MEDIA_STOP, "_Stop", "<control>S",
      NULL,
       G_CALLBACK(stop_action) },
    { "Next", GTK_STOCK_MEDIA_NEXT, "_Next", "<control>N",
      NULL,
       G_CALLBACK(next_action) },
    { "Previous", GTK_STOCK_MEDIA_PREVIOUS, "_Previous", "<control>B",
      NULL,
      G_CALLBACK(prev_action) },
};

void
kilikalisb_entry(HildonStatusBarPluginFn_st *fn)
{
    g_debug("entry");

    if (fn == NULL) {
        return;
    }
    fn->initialize = kilikalisb_initialize;
    fn->destroy = kilikalisb_destroy;
    fn->get_priority = kilikalisb_get_priority;
    fn->set_conditional = kilikalisb_set_conditional;
}



void *
kilikalisb_initialize(HildonStatusBarItem *item, GtkWidget **button)
{
    struct kilikali_plugin *kilikali;
    DBusConnection* conn;

    g_debug("initialize: item: %s", (item == NULL ? "NULL" : "not NULL"));

    kilikali = g_new0(struct kilikali_plugin, 1);

    kilikali->item = item;
    *button = NULL;

    /* Initialize libosso */
    kilikali->osso = osso_initialize("kilikalisb", "0.0", FALSE, NULL);
    if (kilikali->osso == NULL) {
        g_warning("Failed to initialize libosso");
        return FALSE;
    }

    kilikali->uimanager = gtk_ui_manager_new();

    kilikali->actions = gtk_action_group_new("MenuActions");
    gtk_action_group_add_actions (kilikali->actions, entries,
                                  G_N_ELEMENTS(entries), kilikali);
    gtk_ui_manager_insert_action_group(kilikali->uimanager,
                                       kilikali->actions, -1);

    if (gtk_ui_manager_add_ui_from_file(kilikali->uimanager,
                                        PACKAGE_DATA_DIR "/sbactions.xml",
                                        NULL) == 0) {
        g_error("Failed to load actions files %s/%s",
                PACKAGE_DATA_DIR, "actions.xml");
    }

    conn = osso_get_dbus_connection(kilikali->osso);
    if (!conn) {
        g_warning("Failed to get D-BUS connection");
        return (void *)kilikali;
    }
    
    /* listening to messages from all objects as no path is specified */
    dbus_bus_add_match(conn,
                       "type='signal',interface='org.freedesktop.MusicPlayer'",
                       NULL);
    dbus_connection_add_filter(conn, signal_filter,
                               (void *)kilikali,
                               NULL);
    return (void *)kilikali;
}



/* Called by the desktop process whenever the
 * hildon_status_bar_update_conditional is emitted on the
 * SB item */ 
void kilikalisb_set_conditional(void *data, gboolean visible)
{ 
    struct kilikali_plugin *kilikali;
    GtkWidget *image = NULL;

    g_debug("set_conditional");

    if (!data) {
        return; 
    }

    kilikali = data;

    if (visible) { 
        if (!kilikali->button) { 
            
            kilikali->button = gtk_button_new();

            /* Should be 40x40 */
            image = gtk_image_new_from_icon_name("kilikali",
                                                 GTK_ICON_SIZE_BUTTON);
            gtk_container_add(GTK_CONTAINER(kilikali->button),
                              GTK_WIDGET(image));
                        

            g_signal_connect(kilikali->button, "clicked",
                             G_CALLBACK(kilikalisb_clicked), kilikali);
            gtk_widget_show_all(kilikali->button);
        } 
    } else { 
        gtk_widget_destroy(kilikali->button); 
        kilikali->button = NULL;
    }

    hildon_status_bar_item_set_button(kilikali->item, kilikali->button);
}


void
kilikalisb_destroy(void *data)
{
    struct kilikali_plugin *kilikali;

    g_debug("destroy");

    if (!data) {
        return;
    }

    kilikali = data;

    osso_deinitialize(kilikali->osso);

    g_free(data);
}



gint
kilikalisb_get_priority(void *data)
{
    g_debug("get_priority");

    return 1;
}



static void
kilikalisb_clicked(GtkButton *button, gpointer user_data)
{
    struct kilikali_plugin *kilikali;
    GtkWidget *menu;

    g_debug("clicked");

    g_return_if_fail(user_data != NULL);

    kilikali = user_data;

    menu = gtk_ui_manager_get_widget(kilikali->uimanager, "/ui/StatusbarMenu");

	gtk_widget_show_all(menu);

	gtk_menu_popup(GTK_MENU(menu), NULL, NULL, popup_place, button, 1,
                   gtk_get_current_event_time());

	g_signal_connect(menu,
                     "selection-done",
                     G_CALLBACK(gtk_widget_hide), NULL);
}



static void
popup_place(GtkMenu *menu, gint *x, gint *y, gboolean *push_in, gpointer data)
{

	GtkWidget *button = GTK_WIDGET(data);

	gdk_window_get_position(button->window, x, y);

	*y += button->allocation.y + button->allocation.height;
	*x += button->allocation.x;
}



static DBusHandlerResult
signal_filter(DBusConnection *connection, DBusMessage *message, void *user_data)
{
    struct kilikali_plugin *kilikali;
    
    g_debug("got dbus signal");
    
    g_return_val_if_fail(user_data != NULL, 
                         DBUS_HANDLER_RESULT_NOT_YET_HANDLED);

    kilikali = user_data;

    if (dbus_message_is_signal(message,
                               "org.freedesktop.MusicPlayer", 
                               "active")) {
        DBusError error;
        gboolean active;
        dbus_error_init (&error);

        if (dbus_message_get_args(message,
                                  &error,
                                  DBUS_TYPE_BOOLEAN, &active,
                                  DBUS_TYPE_INVALID)) {
            g_debug("active signal received: %d", active);

            g_debug("emitting, item: %s", 
                    (kilikali->item == NULL ? "NULL" : "not NULL"));

            g_signal_emit_by_name(kilikali->item,
                                  "hildon_status_bar_update_conditional",
                                  active);

       } else {
            g_warning("Active received, but error getting message: %s\n",
                      error.message);
            dbus_error_free(&error);
        }
    }
    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static void
play_action(GtkAction *action, struct kilikali_plugin *kilikali)
{
   g_return_if_fail(kilikali != NULL);
   kilikali_plugin_play(kilikali->osso);
}


static void
pause_action(GtkAction *action, struct kilikali_plugin *kilikali)
{
   g_return_if_fail(kilikali != NULL);
   kilikali_plugin_pause(kilikali->osso);
}



static void
stop_action(GtkAction *action, struct kilikali_plugin *kilikali)
{
   g_return_if_fail(kilikali != NULL);
   kilikali_plugin_stop(kilikali->osso);
}



static void
prev_action(GtkAction *action, struct kilikali_plugin *kilikali)
{
   g_return_if_fail(kilikali != NULL);
   kilikali_plugin_prev(kilikali->osso);
}



static void
next_action(GtkAction *action, struct kilikali_plugin *kilikali)
{
   g_return_if_fail(kilikali != NULL);
   kilikali_plugin_next(kilikali->osso);
}



/* Emacs indentatation information
   Local Variables:
   indent-tabs-mode:nil
   tab-width:4
   c-set-offset:4
   c-basic-offset:4
   End: 
*/
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4

