/*
 * * Copyright (C) 2007 Tuomas Kulve <tuomas@kulve.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "kilikali.h"
#include "kilikali-ui.h"
#include "kilikali-osso.h"
#include "kilikali-plugin.h"
#include "kilikali-vfs.h"

#include <stdio.h>                /* fprintf */
#include <glib.h>
#include <libosso.h>
#include <gtk/gtk.h>              /* GtkTreeModel */
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

/* module global osso context */
static osso_context_t *osso;

static gint dbus_handler(const gchar * interface,
                         const gchar * method,
                         GArray * arguments,
                         gpointer data, 
                         osso_rpc_t * retval);
static void hw_event_handler(osso_hw_state_t *state, gpointer data);
static void dbus_signal_player_active(osso_context_t *osso, gboolean active);
static DBusHandlerResult action_signal_filter(DBusConnection *connection, 
                                              DBusMessage *message,
                                              void *user_data);
static gboolean _inhibit(gpointer user_data);

gboolean kilikali_osso_init(KilikaliUI *ui)
{
    osso_return_t osso_ret;
    DBusConnection* conn;
    DBusError error;

    /* Initialize libosso */
    osso = osso_initialize(PACKAGE, VERSION, FALSE, NULL);
    if (osso == NULL) {
        fprintf(stderr, "Failed to initialize libosso\n");
        return FALSE;
    }

    /* Set dbus handler to get mime open callbacks */
    osso_ret = osso_rpc_set_cb_f(osso,
                                 KILIKALI_DBUS_SERVICE,
                                 KILIKALI_DBUS_OBJECT_PATH,
                                 KILIKALI_DBUS_INTERFACE,
                                 dbus_handler, ui);
    if (osso_ret != OSSO_OK) {
        fprintf(stderr, "Failed to set mime callback\n");
        return FALSE;
    }

    /* Add handler for hardware D-BUS messages */
    osso_ret = osso_hw_set_event_cb(osso, NULL, hw_event_handler, NULL);
    if (osso_ret != OSSO_OK) {
        fprintf(stderr, "Error setting HW state callback (%d)\n", osso_ret);
        return FALSE;
    }  

    dbus_error_init (&error);

    conn = osso_get_dbus_connection(osso);
    if (!conn) {
        g_warning("Failed to connect to the D-BUS daemon: %s", error.message);
        dbus_error_free(&error);
        return TRUE;
    }

    /* listening to messages from all objects as no path is specified */
    dbus_bus_add_match(conn,
                       "type='signal',interface='org.freedesktop.MusicPlayer'",
                       NULL);
    dbus_connection_add_filter(conn, action_signal_filter,
                               ui,
                               NULL);

    return TRUE;
}


void kilikali_osso_deinit(void)
{
    osso_deinitialize(osso);
}



/*
 * Private functions
 */



static gint dbus_handler(const gchar *interface,
                         const gchar *method,
                         GArray *arguments,
                         gpointer data,
                         osso_rpc_t *retval)
{
    KilikaliUI *ui;
    GtkTreeModel *model;

    g_debug("in dbus_handler");

    if (data == NULL) {
        return OSSO_ERROR;
    }        

    if (arguments == NULL) {
        return OSSO_OK;
    }

    /* get model ui property */
    ui = data;
    g_object_get(G_OBJECT(ui), "model", &model, NULL);

    /* check for mime open */
    if (g_ascii_strcasecmp(method, "mime_open") == 0) {
        osso_rpc_t val;
        gint i;

        for(i = 0; i < arguments->len; ++i) {
            val = g_array_index(arguments, osso_rpc_t, i);
            if (val.type == DBUS_TYPE_STRING && val.value.s != NULL) {
                g_idle_add(vfs_add_file_from_idle,
                           vfs_modeluri(model, val.value.s));
            }
        }
    }

    return OSSO_OK;
}



/* Callback for hardware D-BUS events */
static void hw_event_handler(osso_hw_state_t *state, gpointer data)
{

    /* The device is about to be shut down */
    if (state->shutdown_ind)
    {
        g_debug("The device is about to be shut down");
        kilikali_quit();
    }

    /* The system is low on memory */
    if (state->memory_low_ind)
    {
        g_debug("The system is low on memory");
        /* FIXME: implement user setting for this */
        /*
          if (TRUE)
          save_auto_playlist();
        */
        kilikali_quit();
        
    }

    /* The application should save unsaved user data */
    if (state->save_unsaved_data_ind)
    {
        g_debug("The application should save unsaved user data");        

        /* FIXME: implement user setting for this */
        /*
          if (TRUE)
          save_auto_playlist();
        */
    }

    /*
     * libosso header: The application should reduce its activity 
     * truth         : display was blanked.
     */
    if (state->system_inactivity_ind)
    {
        g_debug("The display was blanked");
    }
}


gboolean kilikali_osso_start_sb(gpointer data)
{
    g_debug("Loading sb plugin");

    dbus_signal_player_active(osso, TRUE);

    return FALSE;
}


gboolean kilikali_osso_stop_sb(gpointer data)
{
    g_debug("Unloading sb plugin");

    dbus_signal_player_active(osso, FALSE);

    return FALSE;
}



static void
dbus_signal_player_active(osso_context_t *osso, gboolean active)
{
    DBusConnection* conn;
    DBusMessage* msg;
    DBusError error;

    dbus_error_init (&error);

    conn = osso_get_dbus_connection(osso);
    if (!conn) {
        g_warning ("Failed to connect to the D-BUS daemon: %s", error.message);
        dbus_error_free (&error);
        return;
    }
    
    /* send signal telling that mediaplayer is running */

    /* create a signal and check for errors  */
    msg = dbus_message_new_signal
        ("/org/freedesktop/MusicPlayer", /* object name of the signal */
         "org.freedesktop.MusicPlayer", /* interface name of the signal */
         "active");                     /* name of the signal */
    if (msg == NULL) { 
        g_warning("Failed to create dbus message"); 
        return;
    }

   /* append arguments onto signal */
    dbus_message_append_args(msg,
                             DBUS_TYPE_BOOLEAN, &active,
                             DBUS_TYPE_INVALID);

   /* send the message and flush the connection */
   if (!dbus_connection_send(conn, msg, NULL)) { 
       g_warning("Failed to send dbus msg"); 
      return;
   }

   dbus_connection_flush(conn);

   /* free the message */
   dbus_message_unref(msg);

}



static DBusHandlerResult
action_signal_filter(DBusConnection *connection,
                     DBusMessage *message,
                     void *user_data)
{
    KilikaliUI *ui;

    g_debug("got dbus signal");

    g_return_val_if_fail(user_data != NULL, 
                         DBUS_HANDLER_RESULT_NOT_YET_HANDLED);

    ui = user_data;

    if (dbus_message_is_signal(message,
                               "org.freedesktop.MusicPlayer", 
                               "action")) {
        DBusError error;
        gint action;
        dbus_error_init(&error);

        if (dbus_message_get_args(message,
                                  &error,
                                  DBUS_TYPE_INT32, &action,
                                  DBUS_TYPE_INVALID)) {
            g_debug("action signal received: %d", action);

            kilikali_ui_action(ui, action);

        } else {
            g_warning("Action received, but error getting message: %s\n",
                      error.message);
            dbus_error_free(&error);
        }
    }
    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

guint inhibit_timer;

static
gboolean _inhibit(gpointer user_data)
{
    osso_display_blanking_pause(osso);
    return TRUE;
}

void kilikali_osso_inhibit_blanking()
{
    _inhibit(NULL);
    if (inhibit_timer > 0)
    {
        return;
    }
    
    /* The inhibit function needs to be called every 60s to keep the display
     * on (bleh). g_timeouts are not guaranteed to be exact so we use 55s to
     * be sure.
     */
    inhibit_timer = g_timeout_add(55000, _inhibit, NULL);
}

void kilikali_osso_allow_blanking()
{
    if (inhibit_timer > 0)
    {
        g_source_remove(inhibit_timer);
        inhibit_timer = 0;
    }
}

/* Emacs indentatation information
   Local Variables:
   indent-tabs-mode:nil
   tab-width:4
   c-set-offset:4
   c-basic-offset:4
   End: 
*/
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4
