/*
 * * Copyright (C) 2007 Joni Valtanen <jvaltane@kapsi.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

#include <glib.h>
#include <gst/gst.h>

#include <string.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

/* Base bins */
#include "kilikali-base-bin.h"

#define KILIKALI_TYPE_BIN               (kilikali_bin_get_type())
#define KILIKALI_BIN(obj)               (G_TYPE_CHECK_INSTANCE_CAST((obj),KILIKALI_TYPE_BIN,KilikaliBin))
#define KILIKALI_BIN_CLASS(klass)       (G_TYPE_CHECK_CLASS_CAST((klass),KILIKALI_TYPE_BIN,KilikaliBinClass))
#define KILIKALI_IS_BIN(obj)            (G_TYPE_CHECK_INSTANCE_TYPE((obj),KILIKALI_TYPE_BIN))
#define KILIKALI_IS_BIN_CLASS(klass)    (G_TYPE_CHECK_CLASS_TYPE((klass),KILIKALI_TYPE_BIN))


typedef struct _KilikaliBin KilikaliBin;
typedef struct _KilikaliBinClass KilikaliBinClass;


struct _KilikaliBinClass
{
    KilikaliBaseBinClass parent_class;
};

struct _KilikaliBin 
{
    KilikaliBaseBin parent;

    GList *factories;

    gboolean mute;
    gdouble volume;
    gchar *uri;
    gchar *iradioname;

    /* SID stuff */
    gint sid_tune;
    gint sid_clock;
    gint sid_memory;
    gboolean sid_filter;
    gboolean sid_measured_volume;
    gboolean sid_mos8580;
    gboolean sid_force_speed;

};


/* properties */
enum 
    {
        ARG_0 = 0,
        ARG_MUTE,
        ARG_VOLUME,
        ARG_IRADIO_NAME,
        ARG_URI,
        ARG_SID_TUNE,
        ARG_SID_CLOCK,
        ARG_SID_MEMORY,
        ARG_SID_FILTER,
        ARG_SID_MEASURED_VOLUME,
        ARG_SID_MOS8580,
        ARG_SID_FORCE_SPEED
    };


static GType kilikali_bin_get_type (void);


static void kilikali_bin_class_init (KilikaliBinClass *klass);
static void kilikali_bin_init (KilikaliBin *bin);
static void kilikali_bin_dispose (GObject *object);


static void kilikali_bin_set_property (GObject *object, guint prop_id,
                                              const GValue *value, 
                                              GParamSpec *spec);
static void kilikali_bin_get_property (GObject *object, guint prop_id,
                                              GValue *value, GParamSpec *spec);

//static void kilikali_bin_handle_message (GstBin *bin, GstMessage *msg);

static GstStateChangeReturn 
kilikali_bin_change_state (GstElement *element,
                                  GstStateChange transition);

static GstElementClass *parent_class;


static const GstElementDetails kilikali_bin_details =
    GST_ELEMENT_DETAILS ("Kilikali Bin",
                         "Audio/Bin/Player",
                         "Autoplug and play audio from an uri",
                         "Joni Valtanen <jvaltane@kapsi.fi>");

static void
kilikali_bin_class_init (KilikaliBinClass *klass)
{
    GObjectClass *gobject_klass;
    GstElementClass *gstelement_klass;
    GstBinClass *gstbin_klass;

    gobject_klass = (GObjectClass *)klass;
    gstelement_klass = (GstElementClass *)klass;
    gstbin_klass = (GstBinClass *)klass;

    parent_class = g_type_class_peek_parent (klass);

    gobject_klass->set_property = kilikali_bin_set_property;
    gobject_klass->get_property = kilikali_bin_get_property;
    gobject_klass->dispose = kilikali_bin_dispose;


    //    gstbin_klass->handle_message = kilikali_bin_handle_message;

    gstelement_klass->change_state = kilikali_bin_change_state;


    g_object_class_install_property (gobject_klass, ARG_URI,
        g_param_spec_string ("uri", "URI", "URI to play", NULL,
            G_PARAM_READWRITE));

    g_object_class_install_property (gobject_klass, ARG_VOLUME,
        g_param_spec_double ("volume", "Volume", "Audio volume", 0.0, 1.0, 1.0, 
            G_PARAM_READWRITE));

    g_object_class_install_property (gobject_klass, ARG_MUTE,
        g_param_spec_boolean ("mute", "Mute", "Audio mute", FALSE, 
            G_PARAM_READWRITE));

    g_object_class_install_property (gobject_klass, ARG_IRADIO_NAME,
        g_param_spec_string ("iradio-name", "internet radio name", 
            "Currently playing internet radio name", NULL, G_PARAM_READABLE));

    g_object_class_install_property (gobject_klass, ARG_SID_TUNE,
        g_param_spec_int ("sid-tune", "sid tune", "Selected SID tune", 0, 100, 
                          0, G_PARAM_READWRITE));

    g_object_class_install_property (gobject_klass, ARG_SID_CLOCK,
        g_param_spec_int ("sid-clock", "sid clock", "Selected SID clock pal", 1,
                          2, 1, G_PARAM_READWRITE));

    g_object_class_install_property (gobject_klass, ARG_SID_MEMORY,
        g_param_spec_int ("sid-memory", "sid memory", "Selected SID memorybank",
                          32, 34, 32, G_PARAM_READWRITE));

    g_object_class_install_property (gobject_klass, ARG_SID_FILTER,
        g_param_spec_boolean ("sid-filter", "SID filter", "Use of SID filter", 
                              FALSE, G_PARAM_READWRITE));

    g_object_class_install_property (gobject_klass, ARG_SID_MEASURED_VOLUME,
        g_param_spec_boolean ("sid-measured-volume", "SID measured volume", 
                              "Use of SID measured volume", 
                              FALSE, G_PARAM_READWRITE));

    g_object_class_install_property (gobject_klass, ARG_SID_MOS8580,
        g_param_spec_boolean ("sid-mos8580", "SID mos8580", "Use of SID mos8580", 
                              FALSE, G_PARAM_READWRITE));

    g_object_class_install_property (gobject_klass, ARG_SID_FORCE_SPEED,
        g_param_spec_boolean ("sid-force-speed", "SID mforce speed", 
                              "Use of SID force speed", FALSE, 
                              G_PARAM_READWRITE));

    gst_element_class_set_details (gstelement_klass, &kilikali_bin_details);

}

static void
kilikali_bin_init (KilikaliBin *bin)
{
    bin->mute = FALSE; /* No mute */
    bin->uri = NULL;
    bin->volume = 1.0; /* Max volume */

    /* Default sid values */
    bin->sid_tune = 0;
    bin->sid_clock = 1;
    bin->sid_memory = 32;
    bin->sid_filter = FALSE;
    bin->sid_measured_volume = FALSE;
    bin->sid_mos8580 = FALSE;
    bin->sid_force_speed = FALSE;

}

static void
kilikali_bin_dispose (GObject *object)
{
    KilikaliBin *bin = KILIKALI_BIN (object);

    g_debug (__FUNCTION__);

    if (bin->uri != NULL)
        g_free (bin->uri);
    if (bin->iradioname != NULL)
        g_free (bin->iradioname);

}

static void
kilikali_bin_set_property (GObject *object, guint pid, 
                                  const GValue *value, GParamSpec *pspec)
{
    KilikaliBin *bin = KILIKALI_BIN (object);
    KilikaliBaseBin *base_bin = KILIKALI_BASE_BIN (object);

    g_return_if_fail (KILIKALI_IS_BIN (object));
      
    switch (pid) {
    case ARG_URI: {
        const gchar *uri = g_value_get_string (value);

        if (uri == NULL) {
            return;
        }

        if (base_bin->uri == NULL || strcmp (base_bin->uri, uri) != 0) {
            g_free (base_bin->uri);
            base_bin->uri = g_strdup (uri);
        }
        break;
    }
    case ARG_MUTE:
        bin->mute = g_value_get_boolean (value);
        if (base_bin != NULL) {
            base_bin->set_mute(base_bin, bin->mute);            
        }
        break;
    case ARG_VOLUME:
        bin->volume = g_value_get_double (value);
        if(base_bin != NULL) {
            base_bin->set_volume(base_bin, bin->volume);
        }
        break;
    case ARG_SID_TUNE:
        bin->sid_tune = g_value_get_int (value);
        if(base_bin->siddec != NULL) {
            g_object_set( G_OBJECT (base_bin->siddec),
                          "tune", (gint)bin->sid_tune,
                          NULL);
        }
        break;
    case ARG_SID_CLOCK:
        bin->sid_clock = g_value_get_int (value);
        if(base_bin->siddec != NULL) {
            g_object_set( G_OBJECT (base_bin->siddec),
                          "clock", bin->sid_clock,
                          NULL);
        }
        break;
    case ARG_SID_MEMORY:
        bin->sid_memory = g_value_get_int (value);
        if(base_bin->siddec != NULL) {
            g_object_set( G_OBJECT (base_bin->siddec),
                          "memory", bin->sid_memory,
                          NULL);
        }
        break;
    case ARG_SID_FILTER:
        bin->sid_filter = g_value_get_boolean (value);
        if(base_bin->siddec != NULL) {
            g_object_set( G_OBJECT (base_bin->siddec),
                          "filter", bin->sid_filter,
                          NULL);
        }
        break;
    case ARG_SID_MEASURED_VOLUME:
        bin->sid_measured_volume = g_value_get_boolean (value);
        if(base_bin->siddec != NULL) {
            g_object_set( G_OBJECT (base_bin->siddec),
                          "measured-volume", bin->sid_measured_volume,
                          NULL);
        }
        break;
    case ARG_SID_MOS8580:
        bin->sid_mos8580 = g_value_get_boolean (value);
        if(base_bin->siddec != NULL) {
            g_object_set( G_OBJECT (base_bin->siddec),
                          "mos8580", bin->sid_mos8580,
                          NULL);
        }
        break;
    case ARG_SID_FORCE_SPEED:
        bin->sid_force_speed = g_value_get_boolean (value);
        if(base_bin->siddec != NULL) {
            g_object_set( G_OBJECT (base_bin->siddec),
                          "force-speed", bin->sid_force_speed,
                          NULL);
        }
        break;
     default:
        break;
    }

}

static void
kilikali_bin_get_property (GObject *object, guint id, GValue *value, 
                                  GParamSpec *pspec)
{
    KilikaliBin *bin = NULL;
    KilikaliBaseBin *base_bin = NULL;

    g_return_if_fail (KILIKALI_IS_BIN (object));

    bin = KILIKALI_BIN (object);
    base_bin = KILIKALI_BASE_BIN (object);

    switch (id) {
    case ARG_URI:
        g_value_set_string (value, bin->uri);
        break;
    case ARG_MUTE:
        g_value_set_boolean (value, bin->mute);
        break;
    case ARG_VOLUME:
        g_value_set_double (value, bin->volume);
        break;
    case ARG_IRADIO_NAME:
        g_value_set_string (value, base_bin->iradioname);
        break;
    case ARG_SID_TUNE:
        g_value_set_uint (value, bin->sid_tune);
        break;
    case ARG_SID_CLOCK:
        g_value_set_uint (value, bin->sid_clock);
        break;
    case ARG_SID_MEMORY:
        g_value_set_uint (value, bin->sid_memory);
        break;
    case ARG_SID_FILTER:
        g_value_set_boolean (value, bin->sid_filter);
        break;
    case ARG_SID_MEASURED_VOLUME:
        g_value_set_boolean (value, bin->sid_measured_volume);
        break;
    case ARG_SID_MOS8580:
        g_value_set_boolean (value, bin->sid_mos8580);
        break;
    case ARG_SID_FORCE_SPEED:
        g_value_set_boolean (value, bin->sid_force_speed);
        break;
    default:
        break;
    }
}


static GType
kilikali_bin_get_type (void)
{
    static GType kilikali_bin_type = 0;

    if (!kilikali_bin_type) {
        static const GTypeInfo kilikali_bin_info = {
            sizeof (KilikaliBinClass),
            NULL,
            NULL,
            (GClassInitFunc) kilikali_bin_class_init,
            NULL,
            NULL,
            sizeof (KilikaliBin),
            0,
            (GInstanceInitFunc) kilikali_bin_init,
            NULL
    };

        kilikali_bin_type = 
            g_type_register_static (KILIKALI_TYPE_BASE_BIN, "KilikaliBin", 
                &kilikali_bin_info, 0);
    }

    return kilikali_bin_type;
}


#if 0
static void
kilikali_bin_handle_message (GstBin *bin, GstMessage *msg)
{
    GST_BIN_CLASS (parent_class)->handle_message (bin, msg);
}
#endif

static GstStateChangeReturn
kilikali_bin_change_state (GstElement *element, GstStateChange transition)
{
    GstStateChangeReturn ret;
    KilikaliBin *bin = KILIKALI_BIN(element);
    KilikaliBaseBin *base_bin = KILIKALI_BASE_BIN(element);

    g_debug (__FUNCTION__);


    ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);
    if (ret == GST_STATE_CHANGE_FAILURE) {
        return ret;
    }

    if (base_bin != NULL) {
        if(base_bin->siddec != NULL) {
            g_object_set( G_OBJECT (base_bin->siddec),
                          "tune", (gint)bin->sid_tune,
                          "clock", bin->sid_clock,
                          "memory", bin->sid_memory,
                          "filter", bin->sid_filter,
                          "measured-volume", bin->sid_measured_volume,
                          "mos8580", bin->sid_mos8580,
                          "force-speed", bin->sid_force_speed,
                           NULL);
         }
       
    }

    /* Setup volume and mute */
    base_bin->set_volume(base_bin, bin->volume);
    base_bin->set_mute(base_bin, bin->mute);

    return ret;

}


/* gstreamer plugin stuff */
static gboolean
register_elements (GstPlugin *plugin)
{
    return gst_element_register (plugin, "kilikalibin",
                                 GST_RANK_NONE, KILIKALI_TYPE_BIN);
}

GST_PLUGIN_DEFINE_STATIC (
                          GST_VERSION_MAJOR,
                          GST_VERSION_MINOR,
                          "kilikalibin",
                          "kilikali-plugins",
                          register_elements,
                          VERSION,
                          "GPL",
                          "kilikali",
                          "http://kilikali.garage.maemo.org"
                          )


/* Emacs indentatation information
   Local Variables:
   indent-tabs-mode:nil
   tab-width:4
   c-set-offset:4
   c-basic-offset:4
   End: 
*/
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4
