/*
 * * Copyright (C) 2007 Joni Valtanen <jvaltane@kapsi.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

#include <glib.h>
#include <gst/gst.h>

#include <string.h>

#include "kilikali-base-bin.h"

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#define MAX_VOLUME 65535.0

/* signals */
enum
{
  SIGNAL_TYPE_SID,
  SIGNAL_TYPE_UNKNOWN,
  LAST_SIGNAL
};

static void kilikali_base_bin_class_init (KilikaliBaseBinClass *klass);
static void kilikali_base_bin_init (KilikaliBaseBin *base_bin);
static void kilikali_base_bin_dispose (GObject *object);

static GstStateChangeReturn 
kilikali_base_bin_change_state (GstElement *element,
                                  GstStateChange transition);


/* Helpers */
static gboolean _element_remove (gpointer key, gpointer value, gpointer data);
static void _cleanup(KilikaliBaseBin *base_bin);
static gboolean _build_pipeline (KilikaliBaseBin *base_bin);

static gboolean _build_special_sid (KilikaliBaseBin *base_bin, GstPad *pad);
static gboolean _normal_plug (KilikaliBaseBin *base_bin, GstPad *pad, 
                              const GstCaps *caps);

static gboolean _pipeline_failed (KilikaliBaseBin *base_bin);
static gboolean _get_iradio_name (KilikaliBaseBin *base_bin);

/* Callbacks */
static gint _compare_ranks (GstPluginFeature *f1, GstPluginFeature *f2);
static gboolean _feature_filter (GstPluginFeature *feature, gpointer data);
static void _typefound(GstElement *typefind, guint probability, GstCaps *caps,
                       gpointer data);

static void _new_decoded_pad (GstElement* dec, GstPad* pad, gboolean arg1, gpointer data);
static gboolean _build_decodebin (KilikaliBaseBin *base_bin, GstPad *pad);
static void _unknown_type(GstElement *dec, GstPad *pad, GstCaps *caps, gpointer data);
static gboolean _type_sid (KilikaliBaseBin *base_bin);

/* other stuff */
static void _kilikali_base_bin_set_volume (KilikaliBaseBin *base_bin, gdouble volume);
static void _kilikali_base_bin_set_mute (KilikaliBaseBin *base_bin, gboolean mute);

static guint kilikali_base_bin_signals[LAST_SIGNAL] = { 0 };

static GstElementClass *parent_class;


static void
kilikali_base_bin_class_init (KilikaliBaseBinClass *klass)
{
    GObjectClass *gobject_klass;
    GstElementClass *gstelement_klass;
    GstBinClass *gstbin_klass;

    gobject_klass = (GObjectClass *)klass;
    gstelement_klass = (GstElementClass *)klass;
    gstbin_klass = (GstBinClass *)klass;

    parent_class = g_type_class_peek_parent (klass);

    gobject_klass->dispose = kilikali_base_bin_dispose;

    gstelement_klass->change_state = kilikali_base_bin_change_state;

    kilikali_base_bin_signals[SIGNAL_TYPE_UNKNOWN] =
            g_signal_newv ("type-unknown",
                           G_TYPE_FROM_CLASS (klass),
                           G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | 
                           G_SIGNAL_NO_HOOKS,
                           NULL,
                           NULL,
                           NULL,
                           g_cclosure_marshal_VOID__VOID,
                           G_TYPE_NONE,
                           0,
                           NULL);

    kilikali_base_bin_signals[SIGNAL_TYPE_SID] =
            g_signal_newv ("sid-playing",
                           G_TYPE_FROM_CLASS (klass),
                           G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | 
                           G_SIGNAL_NO_HOOKS,
                           NULL,
                           NULL,
                           NULL,
                           g_cclosure_marshal_VOID__VOID,
                           G_TYPE_NONE,
                           0,
                           NULL);

}

static void
kilikali_base_bin_init (KilikaliBaseBin *base_bin)
{
    base_bin->uri = NULL;
    base_bin->iradioname = NULL;

    base_bin->factories = gst_registry_feature_filter (gst_registry_get_default (),
                         (GstPluginFeatureFilter)_feature_filter, FALSE, NULL);
    base_bin->factories = g_list_sort (base_bin->factories, (GCompareFunc)_compare_ranks);

    base_bin->elements = g_hash_table_new_full (g_str_hash, g_str_equal, NULL, 
                         (GDestroyNotify)gst_object_unref);
    base_bin->fakesink = NULL;

    base_bin->set_volume = _kilikali_base_bin_set_volume;
    base_bin->set_mute = _kilikali_base_bin_set_mute;
}

static void
kilikali_base_bin_dispose (GObject *object)
{
    KilikaliBaseBin *base_bin = KILIKALI_BASE_BIN (object);
    g_debug (__FUNCTION__);

    if (base_bin->uri != NULL)
        g_free (base_bin->uri);
    if (base_bin->iradioname != NULL)
        g_free (base_bin->iradioname);
    g_hash_table_foreach_remove (base_bin->elements, _element_remove, (gpointer)base_bin);

}

GType
kilikali_base_bin_get_type (void)
{
    static GType kilikali_base_bin_type = 0;

    if (!kilikali_base_bin_type) {
        static const GTypeInfo kilikali_base_bin_info = {
            sizeof (KilikaliBaseBinClass),
            NULL,
            NULL,
            (GClassInitFunc) kilikali_base_bin_class_init,
            NULL,
            NULL,
            sizeof (KilikaliBaseBin),
            0,
            (GInstanceInitFunc) kilikali_base_bin_init,
            NULL
    };

        kilikali_base_bin_type = 
            g_type_register_static (GST_TYPE_PIPELINE, "KilikaliBaseBin", 
                &kilikali_base_bin_info, 0);
    }

    return kilikali_base_bin_type;
}


static GstStateChangeReturn
kilikali_base_bin_change_state (GstElement * element, GstStateChange transition)
{
    GstStateChangeReturn ret;
    KilikaliBaseBin *base_bin = NULL;

    g_debug (__FUNCTION__);

    base_bin = KILIKALI_BASE_BIN (element);

    ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

    switch (transition) {
    case GST_STATE_CHANGE_READY_TO_PAUSED:
        if (_build_pipeline (base_bin) == FALSE)
            goto building_pipeline_error;
        break;
    case GST_STATE_CHANGE_PAUSED_TO_READY:
        break;
    case GST_STATE_CHANGE_READY_TO_NULL:
        _cleanup (base_bin);
        break;
    default:
        break;
    }

    return ret;
 building_pipeline_error:
    return GST_STATE_CHANGE_FAILURE;
}


/* Other stuff */
static void 
_kilikali_base_bin_set_volume (KilikaliBaseBin *base_bin, gdouble volume)
{
    if(base_bin->sink) {
        g_object_set( G_OBJECT (base_bin->sink),
                      "volume", (gint)(volume * MAX_VOLUME),
                      NULL);
    }
}

static void 
_kilikali_base_bin_set_mute (KilikaliBaseBin *base_bin, gboolean mute)
{
    if(base_bin->sink) {
        g_object_set( G_OBJECT (base_bin->sink),
                      "mute", mute,
                      NULL);
    }
}


/* Helpers */
static gboolean 
_element_remove (gpointer key, gpointer value, gpointer data)
{
    GstElement *element = GST_ELEMENT (value);
    KilikaliBaseBin *base_bin = KILIKALI_BASE_BIN (data);

    if (base_bin == NULL && element != NULL)
        return TRUE;

    if (element != NULL) {
        gst_element_set_state ( element, GST_STATE_NULL);
        gst_bin_remove (GST_BIN (base_bin), element);
        return TRUE;
    }

    return FALSE;
}

static void
_cleanup(KilikaliBaseBin *base_bin)
{
    g_debug (__FUNCTION__);

    g_return_if_fail (KILIKALI_IS_BASE_BIN (base_bin));

    g_hash_table_foreach_remove (base_bin->elements, _element_remove, (gpointer)base_bin);
    base_bin->elements = g_hash_table_new_full (g_str_hash, g_str_equal, NULL, 
                         (GDestroyNotify)gst_object_unref);
    base_bin->sink = NULL;
    base_bin->siddec = NULL;
}


static gboolean
_build_pipeline(KilikaliBaseBin *base_bin)
{
    GstElement *src = NULL;
    GstElement *conv = NULL;
    GstElement *typefind = NULL;
    base_bin->siddec = NULL;

    g_return_val_if_fail (KILIKALI_IS_BASE_BIN (base_bin), FALSE);

    g_debug (__FUNCTION__);

    if (base_bin->uri == NULL) {
        goto uri_error;
    }

    if (gst_uri_is_valid (base_bin->uri) == FALSE) {
        goto uri_error;
    }

    if (base_bin->iradioname != NULL) 
        g_free (base_bin->iradioname);
    base_bin->iradioname = NULL;

    conv = gst_element_factory_make ("audioconvert", "audioconvert");
    if (conv != NULL) {
        gst_bin_add (GST_BIN_CAST (base_bin), conv);
        gst_object_ref (conv);
        g_hash_table_insert (base_bin->elements, "audioconvert", conv);
    }

    src = gst_element_factory_make ("gnomevfssrc", "src");
    if (src == NULL) {
        goto uri_error;
    }

    g_object_set (G_OBJECT (src), "location", base_bin->uri, NULL);
    gst_bin_add (GST_BIN_CAST (base_bin), src);

    gst_object_ref (src);
    g_hash_table_insert (base_bin->elements, "src", src);

    if (g_strrstr (base_bin->uri, "http://")) {
        g_object_set (G_OBJECT (src), 
                      "blocksize", 64, /* 128 */
                      "iradio-mode", TRUE, 
                      NULL);

        gst_element_set_state (src, GST_STATE_PLAYING);
        _get_iradio_name (base_bin);
        gst_element_set_state (src, GST_STATE_NULL);
        g_object_set (G_OBJECT (src), 
                      "iradio-mode", FALSE, 
                      NULL);

    }
    
    typefind = gst_element_factory_make ("typefind", "typefinder");
    if (typefind == NULL) {
        goto typefind_error;
    }

    gst_bin_add (GST_BIN_CAST (base_bin), typefind);
    gst_object_ref (typefind);
    g_hash_table_insert (base_bin->elements, "typefind", typefind);

    gst_element_link_many (src, typefind, NULL); 

    g_signal_connect (typefind, "have-type", G_CALLBACK (_typefound),
                      (gpointer)base_bin);
    return TRUE;

 typefind_error:
    gst_object_unref (GST_OBJECT (src));
 uri_error:
    return FALSE;
}


static gboolean
_build_special_sid (KilikaliBaseBin *base_bin, GstPad *pad)
{
    GstPad *decpad = NULL;

    base_bin->siddec = gst_element_factory_make("siddec", "siddec");
    base_bin->sink = gst_element_factory_make("dsppcmsink", "sink");

    if (pad == NULL || base_bin->siddec == NULL || base_bin->sink == NULL) {
        goto error_sid;
    }
   
    decpad = gst_element_get_pad (base_bin->siddec, "sink");
    if (decpad == NULL)
        goto error_sid;

    gst_bin_add_many( GST_BIN (base_bin), base_bin->siddec, base_bin->sink, NULL );

    gst_object_ref (base_bin->siddec);
    gst_object_ref (base_bin->sink);

    g_hash_table_insert (base_bin->elements, "dec", base_bin->siddec);
    g_hash_table_insert (base_bin->elements, "sink", base_bin->sink);

    gst_pad_link (pad, decpad);
    gst_object_unref (decpad);

    if (!gst_element_link (base_bin->siddec, base_bin->sink))
        goto error_link_sid;
  
    g_object_set( G_OBJECT (base_bin->sink),
                  "preroll-queue-len", 0,
                  "sync", FALSE,
                  "max-lateness", -1,
                  "priority", 0,
                  NULL );

    /* FIXME: dirty hack */
    gst_element_set_state (GST_ELEMENT (base_bin), GST_STATE_PLAYING);
  
    return TRUE;

 error_sid:
    g_debug ("error sid");
    if (base_bin->sink != NULL)
        gst_object_unref (base_bin->sink);
    if (base_bin->siddec != NULL)
        gst_object_unref (base_bin->siddec);
    if (decpad != NULL)
        gst_object_unref (decpad);
    base_bin->siddec = NULL;
 error_link_sid:
    return FALSE;
}

static gboolean
_build_decodebin (KilikaliBaseBin *base_bin, GstPad *pad)
{
    GstPad *decpad = NULL;
    GstElement *dec = gst_element_factory_make("decodebin", "dec");
    g_debug (__FUNCTION__);
    base_bin->sink = gst_element_factory_make("dsppcmsink", "sink");

    if (pad == NULL || dec == NULL || base_bin->sink == NULL) {
        goto error_decodebin;
    }
   
    decpad = gst_element_get_pad (dec, "sink");
    if (decpad == NULL)
        goto error_decodebin;

    gst_bin_add_many( GST_BIN (base_bin), dec, base_bin->sink, NULL );


    gst_object_ref (dec);
    gst_object_ref (base_bin->sink);

    g_hash_table_insert (base_bin->elements, "dec", dec);
    g_hash_table_insert (base_bin->elements, "sink", base_bin->sink);

    gst_pad_link (pad, decpad);
    gst_object_unref (decpad);

 
    g_object_set( G_OBJECT (base_bin->sink),
                  "preroll-queue-len", 0,
                  "sync", FALSE,
                  "max-lateness", -1,
                  "priority", 0,
                  NULL );

    g_signal_connect (dec, "new-decoded-pad", 
                      G_CALLBACK (_new_decoded_pad), 
                      (gpointer)base_bin);

    /* Add signal to "unknown-type" */
    g_signal_connect (dec, "unknown-type", 
                      G_CALLBACK (_unknown_type), 
                      (gpointer)base_bin);

    /* FIXME: dirty hack */
    gst_element_set_state (GST_ELEMENT (base_bin), GST_STATE_PLAYING);
  
    return TRUE;

 error_decodebin:
    g_debug ("error decodebin");
    if (base_bin->sink != NULL)
        gst_object_unref (base_bin->sink);
    if (dec != NULL)
        gst_object_unref (dec);
    if (decpad != NULL)
        gst_object_unref (decpad);
    g_debug ("error decodebin failed");
    return FALSE;
}


/* Normal at least for internet tablets */
static gboolean
_normal_plug (KilikaliBaseBin *base_bin, GstPad *pad, const GstCaps *caps)
{
    GstElement *element = NULL;
    GstElement *queue = NULL;
    GstPad *queuepad = NULL;
    GstCaps *res;

    const GList *item;
    gchar *support = "dsp";

    for (item = base_bin->factories; item != NULL; item = item->next) {
        GstElementFactory *factory = GST_ELEMENT_FACTORY (item->data);
        const GList *pads;

        for (pads = gst_element_factory_get_static_pad_templates (factory);
             pads != NULL; pads = pads->next) {
            GstStaticPadTemplate *templ = pads->data;

            if (templ->direction != GST_PAD_SINK || 
                templ->presence != GST_PAD_ALWAYS) {
                continue;
            }

            res = gst_caps_intersect (caps,
                        gst_static_caps_get (&templ->static_caps));

            if (res && !gst_caps_is_empty (res)) {
                gst_caps_unref (res);
                element = gst_element_factory_create (factory, NULL);
        
                /* we wat here only dspsinks */
                if (g_strrstr (gst_object_get_name (GST_OBJECT (element)),
                               support) != NULL) {
                    
                    queue = gst_element_factory_make ("queue", "queue");
                    if (queue == NULL)
                        goto element_error;

                    queuepad = gst_element_get_pad (queue, "sink");
                    if (queuepad == NULL)
                        goto element_error;

                    g_debug("found from tablets special sinks: %s", 
                            gst_object_get_name (GST_OBJECT (element)));

                    g_object_set(G_OBJECT (element),
                                 "preroll-queue-len", 0,
                                 "sync", FALSE,
                                 "max-lateness", -1,
                                 "priority", 0,
                                 NULL );

                    g_object_set (G_OBJECT (queue), "min-threshold-bytes", 
                                  4096*2, NULL);

                    base_bin->sink = element;
                    gst_bin_add_many (GST_BIN (base_bin), queue, base_bin->sink, NULL);

                    gst_object_ref (queue);
                    gst_object_ref (element);

                    g_hash_table_insert (base_bin->elements, "queue", queue);
                    g_hash_table_insert (base_bin->elements, "sink", element);

                    gst_pad_link (pad, queuepad);
                    gst_object_unref (queuepad);
                    queuepad = NULL;

                    gst_element_link (queue, base_bin->sink);

                    gst_element_set_state (GST_ELEMENT (base_bin), GST_STATE_PLAYING);
                    return TRUE;
                }
                continue;
                
                gst_object_unref (element);
                element = NULL;
                return FALSE;
            }
            gst_caps_unref (res);
            break;
        }
    }
    return FALSE;
element_error:
    g_debug ("Error: element");
    if (queue != NULL)
        gst_object_unref (queue);
    if (queuepad != NULL)
        gst_object_unref (queuepad);
    if (element != NULL)
        gst_object_unref (element);

    return FALSE;
}

static gboolean
_pipeline_failed (KilikaliBaseBin *base_bin)
{
    g_debug (__FUNCTION__);
    _cleanup (base_bin);

    /* type-unknown signal */
    g_signal_emit (G_OBJECT (base_bin), 
                   kilikali_base_bin_signals[SIGNAL_TYPE_UNKNOWN], 0);

    return FALSE;
}

static gboolean
_type_sid (KilikaliBaseBin *base_bin)
{
    /* type-unknown signal */
    g_signal_emit (G_OBJECT (base_bin), 
                   kilikali_base_bin_signals[SIGNAL_TYPE_SID], 0);

    return FALSE;
}


static gboolean
_get_iradio_name (KilikaliBaseBin *base_bin)
{
    GstElement *src = NULL;
    gchar *tmpstring = NULL;
    
    g_debug (__FUNCTION__);

    src = gst_bin_get_by_name (GST_BIN (base_bin), "src");
    if (src == NULL) {
        g_debug ("No source element");   
        return FALSE;
    }
    g_object_set (G_OBJECT (src), 
                      "iradio-mode", TRUE, 
                      NULL);

    g_object_get (G_OBJECT (src), "iradio-name", &tmpstring, NULL);
    if (tmpstring != NULL) {
        g_debug ("name: %s", tmpstring);

        if (base_bin->iradioname != NULL)
            g_free (base_bin->iradioname);
        base_bin->iradioname = tmpstring;
        /* notify to radioname */

        g_object_notify (G_OBJECT (base_bin), "iradio-name");
    }
    return FALSE;
}




/* Callbacks */
static void 
_unknown_type(GstElement *dec, GstPad *pad, GstCaps *caps, gpointer data)
{
    g_debug(__FUNCTION__);
    g_assert (KILIKALI_IS_BASE_BIN (data));

    g_idle_add ((GSourceFunc)_pipeline_failed, data);
}

static gint
_compare_ranks (GstPluginFeature *f1, GstPluginFeature *f2)
{
    return gst_plugin_feature_get_rank (f2) - gst_plugin_feature_get_rank (f1);
}

static gboolean
_feature_filter (GstPluginFeature *feature, gpointer data)
{
    const gchar *klass;
    if (GST_IS_ELEMENT_FACTORY (feature) == FALSE)
        return FALSE;

    klass = gst_element_factory_get_klass (GST_ELEMENT_FACTORY (feature));
    if (g_strrstr (klass, "Sink") != NULL)
        return TRUE;

    return FALSE;
}

static void 
_typefound (GstElement *typefind, guint prob, GstCaps *caps, gpointer data)
{
    KilikaliBaseBin *base_bin = KILIKALI_BASE_BIN (data);
    GstPad *pad = NULL;
    gchar *type = NULL;

    base_bin->siddec = NULL;

    type = gst_caps_to_string (caps);
    g_debug ("%s - type %s prob %d%%\n", __FUNCTION__, type, prob);

    if (g_strrstr (type, "audio/x-sid") != NULL ) {

        /* we want special pipeline to sid audio */
        pad = gst_element_get_pad (typefind, "src");
        if (_build_special_sid (base_bin, pad) == FALSE) {
            gst_object_unref (GST_OBJECT (pad));
            _pipeline_failed (base_bin);
        } else {
            g_idle_add ((GSourceFunc)_type_sid, base_bin);
        }
        gst_object_unref(pad);

    } else {

        /* Try to plug normal way */
        pad = gst_element_get_pad (typefind, "src");
        gst_object_ref (pad);
        if (_normal_plug (base_bin, pad, caps) == FALSE) {
            if (_build_decodebin (base_bin, pad) == FALSE) 
            {
                gst_object_unref (GST_OBJECT (pad));
                _pipeline_failed (base_bin);
            }
        }
        gst_object_unref (GST_OBJECT (pad));
    }
    g_free (type);

}

static void
_new_decoded_pad (GstElement* dec, GstPad* pad, gboolean arg1, gpointer data)
{
    GstCaps *caps;
    GstStructure *str;
    GstElement *linkto = GST_ELEMENT (KILIKALI_BASE_BIN (data)->sink);
    GstPad *audiopad = gst_element_get_pad (linkto, "sink");
    GstPadLinkReturn ret;

    if (GST_PAD_IS_LINKED (audiopad)) {
        g_object_unref (audiopad);
        return;
    }
 
    caps = gst_pad_get_caps (pad);
    str = gst_caps_get_structure (caps, 0);
    if (!g_strrstr (gst_structure_get_name (str), "audio")) {
        gst_caps_unref (caps);
        gst_object_unref (audiopad);
        return;
    }
    gst_caps_unref (caps);

    ret = gst_pad_link (pad, audiopad);
    gst_object_unref (audiopad);

    /* Fallback to link with convert*/
    if (GST_PAD_LINK_FAILED (ret)) {
        g_debug ("Fallback to link with convert");
        linkto = gst_bin_get_by_name (GST_BIN (data), "audioconvert");
        if (linkto == NULL) {
            return;
        }
        gst_element_link (linkto, KILIKALI_BASE_BIN (data)->sink);

        audiopad = gst_element_get_pad (linkto, "sink");
        ret = gst_pad_link (pad, audiopad);
        gst_object_unref (audiopad);
    }
}


/* Emacs indentatation information
   Local Variables:
   indent-tabs-mode:nil
   tab-width:4
   c-set-offset:4
   c-basic-offset:4
   End: 
*/
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4
