/*
 * * Copyright (C) 2007 Joni Valtanen <jvaltane@kapsi.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */
#include <glib.h>
#include <gst/gst.h>

#include <string.h>

#include "kilikali-base-bin.h"

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

/* signals */
enum
{
  SIGNAL_TYPE_SID,
  SIGNAL_TYPE_VIDEO,
  SIGNAL_TYPE_UNKNOWN,
  LAST_SIGNAL
};

static void kilikali_base_bin_class_init (KilikaliBaseBinClass *klass);
static void kilikali_base_bin_init (KilikaliBaseBin *base_bin);
static void kilikali_base_bin_dispose (GObject *object);

static GstStateChangeReturn 
kilikali_base_bin_change_state (GstElement *element,
                                  GstStateChange transition);


/* Helpers */
static gboolean _element_remove (gpointer key, gpointer value, gpointer data);
static void _cleanup(KilikaliBaseBin *base_bin);
static gboolean _build_pipeline (KilikaliBaseBin *base_bin);

static gboolean _build_special_sid (KilikaliBaseBin *base_bin, GstPad *pad);

static gboolean _pipeline_failed (KilikaliBaseBin *base_bin);
static gboolean _get_iradio_name (KilikaliBaseBin *base_bin);

/* Callbacks */
static void _typefound(GstElement *typefind, guint probability, GstCaps *caps,
                       gpointer data);
static gboolean _type_sid (KilikaliBaseBin *base_bin);

static void _new_decoded_pad (GstElement* dec, GstPad* pad, gboolean arg1, gpointer data);
static gboolean _build_decodebin (KilikaliBaseBin *base_bin, GstPad *pad, gboolean has_video);
static void _unknown_type(GstElement *dec, GstPad *pad, GstCaps *caps, gpointer data);

#if 0
static void _empty_queue (GstElement *queue, KilikaliBaseBin *base_bin);
static void _full_queue (GstElement *queue, KilikaliBaseBin *base_bin);
static void _possible_queue_deadlock (GstElement *queue, KilikaliBaseBin *base_bin);
#endif

/* other stuff */
static void _kilikali_base_bin_set_volume (KilikaliBaseBin *base_bin, gdouble volume);
static void _kilikali_base_bin_set_mute (KilikaliBaseBin *base_bin, gboolean mute);

static guint kilikali_base_bin_signals[LAST_SIGNAL] = { 0 };

static GstElementClass *parent_class;


static void
kilikali_base_bin_class_init (KilikaliBaseBinClass *klass)
{
    GObjectClass *gobject_klass;
    GstElementClass *gstelement_klass;
    GstBinClass *gstbin_klass;

    gobject_klass = (GObjectClass *)klass;
    gstelement_klass = (GstElementClass *)klass;
    gstbin_klass = (GstBinClass *)klass;

    parent_class = g_type_class_peek_parent (klass);

    gobject_klass->dispose = kilikali_base_bin_dispose;

    gstelement_klass->change_state = kilikali_base_bin_change_state;

    kilikali_base_bin_signals[SIGNAL_TYPE_UNKNOWN] =
            g_signal_newv ("type-unknown",
                           G_TYPE_FROM_CLASS (klass),
                           G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | 
                           G_SIGNAL_NO_HOOKS,
                           NULL,
                           NULL,
                           NULL,
                           g_cclosure_marshal_VOID__VOID,
                           G_TYPE_NONE,
                           0,
                           NULL);

    kilikali_base_bin_signals[SIGNAL_TYPE_SID] =
            g_signal_newv ("sid-playing",
                           G_TYPE_FROM_CLASS (klass),
                           G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | 
                           G_SIGNAL_NO_HOOKS,
                           NULL,
                           NULL,
                           NULL,
                           g_cclosure_marshal_VOID__VOID,
                           G_TYPE_NONE,
                           0,
                           NULL);

    kilikali_base_bin_signals[SIGNAL_TYPE_VIDEO] =
            g_signal_newv ("type-video",
                           G_TYPE_FROM_CLASS (klass),
                           G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | 
                           G_SIGNAL_NO_HOOKS,
                           NULL,
                           NULL,
                           NULL,
                           g_cclosure_marshal_VOID__VOID,
                           G_TYPE_NONE,
                           0,
                           NULL);

}

static void
kilikali_base_bin_init (KilikaliBaseBin *base_bin)
{
    base_bin->volumeelem = NULL;
    base_bin->videosink = NULL;
    base_bin->colorspace = NULL;

    base_bin->elements = g_hash_table_new_full (g_str_hash, g_str_equal, NULL, 
                         (GDestroyNotify)gst_object_unref);
    base_bin->fakesink = NULL;
    base_bin->siddec = NULL;

    /* Set mute & volume */
    base_bin->set_volume =  _kilikali_base_bin_set_volume;
    base_bin->set_mute =  _kilikali_base_bin_set_mute;

}

static void
kilikali_base_bin_dispose (GObject *object)
{
    KilikaliBaseBin *base_bin = KILIKALI_BASE_BIN (object);
    g_debug (__FUNCTION__);

    if (base_bin->iradioname != NULL)
        g_free (base_bin->iradioname);
    g_hash_table_foreach_remove (base_bin->elements, _element_remove, (gpointer)base_bin);

    G_OBJECT_CLASS (parent_class)->dispose (object);

}

GType
kilikali_base_bin_get_type (void)
{
    static GType kilikali_base_bin_type = 0;

    if (!kilikali_base_bin_type) {
        static const GTypeInfo kilikali_base_bin_info = {
            sizeof (KilikaliBaseBinClass),
            NULL,
            NULL,
            (GClassInitFunc) kilikali_base_bin_class_init,
            NULL,
            NULL,
            sizeof (KilikaliBaseBin),
            0,
            (GInstanceInitFunc) kilikali_base_bin_init,
            NULL
    };

        kilikali_base_bin_type = 
            g_type_register_static (GST_TYPE_PIPELINE, "KilikaliBaseBin", 
                &kilikali_base_bin_info, 0);
    }

    return kilikali_base_bin_type;
}

static GstStateChangeReturn
kilikali_base_bin_change_state (GstElement *element, GstStateChange transition)
{
    GstStateChangeReturn ret;
    KilikaliBaseBin *base_bin = KILIKALI_BASE_BIN (element);

    g_debug (__FUNCTION__);

    switch (transition) {
    case GST_STATE_CHANGE_READY_TO_PAUSED:
        g_debug ("Ready To Paused");
        //  gst_pipeline_set_new_stream_time ( GST_PIPELINE (base_bin), 0);
        if (_build_pipeline (base_bin) == FALSE)
            goto building_pipeline_error;
        break;
    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
        g_debug ("Paused To Playing");
        // gst_pipeline_set_new_stream_time ( GST_PIPELINE (base_bin), GST_CLOCK_TIME_NONE);
        break;
    case GST_STATE_CHANGE_PAUSED_TO_READY:
        g_debug ("Paused To Ready");
        break;
    case GST_STATE_CHANGE_READY_TO_NULL:
        g_debug ("Ready To NULL");
        _cleanup (base_bin);
        break;
    default:
        break;
    }

    ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

    return ret;
 building_pipeline_error:
    return GST_STATE_CHANGE_FAILURE;
}

static gboolean
_type_sid (KilikaliBaseBin *base_bin)
{
    /* type-sid signal */
    g_signal_emit (G_OBJECT (base_bin), 
                   kilikali_base_bin_signals[SIGNAL_TYPE_SID], 0);

    return FALSE;
}

static gboolean
_type_video (KilikaliBaseBin *base_bin)
{
    /* type-video signal */
    g_signal_emit (G_OBJECT (base_bin), 
                   kilikali_base_bin_signals[SIGNAL_TYPE_VIDEO], 0);

    return FALSE;
}

static void
_kilikali_base_bin_set_volume (KilikaliBaseBin *base_bin, gdouble volume)
{
    if (base_bin->volumeelem) {
        g_object_set (G_OBJECT (base_bin->volumeelem), 
                      "volume", volume,
                      NULL);
    }
}

static void 
_kilikali_base_bin_set_mute (KilikaliBaseBin *base_bin, gboolean mute)
{
    if (base_bin->volumeelem) {
        g_object_set( G_OBJECT (base_bin->volumeelem),
                      "mute", mute,
                      NULL);
    }
}

/* Helpers */
static gboolean 
_element_remove (gpointer key, gpointer value, gpointer data)
{
    GstElement *element = GST_ELEMENT (value);
    KilikaliBaseBin *base_bin = KILIKALI_BASE_BIN (data);

    if (base_bin == NULL && element != NULL)
        return TRUE;

    if (element != NULL) {
        gst_element_set_state (element, GST_STATE_NULL);
        /* This should wait if element is async state */
        gst_element_get_state (element, NULL, NULL, GST_CLOCK_TIME_NONE); 
        gst_bin_remove (GST_BIN (base_bin), element);
        return TRUE;
    }

    return FALSE;
}

static void
_cleanup(KilikaliBaseBin *base_bin)
{
    g_debug (__FUNCTION__);

    g_return_if_fail (KILIKALI_IS_BASE_BIN (base_bin));


    /* FIXME: looks dirty but works quite well. No more wrong state problems  */
    if (base_bin->videosink != NULL && GST_IS_ELEMENT (base_bin->videosink)) {
        gst_element_set_state (base_bin->videosink, GST_STATE_NULL);
        gst_element_get_state (base_bin->videosink, NULL, NULL, 
                               GST_CLOCK_TIME_NONE); 
    }
    if (base_bin->vqueue != NULL && GST_IS_ELEMENT (base_bin->vqueue)) {
        gst_element_set_state (base_bin->vqueue, GST_STATE_NULL);
        gst_element_get_state (base_bin->vqueue, NULL, NULL, 
                               GST_CLOCK_TIME_NONE); 
    }
    if (base_bin->queue != NULL && GST_IS_ELEMENT (base_bin->queue)) {
        gst_element_set_state (base_bin->queue, GST_STATE_NULL);
        gst_element_get_state (base_bin->queue, NULL, NULL, 
                               GST_CLOCK_TIME_NONE); 
    }

    g_hash_table_foreach_remove (base_bin->elements, _element_remove, (gpointer)base_bin);
    base_bin->elements = g_hash_table_new_full (g_str_hash, g_str_equal, NULL, 
                         (GDestroyNotify)gst_object_unref);
    base_bin->sink = NULL;
    base_bin->volumeelem = NULL;
    base_bin->siddec = NULL;

    base_bin->videosink = NULL;
    base_bin->vqueue = NULL;
    base_bin->colorspace = NULL;

}


static gboolean
_build_pipeline(KilikaliBaseBin *base_bin)
{
    GstElement *src = NULL;
    GstElement *typefind = NULL;

    g_return_val_if_fail (KILIKALI_IS_BASE_BIN (base_bin), FALSE);

    g_debug (__FUNCTION__);

    if (base_bin->uri == NULL) {
        goto uri_error;
    }

    if (gst_uri_is_valid (base_bin->uri) == FALSE) {
        goto uri_error;
    }

    if (base_bin->iradioname != NULL) {
        g_free (base_bin->iradioname);
    }
    base_bin->iradioname = NULL;

    src = gst_element_make_from_uri (GST_URI_SRC, base_bin->uri, "src");
    if (src == NULL) {
        goto uri_error;
    }


    gst_bin_add (GST_BIN_CAST (base_bin), src);

    gst_object_ref (src);
    g_hash_table_insert (base_bin->elements, "src", src);

    if (g_strrstr (base_bin->uri, "http://")) {
        g_object_set (G_OBJECT (src), 
                      "iradio-mode", TRUE, 
                      NULL);

        gst_element_set_state (GST_ELEMENT_CAST (src), GST_STATE_PLAYING);
        _get_iradio_name (KILIKALI_BASE_BIN(src));
        gst_element_set_state (src, GST_STATE_NULL);
        g_object_set (G_OBJECT (src),
                      "iradio-mode", FALSE, 
                      NULL);

    }

    typefind = gst_element_factory_make ("typefind", "typefind");
    if (typefind == NULL) {
        goto typefind_error;
    }
    gst_object_ref (typefind);
    g_hash_table_insert (base_bin->elements, "typefind", typefind);

    gst_bin_add (GST_BIN_CAST (base_bin), typefind);

    gst_element_link_many (src, typefind, NULL); 

    g_signal_connect (typefind, "have-type", G_CALLBACK (_typefound),
                      (gpointer)base_bin);
    return TRUE;

 typefind_error:
    g_debug ("%s - typefind error", __FUNCTION__);
    gst_object_unref (GST_OBJECT (src));
 uri_error:
    g_debug ("%s - uri error", __FUNCTION__);
    return FALSE;
}


static gboolean
_build_special_sid (KilikaliBaseBin *base_bin, GstPad *pad)
{
    GstPad *decpad = NULL;
    GstElement *dec = gst_element_factory_make("siddec", "siddec");
    GstElement *audioconv = gst_element_factory_make ("audioconvert", "conv");
    base_bin->sink = gst_element_factory_make("alsasink", "sink");
    base_bin->volumeelem = gst_element_factory_make ("volume", "volume");
    base_bin->siddec = dec;

    if (pad == NULL || dec == NULL || base_bin->sink == NULL || audioconv == NULL || 
        base_bin->volumeelem == NULL) {
        goto error_sid;
    }
   
    decpad = gst_element_get_pad (dec, "sink");
    if (decpad == NULL)
        goto error_sid;

    gst_bin_add_many( GST_BIN (base_bin), dec, audioconv, base_bin->volumeelem, 
                      base_bin->sink, NULL );

    gst_object_ref (dec);
    gst_object_ref (audioconv);
    gst_object_ref (base_bin->volumeelem);
    gst_object_ref (base_bin->sink);

    g_hash_table_insert (base_bin->elements, "dec", dec);
    g_hash_table_insert (base_bin->elements, "conv", audioconv);
    g_hash_table_insert (base_bin->elements, "volume", base_bin->volumeelem);
    g_hash_table_insert (base_bin->elements, "sink", base_bin->sink);

    gst_pad_link (pad, decpad);
    gst_object_unref (decpad);

    gst_element_link_many( dec, audioconv, base_bin->volumeelem, base_bin->sink, NULL );

    return TRUE;

 error_sid:
    g_debug ("error sid");
    if (base_bin->sink != NULL)
        gst_object_unref (base_bin->sink);
    if (base_bin->volumeelem != NULL)
        gst_object_unref (base_bin->volumeelem);
    if (dec != NULL)
        gst_object_unref (dec);
    if (decpad != NULL)
        gst_object_unref (decpad);
    base_bin->siddec = NULL;
    return FALSE;
}

static gboolean
_build_decodebin (KilikaliBaseBin *base_bin, GstPad *pad, gboolean has_video)
{
    GstPad *decpad = NULL;
    GstElement *dec = gst_element_factory_make ("decodebin", "dec");
    GstElement *audioconv = gst_element_factory_make ("audioconvert", "conv");
    GstElement *audioscale = gst_element_factory_make ("audioresample", "scale");

    g_debug (__FUNCTION__);

    base_bin->queue = gst_element_factory_make ("queue", "queue2");
    base_bin->sink = gst_element_factory_make ("alsasink", "sink");
    base_bin->volumeelem = gst_element_factory_make ("volume", "volume");

    if (has_video) {

        /* setup video */
        base_bin->videosink = gst_element_factory_make ("xvimagesink", "vsink");
        if (base_bin->videosink == NULL) {
            /* Fallback to ximagesink */
            base_bin->videosink = gst_element_factory_make ("ximagesink", "vsink");
            if (base_bin->videosink == NULL) {
                goto error_decodebin;
            }
        }
        g_object_set(base_bin->videosink, "force-aspect-ratio", TRUE, NULL);
        gst_object_ref (base_bin->videosink);
        g_hash_table_insert (base_bin->elements, "vsink", base_bin->videosink);

        /* colorspace */
        base_bin->colorspace = gst_element_factory_make ("ffmpegcolorspace", "cspace");
        if (base_bin->colorspace == NULL) {
            g_debug ("colorspace == NULL");
        } else {
            gst_object_ref (base_bin->colorspace);
            g_hash_table_insert (base_bin->elements, "cspace", base_bin->colorspace);
        }

        /* queue to video */
        base_bin->vqueue = gst_element_factory_make ("queue", "vqueue");
        if (base_bin->vqueue == NULL) {
                goto error_decodebin;
        }
        gst_object_ref (base_bin->vqueue);
        g_hash_table_insert (base_bin->elements, "vqueue", base_bin->vqueue);

        gst_bin_add_many( GST_BIN (base_bin),
                          base_bin->videosink,
                          base_bin->colorspace,
                          base_bin->vqueue,
                          NULL );

        if (base_bin->colorspace != NULL) {
            gst_element_link_many (base_bin->vqueue, base_bin->colorspace, 
                                   base_bin->videosink, NULL);
        } else {
            gst_element_link_many (base_bin->vqueue, base_bin->videosink, NULL);
      
        }
    }


#if 0
    {
        g_signal_connect (G_OBJECT (base_bin->vqueue), "underrun",
                          G_CALLBACK (_empty_queue), base_bin);
        g_signal_connect (base_bin->vqueue, "overrun",
                          G_CALLBACK (_possible_queue_deadlock), base_bin);
        g_object_set( G_OBJECT (base_bin->vqueue),
                      "max-size-buffers", 0,
                      "max-size-bytes", 1024*1024*25,
                      "max-size-time", GST_SECOND*3,
                      "min-threshold-time", GST_SECOND,
                      NULL );
    }
#endif

    if (pad == NULL || dec == NULL || base_bin->sink == NULL || 
        base_bin->queue == NULL || audioconv == NULL || audioscale == NULL ||  
        base_bin->volumeelem == NULL ) {
        goto error_decodebin;
    }

    g_object_set( G_OBJECT (base_bin->sink),
                  "sync", FALSE,
                   NULL );

    decpad = gst_element_get_pad (dec, "sink");
    if (decpad == NULL)
        goto error_decodebin;

    gst_bin_add_many( GST_BIN (base_bin), dec, base_bin->queue, audioconv, 
                      audioscale, base_bin->volumeelem, base_bin->sink, NULL );

    gst_element_link_many( base_bin->queue, audioconv, audioscale, 
                           base_bin->volumeelem, base_bin->sink, NULL );

    gst_object_ref (dec);
    gst_object_ref (base_bin->queue);
    gst_object_ref (base_bin->sink);
    gst_object_ref (audioconv);
    gst_object_ref (audioscale);
    gst_object_ref (base_bin->volumeelem);

    g_hash_table_insert (base_bin->elements, "dec", dec);
    g_hash_table_insert (base_bin->elements, "queue", base_bin->queue);
    g_hash_table_insert (base_bin->elements, "sink", base_bin->sink);
    g_hash_table_insert (base_bin->elements, "volume", base_bin->volumeelem);
    g_hash_table_insert (base_bin->elements, "scale", audioscale);
    g_hash_table_insert (base_bin->elements, "conv", audioconv);

    gst_pad_link (pad, decpad);
    gst_object_unref (decpad);


#if 0
    {
        g_signal_connect (G_OBJECT (base_bin->queue), "underrun",
                          G_CALLBACK (_empty_queue), base_bin);
        g_signal_connect (base_bin->queue, "overrun",
                          G_CALLBACK (_possible_queue_deadlock), base_bin);
        g_object_set( G_OBJECT (base_bin->queue),
                      "max-size-buffers", 0,
                      "max-size-bytes", 1024*1024,
                      "max-size-time", GST_SECOND*3,
                      "min-threshold-time", GST_SECOND,
                      NULL );
    }
#endif


    g_signal_connect (dec, "new-decoded-pad", 
                      G_CALLBACK (_new_decoded_pad), 
                      (gpointer)base_bin);

    /* Add signal to "unknown-type" */
    g_signal_connect (dec, "unknown-type", 
                      G_CALLBACK (_unknown_type), 
                      (gpointer)base_bin);

    /* FIXME: dirty hack... or is it? */
    //    gst_element_set_state (GST_ELEMENT (base_bin), GST_STATE_PLAYING);

    return TRUE;

 error_decodebin:
    g_debug ("error decodebin");
    if (base_bin->sink != NULL)
        gst_object_unref (base_bin->sink);
    if (base_bin->queue != NULL)
        gst_object_unref (base_bin->queue);
    if (dec != NULL)
        gst_object_unref (dec);
    if (decpad != NULL)
        gst_object_unref (decpad);
    if (audioconv != NULL)
        gst_object_unref (audioconv);
    if (audioscale != NULL)
        gst_object_unref (audioscale);
    if (base_bin->volumeelem)
        gst_object_unref (base_bin->volumeelem);
    if (base_bin->videosink)
        gst_object_unref (base_bin->videosink);
    if (base_bin->vqueue)
        gst_object_unref (base_bin->vqueue);
    if (base_bin->colorspace)
        gst_object_unref (base_bin->colorspace);

    return FALSE;
}

static gboolean
_pipeline_failed (KilikaliBaseBin *base_bin)
{
    g_debug (__FUNCTION__);
    _cleanup (base_bin);

    /* type-unknown signal */
    g_signal_emit (G_OBJECT (base_bin), 
                   kilikali_base_bin_signals[SIGNAL_TYPE_UNKNOWN], 0);

    return FALSE;
}

static gboolean
_get_iradio_name (KilikaliBaseBin *base_bin)
{
    GstElement *src = NULL;
    gchar *tmpstring = NULL;
    
    g_debug (__FUNCTION__);

    src = gst_bin_get_by_name (GST_BIN (base_bin), "src");
    if (src == NULL) {
        g_debug ("No source element");   
        return FALSE;
    }
    g_object_set (G_OBJECT (src), 
                      "iradio-mode", TRUE, 
                      NULL);

    g_object_get (G_OBJECT (src), "iradio-name", &tmpstring, NULL);
    if (tmpstring != NULL) {
        g_debug ("name: %s", tmpstring);

        if (base_bin->iradioname != NULL)
            g_free (base_bin->iradioname);
        base_bin->iradioname = tmpstring;
        /* notify to radioname */

        g_object_notify (G_OBJECT (base_bin), "iradio-name");
    }
    return FALSE;
}




/* Callbacks */
static void 
_unknown_type(GstElement *dec, GstPad *pad, GstCaps *caps, gpointer data)
{
    g_debug(__FUNCTION__);
    g_assert (KILIKALI_IS_BASE_BIN (data));

    g_idle_add ((GSourceFunc)_pipeline_failed, data);
}

static void 
_typefound (GstElement *typefind, guint prob, GstCaps *caps, gpointer data)
{
    KilikaliBaseBin *base_bin = KILIKALI_BASE_BIN (data);
    GstPad *pad = NULL;
    gchar *type = NULL;

    type = gst_caps_to_string (caps);
    g_debug ("%s - type %s prob %d%%\n", __FUNCTION__, type, prob);

    /* FIXME: do some blacklist system to check unwanted types */
    if (g_strrstr (type, "text/plain") != NULL || 
        g_strrstr (type, "application/x-subtitle") != NULL ||
        prob < 90) {

        _pipeline_failed (base_bin);

    } else if (g_strrstr (type, "audio/x-sid") != NULL ) {

        /* we want special pipeline to sid audio 
           from sid we can not get duration anyway
         */
        pad = gst_element_get_pad (typefind, "src");
        if (_build_special_sid (base_bin, pad) == FALSE) {
            gst_object_unref (GST_OBJECT (pad));
            _pipeline_failed (base_bin);
        } else {
            g_idle_add ((GSourceFunc)_type_sid, base_bin);
        }
        gst_object_unref(pad);

    } else {
        GstElement *src = NULL;
        gboolean has_video = FALSE;
        src = gst_bin_get_by_name (GST_BIN (base_bin), "src");
        if (src == NULL) {
            g_debug ("%s : source element didn't found!!", __FUNCTION__);
            gst_object_unref (GST_OBJECT (pad));
            _pipeline_failed (base_bin);
            g_free (type);
            return;
        }
        gst_element_set_state (GST_ELEMENT (typefind), GST_STATE_NULL);
        gst_element_unlink (src, typefind);

        gst_element_set_state (GST_ELEMENT (src), GST_STATE_NULL);

        /* Try to plug normal way */
        pad = gst_element_get_pad (src, "src");
        
        if (g_str_has_prefix(type, "video"))
        {
            has_video = TRUE;
        }
        
        gst_object_ref (pad);
        if (_build_decodebin (base_bin, pad, has_video) == FALSE) 
        {
            gst_object_unref (GST_OBJECT (pad));
            _pipeline_failed (base_bin);
        }
        gst_object_unref (GST_OBJECT (pad));
    }
    g_free (type);

}

static void
_new_decoded_pad (GstElement *dec, GstPad* pad, gboolean arg1, gpointer data)
{
    KilikaliBaseBin *base_bin = NULL;
    GstCaps *caps = NULL;
    GstStructure *str = NULL;
    GstElement *linkto = NULL; 
    GstPad *mediapad = NULL;

    g_assert (KILIKALI_IS_BASE_BIN (data));
    
    caps = gst_pad_get_caps (pad);
    str = gst_caps_get_structure (caps, 0);

    base_bin = KILIKALI_BASE_BIN (data);

    if (g_strrstr (gst_structure_get_name (str), "video")) {
        /* Have videopad */
        linkto = GST_ELEMENT (base_bin->vqueue);

        /* FIXME: Not sure is this working while playign */
        /*Set audio sync to TRUE if we have video */
        g_object_set( G_OBJECT (base_bin->sink),
                      "sync", TRUE, NULL );
        _type_video(base_bin);
    } else if (g_strrstr (gst_structure_get_name (str), "audio")) {
        /* Have audiopad */
        linkto = GST_ELEMENT (base_bin->queue);
    }

    if (linkto == NULL) {
        goto new_decpad_error;
    }

    mediapad = gst_element_get_pad (linkto, "sink");
    if (mediapad == NULL || GST_PAD_IS_LINKED (mediapad)) {
         goto new_decpad_error;
    }

    if(gst_pad_link (pad, mediapad)) {
         g_debug ("Could not link pads");
    }
    gst_object_unref (mediapad);

 new_decpad_error:
    if (caps != NULL)
        gst_caps_unref (caps);

}

#if 0
static void
_empty_queue (GstElement *queue, KilikaliBaseBin *base_bin)
{
    g_debug ("full queue");

    base_bin->pushing_id = 
        g_signal_connect (queue, "pushing", G_CALLBACK (_full_queue), base_bin);
    g_object_set (queue, "min-threshold-time", GST_SECOND, NULL);

}

static void
_full_queue (GstElement *queue, KilikaliBaseBin *base_bin)
{
    gpointer data;

    g_debug (__FUNCTION__);

    g_signal_handler_disconnect (queue, base_bin->pushing_id);

    data = g_object_get_data (G_OBJECT (queue), "eos");
    if (data) {
        g_object_set (queue, "min-threshold-time", 0, NULL);
    } else {
        g_object_set (queue, "min-threshold-time", GST_SECOND, NULL);
    }
}

static void
_possible_queue_deadlock (GstElement *queue, KilikaliBaseBin *base_bin)
{
    g_debug (__FUNCTION__);

}
#endif

/* Emacs indentatation information
   Local Variables:
   indent-tabs-mode:nil
   tab-width:4
   c-set-offset:4
   c-basic-offset:4
   End: 
*/
// vim: filetype=c:expandtab:shiftwidth=4:tabstop=4:softtabstop=4
