/*
 * * Copyright (C) 2007 Joni Valtanen <jvaltane@kapsi.fi>
 *                    Tuomas Kulve <tuomas@kulve.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "kilikali-audio-gst.h"
#include "kilikali-audio.h"

#include <unistd.h>

#include <glib/gtypes.h>
#include <glib/gstring.h>
#include <glib/gprintf.h>           /* g_printf */

#include <gst/gst.h>
#include <gst/interfaces/xoverlay.h>
#include <string.h>

#define NETWORK_BUFFERBYTES 128


static void
kilikali_audio_gst_class_init (KilikaliAudioGstClass *klass);


/* Virtual functions */
static gboolean 
kilikali_audio_gst_init( KilikaliAudioGst *audiogst );
static void 
kilikali_audio_gst_play (KilikaliAudioGst *self);
static void 
kilikali_audio_gst_pause (KilikaliAudioGst *self);
static void 
kilikali_audio_gst_stop (KilikaliAudioGst *self);
static void 
kilikali_audio_gst_absolute (KilikaliAudioGst *self, gulong time);

static void
set_volume (KilikaliAudioGst *self);
static void
set_mute (KilikaliAudioGst *self);

static gboolean 
kilikali_audio_gst_is_supported (KilikaliAudio *self, const gchar *uri);

static GstBusSyncReply 
kilikali_audio_gst_play_bus_callback (GstBus *bus, GstMessage *msg, 
                                      gpointer data);

#ifndef USE_PLAYBIN
static void unknown_type (GObject *obj, gpointer data);
static void sid_playing (KilikaliAudioGst *obj, gpointer data);

static void avbin_notify_iradio (GObject *obj, gpointer data, gpointer data2);

static gboolean sid_playing_emit(KilikaliAudioGst *self);
static gboolean unknown_type_emit (KilikaliAudioGst *self);
#else
static void init_playbin (KilikaliAudioGst *self);
#endif

static gboolean video_playing_emit(KilikaliAudioGst *self);
static gboolean emit_eos (KilikaliAudioGst *self);
static gboolean get_position (KilikaliAudioGst *self);
static gboolean get_length (KilikaliAudioGst *self);
static gboolean tag_found_emit (KilikaliAudioGst *self);

static void remove_timeouts (KilikaliAudioGst *self);
static void free_tags (KilikaliAudioGst *self);

enum KilikaliAudioGstMode
{
    KILIKALI_AUDIO_GST_MODE_NONE = 0,
    KILIKALI_AUDIO_GST_MODE_FILE,
    KILIKALI_AUDIO_GST_MODE_IRADIO
};

struct _KilikaliAudioGstPrivate
{
    /* FIXME: Does not look good */
    enum KilikaliAudioGstMode mode; 
    enum _kilikali_audio_states state;

    guint sid_timeout_id;
    guint position_timeout_id;
    guint length_timeout_id;

    guint timeout;

    gchar *device;
    gchar *sink;
    gchar *uri;
    gchar *mime_type;
    gdouble volume;
    gdouble mute_backup;
    gboolean mute;
    gchar *iradioname;

    guint position;
    guint length;
    
    KilikaliAudioTags tags;

    /* SID stuff */
    gint sid_time; /* time in ms to play tune */

    gint sid_tune;
    gint sid_clock;
    gint sid_memory;
    gboolean sid_filter;
    gboolean sid_measured_volume;
    gboolean sid_mos8580;
    gboolean sid_force_speed;

    /* video stuff */
    gulong xid;

};



static GObjectClass *parent_class = NULL;



static void
kilikali_audio_gst_interface_init( gpointer   g_iface,
								   gpointer   iface_data)
{
    KilikaliAudioInterface *iface = (KilikaliAudioInterface *)g_iface;

    iface->init = (gboolean (*) (KilikaliAudio *self))kilikali_audio_gst_init;
    iface->play = (void (*) (KilikaliAudio *self))kilikali_audio_gst_play;
    iface->stop = (void (*) (KilikaliAudio *self))kilikali_audio_gst_stop;
    iface->pause = (void (*) (KilikaliAudio *self))kilikali_audio_gst_pause;

    iface->absolute = (void (*) (KilikaliAudio *self, 
                                 gulong time))kilikali_audio_gst_absolute;

    iface->is_supported = 
        (gboolean (*) (KilikaliAudio *self, const gchar *uri)) 
            kilikali_audio_gst_is_supported;

    /* do something at the start from interface */
}



static void
kilikali_audio_gst_instance_init ( GTypeInstance *instance,
								   gpointer g_class )
{
    KilikaliAudioGst *self = (KilikaliAudioGst *)instance;

    /* create and init priv "area" */
    self->priv = g_new (KilikaliAudioGstPrivate, 1);

    self->priv->sid_timeout_id = 0;
    self->priv->position_timeout_id = 0;
    self->priv->length_timeout_id = 0;

    self->priv->timeout = 0;

    self->priv->position = 0;
    self->priv->length = 0;

    self->priv->mime_type = NULL;
    self->priv->device = NULL;
    self->priv->sink = NULL;
    self->priv->uri = NULL;
    self->priv->state = AUDIO_STATE_VOID;
    self->priv->volume = 0.9;
    self->priv->mute_backup = 0.0;
    self->priv->mute = FALSE;
    self->priv->mode = KILIKALI_AUDIO_GST_MODE_NONE;
    self->priv->iradioname = NULL;

    /* Default sid values */
    self->priv->sid_time = 180000; /* Default three minutes */
 
    self->priv->sid_tune = 0; 
    self->priv->sid_clock = 1;
    self->priv->sid_memory = 32;
    self->priv->sid_filter = FALSE;
    self->priv->sid_measured_volume = FALSE;
    self->priv->sid_mos8580 = FALSE;
    self->priv->sid_force_speed = FALSE;

    self->priv->xid = 0;

    self->priv->tags.artist = NULL;
    self->priv->tags.album = NULL;
    self->priv->tags.title = NULL;

}



static void
kilikali_audio_gst_finalize (GObject *obj)
{
    KilikaliAudioGst *self = (KilikaliAudioGst *)obj;

    kilikali_audio_gst_stop (self);
    if (self->avbin) {
        gst_object_unref (GST_OBJECT (self->avbin));
    }

    if (self->priv->device != NULL)
        g_free (self->priv->device);
    if (self->priv->sink != NULL)
        g_free (self->priv->sink);
    if (self->priv->mime_type != NULL)
        g_free (self->priv->mime_type);
    if (self->priv->uri != NULL)
        g_free (self->priv->uri);
    if (self->priv->iradioname != NULL)
        g_free (self->priv->iradioname);
    free_tags (self);
    g_free (self->priv);
}



static void
kilikali_audio_gst_set_property ( GObject * object, 
                                  guint prop_id,
                                  const GValue * value, 
                                  GParamSpec * pspec)
{
    KilikaliAudioGst *audiogst;
    /*GObject *obj;*/

    /* it's not null if we got it, but it might not be ours */
    g_return_if_fail (KILIKALI_IS_AUDIO_GST(object));

    audiogst = KILIKALI_AUDIO_GST(object);

    switch (prop_id) {
    case  KILIKALI_AUDIO_SINK:
        g_free(audiogst->priv->sink);
        audiogst->priv->sink = g_value_dup_string(value);
        break;
    case  KILIKALI_AUDIO_DEVICE:
        g_free(audiogst->priv->device);
        audiogst->priv->device = g_value_dup_string(value);
        break;
    case  KILIKALI_AUDIO_MIME_TYPE:
        g_free(audiogst->priv->mime_type);
        audiogst->priv->mime_type = g_value_dup_string(value);
        break;
    case  KILIKALI_AUDIO_URI:
        g_free(audiogst->priv->uri);
        audiogst->priv->uri = g_value_dup_string(value);
#ifdef USE_PLAYBIN
        if (audiogst->avbin != NULL) {
            gst_object_unref (GST_OBJECT (audiogst->avbin));
            audiogst->avbin = NULL;
        }
#endif
        free_tags (audiogst);
        break;
#if 0
    case  KILIKALI_AUDIO_STATE:
        audiogst->priv->state = g_value_get_uint (value);
        break;
#endif
    case  KILIKALI_AUDIO_VOLUME:
        audiogst->priv->volume = g_value_get_double (value);
        set_volume (audiogst);
        break;
    case  KILIKALI_AUDIO_MUTE:
        audiogst->priv->mute = g_value_get_boolean (value);
        set_mute (audiogst);
        break;
    case KILIKALI_AUDIO_SID_TIME:
        audiogst->priv->sid_time = g_value_get_uint (value);
        break;
    case KILIKALI_AUDIO_SID_TUNE:
        audiogst->priv->sid_tune = g_value_get_int (value);
        if (audiogst->avbin != NULL) {
            g_object_set ( G_OBJECT (audiogst->avbin),
                           "sid-tune", audiogst->priv->sid_tune,
                           NULL);
        }
        break;
    case KILIKALI_AUDIO_SID_CLOCK:
        audiogst->priv->sid_clock = g_value_get_int (value);
        if (audiogst->avbin != NULL) {
            g_object_set ( G_OBJECT (audiogst->avbin),
                           "sid-clock", audiogst->priv->sid_clock,
                           NULL);
        }
        break;
    case KILIKALI_AUDIO_SID_MEMORY:
        audiogst->priv->sid_memory = g_value_get_int (value);
        if (audiogst->avbin != NULL) {
            g_object_set ( G_OBJECT (audiogst->avbin),
                           "sid-memory", audiogst->priv->sid_memory,
                           NULL);
        }
        break;
    case KILIKALI_AUDIO_SID_FILTER:
        audiogst->priv->sid_filter = g_value_get_boolean (value);
        if (audiogst->avbin != NULL) {
            g_object_set ( G_OBJECT (audiogst->avbin),
                           "sid-filter", audiogst->priv->sid_filter,
                           NULL);
        }
        break;
    case KILIKALI_AUDIO_SID_MEASURED_VOLUME:
        audiogst->priv->sid_measured_volume = g_value_get_boolean (value);
        if (audiogst->avbin != NULL) {
            g_object_set ( G_OBJECT (audiogst->avbin),
                           "sid-measured-volume", audiogst->priv->sid_measured_volume,
                           NULL);
        }
        break;
    case KILIKALI_AUDIO_SID_MOS8580:
        audiogst->priv->sid_mos8580 = g_value_get_boolean (value);
        if (audiogst->avbin != NULL) {
            g_object_set ( G_OBJECT (audiogst->avbin),
                           "sid-mos8580", audiogst->priv->sid_mos8580,
                           NULL);
        }
        break;
    case KILIKALI_AUDIO_SID_FORCE_SPEED:
        audiogst->priv->sid_force_speed = g_value_get_boolean (value);
        if (audiogst->avbin != NULL) {
            g_object_set ( G_OBJECT (audiogst->avbin),
                           "sid-force-speed", audiogst->priv->sid_force_speed,
                           NULL);
        }
        break;
    case KILIKALI_AUDIO_XID:
        audiogst->priv->xid = g_value_get_ulong (value);
#ifdef DEBUG
        g_debug("XID set to %li", audiogst->priv->xid);
#endif
        break;
    case KILIKALI_AUDIO_TAGS:
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}



static void
kilikali_audio_gst_get_property (GObject * object, 
                                 guint prop_id,
                                 GValue * value,
                                 GParamSpec * pspec)
{
    KilikaliAudioGst *audiogst;
    /*GObject *obj;*/

    /* it's not null if we got it, but it might not be ours */
    g_return_if_fail (KILIKALI_IS_AUDIO_GST(object));

    audiogst = KILIKALI_AUDIO_GST(object);

    switch (prop_id) {
    case  KILIKALI_AUDIO_SINK:
        g_value_set_string (value, audiogst->priv->sink);
        break;
    case  KILIKALI_AUDIO_DEVICE:
        g_value_set_string (value, audiogst->priv->device);
        break;
    case  KILIKALI_AUDIO_MIME_TYPE:
        g_value_set_string (value, audiogst->priv->mime_type);
        break;
    case  KILIKALI_AUDIO_URI:
        g_value_set_string (value, audiogst->priv->uri);
        break;
    case  KILIKALI_AUDIO_STATE:
        g_value_set_uint (value, audiogst->priv->state );
        break;
    case  KILIKALI_AUDIO_VOLUME:
        g_value_set_double (value, audiogst->priv->volume );
        break;
    case  KILIKALI_AUDIO_MUTE:
        g_value_set_boolean (value, audiogst->priv->mute );
        break;
    case  KILIKALI_AUDIO_IRADIONAME:
        g_value_set_string (value, audiogst->priv->iradioname);
        break;
    case  KILIKALI_AUDIO_POSITION:
        g_value_set_uint (value, audiogst->priv->position);
        break;
    case  KILIKALI_AUDIO_LENGTH:
        g_value_set_uint (value, audiogst->priv->length);
        break;
    case  KILIKALI_AUDIO_SID_TIME:
        g_value_set_uint (value, audiogst->priv->sid_time);
        break;
    case  KILIKALI_AUDIO_SID_TUNE:
        g_value_set_int (value, audiogst->priv->sid_tune);
        break;
    case  KILIKALI_AUDIO_SID_CLOCK:
        g_value_set_int (value, audiogst->priv->sid_clock);
        break;
    case  KILIKALI_AUDIO_SID_MEMORY:
        g_value_set_int (value, audiogst->priv->sid_memory);
        break;
    case  KILIKALI_AUDIO_SID_FILTER:
        g_value_set_boolean (value, audiogst->priv->sid_filter);
        break;
    case  KILIKALI_AUDIO_SID_MEASURED_VOLUME:
        g_value_set_boolean (value, audiogst->priv->sid_measured_volume);
        break;
    case  KILIKALI_AUDIO_SID_MOS8580:
        g_value_set_boolean (value, audiogst->priv->sid_mos8580);
        break;
    case  KILIKALI_AUDIO_SID_FORCE_SPEED:
        g_value_set_boolean (value, audiogst->priv->sid_force_speed);
        break;
    case  KILIKALI_AUDIO_XID:
        g_value_set_ulong (value, audiogst->priv->xid);
        break;
    case KILIKALI_AUDIO_TAGS:
        g_value_set_pointer (value, &audiogst->priv->tags);

        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}



static void
kilikali_audio_gst_class_init(KilikaliAudioGstClass * klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;

    parent_class = g_type_class_ref (G_TYPE_OBJECT);

    gobject_class->finalize = kilikali_audio_gst_finalize;
    gobject_class->set_property = kilikali_audio_gst_set_property;
    gobject_class->get_property = kilikali_audio_gst_get_property;

    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_VOLUME, 
                                      "volume");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_MUTE, 
                                      "mute");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_STATE, 
                                      "state");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_DEVICE,
                                      "device");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_MIME_TYPE,
                                      "mime-type");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_SINK,
                                      "sink");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_URI,
                                      "uri");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_IRADIONAME,
                                      "iradio-name");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_SID_TIME,
                                      "sid-time");

    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_POSITION,
                                      "stream-position");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_LENGTH,
                                      "stream-length");

    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_SID_TUNE,
                                      "sid-tune");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_SID_CLOCK,
                                      "sid-clock");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_SID_MEMORY,
                                      "sid-memory");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_SID_FILTER,
                                      "sid-filter");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_SID_MEASURED_VOLUME,
                                      "sid-measured-volume");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_SID_MOS8580,
                                      "sid-mos8580");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_SID_FORCE_SPEED,
                                      "sid-force-speed");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_XID,
                                      "xid");
    g_object_class_override_property (gobject_class,KILIKALI_AUDIO_TAGS,
                                      "tags");



}



GType kilikali_audio_gst_get_type (void)
{
    static GType type = 0;
    if (type == 0) {
        static const GTypeInfo info = {
            sizeof (KilikaliAudioGstClass),
            NULL,   /* base_init */
            NULL,   /* base_finalize */
            (GClassInitFunc)kilikali_audio_gst_class_init,   /*   class_init */
            NULL,   /* class_finalize */
            NULL,   /* class_data */
            sizeof (KilikaliAudioGst),
            0,      /* n_preallocs */
            kilikali_audio_gst_instance_init    /* instance_init */
        };

        static const GInterfaceInfo kilikali_audio_info = {
            (GInterfaceInitFunc) kilikali_audio_gst_interface_init, 
            NULL,               /* interface_finalize */
            NULL                /* interface_data */
        };

        type = g_type_register_static (G_TYPE_OBJECT,
                                       "KilikaliAudioGstType",
                                       &info, 0);
        g_type_add_interface_static (type,
                                     KILIKALI_TYPE_AUDIO,
                                     &kilikali_audio_info);  }
    return type;
}



KilikaliAudioGst *
kilikali_audio_gst_new ( void )
{
    KilikaliAudioGst *audiogst = g_object_new (KILIKALI_TYPE_AUDIO_GST, NULL);
    return audiogst;
}


static void
free_tags (KilikaliAudioGst *self)
{
    if (self->priv->tags.artist != NULL)
        g_free (self->priv->tags.artist);
    if (self->priv->tags.album != NULL)
        g_free (self->priv->tags.album);
    if (self->priv->tags.title != NULL)
        g_free (self->priv->tags.title);
    self->priv->tags.artist = NULL;
    self->priv->tags.album = NULL;
    self->priv->tags.title = NULL;

}

#ifdef USE_PLAYBIN
static void
init_playbin (KilikaliAudioGst *self)
{
    GstElement *videosink = NULL;
    self->avbin = gst_element_factory_make ("playbin", "kbin");
    videosink = gst_element_factory_make ("xvimagesink", "vsink");
    if (videosink != NULL) {
        g_object_set (videosink, "force-aspect-ratio", TRUE, NULL);
        g_object_set (self->avbin, "video-sink", videosink, NULL);
    }
}
#endif

/* Init */
static gboolean 
kilikali_audio_gst_init (KilikaliAudioGst *self)
{
    GstBus *bus = NULL;
#ifdef DEBUG
    g_debug("%s", __FUNCTION__);
#endif

#ifdef USE_PLAYBIN
    init_playbin (self);
#else
    self->avbin = gst_element_factory_make ("kilikalibin", "kbin");
#endif
    if (self->avbin == NULL) {
        g_error ("Could not create avbin element");
        return FALSE;
    }

#ifndef USE_PLAYBIN
    g_signal_connect (G_OBJECT (self->avbin), "type-unknown", 
                      G_CALLBACK (unknown_type), (gpointer)self);
    g_signal_connect (G_OBJECT (self->avbin), "sid-playing", 
                      G_CALLBACK (sid_playing), (gpointer)self);
#endif

    bus = gst_pipeline_get_bus (GST_PIPELINE (self->avbin));
    if (bus == NULL) {
        g_error ("Could not get bus from avbin element");
        gst_object_unref (self->avbin);
        self->avbin = NULL;
        return FALSE;
    }

    gst_bus_set_sync_handler (bus, kilikali_audio_gst_play_bus_callback, 
                              (gpointer)self);
    gst_object_unref (bus);


#ifndef USE_PLAYBIN
    g_signal_connect ( self->avbin, "notify::iradio-name", 
                             (GCallback)avbin_notify_iradio, 
                              (gpointer)self);
#endif
    return TRUE;
}

static gboolean
set_sound (gpointer data) 
{
    KilikaliAudioGst *self = NULL;
    
    if (data == NULL)
        return FALSE;

    g_return_val_if_fail (KILIKALI_IS_AUDIO_GST (data), FALSE);

    self = KILIKALI_AUDIO_GST (data);

#ifndef USE_PLAYBIN
    if(self->avbin != NULL) {
        g_object_set ( G_OBJECT (self->avbin),
                       "sid-tune", self->priv->sid_tune,
                       "sid-clock", self->priv->sid_clock,
                       "sid-memory", self->priv->sid_memory,
                       "sid-filter", self->priv->sid_filter,
                       "sid-measured-volume", self->priv->sid_measured_volume,
                       "sid-mos8580", self->priv->sid_mos8580,
                       "sid-force-speed", self->priv->sid_force_speed,
                       NULL);
    }

#endif
    set_volume (self);
    set_mute (self);

    return FALSE;
}


/* Play */
static void 
kilikali_audio_gst_play (KilikaliAudioGst *self)
{
    g_assert (KILIKALI_IS_AUDIO_GST (self));

    if (self->avbin == NULL) {
        (void)kilikali_audio_gst_init (self);

        if (self->priv->iradioname != NULL) {
            g_free (self->priv->iradioname);
            self->priv->iradioname = NULL;
        }

        self->priv->length = 0;

        remove_timeouts (self);
        self->priv->timeout = 0;

        g_object_set (G_OBJECT (self->avbin), "uri", self->priv->uri, NULL);
        g_idle_add (set_sound, (gpointer)self);
    }
    gst_element_set_state(self->avbin, GST_STATE_PLAYING);
}



/* Pause */
static void 
kilikali_audio_gst_pause (KilikaliAudioGst *self )
{
    if (self->avbin == NULL) {
        return;
    }

    gst_element_set_state(self->avbin, GST_STATE_PAUSED);

}



/* Stop */
static void 
kilikali_audio_gst_stop (KilikaliAudioGst *self )
{

    if (self->avbin == NULL) {
        return;
    }

    gst_element_set_state(self->avbin, GST_STATE_NULL);

/*    KILIKALI_AUDIO_GET_IFACE(self)->state = AUDIO_STATE_STOPPED; */
    if (self->priv->iradioname != NULL) {
        g_free (self->priv->iradioname);
        self->priv->iradioname = NULL;
    }

    remove_timeouts (self);

    /* If this is the problem add, state checking before */
    gst_object_unref (self->avbin);
    self->avbin = NULL;
    self->priv->position = 0;
}



/* Absolute seek of the stream */
static void 
kilikali_audio_gst_absolute (KilikaliAudioGst *self, gulong time )
{

    if (self->avbin == NULL) {
        return;
    }

    if (self->priv->length == 0) {
        g_debug ("%s - Could not seek", __FUNCTION__);
        return;
    }

    if(time > self->priv->length )
        time = self->priv->length-10;
  
    if (!gst_element_seek( self->avbin, 1.0, 
                           GST_FORMAT_TIME, 
                           GST_SEEK_FLAG_FLUSH, 
                           GST_SEEK_TYPE_SET, 
                           time*GST_MSECOND, 
                           GST_SEEK_TYPE_NONE, 
                           GST_CLOCK_TIME_NONE ) ) {
        g_debug ( "%s : Seek failed\n", __FUNCTION__ );
    } else {
        /* Accept whatever position update comes next */
        self->priv->position = 0;
    }
    gst_element_get_state(self->avbin, NULL, NULL, 100*GST_MSECOND);

#ifdef DEBUG
    g_debug ("%s : Seeked to %lu\n",__FUNCTION__, time );
#endif
}


/* Get stream position */
static gboolean
get_position (KilikaliAudioGst *self)
{
  GstFormat format = GST_FORMAT_TIME;
  gint64 position = 0;

  if (!gst_element_query_position (self->avbin, &format, &position)) {
      return TRUE;
  }

  if (self->priv->position < (guint)(position/GST_MSECOND)) {
      self->priv->position = (guint)(position/GST_MSECOND);
      if (self->priv->position > self->priv->length)
          self->priv->position = self->priv->length;

      g_signal_emit (self, 
                     KILIKALI_AUDIO_GET_IFACE(self)->position_signal_id,
                     0);
  }

  return TRUE;
}

/* Get stream duration */
static gboolean
get_length (KilikaliAudioGst *self)
{
    GstFormat fmt = GST_FORMAT_TIME;
    gint64 value = -1;

    g_assert (KILIKALI_IS_AUDIO_GST (self));

    /* This shouldn't never true, but depends on place wheree this is called */
    if (self->avbin == NULL) {
        g_debug("%s - avbin == NULL", __FUNCTION__);
        self->priv->length_timeout_id = 0;
        return FALSE;
    }

    /* Seems that this is working better if called after position
       Bad luck or something?
     */
    if (self->priv->position_timeout_id == 0) {
          /* Let's set position every one sec */
           self->priv->position_timeout_id = g_timeout_add (1000, 
                                                   (GSourceFunc) get_position,
                                                     self);
    }

    /* we drop timeout when we got all we want */
    if( self->priv->length > 0 || self->priv->iradioname != NULL || 
        self->priv->sid_timeout_id > 0 ||  self->priv->timeout > 10) {
        self->priv->timeout = 0;
        self->priv->length_timeout_id = 0;
        return FALSE;
    }

    if (gst_element_query_duration (GST_ELEMENT (self->avbin), &fmt, 
                                    &value)) {
           if(value != -1 && fmt == GST_FORMAT_TIME) {
               self->priv->length = (guint)(value/GST_MSECOND);
               g_signal_emit (self, 
                              KILIKALI_AUDIO_GET_IFACE(self)->length_signal_id,
                              0);
           }
    }

    self->priv->timeout++;
    return TRUE;

}
#ifndef USE_PLAYBIN
static gboolean
sid_playing_emit(KilikaliAudioGst *self)
{
    g_assert (KILIKALI_IS_AUDIO_GST (self));

    g_signal_emit ( self, 
                    KILIKALI_AUDIO_GET_IFACE(self)->sid_playing_signal_id,
                    0 );
    return FALSE;
}
#endif

static gboolean
video_playing_emit(KilikaliAudioGst *self)
{
    g_assert (KILIKALI_IS_AUDIO_GST (self));

    g_signal_emit ( self, 
                    KILIKALI_AUDIO_GET_IFACE(self)->video_playing_signal_id,
                    0 );
    return FALSE;
}

static void
remove_timeouts (KilikaliAudioGst *self)
{
    /* remove time outs */
    if (self->priv->sid_timeout_id > 0) {
        if (g_source_remove (self->priv->sid_timeout_id) == TRUE) {
            self->priv->sid_timeout_id = 0;
        }
    }
    if (self->priv->position_timeout_id > 0) {
        if (g_source_remove (self->priv->position_timeout_id) == TRUE) {
            self->priv->position_timeout_id = 0;
        }
    }
    if (self->priv->length_timeout_id > 0) {
        if (g_source_remove (self->priv->length_timeout_id) == TRUE) {
            self->priv->length_timeout_id = 0;
        }
    }
}

static gboolean
emit_eos (KilikaliAudioGst *self)
{ 
    /* Notify outsiders we are out of data */
    if (self->priv->iradioname == NULL) {
        g_signal_emit ( self, 
                        KILIKALI_AUDIO_GET_IFACE(self)->eos_signal_id,
                        0 );
    } else {
        kilikali_audio_gst_stop (self);
        kilikali_audio_gst_play (self);
    }


    return FALSE;
}

/* GStreamer */
/* Bus callback to get end of stream and errors */
static GstBusSyncReply
kilikali_audio_gst_play_bus_callback (GstBus *bus, GstMessage *msg, 
                                      gpointer data)
{

    KilikaliAudioGst *self = NULL;
    g_assert( data != NULL );

    self = KILIKALI_AUDIO_GST(data);

    switch (GST_MESSAGE_TYPE (msg)) {
    case GST_MESSAGE_STATE_CHANGED: 
    {
        GstState state=0;

        gst_message_parse_state_changed ( msg,
                                          NULL, /*GstState *oldstate,*/
                                          &state, /*GstState *newstate,*/
                                          NULL /*GstState *pending*/ );
        if( state == GST_STATE_PLAYING) {
            self->priv->state = AUDIO_STATE_PLAYING;
            
            /* FIXME: dirty hack. probaply */
            if (self->priv->length_timeout_id == 0) {
                self->priv->length_timeout_id = g_timeout_add (1000, 
                                                     (GSourceFunc) get_length, 
                                                       self);
#ifndef MAEMO_CHANGES
    /* n8x0 (itos08) does not like if volume is setted while playbin is 
     * playing? (somewhere between pause and playing???)
     */

                set_volume (self);
#endif
            }

        } else if (state == GST_STATE_PAUSED ) {
            self->priv->state = AUDIO_STATE_PAUSED;
        } else if (state == GST_STATE_NULL ) {
            self->priv->state = AUDIO_STATE_STOPPED;
        } else if (state == GST_STATE_READY ) {
            self->priv->state = AUDIO_STATE_STOPPED;
        } else {
            self->priv->state = AUDIO_STATE_VOID;
        }
#ifdef DEBUG
        /* set states to ask net  */
        /* FIXME */
        if( state == GST_STATE_VOID_PENDING)
            g_debug("void pending");
        if( state == GST_STATE_NULL)
            g_debug("null");
        if( state == GST_STATE_READY) 
            g_debug("ready ");
        if( state == GST_STATE_PAUSED)
            g_debug("paused");
        if( state == GST_STATE_PLAYING)
            g_debug("playing");
#endif
    }
    break;
    case GST_MESSAGE_EOS:
        /* TODO : Notify playlist to give us new file */
      
        g_idle_add((GSourceFunc)emit_eos, self);

        break;
    case GST_MESSAGE_ERROR: {
        gchar *debug;
        GError *err;

        gst_message_parse_error (msg, &err, &debug);
        g_free (debug);

        g_critical ("Error: %s\n", err->message);

        /* TODO: emit some error */
        /*
        g_idle_add((GSourceFunc)emit_error, self);
        */

        g_error_free (err);

        break;
    }
    case GST_MESSAGE_ELEMENT: {

        if (gst_structure_has_name (msg->structure, "prepare-xwindow-id")
            && self->priv->xid != 0) {
#ifdef DEBUG
            g_debug ("%s - prepare-xwindow-id", __FUNCTION__);
#endif
            gst_x_overlay_set_xwindow_id(GST_X_OVERLAY(GST_MESSAGE_SRC (msg)),
                                         self->priv->xid);
            g_idle_add((GSourceFunc)video_playing_emit, data);
        }

        break;
    }
    case GST_MESSAGE_TAG: {
        GstTagList* taglist = NULL;
        char *tag;
        gst_message_parse_tag (msg, &taglist);
        if (gst_tag_list_get_string (taglist, GST_TAG_ARTIST, &tag)) {
            self->priv->tags.artist = tag;
        }
        if (gst_tag_list_get_string (taglist, GST_TAG_TITLE, &tag)) {
            self->priv->tags.title = tag;
        }
        if (gst_tag_list_get_string (taglist, GST_TAG_ALBUM, &tag)) {
            self->priv->tags.album = tag;
        }
        gst_tag_list_free (taglist);
        g_idle_add ((GSourceFunc)tag_found_emit, self);
        break;
    }

    default:
        break;
    }

    gst_message_unref( msg );
    return GST_BUS_DROP;
}



static void
set_volume (KilikaliAudioGst *self)
{
    

    if (self->avbin == NULL)
        return;

#ifdef USE_PLAYBIN
#ifdef MAEMO_CHANGES
    /* Ok, here are the rules that seem to be in effect:
     *  - standard playbin really only has the range 0.0...1.0 since anything
     *    over 1.0 starts to distort the sound
     *  - maemo playbin seems to set the volume element volume to 0.0, dunno why
     *  - maemo playbin uses the whole 0.0...10.0 range to control the dsp sink
     *
     * So in principle you could adjust volume the same way everywhere but
     * you just won't like the result (1/10th of max volume) on maemo.
     *
     * So we hack around, as usual.
     */
     g_object_set (G_OBJECT(self->avbin), "volume", 
                   self->priv->volume*10.0, NULL);

    {
      GstElement *sink = gst_bin_get_by_name(GST_BIN(self->avbin), "volume");
      if (sink != NULL)
      {
        g_object_set (G_OBJECT(sink), "volume", 1.0, NULL);
      }
    }
#else

#ifdef DEBUG
    g_debug (G_STRLOC ": volume: %f", self->priv->volume);
#endif
    g_object_set (G_OBJECT(self->avbin), "volume", 
                  self->priv->volume, NULL);
#endif
#else
    g_object_set (G_OBJECT(self->avbin), "volume", 
                  (gdouble)(65535/100*volume)/65535.0, NULL);
#endif

}

static void
set_mute (KilikaliAudioGst *self)
{
    if (self->avbin == NULL)
        return;
#ifdef USE_PLAYBIN
    if (self->priv->mute == TRUE) {
        g_object_get (G_OBJECT(self->avbin), "volume", 
                      &self->priv->mute_backup, NULL);
        g_object_set (G_OBJECT(self->avbin), "volume", 0.0, NULL);
    } else {
        g_object_set (G_OBJECT(self->avbin), "volume", 
                      self->priv->mute_backup, NULL);
    }
#else
    g_object_set (G_OBJECT(self->avbin), "mute", self->priv->mute, NULL);
#endif
}

static gboolean 
kilikali_audio_gst_is_supported (KilikaliAudio *self, const gchar *uri)
{
    GstElement *pipeline = gst_pipeline_new ("testpipe");
    GstElement *fakesink = gst_element_factory_make ("fakesink", "fakesink");
    GstElement *typefind = gst_element_factory_make ("typefind", "typefind");
    GstElement *src = gst_element_factory_make ("gnomevfssrc", "src");
    GstStateChangeReturn ret = 0;

    gst_bin_add_many (GST_BIN (pipeline), src, typefind, fakesink, NULL);
    gst_element_link (src, typefind);
    gst_element_link (typefind, fakesink);
    g_object_set (G_OBJECT (src), "location", uri, NULL);

    gst_element_set_state (GST_ELEMENT (pipeline), GST_STATE_PLAYING);

    ret = gst_element_set_state (GST_ELEMENT (pipeline), GST_STATE_PAUSED);
    switch (ret) {
    case GST_STATE_CHANGE_FAILURE:
        gst_element_set_state (GST_ELEMENT (pipeline), GST_STATE_NULL);
        gst_object_unref (GST_OBJECT (pipeline)); 
        return FALSE;
        break;
    default:
        break;
    }

    gst_element_set_state (GST_ELEMENT (pipeline), GST_STATE_NULL);
    gst_object_unref (GST_OBJECT (pipeline)); 

    return TRUE;
}

#ifndef USE_PLAYBIN
static void 
unknown_type (GObject *obj, gpointer data)
{
    /* TODO: emit unknown type */
    g_assert (GST_IS_ELEMENT (obj));
    gst_element_set_state (GST_ELEMENT (obj), GST_STATE_NULL);
    g_idle_add ((GSourceFunc)unknown_type_emit, data);
}

static gboolean
unknown_type_emit (KilikaliAudioGst *self)
{
#ifdef DEBUG
    g_debug ("%s %s",__FILE__,  __FUNCTION__);
#endif
    g_assert (KILIKALI_IS_AUDIO_GST (self));

    g_signal_emit ( self, 
                    KILIKALI_AUDIO_GET_IFACE(self)->type_unknown_signal_id,
                    0 );
    return FALSE;
}

static void 
sid_playing (KilikaliAudioGst *obj, gpointer data)
{
    KilikaliAudioGst *self = NULL;
    g_assert(KILIKALI_IS_AUDIO_GST (data));
    self = KILIKALI_AUDIO_GST (data);

    /* Just in case with idleadd */
    g_idle_add((GSourceFunc)sid_playing_emit, data);
    self->priv->sid_timeout_id = g_timeout_add(self->priv->sid_time, 
                                               (GSourceFunc) emit_eos, data);

}


static void
avbin_notify_iradio (GObject *obj, gpointer data, gpointer data2)
{
    gchar *name = NULL;
    KilikaliAudioGst *self = NULL;
    g_return_if_fail (KILIKALI_IS_AUDIO_GST (data2));

#ifdef DEBUG
    g_debug ("%s", __FUNCTION__);
#endif
    self = KILIKALI_AUDIO_GST (data2);

    g_object_get (obj, "iradio-name", &name, NULL);
    self->priv->iradioname = name;
    g_object_notify (G_OBJECT (self), "iradio-name");
}
#endif

static gboolean 
tag_found_emit (KilikaliAudioGst *self)
{
    g_signal_emit ( self, 
                    KILIKALI_AUDIO_GET_IFACE(self)->tag_found_signal_id,
                    0 );
    return FALSE;
}

/* Emacs indentatation information
 * Local Variables:
 * c-file-style:"stroustrup"
 * c-basic-offset:4
 * indent-tabs-mode:nil
 * End:
 */
