/*
  Kandis - A tea steepming timer.
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Selector.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/Value.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/Slider.h>
#include <Lum/Space.h>
#include <Lum/Table.h>
#include <Lum/TextValue.h>

#include "Configuration.h"

Selector::Selector(Lum::Model::Int* minutes, Lum::Model::Int *seconds)
 : value(new Lum::Model::String()),
   minutes(minutes),
   seconds(seconds),
   list(new Lum::Model::ListTable(2)),
   selection(new Lum::Model::SingleLineSelection()),
   addAction(new Lum::Model::Action()),
   updateAction(new Lum::Model::Action()),
   removeAction(new Lum::Model::Action()),
   okAction(new Lum::Model::Action())
{
  for (size_t i=0; i<teas.size(); i++) {
    Lum::Model::ListTable::StdEntry *entry;

    entry=new Lum::Model::ListTable::StdEntry(list);

    entry->SetString(1,teas[i].name);
    entry->SetString(2,GetTimeString(teas[i].minutes,teas[i].seconds));

    list->Append(entry);
  }

  updateAction->Disable();
  removeAction->Disable();

  AttachModel(minutes);
  AttachModel(seconds);

  AttachModel(selection);

  AttachModel(addAction);
  AttachModel(updateAction);
  AttachModel(removeAction);
  AttachModel(okAction);

  AttachModel(GetClosedAction());
}

std::wstring Selector::GetTimeString(size_t minutes, size_t seconds) const
{
  std::wstring tmp;

  tmp=Lum::Base::NumberToWString(minutes);
  tmp.append(L":");
  if (seconds<10) {
    tmp.append(L"0");
  }
  tmp.append(Lum::Base::NumberToWString(seconds));

  return tmp;
}

void Selector::PreInit()
{
  Lum::Button           *button;
  Lum::ButtonRow        *buttonRow;
  Lum::Label            *label;
  Lum::Panel            *panel,*horiz,*vert;
  Lum::Slider           *slider;
  Lum::Table            *table;
  Lum::TextValue        *value;
  Lum::Model::HeaderRef headerModel;

  panel=new Lum::VPanel();

  label=new Lum::Label();
  label->SetFlex(true,true);

  value=new Lum::TextValue();
  value->SetFlex(true,false);
  value->SetModel(this->value);
  value->SetFont(Lum::OS::display->GetFont(Lum::OS::Display::fontTypeProportional,
                                           Lum::OS::Display::fontScaleCaption2));
  value->SetStyle(Lum::OS::Font::bold);
  value->SetAlignment(Lum::TextValue::centered);
  panel->Add(value);

  panel->Add(new Lum::VSpace());

  slider=new Lum::HSlider();
  slider->SetWidth(Lum::Base::Size::stdCharWidth,30);
  slider->SetFlex(true,false);
  slider->SetModel(minutes);
  slider->SetScale(true);
  label->AddLabel(_(L"LBL_MINUTES",L"Minutes"),slider);

  slider=new Lum::HSlider();
  slider->SetWidth(Lum::Base::Size::stdCharWidth,30);
  slider->SetFlex(true,false);
  slider->SetScale(true);
  slider->SetModel(seconds);

  label->AddLabel(_(L"LBL_SECONDS",L"Seconds"),slider);
  panel->Add(label);

  panel->Add(new Lum::VSpace());

  horiz=new Lum::HPanel();
  horiz->SetFlex(true,true);

  vert=new Lum::VPanel();
  vert->SetFlex(false,true);

  button=new Lum::Button(_(L"BTN_ADD",L"_Add"));
  button->SetFlex(true,false);
  button->SetModel(addAction);
  vert->Add(button);

  button=new Lum::Button(_(L"BTN_UPDATE",L"_Update"));
  button->SetFlex(true,false);
  button->SetModel(updateAction);
  vert->Add(button);

  button=new Lum::Button(_(L"BTN_REMOVE",L"_Remove"));
  button->SetFlex(true,false);
  button->SetModel(removeAction);
  vert->Add(button);

  horiz->Add(vert);

  horiz->Add(new Lum::HSpace());

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(_(L"TBL_HEADER_TEA",L"Tea"),Lum::Base::Size::stdCharWidth,20,true);
  headerModel->AddColumn(_(L"TBL_HEADER_TIME",L"Time"),Lum::Base::Size::stdCharWidth,5);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetShowHeader(true);
  //table->GetTableView()->SetAutoHFit(true);
  table->SetModel(list);
  table->SetHeaderModel(headerModel);
  table->SetSelection(selection);
  table->SetDoubleClickAction(okAction);
  horiz->Add(table);

  panel->Add(horiz);

  panel->Add(new Lum::VSpace());

  buttonRow=Lum::ButtonRow::CreateOkCancel(okAction,GetClosedAction());
  buttonRow->SetFlex(true,false);
  panel->Add(buttonRow);

  SetMain(panel);

  Dialog::PreInit();
}

void Selector::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
    Exit();
  }
  else if (model==okAction && okAction->IsFinished()) {
    minutes->Save();
    seconds->Save();
    Exit();
  }
  else if (model==minutes || model==seconds) {
    std::wstring text;

    text=Lum::Base::NumberToWString(minutes->Get());
    text.append(L":");
    if (seconds->Get()<10) {
      text.append(L"0");
    }
    text.append(Lum::Base::NumberToWString(seconds->Get()));

    value->Set(text);
  }
  else if (model==selection && dynamic_cast<const Lum::Model::Selection::Selected*>(&msg)!=NULL) {
    if (selection->HasSelection()) {
      updateAction->Enable();
      removeAction->Enable();

      minutes->Set(teas[selection->GetLine()-1].minutes);
      seconds->Set(teas[selection->GetLine()-1].seconds);
    }
    else {
      updateAction->Disable();
      removeAction->Disable();
    }
  }
  else if (model==addAction && addAction->IsFinished()) {
    Lum::Model::StringRef name=new Lum::Model::String(_(L"TEA_NAME_DEFAULT",L"Some tea"));

    if (Lum::Dlg::TextInput::GetText(this->GetWindow(),
                                     L"",
                                     _(L"DLG_TEA_NAME_TEXT",
                                       L"Please enter the name of the tea:"),
                                       name)) {
      Lum::Model::ListTable::StdEntry *entry;
      Tea                             tea;

      // The tea list
      tea.name=name->Get();
      tea.minutes=minutes->Get();
      tea.seconds=seconds->Get();

      teas.push_back(tea);

      // The selection list
      entry=new Lum::Model::ListTable::StdEntry(list);

      entry->SetString(1,name->Get());
      entry->SetString(2,GetTimeString(minutes->Get(),seconds->Get()));

      list->Append(entry);

      configurationChanged=true;
    }
  }
  else if (model==updateAction && updateAction->IsFinished()) {
    Lum::Model::ListTable::StdEntry *entry;

    assert(selection->HasSelection());

    entry=dynamic_cast<Lum::Model::ListTable::StdEntry*>(list->GetEntry(selection->GetLine()));

    assert(entry!=NULL);

    teas[selection->GetLine()-1].minutes=minutes->Get();
    teas[selection->GetLine()-1].seconds=seconds->Get();

    entry->SetString(2,GetTimeString(minutes->Get(),seconds->Get()));

    configurationChanged=true;
  }
  else if (model==removeAction && removeAction->IsFinished()) {
    assert(selection->HasSelection());

    teas.erase(teas.begin()+selection->GetLine()-1);
    list->Delete(selection->GetLine());

    configurationChanged=true;
  }

  Dialog::Resync(model,msg);
}

