/*
  Kandis - A tea steepming timer.
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <Lum/Audio/PlaySound.h>

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/Msg.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Integer.h>

#include <Lum/OS/Main.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Dialog.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>
#include <Lum/WindowGroup.h>

#include "Configuration.h"
#include "Selector.h"
#include "StopWatch.h"

#include "config.h"

static Lum::AppInfo info;

class MainDialog : public Lum::Dialog
{
private:
  enum State {
    stateUninitialized,
    stateStoped,
    stateRunning,
    stateAlerting
  };

private:
  Lum::Model::ActionRef aboutAction;
  Lum::Model::ActionRef startAction;
  Lum::Model::ActionRef restartAction;
  Lum::Model::ActionRef stopAction;
  Lum::Model::ActionRef timer;
  Lum::Model::IntRef    minutes;
  Lum::Model::IntRef    seconds;
  size_t                lastMinutes;
  size_t                lastSeconds;

  State                 state;
  Lum::Base::SystemTime alertTime;

  StopWatch             *value;
  Lum::Audio::PlaySound *beepSound;

public:
  MainDialog()
  : aboutAction(new Lum::Model::Action()),
    startAction(new Lum::Model::Action()),
    restartAction(new Lum::Model::Action()),
    stopAction(new Lum::Model::Action()),
    timer(new Lum::Model::Action()),
    minutes(new Lum::Model::Int()),
    seconds(new Lum::Model::Int()),
    lastMinutes(0),
    lastSeconds(0),
    state(stateUninitialized),
    value(new StopWatch()),
    beepSound(NULL)
  {
    Lum::Base::Path soundPath;

    soundPath.SetNativeDir(Lum::Base::Path::GetCWD());
    soundPath.AppendDir(L"data");
    soundPath.AppendDir(L"sounds");
    soundPath.SetBaseName(L"alarm.wav");

#if defined(APP_DATADIR)
    if (!soundPath.Exists()) {
      soundPath.SetNativeDir(Lum::Base::StringToWString(APP_DATADIR));
      soundPath.AppendDir(L"sounds");
      soundPath.SetBaseName(L"alarm.wav");
    }
#endif

    if (soundPath.Exists()) {
      beepSound=Lum::Audio::GetPlaySound();
      if (beepSound!=NULL) {
        beepSound->SetFilename(soundPath.GetPath(),Lum::Audio::PlaySound::typeWav);
      }
    }

    restartAction->Disable();

    seconds->SetRange(0,59);
    minutes->SetRange(0,15);

    minutes->Set(4);
    seconds->Set(0);

    AttachModel(GetOpenedAction());
    AttachModel(GetClosedAction());

    AttachModel(minutes);
    AttachModel(seconds);

    AttachModel(value->GetClickAction());

    AttachModel(aboutAction);
    AttachModel(startAction);
    AttachModel(restartAction);
    AttachModel(stopAction);
    AttachModel(timer);
  }

  ~MainDialog()
  {
    UnattachModel(timer);
    UnattachModel(stopAction);
    UnattachModel(restartAction);
    UnattachModel(startAction);
    UnattachModel(aboutAction);

    UnattachModel(value->GetClickAction());

    UnattachModel(minutes);
    UnattachModel(seconds);

    UnattachModel(GetClosedAction());
    UnattachModel(GetOpenedAction());

    delete beepSound;
  }

  void SetValue()
  {
    std::wstring text;

    text=Lum::Base::NumberToWString(minutes->Get());
    text.append(L":");
    if (seconds->Get()<10) {
      text.append(L"0");
    }
    text.append(Lum::Base::NumberToWString(seconds->Get()));

    if (value!=NULL) {
      value->SetText(text);
    }
  }

  void PlayAlert()
  {
    if (beepSound!=NULL) {
      beepSound->Play();
    }
    else {
      Lum::OS::display->Beep();
    }
  }

  void SetState(State state)
  {
    if (this->state==state) {
      return;
    }

    this->state=state;

    switch (state) {
    case stateUninitialized:
      assert(false);
      break;
    case stateStoped:
      startAction->Enable();
      stopAction->Disable();
      break;
    case stateRunning:
      restartAction->Enable();
    case stateAlerting:
      startAction->Disable();
      stopAction->Enable();
      break;
    }

    switch (state) {
    case stateUninitialized:
      assert(false);
      break;
    case stateStoped:
      value->SetAlarm(false);
      Lum::OS::display->RemoveTimer(timer);
      break;
    case stateRunning:
      value->SetAlarm(false);
      lastMinutes=minutes->Get();
      lastSeconds=seconds->Get();
      alertTime.SetToNow();
      alertTime.Add(minutes->Get()*60+seconds->Get());
      Lum::OS::display->AddTimer(0,100000,timer);
      break;
    case stateAlerting:
      value->SetAlarm(true);
      minutes->Set(0);
      seconds->Set(0);
      PlayAlert();
      alertTime.SetToNow();
      alertTime.Add(3);
      Lum::OS::display->RemoveTimer(timer);
      Lum::OS::display->AddTimer(0,100000,timer);
      break;
    }
  }

  virtual void PreInit()
  {
    Lum::Button      *button;
    Lum::ButtonRow   *buttonRow;
    Lum::Panel       *panel;
    Lum::WindowGroup *wGroup;

    wGroup=new Lum::WindowGroup();
    wGroup->SetFlex(true,true);

    panel=new Lum::VPanel();
    panel->SetFlex(true,true);

    value->SetText(L"4:00");
    value->SetFlex(true,true);
    panel->Add(value);

    panel->Add(new Lum::VSpace());

    buttonRow=new Lum::ButtonRow();
    buttonRow->SetFlex(true,false);

    button=new Lum::Button(_ld(dlgButtonAbout));
    button->SetFlex(true,true);
    button->SetModel(aboutAction);
    buttonRow->AddOptional(button);

    button=new Lum::Button(_(L"BUTTON_START",L"_Start"));
    button->SetFlex(true,true);
    button->SetModel(startAction);
    buttonRow->Add(button);

    button=new Lum::Button(_(L"BUTTON_RESTART",L"_Restart"));
    button->SetFlex(true,true);
    button->SetModel(restartAction);
    buttonRow->Add(button);

    button=new Lum::Button(_(L"BUTTON_STOP",L"Sto_p"));
    button->SetFlex(true,true);
    button->SetModel(stopAction);
    buttonRow->Add(button);

    button=new Lum::Button(_ld(dlgButtonQuit));
    button->SetFlex(true,true);
    button->SetModel(GetClosedAction());
    buttonRow->Add(button);

    panel->Add(buttonRow);

    wGroup->SetMain(panel);

    SetTop(wGroup);

    SetState(stateStoped);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      LoadConfig();
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      if (configurationChanged && !SaveConfig()) {
        Lum::Dlg::Msg::ShowOk(this,
                              _(L"ERR_CONFIG_SAVE_CAPTION",
                                L"Error while storing configuration..."),
                              _(L"ERR_CONFIG_SAVE_TEXT",
                                L"There was an error while storing the configuration!"));
      }
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==value->GetClickAction() &&
             value->GetClickAction()->IsFinished() &&
             state==stateStoped) {
      Selector *selector;

      minutes->Push();
      seconds->Push();

      selector=new Selector(minutes,seconds);
      selector->SetParent(GetWindow());
      selector->Open();
      selector->EventLoop();
      selector->Close();

      minutes->Pop();
      seconds->Pop();

      delete selector;
    }
    else if (model==startAction && startAction->IsFinished()) {
      SetState(stateRunning);
    }
    else if (model==restartAction && restartAction->IsFinished()) {
      SetState(stateStoped);
      minutes->Set(lastMinutes);
      seconds->Set(lastSeconds);
      SetState(stateRunning);
    }
    else if (model==stopAction && stopAction->IsFinished()) {
      SetState(stateStoped);
    }
    else if (model==timer && timer->IsFinished()) {
      if (state==stateRunning) {
        Lum::Base::SystemTime now;

        if (now>=alertTime) {
          SetState(stateAlerting);
        }
        else {
          minutes->Set((alertTime.GetTime()-now.GetTime())/60);
          seconds->Set((alertTime.GetTime()-now.GetTime())%60);
          Lum::OS::display->AddTimer(0,100000,timer);
        }
      }
      else if (state==stateAlerting) {
        Lum::Base::SystemTime now;

        if (now.GetTime()>alertTime) {
          PlayAlert();
          alertTime.SetToNow();
          alertTime.Add(3);
        }

        Lum::OS::display->AddTimer(0,100000,timer);
      }
    }
    else if (model==minutes || model==seconds) {
      SetValue();
    }

    Dialog::Resync(model,msg);
  }

};

class Main : public Lum::OS::MainDialog<MainDialog>
{
public:
  bool Prepare()
  {
#if defined(APP_DATADIR)
    Lum::Base::Path::SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(_(L"ABOUT_DESC",L"A tea steeping timer"));
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2007, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    return Lum::OS::MainDialog<MainDialog>::Prepare();
  }

  void Cleanup()
  {
    FreeConfig();

    Lum::OS::MainDialog<MainDialog>::Cleanup();
  }
};

LUM_MAIN(Main,L"Kandis")
