#!/usr/bin/env python
# -*- coding: utf-8 -*-

import sys, time, os
from PyQt4 import QtCore
from PyQt4 import QtGui
from PyQt4.QtCore import Qt
from PyQt4.QtGui import QApplication
try:
    from PyQt4.QtMaemo5 import QMaemo5InformationBox
except:
    pass


greekSymbols = ["alpha", "beta", "chi", "delta", "Delta", "epsilon", "eta", "gamma", "Gamma", "iota", "kappa", "nu", "omega", "Omega", "omicron", "phi", "Phi", "pi", "Pi", "psi", "Psi", "rho", "sigma", "Sigma", "tau", "theta", "upsilon", "xi", "Xi"]

sympy_symbols_functions = ["acos()", "acosh()", "acot()", "acoth()", "asin()", "asinh()", "atan()", "atanh()", "cos()", "cosh()", "cot()", "coth()", "DiracDelta()", "E", "EulerGamma", "exp()", "factorial()", "fibonacci()", "gamma()", "GoldenRatio", "harmonic()", "hermite(n,x)", "I", "im()", "legendre(n,x)", "ln()", "ln(n,b)", "log()", "log(n,b)", "oo", "re()", "sin()", "sinh()", "sqrt()", "tan()", "tanh()", "zeta()"]

historyIntegrandSamples = ["1/(alpha*x+beta)**3","1/sqrt(alpha*x+beta)","log(sigma*x)**2","sin(kappa*x)**5", "x**3/(alpha*x**2+beta)**2", "exp(omega*t**2)*t"]

historyLimitSamples = ["pi/2","-pi/2","pi/3","-pi/3","pi/4","-pi/4","pi/6","-pi/6","-pi","-oo","0","1","-1"]

numerIntegralTypes = {'approx':0, 'infinities':1, 'smooth':2}
integralType = {'indefinite':1, 'definite':2, 'numerical':3}
simplifyType = {'none':0, 'expandterms':1, 'simplifyterms':2, 'expandall':3, 'simplifyall':4}
outputType = {'simple':0, 'bidimensional':1, 'typesetting':2, 'latex':3, 'mathml':4, 'c':5, 'fortran':6}
integralErrorMessage = u'Error: integral not calculated'
integralNumerErrorMessage = u'Error: numerical integral not calculated'

zoomFontSize = (10, 12, 14, 16, 20, 24, 28, 32)
zoomNumColumnsMaemo4 = (84, 68, 62, 52, 43, 36, 32, 27)
zoomNumColumnsMaemo5 = (97, 77, 70, 59, 49, 41, 36, 30)

maemo5InformationBoxTime = 3000


class IntegralWindow(QtGui.QMainWindow):
    def __init__(self, parent=None):
        QtGui.QMainWindow.__init__(self, parent)
        self.setWindowTitle("Integral")

        self.window = QtGui.QWidget()
        self.setCentralWidget(self.window)

        if os.path.exists("/proc/component_version"):
            self.isMaemo = True
            try:
                self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
                self.setAttribute(Qt.WA_Maemo5StackedWindow, True)
                self.zoomNumColumns = zoomNumColumnsMaemo5
                self.isMaemo5 = True
            except:
                self.isMaemo5 = False
                self.zoomNumColumns = zoomNumColumnsMaemo4
        else:
            self.zoomNumColumns = zoomNumColumnsMaemo5
            self.isMaemo = False
            self.isMaemo5 = False

        if not self.isMaemo:
            self.resize(800, 600)

        self.settings = QtCore.QSettings()
        self.loadSettings()
        # Setting the pretty printing of SymPy to use unicode characters and correct number of columns.
        init_printing(use_unicode=True, num_columns=self.zoomNumColumns[self.zoomLevel])

        self.historyInitIntegrand = QtCore.QStringList()
        for i in greekSymbols+sympy_symbols_functions+historyIntegrandSamples:
            self.historyInitIntegrand.append(QtCore.QString(i))
        self.historyIntegrand=self.historyInitIntegrand+self.historyUserIntegrand

        self.historyInitLimits = QtCore.QStringList()
        for i in greekSymbols+sympy_symbols_functions+historyLimitSamples:
            self.historyInitLimits.append(QtCore.QString(i))
        self.historyLimits=self.historyInitLimits+self.historyUserLimits

        self.historyDifferentials = QtCore.QStringList()
        for i in greekSymbols:
            self.historyDifferentials.append(QtCore.QString(i))

        self.nonCalculatedIntegral = u""
        self.resultIntegral = u""
        self.resultIntegralSimp = u""
        self.resultOutput = u""
        self.nonCalculatedIntegralOutput = u""
        self.timeIntegral = 0.0
        self.configUpdateResult = False
        self.createMenuAndActions()
        self.createItems()
        self.showHideItems()
        self.createLayouts()

    def createItems(self):
        self.integrandEditCompleter = QtGui.QCompleter()
        self.integrandEditCompleter.setCaseSensitivity(QtCore.Qt.CaseInsensitive)
        self.integrandEditModel = QtGui.QStringListModel()
        self.integrandEditCompleter.setModel(self.integrandEditModel)
        self.integrandEditModel.setStringList(self.historyIntegrand)

        self.integrandLabel = QtGui.QLabel(self.tr("Integrand"))
        self.integrandEdit = QtGui.QLineEdit("")
        self.integrandEdit.setCompleter(self.integrandEditCompleter)

        self.diffEditCompleter = QtGui.QCompleter()
        self.diffEditCompleter.setCaseSensitivity(QtCore.Qt.CaseInsensitive)
        self.diffEditModel = QtGui.QStringListModel()
        self.diffEditCompleter.setModel(self.diffEditModel)
        self.diffEditModel.setStringList(self.historyDifferentials)

        self.diff1Label = QtGui.QLabel("d")
        self.diff1Edit = QtGui.QLineEdit("x")
        self.diff1Edit.setMaximumSize(QtCore.QSize(80,100))
        self.diff1Edit.setCompleter(self.diffEditCompleter)
        self.diff2Label = QtGui.QLabel("d")
        self.diff2Edit = QtGui.QLineEdit("y")
        self.diff2Edit.setMaximumSize(QtCore.QSize(80,100))
        self.diff2Edit.setCompleter(self.diffEditCompleter)
        self.diff3Label = QtGui.QLabel("d")
        self.diff3Edit = QtGui.QLineEdit("z")
        self.diff3Edit.setMaximumSize(QtCore.QSize(80,100))
        self.diff3Edit.setCompleter(self.diffEditCompleter)

        self.limEditCompleter = QtGui.QCompleter()
        self.limEditCompleter.setCaseSensitivity(QtCore.Qt.CaseInsensitive)
        self.limEditModel = QtGui.QStringListModel()
        self.limEditCompleter.setModel(self.limEditModel)
        self.limEditModel.setStringList(self.historyLimits)

        self.limSupLabel = QtGui.QLabel(self.tr("Upper limit"))
        self.limSup1Edit = QtGui.QLineEdit("1")
        self.limSup1Edit.setCompleter(self.limEditCompleter)
        self.limSup2Edit = QtGui.QLineEdit("1")
        self.limSup2Edit.setCompleter(self.limEditCompleter)
        self.limSup3Edit = QtGui.QLineEdit("1")
        self.limSup3Edit.setCompleter(self.limEditCompleter)
        self.limInfLabel = QtGui.QLabel(self.tr("Lower limit"))
        self.limInf1Edit = QtGui.QLineEdit("0")
        self.limInf1Edit.setCompleter(self.limEditCompleter)
        self.limInf2Edit = QtGui.QLineEdit("0")
        self.limInf2Edit.setCompleter(self.limEditCompleter)
        self.limInf3Edit = QtGui.QLineEdit("0")
        self.limInf3Edit.setCompleter(self.limEditCompleter)

        self.font = QtGui.QFont()
#        self.font.setFamily("Monospace")
        self.font.setFamily("Dejavu Sans Mono")
        self.font.setPointSize(zoomFontSize[self.zoomLevel])
        self.resultEdit = QtGui.QTextBrowser()
        self.resultEdit.setReadOnly(False)
        self.resultEdit.setFont(self.font)

        # Connect buttons
        self.connect(self.limSup1Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.limSup2Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.limSup3Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.diff1Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.diff2Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.diff3Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.limInf1Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.limInf2Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.limInf3Edit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)
        self.connect(self.integrandEdit,QtCore.SIGNAL("returnPressed()"),self.updateHistoriesAndResult)

    def createLayouts(self):    
        self.h_box_limSup = QtGui.QHBoxLayout()
        self.h_box_limSup.addWidget(self.limSupLabel)
        self.h_box_limSup.addWidget(self.limSup3Edit)
        self.h_box_limSup.addWidget(self.limSup2Edit)
        self.h_box_limSup.addWidget(self.limSup1Edit)
        self.h_box_integrand = QtGui.QHBoxLayout()
        self.h_box_integrand.addWidget(self.integrandLabel)
        self.h_box_integrand.addWidget(self.integrandEdit)
        self.h_box_integrand.addWidget(self.diff1Label)
        self.h_box_integrand.addWidget(self.diff1Edit)
        self.h_box_integrand.addWidget(self.diff2Label)
        self.h_box_integrand.addWidget(self.diff2Edit)
        self.h_box_integrand.addWidget(self.diff3Label)
        self.h_box_integrand.addWidget(self.diff3Edit)
        self.h_box_limInf = QtGui.QHBoxLayout()
        self.h_box_limInf.addWidget(self.limInfLabel)
        self.h_box_limInf.addWidget(self.limInf3Edit)
        self.h_box_limInf.addWidget(self.limInf2Edit)
        self.h_box_limInf.addWidget(self.limInf1Edit)
        self.v_box = QtGui.QVBoxLayout()
        self.v_box.addLayout(self.h_box_limSup)
        self.v_box.addLayout(self.h_box_integrand)
        self.v_box.addLayout(self.h_box_limInf)
        self.v_box.addWidget(self.resultEdit)
        self.window.setLayout(self.v_box)
        
    def showHideItems(self):
        if self.typeIntegral == integralType['indefinite']:
            self.limSupLabel.hide()
            self.limSup1Edit.hide()
    	    self.limSup2Edit.hide()
    	    self.limSup3Edit.hide()
            self.limInfLabel.hide()
            self.limInf1Edit.hide()
    	    self.limInf2Edit.hide()
    	    self.limInf3Edit.hide()
        else:
            self.limSupLabel.show()
            self.limSup1Edit.show()
            self.limInfLabel.show()
            self.limInf1Edit.show()
            if self.numDimensions > 1:
        	    self.limSup2Edit.show()
        	    self.limInf2Edit.show()
            else:
        	    self.limSup2Edit.hide()
        	    self.limInf2Edit.hide()
            if self.numDimensions > 2:
        	    self.limSup3Edit.show()
        	    self.limInf3Edit.show()
            else:
        	    self.limSup3Edit.hide()
        	    self.limInf3Edit.hide()
        if self.numDimensions > 1:
            self.diff2Label.show()
            self.diff2Edit.show()
        else:
           self.diff2Label.hide()
           self.diff2Edit.hide()
        if self.numDimensions > 2:
            self.diff3Label.show()
            self.diff3Edit.show()
        else:
           self.diff3Label.hide()
           self.diff3Edit.hide()

    def createMenuAndActions(self):
        if not self.isMaemo:
            fileMenu = self.menuBar().addMenu(self.tr("&File"))
#           saveAction = fileMenu.addAction(self.tr("&Salvar"))
#           shareAction = fileMenu.addAction(self.tr("&Share"))
            fileMenu.addSeparator()
            exitAction = QtGui.QAction(self.tr("E&xit"), self, shortcut="Ctrl+Q")
            fileMenu.addAction(exitAction)

        viewMenu = self.menuBar().addMenu(self.tr("&View"))
        zoomInAction = viewMenu.addAction(self.tr("Zoom in"))
        zoomOutAction = viewMenu.addAction(self.tr("Zoom out"))

        integralMenu = self.menuBar().addMenu(self.tr("&Integral"))
        undefinedAction = QtGui.QAction(self.tr("Indefinite"), self, checkable=True)
        definedAction = QtGui.QAction(self.tr("Definite"), self, checkable=True)
        numericalAction = QtGui.QAction(self.tr("Numerical"), self, checkable=True)
        typeIntegralGroup = QtGui.QActionGroup(self)
        typeIntegralGroup.addAction(undefinedAction)
        typeIntegralGroup.addAction(definedAction)
        typeIntegralGroup.addAction(numericalAction)
        if self.typeIntegral == integralType['indefinite']:
            undefinedAction.setChecked(True)
        elif self.typeIntegral == integralType['definite']:
            definedAction.setChecked(True)
        else:
            numericalAction.setChecked(True)     
        integralMenu.addAction(undefinedAction)
        integralMenu.addAction(definedAction)
        integralMenu.addAction(numericalAction)
        integralMenu.addSeparator()
        n1DAction = QtGui.QAction("1D", self, checkable=True)
        n2DAction = QtGui.QAction("2D", self, checkable=True)
        n3DAction = QtGui.QAction("3D", self, checkable=True)
        dimIntegralGroup = QtGui.QActionGroup(self)
        dimIntegralGroup.addAction(n1DAction)
        dimIntegralGroup.addAction(n2DAction)
        dimIntegralGroup.addAction(n3DAction)
        if self.numDimensions == 1:
            n1DAction.setChecked(True)
        elif self.numDimensions == 2:
            n2DAction.setChecked(True)
        else:
            n3DAction.setChecked(True)
        integralMenu.addAction(n1DAction)
        integralMenu.addAction(n2DAction)
        integralMenu.addAction(n3DAction)
        integralMenu.addSeparator()
        configAction = integralMenu.addAction(self.tr("Settings"))

        helpMenu = self.menuBar().addMenu(self.tr("&Help"))
        helpAction = helpMenu.addAction(self.tr("&Help"))
        helpMenu.addSeparator()
        aboutAction = helpMenu.addAction(self.tr("&About"))

        if self.isMaemo and (not self.isMaemo5):
            exitAction = QtGui.QAction(self.tr("E&xit"), self, shortcut="Ctrl+Q")
            self.menuBar().addAction(exitAction)

        self.connect(zoomInAction, QtCore.SIGNAL("triggered()"), self.zoomInMenu)
        self.connect(zoomOutAction, QtCore.SIGNAL("triggered()"), self.zoomOutMenu)
        self.connect(undefinedAction, QtCore.SIGNAL("triggered()"), self.undefinedMenu)
        self.connect(definedAction, QtCore.SIGNAL("triggered()"), self.definedMenu)
        self.connect(numericalAction, QtCore.SIGNAL("triggered()"), self.numericalMenu)
        self.connect(n1DAction, QtCore.SIGNAL("triggered()"), self.n1DMenu)
        self.connect(n2DAction, QtCore.SIGNAL("triggered()"), self.n2DMenu)
        self.connect(n3DAction, QtCore.SIGNAL("triggered()"), self.n3DMenu)
        if not self.isMaemo5:
            self.connect(exitAction, QtCore.SIGNAL("triggered()"), app, QtCore.SLOT("quit()"))
        self.connect(configAction, QtCore.SIGNAL("triggered()"), self.showConfigWindow)
        self.connect(helpAction, QtCore.SIGNAL("triggered()"), self.showHelpBox)
        self.connect(aboutAction, QtCore.SIGNAL("triggered()"), self.showAboutBox)

    def showHelpBox(self):
        QtGui.QMessageBox.information(self, self.tr("Help on Integral"),
            u'Mathematical operators : + - * / ** (power)\nExample of functions : sqrt, exp, log, sin, acos\n' +\
            u'Examples of integrands : 1/(alpha*x+beta)**3,\n   sin(kappa*x)**5, exp(omega*t**2)*t\n'+\
            u'Examples of variables : x, t, alpha, Omega\n'+u'Examples of limits : 0, -oo, pi/2, E, sqrt(1-y**2)\n'+\
            u'Look at the SymPy site: http://sympy.org/')

    def showAboutBox(self):
        versionIntegral = '1.1.3'
        siteIntegral = 'http://www.RobertoColistete.net/Integral\n'
        infoIntegral = self.tr("Calculation of integrals using ")
        infoPython = "Python v"+python_version()
        infoSymPy = "SymPy v"+__version__+" (http://sympy.org/)"
        QtGui.QMessageBox.information(self, self.tr("About Integral"),
            'Integral v'+versionIntegral+' - LGPLv3 (C) 2011 Roberto Colistete Jr.\n'+\
            siteIntegral+infoIntegral+infoPython+" and\n"+infoSymPy+'\n\n* In loving memory of my wife Lorena *')

    def showConfigWindow(self):
        if self.isMaemo5:
            self.configWin = QtGui.QMainWindow(self)
            self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
            self.configWin.setAttribute(Qt.WA_Maemo5StackedWindow)
            self.configWin.window = QtGui.QWidget()
            self.configWin.setCentralWidget(self.configWin.window)
        else:
            self.configWin = QtGui.QDialog(self)

        self.configWin.setWindowTitle(self.tr("Integral (Settings)"))
        self.configWin.v_box = QtGui.QVBoxLayout()

        self.configUpdateResult = False

        self.configWin.showIntegralCheckBox = QtGui.QCheckBox(self.tr("Show not calculated integral before result"))
        if self.showIntegral:
            self.configWin.showIntegralCheckBox.toggle()

        self.configWin.showTimeCheckBox = QtGui.QCheckBox(self.tr("Show calculation time before integral result"))
        if self.showTime:
            self.configWin.showTimeCheckBox.toggle()

        self.configWin.h_box_simplify = QtGui.QHBoxLayout()
        self.configWin.simplifyLabel = QtGui.QLabel(self.tr("Simplification for non-numerical integration"))
        self.configWin.simplifySpacer = QtGui.QSpacerItem(40, 20, QtGui.QSizePolicy.Expanding,\
                                                          QtGui.QSizePolicy.Minimum)
        self.configWin.simplifyComboBox = QtGui.QComboBox()
        self.configWin.simplifyComboBox.addItem(self.tr("None"))
        self.configWin.simplifyComboBox.addItem(self.tr("Expand terms"))
        self.configWin.simplifyComboBox.addItem(self.tr("Simplify terms"))
        self.configWin.simplifyComboBox.addItem(self.tr("Expand all"))
        self.configWin.simplifyComboBox.addItem(self.tr("Simplify all"))
        self.configWin.simplifyComboBox.setCurrentIndex(self.simplifyResult)
        self.configWin.h_box_simplify.addWidget(self.configWin.simplifyLabel)
        self.configWin.h_box_simplify.addWidget(self.configWin.simplifyComboBox)
        self.configWin.h_box_simplify.addItem(self.configWin.simplifySpacer)

        self.configWin.h_box_output = QtGui.QHBoxLayout()
        self.configWin.outputLabel = QtGui.QLabel(self.tr("Output type for integral result"))
        self.configWin.outputSpacer = QtGui.QSpacerItem(40, 20, QtGui.QSizePolicy.Expanding, QtGui.QSizePolicy.Minimum)
        self.configWin.outputComboBox = QtGui.QComboBox()
        self.configWin.outputComboBox.addItem(self.tr("Simple"))
        self.configWin.outputComboBox.addItem(self.tr("Bidimensional"))
        self.configWin.outputComboBox.addItem(self.tr("Typesetting"))
        self.configWin.outputComboBox.addItem("LaTex")
        self.configWin.outputComboBox.addItem("MathML")
        self.configWin.outputComboBox.addItem("C")
        self.configWin.outputComboBox.addItem("Fortran")
        self.configWin.outputComboBox.setCurrentIndex(self.outputResult)
        self.configWin.h_box_output.addWidget(self.configWin.outputLabel)
        self.configWin.h_box_output.addWidget(self.configWin.outputComboBox)
        self.configWin.h_box_output.addItem(self.configWin.outputSpacer)

        self.configWin.h_box_numDig = QtGui.QHBoxLayout()
        self.configWin.numDigLabel = QtGui.QLabel(self.tr("Number of digits for numerical integration"))
        self.configWin.numDigSpacer = QtGui.QSpacerItem(40, 20, QtGui.QSizePolicy.Expanding, QtGui.QSizePolicy.Minimum)
        self.configWin.numDigEdit = QtGui.QLineEdit(str(self.numDig))
        self.configWin.numDigEdit.setValidator(QtGui.QIntValidator(1,1000000,self.configWin.numDigEdit))
        self.configWin.numDigEdit.setMaximumSize(QtCore.QSize(100,100))
        self.configWin.h_box_numDig.addWidget(self.configWin.numDigLabel)
        self.configWin.h_box_numDig.addWidget(self.configWin.numDigEdit)
        self.configWin.h_box_numDig.addItem(self.configWin.numDigSpacer)

        self.configWin.h_box_numerInt = QtGui.QHBoxLayout()
        self.configWin.numerIntLabel = QtGui.QLabel(self.tr("Numerical integration method"))
        self.configWin.numerIntSpacer = QtGui.QSpacerItem(40, 20, QtGui.QSizePolicy.Expanding,\
                                                          QtGui.QSizePolicy.Minimum)
        self.configWin.numerIntComboBox = QtGui.QComboBox()
        self.configWin.numerIntComboBox.addItem(self.tr("Numerical approximation of definite integral"))
        self.configWin.numerIntComboBox.addItem(self.tr("Optimized for infinities"))
        self.configWin.numerIntComboBox.addItem(self.tr("Optimized for smooth integrands"))
        self.configWin.numerIntComboBox.setCurrentIndex(self.numerIntegralType)
        self.configWin.h_box_numerInt.addWidget(self.configWin.numerIntLabel)
        self.configWin.h_box_numerInt.addWidget(self.configWin.numerIntComboBox)
        self.configWin.h_box_numerInt.addItem(self.configWin.numerIntSpacer)

        if not self.isMaemo5:
            self.configWin.h_box_close = QtGui.QHBoxLayout()
            self.configWin.closeButton = QtGui.QPushButton("Close")
            self.configWin.h_box_close.addStretch(1)
            self.configWin.h_box_close.addWidget(self.configWin.closeButton)

        self.configWin.v_box.addWidget(self.configWin.showIntegralCheckBox)
        self.configWin.v_box.addWidget(self.configWin.showTimeCheckBox)
        self.configWin.v_box.addLayout(self.configWin.h_box_simplify)
        self.configWin.v_box.addLayout(self.configWin.h_box_output)
        self.configWin.v_box.addLayout(self.configWin.h_box_numDig)
        self.configWin.v_box.addLayout(self.configWin.h_box_numerInt)
        if not self.isMaemo5:
            self.configWin.v_box.addStretch(1)
            self.configWin.v_box.addSpacing(12)
            self.configWin.v_box.addLayout(self.configWin.h_box_close)
            self.configWin.setLayout(self.configWin.v_box)
        else:
            self.configWin.window.setLayout(self.configWin.v_box)

        self.connect(self.configWin.showIntegralCheckBox, QtCore.SIGNAL("stateChanged(int)"), self.toggleShowIntegral)
        self.connect(self.configWin.showTimeCheckBox, QtCore.SIGNAL("stateChanged(int)"), self.toggleShowTime)
        self.connect(self.configWin.simplifyComboBox, QtCore.SIGNAL("activated(int)"), self.changeSimplifyResult)
        self.connect(self.configWin.outputComboBox, QtCore.SIGNAL("activated(int)"), self.changeOutputResult)
        self.connect(self.configWin.numDigEdit, QtCore.SIGNAL("textEdited (const QString&)"),self.changeNumDig)
        self.connect(self.configWin.numerIntComboBox, QtCore.SIGNAL("activated(int)"), self.changeNumerIntegralType)
        if not self.isMaemo5:
            self.configWin.closeButton.clicked.connect(self.configWin.close)
        self.configWin.show()

    def loadSettings(self):
        self.zoomLevel = self.settings.value('zoomLevel',3).toInt()[0]
        self.numDimensions = self.settings.value('NumDimensions',1).toInt()[0]
        self.typeIntegral = self.settings.value('TypeIntegral',integralType['indefinite']).toInt()[0]
        self.showIntegral = self.settings.value('ShowIntegral',True).toBool()
        self.showTime = self.settings.value('ShowTime',True).toBool()
        self.simplifyResult = self.settings.value('SimplifyResult',simplifyType['simplifyterms']).toInt()[0]
        self.outputResult = self.settings.value('OutputResult',outputType['bidimensional']).toInt()[0]
        self.numDig = self.settings.value('NumDig',15).toInt()[0]
        self.numerIntegralType = self.settings.value('NumerIntegralType',numerIntegralTypes['infinities']).toInt()[0]
        self.historyUserIntegrand = self.settings.value('HistoryUserIntegrand','').toStringList()
        self.historyUserLimits = self.settings.value('HistoryUserLimits','').toStringList()

    def saveSettings(self):
        self.settings.setValue('zoomLevel',self.zoomLevel)
        self.settings.setValue('NumDimensions',self.numDimensions)
        self.settings.setValue('TypeIntegral',self.typeIntegral) 	
        self.settings.setValue('ShowIntegral',self.showIntegral) 	
        self.settings.setValue('ShowTime',self.showTime) 	
        self.settings.setValue('SimplifyResult',self.simplifyResult)
        self.settings.setValue('OutputResult',self.outputResult)
        self.settings.setValue('NumDig',self.numDig)
        self.settings.setValue('NumerIntegralType',self.numerIntegralType)
        self.settings.setValue('HistoryUserIntegrand',self.historyUserIntegrand)
        self.settings.setValue('HistoryUserLimits',self.historyUserLimits)

    def zoomInMenu(self):
        if self.zoomLevel < (len(zoomFontSize)-1):
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nZoom in to font size "+str(zoomFontSize[self.zoomLevel])+"\n"), \
                                                  maemo5InformationBoxTime)
                QApplication.processEvents()
            self.zoomLevel += 1
            self.font.setPointSize(zoomFontSize[self.zoomLevel])
            self.resultEdit.setFont(self.font)
            # Setting the pretty printing of SymPy to use unicode characters and correct number of columns.
            init_printing(use_unicode=True, num_columns=self.zoomNumColumns[self.zoomLevel])
            self.updateResult(flagCalculate=False)
            self.configUpdateResult = True
            self.saveSettings()          

    def zoomOutMenu(self):
        if self.zoomLevel > 0:
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nZoom out to font size "+str(zoomFontSize[self.zoomLevel])+"\n"), \
                                                  maemo5InformationBoxTime)
                QApplication.processEvents()
            self.zoomLevel -= 1
            self.font.setPointSize(zoomFontSize[self.zoomLevel])
            self.resultEdit.setFont(self.font)
            # Setting the pretty printing of SymPy to use unicode characters and correct number of columns.
            init_printing(use_unicode=True, num_columns=self.zoomNumColumns[self.zoomLevel])
            self.updateResult(flagCalculate=False)
            self.configUpdateResult = True
            self.saveSettings()          

    def toggleShowIntegral(self, status):
        if status == Qt.Checked:
            self.showIntegral = True
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nNot calculated integral is visible\n"), maemo5InformationBoxTime)
                QApplication.processEvents()
        else:
            self.showIntegral = False
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nNot calculated integral is not visible\n"), maemo5InformationBoxTime)
                QApplication.processEvents()
        self.updateResult(flagCalculate=False)
        self.configUpdateResult = True
        self.saveSettings()

    def toggleShowTime(self, status):
        if status == Qt.Checked:
            self.showTime = True
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nCalculation time is visible\n"), maemo5InformationBoxTime)
                QApplication.processEvents()
        else:
            self.showTime = False
            if self.isMaemo5:
                QMaemo5InformationBox.information(self, QtCore.QString("\nCalculation time is not visible\n"), maemo5InformationBoxTime)
                QApplication.processEvents()
        self.updateResult(flagCalculate=False)
        self.configUpdateResult = True
        self.saveSettings()

    def changeSimplifyResult(self, option):
        self.simplifyResult = option
        self.configUpdateResult = True
        self.saveSettings()

    def changeOutputResult(self, option):
        self.outputResult = option
        if self.isMaemo5:
            if self.outputResult == outputType['simple']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nIntegral result converted to simple output\n"), maemo5InformationBoxTime)
            elif self.outputResult == outputType['bidimensional']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nIntegral result converted to bidimensional output\n"), maemo5InformationBoxTime)
            elif self.outputResult == outputType['typesetting']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nIntegral result converted to typesetting output\n"), maemo5InformationBoxTime)
                QApplication.processEvents()
            elif self.outputResult == outputType['latex']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nIntegral result converted to LaTeX output\n"), maemo5InformationBoxTime)
            elif self.outputResult == outputType['mathml']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nIntegral result converted to MathML output\n"), maemo5InformationBoxTime)
            elif self.outputResult == outputType['c']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nIntegral result converted to C output\n"), maemo5InformationBoxTime)
            elif self.outputResult == outputType['fortran']:
                QMaemo5InformationBox.information(self, QtCore.QString("\nIntegral result converted to Fortran output\n"), maemo5InformationBoxTime)
            QApplication.processEvents()
        self.updateResult(flagCalculate=False)
        self.configUpdateResult = True
        self.saveSettings()

    def changeNumerIntegralType(self, option):
        self.numerIntegralType = option
        self.saveSettings()

    def changeNumDig(self):
        if self.configWin.numDigEdit.text():
            self.numDig = int(self.configWin.numDigEdit.text())
        else:
            self.numDig = 0
        self.saveSettings()

    def undefinedMenu(self):
        self.typeIntegral = integralType['indefinite']
        self.showHideItems()
        self.saveSettings()

    def definedMenu(self):
        self.typeIntegral = integralType['definite']
        self.showHideItems()
        self.saveSettings()

    def numericalMenu(self):
        self.typeIntegral = integralType['numerical']
        self.showHideItems()
        self.saveSettings()

    def n1DMenu(self):
        self.numDimensions = 1
        self.showHideItems()
        self.saveSettings()

    def n2DMenu(self):
        self.numDimensions = 2
        self.showHideItems()
        self.saveSettings()

    def n3DMenu(self):
        self.numDimensions = 3
        self.showHideItems()
        self.saveSettings()

    def mapexpr(self,expr,func):
        if isinstance(expr,Add):
            return apply(Add,map(func,expr.args))
        else:
            return func(expr)

    def updateHistoriesAndResult(self):
        text = unicode(self.integrandEdit.text())
        if not text in self.historyIntegrand:
            self.historyUserIntegrand.prepend(text)
            self.historyUserIntegrand=self.historyUserIntegrand[:100]
            self.historyIntegrand=self.historyInitIntegrand+self.historyUserIntegrand
            self.integrandEditModel.setStringList(self.historyIntegrand)
        text = unicode(self.diff1Edit.text())
        if not text in self.historyDifferentials:
            self.historyDifferentials.prepend(text)
            self.diffEditModel.setStringList(self.historyDifferentials)
        text = unicode(self.diff2Edit.text())
        if not text in self.historyDifferentials:
            self.historyDifferentials.prepend(text)
            self.diffEditModel.setStringList(self.historyDifferentials)
        text = unicode(self.diff3Edit.text())
        if not text in self.historyDifferentials:
            self.historyDifferentials.prepend(text)
            self.diffEditModel.setStringList(self.historyDifferentials)
        text = unicode(self.limSup1Edit.text())
        if not text in self.historyLimits:
            self.historyUserLimits.prepend(text)
            self.historyUserLimits=self.historyUserLimits[:100]
            self.historyLimits=self.historyInitLimits+self.historyUserLimits
            self.limEditModel.setStringList(self.historyLimits)
        text = unicode(self.limSup2Edit.text())
        if not text in self.historyLimits:
            self.historyUserLimits.prepend(text)
            self.historyUserLimits=self.historyUserLimits[:100]
            self.historyLimits=self.historyInitLimits+self.historyUserLimits
            self.limEditModel.setStringList(self.historyLimits)
        text = unicode(self.limSup3Edit.text())
        if not text in self.historyLimits:
            self.historyUserLimits.prepend(text)
            self.historyUserLimits=self.historyUserLimits[:100]
            self.historyLimits=self.historyInitLimits+self.historyUserLimits
            self.limEditModel.setStringList(self.historyLimits)
        text = unicode(self.limInf1Edit.text())
        if not text in self.historyLimits:
            self.historyUserLimits.prepend(text)
            self.historyUserLimits=self.historyUserLimits[:100]
            self.historyLimits=self.historyInitLimits+self.historyUserLimits
            self.limEditModel.setStringList(self.historyLimits)
        text = unicode(self.limInf2Edit.text())
        if not text in self.historyLimits:
            self.historyUserLimits.prepend(text)
            self.historyUserLimits=self.historyUserLimits[:100]
            self.historyLimits=self.historyInitLimits+self.historyUserLimits
            self.limEditModel.setStringList(self.historyLimits)
        text = unicode(self.limInf3Edit.text())
        if not text in self.historyLimits:
            self.historyUserLimits.prepend(text)
            self.historyUserLimits=self.historyUserLimits[:100]
            self.historyLimits=self.historyInitLimits+self.historyUserLimits
            self.limEditModel.setStringList(self.historyLimits)
        self.saveSettings()
        self.updateResult()

    def updateResult(self, flagCalculate=True):
        if flagCalculate:
            self.resultEdit.clear()
            self.resultEdit.setTextColor(QtGui.QColor("darkGreen"))
            self.resultEdit.append(u"Calculating integral...")
            self.resultEdit.setTextColor(QtGui.QColor("black"))
            QApplication.processEvents()
            if self.isMaemo5:
                self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator,True)
                QApplication.processEvents()
    	    timet1=time.time()
            integrand = unicode(self.integrandEdit.text())
            diffvar1 = unicode(self.diff1Edit.text())
            if self.numDimensions > 1:
                diffvar2 = unicode(self.diff2Edit.text())
            if self.numDimensions > 2:
                diffvar3 = unicode(self.diff3Edit.text())
            if self.typeIntegral > integralType['indefinite']:
                limSup1 = unicode(self.limSup1Edit.text())
                limInf1 = unicode(self.limInf1Edit.text())
                if self.numDimensions > 1:
                    limSup2 = unicode(self.limSup2Edit.text())
                    limInf2 = unicode(self.limInf2Edit.text())
                if self.numDimensions > 2:
                    limSup3 = unicode(self.limSup3Edit.text())
                    limInf3 = unicode(self.limInf3Edit.text())
            integrateExpr = u'('+integrand+u','
            if self.typeIntegral == integralType['indefinite']:
                integrateExpr += diffvar1
                if self.numDimensions > 1:
                    integrateExpr += u','+diffvar2
                if self.numDimensions > 2:
                    integrateExpr += u','+diffvar3
                integrateExpr += u')'
                try:
                    self.nonCalculatedIntegral = sympify(u'Integral'+integrateExpr)
                except:
                    self.nonCalculatedIntegral = u'Integral'+integrateExpr
                try:
                    self.resultIntegral = sympify(u'integrate'+integrateExpr)
                except:
                    self.resultIntegral = integralErrorMessage
            elif self.typeIntegral == integralType['definite']:
                integrateExpr += u'('+diffvar1+u','+limInf1+u','+limSup1+u')'
                if self.numDimensions > 1:
                    integrateExpr += u',('+diffvar2+u','+limInf2+u','+limSup2+u')'
                if self.numDimensions > 2:
                    integrateExpr += u',('+diffvar3+u','+limInf3+u','+limSup3+u')'
                integrateExpr += u')'
                try:
                    self.nonCalculatedIntegral = sympify(u'Integral'+integrateExpr)
                except:
                    self.nonCalculatedIntegral = u'Integral'+integrateExpr
                try:
                    self.resultIntegral = sympify(u'integrate'+integrateExpr)
                except:
                    self.resultIntegral = integralErrorMessage
            else:
                integrateExpr += u'('+diffvar1+u','+limInf1+u','+limSup1+u')'
                if self.numDimensions > 1:
                    integrateExpr += u',('+diffvar2+u','+limInf2+u','+limSup2+u')'
                if self.numDimensions > 2:
                    integrateExpr += u',('+diffvar3+u','+limInf3+u','+limSup3+u')'
                integrateExpr += u')'
                try:
                    self.nonCalculatedIntegral = sympify(u'Integral'+integrateExpr)
                except:
                    self.nonCalculatedIntegral = u'Integral'+integrateExpr
                if self.numerIntegralType == numerIntegralTypes['approx']:
                    try:
                        self.resultIntegral = sympify(u'N(integrate'+integrateExpr+u','+str(self.numDig)+u')')
                    except:
                        self.resultIntegral = integralErrorMessage
                else:
                    mp.dps=self.numDig
                    integrateExpr = u"(lambda "
                    if self.numDimensions > 2:
                        integrateExpr += diffvar3+u","
                    if self.numDimensions > 1:
                        integrateExpr += diffvar2+u","
                    integrand = self.fixMPMathText(integrand)
                    integrateExpr += diffvar1+u": "+integrand
                    if self.numDimensions > 2:
                        limInf3 = self.fixMPMathText(limInf3)
                        limSup3 = self.fixMPMathText(limSup3)
                        integrateExpr += u',['+limInf3+u','+limSup3+u']'
                    if self.numDimensions > 1:
                        limInf2 = self.fixMPMathText(limInf2)
                        limSup2 = self.fixMPMathText(limSup2)
                        integrateExpr += u',['+limInf2+u','+limSup2+u']'
                    limInf1 = self.fixMPMathText(limInf1)
                    limSup1 = self.fixMPMathText(limSup1)
                    integrateExpr += u",["+limInf1+u","+limSup1+u"])"
                    try:
                        if self.numerIntegralType == numerIntegralTypes['infinities']:
                             self.resultIntegral = eval(str(u"quadts"+integrateExpr))
                        else:
                            self.resultIntegral = eval(str(u"quadgl"+integrateExpr))
                        self.resultIntegral = sympify(self.fixMPMathOutput(str(self.resultIntegral)))
                    except:
                        self.resultIntegral = integralNumerErrorMessage
            if (self.resultIntegral) and (type(self.resultIntegral) != unicode) and (self.typeIntegral <= integralType['definite']):
                if self.simplifyResult == simplifyType['none']:
                    self.resultIntegralSimp = sympify(self.resultIntegral)
                elif self.simplifyResult == simplifyType['expandterms']:
                    self.resultIntegralSimp = sympify(self.mapexpr(self.resultIntegral,expand))
                elif self.simplifyResult == simplifyType['simplifyterms']:
                    self.resultIntegralSimp = sympify(self.mapexpr(self.resultIntegral,simplify))
                elif self.simplifyResult == simplifyType['expandall']:
                    self.resultIntegralSimp = sympify(expand(self.resultIntegral))
                elif self.simplifyResult == simplifyType['simplifyall']:
                    self.resultIntegralSimp= sympify(simplify(self.resultIntegral))
            else:
                self.resultIntegralSimp = self.resultIntegral
    	    timet2=time.time()
            self.timeIntegral=timet2-timet1
            if self.isMaemo5:
                self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator,False)
                QApplication.processEvents()
        self.nonCalculatedIntegralOutput = self.nonCalculatedIntegral
        self.resultOutput = self.resultIntegralSimp
        if self.outputResult == outputType['bidimensional']:
            if (type(self.nonCalculatedIntegral) != unicode):
                self.nonCalculatedIntegralOutput = self.fixUnicodeText(printing.pretty(self.nonCalculatedIntegral))
            if (type(self.resultIntegralSimp) != unicode):
                self.resultOutput = self.fixUnicodeText(printing.pretty(self.resultIntegralSimp))
        elif self.outputResult == outputType['typesetting']:
            if (type(self.nonCalculatedIntegral) != unicode):
                self.nonCalculatedIntegralOutput = self.fixUnicodeText(printing.pretty(self.nonCalculatedIntegral))
            if (type(self.resultIntegralSimp) != unicode):
                self.resultOutput = self.fixUnicodeText(printing.pretty(self.resultIntegralSimp))
        elif self.outputResult == outputType['latex']:
            if (type(self.nonCalculatedIntegral) != unicode):
                self.nonCalculatedIntegralOutput = latex(self.nonCalculatedIntegral)
            if (type(self.resultIntegralSimp) != unicode):
                self.resultOutput = latex(self.resultIntegralSimp)
        elif self.outputResult == outputType['mathml']:
            if (type(self.nonCalculatedIntegral) != unicode):
                self.nonCalculatedIntegralOutput = mathml(self.nonCalculatedIntegral)
            if (type(self.resultIntegralSimp) != unicode):
                self.resultOutput = mathml(self.resultIntegralSimp)
        elif self.outputResult == outputType['c']:
            if (type(self.nonCalculatedIntegral) != unicode):
                self.nonCalculatedIntegralOutput = self.nonCalculatedIntegral
            if (type(self.resultIntegralSimp) != unicode):
                self.resultOutput = ccode(self.resultIntegralSimp)
        elif self.outputResult == outputType['fortran']:
            if (type(self.nonCalculatedIntegral) != unicode):
                self.nonCalculatedIntegralOutput = self.nonCalculatedIntegral
            if (type(self.resultIntegralSimp) != unicode):
                self.resultOutput = fcode(self.resultIntegralSimp)
        self.resultEdit.clear()
        if self.showTime:
            if self.timeIntegral > 0.0:
                self.resultEdit.setTextColor(QtGui.QColor("darkGreen"))
                self.resultEdit.append(u"Calculated after %f seconds :" % self.timeIntegral)
                self.resultEdit.setTextColor(QtGui.QColor("black"))
        if self.showIntegral:
            self.resultEdit.setTextColor(QtGui.QColor("Blue"))
            self.resultEdit.append(u"%s\n=" % self.nonCalculatedIntegralOutput)
            self.resultEdit.setTextColor(QtGui.QColor("black"))
        if (type(self.resultIntegralSimp) != unicode):
            self.resultEdit.setTextColor(QtGui.QColor("black"))
        else:
            self.resultEdit.setTextColor(QtGui.QColor("Red"))
        self.resultEdit.append(u"%s" % unicode(self.resultOutput))

    def fixMPMathText(self, str):
        str = str.replace("oo","inf")
        str = str.replace("E","e")
        str = str.replace("I","j")
        str = str.replace("GoldenRatio","phi")
        return str

    def fixMPMathOutput(self, str):
        str = str.replace("j","i")
        return str

    def fixUnicodeText(self, str):
        str = str.replace(u"ℯ","e")
        str = str.replace(u"ⅈ","i")
        return str


if __name__ == "__main__":
    #This function means this was run directly, not called from another python file.
    app = QtGui.QApplication(sys.argv)
    app.setOrganizationDomain("RobertoColistete.net")
    app.setApplicationName("Integral")

    from platform import python_version
    from sympy import *
    from sympy import __version__
    from sympy.integrals import *
    from sympy.mpmath import *
    from sympy.interactive.printing import init_printing
    from sympy.printing.mathml import mathml

    myapp = IntegralWindow()
    myapp.show()
    sys.exit(app.exec_())

