/*
 * GStreamer
 * Copyright 2007 Danilo Freire <danilo@embedded.ufcg.edu.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gst/gst.h>

#include "gstradiobin.h"
#include "radio_calls.h"

#define DEFAULT_DEVICE "/dev/radio0"
#define DEFAULT_MIXER  "/dev/mixer"
#define DEFAULT_SIGNAL_STRENGTH 0
#define DEFAULT_VOLUME 50

GST_DEBUG_CATEGORY_STATIC (gst_radiobin_debug);
#define GST_CAT_DEFAULT gst_radiobin_debug

/* Filter signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  ARG_0,
  ARG_VOLUME,
  ARG_DEVICE,
  ARG_FREQUENCY,
  ARG_MIXER,
  ARG_SIGNAL_STRENGTH
};

static gboolean
gst_radiobin_interface_supported (GstImplementsInterface *iface,
				                   GType iface_type)
{
  /*g_return_val_if_fail (iface_type == GST_TYPE_TUNER, FALSE);*/

  /* for the sake of this example, we'll always support it. However, normally,
   * you would check whether the device you've opened supports tuning. */
  return TRUE;
}

static void
gst_radiobin_implements_interface_init (GstImplementsInterfaceClass *iface)
{
  iface->supported = gst_radiobin_interface_supported;
}

//static void
//gst_radiobin_tuner_set_frequency(GstTuner        *tuner,
//	     						  GstTunerChannel *channel,
//	     						  gulong           frequency)
//{
//	
//}
//
//static gulong
//gst_radiobin_tuner_get_frequency(GstTuner        *tuner,
//	     						  GstTunerChannel *channel)
//{
//	return 0.0;
//}
//
//static gint
//gst_radiobin_signal_strength(GstTuner        *tuner,
//	     					  GstTunerChannel *channel)
//{
//	return 0;
//}
//
//static const GList *
//gst_radiobin_tuner_list_channels (GstTuner *tuner)
//{
//  GstRadioBin *filter = GST_RADIOBIN (tuner);
//
//  return filter->channels;
//}
//
//static GstTunerChannel *
//gst_radiobin_tuner_get_channel (GstTuner *tuner)
//{
//  GstRadioBin *filter = GST_RADIOBIN (tuner);
//
//  return g_list_nth_data (filter->channels,
//			  filter->active_input);
//}
//
//static void
//gst_radiobin_tuner_set_channel (GstTuner        *tuner,
//				 				 GstTunerChannel *channel)
//{
//  GstRadioBin *filter = GST_RADIOBIN (tuner);
//
//  filter->active_input = g_list_index (filter->channels, channel);
//  g_assert (filter->active_input >= 0);
//}
//
//static const GList *   
//gst_radiobin_list_norms (GstTuner * mixer)
//{
//	return 0;
//}
//
//static void
//gst_radiobin_set_norm_and_notify (GstTuner * mixer,
//								  GstTunerNorm * norm)  
//{
//	
//}
//
//static GstTunerNorm *
//gst_radiobin_get_norm (GstTuner * mixer)
//{
//	return 0;
//}
//
//static void
//gst_radiobin_tuner_interface_init (GstTunerClass *iface)
//{
//  iface->list_channels 	 = gst_radiobin_tuner_list_channels;
//  iface->get_channel   	 = gst_radiobin_tuner_get_channel;
//  iface->set_channel   	 = gst_radiobin_tuner_set_channel;
//  iface->set_frequency 	 = gst_radiobin_tuner_set_frequency;
//  iface->get_frequency   = gst_radiobin_tuner_get_frequency;
//  iface->signal_strength = gst_radiobin_signal_strength;
//  iface->list_norms 	 = gst_radiobin_list_norms;                     
//  iface->set_norm 	 	 = gst_radiobin_set_norm_and_notify;              
//  iface->get_norm 		 = gst_radiobin_get_norm;
//}


void
gst_radiobin_interfaces(GType type)
{
	  static const GInterfaceInfo implements_interface_info = {
	      (GInterfaceInitFunc) gst_radiobin_implements_interface_init,
	      NULL,
	      NULL,
	    };

//	    static const GInterfaceInfo tuner_interface_info = {
//	      (GInterfaceInitFunc) gst_radiobin_tuner_interface_init,
//	      NULL,
//	      NULL,
//	    };

	    g_type_add_interface_static (type,
					 GST_TYPE_IMPLEMENTS_INTERFACE,
					 &implements_interface_info);

//	    g_type_add_interface_static (type,
//					 GST_TYPE_TUNER,
//					 &tuner_interface_info);
}

GST_BOILERPLATE_FULL (GstRadioBin, gst_radiobin, GstElement, GST_TYPE_ELEMENT, gst_radiobin_interfaces );

static void gst_radiobin_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_radiobin_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);
static void gst_radiobin_finalize(GstRadioBin * radiobin);
static void gst_radiobin_dispose(GObject * object);
static GstStateChangeReturn gst_radiobin_change_state(GstElement * element,
								  GstStateChange transition);

static void
gst_radiobin_base_init (gpointer gclass)
{
  static GstElementDetails element_details = {
    "RadioBin",
    "Bin/Audio",
    "RadioBin Element",
    "Danilo Freire <danilo@embedded.ufcg.edu.br>"
  };
  GstElementClass *element_class = GST_ELEMENT_CLASS (gclass);

  gst_element_class_set_details (element_class, &element_details);
}

/* initialize the plugin's class */
static void
gst_radiobin_class_init (GstRadioBinClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;

  gobject_class->set_property = gst_radiobin_set_property;
  gobject_class->get_property = gst_radiobin_get_property;

  g_object_class_install_property (gobject_class, ARG_DEVICE,
        g_param_spec_string ("device", "Radio device Location",
					   "Radio device Location",
		               DEFAULT_DEVICE, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, ARG_MIXER,
        g_param_spec_string ("mixer", "Mixer device Location",
					   "Mixer device Location",
		               DEFAULT_MIXER, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, ARG_VOLUME,
		g_param_spec_int ("volume", "Volume (%)", "Volume Value (0-100%)",
		  0, 100, DEFAULT_VOLUME, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, ARG_FREQUENCY,
		g_param_spec_uint ("frequency", "Frequency (Hz)", "Frequency Value",
		  0, 107000, 93100, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, ARG_SIGNAL_STRENGTH,
		g_param_spec_int ("signal-strength", "Signal Strength", "The Signal Strenght",
		G_MININT, G_MAXINT, DEFAULT_SIGNAL_STRENGTH, G_PARAM_READABLE));

    /* set several parent class virtual functions */
  gobject_class->dispose = gst_radiobin_dispose;
  gobject_class->finalize = (GObjectFinalizeFunc) gst_radiobin_finalize;

  gstelement_class->change_state = GST_DEBUG_FUNCPTR (gst_radiobin_change_state);

}

/* initialize the new element
 * instantiate pads and add them to element
 * set functions
 * initialize structure
 */
static void
gst_radiobin_init (GstRadioBin * filter,
    GstRadioBinClass * gclass)
{
 // GstElementClass *klass = GST_ELEMENT_GET_CLASS (filter);
  filter->vol = DEFAULT_VOLUME;
  filter->freq = 93100;
  filter->radio_dev = g_strdup(DEFAULT_DEVICE);
  filter->mixer_dev = g_strdup(DEFAULT_MIXER);
  filter->silent = FALSE;
}

static void
gst_radiobin_dispose(GObject * object)
{
  GstRadioBin *radiobin = GST_RADIOBIN (object);

  gst_radio_close(radiobin);

  G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
gst_radiobin_finalize(GstRadioBin * radiobin)
{
  g_free(radiobin->radio_dev);
  g_free(radiobin->mixer_dev);
  G_OBJECT_CLASS (parent_class)->finalize ((GObject *) (radiobin));
}

static GstStateChangeReturn
gst_radiobin_change_state(GstElement * element,
					   GstStateChange transition)
{
  GstRadioBin *radiobin;
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;

  radiobin = GST_RADIOBIN (element);
  switch (transition)
  {
	case GST_STATE_CHANGE_NULL_TO_READY:
		/*open radio device*/
		gst_radio_open(radiobin);
		break;
	case GST_STATE_CHANGE_READY_TO_PAUSED:
		break;
	case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
		gst_radio_setfreq(radiobin, radiobin->freq);
		gst_radio_unmute(radiobin);
		gst_radio_setvolume(radiobin, radiobin->vol);
		break;
	case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
		/*mute radio*/
		gst_radio_mute(radiobin);
		/* quick thought: put a little sleep here*/
		gst_radio_setvolume(radiobin, 0);
		break;
	case GST_STATE_CHANGE_PAUSED_TO_READY:
		break;
	case GST_STATE_CHANGE_READY_TO_NULL:
		/*close radio*/
		gst_radio_close(radiobin);
		break;
	default:
		break;
  }

  return ret;
}

static void
gst_radiobin_set_property (GObject * object, guint prop_id,
						   const GValue * value, GParamSpec * pspec)
{
  GstRadioBin *radiobin = GST_RADIOBIN (object);

  switch (prop_id)
  {
     case ARG_VOLUME:
       gst_radio_setvolume(radiobin, g_value_get_int (value));
       break;
     case ARG_DEVICE:
       radiobin->radio_dev = g_strdup( (gchar*) g_value_get_string(value));
       break;
     case ARG_MIXER:
	   radiobin->mixer_dev = g_strdup( (gchar*) g_value_get_string(value));
	   break;
     case ARG_FREQUENCY:
       radiobin->freq = g_value_get_uint (value);
       gst_radio_setfreq(radiobin, g_value_get_uint(value));
       break;
      default:
       G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
       break;
  }
}

static void
gst_radiobin_get_property (GObject * object, guint prop_id,
						   GValue * value, GParamSpec * pspec)
{
  GstRadioBin *radiobin = GST_RADIOBIN (object);

  switch (prop_id)
  {
	  case ARG_VOLUME:
		g_value_set_int(value, radiobin->vol);
	    break;
	  case ARG_DEVICE:
		g_value_set_string(value, radiobin->radio_dev);
	    break;
	  case ARG_MIXER:
	    g_value_set_string(value, radiobin->mixer_dev);
	    break;
	  case ARG_FREQUENCY:
		g_value_set_uint(value, radiobin->freq);
	    break;
	  case ARG_SIGNAL_STRENGTH:
		g_value_set_int(value, gst_radio_get_signal_strength(radiobin));
	    break;
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
  }
}

/* entry point to initialize the plug-in
 * initialize the plug-in itself
 * register the element factories and pad templates
 * register the features
 *
 * exchange the string 'plugin' with your elemnt name
 */
static gboolean
plugin_init (GstPlugin * plugin)
{
  /* exchange the strings 'plugin' and 'Template plugin' with your
   * plugin name and description */
  GST_DEBUG_CATEGORY_INIT (gst_radiobin_debug, "radiobin",
      0, "RadioBin Plugin");

  return gst_element_register (plugin, "radiobin",
      GST_RANK_NONE, GST_TYPE_RADIOBIN);
}

/* this is the structure that gstreamer looks for to register plugins
 *
 * exchange the strings 'plugin' and 'Template plugin' with you plugin name and
 * description
 */
GST_PLUGIN_DEFINE (GST_VERSION_MAJOR,
    GST_VERSION_MINOR,
    "radiobin",
    "RadioBin Plugin",
    plugin_init, VERSION, "LGPL", "GStreamer", "http://gstreamer.net/")
