/*
  GPSJinni - show raw data from the GPS subsystem.
  Copyright (C) 2009  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "TimeLine.h"

#include <cmath>
#include <cstdlib>
#include <iomanip>
#include <sstream>

#include <Lum/Base/String.h>

#include <iostream>

static double borderOffset = 0.2;

static std::wstring DoubleToWString(double value, int precision)
{
  if (!isnan(value)) {
    std::stringstream buffer;

    buffer.imbue(std::locale(""));
    buffer << std::fixed;
    if (precision>0) {
      buffer << std::showpoint;
    }
    buffer.precision(precision);

    buffer << value;

    return Lum::Base::StringToWString(buffer.str());
  }
  else {
    return L"";
  }
}

TimeLine::TimeLine()
 : font(Lum::OS::display->GetFont(Lum::OS::Display::fontScaleFootnote)),
   verticalScaleWidth(0),
   digitWidth(0),
   bottom(0),
   top(0)
{
  SetBackground(Lum::OS::display->GetFill(Lum::OS::Display::graphBackgroundFillIndex));
}

void TimeLine::UpdateDimensions()
{
  if (top==0 && bottom==0) {
    hAdjustment->SetInvalid();
    vAdjustment->SetInvalid();
  }
  else {
    hAdjustment->SetDimension(width,std::max(width,(size_t)GetSecondsCovered()));
    hAdjustment->SetStepSize(60); // one minute
  }
}

unsigned long TimeLine::GetSecondsCovered() const
{
  assert(!data.empty());

  return data.rbegin()->first.GetTime()-data.begin()->first.GetTime()+1;
}

bool TimeLine::SetModel(Lum::Base::Model* model)
{
  this->model=dynamic_cast<Lum::Model::DoubleDataStream*>(model);

  Control::SetModel(this->model);

  return this->model.Valid();
}

void TimeLine::CalcSize()
{
  for (size_t i=0; i<10; i++) {
    digitWidth=std::max(digitWidth,font->StringWidth(Lum::Base::NumberToWString(i)));
  }

  width=100;
  height=100;
  minWidth=width;
  minHeight=height;

  Scrollable::CalcSize();
}

int TimeLine::TransformValue(double y)
{
  return (int)floor(height-(y-bottom)*(height-1)/(top-bottom));
}

void TimeLine::CalculateVerticalScale()
{
  if (lastDrawnTop==top && lastDrawnBottom==bottom) {
    std::cout << "Top/Bottom unchanged => nothing to calculate" << std::endl;
    return;
  }

  size_t effFontHeight;
  bool   timesFive=true;

  if (font->height/4>2) {
    effFontHeight=font->height+(font->height/4);
  }
  else {
    effFontHeight=font->height+2;
  }

  verticalScaleStep=pow(10,lrint(log10(std::abs(bottom))-1));
  std::cout << "Vertical scale step: " << verticalScaleStep << std::endl;
  if (verticalScaleStep<=1) {
    std::cout << "(corrected) Vertical scale step: " << verticalScaleStep << std::endl;
    verticalScaleStep=1;
  }

  while ((TransformValue(bottom)-TransformValue(bottom+verticalScaleStep))<(int)effFontHeight) {
    if (timesFive) {
      verticalScaleStep=verticalScaleStep*5;
    }
    else {
      verticalScaleStep=verticalScaleStep*2;
    }
    timesFive=!timesFive;
  }

  std::cout << "Scaled vertical scale step: " << verticalScaleStep << std::endl;

  if (log10(verticalScaleStep)<0) {
    verticalScalePrecision=(size_t)ceil(std::abs(log10(verticalScaleStep)));
  }
  else {
    verticalScalePrecision=0;
  }

  std::cout << "Vertical scale precision: " << verticalScalePrecision << std::endl;

  verticalScaleWidth=0;
  for (double i=floor((bottom/verticalScaleStep))*verticalScaleStep; i<=top; i+=verticalScaleStep) {
    Lum::OS::FontExtent extent;

    font->StringExtent(DoubleToWString(i,verticalScalePrecision),extent,Lum::OS::Font::normal);

    verticalScaleWidth=std::max(verticalScaleWidth,extent.width);
  }

  std::cout << "Vertical scale width: " << verticalScaleWidth << std::endl;
}

void TimeLine::PrintData(Lum::OS::DrawInfo* draw,
                        const Lum::Base::SystemTime& start,
                        const Lum::Base::SystemTime& end)
{
  int                                                xPos=this->x;
  std::map<Lum::Base::SystemTime,Entry>::const_iterator iter;
  bool                                               pointsDrawn=false; // We have drawn at least one point
  bool                                               pastEnd=false;     // We have beenmoved past the visible end
  bool                                               finished=false;    // We have drawn one point pastthe visible end and can now finish
  int                                                px=0,py=0;         // Coordinates of previous point
  long                                               seconds;
  double                                             lastValue=0;

  iter=data.lower_bound(start);

  while (iter!=data.end() && !finished) {
    if (iter->first>end) {
      pastEnd=true;
    }

    seconds=iter->first.GetTime()-start.GetTime();

    //
    // Draw value points
    //

    if (!isnan(iter->second.value)) {
      draw->PushPen(1,Lum::OS::DrawInfo::penNormal);
      draw->PushForeground(Lum::OS::Display::blackColor);

      int cx,cy; // Coordinates of current point

      cx=xPos+seconds;
      cy=this->y+TransformValue(iter->second.value);

      draw->DrawPoint(cx,cy);

      // Draw line between previous and current weight point
      if (pointsDrawn) {
        int lx1, ly1, lx2, ly2;

        // Project points outside the diagram on the axis of the diagram
        // to avoid overflow errors and similar...
        lx1=px;
        ly1=py;
        lx2=cx;
        ly2=cy;

        if (!(lx1<this->x && lx2<this->x)) {
          // Point outside diagram: Interpolate position on left diagram border
          if (lx1<this->x) {
            lx1=this->x;
            ly1=((lx1-px)*(cy-py))/(cx-px)+py;
          }

          // Point outside diagram: Interpolate position on right diagram border
          if (lx2>this->x+(int)this->width-1) {
            lx2=this->x+this->width-1;
            ly2=((lx2-px)*(cy-py))/(cx-px)+py;
          }

          draw->DrawLine(lx1,ly1,lx2,ly2);
        }
      }

      draw->PopForeground();
      draw->PopPen();

      pointsDrawn=true;
      px=cx;
      py=cy;
      lastValue=iter->second.value;
    }
    else {
      pointsDrawn=false;
    }

    if (pastEnd) {
      finished=true;
    }

    ++iter;
  }
}

void TimeLine::PrintHorizontalScale(Lum::OS::DrawInfo* draw,
                                   const Lum::Base::SystemTime& start)
{
  std::wstring          tmp;
  Lum::Base::SystemTime time;

  draw->PushForeground(Lum::OS::Display::graphScaleColor);
  draw->PushFont(font);

  draw->PushClip(this->x+verticalScaleWidth,this->y,this->width-verticalScaleWidth,this->height);

  time=start;
  int xPos=this->x;
  while (xPos<this->x+(int)width) {
    if (time.GetTime()%60==0) {
      Lum::Base::DateTime dt;

      time.GetLocalDateTime(dt);
      tmp=Lum::Base::NumberToWString(dt.hour);
      tmp+=L":";
      if (dt.minute<10) {
        tmp+=L"0";
      }
      tmp+=Lum::Base::NumberToWString(dt.minute);
      // TODO: Correct by offset
      draw->DrawString(xPos,this->y+height-font->height+font->ascent,tmp);
    }

    time.Add(1,0);
    xPos+=1;
  }

  draw->PopClip();

  draw->PopFont();
  draw->PopForeground();
}

/**
  Printing vertial value scale
 */
void TimeLine::PrintVerticalScale(Lum::OS::DrawInfo* draw)
{
  draw->PushForeground(Lum::OS::Display::graphScaleColor);
  draw->PushFont(font);

  for (double i=floor((bottom/verticalScaleStep))*verticalScaleStep;
       i<=top;
       i+=verticalScaleStep) {
    std::wstring tmp=DoubleToWString(i,verticalScalePrecision);
    draw->DrawString(this->x+verticalScaleWidth-font->StringWidth(tmp),
                     TransformValue(i)-font->height/2+this->y+font->ascent,
                     tmp);
  }

  draw->PopFont();
  draw->PopForeground();
}

void TimeLine::Draw(int x, int y, size_t w, size_t h)
{
  Lum::Scrollable::Draw(x,y,w,h);

  if (!OIntersect(x,y,w,h)) {
    return;
  }

  /* --- */

  if (top==0 && bottom==0) {
    return;
  }

  Lum::OS::DrawInfo     *draw=GetDrawInfo();
  Lum::Base::SystemTime start=data.begin()->first;
  Lum::Base::SystemTime end;

  start.Add(hAdjustment->GetTop()-1,0);
  end=start;
  end.Add(width,0);

  draw->PushClip(this->x,this->y,width,height);
  draw->PushClip(x,y,w,h);

  std::cout << "Calculate vertical scale..." << std::endl;
  CalculateVerticalScale();
  std::cout << "Printing data..." << std::endl;
  PrintData(draw,start,end);
  std::cout << "Printing horizontal scale..." << std::endl;
  PrintHorizontalScale(draw,start);
  std::cout << "Printing vertical scale..." << std::endl;
  PrintVerticalScale(draw);
  std::cout << "done." << std::endl;

  draw->PopClip();
  draw->PopClip();

  lastDrawnTop=top;
  lastDrawnBottom=bottom;
}

void TimeLine::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==hAdjustment->GetTopModel()) {
    Redraw();
  }
  else if (model==this->model && !this->model->IsNull()) {
    Lum::Base::SystemTime now;
    Entry                 entry;

    entry.value=this->model->Get(0);

    std::cout << "New value: " << entry.value << std::endl;

    data[now]=entry;

    if (isnan(entry.value)) {
      std::cout << "New value is not a number, skipping" << std::endl;
      return;
    }

    if (top==0 && bottom==0) {
      if (entry.value>=0) {
        top=entry.value*(1.0+borderOffset);
        bottom=entry.value*(1.0-borderOffset);
      }
      else {
        top=entry.value*(1.0-borderOffset);
        bottom=entry.value*(1.0+borderOffset);
      }
    }
    else {
      if (entry.value>=0) {
        top=std::max(top,entry.value*(1.0+borderOffset));
        bottom=std::min(bottom,entry.value*(1.0-borderOffset));
      }
      else {
        top=std::max(top,entry.value*(1.0-borderOffset));
        bottom=std::min(bottom,entry.value*(1.0+borderOffset));
      }
    }

    std::cout << "Top: " << top << " bottom: " << bottom << std::endl;

    Redraw();
  }

  Scrollable::Resync(model,msg);
}

TimeLine* TimeLine::Create(bool horizontalFlex, bool verticalFlex)
{
  TimeLine *t;

  t=new TimeLine();
  t->SetFlex(horizontalFlex,verticalFlex);

  return t;
}

TimeLine* TimeLine::Create(Lum::Base::Model* model, bool horizontalFlex, bool verticalFlex)
{
  TimeLine *t;

  t=new TimeLine();
  t->SetFlex(horizontalFlex,verticalFlex);
  t->SetModel(model);

  return t;
}


