/*
 * Copyright (C) 2005 Luca De Cicco <ldecicco@gmail.com>
 * Hildonized by Luca De Cicco
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */

#include <time.h>
#include <stdlib.h>
#include <string.h>

#include "sudoku_worksheet.h"
#include "save_game.h"
#include "main_window.h"

extern guint state;
	
gchar *timestamp_to_string()
{
        time_t *t;
        struct tm *tm;
        gchar *date;

        date = (gchar *)g_malloc(26);
        t  = (time_t *)g_malloc(sizeof(time_t));
        tm = (struct tm *)g_malloc(sizeof(struct tm));
        time(t);
        tm = localtime(t);

        snprintf(date,26,"%.4d%.2d%.2d%.2d%.2d%.2d",tm->tm_year+1900,
                                                           tm->tm_mon+1,
                                                           tm->tm_mday,
                                                           tm->tm_hour,
                                                           tm->tm_min,
                                                           tm->tm_sec);
        return(date);
}

/*

 <worksheet name="something_timestamp" difficulty="hard">
    <cell value="0" idx="0">
    	<guess value="3" />
        <guess value="4" />
    </cell>
    <cell value="2" idx="2">
    </cell>
     ...
 </worksheet>
 */
gchar *worksheet_to_xml (struct sudoku_worksheet *ws)
{
	if(!ws)
		return NULL;
	
	int i;
	gchar *ts_string = timestamp_to_string();
	GString *xml = g_string_new("");
	gchar *str;
	g_string_append_printf(xml, "<worksheet name=\"%s\">\n",ts_string);
	for (i=0 ; i<81 ; i++)
	{

		GSList *iter;
		str = g_markup_printf_escaped ("\t<cell value=\"%d\" idx=\"%d\" fixed=\"%d\">\n", ws->s_cells[i]->cell,i, ws->s_cells[i]->fixed);
		g_string_append (xml, str);
		for(iter = ws->s_cells[i]->guesses; iter ; iter=iter->next)
		{
			int value = GPOINTER_TO_INT(iter->data);
		  	str = g_markup_printf_escaped ("\t\t<guess value=\"%d\" />\n",value);
			g_string_append (xml, str);
		}
		g_string_append_printf (xml,"\t</cell>\n");
	
	}
	g_string_append_printf(xml, "</worksheet>\n");
	
	g_free (ts_string);
	return (g_string_free(xml, FALSE));
}


static int in_cell_idx = -1;
void
read_start_element ( GMarkupParseContext *context,
                const gchar *e_n,
                const gchar **attr_names,
                const gchar **attr_values,
                gpointer user_data,
                GError **error)
{
	struct sudoku_worksheet *ws = (struct sudoku_worksheet *)user_data;
	
	guint cell_value=0;
	gint guess_value;
	gboolean fixed=0;
	if (!strcmp (e_n,"worksheet"))
	{
	//	printf("Root element found\n");
	}else if (!strcmp (e_n,"cell")){
		int i=0;
		for (i=0 ; attr_names[i] ; i++)
		{
			if(!strcmp(attr_names[i],"idx")){
				
				in_cell_idx = atoi (attr_values[i]);
			}else if(!strcmp(attr_names[i],"value")){
				cell_value = (guint)atoi(attr_values[i]);
				if(cell_value>9 || cell_value<0){
					cell_value=0;
					g_warning("Invalid Cell value");
				}
			}else if(!strcmp(attr_names[i],"fixed")){
				fixed = (guint)atoi(attr_values[i]);
			}
		}	
		ws->s_cells[in_cell_idx]->cell = cell_value;		
		ws->s_cells[in_cell_idx]->fixed = fixed;
	}else if (!strcmp (e_n, "guess"))
	{
		if(!strcmp(attr_names[0],"value") && in_cell_idx>=0){
			guess_value = atoi(attr_values[0]);
			guess_add(&(ws->s_cells[in_cell_idx]->guesses), guess_value);
		}
	}
}

void
read_end_element ( GMarkupParseContext *context,
                const gchar *e_n,
                const gchar **attr_names,
                const gchar **attr_values,
                gpointer user_data,
                GError **error)
{
		if(!strcmp(e_n,"cell")){
			in_cell_idx = -1;	
		}
}

void fill_worksheet_from_xmlfile (gchar *fname, struct sudoku_worksheet *ws)
{
	
	FILE *fp;
	gint i;
	
	GMarkupParser parser = {read_start_element,
                            read_end_element,NULL, NULL, NULL};
							
	GMarkupParseContext *pc = g_markup_parse_context_new(&parser, 0, ws, NULL);

	fp = fopen (fname, "r");
	//printf("ws->actions: %p\n", ws->actions);
	if (!fp){
		//printf("Can't open file: %s\n",fname);		
		return;
	}else{
		//printf("File exists and is open\n");		
	}
	/* Clean the worksheet */
	g_slist_free(ws->actions);
	ws->actions = NULL;
	state = 0;
	for (i = 0 ; i<81 ; i++ )
	{
		ws->s_cells[i]->cell = 0;
		/* Initialise all guess lists */
		
		sudoku_worksheet_flush_guesses(ws, i);
	}
	//printf("ws->actions after: %p\n", ws->actions);
	while (! feof (fp))
    {
		char buf[1024];
		int n;
		GError *err = NULL;
		n = fread (buf, 1, 1024, fp);
        if (g_markup_parse_context_parse (pc, buf, n, &err) == FALSE)
       	{
   			g_warning (err->message);
        	g_error_free (err);
        	g_markup_parse_context_free (pc);
			fclose(fp);
        	return ;
		}
	}
		
	fclose(fp);
	draw_all_squares();
}
