/*
 * This file is part of eostimesync1
 *
 * Author: Marcell Lengyel <marcell@maemo.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <gtk/gtk.h>
#include <glib-object.h>
#include <libhildondesktop/libhildondesktop.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-caption.h>
#include <hildon/hildon-note.h>
#include <stdio.h>	/* popen */
#include <string.h>	/* strlen... */
#include <gconf/gconf-client.h>

#include "eostimesync-plugin.h"

struct _EosTimeSyncPluginPrivate {
  GtkWidget             *buttonbar;
  GtkWidget             *ntp_button;
  GtkWidget 		*usb_button;
  GtkWidget		*camera_button;
  gboolean		ntp_enabled;
  gboolean		usb_enabled;
  gboolean		camera_enabled;
  gchar			*ntp_server;
  GConfClient		*gconf_client;
};

#define EOSTIMESYNC_GCONF_ROOT "/apps/eostimesync"
#define EOSTIMESYNC_GCONF_NTP_SERVER "/apps/eostimesync/ntp_server"
#define EOSTIMESYNC_GCONF_NTP_ENABLED "/apps/eostimesync/ntp_enabled"
#define EOSTIMESYNC_GCONF_USB_ENABLED "/apps/eostimesync/usb_enabled"
#define EOSTIMESYNC_GCONF_CAMERA_ENABLED "/apps/eostimesync/camera_enabled"


HD_DEFINE_PLUGIN (EosTimeSyncPlugin, eos_time_sync_plugin, HILDON_DESKTOP_TYPE_HOME_ITEM);

/* forward declarations */
static gboolean ntp_clicked(GtkWidget *widget, gpointer data);
static gboolean usb_clicked(GtkWidget *widget, gpointer data);
static gboolean eos_clicked(GtkWidget *widget, gpointer data);
static int usb_mode_get();
static int usb_mode_toggle();
static void eos_time_sync_save_preferences (EosTimeSyncPlugin *home_plugin);

/* the real thing */

/**
 * Displays the Applet Settings dialog
 *
 * @param home_plugin the Hildon Desktop home plugin
 */
static void
eos_time_sync_settings_dialog (EosTimeSyncPlugin *home_plugin)
{
	EosTimeSyncPluginPrivate *priv  = home_plugin->priv;
	GtkWidget *dialog, *settings_notebook;
	GtkWidget *ntp_vbox, *ntp_server_entry, *ntp_server_caption, *ntp_enabled;
	GtkWidget *usb_vbox, *usb_enabled;
	GtkWidget *camera_vbox, *camera_enabled;
	GtkSizeGroup *group;
	gchar *str = NULL;
	gint retval;
	
#ifdef __i386__	
	printf("eos_time_sync_settings_dialog\n");
#endif

    	dialog = gtk_dialog_new_with_buttons ("Eos Time Sync Settings",
        	                                NULL,
                	                        GTK_DIALOG_DESTROY_WITH_PARENT |
                        	                GTK_DIALOG_NO_SEPARATOR,
                                	        GTK_STOCK_OK,
                                        	GTK_RESPONSE_OK,
						GTK_STOCK_CANCEL,
						GTK_RESPONSE_CANCEL,
                                        	NULL);
	g_return_if_fail (dialog);
	
	settings_notebook=gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(settings_notebook),GTK_POS_TOP);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(settings_notebook), FALSE);

	/* -------------------------------------------------------------------------------*
	 * NTP TAB
	 * -------------------------------------------------------------------------------*/
	group = GTK_SIZE_GROUP (gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL));
	
	ntp_vbox = gtk_vbox_new(FALSE, 2);

	ntp_enabled = gtk_check_button_new_with_label ("Time Sync button visible");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(ntp_enabled), (gboolean)priv->ntp_enabled);
	gtk_container_add (GTK_CONTAINER(ntp_vbox),
			   ntp_enabled);
	
	ntp_server_entry = gtk_entry_new();
#ifdef __i386__
          printf("eos_time_sync_settings_dialog - ntp_server: %s\n", priv->ntp_server);
#endif
	str = g_strdup(priv->ntp_server);
	gtk_entry_set_text(GTK_ENTRY(ntp_server_entry),str);
	
	ntp_server_caption = hildon_caption_new (group,
		                                 "NTP server", 
					         ntp_server_entry,
						 NULL,
						 HILDON_CAPTION_OPTIONAL);
	gtk_container_add (GTK_CONTAINER(ntp_vbox),
			   ntp_server_caption);
	/* -------------------------------------------------------------------------------*
	 * USB TAB
	 * -------------------------------------------------------------------------------*/
	usb_vbox = gtk_vbox_new(FALSE, 2);
	usb_enabled = gtk_check_button_new_with_label ("USB Mode Change button visible");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(usb_enabled), (gboolean)priv->usb_enabled);
	gtk_container_add (GTK_CONTAINER(usb_vbox),
			   usb_enabled);
	
	/* -------------------------------------------------------------------------------*
	 * CAMERA TAB
	 * -------------------------------------------------------------------------------*/
	camera_vbox = gtk_vbox_new(FALSE, 2);
	camera_enabled = gtk_check_button_new_with_label ("Camera Sync button visible");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(camera_enabled), (gboolean)priv->camera_enabled);
	gtk_container_add (GTK_CONTAINER(camera_vbox),
			   camera_enabled);
	
	/* -------------------------------------------------------------------------------*
	 * Notebook tabs
	 * -------------------------------------------------------------------------------*/
	gtk_notebook_append_page(GTK_NOTEBOOK(settings_notebook),
				ntp_vbox,
				gtk_label_new("NTP"));
	gtk_notebook_append_page(GTK_NOTEBOOK(settings_notebook),
				usb_vbox,
				gtk_label_new("USB"));
	gtk_notebook_append_page(GTK_NOTEBOOK(settings_notebook),
				camera_vbox,
				gtk_label_new("Camera"));
	
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox),
        	           settings_notebook);
	gtk_widget_show_all (dialog);

	retval = gtk_dialog_run (GTK_DIALOG(dialog));

	if (retval == GTK_RESPONSE_OK) {
	  if (str) {
	    g_free(str);
	  }
	  priv->ntp_server = g_strdup(gtk_entry_get_text (GTK_ENTRY (ntp_server_entry)));
#ifdef __i386__
          printf("eos_time_sync_settings_dialog - OK - ntp_server: %s\n", priv->ntp_server);
#endif
	  priv->ntp_enabled = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ntp_enabled));
	  priv->usb_enabled = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(usb_enabled));
	  priv->camera_enabled = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(camera_enabled));
	  /* Free the dialog (and it's children) */
	  gtk_widget_destroy(GTK_WIDGET(dialog));
	  /* save preferences (after closing the preferences dialog) */
	  eos_time_sync_save_preferences(home_plugin);
	  /* hide the disabled buttons */
	  if (priv->ntp_enabled) {
	    gtk_widget_show (priv->ntp_button);
	  } else {
	    gtk_widget_hide (priv->ntp_button);
	  }
	  if (priv->usb_enabled) {
	    gtk_widget_show (priv->usb_button);
	  } else {
	    gtk_widget_hide (priv->usb_button);
	  }
	  if (priv->camera_enabled) {
	    gtk_widget_show (priv->camera_button);
	  } else {
	    gtk_widget_hide (priv->camera_button);
	  }
	} else {
	  /* Free the dialog (and it's children) */
	  gtk_widget_destroy(GTK_WIDGET(dialog));
	}
}

static GtkWidget * 
eos_time_sync_settings(HildonDesktopHomeItem   *applet,
	               GtkWidget               *parent)
{
	EosTimeSyncPlugin *home_plugin = EOSTIMESYNC_PLUGIN(applet);
	GtkWidget *item;

#ifdef __i386__	
	printf("eos_time_sync_settings - enter\n");
#endif
	item = gtk_menu_item_new_with_label ("Eos Time Sync");

	g_signal_connect_swapped (G_OBJECT (item), "activate",
	                          G_CALLBACK (eos_time_sync_settings_dialog),
	 			                            home_plugin);

	return item;

}

static void
eos_time_sync_plugin_destroy (GtkObject *object)
{
  EosTimeSyncPlugin *home_plugin = EOSTIMESYNC_PLUGIN(object);
  EosTimeSyncPluginPrivate *priv = home_plugin->priv;

  if (priv->ntp_server) {
    g_free (priv->ntp_server);
    priv->ntp_server = NULL;
  }  
  if (priv->gconf_client)  {
    g_object_unref (priv->gconf_client);
    priv->gconf_client = NULL;
  }
#ifdef __i386__	
  printf("eos_time_sync_plugin_destroy - leave\n");
#endif
}

static void
eos_time_sync_load_preferences (EosTimeSyncPlugin *home_plugin)
{
  EosTimeSyncPluginPrivate *priv  = home_plugin->priv;
  GError                   *error = NULL;
  GtkWidget		   *note;

#ifdef __i386__
  printf("eos_time_sync_load_preferences - enter \n");
#endif
  g_return_if_fail (priv && priv->gconf_client);

  /*Add GConf node if absent*/
  if (!gconf_client_dir_exists(priv->gconf_client,
			       EOSTIMESYNC_GCONF_ROOT,
			       NULL)) {
      gconf_client_add_dir(priv->gconf_client,
		           EOSTIMESYNC_GCONF_ROOT,
		           GCONF_CLIENT_PRELOAD_NONE, 
		           NULL);
      priv->ntp_server = g_strdup("pool.ntp.org");
      priv->ntp_enabled = TRUE;
      priv->usb_enabled = TRUE;
      priv->camera_enabled = TRUE;
      eos_time_sync_save_preferences(home_plugin);
      return;
  }
  priv->ntp_server = gconf_client_get_string (priv->gconf_client,
                                              EOSTIMESYNC_GCONF_NTP_SERVER,
                                              &error);
#ifdef __i386__
  printf("eos_time_sync_load_preferences - ntp_server: %s\n", priv->ntp_server);
#endif
  if (error) goto error;
  
  priv->ntp_enabled = gconf_client_get_bool (priv->gconf_client,
                                             EOSTIMESYNC_GCONF_NTP_ENABLED,
                                             &error);
  if (error) goto error;
  
  priv->usb_enabled = gconf_client_get_bool (priv->gconf_client,
                                             EOSTIMESYNC_GCONF_USB_ENABLED,
                                             &error);
  if (error) goto error;
  
  priv->camera_enabled = gconf_client_get_bool (priv->gconf_client,
                                             EOSTIMESYNC_GCONF_CAMERA_ENABLED,
                                             &error);
  if (error) goto error;
  if ((priv->ntp_enabled || priv->usb_enabled || priv->camera_enabled) == FALSE) {
    /* Can't disable all buttons, warn the user and enable ntp */
    note=hildon_note_new_information_with_icon_name(NULL, "Can not disable all buttons!\nEnabling NTP.",  GTK_STOCK_DIALOG_WARNING);
    gtk_widget_show(note);
    gtk_dialog_run(GTK_DIALOG(note));
    gtk_widget_destroy(GTK_WIDGET(note));
    priv->ntp_enabled = TRUE;
  }
  
error:  
  if (error) {
    g_critical ("* Error when getting gconf node: %s", error->message);
    g_error_free (error);
    g_free (priv->ntp_server);
    priv->ntp_server = NULL;
  }
}

static void
eos_time_sync_save_preferences (EosTimeSyncPlugin *home_plugin)
{
  EosTimeSyncPluginPrivate *priv  = home_plugin->priv;
  GError                   *error = NULL;
  GtkWidget		   *note;

#ifdef __i386__
  printf("eos_time_sync_save_preferences - enter \n");
#endif
  g_return_if_fail (priv && priv->gconf_client);
  if ((priv->ntp_enabled || priv->usb_enabled || priv->camera_enabled) == FALSE) {
    /* Can't disable all buttons, warn the user and enable ntp */
    note=hildon_note_new_information_with_icon_name(NULL, "Can not disable all buttons!\nEnabling NTP.",  GTK_STOCK_DIALOG_WARNING);
    gtk_widget_show(note);
    gtk_dialog_run(GTK_DIALOG(note));
    gtk_widget_destroy(GTK_WIDGET(note));
    priv->ntp_enabled = TRUE;
  }
  /*Add GConf node if absent*/
  if (!gconf_client_dir_exists(priv->gconf_client,
			       EOSTIMESYNC_GCONF_ROOT,
			       NULL))
      gconf_client_add_dir(priv->gconf_client,
		           EOSTIMESYNC_GCONF_ROOT,
		           GCONF_CLIENT_PRELOAD_NONE, 
		           NULL);
#ifdef __i386__
  printf("eos_time_sync_save_preferences - ntp_server: %s\n", priv->ntp_server);
#endif
  if (priv->ntp_server)
      gconf_client_set_string (priv->gconf_client,
		               EOSTIMESYNC_GCONF_NTP_SERVER,
			       priv->ntp_server,
			       &error);
  if (error) goto save_error;
  gconf_client_set_bool (priv->gconf_client,
                         EOSTIMESYNC_GCONF_NTP_ENABLED,
		         priv->ntp_enabled,
		        &error);
  if (error) goto save_error;
  gconf_client_set_bool (priv->gconf_client,
                         EOSTIMESYNC_GCONF_USB_ENABLED,
			 priv->usb_enabled,
			 &error);
  if (error) goto save_error;
  gconf_client_set_bool (priv->gconf_client,
                         EOSTIMESYNC_GCONF_CAMERA_ENABLED,
			 priv->camera_enabled,
			 &error);
  if (error) goto save_error;
  

save_error:
  if (error) {
    g_critical ("* Error when setting gconf node: %s", error->message);
    g_error_free (error);
  }
  
}
		  
static void
eos_time_sync_plugin_init (EosTimeSyncPlugin *home_plugin)
{
  EosTimeSyncPluginPrivate *priv;
  GtkWidget *buttonbar, *ntp_button, *usb_button, *eos_button, *frame, *alignment;
  
  priv = home_plugin->priv =
	G_TYPE_INSTANCE_GET_PRIVATE ((home_plugin), 
			             EOS_TIME_SYNC_TYPE_PLUGIN, 
				     EosTimeSyncPluginPrivate);
  priv->gconf_client = gconf_client_get_default ();
  // load config
  eos_time_sync_load_preferences(home_plugin);
#ifdef __i386__
  printf("eos_time_sync_plugin_init: after load preferences\n");
#endif
  
  ntp_button = gtk_button_new ();
  gtk_button_set_image (GTK_BUTTON (ntp_button), 
                        gtk_image_new_from_file (g_strjoin("/", PIXMAP_PATH, "ntp_sync.png", NULL )));

   gtk_widget_set_size_request (ntp_button, 40, 40);
  priv->ntp_button = ntp_button;
  usb_button = gtk_button_new ();
  gtk_button_set_image (GTK_BUTTON (usb_button), 
                        gtk_image_new_from_file (g_strjoin("/", PIXMAP_PATH, "usb_otg.png", NULL )));

  gtk_widget_set_size_request (usb_button, 40, 40);
  priv->usb_button = usb_button;
  eos_button = gtk_button_new ();
  gtk_button_set_image (GTK_BUTTON (eos_button), 
                        gtk_image_new_from_file (g_strjoin("/", PIXMAP_PATH, "eos.png", NULL )));

  gtk_widget_set_size_request (eos_button, 40, 40);
  priv->camera_button = eos_button;

  buttonbar = gtk_hbutton_box_new();
  priv->buttonbar = buttonbar;
  
  gtk_container_add(GTK_CONTAINER (buttonbar), ntp_button);
  gtk_container_add(GTK_CONTAINER (buttonbar), usb_button);
  gtk_container_add(GTK_CONTAINER (buttonbar), eos_button);

  frame = gtk_frame_new (NULL);
  gtk_widget_set_name(frame, "osso-speeddial");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 0);

  alignment = gtk_alignment_new (0.5,
		                 0.5,
				 1.0,
				 1.0);
  gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 15, 15, 15, 15);
  gtk_container_add (GTK_CONTAINER (alignment), buttonbar);
  gtk_container_add (GTK_CONTAINER (frame), alignment);
  gtk_container_add (GTK_CONTAINER (home_plugin), frame); 

  gtk_widget_show_all (frame);
  
  if (! priv->ntp_enabled) gtk_widget_hide (ntp_button);
  if (! priv->usb_enabled) gtk_widget_hide (usb_button);
  if (! priv->camera_enabled) gtk_widget_hide (eos_button);

#ifdef __i386__	
  printf("eos_time_sync_plugin_init - ntp_server: %s\n", priv->ntp_server);
  printf("priv: %x\n", priv);
  printf("home_plugin->priv: %x\n", home_plugin->priv);
  printf("home_plugin: %x\n", home_plugin);
#endif
  g_signal_connect(G_OBJECT(ntp_button), "clicked",
		   G_CALLBACK(ntp_clicked), (gpointer)home_plugin);
  g_signal_connect(G_OBJECT(usb_button), "clicked",
		   G_CALLBACK(usb_clicked), (gpointer)home_plugin);
  g_signal_connect(G_OBJECT(eos_button), "clicked",
		   G_CALLBACK(eos_clicked), (gpointer)home_plugin);
 
  
}

static int usb_mode_get()
{
    FILE *fh;
    char iorbuf[512];
    int retval = USB_MODE_OTG;
    
//    fh = fopen ("/sys/devices/platform/musb_hdrc/mode", "r");
    fh = fopen (USB_MODE_PATH, "r");
    if (fh != NULL) {
	iorbuf[0] = '\0';
	while (fgets (iorbuf,sizeof(iorbuf),fh)) {
            g_strstrip(iorbuf);
	}
	fclose(fh);
	if (strlen(iorbuf) > 1) {
//	    if (strncmp(iorbuf, "a_wait_bcon", 11) == 0 )
	    if (strncmp(iorbuf, "a_", 2) == 0 )
		retval = USB_MODE_HOST;
	}
    }
    return retval;
}

static int usb_mode_toggle()
{
    char cmd[255];
    FILE *handle = NULL;
    int retval, new_mode;
    GtkWidget *dialog;
    
    if (usb_mode_get() == USB_MODE_OTG) {
        strcpy(cmd, USB_HOST_COMMAND);
	new_mode = USB_MODE_HOST;
    } else {
        strcpy(cmd, USB_OTG_COMMAND);
	new_mode = USB_MODE_OTG;
    }

    handle = popen (cmd, "r");
    if (!handle) {
        hildon_banner_show_information(NULL, NULL, "usb_mode_toggle: popen failed");
    } else {
	memset(&cmd, 0, sizeof(cmd));
	fread(cmd, 1, sizeof(cmd), handle);
	
	if (strlen(cmd) > 1) {
            dialog = gtk_dialog_new_with_buttons ("usb_mode_toggle",
                                                  NULL,
                                                  GTK_DIALOG_DESTROY_WITH_PARENT |
                                                  GTK_DIALOG_NO_SEPARATOR,
                                                  "Close",
                                                  GTK_RESPONSE_OK,
                                                  NULL);
            gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox),
                               gtk_label_new (cmd));
            gtk_widget_show_all (dialog);
            gtk_dialog_run (GTK_DIALOG(dialog));
            /* Free the dialog (and it's children) */
            gtk_widget_destroy(GTK_WIDGET(dialog));
        }
        retval = pclose (handle);
        if (retval == -1) {
	        /* Error reported by pclose() */
        } else {
	        /* Use macros described under wait() to inspect `status' in order
		 *        to determine success/failure of command executed by popen() */
        }

    	if (retval != 0) {
        	sprintf(cmd, "usb_mode_toggle: command returned %d", retval);
        	hildon_banner_show_information(NULL, NULL, cmd);
    	}
    }
    return new_mode;
}

/**
 * Callback for the button clicked signal
 *
 * @param widget the button widget
 * @param data helloworld info struct
 */
static gboolean ntp_clicked(GtkWidget *widget, gpointer data)
{
    EosTimeSyncPlugin *home_plugin = EOSTIMESYNC_PLUGIN(data);  
    EosTimeSyncPluginPrivate *priv = home_plugin->priv;
    char cmd[255];
    char buf[1024];
    int retval;
    FILE *handle = NULL;
    GtkWidget *dialog;

    hildon_banner_show_information(NULL, NULL, "Setting the time via NTP");
#ifdef __i386__	
    printf("data: %x \n", data);
    printf("priv: %x \n", priv);
#endif
    
    //strcpy(cmd, NTP_COMMAND);
    snprintf(cmd, 255, NTP_COMMAND, priv->ntp_server);
#ifdef __i386__	
    printf("eos_time_sync_ntp_clicked - ntp_server: %s\n", priv->ntp_server);
    printf("eos_time_sync_ntp_clicked - cmd: %s\n", cmd);
#endif
    handle = popen (cmd, "r");
    if (!handle) {
        hildon_banner_show_information(NULL, NULL, "ntp: popen failed");
	return TRUE;
    }
    
    memset(&buf, 0, sizeof(buf));
    fread(buf, 1, sizeof(buf), handle);
    
    retval = pclose (handle);
    if (retval == -1) {
	        /* Error reported by pclose() */
    } else {
	        /* Use macros described under wait() to inspect `status' in order
		 *        to determine success/failure of command executed by popen() */
    }
    if (retval != 0) {
    	sprintf(cmd, "ntp: ntpdate returned %d", retval);
    } else {
    	sprintf(cmd, "Time sync OK");
    }
    hildon_banner_show_information(NULL, NULL, cmd);
    
    if (retval != 0) {
    	dialog = gtk_dialog_new_with_buttons ("ntpdate",
        	                                NULL,
                	                        GTK_DIALOG_DESTROY_WITH_PARENT |
                        	                GTK_DIALOG_NO_SEPARATOR,
                                	        "Close",
                                        	GTK_RESPONSE_OK,
                                        	NULL);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox),
        	           gtk_label_new (buf));
	gtk_widget_show_all (dialog);

	gtk_dialog_run (GTK_DIALOG(dialog));

	 /* Free the dialog (and it's children) */
	 gtk_widget_destroy(GTK_WIDGET(dialog));
    }

#ifdef RETUTIME_COMMAND
    if (retval == 0) {
	/* if ntpdate was successful, then sync the time to the HW clock */
    	strcpy(cmd, RETUTIME_COMMAND);
    	handle = popen (cmd, "r");
    	if (!handle) {
        	hildon_banner_show_information(NULL, NULL, "retutime: popen failed");
		return TRUE;
    	}

    	retval = pclose (handle);
    	if (retval == -1) {
	        /* Error reported by pclose() */
    	} else {
	        /* Use macros described under wait() to inspect `status' in order
		 *        to determine success/failure of command executed by popen() */
    	}
    	if (retval != 0) {
    		sprintf(cmd, "ntp: retutime returned %d", retval);
    	} else {
    		sprintf(cmd, "Time synced to HW clock");
    	}
    	hildon_banner_show_information(NULL, NULL, cmd);
    }
#endif
    
    return TRUE;
}

/**
 * Callback for the button clicked signal
 *
 * @param widget the button widget
 * @param data helloworld info struct
 */
static gboolean usb_clicked(GtkWidget *widget, gpointer data)
{
    char filename[15];
    
    if (usb_mode_toggle() == USB_MODE_HOST)
	strcpy(filename, "usb_host.png");
    else
	strcpy(filename, "usb_otg.png");
    
    gtk_button_set_image (GTK_BUTTON (widget),
	                  gtk_image_new_from_file (g_strjoin("/", PIXMAP_PATH, filename, NULL )));
    gtk_widget_show_all (widget);

    return TRUE;
}

/**
 * Callback for the button clicked signal
 *
 * @param widget the button widget
 * @param data helloworld info struct
 */
static gboolean eos_clicked(GtkWidget *widget, gpointer data)
{
    char cmd[255];
    char buf[1024];
    int retval;
    FILE *handle = NULL;
    GtkWidget *dialog;

    strcpy(cmd, S10SH_COMMAND);
    handle = popen (cmd, "r");
    if (!handle) {
        hildon_banner_show_information(NULL, NULL, "eos: popen failed");
	return TRUE;
    }
    
    memset(&buf, 0, sizeof(buf));
    fread(buf, 1, sizeof(buf), handle);
    
    dialog = gtk_dialog_new_with_buttons ("s10sh",
                                        NULL,
                                        GTK_DIALOG_DESTROY_WITH_PARENT |
                                        GTK_DIALOG_NO_SEPARATOR,
                                        "Close",
                                        GTK_RESPONSE_OK,
                                        NULL);
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox),
                     gtk_label_new (buf));
    gtk_widget_show_all (dialog);

    gtk_dialog_run (GTK_DIALOG(dialog));

    /* Free the dialog (and it's children) */
    gtk_widget_destroy(GTK_WIDGET(dialog));

    retval = pclose (handle);
    if (retval == -1) {
	        /* Error reported by pclose() */
    } else {
	        /* Use macros described under wait() to inspect `status' in order
		 *        to determine success/failure of command executed by popen() */
    }

    sprintf(cmd, "eos: s10sh returned %d", retval);
    hildon_banner_show_information(NULL, NULL, cmd);

    return TRUE;
}

static void
eos_time_sync_plugin_class_init (EosTimeSyncPluginClass *class)
{
	HildonDesktopHomeItemClass           *applet_class;
	GtkObjectClass                       *object_class;
	
	applet_class = HILDON_DESKTOP_HOME_ITEM_CLASS (class);
	applet_class->settings = eos_time_sync_settings;
	object_class = GTK_OBJECT_CLASS (class);
	object_class->destroy = eos_time_sync_plugin_destroy;

	g_type_class_add_private (class, sizeof (EosTimeSyncPluginPrivate));
}
