/*
  EightyOne - A simple Sudoku solving game
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <iostream>
#include <vector>

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/Help.h>
#include <Lum/Dlg/Msg.h>
#include <Lum/Dlg/Progress.h>

#include <Lum/Array.h>
#include <Lum/Button.h>
#include <Lum/Combo.h>
#include <Lum/Dialog.h>
#include <Lum/Label.h>
#include <Lum/Menu.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>
#include <Lum/Text.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/OS/Main.h>

#include "config.h"
#include "Configuration.h"
#include "Dancing.h"
#include "GameArea.h"
#include "Statistics.h"

static Lum::AppInfo info;

// TODO
static Sudoku riddle,solution;

class GenerateTask : public Lum::Dlg::ProgressTask
{
public:
  Sudoku::Weight weight;
  size_t         minimumPoints;

public:
  void Run()
  {
    Sudoku area;

    SetProgress(0);
    SetAction(_(L"PROGRESS_SOLUTION",L"Calculating a new solution..."));
    area.FillRandom();
    riddle=area;
    solution=area;

    SetProgress(10);
    SetAction(_(L"PROGRESS_RIDDLE",L"Calculating riddle..."));

    Sudoku::GenerateRiddle(riddle,minimumPoints);

    SetProgress(90);
    SetAction(_(L"PROGRESS_WEIGHT",L"Calculating weight..."));

    area=riddle;
    area.CalculateWeight(solution,weight);

    SetProgress(100);
  }
};

class MainDialog : public Lum::Dialog
{
private:
  Lum::Model::ActionRef  generateAction;
  Lum::Model::ActionRef  solveAction;
  Lum::Model::ActionRef  statAction;
  Lum::Model::ActionRef  helpAction;
  Lum::Model::ActionRef  aboutAction;
  Lum::Model::StringRef  minimumPoints;
  GameArea               *gameArea;
  Lum::Text              *filledFields;
  Lum::Text              *directFills;
  Lum::Text              *crossHatch;
  Lum::Text              *singleCandidate;
  Lum::Text              *singleSquare;
  Lum::Text              *guesses;
  Lum::Text              *rating;
  GenerateTask           generateTask;

public:
  MainDialog()
  : generateAction(new Lum::Model::Action),
    solveAction(new Lum::Model::Action),
    statAction(new Lum::Model::Action),
    helpAction(new Lum::Model::Action),
    aboutAction(new Lum::Model::Action),
    minimumPoints(new Lum::Model::String()),
    gameArea(new GameArea()),
    directFills(NULL),
    crossHatch(NULL),
    singleCandidate(NULL),
    singleSquare(NULL),
    guesses(NULL),
    rating(NULL)
  {
    generateTask.SetParent(this);
    generateTask.SetCaption(_(L"PROGRESS_TITLE",L"Generating riddle..."));

    minimumPoints->Set(L"4000");

    AttachModel(generateAction);
    AttachModel(solveAction);
    AttachModel(statAction);
    AttachModel(helpAction);
    AttachModel(aboutAction);
    AttachModel(GetOpenedAction());
    AttachModel(GetClosedAction());
    AttachModel(generateTask.GetFinishedAction());
    gameArea->GetFinishedModel()->Off(); // Switch it off initialy and on after config reading!
    AttachModel(gameArea->GetFinishedModel());
  }

  void PreInit()
  {
    Lum::Array               *array;
    Lum::Button              *button;
    Lum::Combo               *combo;
    Lum::Label               *label;
    Lum::Model::ListTableRef table;
    Lum::Panel               *horiz,*vert;

    horiz=new Lum::HPanel();
    horiz->SetFlex(true,true);

    gameArea->SetFlex(true,true);
    gameArea->SetModel(riddle,solution);

    horiz->Add(gameArea);

    horiz->Add(new Lum::HSpace());

    vert=new Lum::VPanel();
    vert->SetFlex(false,true);

    array=new Lum::Array();
    array->SetFlex(true,false);
    array->SetVerticalCount(1);
    array->SetSpace(true,false);

    button=new Lum::Button;
    button->SetFlex(true,false);
    button->SetModel(generateAction);
    button->SetText(_(L"GENERATE",L"_Generate"));
    array->Add(button);

    button=new Lum::Button;
    button->SetFlex(true,false);
    button->SetModel(solveAction);
    button->SetText(_(L"SOLVE",L"_Solve"));
    array->Add(button);

    vert->Add(array);

    vert->Add(new Lum::VSpace());

    label=new Lum::Label();
    label->SetFlex(false,true);

    table=new Lum::Model::ListTable(1);
    table->AppendString(L"2500");
    table->AppendString(L"4000");
    table->AppendString(L"8000");
    table->AppendString(L"15000");
    table->AppendString(L"25000");

    combo=new Lum::TextEditCombo();
    combo->SetFlex(true,false);
    combo->SetModel(minimumPoints);
    combo->SetTableModel(table);
    dynamic_cast<Lum::TextEditCombo*>(combo)->GetString()->SetAlignment(Lum::String::right);
    label->AddLabel(_(L"POINTS_MIN",L"min. Points:"),combo);

    filledFields=new Lum::Text();
    filledFields->SetFlex(true,false);
    filledFields->SetWidth(Lum::Base::Size::stdCharWidth,2);
    filledFields->SetAlignment(Lum::Text::right);
    filledFields->SetText(L"0");
    label->AddLabel(_(L"POINTS_FILLED_FIELDS",L"Filled fields:"),filledFields);

    directFills=new Lum::Text();
    directFills->SetFlex(true,false);
    directFills->SetWidth(Lum::Base::Size::stdCharWidth,2);
    directFills->SetAlignment(Lum::Text::right);
    directFills->SetText(L"0");
    label->AddLabel(_(L"POINTS_DIRECT_FILLS",L"Direct fills:"),directFills);

    crossHatch=new Lum::Text();
    crossHatch->SetFlex(true,false);
    crossHatch->SetWidth(Lum::Base::Size::stdCharWidth,2);
    crossHatch->SetAlignment(Lum::Text::right);
    crossHatch->SetText(L"0");
    label->AddLabel(_(L"POINTS_CROSSHATCH",L"Cross hatch:"),crossHatch);

    singleCandidate=new Lum::Text();
    singleCandidate->SetFlex(true,false);
    singleCandidate->SetWidth(Lum::Base::Size::stdCharWidth,2);
    singleCandidate->SetAlignment(Lum::Text::right);
    singleCandidate->SetText(L"0");
    label->AddLabel(_(L"POINTS_SINGLE_CAND",L"Single candidate:"),singleCandidate);

    singleSquare=new Lum::Text();
    singleSquare->SetFlex(true,false);
    singleSquare->SetWidth(Lum::Base::Size::stdCharWidth,2);
    singleSquare->SetAlignment(Lum::Text::right);
    singleSquare->SetText(L"0");
    label->AddLabel(_(L"POINTS_SINGLE_SQUARE",L"Single square:"),singleSquare);

    guesses=new Lum::Text();
    guesses->SetFlex(true,false);
    guesses->SetWidth(Lum::Base::Size::stdCharWidth,2);
    guesses->SetAlignment(Lum::Text::right);
    guesses->SetText(L"0");
    label->AddLabel(_(L"POINTS_GUESS",L"Guesses:"),guesses);

    rating=new Lum::Text();
    rating->SetFlex(true,false);
    rating->SetWidth(Lum::Base::Size::stdCharWidth,7);
    rating->SetAlignment(Lum::Text::right);
    rating->SetStyle(Lum::OS::Font::bold);
    rating->SetText(L"0");
    label->AddLabel(_(L"POINTS_WEIGHT",L"Weight:"),rating);

    vert->Add(label);

    horiz->Add(vert);

    SetMain(horiz);

    Lum::MenuDesc *menu=new Lum::MenuDesc();

    menu
      ->AddMenuItemSub(_ld(menuProject))
        ->AddMenuItemAction(_ld(menuProjectQuit),Lum::OS::qualifierControl,L"q",GetClosedAction());
    menu
      ->AddMenuItemSub(_(L"MENU_GAME",L"_Game"))
        ->AddMenuItemAction(_(L"MENU_GAME_NEW",L"_Generate new game"),Lum::OS::qualifierControl,L"g",generateAction)
        ->AddMenuItemAction(_(L"MENU_GAME_SOLVE",L"_Solve game"),Lum::OS::qualifierControl,L"s",solveAction)
        ->AddSeparator()
        ->AddMenuItemAction(_(L"MENU_GAME_STAT",L"S_tatistics"),Lum::OS::qualifierControl,L"t",statAction);
    menu
      ->AddMenuItemSub(_ld(menuHelp))
        ->AddMenuItemAction(_ld(menuHelpHelp),helpAction)
        ->AddMenuItemAction(_ld(menuHelpAbout),aboutAction);

    SetMenu(menu);

    Dialog::PreInit();
  }

  void GenerateRiddle()
  {
    riddle.Clear();
    gameArea->GetFinishedModel()->Off();
    gameArea->SetModel(riddle,solution);

    if (!Lum::Base::WStringToNumber(minimumPoints->Get(),generateTask.minimumPoints) ||
        generateTask.minimumPoints<0) {
      Lum::Dlg::Msg::ShowOk(this,
                            _(L"INVALID_TITLE",L"Invalid value for 'minimum points'!"),
                            _(L"INVALID_BODY",
                              L"The value you have entered as minimum points for\n"
                              L"riddle generation is not a valid number!"));
      return;
    }

    generateTask.Start();
  }

  void SetRiddle()
  {
    char value[20];

    sprintf(value,"%.0f",generateTask.weight.rating);

    gameArea->SetModel(riddle,solution);
    filledFields->SetText(Lum::Base::NumberToWString(generateTask.weight.filledFields));
    directFills->SetText(Lum::Base::NumberToWString(generateTask.weight.directFills));
    crossHatch->SetText(Lum::Base::NumberToWString(generateTask.weight.crossHatch));
    singleCandidate->SetText(Lum::Base::NumberToWString(generateTask.weight.singleCandidate));
    singleSquare->SetText(Lum::Base::NumberToWString(generateTask.weight.singleSquare));
    guesses->SetText(Lum::Base::NumberToWString(generateTask.weight.guesses));
    rating->SetText(Lum::Base::StringToWString(value));
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      if (LoadConfig()) {
        if (game!=NULL) {
          Sudoku         riddle,solution;
          Sudoku::Weight weight;
          char           value[20];

          riddle.SetAreaAsString(game->riddle);
          solution.SetAreaAsString(game->solution);

          riddle.CalculateWeight(solution,weight);

          gameArea->SetGame(*game);

          sprintf(value,"%.0f",weight.rating);

          filledFields->SetText(Lum::Base::NumberToWString(weight.filledFields));
          directFills->SetText(Lum::Base::NumberToWString(weight.directFills));
          crossHatch->SetText(Lum::Base::NumberToWString(weight.crossHatch));
          singleCandidate->SetText(Lum::Base::NumberToWString(weight.singleCandidate));
          singleSquare->SetText(Lum::Base::NumberToWString(weight.singleSquare));
          guesses->SetText(Lum::Base::NumberToWString(weight.guesses));
          rating->SetText(Lum::Base::StringToWString(value));

          delete game;
          game=NULL;
        }
      }
      if (gameArea->GetFinishedModel()->Get()) {
        solveAction->Disable();
      }
      gameArea->GetFinishedModel()->On(false);
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      if (!gameArea->GetFinishedModel()->Get()) {
        game=new Game();
        gameArea->GetGame(*game);
      }
      SaveConfig();
    }
    else if (model==generateAction && generateAction->IsFinished()) {
      GenerateRiddle();
    }
    if (model==generateTask.GetFinishedAction() && generateTask.GetFinishedAction()->IsFinished()) {
      solveAction->Enable();
      SetRiddle();
      gameArea->GetFinishedModel()->On(false);
    }
    else if (model==solveAction && solveAction->IsFinished() && solveAction->IsEnabled()) {
      if (Lum::Dlg::Msg::Ask(this,
                             _(L"SOLVE_TITLE",L"Are you sure?"),
                             _(L"SOLVE_BODY",
                               L"Are you sure that you want to solve the game? \n"
                               L"\n"
                               L"Solving the game cannot be undone!"),
                             _(L"SOLVE_BUTTONS",L"_Solve it!*|_Cancel^"))==0) {
        gameArea->GetFinishedModel()->Off();
        gameArea->Solve();
        gameArea->GetFinishedModel()->On(false);
        solveAction->Disable();
      }
    }
    else if (model==statAction && statAction->IsFinished()) {
      Sudoku game;

      gameArea->GetGame(game);

      StatisticsDialog::Show(GetWindow(),game);
    }
    else if (model==helpAction && helpAction->IsFinished()) {
      Lum::Dlg::Help::Show(this,
                           _(L"HELP_BODY",
                             L"The interface supports setting numbers and marking\n"
                             L"fields as potentially holding a subset of numbers.\n"
                             L"\n"
                             L"To mark a field click on the upperleft quadrant. A\n"
                             L"popup will open, from which you can select/unselect\n"
                             L"multiple numbers for marking.\n"
                             L"To set a number, click on any other quadrant. Again\n"
                             L"a similar dialog will open, this time only allowing\n"
                             L"one (or no) number.\n"
                             L"Marks will shown a small numbers, set values as black,\n"
                             L"predefined number will be drawn grey, and conflicts\n"
                             L"red.\n"
                             L"\n"
                             L"Have fun :-)"));
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==gameArea->GetFinishedModel() && gameArea->GetFinishedModel()->Get()) {
      Lum::Dlg::Msg::ShowOk(this,
                            _(L"CONGRATS_TITLE",L"Yeeeaarrrr!!\n"),
                            _(L"CONGRATS_BODY",
                              L"It looks like you have successfully finished\n"
                              L"the given riddle!\n"
                              L"Congratulations!"));
      solveAction->Disable();
    }


    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::OS::MainDialog<MainDialog>
{
public:
  bool Prepare()
  {
#if defined(APP_DATADIR)
    Lum::Base::Path::SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(_(L"ABOUT_DESC",L"A simple Sudoku implementation..."));
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2006, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    return Lum::OS::MainDialog<MainDialog>::Prepare();
  }

  void Cleanup()
  {
    FreeConfig();

    Lum::OS::MainDialog<MainDialog>::Cleanup();
  }
};

LUM_MAIN(Main,L"EightyOne")
