"""
Wrapper for Remember The Milk API
"""

import datetime

import toolbox
import rtm_api


def fix_url(rturl):
	return "/".join(rturl.split(r"\/"))


class RtmBackend(object):
	"""
	Interface with rememberthemilk.com

	@todo Decide upon an interface that will end up a bit less bloated
	@todo Add interface for task tags
	@todo Add interface for postponing tasks (have way for UI to specify how many days to postpone?)
	@todo Add interface for task recurrence
	@todo Add interface for task estimate
	@todo Add interface for task location
	@todo Add interface for task url 
	@todo Add undo support
	"""
	API_KEY = '71f471f7c6ecdda6def341967686fe05'
	SECRET = '7d3248b085f7efbe'

	def __init__(self, username, password, token):
		self._username = username
		self._password = password
		self._token = token

		self._rtm = rtm_api.RTMapi(self._username, self.API_KEY, self.SECRET, token)
		self._token = token
		resp = self._rtm.timelines.create()
		self._timeline = resp.timeline
		self._lists = []

	def save(self):
		pass

	def load(self):
		pass

	def add_project(self, name):
		rsp = self._rtm.lists.add(
			timeline=self._timeline,
			name=name,
		)
		assert rsp.stat == "ok", "Bad response: %r" % (rsp, )
		self._lists = []
		return rsp.list.id

	def set_project_name(self, projId, name):
		rsp = self._rtm.lists.setName(
			timeline=self._timeline,
			list_id=projId,
			name=name,
		)
		assert rsp.stat == "ok", "Bad response: %r" % (rsp, )
		self._lists = []

	def set_project_visibility(self, projId, visibility):
		action = self._rtm.lists.unarchive if visibility else self._rtm.lists.archive
		rsp = action(
			timeline=self._timeline,
			list_id=projId,
		)
		assert rsp.stat == "ok", "Bad response: %r" % (rsp, )
		self._lists = []

	def get_projects(self):
		if len(self._lists) == 0:
			self._populate_projects()

		for list in self._lists:
			yield list

	def get_project(self, projId):
		projs = [proj for proj in self.get_projects() if projId == proj["id"]]
		assert len(projs) == 1, "%r: %r / %r" % (projId, projs, self._lists)
		return projs[0]

	def lookup_project(self, projName):
		"""
		From a project's name, returns the project's details
		"""
		todoList = [list for list in self.get_projects() if list["name"] == projName]
		assert len(todoList) == 1, "Wrong number of lists found for %s, in %r" % (projName, todoList)
		return todoList[0]

	def get_locations(self):
		rsp = self._rtm.locations.getList()
		assert rsp.stat == "ok", "Bad response: %r" % (rsp, )
		locations = [
			dict((
				("name", t.name),
				("id", t.id),
				("longitude", t.longitude),
				("latitude", t.latitude),
				("address", t.address),
			))
			for t in rsp.locations
		]
		return locations

	def get_tasks_with_details(self, projId):
		for realProjId, taskSeries in self._get_taskseries(projId):
			for task in self._get_tasks(taskSeries):
				taskId = self._pack_ids(realProjId, taskSeries.id, task.id)
				rawTaskDetails = {
					"id": taskId,
					"projId": projId,
					"name": taskSeries.name,
					"url": taskSeries.url,
					"locationId": taskSeries.location_id,
					"dueDate": task.due,
					"isCompleted": task.completed,
					"completedDate": task.completed,
					"priority": task.priority,
					"estimate": task.estimate,
					"notes": dict((
						(note["id"], note)
						for note in self._get_notes(taskId, taskSeries.notes)
					)),
				}
				taskDetails = self._parse_task_details(rawTaskDetails)
				yield taskDetails

	def get_task_details(self, taskId):
		projId, rtmSeriesId, rtmTaskId = self._unpack_ids(taskId)
		for task in self.get_tasks_with_details(projId):
			if task["id"] == taskId:
				return task
		return {}

	def add_task(self, projId, taskName):
		rsp = self._rtm.tasks.add(
			timeline=self._timeline,
			list_id=projId,
			name=taskName,
		)
		assert rsp.stat == "ok", "Bad response: %r" % (rsp, )
		seriesId = rsp.list.taskseries.id
		taskId = rsp.list.taskseries.task.id
		name = rsp.list.taskseries.name

		return self._pack_ids(projId, seriesId, taskId)

	def set_project(self, taskId, newProjId):
		projId, seriesId, taskId = self._unpack_ids(taskId)
		rsp = self._rtm.tasks.moveTo(
			timeline=self._timeline,
			from_list_id=projId,
			to_list_id=newProjId,
			taskseries_id=seriesId,
			task_id=taskId,
		)
		assert rsp.stat == "ok", "Bad response: %r" % (rsp, )

	def set_name(self, taskId, name):
		projId, seriesId, taskId = self._unpack_ids(taskId)
		rsp = self._rtm.tasks.setName(
			timeline=self._timeline,
			list_id=projId,
			taskseries_id=seriesId,
			task_id=taskId,
			name=name,
		)
		assert rsp.stat == "ok", "Bad response: %r" % (rsp, )

	def set_duedate(self, taskId, dueDate):
		assert isinstance(dueDate, toolbox.Optional), (
			"Date being set too definitively: %r" % dueDate
		)

		projId, seriesId, taskId = self._unpack_ids(taskId)
		rsp = self._rtm.tasks.setDueDate(
			timeline=self._timeline,
			list_id=projId,
			taskseries_id=seriesId,
			task_id=taskId,
			due=dueDate,
			parse=1,
		)
		assert rsp.stat == "ok", "Bad response: %r" % (rsp, )

	def set_priority(self, taskId, priority):
		assert isinstance(priority, toolbox.Optional), (
			"Priority being set too definitively: %r" % priority
		)
		priority = str(priority.get_nothrow("N"))
		projId, seriesId, taskId = self._unpack_ids(taskId)

		rsp = self._rtm.tasks.setPriority(
			timeline=self._timeline,
			list_id=projId,
			taskseries_id=seriesId,
			task_id=taskId,
			priority=priority,
		)
		assert rsp.stat == "ok", "Bad response: %r" % (rsp, )

	def complete_task(self, taskId):
		projId, seriesId, taskId = self._unpack_ids(taskId)

		rsp = self._rtm.tasks.complete(
			timeline=self._timeline,
			list_id=projId,
			taskseries_id=seriesId,
			task_id=taskId,
		)
		assert rsp.stat == "ok", "Bad response: %r" % (rsp, )

	def add_note(self, taskId, noteTitle, noteBody):
		projId, seriesId, taskId = self._unpack_ids(taskId)

		rsp = self._rtm.tasks.notes.add(
			timeline=self._timeline,
			list_id=projId,
			taskseries_id=seriesId,
			task_id=taskId,
			note_title=noteTitle,
			note_text=noteBody,
		)
		assert rsp.stat == "ok", "Bad response: %r" % (rsp, )
		noteId = rsp.note.id

		return self._pack_ids(projId, seriesId, taskId, noteId)

	def update_note(self, noteId, noteTitle, noteBody):
		projId, seriesId, taskId, note = self._unpack_ids(noteId)

		rsp = self._rtm.tasks.notes.edit(
			timeline=self._timeline,
			note_id=noteId,
			note_title=noteTitle,
			note_text=noteBody,
		)
		assert rsp.stat == "ok", "Bad response: %r" % (rsp, )

	def delete_note(self, noteId):
		projId, seriesId, taskId, noteId = self._unpack_ids(noteId)

		rsp = self._rtm.tasks.notes.delete(
			timeline=self._timeline,
			note_id=noteId,
		)
		assert rsp.stat == "ok", "Bad response: %r" % (rsp, )

	@staticmethod
	def _pack_ids(*ids):
		"""
		>>> RtmBackend._pack_ids(123, 456)
		'123-456'
		"""
		return "-".join((str(id) for id in ids))

	@staticmethod
	def _unpack_ids(ids):
		"""
		>>> RtmBackend._unpack_ids("123-456")
		['123', '456']
		"""
		return ids.split("-")

	def _get_taskseries(self, projId):
		rsp = self._rtm.tasks.getList(
			list_id=projId,
		)
		assert rsp.stat == "ok", "Bad response: %r" % (rsp, )
		# @note Meta-projects return lists for each project (I think)
		rspTasksList = rsp.tasks.list

		if not isinstance(rspTasksList, list):
			rspTasksList = (rspTasksList, )

		for something in rspTasksList:
			realProjId = something.id
			try:
				something.taskseries
			except AttributeError:
				continue

			if isinstance(something.taskseries, list):
				somethingsTaskseries = something.taskseries
			else:
				somethingsTaskseries = (something.taskseries, )

			for taskSeries in somethingsTaskseries:
				yield realProjId, taskSeries

	def _get_tasks(self, taskSeries):
		if isinstance(taskSeries.task, list):
			tasks = taskSeries.task
		else:
			tasks = (taskSeries.task, )
		for task in tasks:
			yield task

	def _parse_task_details(self, rawTaskDetails):
		taskDetails = {}
		taskDetails["id"] = rawTaskDetails["id"]
		taskDetails["projId"] = rawTaskDetails["projId"]
		taskDetails["name"] = rawTaskDetails["name"]
		taskDetails["url"] = fix_url(rawTaskDetails["url"])

		rawLocationId = rawTaskDetails["locationId"]
		if rawLocationId:
			locationId = toolbox.Optional(rawLocationId)
		else:
			locationId = toolbox.Optional()
		taskDetails["locationId"] = locationId

		if rawTaskDetails["dueDate"]:
			dueDate = datetime.datetime.strptime(
				rawTaskDetails["dueDate"],
				"%Y-%m-%dT%H:%M:%SZ",
			)
			dueDate = toolbox.Optional(dueDate)
		else:
			dueDate = toolbox.Optional()
		taskDetails["dueDate"] = dueDate

		taskDetails["isCompleted"] = len(rawTaskDetails["isCompleted"]) != 0

		if rawTaskDetails["completedDate"]:
			completedDate = datetime.datetime.strptime(
				rawTaskDetails["completedDate"],
				"%Y-%m-%dT%H:%M:%SZ",
			)
			completedDate = toolbox.Optional(completedDate)
		else:
			completedDate = toolbox.Optional()
		taskDetails["completedDate"] = completedDate

		try:
			priority = toolbox.Optional(int(rawTaskDetails["priority"]))
		except ValueError:
			priority = toolbox.Optional()
		taskDetails["priority"] = priority

		if rawTaskDetails["estimate"]:
			estimate = rawTaskDetails["estimate"]
			estimate = toolbox.Optional(estimate)
		else:
			estimate = toolbox.Optional()
		taskDetails["estimate"] = estimate

		taskDetails["notes"] = rawTaskDetails["notes"]

		rawKeys = list(rawTaskDetails.iterkeys())
		rawKeys.sort()
		parsedKeys = list(taskDetails.iterkeys())
		parsedKeys.sort()
		assert rawKeys == parsedKeys, "Missing some, %r != %r" % (rawKeys, parsedKeys)

		return taskDetails

	def _get_notes(self, taskId, notes):
		if not notes:
			return
		elif isinstance(notes.note, list):
			notes = notes.note
		else:
			notes = (notes.note, )

		projId, rtmSeriesId, rtmTaskId = self._unpack_ids(taskId)

		for note in notes:
			noteId = self._pack_ids(projId, rtmSeriesId, rtmTaskId, note.id)
			title = note.title
			body = getattr(note, "$t")
			yield {
				"id": noteId,
				"title": title,
				"body": body,
			}

	def _populate_projects(self):
		rsp = self._rtm.lists.getList()
		assert rsp.stat == "ok", "Bad response: %r" % (rsp, )
		del self._lists[:]
		self._lists.extend((
			dict((
				("name", t.name),
				("id", t.id),
				("isVisible", not int(t.archived)),
				("isMeta", not not int(t.smart)),
			))
			for t in rsp.lists.list
		))
