/*
 *  (C) 2004-2009  Dominik Brodowski <linux@dominikbrodowski.de>
 *
 *  Licensed under the terms of the GNU GPL License version 2.
 */


#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include "cpufreq.h"
#include "interfaces.h"

#define LINE_LEN 10

int cpufreq_cpu_exists(unsigned int cpu)
{
	int ret = sysfs_cpu_exists(cpu);
	if (ret == -ENOSYS)
		ret = proc_cpu_exists(cpu);
	return (ret);
}

unsigned long cpufreq_get_freq_kernel(unsigned int cpu)
{
	unsigned long ret = sysfs_get_freq_kernel(cpu);
	if (!ret)
		ret = proc_get_freq_kernel(cpu);
	return (ret);
}

unsigned long cpufreq_get_freq_hardware(unsigned int cpu)
{
	unsigned long ret = sysfs_get_freq_hardware(cpu);
	if (!ret)
		ret = proc_get_freq_hardware(cpu);
	return (ret);
}

unsigned long cpufreq_get_transition_latency(unsigned int cpu)
{
	unsigned long ret = sysfs_get_transition_latency(cpu);
	if (!ret)
		ret = proc_get_transition_latency(cpu);
	return (ret);
}

int cpufreq_get_hardware_limits(unsigned int cpu,
				unsigned long *min,
				unsigned long *max)
{
	int ret;
	if ((!min) || (!max))
		return -EINVAL;
	ret = sysfs_get_hardware_limits(cpu, min, max);
	if (ret)
		ret = proc_get_hardware_limits(cpu, min, max);
	return (ret);
}

char * cpufreq_get_driver(unsigned int cpu) {
	char * ret;
	ret = sysfs_get_driver(cpu);
	if (!ret) {
		ret = proc_get_driver(cpu);
	}
	return (ret);
}

void cpufreq_put_driver(char * ptr) {
	if (!ptr)
		return;
	free(ptr);
}

struct cpufreq_policy * cpufreq_get_policy(unsigned int cpu) {
	struct cpufreq_policy * ret;
	ret = sysfs_get_policy(cpu);
	if (!ret)
		ret = proc_get_policy(cpu);
	return (ret);
}

void cpufreq_put_policy(struct cpufreq_policy *policy) {
	if ((!policy) || (!policy->governor))
		return;

	free(policy->governor);
	policy->governor = NULL;
	free(policy);
}

struct cpufreq_available_governors * cpufreq_get_available_governors(unsigned int cpu) {
	struct cpufreq_available_governors *ret;
	ret = sysfs_get_available_governors(cpu);
	if (!ret)
		ret = proc_get_available_governors(cpu);
	return (ret);
}

void cpufreq_put_available_governors(struct cpufreq_available_governors *any) {
	struct cpufreq_available_governors *tmp, *next;

	if (!any)
		return;

	tmp = any->first;
	while (tmp) {
		next = tmp->next;
		if (tmp->governor)
			free(tmp->governor);
		free(tmp);
		tmp = next;
	}
}


struct cpufreq_available_frequencies * cpufreq_get_available_frequencies(unsigned int cpu) {
	struct cpufreq_available_frequencies * ret;
	ret = sysfs_get_available_frequencies(cpu);
	if (!ret)
		ret = proc_get_available_frequencies(cpu);
	return (ret);
}

void cpufreq_put_available_frequencies(struct cpufreq_available_frequencies *any) {
	struct cpufreq_available_frequencies *tmp, *next;

	if (!any)
		return;

	tmp = any->first;
	while (tmp) {
		next = tmp->next;
		free(tmp);
		tmp = next;
	}
}


struct cpufreq_affected_cpus * cpufreq_get_affected_cpus(unsigned int cpu) {
	struct cpufreq_affected_cpus * ret;
	ret = sysfs_get_affected_cpus(cpu);
	if (!ret)
		ret = proc_get_affected_cpus(cpu);
	return (ret);
}

void cpufreq_put_affected_cpus(struct cpufreq_affected_cpus *any) {
	struct cpufreq_affected_cpus *tmp, *next;

	if (!any)
		return;

	tmp = any->first;
	while (tmp) {
		next = tmp->next;
		free(tmp);
		tmp = next;
	}
}


struct cpufreq_affected_cpus * cpufreq_get_related_cpus(unsigned int cpu) {
	struct cpufreq_affected_cpus * ret;
	ret = sysfs_get_related_cpus(cpu);
	if (!ret)
		ret = proc_get_related_cpus(cpu);
	return (ret);
}

void cpufreq_put_related_cpus(struct cpufreq_affected_cpus *any) {
	cpufreq_put_affected_cpus(any);
}


int cpufreq_set_policy(unsigned int cpu, struct cpufreq_policy *policy) {
	int ret;
	if (!policy || !(policy->governor))
		return -EINVAL;

	ret = sysfs_set_policy(cpu, policy);
	if (ret)
		ret = proc_set_policy(cpu, policy);
	return (ret);
}


int cpufreq_modify_policy_min(unsigned int cpu, unsigned long min_freq) {
	int ret;

	ret = sysfs_modify_policy_min(cpu, min_freq);
	if (ret)
		ret = proc_modify_policy_min(cpu, min_freq);
	return (ret);
}


int cpufreq_modify_policy_max(unsigned int cpu, unsigned long max_freq) {
	int ret;

	ret = sysfs_modify_policy_max(cpu, max_freq);
	if (ret)
		ret = proc_modify_policy_max(cpu, max_freq);
	return (ret);
}


int cpufreq_modify_policy_governor(unsigned int cpu, char *governor) {
	int ret;

	if ((!governor) || (strlen(governor) > 19))
		return -EINVAL;

	ret = sysfs_modify_policy_governor(cpu, governor);
	if (ret)
		ret = proc_modify_policy_governor(cpu, governor);
	return (ret);
}

int cpufreq_set_frequency(unsigned int cpu, unsigned long target_frequency) {
	int ret;

	ret = sysfs_set_frequency(cpu, target_frequency);
	if (ret)
		ret = proc_set_frequency(cpu, target_frequency);
	return (ret);
}

struct cpufreq_stats * cpufreq_get_stats(unsigned int cpu, unsigned long long *total_time) {
	struct cpufreq_stats *ret;

	ret = sysfs_get_stats(cpu, total_time);
	return (ret);
}

void cpufreq_put_stats(struct cpufreq_stats *any) {
	struct cpufreq_stats *tmp, *next;

	if (!any)
		return;

	tmp = any->first;
	while (tmp) {
		next = tmp->next;
		free(tmp);
		tmp = next;
	}
}

unsigned long cpufreq_get_transitions(unsigned int cpu) {
	unsigned long ret = sysfs_get_transitions(cpu);

	return (ret);
}

unsigned int count_cpus(void)
{
        FILE *fp;
        char value[LINE_LEN];
        unsigned int ret = 0;
        unsigned int cpunr = 0;

        fp = fopen("/proc/stat", "r");
        if(!fp) {
                //printf(gettext("Couldn't count the number of CPUs (%s: %s), assuming 1\n"), "/proc/stat", strerror(errno));
                return 1;
        }

        while (!feof(fp)) {
                if (!fgets(value, LINE_LEN, fp))
                        continue;
                value[LINE_LEN - 1] = '\0';
                if (strlen(value) < (LINE_LEN - 2))
                        continue;
                if (strstr(value, "cpu "))
                        continue;
                if (sscanf(value, "cpu%d ", &cpunr) != 1)
                        continue;
                if (cpunr > ret)
                        ret = cpunr;
        }
        fclose(fp);

        /* cpu count starts from 0, on error return 1 (UP) */
        return (ret+1);
}

unsigned long cpufreq_get_up_threshold(unsigned int cpu)
{
    unsigned long ret = sysfs_get_up_threshold(cpu);
    return (ret);
}

unsigned long cpufreq_get_sampling_rate(unsigned int cpu)
{
    unsigned long ret = sysfs_get_sampling_rate(cpu);
    return (ret);
}

int cpufreq_set_up_threshold(unsigned int cpu, unsigned long target_up_threshold)
{
    int ret;
    ret = sysfs_set_up_threshold(cpu, target_up_threshold);
    return (ret);
}

int cpufreq_set_sampling_rate(unsigned int cpu, unsigned long target_sampling_rate)
{
    int ret;
    ret = sysfs_set_sampling_rate(cpu, target_sampling_rate);
    return (ret);
}

int cpufreq_get_ignore_nice_load(unsigned int cpu)
{
    return sysfs_get_ignore_nice_load(cpu);
}

int cpufreq_set_ignore_nice_load(unsigned int cpu, int target_ignore_nice_load)
{
    return sysfs_set_ignore_nice_load(cpu, target_ignore_nice_load);
}

int cpufreq_get_SmartReflex()
{
    return sysfs_get_SmartReflex();
}

int cpufreq_set_SmartReflex(int target_ignore_nice_load)
{
    sysfs_set_SmartReflex(target_ignore_nice_load);
}
