/*
 * CoverFinder - Find and download cover images from Amazon 
 * Copyright (C) 2007 - Sven Salzwedel
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include <errno.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <curl/curl.h>

#include "net.h"

gchar *net_escape (const gchar * string, GError ** err)
{
	const gchar *p;
	GString *escaped;

	escaped = g_string_new (NULL);
	p = string;

	/* simply escape all chars which are not [A-Z][a-z][0-9] */
	while (*p != '\0') {
		if (*p < 0x30 || (*p > 0x39 && *p < 0x41) || (*p > 0x5a && *p < 0x61) || *p > 0x7a) {
			g_string_append_printf (escaped, "%%%02X", (guchar) *p);
		} else {
			g_string_append_c (escaped, *p);
		}
		p++;
	}
	return g_string_free (escaped, FALSE);
}

gboolean net_getfile_fp (const gchar * url, FILE * fp, GError ** err)
{
	CURL *handle;
	CURLcode ccode;
	gchar errbuf[CURL_ERROR_SIZE];
	gboolean retval = TRUE;

	if ((handle = curl_easy_init ()) == NULL) {
		g_set_error (err, 0, 0,
					 "Failed to initialize network connection: CURL initialization failed.");
		return FALSE;
	}

	if ((ccode = curl_easy_setopt (handle, CURLOPT_ERRORBUFFER, errbuf)) != CURLE_OK) {
		g_set_error (err, 0, 0, "Error while setting curl option: %s.", curl_easy_strerror (ccode));
		retval = FALSE;
	}

	if (retval
		&& (curl_easy_setopt (handle, CURLOPT_URL, url) != CURLE_OK
			|| curl_easy_setopt (handle, CURLOPT_WRITEDATA, fp) != CURLE_OK
			|| curl_easy_perform (handle) != CURLE_OK)) {
		g_set_error (err, 0, 0, "Failed to perform request: %s.", errbuf);
		retval = FALSE;
	}

	curl_easy_cleanup (handle);

	return retval;
}

gboolean net_getfile (const gchar * url, const gchar * filename, GError ** err)
{
	FILE *file_s;
	gboolean net_result;

	if ((file_s = g_fopen (filename, "w+")) == NULL) {
		g_set_error (err, 0, 0, "Error while opening cache file '%s': %s.", filename,
					 g_strerror (errno));
		return FALSE;
	}

	net_result = net_getfile_fp (url, file_s, err);
	fclose (file_s);

	return net_result;
}
