
/*
 * This file is part of Countdown Home Plugin.
 *
 * Copyright (C) 2008 Andrew Olmsted. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
*  
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <hildon/hildon.h>
#include <libhildondesktop/libhildondesktop.h>

#include <libosso.h>

#define HOME_DIR g_get_home_dir()

/* Common struct types declarations */
typedef struct _CountdownHomePlugin CountdownHomePlugin;
typedef struct _CountdownUI CountdownUI;

/* Instance struct */
struct _CountdownHomePlugin
{
	GtkWidget *mainwidget;
	GtkWidget *scrolled_window;
	osso_context_t *osso;
	guint eventID;
	gint refresh;
	GtkWindow *parent;
	gboolean fontCheck, headerFontCheck;
	gchar *fontText, *headerFontText;
	gchar *colourText, *headerColourText;
	gchar *yearString,*weekString,*dayString,*hourString,*minuteString;
	gchar *passedString,*remainingString;
	gboolean hidePassed;
	GtkListStore *entries;
	CountdownUI *UI;
};

struct _CountdownUI
{
	GtkWidget *headerFontButton, *fontButton;
	GtkWidget *headerColourButton, *colourButton;
	GtkTreeView *view;
};


static void on_countdown_settings(GtkWidget *widget, CountdownHomePlugin *setup);

static void system_check_toggle(GtkToggleButton *toggleButton, CountdownHomePlugin *setup)
{
	if (gtk_toggle_button_get_active(toggleButton))
	{
		gtk_widget_hide(gtk_widget_get_parent(setup->UI->fontButton));
		gtk_widget_hide(gtk_widget_get_parent(setup->UI->colourButton));
	}
	else
	{
		gtk_widget_show(gtk_widget_get_parent(setup->UI->fontButton));
		gtk_widget_show(gtk_widget_get_parent(setup->UI->colourButton));
	}
}

static void header_system_check_toggle(GtkToggleButton *toggleButton, CountdownHomePlugin *setup)
{
	if (gtk_toggle_button_get_active(toggleButton))
	{
		gtk_widget_hide(gtk_widget_get_parent(setup->UI->headerFontButton));
		gtk_widget_hide(gtk_widget_get_parent(setup->UI->headerColourButton));
	}
	else
	{
		gtk_widget_show(gtk_widget_get_parent(setup->UI->headerFontButton));
		gtk_widget_show(gtk_widget_get_parent(setup->UI->headerColourButton));
	}
}

void countdown_content_create (CountdownHomePlugin *setup)
{
	GtkWidget *headingLabel, *contentLabel;
	GtkWidget *vbox;
	GDate *startDate=NULL, *endDate=NULL;
	gint startSec=0, endSec=0;
	time_t now;
	struct tm *timeinfo;
	GtkTreeIter iter;
	gboolean exists;

	startDate=g_date_new();
	time(&now);
	g_date_set_time_t(startDate,now);
	timeinfo=localtime(&now);
	startSec=timeinfo->tm_hour*(60*60)+timeinfo->tm_min*60;

	endDate=g_date_new();

	vbox=gtk_vbox_new(FALSE,0);

	exists=gtk_tree_model_get_iter_first(GTK_TREE_MODEL(setup->entries),&iter);
	if (exists)
	{
		while (exists)
		{
			gchar* title;
			gchar* date;
			gchar* time;
			gint years=0;
			gint weeks=0;
			gint days=0;
			gint seconds=0;
			gint minutes=0;
			gint hours=0;
			gint totDays=0;
			gboolean showYears,showWeeks,showDays,showHours,showMinutes;
			gboolean displayCheck;
			gint displayNum;
			gchar *tempContent="";
			gchar **content=g_new0(gchar*,6);
			gchar *contentLabelText;
			gint index=0;
			gchar *tail="";
			gboolean firstShown;

			gtk_tree_model_get(GTK_TREE_MODEL(setup->entries),&iter,0,&title,1,&date,2,&time,4,&showYears,5,&showWeeks,6,&showDays,7,&showHours,8,&showMinutes,9,&displayCheck,10,&displayNum,-1);

			seconds=g_ascii_strtoll(time,NULL,10);


			g_date_set_parse(endDate,date);
			if (!g_date_valid(endDate))
			{
				title=g_strconcat(title,"(INCORRECT DATE!)",NULL);
				totDays=0;
			}
			else
			{
				totDays=g_date_days_between(startDate,endDate);
			}
			endSec=seconds;
			seconds=endSec-startSec;

			if (totDays>0)
			{
				tail=g_strdup(setup->remainingString);
				if (seconds<0)
				{
					totDays--;
					seconds=(60*60*24)+seconds;
				}
			}
			else if (totDays<0)
			{
				tail=g_strdup(setup->passedString);
				if (seconds>0)
				{
					totDays++;
					seconds=(60*60*24)-seconds;
				}
			}
			else
			{
				if (seconds<0)
				{
					tail=g_strdup(setup->passedString);
				}
				else
				{
					tail=g_strdup(setup->remainingString);
				}
			}

			firstShown=FALSE;
			if ((g_strcasecmp(tail,setup->remainingString)==0)||(!setup->hidePassed))
			{
				totDays=abs(totDays);
				seconds=abs(seconds);
				
				if (showYears)
				{
					years=totDays/365;
					if ((firstShown)||(years!=0)||((!showWeeks)&&(!showDays)&&(!showHours)&&(!showMinutes)))
					{
						firstShown=TRUE;
						tempContent=g_strdup_printf("%d",abs(years));
						content[index++]=g_strconcat(tempContent," ",setup->yearString," ",NULL);
						g_free(tempContent);
					}
				}
				if (showWeeks)
				{
					weeks=(totDays-(years*365))/7;
					if ((firstShown)||(weeks!=0)||((!showDays)&&(!showHours)&&(!showMinutes)))
					{
						firstShown=TRUE;
						tempContent=g_strdup_printf("%d",abs(weeks));
						content[index++]=g_strconcat(tempContent," ",setup->weekString," ",NULL);
						g_free(tempContent);
					}
				}
				if (showDays)
				{
					days=totDays-(weeks*7)-(years*365);
					if ((firstShown)||(days!=0)||((!showHours)&&(!showMinutes)))
					{
						firstShown=TRUE;
						tempContent=g_strdup_printf("%d",abs(days));
						content[index++]=g_strconcat(tempContent," ",setup->dayString," ",NULL);
						g_free(tempContent);
					}
				}
				if (showHours)
				{
					hours=((totDays-(days+(weeks*7)+(years*365)))*(24))+seconds/(60*60);
					if ((firstShown)||(hours!=0)||(!showMinutes))
					{
						firstShown=TRUE;
						tempContent=g_strdup_printf("%d",abs(hours));
						content[index++]=g_strconcat(tempContent," ",setup->hourString," ",NULL);
						g_free(tempContent);
					}
				}
				if (showMinutes)
				{
					minutes=((totDays-(days+(weeks*7)+(years*365)))*(24*60))+((seconds/60)-(hours*60));
					firstShown=TRUE;
					tempContent=g_strdup_printf("%d",abs(minutes));
					content[index++]=g_strconcat(tempContent," ",setup->minuteString," ",NULL);
					g_free(tempContent);
				}
				content[index]=g_strconcat(tail,NULL);

				headingLabel=gtk_label_new(title);
				if (setup->headerFontCheck)
				{
					gchar *temp=g_strconcat("<span weight='bold'>",title,"</span>",NULL);
					gtk_label_set_markup(GTK_LABEL(headingLabel),temp);
					g_free(temp);
				}
				else
				{
					gchar *temp=g_strconcat("<span font_desc='",setup->headerFontText,"' foreground='#",setup->headerColourText,"'>",title,"</span>",NULL);
					gtk_label_set_markup(GTK_LABEL(headingLabel),temp);
					g_free(temp);
				}
				gtk_box_pack_start(GTK_BOX(vbox),headingLabel,FALSE,FALSE,0);
				contentLabelText=g_strjoinv(NULL,content);
				contentLabel=gtk_label_new(contentLabelText);
				if (!setup->fontCheck)
				{
					gchar *temp=g_strconcat("<span font_desc='",setup->fontText,"' foreground='#",setup->colourText,"'>",contentLabelText,"</span>",NULL);
					gtk_label_set_markup(GTK_LABEL(contentLabel),temp);
					g_free(temp);
				}
				gtk_box_pack_start(GTK_BOX(vbox),contentLabel,FALSE,FALSE,0);
				g_free(contentLabelText);
			}
			//g_free(tempContent);
			g_free(title);
			g_free(date);
			g_free(time);
			g_free(tail);
			g_strfreev(content);
			
			exists=gtk_tree_model_iter_next(GTK_TREE_MODEL(setup->entries),&iter);
		}
	}
	else
	{
		GtkWidget *settingsButton=gtk_button_new_with_label("Click here to set\nup your countdowns");
		setup->parent=NULL;
		g_signal_connect(settingsButton,"clicked",G_CALLBACK(on_countdown_settings),setup);
		gtk_box_pack_start(GTK_BOX(vbox),settingsButton,FALSE,FALSE,0);
	}

	if (gtk_bin_get_child(GTK_BIN(setup->scrolled_window)))
	{
		GtkWidget *child=gtk_bin_get_child(GTK_BIN(setup->scrolled_window));
		//gtk_container_remove(GTK_CONTAINER(setup->scrolled_window),child);
		gtk_widget_destroy(child);
	}
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(setup->scrolled_window),vbox); 
	gtk_widget_show_all(setup->scrolled_window);

	g_date_free(startDate);
	g_date_free(endDate);
}

void setup_make_view(CountdownHomePlugin *setup)
{
	gchar* filename;
	gboolean fileExists;
	GKeyFile* keyFile;
	GtkTreeIter iter;

	setup->entries=gtk_list_store_new(11,G_TYPE_STRING,G_TYPE_STRING,G_TYPE_STRING,G_TYPE_STRING,G_TYPE_BOOLEAN,G_TYPE_BOOLEAN,G_TYPE_BOOLEAN,G_TYPE_BOOLEAN,G_TYPE_BOOLEAN,G_TYPE_BOOLEAN,G_TYPE_INT);

	keyFile=g_key_file_new();
	filename=g_strconcat(HOME_DIR,"/.countdown_home.rc",NULL);
	fileExists=g_key_file_load_from_file(keyFile,filename,G_KEY_FILE_KEEP_COMMENTS,NULL);

	if (fileExists)
	{
	  int i=0;
	  char entry[5];
	  gboolean found;
	  GError *error=NULL;
	  snprintf(entry,5,"%d",i);
	  found=g_key_file_has_group(keyFile,entry);
	  while (found)
	  {
	    gchar* title;
	    gchar* date;
	    gchar* time;
	    gchar* readableTime;
	    gint seconds;
	    gint minutes;
	    gint hours;
	    gboolean *showWhich;
	    gboolean displayCheck;
	    gint displayNum;
	    gsize length;
	    
	    sprintf(entry,"%d",i);
	    title=g_key_file_get_string(keyFile,entry,"title",&error);
	    if (error)
	    {
	    	title="TITLE MISSING!";
			g_error_free(error);
			error=NULL;
	    }
	    date=g_key_file_get_string(keyFile,entry,"date",&error);
	    if (error)
	    {
	    	date="INCORRECT DATE!";
			g_error_free(error);
			error=NULL;
	    }
	    time=g_key_file_get_string(keyFile,entry,"time",&error);
	    if (error)
	    {
	    	time="0";
	    	g_error_free(error);
	    	error=NULL;
	    }
	    showWhich=g_key_file_get_boolean_list(keyFile,entry,"show",&length,&error);
	    if (error)
	    {
	    	gboolean show[5]={TRUE,TRUE,TRUE,TRUE,TRUE};
	    	showWhich=show;
	    	g_error_free(error);
	    	error=NULL;
	    }
		displayCheck=g_key_file_get_boolean(keyFile,entry,"display check",&error);
		if (error)
		{
			displayCheck=FALSE;
			g_error_free(error);
			error=NULL;
		}
		displayNum=g_key_file_get_integer(keyFile,entry,"display num",&error);
		if (error)
		{
			displayNum=1;
			g_error_free(error);
			error=NULL;
		}
	    seconds=g_ascii_strtoll(time,NULL,10);
	    hours=seconds/(60*60);
  		minutes=seconds/(60)-(hours*60);
  		readableTime=g_strdup_printf("%02d:%02d",hours,minutes);
	    
	    gtk_list_store_append(setup->entries, &iter);
    	gtk_list_store_set(setup->entries, &iter,0,title,1,date,2,time,3,readableTime,4,showWhich[0],5,showWhich[1],6,showWhich[2],7,showWhich[3],8,showWhich[4],9,displayCheck,10,displayNum,-1);
	    snprintf(entry,5,"%d",++i);
	    found=g_key_file_has_group(keyFile,entry);
	    free(readableTime);
	    g_free(title);
	    g_free(date);
	    g_free(time);
	    g_free(showWhich);
	  }
		setup->refresh=g_key_file_get_integer(keyFile,"config","refresh",&error);
		if (error)
		{
			setup->refresh=1;
			g_error_free(error);
			error=NULL;
		}
		setup->hidePassed=g_key_file_get_boolean(keyFile,"config","hide passed",&error);
		if (error)
		{
			setup->hidePassed=FALSE;
			g_error_free(error);
			error=NULL;
		}
		setup->yearString=g_key_file_get_string(keyFile,"config","years label",&error);
		if (error)
		{
			setup->yearString="years";
			g_error_free(error);
			error=NULL;
		}
		setup->weekString=g_key_file_get_string(keyFile,"config","weeks label",&error);
		if (error)
		{
			setup->weekString="weeks";
			g_error_free(error);
			error=NULL;
		}
		setup->dayString=g_key_file_get_string(keyFile,"config","days label",&error);
		if (error)
		{
			setup->dayString="days";
			g_error_free(error);
			error=NULL;
		}
		setup->hourString=g_key_file_get_string(keyFile,"config","hours label",&error);
		if (error)
		{
			setup->hourString="hours";
			g_error_free(error);
			error=NULL;
		}
		setup->minuteString=g_key_file_get_string(keyFile,"config","minutes label",&error);
		if (error)
		{
			setup->minuteString="minutes";
			g_error_free(error);
			error=NULL;
		}
		setup->passedString=g_key_file_get_string(keyFile,"config","passed label",&error);
		if (error)
		{
			setup->passedString="passed";
			g_error_free(error);
			error=NULL;
		}
		setup->remainingString=g_key_file_get_string(keyFile,"config","remaining label",&error);
		if (error)
		{
			setup->remainingString="left";
			g_error_free(error);
			error=NULL;
		}
		setup->fontCheck=g_key_file_get_boolean(keyFile,"config","system style",&error);
		if (error)
		{
			setup->fontCheck=TRUE;
			g_error_free(error);
			error=NULL;
		}
		setup->headerFontCheck=g_key_file_get_boolean(keyFile,"config","header system style",&error);
		if (error)
		{
			setup->headerFontCheck=TRUE;
			g_error_free(error);
			error=NULL;
		}
		setup->colourText=g_key_file_get_string(keyFile,"config","font colour",&error);
		if (error)
		{
			setup->colourText="000000000000";
			g_error_free(error);
			error=NULL;
		}
		setup->headerColourText=g_key_file_get_string(keyFile,"config","header font colour",&error);
		if (error)
		{
			setup->headerColourText="000000000000";
			g_error_free(error);
			error=NULL;
		}
		setup->fontText=g_key_file_get_string(keyFile,"config","font style",&error);
		if (error)
		{
			setup->fontText="Sans 16";
			g_error_free(error);
			error=NULL;
		}
		setup->headerFontText=g_key_file_get_string(keyFile,"config","header font style",&error);
		if (error)
		{
			setup->headerFontText="Sans 16";
			g_error_free(error);
			error=NULL;
		}
	}
	else
	{
		setup->refresh=1;
		setup->hidePassed=FALSE;
		setup->yearString="years";
		setup->weekString="weeks";
		setup->dayString="days";
		setup->hourString="hours";
		setup->minuteString="minutes";
		setup->remainingString="left";
		setup->passedString="passed";
		setup->fontCheck=TRUE;
		setup->headerFontCheck=TRUE;
		setup->colourText="000000000000";
		setup->headerColourText="000000000000";
		setup->fontText="Sans 16";
		setup->headerFontText="Sans 16";
	}
	g_key_file_free(keyFile);
    g_free(filename);
}

void set_font_button_text(CountdownHomePlugin *setup, gchar *text, gboolean header)
{
	if (strlen(text)>15)
	{
		char result[16]="";
		char ell[3]="...";
		strncat(result,text,12);
		strncat(result,ell,3);
		if (header)
			gtk_button_set_label(GTK_BUTTON(setup->UI->headerFontButton),result);
		else
			gtk_button_set_label(GTK_BUTTON(setup->UI->fontButton),result);
	}
	else
	{
		if (header)
			gtk_button_set_label(GTK_BUTTON(setup->UI->headerFontButton),text);
		else
			gtk_button_set_label(GTK_BUTTON(setup->UI->fontButton),text);
	}
	if (header)
		setup->headerFontText=text;
	else
		setup->fontText=text;
}

void dialog_font_select(GtkButton *button, CountdownHomePlugin *setup)
{
	GtkWidget *fsd;
	gint res=0;
	fsd=gtk_font_selection_dialog_new("Select Font");
	gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(fsd),setup->fontText);
	gtk_widget_show_all(fsd);
	res=gtk_dialog_run(GTK_DIALOG(fsd));
	if (res==GTK_RESPONSE_OK)
	{
		gchar *temp=gtk_font_selection_dialog_get_font_name(GTK_FONT_SELECTION_DIALOG(fsd));
		set_font_button_text(setup,temp,FALSE);
		g_free(temp);
	}
	gtk_widget_destroy(fsd);
}

void dialog_header_font_select(GtkButton *button, CountdownHomePlugin *setup)
{
	GtkWidget *fsd;
	gint res;
	fsd=gtk_font_selection_dialog_new("Select Font");
	gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(fsd),setup->headerFontText);
	gtk_widget_show_all(fsd);
	res=gtk_dialog_run(GTK_DIALOG(fsd));
	if (res==GTK_RESPONSE_OK)
	{
		gchar *temp=gtk_font_selection_dialog_get_font_name(GTK_FONT_SELECTION_DIALOG(fsd));
		set_font_button_text(setup,temp,TRUE);
		g_free(temp);
	}
	gtk_widget_destroy(fsd);
}

void add_entry(CountdownHomePlugin *setup)
{
	GtkTreeModel *model=gtk_tree_view_get_model(setup->UI->view);
	GtkWindow *parent=GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(setup->UI->view)));
	GtkWidget *dialog=NULL;
	GtkWidget *caption;
	GtkWidget *titleEntry;
	GtkWidget *datePicker;
	GtkWidget *timePicker;
	GtkWidget *showYear,*showWeek,*showDay,*showHour,*showMinute;
	gint response;
	GtkWidget *scroll;
	GtkWidget *vbox,*hbox;
	GtkWidget *displayCheck,*displayNum;
	GtkWidget *sep;
	
	dialog=gtk_dialog_new_with_buttons("Add entry",parent,GTK_DIALOG_MODAL,GTK_STOCK_OK,GTK_RESPONSE_ACCEPT,GTK_STOCK_CANCEL,GTK_RESPONSE_REJECT,NULL);
	
	scroll=gtk_scrolled_window_new(NULL,NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
	
	vbox=gtk_vbox_new(FALSE,0);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll),vbox);
	
	titleEntry=gtk_entry_new();
	caption=hildon_caption_new(NULL,"Title",titleEntry,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(vbox),caption,FALSE,FALSE,0);
	
	datePicker=hildon_date_editor_new();
	caption=hildon_caption_new(NULL,"Date",datePicker,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(vbox),caption,FALSE,FALSE,0);
	
	timePicker=hildon_time_editor_new();
	caption=hildon_caption_new(NULL,"Time",timePicker,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(vbox),caption,FALSE,FALSE,0);
	
	sep=gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox),sep,FALSE,FALSE,0);
	
	hbox=gtk_hbox_new(FALSE,0);
	showYear=gtk_check_button_new_with_label("Show years?");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showYear),TRUE);
	gtk_box_pack_start(GTK_BOX(hbox),showYear,FALSE,FALSE,0);
	
	showWeek=gtk_check_button_new_with_label("Show weeks?");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showWeek),TRUE);
	gtk_box_pack_start(GTK_BOX(hbox),showWeek,FALSE,FALSE,0);
	gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
	
	showDay=gtk_check_button_new_with_label("Show days?");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showDay),TRUE);
	gtk_box_pack_start(GTK_BOX(vbox),showDay,FALSE,FALSE,0);
	
	hbox=gtk_hbox_new(FALSE,0);
	showHour=gtk_check_button_new_with_label("Show hours?");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showHour),TRUE);
	gtk_box_pack_start(GTK_BOX(hbox),showHour,FALSE,FALSE,0);
		
	showMinute=gtk_check_button_new_with_label("Show minutes?");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showMinute),TRUE);
	gtk_box_pack_start(GTK_BOX(hbox),showMinute,FALSE,FALSE,0);
	gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
	
	sep=gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox),sep,FALSE,FALSE,0);
	
	displayCheck=gtk_check_button_new_with_label("Hide this event until closer to the date");
	gtk_box_pack_start(GTK_BOX(vbox),displayCheck,FALSE,FALSE,0);
	hbox=gtk_hbox_new(FALSE,0);
	caption=gtk_label_new("Don't display until");
	gtk_box_pack_start(GTK_BOX(hbox),caption,FALSE,FALSE,0);
	displayNum=hildon_number_editor_new(1,365);
	gtk_box_pack_start(GTK_BOX(hbox),displayNum,FALSE,FALSE,0);
	caption=gtk_label_new("day(s) before");
	gtk_box_pack_start(GTK_BOX(hbox),caption,FALSE,FALSE,0);
	gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
		
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),scroll,TRUE,TRUE,0);
	
	gtk_widget_set_size_request(dialog,500,350);
	gtk_widget_show_all(dialog);
	response=gtk_dialog_run(GTK_DIALOG(dialog));
	if (response==GTK_RESPONSE_ACCEPT)
	{
		GtkTreeIter iter;
		const gchar *title;
		guint year, month, day;
		guint hours, minutes, seconds;
		gchar *date;
		gchar *time;
		gchar *readableTime;
		
		title=gtk_entry_get_text(GTK_ENTRY(titleEntry));
		
		hildon_date_editor_get_date(HILDON_DATE_EDITOR(datePicker),&year,&month,&day);
		date=g_strdup_printf("%02d/%02d/%02d",year,month,day);
		
		hildon_time_editor_get_time(HILDON_TIME_EDITOR(timePicker),&hours,&minutes,&seconds);
		readableTime=g_strdup_printf("%02d:%02d",hours,minutes);
		time=g_strdup_printf("%d",hours*(60*60)+minutes*60);
		
		gtk_list_store_append(GTK_LIST_STORE(model),&iter);
		gtk_list_store_set(GTK_LIST_STORE(model),&iter,0,title,1,date,2,time,3,readableTime,-1);
		gtk_list_store_set(GTK_LIST_STORE(model),&iter,4,gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(showYear)),-1);
		gtk_list_store_set(GTK_LIST_STORE(model),&iter,5,gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(showWeek)),-1);
		gtk_list_store_set(GTK_LIST_STORE(model),&iter,6,gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(showDay)),-1);
		gtk_list_store_set(GTK_LIST_STORE(model),&iter,7,gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(showHour)),-1);
		gtk_list_store_set(GTK_LIST_STORE(model),&iter,8,gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(showMinute)),-1);
		gtk_list_store_set(GTK_LIST_STORE(model),&iter,9,gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(displayCheck)),10,hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(displayNum)),-1);

		g_free(date);
		g_free(time);
		g_free(readableTime);
	}
	gtk_widget_destroy(dialog);
	
}

void modify_entry(CountdownHomePlugin *setup)
{
	GtkTreeSelection *select=gtk_tree_view_get_selection(setup->UI->view);
	GtkTreeModel *model=gtk_tree_view_get_model(setup->UI->view);
	GtkTreeIter iter;
	if (gtk_tree_selection_get_selected(select,&model,&iter))
	{
		GtkWindow *parent=GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(setup->UI->view)));
		GtkWidget *dialog=NULL;
		GtkWidget *caption;
		GtkWidget *titleEntry;
		GtkWidget *datePicker;
		GtkWidget *timePicker;
		gchar *tempTitle,*tempDate,*tempTime;
		GtkWidget *showYear,*showWeek,*showDay,*showHour,*showMinute;
		gint tempYear,tempMonth,tempDay;
		gboolean tempSY,tempSW,tempSD,tempSH,tempSM;
		gboolean tempDisplayCheck;
		gint tempDisplayNum;
		gint tempHour,tempMinute;
		gchar *endPtr;
		gint response;
		GtkWidget *scroll;
		GtkWidget *vbox,*hbox;
		GtkWidget *displayCheck,*displayNum;
		GtkWidget *sep;
		
		dialog=gtk_dialog_new_with_buttons("Modify entry",parent,GTK_DIALOG_MODAL,GTK_STOCK_OK,GTK_RESPONSE_ACCEPT,GTK_STOCK_CANCEL,GTK_RESPONSE_REJECT,NULL);
				
		scroll=gtk_scrolled_window_new(NULL,NULL);
		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
		
		vbox=gtk_vbox_new(FALSE,0);
		gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll),vbox);
		
		titleEntry=gtk_entry_new();
		caption=hildon_caption_new(NULL,"Title",titleEntry,NULL,HILDON_CAPTION_MANDATORY);
		gtk_box_pack_start(GTK_BOX(vbox),caption,FALSE,FALSE,0);
		
		datePicker=hildon_date_editor_new();
		caption=hildon_caption_new(NULL,"Date",datePicker,NULL,HILDON_CAPTION_MANDATORY);
		gtk_box_pack_start(GTK_BOX(vbox),caption,FALSE,FALSE,0);
		
		timePicker=hildon_time_editor_new();
		caption=hildon_caption_new(NULL,"Time",timePicker,NULL,HILDON_CAPTION_MANDATORY);
		gtk_box_pack_start(GTK_BOX(vbox),caption,FALSE,FALSE,0);
	
		sep=gtk_hseparator_new();
		gtk_box_pack_start(GTK_BOX(vbox),sep,FALSE,FALSE,0);
		
		hbox=gtk_hbox_new(FALSE,0);
		showYear=gtk_check_button_new_with_label("Show years?");
		gtk_box_pack_start(GTK_BOX(hbox),showYear,FALSE,FALSE,0);
		
		showWeek=gtk_check_button_new_with_label("Show weeks?");
		gtk_box_pack_start(GTK_BOX(hbox),showWeek,FALSE,FALSE,0);
		gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
		
		showDay=gtk_check_button_new_with_label("Show days?");
		gtk_box_pack_start(GTK_BOX(vbox),showDay,FALSE,FALSE,0);
		
		hbox=gtk_hbox_new(FALSE,0);
		showHour=gtk_check_button_new_with_label("Show hours?");
		gtk_box_pack_start(GTK_BOX(hbox),showHour,FALSE,FALSE,0);
			
		showMinute=gtk_check_button_new_with_label("Show minutes?");
		gtk_box_pack_start(GTK_BOX(hbox),showMinute,FALSE,FALSE,0);
		gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
	
		sep=gtk_hseparator_new();
		gtk_box_pack_start(GTK_BOX(vbox),sep,FALSE,FALSE,0);
			
		displayCheck=gtk_check_button_new_with_label("Hide this event until closer to the date");
		gtk_box_pack_start(GTK_BOX(vbox),displayCheck,FALSE,FALSE,0);
		hbox=gtk_hbox_new(FALSE,0);
		caption=gtk_label_new("Don't display until");
		gtk_box_pack_start(GTK_BOX(hbox),caption,FALSE,FALSE,0);
		displayNum=hildon_number_editor_new(1,365);
		gtk_box_pack_start(GTK_BOX(hbox),displayNum,FALSE,FALSE,0);
		caption=gtk_label_new("day(s) before");
		gtk_box_pack_start(GTK_BOX(hbox),caption,FALSE,FALSE,0);
		gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
	
		
		gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),scroll,TRUE,TRUE,0);
		

		gtk_tree_model_get(model,&iter,0,&tempTitle,1,&tempDate,3,&tempTime,4,&tempSY,5,&tempSW,6,&tempSD,7,&tempSH,8,&tempSM,9,&tempDisplayCheck,10,&tempDisplayNum,-1);
		gtk_entry_set_text(GTK_ENTRY(titleEntry),tempTitle);

		tempYear=g_ascii_strtoll(tempDate,&endPtr,10);
		tempDate=endPtr+1;
		tempMonth=g_ascii_strtoll(tempDate,&endPtr,10);
		tempDate=endPtr+1;
		tempDay=g_ascii_strtoll(tempDate,&endPtr,10);
		hildon_date_editor_set_date(HILDON_DATE_EDITOR(datePicker),tempYear,tempMonth,tempDay);
		
		tempHour=g_ascii_strtoll(tempTime,&endPtr,10);
		tempTime=endPtr+1;
		tempMinute=g_ascii_strtoll(tempTime,&endPtr,10);
		hildon_time_editor_set_time(HILDON_TIME_EDITOR(timePicker),tempHour,tempMinute,0);
		
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showYear),tempSY);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showWeek),tempSW);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showDay),tempSD);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showHour),tempSH);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(showMinute),tempSM);
		
		hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(displayNum),tempDisplayNum);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(displayCheck),tempDisplayCheck);

		gtk_widget_set_size_request(dialog,500,350);
		gtk_widget_show_all(dialog);
		response=gtk_dialog_run(GTK_DIALOG(dialog));

		if (response==GTK_RESPONSE_ACCEPT)
		{
			const gchar *title;
			guint year, month, day;
			guint hours, minutes, seconds;
			gchar *date;
			gchar *time;
			gchar *readableTime;
			
			title=gtk_entry_get_text(GTK_ENTRY(titleEntry));
			
			hildon_date_editor_get_date(HILDON_DATE_EDITOR(datePicker),&year,&month,&day);
			date=g_strdup_printf("%02d/%02d/%02d",year,month,day);
			
			hildon_time_editor_get_time(HILDON_TIME_EDITOR(timePicker),&hours,&minutes,&seconds);
			readableTime=g_strdup_printf("%02d:%02d",hours,minutes);
			time=g_strdup_printf("%d",hours*(60*60)+minutes*60);
			
			gtk_list_store_set(GTK_LIST_STORE(model),&iter,0,title,1,date,2,time,3,readableTime,-1);
			gtk_list_store_set(GTK_LIST_STORE(model),&iter,4,gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(showYear)),-1);
			gtk_list_store_set(GTK_LIST_STORE(model),&iter,5,gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(showWeek)),-1);
			gtk_list_store_set(GTK_LIST_STORE(model),&iter,6,gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(showDay)),-1);
			gtk_list_store_set(GTK_LIST_STORE(model),&iter,7,gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(showHour)),-1);
			gtk_list_store_set(GTK_LIST_STORE(model),&iter,8,gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(showMinute)),-1);
			gtk_list_store_set(GTK_LIST_STORE(model),&iter,9,gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(displayCheck)),10,hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(displayNum)),-1);

			g_free(date);
			g_free(time);
			g_free(readableTime);
		}
		gtk_widget_destroy(dialog);
	}
}

void delete_entry(CountdownHomePlugin *setup)
{
	GtkTreeSelection *select=gtk_tree_view_get_selection(setup->UI->view);
	GtkTreeModel *model=gtk_tree_view_get_model(setup->UI->view);
	GtkTreeIter iter;
	if (gtk_tree_selection_get_selected(select,&model,&iter))
	{
		gtk_list_store_remove(GTK_LIST_STORE(model),&iter);	
	}
}

void move_up(CountdownHomePlugin *setup)
{
	GtkTreeSelection *select=gtk_tree_view_get_selection(setup->UI->view);
	GtkTreeModel *model=gtk_tree_view_get_model(setup->UI->view);
	GtkTreeIter iterFirst,iterSecond;
	GtkTreePath *path;
	if (gtk_tree_selection_get_selected(select,&model,&iterFirst))
	{
		path=gtk_tree_model_get_path(model,&iterFirst);
		gtk_tree_path_prev(path);
		gtk_tree_model_get_iter(model,&iterSecond,path);
		gtk_list_store_swap(GTK_LIST_STORE(model),&iterFirst,&iterSecond);
		gtk_tree_view_scroll_to_cell(setup->UI->view,path,NULL,FALSE,0,0);
		gtk_tree_path_free(path);
	}
}

void move_down(CountdownHomePlugin *setup)
{
	GtkTreeSelection *select=gtk_tree_view_get_selection(setup->UI->view);
	GtkTreeModel *model=gtk_tree_view_get_model(setup->UI->view);
	GtkTreeIter iterFirst,iterSecond;
	GtkTreePath *path;
	if (gtk_tree_selection_get_selected(select,&model,&iterFirst))
	{
		iterSecond=iterFirst;
		if (gtk_tree_model_iter_next(model,&iterSecond))
		{
			gtk_list_store_swap(GTK_LIST_STORE(model),&iterFirst,&iterSecond);
			path=gtk_tree_model_get_path(model,&iterFirst);
			gtk_tree_view_scroll_to_cell(setup->UI->view,path,NULL,FALSE,0,0);
			gtk_tree_path_free(path);
		}
	}
}

static void setting_button_press(GtkButton *button, CountdownHomePlugin *setup)
{
	if (g_strcasecmp(gtk_widget_get_name(GTK_WIDGET(button)),"addButton")==0)
	{
		add_entry(setup);
	}
	else if (g_strcasecmp(gtk_widget_get_name(GTK_WIDGET(button)),"modifyButton")==0)
	{
		modify_entry(setup);
	}
	else if (g_strcasecmp(gtk_widget_get_name(GTK_WIDGET(button)),"deleteButton")==0)
	{
		delete_entry(setup);
	}
	else if (g_strcasecmp(gtk_widget_get_name(GTK_WIDGET(button)),"upButton")==0)
	{
		move_up(setup);
	}
	else if (g_strcasecmp(gtk_widget_get_name(GTK_WIDGET(button)),"downButton")==0)
	{
		move_down(setup);
	}
}

void update_content(CountdownHomePlugin *setup)
{
	countdown_content_create(setup);
}

static void on_countdown_settings(GtkWidget *widget, CountdownHomePlugin *setup)
{
	GtkWidget *dialog;
	GtkWidget *button=NULL;
	GtkWidget *label,*numberEditor;
	GtkWidget *settingHBox=NULL;
	GtkWidget *settingVBox=NULL;
	GtkWidget *smallHBox=NULL;
	GtkWidget *smallVBox=NULL;
	GtkWidget *notebook,*noteLabel;
	GtkWidget *yearEntry,*weekEntry,*dayEntry,*hourEntry,*minuteEntry;
	GtkWidget *passedEntry,*remainingEntry;
	GtkWidget *defaultRegular, *defaultHeading;
	GtkWidget *scroll,*fontScroll,*viewScroll;
	GtkWidget *hidePassed;
	gint result=0;
	GtkSizeGroup *group;
	GtkWidget *frame,*fontFrame;
	GtkWidget *align;
	GtkWidget *regularBox,*headerBox;
	GdkColor colour;
	char s[5]="";
	guint16 x;
	
	/* Create the widgets */
	dialog = gtk_dialog_new_with_buttons("Countdown Settings",setup->parent, GTK_DIALOG_DESTROY_WITH_PARENT,GTK_STOCK_OK,GTK_RESPONSE_ACCEPT,NULL);

	setup->parent=NULL;
	notebook=gtk_notebook_new();
	noteLabel=gtk_label_new("Arrange Items");

	settingHBox=gtk_hbox_new(FALSE,0);

	viewScroll = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(viewScroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(settingHBox),viewScroll,TRUE,TRUE,0);

	setup->UI->view = GTK_TREE_VIEW(gtk_tree_view_new_with_model(GTK_TREE_MODEL(setup->entries)));
	gtk_tree_view_append_column(setup->UI->view,gtk_tree_view_column_new_with_attributes(NULL,gtk_cell_renderer_text_new(),"text",0,NULL));
	gtk_tree_view_append_column(setup->UI->view,gtk_tree_view_column_new_with_attributes(NULL,gtk_cell_renderer_text_new(),"text",1,NULL));
	gtk_tree_view_append_column(setup->UI->view,gtk_tree_view_column_new_with_attributes(NULL,gtk_cell_renderer_text_new(),"text",3,NULL));
	
	gtk_tree_selection_set_mode(gtk_tree_view_get_selection(setup->UI->view),GTK_SELECTION_BROWSE);

	gtk_container_add(GTK_CONTAINER(viewScroll),GTK_WIDGET(setup->UI->view));
	gtk_widget_set_size_request(viewScroll,-1,250);

	settingVBox=gtk_vbox_new(FALSE,0);
	gtk_box_pack_start(GTK_BOX(settingHBox),settingVBox,FALSE,FALSE,0);
	
	smallHBox=gtk_hbox_new(TRUE,0);
	gtk_box_pack_start(GTK_BOX(settingVBox),smallHBox,TRUE,TRUE,0);
	
	button=gtk_button_new_with_label("Edit");
	gtk_widget_set_name(button,"modifyButton");
	gtk_box_pack_start(GTK_BOX(smallHBox),button,TRUE,TRUE,0);
	g_signal_connect(button,"clicked",G_CALLBACK(setting_button_press),setup);

	button=gtk_button_new_with_label("Add");
	gtk_widget_set_name(button,"addButton");
	gtk_box_pack_start(GTK_BOX(smallHBox),button,TRUE,TRUE,0);
	g_signal_connect(button,"clicked",G_CALLBACK(setting_button_press),setup);
	
	button=gtk_button_new_with_label("Delete");
	gtk_widget_set_name(button,"deleteButton");
	gtk_box_pack_start(GTK_BOX(smallHBox),button,TRUE,TRUE,0);
	g_signal_connect(button,"clicked",G_CALLBACK(setting_button_press),setup);

	smallHBox=gtk_hbox_new(FALSE,0);
	gtk_box_pack_start(GTK_BOX(settingVBox),smallHBox,TRUE,TRUE,0);
	
	button = gtk_button_new();
	gtk_widget_set_name(button,"upButton");
	gtk_button_set_image(GTK_BUTTON(button), gtk_image_new_from_icon_name("qgn_indi_arrow_up", GTK_ICON_SIZE_BUTTON));
	gtk_box_pack_start(GTK_BOX(smallHBox),button,TRUE,TRUE,0);
	g_signal_connect(button,"clicked",G_CALLBACK(setting_button_press),setup);
	
	button = gtk_button_new();
	gtk_widget_set_name(button,"downButton");
	gtk_button_set_image(GTK_BUTTON(button), gtk_image_new_from_icon_name("qgn_indi_arrow_down", GTK_ICON_SIZE_BUTTON));
	gtk_box_pack_start(GTK_BOX(smallHBox),button,TRUE,TRUE,0);
	g_signal_connect(button,"clicked",G_CALLBACK(setting_button_press),setup);
	
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook),settingHBox,noteLabel);

	fontScroll=gtk_scrolled_window_new(NULL,NULL);

	noteLabel=gtk_label_new("Settings");
	settingVBox=gtk_vbox_new(FALSE,0);
	
	frame=gtk_frame_new("Changing requires a reboot");
	gtk_box_pack_start(GTK_BOX(settingVBox),frame,FALSE,FALSE,0);
	
	smallVBox=gtk_vbox_new(FALSE,0);
	gtk_container_add(GTK_CONTAINER(frame),smallVBox);

	numberEditor=hildon_number_editor_new(0,60);
	label=hildon_caption_new(NULL,"Minutes between updates",numberEditor,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(smallVBox),label,FALSE,FALSE,0);
	
	label=gtk_label_new("'0' will refresh only when focus is changed to the desktop");
	gtk_label_set_markup(GTK_LABEL(label),"<small>'0' will refresh only when focus is changed to the desktop</small>");
	align=gtk_alignment_new(0,0.5,0,0);
	gtk_container_add(GTK_CONTAINER(align),label);
	gtk_box_pack_start(GTK_BOX(smallVBox),align,FALSE,FALSE,0);
	
	frame=gtk_frame_new("Passed entries");
	gtk_box_pack_start(GTK_BOX(settingVBox),frame,FALSE,FALSE,0);
	
	hidePassed=gtk_check_button_new_with_label("Hide entries passed date");
	gtk_container_add(GTK_CONTAINER(frame),hidePassed);
	
	fontFrame=gtk_frame_new("Font and colour options");
	gtk_box_pack_start(GTK_BOX(settingVBox),fontFrame,FALSE,FALSE,0);
	
	smallVBox=gtk_vbox_new(FALSE,0);
	gtk_container_add(GTK_CONTAINER(fontFrame),smallVBox);
	regularBox=gtk_vbox_new(FALSE,0);
	gtk_box_pack_start(GTK_BOX(smallVBox),regularBox,FALSE,FALSE,0);
	
	defaultRegular=gtk_check_button_new_with_label("Use system font style for counter text");
	gtk_box_pack_start(GTK_BOX(regularBox),defaultRegular,FALSE,FALSE,0);
	g_signal_connect(G_OBJECT(defaultRegular),"toggled",G_CALLBACK(system_check_toggle),setup);

	smallHBox=gtk_hbox_new(FALSE,0);	
	setup->UI->fontButton=gtk_button_new_with_label("Select");
	g_signal_connect(G_OBJECT(setup->UI->fontButton),"clicked",G_CALLBACK(dialog_font_select),setup);
	label=hildon_caption_new(NULL,"Font Style",setup->UI->fontButton,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(regularBox),label,FALSE,FALSE,0);
	
	setup->UI->colourButton=gtk_color_button_new();
	label=hildon_caption_new(NULL,"Font Colour",setup->UI->colourButton,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(regularBox),label,FALSE,FALSE,0);
	
	headerBox=gtk_vbox_new(FALSE,0);
	gtk_box_pack_start(GTK_BOX(smallVBox),headerBox,FALSE,FALSE,0);
	
	defaultHeading=gtk_check_button_new_with_label("Use system font style for headings");
	gtk_box_pack_start(GTK_BOX(headerBox),defaultHeading,FALSE,FALSE,0);
	g_signal_connect(G_OBJECT(defaultHeading),"toggled",G_CALLBACK(header_system_check_toggle),setup);
	
	smallHBox=gtk_hbox_new(FALSE,0);	
	setup->UI->headerFontButton=gtk_button_new_with_label("Select");
	g_signal_connect(G_OBJECT(setup->UI->headerFontButton),"clicked",G_CALLBACK(dialog_header_font_select),setup);
	label=hildon_caption_new(NULL,"Font Style",setup->UI->headerFontButton,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(headerBox),label,FALSE,FALSE,0);
	
	setup->UI->headerColourButton=gtk_color_button_new();
	label=hildon_caption_new(NULL,"Font Colour",setup->UI->headerColourButton,NULL,HILDON_CAPTION_MANDATORY);
	gtk_box_pack_start(GTK_BOX(headerBox),label,FALSE,FALSE,0);
	
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(fontScroll),GTK_POLICY_NEVER,GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(fontScroll),settingVBox);
	
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook),fontScroll,noteLabel);
	
	scroll=gtk_scrolled_window_new(NULL,NULL);
	
	noteLabel=gtk_label_new("Translation");
	settingVBox=gtk_vbox_new(FALSE,0);
	
	group=gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);
	
	frame=gtk_frame_new("Change how the following terms are displayed");
	gtk_box_pack_start(GTK_BOX(settingVBox),frame,FALSE,FALSE,0);
	
	smallVBox=gtk_table_new(4,2,FALSE);
	gtk_container_add(GTK_CONTAINER(frame),smallVBox);
	
	yearEntry=gtk_entry_new();
	label=hildon_caption_new(group,"\"Years\"",yearEntry,NULL,HILDON_CAPTION_MANDATORY);
	gtk_table_attach_defaults(GTK_TABLE(smallVBox),label,0,1,0,1);
	
	weekEntry=gtk_entry_new();
	label=hildon_caption_new(group,"\"Weeks\"",weekEntry,NULL,HILDON_CAPTION_MANDATORY);
	gtk_table_attach_defaults(GTK_TABLE(smallVBox),label,1,2,0,1);
	
	dayEntry=gtk_entry_new();
	label=hildon_caption_new(group,"\"Days\"",dayEntry,NULL,HILDON_CAPTION_MANDATORY);
	gtk_table_attach_defaults(GTK_TABLE(smallVBox),label,0,1,1,2);
	
	hourEntry=gtk_entry_new();
	label=hildon_caption_new(group,"\"Hours\"",hourEntry,NULL,HILDON_CAPTION_MANDATORY);
	gtk_table_attach_defaults(GTK_TABLE(smallVBox),label,0,1,2,3);;
	
	minuteEntry=gtk_entry_new();
	label=hildon_caption_new(group,"\"Minutes\"",minuteEntry,NULL,HILDON_CAPTION_MANDATORY);
	gtk_table_attach_defaults(GTK_TABLE(smallVBox),label,1,2,2,3);
	
	remainingEntry=gtk_entry_new();
	label=hildon_caption_new(group,"\"Remaining\"",remainingEntry,NULL,HILDON_CAPTION_MANDATORY);
	gtk_table_attach_defaults(GTK_TABLE(smallVBox),label,0,1,3,4);

	passedEntry=gtk_entry_new();
	label=hildon_caption_new(group,"\"Passed\"",passedEntry,NULL,HILDON_CAPTION_MANDATORY);
	gtk_table_attach_defaults(GTK_TABLE(smallVBox),label,1,2,3,4);
	
	//gtk_widget_set_size_request(GTK_WIDGET(scroll), -1, 250);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),GTK_POLICY_NEVER,GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll),settingVBox);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook),scroll,noteLabel);

	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),notebook,FALSE,FALSE,0);

	hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(numberEditor),setup->refresh);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(hidePassed),setup->hidePassed);
	gtk_entry_set_text(GTK_ENTRY(yearEntry),setup->yearString);
	gtk_entry_set_text(GTK_ENTRY(weekEntry),setup->weekString);
	gtk_entry_set_text(GTK_ENTRY(dayEntry),setup->dayString);
	gtk_entry_set_text(GTK_ENTRY(hourEntry),setup->hourString);
	gtk_entry_set_text(GTK_ENTRY(minuteEntry),setup->minuteString);
	gtk_entry_set_text(GTK_ENTRY(passedEntry),setup->passedString);
	gtk_entry_set_text(GTK_ENTRY(remainingEntry),setup->remainingString);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(defaultRegular),setup->fontCheck);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(defaultHeading),setup->headerFontCheck);
	strncat(s,setup->colourText,strlen(setup->colourText)/3);
	x=strtol(s,NULL,16);
	colour.red=x;
	s[0]=NULL;
	strncat(s,setup->colourText+strlen(setup->colourText)/3,strlen(setup->colourText)/3);
	x=strtol(s,NULL,16);
	colour.green=x;
	s[0]=NULL;
	strncat(s,setup->colourText+(strlen(setup->colourText)/3)*2,strlen(setup->colourText)/3);
	x=strtol(s,NULL,16);
	colour.blue=x;
	gtk_color_button_set_color(GTK_COLOR_BUTTON(setup->UI->colourButton),&colour);
	strncat(s,setup->headerColourText,strlen(setup->headerColourText)/3);
	x=strtol(s,NULL,16);
	colour.red=x;
	s[0]=NULL;
	strncat(s,setup->headerColourText+strlen(setup->headerColourText)/3,strlen(setup->headerColourText)/3);
	x=strtol(s,NULL,16);
	colour.green=x;
	s[0]=NULL;
	strncat(s,setup->headerColourText+(strlen(setup->headerColourText)/3)*2,strlen(setup->headerColourText)/3);
	x=strtol(s,NULL,16);
	colour.blue=x;
	gtk_color_button_set_color(GTK_COLOR_BUTTON(setup->UI->headerColourButton),&colour);
	set_font_button_text(setup,setup->fontText,FALSE);
	set_font_button_text(setup,setup->headerFontText,TRUE);
	
	gtk_window_set_modal(GTK_WINDOW(dialog),TRUE);
	gtk_widget_show_all (GTK_WIDGET(dialog));
	system_check_toggle(GTK_TOGGLE_BUTTON(defaultRegular),setup);
	gtk_container_check_resize(GTK_CONTAINER(regularBox));
	header_system_check_toggle(GTK_TOGGLE_BUTTON(defaultHeading),setup);
	gtk_container_check_resize(GTK_CONTAINER(headerBox));
	gtk_container_check_resize(GTK_CONTAINER(fontFrame));
	gtk_container_check_resize(GTK_CONTAINER(fontScroll));
	result=gtk_dialog_run(GTK_DIALOG(dialog));
	if (result==GTK_RESPONSE_ACCEPT)
	{
		GKeyFile *keyFile;
		gchar* fileData;
		FILE* rcFile;
		gsize size;
		GtkTreeIter iter;
		GtkTreeModel *model=gtk_tree_view_get_model(setup->UI->view);
		gboolean exists;
		gchar *filename;
		gint i=0;
		GdkColor colour;

		keyFile=g_key_file_new();
		exists=gtk_tree_model_get_iter_first(model,&iter);
		while (exists)
		{
			gchar *title,*date,*time;
			gboolean showWhich[5];
			gboolean displayCheck;
			gint displayNum;
			gchar entry[5];
			snprintf(entry,5,"%d",i++);
			gtk_tree_model_get(model,&iter,0,&title,1,&date,2,&time,4,&showWhich[0],5,&showWhich[1],6,&showWhich[2],7,&showWhich[3],8,&showWhich[4],9,&displayCheck,10,&displayNum,-1);
			g_key_file_set_string(keyFile,entry,"title",title);
			g_key_file_set_string(keyFile,entry,"date",date);
			g_key_file_set_string(keyFile,entry,"time",time);
			g_key_file_set_boolean_list(keyFile,entry,"show",showWhich,5);
			g_key_file_set_boolean(keyFile,entry,"display check",displayCheck);
			g_key_file_set_integer(keyFile,entry,"display num",displayNum);
			exists=gtk_tree_model_iter_next(model,&iter);
			g_free(title);
			g_free(date);
			g_free(time);
		}
		
		g_key_file_set_integer(keyFile,"config","refresh",hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(numberEditor)));
		setup->refresh=hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(numberEditor));
		g_key_file_set_boolean(keyFile,"config","hide passed",gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(hidePassed)));
		setup->hidePassed=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(hidePassed));
		g_key_file_set_string(keyFile,"config","years label",gtk_entry_get_text(GTK_ENTRY(yearEntry)));
		setup->yearString=g_strdup(gtk_entry_get_text(GTK_ENTRY(yearEntry)));
		g_key_file_set_string(keyFile,"config","weeks label",gtk_entry_get_text(GTK_ENTRY(weekEntry)));
		setup->weekString=g_strdup(gtk_entry_get_text(GTK_ENTRY(weekEntry)));
		g_key_file_set_string(keyFile,"config","days label",gtk_entry_get_text(GTK_ENTRY(dayEntry)));
		setup->dayString=g_strdup(gtk_entry_get_text(GTK_ENTRY(dayEntry)));
		g_key_file_set_string(keyFile,"config","hours label",gtk_entry_get_text(GTK_ENTRY(hourEntry)));
		setup->hourString=g_strdup(gtk_entry_get_text(GTK_ENTRY(hourEntry)));
		g_key_file_set_string(keyFile,"config","minutes label",gtk_entry_get_text(GTK_ENTRY(minuteEntry)));
		setup->minuteString=g_strdup(gtk_entry_get_text(GTK_ENTRY(minuteEntry)));
		g_key_file_set_string(keyFile,"config","remaining label",gtk_entry_get_text(GTK_ENTRY(remainingEntry)));
		setup->remainingString=g_strdup(gtk_entry_get_text(GTK_ENTRY(remainingEntry)));
		g_key_file_set_string(keyFile,"config","passed label",gtk_entry_get_text(GTK_ENTRY(passedEntry)));
		setup->passedString=g_strdup(gtk_entry_get_text(GTK_ENTRY(passedEntry)));
		g_key_file_set_boolean(keyFile,"config","system style",gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(defaultRegular)));
		setup->fontCheck=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(defaultRegular));
		g_key_file_set_string(keyFile,"config","font style",setup->fontText);
		gtk_color_button_get_color(GTK_COLOR_BUTTON(setup->UI->colourButton),&colour);
		setup->colourText=g_strdup_printf("%04X%04X%04X",colour.red,colour.green,colour.blue);
		g_key_file_set_string(keyFile,"config","font colour",setup->colourText);
		g_key_file_set_boolean(keyFile,"config","header system style",gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(defaultHeading)));
		setup->headerFontCheck=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(defaultHeading));
		g_key_file_set_string(keyFile,"config","header font style",setup->headerFontText);
		gtk_color_button_get_color(GTK_COLOR_BUTTON(setup->UI->headerColourButton),&colour);
		setup->headerColourText=g_strdup_printf("%04X%04X%04X",colour.red,colour.green,colour.blue);
		g_key_file_set_string(keyFile,"config","header font colour",setup->headerColourText);
		
		filename=g_strdup(HOME_DIR);
		filename=g_strconcat(filename,"/.countdown_home.rc",NULL);
		fileData=g_key_file_to_data(keyFile,&size,NULL);
		rcFile=fopen(filename,"w");
		fputs(fileData,rcFile);
		fclose(rcFile);
		g_key_file_free(keyFile);
		g_free(fileData);
		g_free(filename);
	}
	gtk_widget_destroy(dialog);
	update_content(setup);
}

void *hildon_home_applet_lib_initialize (void *state_data,int *state_size,GtkWidget **widget)
{
	g_debug("Starting countdown-home");
	g_type_init();
	
	CountdownHomePlugin *setup=g_new0(CountdownHomePlugin,1);
	CountdownUI *UI=g_new0(CountdownUI,1);
	
	setup->UI=UI;
	
	setup->osso = osso_initialize ("countdown", "0.2", FALSE, NULL);
	if (!setup->osso) {
		g_debug ("Error initializing the osso countdown home plugin");
		return NULL;
	}

	setup_make_view(setup);
	g_object_ref(setup->entries);

	setup->mainwidget = gtk_frame_new(NULL);
	gtk_widget_set_name (setup->mainwidget, "osso-speeddial");
	*widget = setup->mainwidget;
  
  	gtk_container_set_border_width (GTK_CONTAINER (setup->mainwidget), 0);

  	GtkWidget* alignment = gtk_alignment_new (0.5,0.5,1.0,1.0);

  	gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 15, 15, 15, 15);

	setup->scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(setup->scrolled_window),GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (setup->mainwidget), alignment);
	gtk_container_add( GTK_CONTAINER( alignment ), setup->scrolled_window );

	gtk_widget_show_all(GTK_WIDGET(setup->mainwidget));
	countdown_content_create(setup);
	
	if (setup->refresh>0)
		setup->eventID=g_timeout_add(setup->refresh*1000*60,(GSourceFunc)update_content,setup);
	return (void*)setup;
}

void hildon_home_applet_lib_deinitialize (CountdownHomePlugin *setup)
{
	if (setup->osso)
		osso_deinitialize (setup->osso);
	if (setup->eventID)
		g_source_remove(setup->eventID);
    /*if (setup->mainwidget) g_free(setup->mainwidget);
	setup->mainwidget=NULL;
	g_object_unref(setup->entries);*/
}

int hildon_home_applet_lib_save_state (CountdownHomePlugin *setup,void **state_data,int *state_size)
{
    (*state_data) = NULL;
   
    if (state_size) {
        (*state_size) = 0; 
    } else {
        g_debug ("State_size pointer was not pointing to right place");
    }
    
    return 1;
}

void hildon_home_applet_lib_background (CountdownHomePlugin *setup)
{
	if (setup->eventID)
		g_source_remove(setup->eventID);
	return;
}

void hildon_home_applet_lib_foreground (CountdownHomePlugin *setup)
{
	countdown_content_create(setup);
	if (setup->refresh>0)
		setup->eventID=g_timeout_add(setup->refresh*1000*60,(GSourceFunc)update_content,setup);
	return;
}

GtkWidget *hildon_home_applet_lib_settings (CountdownHomePlugin *setup, GtkWindow *parent)
{
	GtkWidget *settings;
	setup->parent=parent;

	settings = gtk_menu_item_new_with_label(("Countdown"));
	g_signal_connect (settings, "activate", G_CALLBACK (on_countdown_settings), setup);

	return settings;
	return NULL;
}
