/*
 * This file is part of connection-switcher
 * Copyright (C) 2008 Kate Alhola
 * Copyright (C) 2009 Kyller Gorgonio
 * Copyright (C) 2009 Marcos Morais
 * Copyright (C) 2009 Walter Guerra
 * Copyright (C) 2009 Nokia Corporation. All rights reserved.
 *
 * portions of code from connection-switch-plugin
 * Copyright (C) 2008 Kate Alhola
 *
 * Authors: Kyller Gorgonio <kyllercg@gmail.com>
 *          Marcos Morais <morais@embedded.ufcg.edu.br>
 *          Walter Guerra <walter.guerra@signove.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 */

/* Hildon includes */
#include <hildon/hildon-note.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-sound.h>
#include <hildon/hildon-defines.h>
#include <libhildondesktop/libhildondesktop.h>

#include <log-functions.h>
#include <libosso.h>
#include <osso-log.h>

/* GTK includes */
#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkpixbuf.h>

/* Systems includes */
#include <string.h>



#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <nm_wlan_utils.h>
#include <nm_bt_utils.h>
#include <nm_settings.h>
#include <nm_environment_manager.h>
#include <nm_usb_setup.h>
#include <nm_bt_setup.h>
#include <nm_wlan_setup.h>
#include <nm_sbrsh_setup.h>
#include <nm_unfs_setup.h>
#include <nm_vnc_setup.h>

#include "connection-switcher-statusbar.h"

HD_DEFINE_PLUGIN (CsStatusBar, cs_status_bar, STATUSBAR_TYPE_ITEM)

#define CS_STATUS_BAR_GET_PRIVATE(x) \
    (G_TYPE_INSTANCE_GET_PRIVATE((x), CS_STATUS_BAR_TYPE, CsStatusBarPrivate))

typedef struct _CsStatusBarPrivate CsStatusBarPrivate;

struct _CsStatusBarPrivate
{
    osso_context_t *osso;           /* osso */
    GtkWidget *icon;                /* icon button */
    GtkWidget *button;              /* button StatusBar */

    // The menu and menu options
    GtkWidget *menu;
    GtkWidget *menu_usb_storage;    /* USB file storage */
    GtkWidget *menu_usb_netdev;     /* USB network */
    GtkWidget *menu_bt_netdev;      /* Bluetooth network */
    GtkWidget *menu_wlan_netdev;    /* Wlan adhoc network */

    // The icon
    GdkPixbuf *pixbuf;
    GdkPixbuf *pixbuf_host;
    gboolean button_released;

    // To control active connections
    gboolean usb_storage;
    gboolean usb_netdev;
    gboolean bt_netdev;
    gboolean wlan_netdev;

    // This is for bluetooth search and connect
    NmBtUtils *bt_utils;
    gchar *bt_addr;
    
    GtkWidget *environments; /* Environments submenu */
};

static CsStatusBarPrivate *options;

gboolean USB_FLAG;
gboolean BT_FLAG;
gboolean WLAN_FLAG;

static void cs_status_bar_class_init (CsStatusBarClass *);
static void cs_status_bar_init (CsStatusBar *);
static void cs_status_bar_finalize (GObject *);

static void cs_set_icon (const gchar *);
static void cs_menu_setup (void);
static void cs_enviroment_menu_setup ();
static void cs_environment_change(GtkWidget *, gpointer);

static void cs_icon_pressed (GtkWidget *, gpointer);
static void cs_activate_usb_storage (GtkWidget *, gpointer);
static void cs_activate_usb_netdev (GtkWidget *, gpointer);
static void cs_activate_bt_netdev (GtkWidget *, gpointer);
static void cs_activate_wlan_netdev (GtkWidget *, gpointer);

static void
cs_status_bar_class_init (CsStatusBarClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);
    object_class->finalize = cs_status_bar_finalize;
    g_type_class_add_private (klass, sizeof(CsStatusBarPrivate));
}

static void
cs_status_bar_init (CsStatusBar *self)
{
    CsStatusBarPrivate *priv = CS_STATUS_BAR_GET_PRIVATE (self);

    USB_FLAG = TRUE;
    BT_FLAG = TRUE;
    WLAN_FLAG = TRUE;

    options = priv;  // Save static pointer
    ULOG_OPEN("connection-switcher-statusbar");
    ULOG_WARN ("%s: initializing connection-switcher\n", __FUNCTION__);

    g_return_if_fail (priv);

    priv->icon = gtk_image_new_from_pixbuf (NULL);
    priv->button = gtk_toggle_button_new ();

    /* Init BT data, move to BT stuff later */
    priv->bt_addr = NULL;
    priv->bt_utils = nm_bt_utils_new ();

    cs_set_icon ("connection-switcher");

    gtk_container_add (GTK_CONTAINER (priv->button), GTK_WIDGET (priv->icon));
    gtk_container_add (GTK_CONTAINER (self), priv->button);

    /* Signal for icon (button) */
    g_signal_connect (G_OBJECT (priv->button), "button-press-event",
                       G_CALLBACK(cs_icon_pressed), NULL);

    /* Initialize osso */
    priv->osso = osso_initialize ("connection_switcher_statusbar", "0.1",
                                   FALSE, NULL);

    if (!priv->osso)
        ULOG_WARN ("%s: error while initializing osso\n", __FUNCTION__);

    gtk_widget_show_all (GTK_WIDGET (self));
}

static void
cs_status_bar_finalize (GObject *object)
{
    CsStatusBarPrivate *priv = CS_STATUS_BAR_GET_PRIVATE (object);

    ULOG_WARN ("%s: finalizing connection-switcher\n", __FUNCTION__);

    g_free (priv->bt_addr);
    g_object_unref (priv->bt_utils);

    osso_deinitialize (priv->osso);

    LOG_CLOSE ();

    G_OBJECT_CLASS (g_type_class_peek_parent (G_OBJECT_GET_CLASS (object)))->finalize (object);
}

static void
cs_set_icon (const gchar *name)
{
    GtkIconTheme *icon_theme;

    icon_theme = gtk_icon_theme_get_default();

    options->pixbuf = (name != NULL) ?
            gtk_icon_theme_load_icon (icon_theme, name,
                                       CS_STATUS_BAR_ICON_SIZE,
                                       GTK_ICON_LOOKUP_NO_SVG, NULL) : NULL;

    gtk_image_set_from_pixbuf (GTK_IMAGE (options->icon), options->pixbuf);
}

static void
cs_enviroment_menu_setup()
{
        openlog("Walter", LOG_NDELAY, LOG_USER);
        syslog(LOG_ALERT, "cs_environment_menu_setup");
               
        
        if (options->environments != NULL) {
            gtk_container_remove(GTK_CONTAINER(options->menu), options->environments );
        }
        
        /* Put the environments at the menu*/
        GtkWidget *env_menu = gtk_menu_new();
        
        // Envs
        options->environments = gtk_menu_item_new_with_label ("Environments");
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(options->environments), env_menu);
        
        // Iterate to get all the available environments
        GValue vt = {0,};
        g_value_init (&vt, G_TYPE_BOOLEAN);
        g_value_set_boolean (&vt, TRUE);

        gchar *current = nm_environment_manager_get_active();
        g_debug("[%s] - Active env: %s", __FUNCTION__, current);

        /* Add the None */        
        GtkWidget *none = gtk_check_menu_item_new_with_label(NM_ENVIRONMENT_MANAGER_NONE);
        gtk_menu_shell_append (GTK_MENU_SHELL (env_menu),
                                none);
        g_signal_connect (G_OBJECT (none), "activate",
                            G_CALLBACK (cs_environment_change), NULL); 
                                   
        if ( !strcmp(NM_ENVIRONMENT_MANAGER_NONE, current) ) {
            g_object_set_property (G_OBJECT (none),
                                    "active", &vt);
        }
        
        gchar **list;
        int i;
        gsize length = 0;
        
        list = nm_environment_manager_get_list(&length);
        syslog(LOG_ALERT, "size: %d", length);
        GtkWidget *items[length];
        for(i = 0; i < length; i++) {
            items[i] = gtk_check_menu_item_new_with_label(list[i]);
            gtk_menu_shell_append (GTK_MENU_SHELL (env_menu),
                                    items[i]);
            g_signal_connect (G_OBJECT (items[i]), "activate",
                              G_CALLBACK (cs_environment_change), NULL);
                              
            // Mark the current one
            if ( !strcmp(list[i], current) ) {
                g_object_set_property (G_OBJECT (items[i]),
                                        "active", &vt);
            }
        }
        
        // Add the submenu to the menu
        gtk_menu_shell_prepend (GTK_MENU_SHELL (options->menu),
                                options->environments);

    
        gtk_widget_show_all(options->environments);
        closelog(); 
}

static void
cs_environment_change(GtkWidget *widget, gpointer data)
{
    
    gchar *clicked_env = gtk_label_get_text( GTK_LABEL(GTK_BIN(widget)->child) );
    gchar *active_env = nm_environment_manager_get_active();
    
    if (strcmp(clicked_env, active_env)) {
        gboolean error = FALSE;
        
        if ( !nm_usb_setup_activate_environment(clicked_env) ) {
            g_debug("Error applying usb settings");
            error = TRUE;
        }
        
        if ( !nm_bt_setup_activate_environment(clicked_env) ) {
            g_debug("Error applying bt settings");
            error = TRUE;
        }
        
        if ( !nm_wlan_setup_activate_environment(clicked_env) ) {
            g_debug("Error applying wlan settings");
            error = TRUE;
        }
        
        if ( !nm_sbrsh_setup_activate_environment(clicked_env) ) {
            g_debug("Error applying sbrsh settings");
            error = TRUE;
        }

        if ( !nm_unfs_setup_activate_environment(clicked_env) ) {
            g_debug("Error applying unfs settings");
            error = TRUE;
        }

        if ( !nm_vnc_setup_activate_environment(clicked_env) ) {
            g_debug("Error applying x11vnc settings");
            error = TRUE;
        }
        
        nm_environment_manager_set_active(clicked_env);
        
        gchar message[80];
        if (error == FALSE) {
            sprintf(message, "Environment %s applied successfully", clicked_env);
        }
        else {
            sprintf(message, "Environment %s applied with errors", clicked_env);
        }
        
        g_debug("message: %s", message);
        hildon_banner_show_information(NULL, NULL, message);
        
    }
}

static void
cs_menu_setup (void)
{
    if (!GTK_IS_MENU (options->menu)) {

        options->menu = gtk_menu_new ();
        
        cs_enviroment_menu_setup();
                                
        // Add the separator
        GtkWidget *sep = gtk_separator_menu_item_new();
        gtk_menu_shell_append (GTK_MENU_SHELL (options->menu),
                                sep);                                                                           

        /* USB file storage mode */
        options->menu_usb_storage =
                gtk_check_menu_item_new_with_label ("USB file storage");
        gtk_menu_shell_append (GTK_MENU_SHELL (options->menu),
                                options->menu_usb_storage);
        g_signal_connect (G_OBJECT (options->menu_usb_storage), "activate",
                           G_CALLBACK (cs_activate_usb_storage), NULL);

        /* USB network device mode */
        options->menu_usb_netdev =
                gtk_check_menu_item_new_with_label ("USB network");
        gtk_menu_shell_append (GTK_MENU_SHELL (options->menu),
                                options->menu_usb_netdev);
        g_signal_connect (G_OBJECT (options->menu_usb_netdev), "activate",
                           G_CALLBACK (cs_activate_usb_netdev), NULL);

        /* Bluetooth network mode */
        options->menu_bt_netdev =
                gtk_check_menu_item_new_with_label ("Bluetooth network");
        gtk_menu_shell_append (GTK_MENU_SHELL (options->menu),
                                options->menu_bt_netdev);
        g_signal_connect (G_OBJECT (options->menu_bt_netdev), "activate",
                           G_CALLBACK (cs_activate_bt_netdev), NULL);

        /* WLAN Ad-Hoc network mode */
        options->menu_wlan_netdev =
                gtk_check_menu_item_new_with_label ("Wlan adhoc network");
        gtk_menu_shell_append (GTK_MENU_SHELL (options->menu),
                                options->menu_wlan_netdev);
        g_signal_connect (G_OBJECT (options->menu_wlan_netdev), "activate",
                           G_CALLBACK (cs_activate_wlan_netdev), NULL);

        gtk_widget_show_all (options->menu);
    }
}

/* CALLBACKS */
static void
cs_icon_pressed (GtkWidget *widget, gpointer data)
{
    GValue vt = {0,};
    GValue vf = {0,};

    g_value_init (&vt, G_TYPE_BOOLEAN);
    g_value_init (&vf, G_TYPE_BOOLEAN);

    g_value_set_boolean (&vt, TRUE);
    g_value_set_boolean (&vf, FALSE);

    /* Create menu */
    cs_menu_setup ();
    cs_enviroment_menu_setup();

    options->usb_storage = (system("/usr/sbin/nm-usb-setup status") != 0)? TRUE : FALSE;
    options->usb_netdev = (system("/usr/sbin/nm-usb-setup status") == 0)? TRUE : FALSE;
    options->bt_netdev =
            (nm_bt_utils_radio_is_enabled (options->bt_utils) &&
              nm_bt_utils_has_network_connection (options->bt_utils));
    options->wlan_netdev = nm_wlan_utils_is_active ();

    if (options->usb_netdev) {

        USB_FLAG = FALSE;
        g_object_set_property (G_OBJECT (options->menu_usb_storage),
                                "active", &vf);
        g_object_set_property (G_OBJECT (options->menu_usb_netdev),
                                "active", &vt);
        USB_FLAG = TRUE;
    } else if (options->usb_storage) {

        USB_FLAG = FALSE;
        g_object_set_property (G_OBJECT (options->menu_usb_storage),
                                "active", &vt);
        g_object_set_property (G_OBJECT (options->menu_usb_netdev),
                                "active", &vf);
        USB_FLAG = TRUE;
    }

    if (options->bt_netdev) {

        BT_FLAG = FALSE;
        g_object_set_property (G_OBJECT (options->menu_bt_netdev),
                                "active", &vt);
        BT_FLAG = TRUE;
    } else {

        BT_FLAG = FALSE;
        g_object_set_property (G_OBJECT (options->menu_bt_netdev),
                                "active", &vf);
        BT_FLAG = TRUE;
    }

    if (options->wlan_netdev) {

        WLAN_FLAG = FALSE;
        g_object_set_property (G_OBJECT (options->menu_wlan_netdev),
                                "active", &vt);
        WLAN_FLAG = TRUE;
    } else {

        WLAN_FLAG = FALSE;
        g_object_set_property (G_OBJECT (options->menu_wlan_netdev),
                                "active", &vf);
        WLAN_FLAG = TRUE;
    }

    gtk_menu_popup (GTK_MENU (options->menu),
                     NULL,
                     NULL,
                     NULL,
                     NULL,
                     1, /* left mouse button */
                     gtk_get_current_event_time ());

    g_signal_emit_by_name (G_OBJECT (options->button), "released");
    options->button_released = FALSE;
}

static void
cs_activate_usb_storage (GtkWidget *widget, gpointer data)
{
    GValue vt = {0,};
    GValue vf = {0,};

    g_value_init (&vt, G_TYPE_BOOLEAN);
    g_value_init (&vf, G_TYPE_BOOLEAN);

    g_value_set_boolean (&vt, TRUE);
    g_value_set_boolean (&vf, FALSE);

    if (!options->usb_storage && USB_FLAG) {

        USB_FLAG = !USB_FLAG;
        g_object_set_property (G_OBJECT (options->menu_usb_storage),
                                "active", &vt);
        g_object_set_property (G_OBJECT (options->menu_usb_netdev),
                                "active", &vf);
        USB_FLAG = !USB_FLAG;

        system("sudo /usr/sbin/nm-usb-setup stop");

        options->usb_netdev = FALSE;
        options->usb_storage = TRUE;

        hildon_banner_show_information(NULL, NULL, "USB file storage enabled");
    }
}

static void
cs_activate_usb_netdev (GtkWidget *widget, gpointer data)
{
    GValue vt = {0,};
    GValue vf = {0,};

    g_value_init (&vt, G_TYPE_BOOLEAN);
    g_value_init (&vf, G_TYPE_BOOLEAN);

    g_value_set_boolean (&vt, TRUE);
    g_value_set_boolean (&vf, FALSE);

    if (!options->usb_netdev && USB_FLAG) {

        gchar *gw;
        NmSettings *settings = nm_settings_new ();

        if (!nm_settings_start (settings)) {

            hildon_banner_show_information(NULL, NULL,
                                            "Please, open PC-Connectivity manager and configure it.");
            return;
        }

        gw = nm_settings_get_value (settings, nm_environment_manager_get_active(), NM_USB_SETTINGS_GROUP,
                                     NM_SETTINGS_GATEWAY);

        USB_FLAG = !USB_FLAG;
        g_object_set_property (G_OBJECT (options->menu_usb_storage), "active", &vf);
        g_object_set_property (G_OBJECT (options->menu_usb_netdev), "active", &vt);
        USB_FLAG = !USB_FLAG;

        system("sudo /usr/sbin/nm-usb-setup start");
        system (g_strconcat ("sudo /sbin/ip route del default via ", gw,
                              " dev usb0", NULL));
        system (g_strconcat ("sudo /sbin/ip route append default via ", gw,
                              " dev usb0", NULL));

        options->usb_netdev = TRUE;
        options->usb_storage = FALSE;

        hildon_banner_show_information(NULL, NULL, "USB network enabled");
    }
}

static void
cs_activate_bt_netdev (GtkWidget *widget, gpointer data)
{
    GValue vt = {0,};
    GValue vf = {0,};

    g_value_init (&vt, G_TYPE_BOOLEAN);
    g_value_init (&vf, G_TYPE_BOOLEAN);

    g_value_set_boolean (&vt, TRUE);
    g_value_set_boolean (&vf, FALSE);

    if (!options->bt_netdev && BT_FLAG) {

        gchar *btaddr, *gw;
        NmSettings *settings = nm_settings_new ();

        if (!nm_settings_start (settings)) {

            hildon_banner_show_information(NULL, NULL,
                                            "Please, open PC-Connectivity manager and configure it.");
            return;
        }

        btaddr = nm_settings_get_value (settings, nm_environment_manager_get_active(), NM_BT_SETTINGS_GROUP,
                                         NM_SETTINGS_MAC);
        gw = nm_settings_get_value (settings, nm_environment_manager_get_active(), NM_BT_SETTINGS_GROUP,
                                     NM_SETTINGS_GATEWAY);

        if (strcmp (btaddr, NM_BT_DEFAULT_MAC) == 0) {

            hildon_banner_show_information (NULL, NULL,
                                             "Please, open network manager and\nconfigure bluetooth interface!");
            return;
        } else if (nm_bt_utils_check_pand (options->bt_utils, btaddr) == NULL) {

            hildon_banner_show_information (NULL, NULL,
                                            "Unable to connect. Check if PAND is running on host!");
            return;
        }

        if (!nm_bt_utils_connect (options->bt_utils, btaddr, gw)) {

            hildon_banner_show_information (NULL, NULL,
                                            "Could not connect to PC");
            return;
        }

        hildon_banner_show_information(NULL, NULL, "Bluetooth network enabled");

        BT_FLAG = !BT_FLAG;
        g_object_set_property (G_OBJECT (options->menu_bt_netdev),
                                "active", &vt);
        BT_FLAG = !BT_FLAG;
        options->bt_netdev = TRUE;
    } else if (BT_FLAG) {

        nm_bt_utils_remove_connection (options->bt_utils);

        hildon_banner_show_information(NULL, NULL,
                                        "Bluetooth network disabled");

        BT_FLAG = !BT_FLAG;
        g_object_set_property (G_OBJECT (options->menu_bt_netdev),
                                "active", &vf);
        BT_FLAG = !BT_FLAG;
        options->bt_netdev = FALSE;
    }
}

static void
cs_activate_wlan_netdev (GtkWidget *widget, gpointer data)
{
    GValue vt = {0,};
    GValue vf = {0,};

    g_value_init (&vt, G_TYPE_BOOLEAN);
    g_value_init (&vf, G_TYPE_BOOLEAN);

    g_value_set_boolean (&vt, TRUE);
    g_value_set_boolean (&vf, FALSE);

    gboolean status;

    if (!options->wlan_netdev && WLAN_FLAG) {

        status = system ("dbus-send --system --dest=com.nokia.icd /com/nokia/icd_ui com.nokia.icd_ui.disconnect boolean:true");
        g_usleep(2 * 1000000);
        status = system ("dbus-send --type=method_call --system --dest=com.nokia.icd /com/nokia/icd com.nokia.icd.connect string:devel_adhoc uint32:0");
        g_usleep(2 * 1000000);
        WLAN_FLAG = !WLAN_FLAG;
        g_object_set_property (G_OBJECT (options->menu_wlan_netdev), "active",
                                &vt);
        WLAN_FLAG = !WLAN_FLAG;
        options->wlan_netdev = TRUE;

        hildon_banner_show_information(NULL, NULL, "WLAN adhoc network enabled");
    } else if (options->wlan_netdev && WLAN_FLAG) {

        // FIXME: reconnect to previous wlan network
        status = system ("dbus-send --system --dest=com.nokia.icd /com/nokia/icd_ui com.nokia.icd_ui.disconnect boolean:true");
        WLAN_FLAG = !WLAN_FLAG;
        g_object_set_property (G_OBJECT (options->menu_wlan_netdev), "active",
                                &vf);
        WLAN_FLAG = !WLAN_FLAG;
        options->wlan_netdev = FALSE;

        hildon_banner_show_information(NULL, NULL, "WLAN adhoc network disabled");
    }
}
